/************************************************************************/
/*									*/
/*		gelodata.c						*/
/*									*/
/*	DATA flavor object routines for GELO				*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"






/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_SIZE		15
#define MIN_DEFAULT_SIZE	3

#define CIRCLE_SPACE		4
#define POLY_SPACE		4




/************************************************************************/
/*									*/
/*	Access macro definitions					*/
/*									*/
/************************************************************************/


#define SHAPE(o)	((GELO_SHAPE) ((o)->attributes[GELO_DATA_ATTR_SHAPE]))
#define FILL(o) 	((Integer) ((o)->attributes[GELO_DATA_ATTR_FILL]))
#define TEXT(o) 	((String) ((o)->attributes[GELO_DATA_ATTR_TEXT]))
#define INVERT(o)	((Boolean) ((o)->attributes[GELO_DATA_ATTR_INVERT]))
#define STYLE(o)	((Integer) ((o)->attributes[GELO_DATA_ATTR_STYLE]))
#define FONT(o) 	((Integer) ((o)->attributes[GELO_DATA_ATTR_FONT]))





/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


typedef struct _FONT_DATA {
   Integer size;
   String name;
   Integer font;
   Integer x_size;
   Integer y_size;
   Integer x_offset;
   Integer y_offset;
} FONT_DATA;





/************************************************************************/
/*									*/
/*	Table definitions						*/
/*									*/
/************************************************************************/


static	FONT_DATA	font_data[] = {
   { 3 },
   { 5 },
   { 8	},
   { 10 },
   { 14 },
   { 18 },
   { 0 }
};

#define MIN_FONT_SIZE		0
#define DEFAULT_FONT_SIZE	3
#define MAX_FONT_SIZE		5





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	GELO_ATTR_DATA_B	data_attrs[] = {
   { "SHAPE" },
   { "FILL" },
   { "TEXT" },
   { "INVERT" },
   { "STYLE" },
   { "FONT" },
   { NULL }
};


	String		GELO__data_fg;
	String		GELO__data_bg;






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	gelo_data_size();
static	void		gelo_data_layout();
static	void		gelo_data_draw();
static	void		gelo_data_inq_port_offset();
static	Boolean 	gelo_data_compare();
static	void		draw_text();
static	void		text_size();
static	void		text_draw();
static	void		text_extra_space();
static	void		draw_semicircle();




/************************************************************************/
/*									*/
/*	GELO_data_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
GELO_data_init()
{
   register Integer i;
   Integer x,y;

   ITRACE("GELO_data_init");

   GELO_flavor_define(GELO_FLAVOR_DATA,
			 "DATA",
			 data_attrs,
			 gelo_data_size,
			 gelo_data_layout,
			 gelo_data_draw,
			 NULL,NULL,NULL,NULL,
			 gelo_data_inq_port_offset,
			 gelo_data_compare);

   for (i = 0; font_data[i].size != 0; ++i) {
      font_data[i].name = ASHfont_name("fixed",font_data[i].size,ASH_FONT_ROMAN);
      font_data[i].font = ASHloadfont(font_data[i].name);
      ASHinq_text_next(font_data[i].font,"X",&x,&y);
      font_data[i].x_size = x;
      ASHinq_text_next(font_data[i].font,"Xp_^",&x,&y);
      font_data[i].y_size = y;
      ASHinq_text_offset(font_data[i].font,"Xp_^",&x,&y);
      font_data[i].x_offset = x;
      font_data[i].y_offset = y;
    };

   GELO__data_fg = ASHinq_resource("gelo.data.fg");
   GELO__data_bg = ASHinq_resource("gelo.data.bg");
   if (GELO__data_bg == NULL) GELO__data_bg = "white";
   else if (STREQL(GELO__data_bg,"*")) GELO__data_bg = NULL;
};





/************************************************************************/
/*									*/
/*	gelo_data_size -- set sizes for a DATA object			*/
/*									*/
/************************************************************************/


static Boolean
gelo_data_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   register Integer i;
   Integer xsz,ysz,minx,miny;
   Integer exx,exy;

   ITRACE("gelo_data_size 0x%x",o);

   minx = (GELOinq_zero_size_x(o) ? 0 : MIN_DEFAULT_SIZE);
   miny = (GELOinq_zero_size_y(o) ? 0 : MIN_DEFAULT_SIZE);

   if (minx+1 > mxx || miny+1 > mxy) return FALSE;

   switch (SHAPE(o)) {
      default :
	 break;
      case GELO_SHAPE_NONE :
	 if (FILL(o) == 0 &&
		!GELOinq_use_default_x(o) && !GELOinq_use_default_y(o)) {
	    minx = miny = 0;
	    GELOuse_min_size(o,TRUE);
	  };
	 break;
    };

   if (TEXT(o) == NULL) {
      xsz = DEFAULT_SIZE;
      ysz = DEFAULT_SIZE;
      exx = 1;
      exy = 1;
    }
   else {
      i = DEFAULT_FONT_SIZE + FONT(o);
      if (i < MIN_FONT_SIZE) i = MIN_FONT_SIZE;
      else if (i > MAX_FONT_SIZE) i = MAX_FONT_SIZE;
      for (  ; i >= MIN_FONT_SIZE; --i) {
	 text_size(TEXT(o),i,&xsz,&ysz);
	 text_extra_space(o,xsz,ysz,&exx,&exy);
	 xsz += 2*exx;
	 ysz += 2*exy;
	 switch (SHAPE(o)) {
	    case GELO_SHAPE_CIRCLE :
	    case GELO_SHAPE_SQUARE :
	       xsz = ysz = MAX(xsz,ysz);
	       break;
	  };
	 if (xsz+exx < mxx && ysz+exy < mxy) break;
       };
      if (i < MIN_FONT_SIZE) {
	 xsz = DEFAULT_SIZE;
	 ysz = DEFAULT_SIZE;
	 exx = 1;
	 exy = 1;
       };
    };

   if (xsz+exx > mxx) {
      ysz = (ysz*(mxx-exx))/xsz;
      xsz = mxx-exx;
    };
   if (ysz+exy > mxy) {
      xsz = (xsz*(mxy-exy))/ysz;
      ysz = mxy-exy;
    };

   if (xsz < minx) xsz = minx;
   if (ysz < miny) ysz = miny;

   xsz += exx;
   ysz += exy;
   if (minx != 0) minx += exx;
   if (miny != 0) miny += exy;

   GELO_set_x_size(o,xsz);
   GELO_set_y_size(o,ysz);
   GELO_set_x_min_size(o,minx);
   GELO_set_y_min_size(o,miny);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	gelo_data_layout -- layout a data object			*/
/*									*/
/************************************************************************/


static void
gelo_data_layout(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   ITRACE("gelo_data_layout 0x%x 0x%x",gw,o);
};





/************************************************************************/
/*									*/
/*	gelo_data_draw -- draw a data object				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
gelo_data_draw(gw,o,hifg)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_DRAW_FLAGS hifg;
{
   Integer lx,by,rx,ty;
   Integer r;
   Integer x[10],y[10],ct;
   Integer fill;
   Integer xcl,tb,fgd,bgd,dfgd,dbgd;

   ITRACE("gelo_data_draw 0x%x 0x%x",gw,o);

   if (!GELO_inq_position(gw,o,&lx,&by,&rx,&ty)) return;

   fill = FILL(o);
   fgd = ASHinq_color(gw->window);
   bgd = ASHinq_background_color(gw->window);
   tb = ASHinq_text_background_color(gw->window);

   if (hifg == GELO_DRAW_HILITE) {
      dfgd = fgd;
      dbgd = bgd;
    }
   else {
      if (GELO__data_fg == NULL) dfgd = fgd;
      else dfgd = ASHlookup_color(gw->window,GELO__data_fg);
      if (GELO__data_bg == NULL) dbgd = bgd;
      else dbgd = ASHlookup_color(gw->window,GELO__data_bg);
    };

   if (INVERT(o)) {
      if (fill == 0) fill = 1;
      else {
	 xcl = dfgd;
	 dfgd = dbgd;
	 dbgd = xcl;
       };
    };

   ASHcolor(gw->window,dfgd);
   ASHtext_color(gw->window,dfgd);
   ASHbackground_color(gw->window,dbgd);
   ASHtext_background_color(gw->window,dbgd);

   switch (SHAPE(o)) {
      case GELO_SHAPE_TRIANGLE_UP :
      case GELO_SHAPE_SEMI_UP :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = by;
	 x[2] = rx; y[2] = by;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_DOWN :
      case GELO_SHAPE_SEMI_DOWN :
	 x[0] = (lx+rx)/2; y[0] = by;
	 x[1] = lx; y[1] = ty;
	 x[2] = rx; y[2] = ty;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_LEFT :
      case GELO_SHAPE_SEMI_LEFT :
      case GELO_SHAPE_TBOX_LEFT :
	 x[0] = lx; y[0] = (by+ty)/2;
	 x[1] = rx; y[1] = ty;
	 x[2] = rx; y[2] = by;
	 ct = 3;
	 break;
      case GELO_SHAPE_TRIANGLE_RIGHT :
      case GELO_SHAPE_SEMI_RIGHT :
      case GELO_SHAPE_TBOX_RIGHT :
	 x[0] = rx; y[0] = (by+ty)/2;
	 x[1] = lx; y[1] = by;
	 x[2] = lx; y[2] = ty;
	 ct = 3;
	 break;
      case GELO_SHAPE_DIAMOND :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = (by+ty)/2;
	 x[2] = (lx+rx)/2; y[2] = by;
	 x[3] = rx; y[3] = (by+ty)/2;
	 ct = 4;
	 break;
      case GELO_SHAPE_PENTAGON :
	 x[0] = (lx+rx)/2; y[0] = ty;
	 x[1] = lx; y[1] = (by+ty)/2;
	 x[2] = (lx+lx+rx)/3; y[2] = by;
	 x[3] = (lx+rx+rx)/3; y[3] = by;
	 x[4] = rx; y[4] = (by+ty)/2;
	 ct = 5;
	 break;
      case GELO_SHAPE_HEXAGON :
	 x[0] = (lx+lx+rx)/3; y[0] = ty;
	 x[1] = lx; y[1] = (ty+by)/2;
	 x[2] = (lx+lx+rx)/3; y[2] = by;
	 x[3] = (lx+rx+rx)/3; y[3] = by;
	 x[4] = rx; y[4] = (ty+by)/2;
	 x[5] = (lx+rx+rx)/3; y[5] = ty;
	 ct = 6;
	 break;
      case GELO_SHAPE_OCTAGON :
	 x[0] = (lx+lx+rx)/3; y[0] = ty;
	 x[1] = lx; y[1] = (ty+ty+by)/3;
	 x[2] = lx; y[2] = (ty+by+by)/3;
	 x[3] = (lx+lx+rx)/3; y[3] = by;
	 x[4] = (lx+rx+rx)/3; y[4] = by;
	 x[5] = rx; y[5] = (ty+by+by)/3;
	 x[6] = rx; y[6] = (ty+ty+by)/3;
	 x[7] = (lx+rx+rx)/3; y[7] = ty;
	 ct = 8;
	 break;
      case GELO_SHAPE_RECTANGLE :
      case GELO_SHAPE_SQUARE :
	 x[0] = lx; y[0] = by;
	 x[1] = lx; y[1] = ty;
	 x[2] = rx; y[2] = ty;
	 x[3] = rx; y[3] = by;
	 ct = 4;
	 break;
      case GELO_SHAPE_VLINE :
	 x[0] = lx; y[0] = by;
	 x[1] = lx; y[1] = ty;
	 x[2] = lx+1; y[2] = ty;
	 x[3] = lx+1; y[3] = by;
	 ct = 4;
	 break;
      default :
	 ct = 0;
	 break;
    };

   if (fill == 0) {
      ASHclear_box(gw->window,lx,by,rx,ty);
      if (STYLE(o) != (int) GELO_STYLE_INVIS) {
	 ASHline_style(gw->window,STYLE(o));
	 switch (SHAPE(o)) {
	    case GELO_SHAPE_NONE :
	       break;
	    case GELO_SHAPE_NULL_PTR :
	       ASHbox(gw->window,lx,by,rx,ty);
	       ASHline(gw->window,lx,ty,rx,by);
	       break;
	    case GELO_SHAPE_FLD_PTR :
	       ASHbox(gw->window,lx,by,rx,ty);
	       ASHfill_box(gw->window,(lx+rx)/2-2,(by+ty)/2-2,(lx+rx)/2+2,(by+ty)/2+2);
	       break;
	    case GELO_SHAPE_CIRCLE :
	    case GELO_SHAPE_ELLIPSE :
	       ASHellipse(gw->window,(lx+rx)/2,(ty+by)/2,abs(lx-rx)/2,abs(ty-by)/2);
	       break;
	    case GELO_SHAPE_ROUND_RECTANGLE :
	       if (abs(rx-lx) > abs(ty-by)) r = abs(ty-by)/4;
	       else r = abs(rx-lx)/4;
	       ASHround_box(gw->window,lx,by,rx,ty,r);
	       break;
	    case GELO_SHAPE_SEMI_UP :
	       draw_semicircle(gw->window,lx,by,rx,ty,1,0);
	       break;
	    case GELO_SHAPE_SEMI_DOWN :
	       draw_semicircle(gw->window,lx,by,rx,ty,3,0);
	       break;
	    case GELO_SHAPE_SEMI_LEFT :
	       draw_semicircle(gw->window,lx,by,rx,ty,2,0);
	       break;
	    case GELO_SHAPE_SEMI_RIGHT :
	       draw_semicircle(gw->window,lx,by,rx,ty,0,0);
	       break;
	    case GELO_SHAPE_TBOX_LEFT :
	    case GELO_SHAPE_TBOX_RIGHT :
	       ASHbox(gw->window,lx,by,rx,ty);
	       x[ct] = x[0];
	       y[ct] = y[0];
	       ASHpolyline(gw->window,ct+1,x,y);
	       break;
	    case GELO_SHAPE_RECTANGLE :
	    case GELO_SHAPE_SQUARE :
	    case GELO_SHAPE_TRIANGLE_UP :
	    case GELO_SHAPE_TRIANGLE_DOWN :
	    case GELO_SHAPE_TRIANGLE_LEFT :
	    case GELO_SHAPE_TRIANGLE_RIGHT :
	    case GELO_SHAPE_DIAMOND :
	    case GELO_SHAPE_PENTAGON :
	    case GELO_SHAPE_HEXAGON :
	    case GELO_SHAPE_OCTAGON :
	    case GELO_SHAPE_VLINE :
	       x[ct] = x[0];
	       y[ct] = y[0];
	       ASHpolyline(gw->window,ct+1,x,y);
	       break;
	  };
	 ASHline_style(gw->window,ASH_STYLE_SOLID);
       };
    }
   else {
      ASHclear_box(gw->window,lx,by,rx,ty);
      ASHfill(gw->window,fill);
      ASHcombination_rule(gw->window,3);
      switch (SHAPE(o)) {
	 case GELO_SHAPE_NONE :
	 case GELO_SHAPE_RECTANGLE :
	 case GELO_SHAPE_SQUARE :
	    ASHrectangle(gw->window,lx,by,rx,ty);
	    break;
	 case GELO_SHAPE_NULL_PTR :
	    ASHrectangle(gw->window,lx,by,rx,ty);
	    ASHline(gw->window,lx,by,rx,ty);
	    break;
	 case GELO_SHAPE_FLD_PTR :
	    ASHrectangle(gw->window,lx,by,rx,ty);
	    ASHfill_box(gw->window,(lx+rx)/2-2,(by+ty)/2-2,(lx+rx)/2+2,(by+ty)/2+2);
	    break;
	 case GELO_SHAPE_CIRCLE :
	 case GELO_SHAPE_ELLIPSE :
	    ASHfilled_ellipse(gw->window,(lx+rx)/2,(ty+by)/2,abs(lx-rx)/2,abs(ty-by)/2);
	    break;
	 case GELO_SHAPE_ROUND_RECTANGLE :
	    if (abs(rx-lx) > abs(ty-by)) r = abs(ty-by)/4;
	    else r = abs(rx-lx)/4;
	    ASHround_rectangle(gw->window,lx,by,rx,ty,r);
	    break;
	 case GELO_SHAPE_SEMI_UP :
	    draw_semicircle(gw->window,lx,by,rx,ty,1,fill);
	    break;
	 case GELO_SHAPE_SEMI_DOWN :
	    draw_semicircle(gw->window,lx,by,rx,ty,3,fill);
	    break;
	 case GELO_SHAPE_SEMI_LEFT :
	    draw_semicircle(gw->window,lx,by,rx,ty,2,fill);
	    break;
	 case GELO_SHAPE_SEMI_RIGHT :
	    draw_semicircle(gw->window,lx,by,rx,ty,0,fill);
	    break;
	 case GELO_SHAPE_TBOX_LEFT :
	 case GELO_SHAPE_TBOX_RIGHT :
	    ASHbox(gw->window,lx,by,rx,ty);
	    ASHpolygon(gw->window,ct,x,y);
	    break;
	 case GELO_SHAPE_TRIANGLE_UP :
	 case GELO_SHAPE_TRIANGLE_DOWN :
	 case GELO_SHAPE_TRIANGLE_LEFT :
	 case GELO_SHAPE_TRIANGLE_RIGHT :
	 case GELO_SHAPE_DIAMOND :
	 case GELO_SHAPE_PENTAGON :
	 case GELO_SHAPE_HEXAGON :
	 case GELO_SHAPE_OCTAGON :
	    ASHpolygon(gw->window,ct,x,y);
	    break;
	 case GELO_SHAPE_VLINE :
	    x[ct] = x[0];
	    y[ct] = y[0];
	    ASHpolyline(gw->window,ct+1,x,y);
	    break;
       };
      ASHfill(gw->window,1);
    };

   draw_text(gw,o,lx,by,rx,ty,fill);

   ASHcombination_rule(gw->window,3);

   ASHcolor(gw->window,fgd);
   ASHbackground_color(gw->window,bgd);
   ASHtext_color(gw->window,fgd);
   ASHtext_background_color(gw->window,tb);
};





/************************************************************************/
/*									*/
/*	gelo_data_inq_port_offset -- get offset for ports		*/
/*									*/
/************************************************************************/


static void
gelo_data_inq_port_offset(obj,port,lx,by,rx,ty,dxp,dyp)
   GELO_OBJECT obj;
   GELO_PORT port;
   Integer lx,by,rx,ty;
   Integer * dxp, * dyp;
{
   float z;
   Integer dx,dy,dz;

   DTRACE("gelo_data_inq_port_offset 0x%x %d %d %d %d %d",obj,port,lx,by,rx,ty);

   dx = dy = 0;

   switch (SHAPE(obj)) {
      default :
      case GELO_SHAPE_RECTANGLE :
      case GELO_SHAPE_SQUARE :
      case GELO_SHAPE_NULL_PTR :
      case GELO_SHAPE_TBOX_LEFT :
      case GELO_SHAPE_TBOX_RIGHT :
	 break;
      case GELO_SHAPE_ELLIPSE :
      case GELO_SHAPE_CIRCLE :
	 z = abs(rx-lx)/2;
	 z *= 0.29289322;
	 dx = z + 0.5;
	 z = abs(ty-by)/2;
	 z *= 0.29289322;
	 dy = z + 0.5;
	 switch (port) {
	    case GELO_PORT_TOP_LEFT :
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = -dx;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dy = -dy;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = -dx;
	       dy = -dy;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_ROUND_RECTANGLE :
	 if (abs(rx-lx) > abs(ty-by)) z = abs(ty-by)/4;
	 else z = abs(rx-lx)/4;
	 z *= (1.0-0.7071);
	 dx = z;
	 dy = z;
	 switch (port) {
	    case GELO_PORT_TOP_LEFT :
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = -dx;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dy = -dy;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = -dx;
	       dy = -dy;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_TRIANGLE_UP :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dx = (rx-lx)/2;
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = (lx-rx)/2;
	       break;
	    case GELO_PORT_LEFT :
	       dx = (rx-lx)/4;
	       break;
	    case GELO_PORT_RIGHT :
	       dx = (lx-rx)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_TRIANGLE_DOWN :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dx = (rx-lx)/2;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = (lx-rx)/2;
	       break;
	    case GELO_PORT_LEFT :
	       dx = (rx-lx)/4;
	       break;
	    case GELO_PORT_RIGHT :
	       dx = (lx-rx)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_TRIANGLE_LEFT :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dy = (by-ty)/2;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dy = (ty-by)/2;
	       break;
	    case GELO_PORT_TOP :
	    case GELO_PORT_TOP_SEQ :
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_BOTTOM :
	    case GELO_PORT_BOTTOM_SEQ :
	       dy = (ty-by)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_TRIANGLE_RIGHT :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dy = (by-ty)/2;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dy = (ty-by)/2;
	       break;
	    case GELO_PORT_TOP :
	    case GELO_PORT_TOP_SEQ :
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_BOTTOM :
	    case GELO_PORT_BOTTOM_SEQ :
	       dy = (ty-by)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_SEMI_UP :
	 z = abs(rx-lx)/2;
	 dz = z*0.1339746;
	 z *= 0.29289322;
	 dx = z + 0.5;
	 z = abs(ty-by);
	 z *= 0.29289322;
	 dy = z + 0.5;
	 switch (port) {
	    case GELO_PORT_TOP_LEFT :
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = -dx;
	       break;
	    case GELO_PORT_LEFT :
	       dx = dz;
	       dy = 0;
	       break;
	    case GELO_PORT_RIGHT :
	       dx = -dz;
	       dy = 0;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_SEMI_DOWN :
	 z = abs(rx-lx)/2;
	 dz = z*0.1339746;
	 z *= 0.29289322;
	 dx = z + 0.5;
	 z = abs(ty-by);
	 z *= 0.29289322;
	 dy = z + 0.5;
	 switch (port) {
	    case GELO_PORT_BOTTOM_LEFT :
	       dy = -dy;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dy = -dy;
	       dx = -dx;
	       break;
	    case GELO_PORT_LEFT :
	       dx = dz;
	       dy = 0;
	       break;
	    case GELO_PORT_RIGHT :
	       dx = -dz;
	       dy = 0;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_SEMI_LEFT :
	 z = abs(rx-lx);
	 z *= 0.29289322;
	 dx = z + 0.5;
	 z = abs(ty-by)/2;
	 dz = z*0.1339746;
	 z *= 0.29289322;
	 dy = z + 0.5;
	 switch (port) {
	    case GELO_PORT_TOP_LEFT :
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dy = -dy;
	       break;
	    case GELO_PORT_TOP :
	       dy = dz;
	       dx = 0;
	       break;
	    case GELO_PORT_BOTTOM :
	       dy = -dz;
	       dx = 0;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_SEMI_RIGHT :
	 z = abs(rx-lx);
	 z *= 0.29289322;
	 dx = z + 0.5;
	 z = abs(ty-by)/2;
	 dz = z*0.1339746;
	 z *= 0.29289322;
	 dy = z + 0.5;
	 switch (port) {
	    case GELO_PORT_TOP_RIGHT :
	       dx = -dx;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dy = -dy;
	       dx = -dx;
	       break;
	    case GELO_PORT_TOP :
	       dy = dz;
	       dx = 0;
	       break;
	    case GELO_PORT_BOTTOM :
	       dy = -dz;
	       dx = 0;
	       break;
	    default :
	       dx = dy = 0;
	       break;
	  };
	 break;
      case GELO_SHAPE_PENTAGON :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dx = (rx-lx)/4;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = (lx-rx)/4;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dx = (rx-lx)/6;
	       dy = (ty-by)/4;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = (lx-rx)/6;
	       dy = (ty-by)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_HEXAGON :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dx = (rx-lx)/6;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = (lx-rx)/6;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dx = (rx-lx)/6;
	       dy = (ty-by)/4;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = (lx-rx)/6;
	       dy = (ty-by)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_OCTAGON :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dx = (rx-lx)/6;
	       dy = (by-ty)/6;
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = (lx-rx)/6;
	       dy = (by-ty)/6;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dx = (rx-lx)/6;
	       dy = (ty-by)/6;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = (lx-rx)/6;
	       dy = (ty-by)/6;
	       break;
	  };
	 break;
      case GELO_SHAPE_DIAMOND :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP_LEFT :
	       dx = (rx-lx)/4;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_TOP_RIGHT :
	       dx = (lx-rx)/4;
	       dy = (by-ty)/4;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	       dx = (rx-lx)/4;
	       dy = (ty-by)/4;
	       break;
	    case GELO_PORT_BOTTOM_RIGHT :
	       dx = (lx-rx)/4;
	       dy = (ty-by)/4;
	       break;
	  };
	 break;
      case GELO_SHAPE_VLINE :
	 switch (port) {
	    default :
	       break;
	    case GELO_PORT_TOP :
	    case GELO_PORT_BOTTOM :
	    case GELO_PORT_CENTER :
	    case GELO_PORT_TOP_SEQ :
	    case GELO_PORT_BOTTOM_SEQ :
	       dx = (lx-rx)/2+1;
	       break;
	    case GELO_PORT_RIGHT :
	    case GELO_PORT_TOP_RIGHT :
	    case GELO_PORT_BOTTOM_RIGHT :
	    case GELO_PORT_RIGHT_SEQ :
	       dx = (lx-rx)+1;
	       break;
	  };
	 switch (port) {
	    case GELO_PORT_TOP :
	    case GELO_PORT_TOP_SEQ :
	       dy = (by-ty)/6;
	       break;
	    case GELO_PORT_TOP_LEFT :
	    case GELO_PORT_TOP_RIGHT :
	       dy = (by-ty)/3;
	       break;
	    case GELO_PORT_BOTTOM_LEFT :
	    case GELO_PORT_BOTTOM_RIGHT :
	       dy = (ty-by)/3;
	       break;
	    case GELO_PORT_BOTTOM :
	    case GELO_PORT_BOTTOM_SEQ :
	       dy = (ty-by)/6;
	       break;
	    default :
	       break;
	  };
	 break;
    };

   *dxp = dx;
   *dyp = dy;
};





/************************************************************************/
/*									*/
/*	gelo_data_compare -- compare two data objects			*/
/*									*/
/************************************************************************/


static Boolean
gelo_data_compare(o1,o2,o1p,o2p)
   GELO_OBJECT o1;
   GELO_OBJECT o2;
   GELO_OBJECT *o1p;
   GELO_OBJECT *o2p;
{
   register Boolean fg;

   DTRACE("gelo_data_compare 0x%x 0x%x",o1,o2);

   if (SHAPE(o1) != SHAPE(o2)) fg = FALSE;
   else if (FILL(o1) != FILL(o2)) fg = FALSE;
   else if (TEXT(o1) != TEXT(o2) &&
	       (TEXT(o1) == NULL || TEXT(o2) == NULL ||
		   STRNEQ(TEXT(o1),TEXT(o2)))) fg = FALSE;
   else if (INVERT(o1) != INVERT(o2)) fg = FALSE;
   else if (STYLE(o1) != STYLE(o2)) fg = FALSE;
   else if (FONT(o1) != FONT(o2)) fg = FALSE;
   else fg = TRUE;

   if (!fg) {
      *o1p = o1;
      *o2p = o2;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	draw_text -- draw text string of appropriate size		*/
/*									*/
/************************************************************************/


static void
draw_text(gw,o,lx,by,rx,ty,fill)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer lx,by,rx,ty;
   Integer fill;
{
   Integer xsz,ysz;
   Integer fmn,fmx,fsz;
   Integer exx,exy;

   DTRACE("draw_text 0x%x 0x%x %d %d %d %d",gw,o,lx,by,rx,ty);

   if (TEXT(o) == NULL) return;

   fmn = MIN_FONT_SIZE;
   fmx = MAX_FONT_SIZE;

   while (fmn <= fmx) {
      fsz = (fmn+fmx+1)/2;
      text_size(TEXT(o),fsz,&xsz,&ysz);
      text_extra_space(o,xsz,ysz,&exx,&exy);
      if (xsz > rx-lx+1-2*exx || ysz > by-ty+1-2*exy) {
	 fmx = fsz-1;
	 if (fmn > fmx) {
	    text_size(TEXT(o),fmn-1,&xsz,&ysz);
	    text_extra_space(o,xsz,ysz,&exx,&exy);
	  };
       }
      else {
	 fmn = fsz+1;
       };
    };

   lx += exx;
   rx -= exx;
   by -= exy;
   ty += exy;

   fsz = fmn-1;
   if (fsz >= 0) {
      if (fill) {
	 ASHclear_box(gw->window,(lx+rx-xsz)/2,(by+ty+ysz)/2,(lx+rx+xsz)/2,(by+ty-ysz)/2);
       };
      ASHfont(gw->window,font_data[fsz].font);
      text_draw(gw->window,fsz,(lx+rx-xsz)/2,(ty+by-ysz)/2,TEXT(o));
    };
};





/************************************************************************/
/*									*/
/*	text_size -- return expected size for text string		*/
/*									*/
/************************************************************************/


static void
text_size(s,sz,xp,yp)
   String s;
   Integer sz;
   Integer *xp;
   Integer *yp;
{
   register Integer i,j;

   *xp = font_data[sz].x_size;
   *yp = font_data[sz].y_size;

   j = 0;
   for (i = 0; s[i] != 0; ++i) {
      if (s[i] == '\n') {
	 *xp = MAX(*xp,(j)*font_data[sz].x_size+2);
	 *yp += font_data[sz].y_size+1;
	 j = 0;
       }
      else if (s[i] == '\t') {
	 ++j;
	 while ((j&3) != 0) ++j;
       }
      else ++j;
    };

   if (j != 0) {
      *xp = MAX(*xp,(j)*font_data[sz].x_size);
    };

   DTRACE("text_size %s %d => %d %d",s,sz,*xp,*yp);
};





/************************************************************************/
/*									*/
/*	text_draw -- actually output text				*/
/*									*/
/************************************************************************/


static void
text_draw(w,ft,lx,ty,s)
   ASH_WINDOW w;
   Integer ft;
   Integer lx;
   Integer ty;
   String s;
{
   register Integer i,j;
   Character buf[1024];

   DTRACE("text_draw 0x%x %d %d %d %s",w,ft,lx,ty,s);

   j = 0;
   for (i = 0; s[i] != 0; ++i) {
      if (s[i] == '\n') {
	 buf[j] = 0;
	 ASHtext(w,lx+font_data[ft].x_offset+1,
		    ty+font_data[ft].y_size-font_data[ft].y_offset,
		    buf);
	 ty += font_data[ft].y_size+1;
	 j = 0;
       }
      else if (s[i] == '\t') {
	 buf[j++] = ' ';
	 while ((j&3) != 0) buf[j++] = ' ';
       }
      else buf[j++] = s[i];
    };

   if (j != 0) {
      buf[j] = 0;
      ASHtext(w,lx+font_data[ft].x_offset+1,
		 ty+font_data[ft].y_size-font_data[ft].y_offset,
		 buf);
    };
};





/************************************************************************/
/*									*/
/*	text_extra_space -- get extra space for text			*/
/*									*/
/************************************************************************/


static void
text_extra_space(o,x,y,xp,yp)
   GELO_OBJECT o;
   Integer x,y;
   Integer *xp, *yp;
{
   register Integer xsz,ysz;

   DTRACE("text_extra_space 0x%x",o);

   switch (SHAPE(o)) {
      case GELO_SHAPE_OCTAGON :
      case GELO_SHAPE_ROUND_RECTANGLE :
      default :
	 xsz = 2;
	 ysz = 1;
	 break;
      case GELO_SHAPE_VLINE :
	 ysz = 1;
	 xsz = 3;
	 break;
      case GELO_SHAPE_HEXAGON :
      case GELO_SHAPE_ELLIPSE :
      case GELO_SHAPE_CIRCLE :
	 xsz = CIRCLE_SPACE;
	 ysz = CIRCLE_SPACE;
	 break;
      case GELO_SHAPE_NONE :
	 xsz = 0;
	 ysz = 0;
	 break;
      case GELO_SHAPE_TRIANGLE_UP :
      case GELO_SHAPE_SEMI_UP :
      case GELO_SHAPE_TRIANGLE_DOWN :
      case GELO_SHAPE_SEMI_DOWN :
	 xsz = x/2;
	 ysz = y/2;
	 break;
      case GELO_SHAPE_TRIANGLE_LEFT :
      case GELO_SHAPE_SEMI_LEFT :
      case GELO_SHAPE_TRIANGLE_RIGHT :
      case GELO_SHAPE_SEMI_RIGHT :
	 xsz = x/2;
	 ysz = y/2;
	 break;
      case GELO_SHAPE_DIAMOND :
	 xsz = x/3;
	 ysz = y/3;
	 break;
      case GELO_SHAPE_PENTAGON :
	 xsz = POLY_SPACE;
	 ysz = POLY_SPACE;
	 break;
    };

   *xp = xsz;
   *yp = ysz;
};





/************************************************************************/
/*									*/
/*	draw_semicircle -- draw a semicircle				*/
/*									*/
/************************************************************************/


static void
draw_semicircle(w,lx,by,rx,ty,dir,fill)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
   Integer dir;
   Integer fill;
{
   Integer xc,yc,rdx,rdy;
   Integer x0,y0,x1,y1;
   Integer clip;

   DTRACE("draw_semicircle 0x%x %d %d %d %d %d %d",w,lx,by,rx,ty,dir,fill);

   switch (dir) {
      case 0 :
	 xc = lx;
	 yc = (by+ty)/2;
	 rdx = rx-lx;
	 rdy = (by-ty)/2;
	 x0 = x1 = lx;
	 y0 = ty;
	 y1 = by;
	 break;
      case 1 :
	 xc = (lx+rx)/2;
	 yc = by;
	 rdx = (rx-lx)/2;
	 rdy = by-ty;
	 x0 = lx;
	 x1 = rx;
	 y0 = y1 = by;
	 break;
      case 2 :
	 xc = rx;
	 yc = (by+ty)/2;
	 rdx = rx-lx;
	 rdy = (by-ty)/2;
	 x0 = x1 = rx;
	 y0 = ty;
	 y1 = by;
	 break;
      case 3 :
	 xc = (lx+rx)/2;
	 yc = ty;
	 rdx = (rx-lx)/2;
	 rdy = by-ty;
	 x0 = lx;
	 x1 = rx;
	 y0 = y1 = ty;
	 break;
    };

   ASHclip_region(w,lx,by,rx,ty);
   clip = ASHclip(w,TRUE);

   if (fill == 0) {
      ASHellipse(w,xc,yc,rdx,rdy);
      ASHline(w,x0,y0,x1,y1);
    }
   else {
      ASHfilled_ellipse(w,xc,yc,rdx,rdy);
    };

   ASHclip(w,clip);
};





/* end of gelodata.c */


