/************************************************************************/
/*									*/
/*		geloconn.c						*/
/*									*/
/*	Connection routines for GELO					*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"
#include "gelo_layout.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define ARROW_SIZE	7
#define LABEL_FONT	"fixed",10,ASH_FONT_ROMAN
#define MAX_CORRELATE	1
#define EPSILON 	3
#define DEFAULT_SUBPORT (MAX_SEQ_PORTS/2)





/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


typedef struct _GELO_CONNECT {
   GELO_OBJECT	from;
   GELO_PORT	from_port;
   Integer	from_subport;
   GELO_OBJECT	to;
   GELO_PORT	to_port;
   Integer	to_subport;
   GELO_OBJECT	arc_object;
   ASH_LINE_STYLE style;
   GELO_ARC_ARROW arrow;
   GELO_ARC_LABEL_POS start_label_pos;
   Boolean	start_label_box;
   String	start_label;
   GELO_ARC_LABEL_POS mid_label_pos;
   Boolean	mid_label_box;
   String	mid_label;
   GELO_ARC_LABEL_POS end_label_pos;
   Boolean	end_label_box;
   String	end_label;
   Sequence	pivots;
   Boolean	from_any;
   Boolean	to_any;
   Integer	lbllx,lblby,lblrx,lblty;
   String	from_data,to_data;
} GELO_CONNECT_B;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	label_font;
static	Function_Ptr	port_rtn;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		hilite_line();
static	void		hilite_lines();
static	Boolean 	checkline();
static	GELO_PORT	find_port();
static	Integer 	real_subport();
static	Boolean 	connect_loc();
static	void		draw_arrow();
static	void		draw_label();
static	void		middle_label();
static	void		source_label();




/************************************************************************/
/*									*/
/*	GELO_conn_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
GELO_conn_init()
{
   ITRACE("GELO_conn_init");

   label_font = ASHfont_id(LABEL_FONT);
   port_rtn = NULL;
};





/************************************************************************/
/*									*/
/*	GELOnew_connect -- define a new connection			*/
/*									*/
/************************************************************************/


GELO_CONNECT
GELOnew_connect(f,fp,t,tp)
   GELO_OBJECT f;
   GELO_PORT fp;
   GELO_OBJECT t;
   GELO_PORT tp;
{
   register GELO_CONNECT c;

   c = (GELO_CONNECT) malloc(sizeof(GELO_CONNECT_B));

   c->from = f;
   c->from_port = fp;
   c->from_subport = -1;
   c->to = t;
   c->to_port = tp;
   c->to_subport = -1;
   c->arc_object = NULL;
   c->style = ASH_STYLE_SOLID;
   c->arrow = GELO_ARROW_SINGLE_ALL;
   c->start_label_pos = GELO_LABEL_NONE;
   c->start_label_box = TRUE;
   c->start_label = NULL;
   c->mid_label_pos = GELO_LABEL_NONE;
   c->mid_label_box = TRUE;
   c->mid_label = NULL;
   c->end_label_pos = GELO_LABEL_NONE;
   c->end_label_box = TRUE;
   c->end_label = NULL;
   c->pivots = NULL;
   c->from_any = (fp == GELO_PORT_ANY ? TRUE : FALSE);
   c->to_any = (tp == GELO_PORT_ANY ? TRUE : FALSE);
   c->lbllx = c->lblby = c->lblrx = c->lblty = -1;
   c->from_data = NULL;
   c->to_data = NULL;

   TRACE("GELOnew_connect 0x%x %d 0x%x %d => 0x%x",f,fp,t,tp,c);

   return c;
};





/************************************************************************/
/*									*/
/*	GELOconnect_label -- set label for connection			*/
/*	GELOconnect_arc_style -- describe arc style for connection	*/
/*	GELOconnect_label_style -- describe label style for connection	*/
/*									*/
/************************************************************************/


void
GELOconnect_label(c,loc,lbl)
   GELO_CONNECT c;
   GELO_ARC_LABEL_LOC loc;
   String lbl;
{
   TRACE("GELOconnect_label 0x%x %d 0x%x",c,loc,lbl);

   if (loc == GELO_LABEL_ANY) {
      if (c->mid_label_pos != GELO_LABEL_NONE) loc = GELO_LABEL_MIDDLE;
      else if (c->start_label_pos != GELO_LABEL_NONE) loc = GELO_LABEL_START;
      else if (c->end_label_pos != GELO_LABEL_NONE) loc = GELO_LABEL_END;
      else return;
    };

   switch (loc) {
      case GELO_LABEL_START :
	 c->start_label = lbl;
	 break;
      case GELO_LABEL_MIDDLE :
	 c->mid_label = lbl;
	 break;
      case GELO_LABEL_END :
	 c->end_label = lbl;
	 break;
    };
};





void
GELOconnect_arc_style(c,st,ar)
   GELO_CONNECT c;
   ASH_LINE_STYLE st;
   GELO_ARC_ARROW ar;
{
   TRACE("GELOconnect_arc_style 0x%x %d %d",c,st,ar);

   c->style = st;
   c->arrow = ar;
};





void
GELOconnect_label_style(c,loc,pos,box)
   GELO_CONNECT c;
   GELO_ARC_LABEL_LOC loc;
   GELO_ARC_LABEL_POS pos;
   Boolean box;
{
   TRACE("GELOconnect_label_style 0x%x %d %d %d",c,loc,pos,box);

   switch (loc) {
      case GELO_LABEL_START :
	 c->start_label_pos = pos;
	 c->start_label_box = box;
	 break;
      case GELO_LABEL_MIDDLE :
	 c->mid_label_pos = pos;
	 c->mid_label_box = box;
	 break;
      case GELO_LABEL_END :
	 c->end_label_pos = pos;
	 c->end_label_box = box;
	 break;
    };
};





/************************************************************************/
/*									*/
/*	GELOconnect_port_routine -- set port routine			*/
/*									*/
/************************************************************************/


void
GELOconnect_port_routine(rtn)
   Function_Ptr rtn;
{
   ENTER("GELOconnect_port_routine 0x%x",rtn);

   port_rtn = rtn;
};





/************************************************************************/
/*									*/
/*	GELOconnect_set_arc_object -- set GELO_OBJECT associated with arc */
/*									*/
/************************************************************************/


void
GELOconnect_set_arc_object(conn,obj)
   GELO_CONNECT conn;
   GELO_OBJECT obj;
{
   TRACE("GELOconnect_set_arc_object 0x%x 0x%x",conn,obj);

   conn->arc_object = obj;
};





/************************************************************************/
/*									*/
/*	GELO_connect_pivot -- add pivot to connection			*/
/*	GELO_clear_pivots -- clear pivots from connection		*/
/*	GELO_connect_inq_pivots -- return list of pivots of connection	*/
/*									*/
/************************************************************************/


void
GELO_connect_pivot(c,x,y)
   GELO_CONNECT c;
   GELO_COORD x,y;
{
   ITRACE("GELO_connect_pivot 0x%x %d %d",c,x,y);

   c->pivots = APPEND(x,c->pivots);
   c->pivots = APPEND(y,c->pivots);
};





void
GELO_clear_pivots(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_clear_pivots 0x%x",c);

   if (c->pivots != NULL) {
      LFREE(c->pivots);
      c->pivots = NULL;
    };
};





Sequence
GELO_connect_inq_pivots(c)
   GELO_CONNECT c;
{
   return c->pivots;
};





/************************************************************************/
/*									*/
/*	GELO_draw_connections -- draw a list of connections		*/
/*	GELO_draw_connect -- draw a single connection			*/
/*									*/
/************************************************************************/


void
GELO_draw_connections(gw,cl,hifg)
   GELO_WINDOW gw;
   Sequence cl;
   GELO_DRAW_FLAGS hifg;
{
   register Sequence l;
   register GELO_CONNECT c;

   ITRACE("GELO_draw_connections 0x%x 0x%x",gw,cl);

   forin (c,GELO_CONNECT,l,cl) {
      GELO_draw_connect(gw,c,hifg);
    };
};





void
GELO_draw_connect(gw,c,hifg)
   GELO_WINDOW gw;
   GELO_CONNECT c;
   GELO_DRAW_FLAGS hifg;
{
   register Integer i,j;
   register Sequence l;
   Integer x1,y1,x2,y2,x3,y3,x4,y4;
   Integer xpt[64],ypt[64];
   register GELO_OBJECT par;
   register Boolean fg;
   Integer lcl;

   ITRACE("GELO_draw_connect 0x%x 0x%x",gw,c);

   if (hifg == GELO_DRAW_NORM && c->arc_object != NULL) GELO_draw(gw,c->arc_object,hifg);

   c->lbllx = c->lblby = c->lblrx = c->lblty = -1;

   fg = connect_loc(gw,c->from,c->from_port,c->from_subport,TRUE,&x1,&y1);
   fg |= connect_loc(gw,c->to,c->to_port,c->to_subport,TRUE,&x2,&y2);
   if (!fg) return;

   ASHline_style(gw->window,c->style);

   if (hifg == GELO_DRAW_HILITE) lcl = ASHcolor(gw->window,ASHinq_background_color(gw->window));

   if (c->pivots == NULL) {
      ASHline(gw->window,x1,y1,x2,y2);
      if (hifg == GELO_DRAW_HILITE) ASHcolor(gw->window,lcl);
      draw_label(gw->window,c,GELO_LABEL_END,x1,y1,x2,y2);
      draw_label(gw->window,c,GELO_LABEL_START,x1,y1,x2,y2);
      draw_label(gw->window,c,GELO_LABEL_MIDDLE,x1,y1,x2,y2);
      x3 = x2;
      y3 = y2;
      x4 = x1;
      y4 = y1;
    }
   else {
      par = c->from->parent;
      while (par != NULL && par->flavor != GELO_FLAVOR_LAYOUT)
	 par = par->parent;
      xpt[0] = x1;
      ypt[0] = y1;
      i = 1;
      for (l = c->pivots; l != NULL; l = CDDR(l)) {
	 x3 = CAR(Integer,l);
	 y3 = CADR(Integer,l);
	 GELO_scale_position(gw,par,x3,y3,&xpt[i],&ypt[i]);
	 ++i;
       };
      xpt[i] = x2;
      ypt[i] = y2;
      ASHpolyline(gw->window,i+1,xpt,ypt);
      if (c->arrow == GELO_ARROW_SINGLE_ALL || c->arrow == GELO_ARROW_MARK_ALL ||
	     c->arrow == GELO_ARROW_DMARK_ALL) {
	 for (j = 1; j < i; ++j) {
	    draw_arrow(gw->window,GELO_ARROW_SINGLE,xpt[j-1],ypt[j-1],xpt[j],ypt[j]);
	  };
       };
      if (hifg == GELO_DRAW_HILITE) ASHcolor(gw->window,lcl);
      draw_label(gw->window,c,GELO_LABEL_END,xpt[i-1],ypt[i-1],xpt[i],ypt[i]);
      draw_label(gw->window,c,GELO_LABEL_START,xpt[0],ypt[0],xpt[1],ypt[1]);
      j = (i-1)/2;
      draw_label(gw->window,c,GELO_LABEL_MIDDLE,xpt[j],ypt[j],xpt[j+1],ypt[j+1]);
      x3 = xpt[1];
      y3 = ypt[1];
      x4 = xpt[i-1];
      y4 = ypt[i-1];
    };

   if (hifg == GELO_DRAW_HILITE) lcl = ASHcolor(gw->window,ASHinq_background_color(gw->window));

   switch (c->arrow) {
      default :
      case GELO_ARROW_NONE :
	 break;
      case GELO_ARROW_SINGLE :
      case GELO_ARROW_SINGLE_ALL :
      case GELO_ARROW_MARK :
      case GELO_ARROW_MARK_ALL :
      case GELO_ARROW_DMARK:
      case GELO_ARROW_DMARK_ALL :
	 draw_arrow(gw->window,c->arrow,x4,y4,x2,y2);
	 break;
      case GELO_ARROW_DOUBLE :
	 draw_arrow(gw->window,GELO_ARROW_SINGLE,x3,y3,x1,y1);
	 draw_arrow(gw->window,GELO_ARROW_SINGLE,x4,y4,x2,y2);
	 break;
    };

   if (hifg == GELO_DRAW_HILITE) ASHcolor(gw->window,lcl);

   ASHline_style(gw->window,ASH_STYLE_SOLID);
};





/************************************************************************/
/*									*/
/*	GELO_hilite_connect -- hilight a connection			*/
/*	hilite_line -- draw a highlighted line				*/
/*	hilite_lines -- draw a highlighted path 			*/
/*									*/
/************************************************************************/


void
GELO_hilite_connect(gw,c,sty)
   GELO_WINDOW gw;
   GELO_CONNECT c;
   Integer sty;
{
   register Integer i;
   register Sequence l;
   Integer x1,y1,x2,y2,x3,y3;
   Integer xpt[16],ypt[16];
   Integer cr,fl;
   register GELO_OBJECT par;
   register Boolean fg;

   ITRACE("GELO_hilite_connect 0x%x 0x%x %d",gw,c,sty);

   fg = connect_loc(gw,c->from,c->from_port,c->from_subport,TRUE,&x1,&y1);
   fg |= connect_loc(gw,c->to,c->to_port,c->to_subport,TRUE,&x2,&y2);
   if (!fg) return;

   if (c->pivots == NULL) {
      hilite_line(gw->window,sty,x1,y1,x2,y2);
    }
   else {
      par = c->from->parent;
      xpt[0] = x1;
      ypt[0] = y1;
      i = 1;
      for (l = c->pivots; l != NULL; l = CDDR(l)) {
	 x3 = CAR(Integer,l);
	 y3 = CADR(Integer,l);
	 GELO_scale_position(gw,par,x3,y3,&xpt[i],&ypt[i]);
	 ++i;
       };
      xpt[i] = x2;
      ypt[i] = y2;
      hilite_lines(gw->window,sty,i+1,xpt,ypt);
    };

   if (c->lbllx > 0) {
      cr = ASHcombination_rule(gw->window,10);
      fl = ASHfill(gw->window,sty);
      ASHrectangle(gw->window,c->lbllx,c->lblby,c->lblrx,c->lblty);
      ASHfill(gw->window,fl);
      ASHcombination_rule(gw->window,cr);
    };
};





static void
hilite_line(w,sty,x0,y0,x1,y1)
   ASH_WINDOW w;
   Integer sty;
   Integer x0,y0,x1,y1;
{
   register Integer dx,dy;
   Integer x[4],y[4];
   Integer cr,fl;

   DTRACE("hilite_line 0x%x %d %d %d %d %d",w,sty,x0,y0,x1,y1);

   dy = (x0 == x1 ? 0 : 2);
   dx = (y0 == y1 ? 0 : 2);

   x[0] = x0+dx;
   y[0] = y0+dy;
   x[1] = x0-dx;
   y[1] = y0-dy;
   x[2] = x1-dx;
   y[2] = y1-dy;
   x[3] = x1+dx;
   y[3] = y1+dy;

   cr = ASHcombination_rule(w,10);
   fl = ASHfill(w,sty);

   ASHpolygon(w,4,x,y);

   ASHcombination_rule(w,cr);
   ASHfill(w,fl);
};






static void
hilite_lines(w,sty,npt,xl,yl)
   ASH_WINDOW w;
   Integer sty;
   Integer npt;
   Integer xl[],yl[];
{
   register Integer i,j;
   register Integer dx,dy;
   Integer x[64],y[64];
   Integer cr,fl;

   DTRACE("hilite_lines 0x%x %d %d",w,sty,npt);

   dx = dy = 0;
   j = 0;
   for (i = 0; i < npt; ++i) {
      if (i != 0) {
	 x[j] = xl[i]+dx;
	 y[j] = yl[i]+dy;
	 ++j;
       };
      if (i+1 < npt) {
	 dy = (xl[i] == xl[i+1] ? 0 : xl[i] < xl[i+1] ? -2 : 2);
	 dx = (yl[i] == yl[i+1] ? 0 : yl[i] < yl[i+1] ? 2 : -2);
       }
      if (i == 0 || xl[i]+dx != x[j-1] || yl[i]+dy != y[j-1]) {
	 x[j] = xl[i]+dx;
	 y[j] = yl[i]+dy;
	 ++j;
       };
    };
   for (i = npt-1; i >= 0; --i) {
      if (i != npt-1) {
	 x[j] = xl[i]+dx;
	 y[j] = yl[i]+dy;
	 ++j;
       };
      if (i > 0) {
	 dy = (xl[i] == xl[i-1] ? 0 : xl[i] > xl[i-1] ? 2 : -2);
	 dx = (yl[i] == yl[i-1] ? 0 : yl[i] > yl[i-1] ? -2 : 2);
       }
      if (i == npt-1 || xl[i]+dx != x[j-1] || yl[i]+dy != y[j-1]) {
	 x[j] = xl[i]+dx;
	 y[j] = yl[i]+dy;
	 ++j;
       };
    };

   cr = ASHcombination_rule(w,10);
   fl = ASHfill(w,sty);

   ASHgeneral_polygon(w,j,x,y);

   ASHcombination_rule(w,cr);
   ASHfill(w,fl);
};






/************************************************************************/
/*									*/
/*	GELO_connect_list_correlate -- correlate list of connections	*/
/*	GELO_connect_correlate -- check if we correlate with one arc	*/
/*	checkline -- check if point is near line			*/
/*									*/
/************************************************************************/


GELO_CONNECT
GELO_connect_list_correlate(gw,cl,x,y)
   GELO_WINDOW gw;
   Sequence cl;
   Integer x,y;
{
   register Sequence l;
   register GELO_CONNECT c;

   ITRACE("GELO_connect_correlate 0x%x 0x%x %d %d",gw,cl,x,y);

   forin (c,GELO_CONNECT,l,cl) {
      if (GELO_connect_correlate(gw,c,x,y)) return c;
    };

   return NULL;
};





Boolean
GELO_connect_correlate(gw,c,x,y)
   GELO_WINDOW gw;
   GELO_CONNECT c;
   Integer x,y;
{
   register Sequence l;
   Integer x1,y1,x2,y2,x3,y3;
   register GELO_OBJECT par;
   register Boolean fg;

   ITRACE("GELO_connect_correlate 0x%x 0x%x %d %d",gw,c,x,y);

   fg = connect_loc(gw,c->from,c->from_port,c->from_subport,TRUE,&x1,&y1);
   fg |= connect_loc(gw,c->to,c->to_port,c->to_subport,TRUE,&x2,&y2);
   if (!fg) return FALSE;

   if (c->pivots == NULL) {
      if (checkline(x,y,x1,y1,x2,y2)) return TRUE;
    }
   else {
      par = c->from->parent;
      for (l = c->pivots; l != NULL; l = CDDR(l)) {
	 x3 = CAR(Integer,l);
	 y3 = CADR(Integer,l);
	 GELO_scale_position(gw,par,x3,y3,&x3,&y3);
	 if (checkline(x,y,x1,y1,x3,y3)) return TRUE;
	 x1 = x3;
	 y1 = y3;
       };
      if (checkline(x,y,x1,y1,x2,y2)) return TRUE;
    };

   if (c->lbllx > 0) {
      if (x >= c->lbllx && x <= c->lblrx &&
	     y >= c->lblty && y <= c->lblby) return TRUE;
    };

   return FALSE;
};




static Boolean
checkline(x,y,x0,y0,x1,y1)
   Integer x,y;
   Integer x0,y0;
   Integer x1,y1;
{
   Integer t,t1;
   Float f;

   DTRACE("checkline %d %d %d %d %d %d",x,y,x0,y0,x1,y1);

   if (x0 == x1) {
      if (y0 > y1) {
	 t = y0;
	 y0 = y1;
	 y1 = t;
       };
      if (abs(x-x0) <= MAX_CORRELATE && (y >= y0 && y <= y1))
	 return TRUE;
    }
   else if (y0 == y1) {
      if (x0 > x1) {
	 t = x0;
	 x0 = x1;
	 x1 = t;
       };
      if (abs(y-y0) <= MAX_CORRELATE && (x >= x0 && x <= x1))
	 return TRUE;
    }
   else if ((x0 < x1 && x >= x0 && x <= x1) ||
	       (x0 > x1 && x >= x1 && x <= x0)) {
      t = (y1-y0)*(x-x0) - (x1-x0)*(y-y0);
      t1 = (y1-y0)*(y1-y0)+(x1-x0)*(x1-x0);
      f = t;
      f = (f*f)/t1;
      t = f;
      if (t < (MAX_CORRELATE+1)*(MAX_CORRELATE+1)) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	GELO_inq_connect_from -- get from node of connect		*/
/*	GELO_inq_connect_to -- get to node of connect			*/
/*	GELO_inq_connect_from_port -- get port of from node		*/
/*	GELO_inq_connect_to_port -- get port of to node 		*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELO_inq_connect_from(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_inq_connect_from 0x%x => 0x%x",c,c->from);

   return c->from;
};





GELO_OBJECT
GELO_inq_connect_to(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_inq_connect_to 0x%x => 0x%x",c,c->to);

   return c->to;
};





GELO_PORT
GELO_inq_connect_from_port(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_inq_connect_from_port 0x%x => 0x%x",c,c->from_port);

   return c->from_port;
};





GELO_PORT
GELO_inq_connect_to_port(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_inq_connect_to_port 0x%x => 0x%x",c,c->to_port);

   return c->to_port;
};





/************************************************************************/
/*									*/
/*	GELO_inq_connect_from_location -- return from location point	*/
/*	GELO_inq_connect_to_location -- return to location point	*/
/*									*/
/************************************************************************/


Boolean
GELO_inq_connect_from_location(gw,o,c,xp,yp)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_CONNECT c;
   Integer *xp, *yp;
{
   GELO_OBJECT p;

   connect_loc(gw,c->from,c->from_port,c->from_subport,FALSE,xp,yp);

   for (p = c->from->parent; p != NULL && p != o; p = p->parent) {
      *xp += p->x_pos;
      *yp += p->y_pos;
    };

   ITRACE("GELO_inq_connect_from_location 0x%x => %d %d",c,*xp,*yp);

   return TRUE;
};





Boolean
GELO_inq_connect_to_location(gw,o,c,xp,yp)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_CONNECT c;
   Integer *xp, *yp;
{
   GELO_OBJECT p;

   connect_loc(gw,c->to,c->to_port,c->to_subport,FALSE,xp,yp);

   for (p = c->to->parent; p != NULL && p != o; p = p->parent) {
      *xp += p->x_pos;
      *yp += p->y_pos;
    };

   ITRACE("GELO_inq_connect_to_location 0x%x => %d %d",c,*xp,*yp);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	GELO_conn_set_from_data -- set from data for connection 	*/
/*	GELO_conn_set_to_data -- set to data for connection		*/
/*	GELO_conn_inq_from_data -- inq from data for connection 	*/
/*	GELO_conn_inq_to_data -- inq to data for connection		*/
/*									*/
/************************************************************************/


void
GELO_conn_set_from_data(c,d)
   GELO_CONNECT c;
   String d;
{
   c->from_data = d;
};




void
GELO_conn_set_to_data(c,d)
   GELO_CONNECT c;
   String d;
{
   c->to_data = d;
};




String
GELO_conn_inq_from_data(c)
   GELO_CONNECT c;
{
   return c->from_data;
};





String
GELO_conn_inq_to_data(c)
   GELO_CONNECT c;
{
   return c->to_data;
};





/************************************************************************/
/*									*/
/*	GELO_conn_set_ports -- set ports on a connection		*/
/*	find_port -- choose a port for a connection			*/
/*									*/
/************************************************************************/


void
GELO_conn_set_ports(c,c1,c2,mthd,cmthd,okpvts)
   GELO_CONNECT c;
   COMPONENT * c1;
   COMPONENT * c2;
   GELO_METHOD mthd;
   GELO_CONN_METHOD cmthd;
   Boolean okpvts;
{
   GELO_PORT port,tp;
   Integer sport;
   Integer x0,y0;
   Integer x1,y1;
   Boolean pvts;
   Sequence l;

   ITRACE("GELO_conn_set_ports 0x%x %d %d %d %d",c,x0,y0,x1,y1);

   pvts = FALSE;
   if (okpvts && (mthd & (GELO_METHOD_PERTBITS|GELO_METHOD_BIT_GIOTTO)) != 0 &&
	  c->pivots != NULL && cmthd != GELO_CONN_METHOD_DIRECT)
      pvts = TRUE;

   c->from_subport = -1;
   c->to_subport = -1;

   if (c->from_any) c->from_port = GELO_PORT_ANY;
   if (c->to_any) c->to_port = GELO_PORT_ANY;

   x0 = c1->pos_x;
   y0 = c1->pos_y;
   if (!pvts) {
      x1 = c2->pos_x;
      y1 = c2->pos_y;
      tp = c->to_port;
    }
   else {
      x1 = CAR(Integer,c->pivots);
      y1 = CADR(Integer,c->pivots);
      tp = GELO_PORT_CENTER;
    };

   if (c->from_port == GELO_PORT_USER) {
      if (port_rtn != NULL && (*port_rtn)(c->from,c->to,c->arc_object,FALSE,&port,&sport)) {
	 c->from_port = port;
	 c->from_subport = sport;
       }
      else c->from_port = GELO_PORT_ANY;
    };
   if (c->from_port == GELO_PORT_ANY) {
      c->from_port = find_port(x1-x0,y1-y0,FALSE,tp,mthd);
    };

   x1 = c2->pos_x;
   y1 = c2->pos_y;
   if (!pvts) {
      x0 = c1->pos_x;
      y0 = c1->pos_y;
      tp = c->from_port;
    }
   else {
      for (l = c->pivots; l != NULL; l = CDDR(l)) {
	 x0 = CAR(Integer,l);
	 y0 = CADR(Integer,l);
       };
      tp = GELO_PORT_CENTER;
    };

   if (c->to_port == GELO_PORT_USER) {
      if (port_rtn != NULL && (*port_rtn)(c->from,c->to,c->arc_object,TRUE,&port,&sport)) {
	 c->to_port = port;
	 c->to_subport = sport;
       }
      else c->to_port = GELO_PORT_ANY;
    };
   if (c->to_port == GELO_PORT_ANY)
      c->to_port = find_port(x0-x1,y0-y1,TRUE,tp,mthd);
};





static GELO_PORT
find_port(dx1,dy1,tfg,port,mthd)
   Integer dx1,dy1;
   Boolean tfg;
   GELO_PORT port;
   GELO_METHOD mthd;
{
   register Integer dx,dy;
   register GELO_PORT p;
   Boolean top,bot,lft,rgt;

   DTRACE("find_port %d %d %d %d",dx1,dy1,tfg,port);

   if (mthd & GELO_METHOD_BIT_GIOTTO) {
      if (dx1 != 0 && abs(dy1/dx1) >= 3) dx1 = 0;
      if (dy1 != 0 && abs(dx1/dy1) >= 3) dy1 = 0;
    };

   dx = (dx1 < 0 ? 0 : dx1 == 0 ? 1 : 2);
   dy = (dy1 < 0 ? 0 : dy1 == 0 ? 1 : 2);

   if (mthd & GELO_METHOD_BIT_GIOTTO) {
      dx1 = dx-1;
      dy1 = dy-1;
    };

   switch (port) {
      case GELO_PORT_LEFT :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_LEFT_SEQ :
	 lft = TRUE;
	 rgt = FALSE;
	 break;
      case GELO_PORT_RIGHT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_RIGHT_SEQ :
	 rgt = TRUE;
	 lft = FALSE;
	 break;
      default :
	 rgt = FALSE;
	 lft = FALSE;
	 break;
    };

   switch (port) {
      case GELO_PORT_TOP :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_TOP_SEQ :
	 top = TRUE;
	 bot = FALSE;
	 break;
      case GELO_PORT_BOTTOM :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_BOTTOM_SEQ :
	 bot = TRUE;
	 top = FALSE;
	 break;
      default :
	 top = FALSE;
	 bot = FALSE;
	 break;
    };

   switch (dy*3+dx) {
      case 0 :
	 p = GELO_PORT_TOP_LEFT;
	 break;
      case 1 :
	 if (lft) p = GELO_PORT_TOP_LEFT;
	 else if (rgt) p = GELO_PORT_TOP_RIGHT;
	 else if (abs(dy1) > 1) p = (tfg ? GELO_PORT_TOP_RIGHT : GELO_PORT_TOP_LEFT);
	 else p = GELO_PORT_TOP;
	 break;
      case 2 :
	 p = GELO_PORT_TOP_RIGHT;
	 break;
      case 3 :
	 if (top) p = GELO_PORT_TOP_LEFT;
	 else if (bot || abs(dx1) > 1) p = GELO_PORT_BOTTOM_LEFT;
	 else p = GELO_PORT_LEFT;
	 break;
      case 4 :
	 p = (tfg ? GELO_PORT_BOTTOM_LEFT : GELO_PORT_BOTTOM_RIGHT);
	 break;
      case 5 :
	 if (top) p = GELO_PORT_TOP_RIGHT;
	 else if (bot || abs(dx1) > 1) p = GELO_PORT_BOTTOM_RIGHT;
	 else p = GELO_PORT_RIGHT;
	 break;
      case 6 :
	 p = GELO_PORT_BOTTOM_LEFT;
	 break;
      case 7 :
	 if (lft) p = GELO_PORT_BOTTOM_LEFT;
	 else if (rgt) p = GELO_PORT_BOTTOM_RIGHT;
	 else if (abs(dy1) > 1) p = (tfg ? GELO_PORT_BOTTOM_LEFT : GELO_PORT_BOTTOM_RIGHT);
	 else p = GELO_PORT_BOTTOM;
	 break;
      case 8 :
	 p = GELO_PORT_BOTTOM_RIGHT;
	 break;
    };

   return p;
};





/************************************************************************/
/*									*/
/*	GELO_conn_inq_from_subport -- return from subport		*/
/*	GELO_conn_set_from_subport -- set from subport of connection	*/
/*	GELO_conn_inq_to_subport -- return to subport			*/
/*	GELO_conn_set_to_subport -- set to subport of connection	*/
/*									*/
/************************************************************************/


Integer
GELO_conn_inq_from_subport(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_conn_inq_from_subport 0x%x",c);

   return c->from_subport;
};





void
GELO_conn_set_from_subport(c,n,mx)
   GELO_CONNECT c;
   Integer n;
   Integer mx;
{
   ITRACE("GELO_conn_set_from_subport 0x%x %d %d",c,n,mx);

   n = real_subport(n,mx);

   c->from_subport = n;
};





Integer
GELO_conn_inq_to_subport(c)
   GELO_CONNECT c;
{
   ITRACE("GELO_conn_inq_to_subport 0x%x",c);

   return c->to_subport;
};





void
GELO_conn_set_to_subport(c,n,mx)
   GELO_CONNECT c;
   Integer n;
   Integer mx;
{
   ITRACE("GELO_conn_set_to_subport 0x%x %d %d",c,n,mx);

   n = real_subport(n,mx);

   c->to_subport = n;
};





static Integer
real_subport(n,mx)
   Integer n,mx;
{
   register Integer x,y;

   DTRACE("real_subport %d %d",n,mx);

   x = MAX_SEQ_PORTS/2;
   y = 1;

   while (mx < x) {
      x = x/2;
      y = y*2;
    };

   return (n+1)*y-1;
};




/************************************************************************/
/*									*/
/*	GELO_free_connects -- free connections				*/
/*									*/
/************************************************************************/


void
GELO_free_connects(gw,cl,keep)
   GELO_WINDOW gw;
   Sequence cl;
   GELO_OBJECT keep;
{
   register Sequence l;
   register GELO_CONNECT c;

   ITRACE("GELO_free_connects 0x%x 0x%x 0x%x",gw,cl,keep);

   forin (c,GELO_CONNECT,l,cl) {
      if (c->arc_object != NULL) GELO_object_free(gw,c->arc_object,keep);
      c->arc_object = NULL;
      if (c->pivots != NULL) LFREE(c->pivots);
      c->pivots = NULL;
      free(c);
    };
};





/************************************************************************/
/*									*/
/*	GELO_portal_location -- find PORT location for portal		*/
/*									*/
/************************************************************************/


GELO_PORT
GELO_portal_location(obj,port)
   GELO_OBJECT obj;
   GELO_PORT port;
{
   GELO_PORT nport;
   Integer sport;

   ITRACE("GELO_portal_location 0x%x %d",obj,port);

   if (port == GELO_PORT_USER) {
      if (port_rtn != NULL && (*port_rtn)(obj,NULL,obj,FALSE,&nport,&sport)) {
	 port = nport;
       }
      else port = GELO_PORT_ANY;
    };

   return port;
};





/************************************************************************/
/*									*/
/*	GELO_connect_replace -- replace component in list of connects	*/
/*									*/
/************************************************************************/


void
GELO_connect_replace(cl,old,new)
   Sequence cl;
   GELO_OBJECT old,new;
{
   register GELO_CONNECT c;
   register Sequence l;

   ITRACE("GELO_connect_replace 0x%x 0x%x 0x%x",cl,old,new);

   forin (c,GELO_CONNECT,l,cl) {
      if (c->from == old) c->from = new;
      if (c->to == old) c->to = new;
    };
};





/************************************************************************/
/*									*/
/*	GELO_connect_elim -- eliminate connections using component	*/
/*									*/
/************************************************************************/


Sequence
GELO_connect_elim(gw,cl,old)
   GELO_WINDOW gw;
   Sequence cl;
   GELO_OBJECT old;
{
   register GELO_CONNECT c;
   register Sequence l;
   Sequence nl;

   ITRACE("GELO_connect_elim 0x%x 0x%x 0x%x",gw,cl,old);

   nl = NULL;
   forin (c,GELO_CONNECT,l,cl) {
      if (c->from == old || c->to == old) nl = CONS(c,nl);
    };

   if (nl != NULL) {
      forin (c,GELO_CONNECT,l,nl) {
	 cl = REMOB(c,cl);
	 c->arc_object = NULL;
	 if (c->pivots != NULL) LFREE(c->pivots);
	 c->pivots = NULL;
	 free(c);
       };
    };

   return cl;
};





/************************************************************************/
/*									*/
/*	GELO_connect_compare -- compare two connections 		*/
/*									*/
/************************************************************************/


Boolean
GELO_connect_compare(c1,c2)
   GELO_CONNECT c1;
   GELO_CONNECT c2;
{
   register Boolean fg;

   ITRACE("GELO_connect_compare 0x%x 0x%x",c1,c2);

   if (c1->from_port != c2->from_port &&
	  c1->from_port != GELO_PORT_ANY &&
	  c1->from_port != GELO_PORT_USER &&
	  c2->from_port != GELO_PORT_ANY &&
	  c2->from_port != GELO_PORT_USER) {
      fg = FALSE;
    }
   else if (c1->to_port != c2->to_port &&
	  c1->to_port != GELO_PORT_ANY &&
	  c1->to_port != GELO_PORT_USER &&
	  c2->to_port != GELO_PORT_ANY &&
	  c2->to_port != GELO_PORT_USER) {
      fg = FALSE;
    }
   else if (c1->from_subport != c2->from_subport &&
	  c1->from_port != GELO_PORT_ANY &&
	  c1->from_port != GELO_PORT_USER &&
	  c2->from_port != GELO_PORT_ANY &&
	  c2->from_port != GELO_PORT_USER) {
      fg = FALSE;
    }
   else if (c1->to_subport != c2->to_subport &&
	  c1->to_port != GELO_PORT_ANY &&
	  c1->to_port != GELO_PORT_USER &&
	  c2->to_port != GELO_PORT_ANY &&
	  c2->to_port != GELO_PORT_USER) {
      fg = FALSE;
    }
   else if (c1->style != c2->style || c1->arrow != c2->arrow)
      fg = FALSE;
   else if (c1->start_label_pos != c2->start_label_pos ||
	       c1->start_label_box != c2->start_label_box)
      fg = FALSE;
   else if (c1->start_label != c2->start_label &&
	       (c1->start_label == NULL || c2->start_label == NULL ||
		   STRNEQ(c1->start_label,c2->start_label)))
      fg = FALSE;
   else if (c1->mid_label_pos != c2->mid_label_pos ||
	       c1->mid_label_box != c2->mid_label_box)
      fg = FALSE;
   else if (c1->mid_label != c2->mid_label &&
	       (c1->mid_label == NULL || c2->mid_label == NULL ||
		   STRNEQ(c1->mid_label,c2->mid_label)))
      fg = FALSE;
   else if (c1->end_label_pos != c2->end_label_pos ||
	       c1->end_label_box != c2->end_label_box)
      fg = FALSE;
   else if (c1->end_label != c2->end_label &&
	       (c1->end_label == NULL || c2->end_label == NULL ||
		   STRNEQ(c1->end_label,c2->end_label)))
      fg = FALSE;
   else fg = TRUE;

   return fg;
};





/************************************************************************/
/*									*/
/*	connect_loc -- get location on box for connection		*/
/*									*/
/************************************************************************/


static Boolean
connect_loc(gw,obj,port,subport,absfg,xp,yp)
   GELO_WINDOW gw;
   GELO_OBJECT obj;
   GELO_PORT port;
   Integer subport;
   Boolean absfg;
   Integer *xp, *yp;
{
   register Integer x,y;
   register Boolean fg;
   Integer lx,by,rx,ty;
   Integer dx,dy;

   DTRACE("connect_loc 0x%x 0x%x %d %d %d",gw,obj,port,subport,absfg);

   fg = TRUE;

   if (subport < 0) subport = DEFAULT_SUBPORT;

   if (absfg) {
      if (!GELO_inq_position(gw,obj,&lx,&by,&rx,&ty)) fg = FALSE;
    }
   else {
      lx = GELO_inq_x_position(obj);
      ty = GELO_inq_y_position(obj);
      rx = lx+GELO_inq_x_size(obj)-1;
      by = ty+GELO_inq_y_size(obj)-1;
    };

   switch (port) {
      case GELO_PORT_LEFT :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_LEFT_SEQ :
	 x = lx;
	 break;
      case GELO_PORT_RIGHT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_RIGHT_SEQ :
	 x = rx;
	 break;
      case GELO_PORT_TOP :
      case GELO_PORT_BOTTOM :
      case GELO_PORT_CENTER :
	 x = (lx+rx)/2;
	 break;
      case GELO_PORT_TOP_SEQ :
      case GELO_PORT_BOTTOM_SEQ :
	 x = lx + ((rx-lx)*(subport+1))/(MAX_SEQ_PORTS+1);
	 break;
    };

   switch (port) {
      case GELO_PORT_TOP :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_TOP_SEQ :
	 y = ty;
	 break;
      case GELO_PORT_BOTTOM :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_BOTTOM_SEQ :
	 y = by;
	 break;
      case GELO_PORT_LEFT :
      case GELO_PORT_RIGHT :
      case GELO_PORT_CENTER :
	 y = (by+ty)/2;
	 break;
      case GELO_PORT_LEFT_SEQ :
      case GELO_PORT_RIGHT_SEQ :
	 y = ty + ((by-ty)*(subport+1))/(MAX_SEQ_PORTS+1);
	 break;
    };

   GELO_inq_port_offset(obj,port,lx,by,rx,ty,&dx,&dy);
   x += dx;
   y += dy;

   *xp = x;
   *yp = y;

   return fg;
};





/************************************************************************/
/*									*/
/*	draw_arrow -- draw an arrow head at the tail of a line		*/
/*									*/
/************************************************************************/


static void
draw_arrow(w,ast,x0,y0,x1,y1)
   ASH_WINDOW w;
   GELO_ARC_ARROW ast;
   Integer x0,y0,x1,y1;
{
   float x,y,dx,dy,z;
   int x2,y2,i;
   int x3,y3;
   ASH_LINE_STYLE sty;

   DTRACE("draw_arrow 0x%x %d %d %d %d",w,x0,y0,x1,y1);

   z = (y1-y0)*(y1-y0)+(x1-x0)*(x1-x0);
   if (z == 0) return;
   x = z/2.0;
   for (i = 0; x != 0 && i < 16; ++i) x = (x+z/x)/2.0;
   z = x;
   if (z == 0) return;

   x = (x0-x1)*ARROW_SIZE/z+x1;
   y = (y0-y1)*ARROW_SIZE/z+y1;
   dx = (y1-y0)*ARROW_SIZE/z/2.0;
   dy = (x0-x1)*ARROW_SIZE/z/2.0;

   x3 = (dx < 0 ? dx - 0.49 : dx + 0.49);
   y3 = (dy < 0 ? dy - 0.49 : dy + 0.49);

   sty = ASHinq_line_style(w);
   if ((sty % 3) != 0) ASHline_style(w,sty-(sty%3));

   x2 = x+x3;
   y2 = y+y3;
   ASHline(w,x2,y2,x1,y1);
   DTRACE("\t%d %d %d %d",x2,y2,x3,y3);

   x2 = x-x3;
   y2 = y-y3;
   ASHline(w,x2,y2,x1,y1);
   DTRACE("\t%d %d %d %d",x1,y1,x2,y2);

   if (ast == GELO_ARROW_MARK || ast == GELO_ARROW_MARK_ALL || ast == GELO_ARROW_DMARK ||
	  ast == GELO_ARROW_DMARK_ALL) {
      x3 *= 2;
      y3 *= 2;
      x = (x0-x1)*1.5*ARROW_SIZE/z+x1;
      y = (y0-y1)*1.5*ARROW_SIZE/z+y1;
      ASHline(w,((int) x+x3),((int) y+y3),((int) x-x3),((int) y-y3));
    };

   if (ast == GELO_ARROW_DMARK || ast == GELO_ARROW_DMARK_ALL) {
      x = (x0-x1)*2*ARROW_SIZE/z+x1;
      y = (y0-y1)*2*ARROW_SIZE/z+y1;
      ASHline(w,((int) x+x3),((int) y+y3),((int) x-x3),((int) y-y3));
    };

   ASHline_style(w,sty);
};





/************************************************************************/
/*									*/
/*	draw_label -- draw label on an arc				*/
/*									*/
/************************************************************************/


static void
draw_label(w,c,loc,x0,y0,x1,y1)
   ASH_WINDOW w;
   GELO_CONNECT c;
   GELO_ARC_LABEL_LOC loc;
   Integer x0,y0;
   Integer x1,y1;
{
   Integer x2,y2,x3,y3;
   Integer xsz,ysz;
   Integer ft;
   String lbl;
   GELO_ARC_LABEL_POS pos;
   Boolean box;

   DTRACE("draw_label 0x%x 0x%x %d %d %d %d",w,c,x0,y0,x1,y1);

   switch (loc) {
      case GELO_LABEL_START :
	 lbl = c->start_label;
	 box = c->start_label_box;
	 pos = c->start_label_pos;
	 break;
      case GELO_LABEL_MIDDLE :
	 lbl = c->mid_label;
	 box = c->mid_label_box;
	 pos = c->mid_label_pos;
	 break;
      case GELO_LABEL_END :
	 lbl = c->end_label;
	 box = c->end_label_box;
	 pos = c->end_label_pos;
	 break;
      default :
	 return;
    };

   if (pos == GELO_LABEL_NONE || lbl == NULL || *lbl == 0) return;

   switch (loc) {
      default :
	 return;
      case GELO_LABEL_START :
	 source_label(FALSE,lbl,pos,x0,y0,x1,y1,&x2,&y2);
	 break;
      case GELO_LABEL_END :
	 source_label(TRUE,lbl,pos,x0,y0,x1,y1,&x2,&y2);
	 break;
      case GELO_LABEL_MIDDLE :
	 middle_label(lbl,pos,x0,y0,x1,y1,&x2,&y2);
	 break;
    };

   ASHinq_text_extent(label_font,lbl,&xsz,&ysz);

   x3 = x2+xsz;
   y3 = y2-ysz;

   ASHclear_box(w,x2,y2,x3,y3);
   ASHinq_text_offset(label_font,lbl,&xsz,&ysz);
   ft = ASHfont(w,label_font);
   ASHtext(w,x2+xsz,y2-ysz,lbl);
   ASHfont(w,ft);

   if (box) ASHbox(w,x2,y2,x3,y3);

   c->lbllx = x2;
   c->lblby = y2;
   c->lblrx = x3;
   c->lblty = y3;
};





/************************************************************************/
/*									*/
/*	middle_label -- get position for label in middle		*/
/*									*/
/************************************************************************/


static void
middle_label(lbl,pos,x0,y0,x1,y1,lxp,byp)
   String lbl;
   GELO_ARC_LABEL_POS pos;
   Integer x0,y0;
   Integer x1,y1;
   Integer *lxp,*byp;
{
   register Integer x2,y2;
   register Integer dx,dy;
   Integer xsz,ysz;

   DTRACE("middle_label %s %d %d %d %d %d",lbl,pos,x0,y0,x1,y1);

   x2 = (x0+x1)/2;
   y2 = (y0+y1)/2;

   ASHinq_text_extent(label_font,lbl,&xsz,&ysz);

   dx = (x0 < x1 ? 1 : x1 < x0 ? -1 : 0);
   dy = (y0 < y1 ? 1 : y1 < y0 ? -1 : 0);

   if (pos == GELO_LABEL_CENTER) {
      x2 = x2-xsz/2;
      y2 = y2+ysz/2;
    }
   else {
      if (pos == GELO_LABEL_RIGHT) {
	 dx = -dx;
	 dy = -dy;
       };
      if (dx == 0) {
	 x2 = (dy > 0 ? x2-xsz-2 : x2+2);
	 y2 = y2+ysz/2;
       }
      else if (dy == 0) {
	 x2 = x2-xsz/2;
	 y2 = (dx > 0 ? y2-2 : y2+ysz+2);
       }
      else if (dx != dy) {
	 if (dx > 0) x2 = x2-xsz-2;
	 else {
	    x2 = x2+2;
	    y2 = y2+ysz;
	  };
       }
      else {
	 if (dx < 0) {
	    x2 = x2-xsz-2;
	    y2 = y2+ysz;
	  }
	 else {
	    x2 = x2+2;
	  };
       };
    };

   *lxp = x2;
   *byp = y2;
};





/************************************************************************/
/*									*/
/*	source_label -- get position for label at source		*/
/*									*/
/************************************************************************/


static void
source_label(flip,lbl,pos,x0,y0,x1,y1,lxp,byp)
   Boolean flip;
   String lbl;
   GELO_ARC_LABEL_POS pos;
   Integer x0,y0;
   Integer x1,y1;
   Integer *lxp,*byp;
{
   register Integer x2,y2;
   register Integer dx,dy;
   Integer xsz,ysz;

   DTRACE("source_label %d %s %d %d %d %d %d",flip,lbl,pos,x0,y0,x1,y1);

   if (flip) {
      x2 = x0;
      x0 = x1;
      x1 = x2;
      y2 = y0;
      y0 = y1;
      y1 = y2;
      if (pos == GELO_LABEL_LEFT) pos = GELO_LABEL_RIGHT;
      else if (pos == GELO_LABEL_RIGHT) pos = GELO_LABEL_LEFT;
    };

   x2 = x0;
   y2 = y0;

   ASHinq_text_extent(label_font,lbl,&xsz,&ysz);

   if (abs(x0-x1) < EPSILON) x1 = x0;
   if (abs(y0-y1) < EPSILON) y1 = y0;

   dx = (x0 < x1 ? 1 : x1 < x0 ? -1 : 0);
   dy = (y0 < y1 ? 1 : y1 < y0 ? -1 : 0);

   if (pos == GELO_LABEL_CENTER) {
      if (dx == 0) {
	 x2 = x2-xsz/2;
	 if (dy == 1) y2 = y2+ysz+2;
	 else y2 = y2-2;
       }
      else if (dy == 0) {
	 y2 = y2+ysz/2;
	 if (dx == 1) x2 = x2+2;
	 else x2 = x2-xsz-2;
       }
      else {
	 x2 = x2-xsz/2;
	 y2 = y2+2*dy;
       };
    }
   else if (pos == GELO_LABEL_LEFT) {
      if (dx == 0) {
	 if (dy == 1) {
	    x2 = x2+2;
	    y2 = y2+ysz+2;
	  }
	 else {
	    x2 = x2-xsz-2;
	    y2 = y2-2;
	  };
       }
      else if (dy == 0) {
	 if (dx == 1) {
	    x2 = x2+2;
	    y2 = y2-2;
	  }
	 else {
	    x2 = x2-xsz-2;
	    y2 = y2+ysz+2;
	  };
       }
      else if (dy == 1) {
	 x2 = x2+2;
	 y2 = y2+ysz+2;
       }
      else {
	 x2 = x2-xsz-2;
	 y2 = y2-2;
       };
    }
   else {
      if (dx == 0) {
	 if (dy == 1) {
	    x2 = x2-xsz-2;
	    y2 = y2+ysz+2;
	  }
	 else {
	    x2 = x2+2;
	    y2 = y2-2;
	  };
       }
      else if (dy == 0) {
	 if (dx == 1) {
	    x2 = x2+2;
	    y2 = y2+ysz+2;
	  }
	 else {
	    x2 = x2-xsz-2;
	    y2 = y2-2;
	  };
       }
      else if (dy == 1) {
	 x2 = x2-xsz-2;
	 y2 = y2+ysz+2;
       }
      else {
	 x2 = x2+2;
	 y2 = y2-2;
       };
    };

   *lxp = x2;
   *byp = y2;
};





/* end of geloconn.c */


