/************************************************************************/
/*                                                                      */
/*              spline.c                                                */
/*                                                                      */
/*      This is an example of a BWE program that uses several of the    */
/*      basic interaction facilities.  Its function is to provide a     */
/*      package whereby the user can draw and edit splines.             */
/*                                                                      */
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include <DATATYPES.h>
#include <stdio.h>
#include <math.h>

#include <ash.h>
#include <stem.h>
#include <leaf.h>
#include <wind.h>
#include <rip.h>





/************************************************************************/
/*                                                                      */
/*      Parameters                                                      */
/*                                                                      */
/************************************************************************/


#define INIT_POINTS             16

#define POINT_COLOR             "red"
#define TEXT_COLOR              "black"
#define SPLINE_COLOR            "blue"
#define FILL_COLOR              "lightblue"
#define BAND_COLOR              "cyan"




/************************************************************************/
/*                                                                      */
/*      Type Definitions                                                */
/*                                                                      */
/************************************************************************/


typedef struct _POINT {         /* A point in the spline                */
   Float x,y;                   /*    Relative coordinates [0,1]        */
   Integer mapx,mapy;           /*    Coordinates in the window         */
} POINT;



typedef enum _STYLE {           /* Spline drawing style                 */
   STYLE_OPEN,                  /*    Open spline                       */
   STYLE_CLOSED,                /*    Closed spline                     */
   STYLE_FILLED,                /*    Closed filled spline              */
} STYLE;





/************************************************************************/
/*                                                                      */
/*      Forward Definitions                                             */
/*                                                                      */
/************************************************************************/


static  void            setup_windows();

static  int             handle_clear_btn();
static  int             handle_quit_btn();
static  int             handle_style_btn();
static  void            select_style();

static  void            spline_refresh();

static  int             handle_draw_hit();
static  int             handle_point_hit();

static  Boolean         rubber_band();
static  int             spline_track();
static  void            draw_spline();

static  Integer         add_point();
static  Integer         duplicate_point();
static  void            remove_point();




/************************************************************************/
/*                                                                      */
/*      Local Storage                                                   */
/*                                                                      */
/************************************************************************/


static  Integer         num_points;     /* current # points             */
static  Integer         max_points;     /* size of points array         */
static  POINT *         point;          /* current set of points        */
static  Integer *       xpoint;         /* x array for spline           */
static  Integer *       ypoint;         /* y array for spline           */


static  STYLE           style;          /* current drawing style        */

static  ASH_COLOR       point_color;    /* color for drawing points     */
static  ASH_COLOR       text_color;     /* color for point text         */
static  ASH_COLOR       spline_color;   /* color for drawing spline     */
static  ASH_COLOR       fill_color;     /* color for filling spline     */
static  ASH_COLOR       band_color;     /* color for rubber banding     */

static  ASH_WINDOW      draw_win;       /* window for drawing spline    */
static  ASH_WINDOW      menu_win;       /* pdm menu window              */

static  Integer         band_pt;        /* point being rubber banded    */
static  Boolean         band_status;    /* status of rubber banding     */




/************************************************************************/
/*                                                                      */
/*      Tables:  Window layout                                          */
/*                                                                      */
/************************************************************************/


#define MENU_WIN        1
#define DRAW_WIN        2
#define L_MOUSE         3
#define M_MOUSE         4
#define R_MOUSE         5
#define STATUS_WIN      6



static LEAF_DATA spline_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DRAW_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_LOWER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT_LINE(L_MOUSE),
	     LEAF_COORD_RX, LEAF_COORD_NEXT_LINE(MENU_WIN) },
	NULL, NULL },
   { L_MOUSE, LEAF_TYPE_TEXT_CENTERED,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(STATUS_WIN),
	     LEAF_COORD_REL_LINE(33), LEAF_COORD_TEXT },
	NULL, (int) "Add Point" },
   { M_MOUSE, LEAF_TYPE_TEXT_CENTERED,
	{ LEAF_COORD_NEXT(L_MOUSE), LEAF_COORD_SAME(L_MOUSE),
	     LEAF_COORD_REL_LINE(67), LEAF_COORD_SAME(L_MOUSE) },
	NULL, (int) "Move Point" },
   { R_MOUSE, LEAF_TYPE_TEXT_CENTERED,
	{ LEAF_COORD_NEXT(M_MOUSE), LEAF_COORD_SAME(L_MOUSE),
	     LEAF_COORD_RX, LEAF_COORD_SAME(L_MOUSE) },
	NULL, (int) "Delete Point" },
   { STATUS_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY, LEAF_COORD_RX, LEAF_COORD_TEXT_LINE },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*                                                                      */
/*      Tables:  Pull down menus                                        */
/*                                                                      */
/************************************************************************/


static STEM_PDM_DATA spline_pdm[] = {
   { STEM_PSTATE_MENU,  "System",       NULL },
      { STEM_PSTATE_BTN,        "Clear",        handle_clear_btn },
      { STEM_PSTATE_BTN,        "Quit",         handle_quit_btn },
   { STEM_PSTATE_MENU,  "Style",        NULL },
      { STEM_PSTATE_BTN,        "Open",         handle_style_btn },
      { STEM_PSTATE_BTN,        "Closed",       handle_style_btn },
      { STEM_PSTATE_BTN,        "Filled",       handle_style_btn },
   { STEM_PSTATE_END }
};





/************************************************************************/
/*                                                                      */
/*      Tables:  Icons                                                  */
/*                                                                      */
/************************************************************************/


static  Integer spline_pt_icon[] = {
   0x80100000,                          /* X          X */
   0x40200000,                          /*  X        X  */
   0x20400000,                          /*   X      X   */
   0x10800000,                          /*    X    X    */
   0x0f000000,                          /*     XXXX     */
   0x0f000000,                          /*     XXXX     */
   0x0f000000,                          /*     XXXX     */
   0x0f000000,                          /*     XXXX     */
   0x10800000,                          /*    X    X    */
   0x20400000,                          /*   X      X   */
   0x40200000,                          /*  X        X  */
   0x80100000,                          /* X          X */
};

#define SPLINE_PT_DEF ASHicon_define("SPLINE_PT",12,12,spline_pt_icon)
#define SPLINE_PT_OFF 6





/************************************************************************/
/*                                                                      */
/*      main -- main program                                            */
/*                                                                      */
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   ASH_WINDOW w;

   argc = ASHset_application(argc,argv);

   w = ASHinit(ASH_MODE_INQUIRE);

   SPLINE_PT_DEF;
   point_color = ASHlookup_color(w,POINT_COLOR);
   text_color = ASHlookup_color(w,TEXT_COLOR);
   spline_color = ASHlookup_color(w,SPLINE_COLOR);
   fill_color = ASHlookup_color(w,FILL_COLOR);
   band_color = ASHlookup_color(w,BAND_COLOR);

   style = STYLE_OPEN;
   max_points = INIT_POINTS;
   point = (POINT *) calloc(max_points,sizeof(POINT));
   xpoint = (Integer *) calloc(max_points,sizeof(Integer));
   ypoint = (Integer *) calloc(max_points,sizeof(Integer));
   num_points = 0;

   setup_windows(w);

   while (1) {
      RIPuser_pick(NULL);
    };
};





/************************************************************************/
/*                                                                      */
/*      setup_windows -- setup windows                                  */
/*                                                                      */
/************************************************************************/


static void
setup_windows(top)
   ASH_WINDOW top;
{
   ASH_WINDOW w;

   LEAFsetup_window(top,spline_leaf,NULL);
   LEAFredraw(top);

   draw_win = LEAFinq_window(top,DRAW_WIN,0);
   if (draw_win == NULL) {
      printf("Window too small to run spline program\n");
      exit(1);
    };

   menu_win = LEAFinq_window(top,MENU_WIN,0);
   STEMpdm_define(menu_win,NULL,spline_pdm);
   select_style();
   ASHvisible(menu_win,TRUE);

   w = LEAFinq_window(top,STATUS_WIN,0);
   RIParea(RIP_AREA_PROMPT,w);
   RIParea(RIP_AREA_STATUS,w);
   ASHvisible(w,TRUE);

   ASHset_refresh(draw_win,spline_refresh);
   ASHvisible(draw_win,TRUE);
};





/************************************************************************/
/*                                                                      */
/*      handle_clear_btn -- clear current spline                        */
/*      handle_quit_btn -- exit                                         */
/*      handle_style_btn -- change spline style                         */
/*      select_style -- change current selection in pdm                 */
/*                                                                      */
/************************************************************************/


static int
handle_clear_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   num_points = 0;

   spline_refresh(draw_win);

   return TRUE;
};





static int
handle_quit_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   exit(0);
};





static int
handle_style_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   if (STREQL(btn,"Open")) style = STYLE_OPEN;
   else if (STREQL(btn,"Closed")) style = STYLE_CLOSED;
   else if (STREQL(btn,"Filled")) style = STYLE_FILLED;
   else return FALSE;

   select_style();

   spline_refresh(draw_win);

   return TRUE;
};




static void
select_style()
{
   String b;

   if (style == STYLE_OPEN) b = "Open";
   else if (style == STYLE_CLOSED) b = "Closed";
   else if (style == STYLE_FILLED) b = "Filled";

   STEMpdm_btn_select(menu_win,"Style","Open",FALSE);
   STEMpdm_btn_select(menu_win,"Style","Closed",FALSE);
   STEMpdm_btn_select(menu_win,"Style","Filled",FALSE);

   STEMpdm_btn_select(menu_win,"Style",b,TRUE);
};






/************************************************************************/
/*                                                                      */
/*      spline_refresh -- redraw current spline                         */
/*                                                                      */
/************************************************************************/



static void
spline_refresh(w)
   ASH_WINDOW w;
{
   Integer lx,by,rx,ty;
   Integer xs,ys,xo,yo;
   Character buf[64];
   Integer i;
   RIP_REGION rgn;
   ASH_COLOR cl;

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   ASHclear_box(w,lx,by,rx,ty);

   RIPremove_window_regions(w);
   RIPdefine_region(w,0,0,0,0,RIP_NO_CHARS,RIP_BTN_ANY_EITHER,
		       handle_draw_hit,ASH_SENSE_NO_CHANGE);

   for (i = 0; i < num_points; ++i) {           /* map the points       */
      point[i].mapx = lx + (rx-lx)*point[i].x;
      point[i].mapy = by + (ty-by)*point[i].y;
      xpoint[i] = point[i].mapx;
      ypoint[i] = point[i].mapy;
    };

   cl = ASHcolor(draw_win,spline_color);

   if (num_points > 2) {
      switch (style) {
	 case STYLE_OPEN :
	    ASHspline(draw_win,num_points,xpoint,ypoint);
	    break;
	 case STYLE_CLOSED :
	    ASHspline_closed(draw_win,num_points,xpoint,ypoint);
	    break;
	 case STYLE_FILLED :
	    cl = ASHcolor(draw_win,fill_color);
	    ASHspline_filled(draw_win,num_points,xpoint,ypoint);
	    cl = ASHcolor(draw_win,spline_color);
	    ASHspline_closed(draw_win,num_points,xpoint,ypoint);
	    break;
       };
    };

   ASHcolor(draw_win,cl);

   for (i = 0; i < num_points; ++i) {
      if (i != 0 && point[i].mapx == point[i-1].mapx &&
	     point[i].mapy == point[i-1].mapy) continue;

      cl = ASHtext_color(draw_win,point_color);
      ASHicon_draw(draw_win,point[i].mapx-SPLINE_PT_OFF,
		      point[i].mapy-SPLINE_PT_OFF,"SPLINE_PT");
      sprintf(buf,"%d",i);
      ASHinq_text_extent(draw_win,buf,&xs,&ys);
      ASHinq_text_offset(draw_win,buf,&xo,&yo);
      ASHtext_color(draw_win,text_color);
      ASHtext_background_color(draw_win,-1);
      ASHtext(draw_win,point[i].mapx - xs/2 + xo, point[i].mapy - ys/2 + yo, buf);
      if (xs < 2*SPLINE_PT_OFF) xs = 2*SPLINE_PT_OFF;
      if (ys < 2*SPLINE_PT_OFF) ys = 2*SPLINE_PT_OFF;
      rgn = RIPdefine_region(draw_win,
				point[i].mapx-xs/2,point[i].mapy-ys/2,
				point[i].mapx+xs/2,point[i].mapy+ys/2,
				RIP_NO_CHARS,RIP_BTN_ANY_EITHER,
				handle_point_hit,ASH_SENSE_BOX);
      RIPset_data(rgn,i);

    };
};





/************************************************************************/
/*                                                                      */
/*      handle_draw_hit -- handle hits in draw window                   */
/*                                                                      */
/************************************************************************/


static int
handle_draw_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   Integer i;
   Boolean fg;

   if (btn & RIP_BTN_NONE) return FALSE;

   fg = FALSE;

   ASHmap(ASHinq_top_window(draw_win),x,y,draw_win,&x,&y);

   if (btn & RIP_BTN_TAP) {
      add_point(x,y);
      fg = TRUE;
    }
   else if (btn & RIP_BTN_DOWN) {
      i = add_point(x,y);
      fg = rubber_band("Locate new spline point",i);
      if (!fg) {
	 remove_point(i);
       };
    }
   else if (btn & RIP_BTN_UP) return FALSE;

   spline_refresh(draw_win);

   return fg;
};





/************************************************************************/
/*                                                                      */
/*      handle_point_hit -- handle hits on points                       */
/*                                                                      */
/************************************************************************/


static int
handle_point_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   Integer pt;
   Boolean fg;

   fg = FALSE;

   pt = (Integer) RIPinq_data(rgn);

   if (btn & RIP_BTN_NONE) return FALSE;

   if (btn & RIP_BTN_TAP) {
      if (btn & RIP_BTN_LEFT) {
	 duplicate_point(pt);
	 fg = TRUE;
       }
      else if (btn & RIP_BTN_RIGHT) {
	 remove_point(pt);
	 fg = TRUE;
       };
    }
   else if (btn & RIP_BTN_UP) return FALSE;
   else {
      fg = rubber_band("Move control point",pt);
    };

   spline_refresh(draw_win);

   return TRUE;
};





/************************************************************************/
/*                                                                      */
/*      rubber_band -- handle rubberbanding a point                     */
/*      spline_track -- handle tracking during spline rb'ing            */
/*      draw_spline -- draw spline for rubberbanding                    */
/*                                                                      */
/************************************************************************/


static Boolean
rubber_band(prompt,pt)
   String prompt;
   Integer pt;
{
   Integer px,py;
   ASH_COLOR cb;
   Integer pm,cr;
   Integer lx,by,rx,ty;
   Float fx,fy;

   px = point[pt].mapx;
   py = point[pt].mapy;

   cb = ASHinq_background_color(draw_win);
   pm = ASHplane_mask(draw_win,cb^band_color);
   cr = ASHcombination_rule(draw_win,10);

   band_pt = pt;
   band_status = FALSE;

   RIPinput_track_down(draw_win,prompt,spline_track,-1);

   ASHcombination_rule(draw_win,cr);
   ASHplane_mask(draw_win,pm);

   if (!band_status) {
      point[pt].mapx = px;
      point[pt].mapy = py;
      return FALSE;
    }   
   else {
      ASHinq_size(draw_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      fx = point[pt].mapx-lx;
      fx /= (rx-lx);
      fy = point[pt].mapy-by;
      fy /= (ty-by);
      point[pt].x = fx;
      point[pt].y = fy;
    };

   return TRUE;
};




static int
spline_track(x,y,n,trans,max,w,btn,ch)
   Integer x,y;
   Integer n;
   RIP_TRANS trans;
   Integer max;
   ASH_WINDOW w;
   Integer btn;
   Integer ch;
{
   if (trans == RIP_TRANS_NONE) {
      draw_spline(draw_win);
      return 1;
    };

   draw_spline(draw_win);

   point[band_pt].mapx = x;
   point[band_pt].mapy = y;

   switch (trans) {
      case RIP_TRANS_MOVE :
	 draw_spline(draw_win);
	 break;
      case RIP_TRANS_DOWN :
      case RIP_TRANS_CHAR :
	 return 0;
      case RIP_TRANS_UP :
	 band_status = TRUE;
	 return 0;
    };

   return 1;
};




static void
draw_spline(w)
   ASH_WINDOW w;
{
   Integer i;

   for (i = 0; i < num_points; ++i) {           /* map the points       */
      xpoint[i] = point[i].mapx;
      ypoint[i] = point[i].mapy;
    };

   if (num_points > 2) {
      if (style == STYLE_OPEN) 
	 ASHspline(draw_win,num_points,xpoint,ypoint);
      else
	 ASHspline_closed(draw_win,num_points,xpoint,ypoint);
    };
};





/************************************************************************/
/*                                                                      */
/*      add_point -- add a new point                                    */
/*      duplicate_point -- duplicate an existing point                  */
/*      remove_point -- delete a point                                  */
/*                                                                      */
/************************************************************************/


static Integer
add_point(x,y)
   Integer x,y;
{
   Float fx,fy;
   Integer i;
   Integer lx,by,rx,ty;

   ASHinq_size(draw_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   fx = x-lx;
   fx /= (rx-lx);
   fy = y-by;
   fy /= (ty-by);

   if (num_points >= max_points) {
      max_points *= 2;
      point = (POINT *) realloc(point,max_points*sizeof(POINT));
      xpoint = (Integer *) realloc(xpoint,max_points*sizeof(Integer));
      ypoint = (Integer *) realloc(ypoint,max_points*sizeof(Integer));
    };

   i = num_points++;
   point[i].x = fx;
   point[i].y = fy;
   point[i].mapx = x;
   point[i].mapy = y;

   return i;
};





static Integer
duplicate_point(pt)
   Integer pt;
{
   Integer i,lst;

   lst = add_point(0,0);

   for (i = lst; i > pt; --i) point[i] = point[i-1];

   return pt+1;
};





static void
remove_point(pt)
   Integer pt;
{
   Integer i;

   for (i = pt; i < num_points-1; ++i) {
      point[i] = point[i+1];
    };

   num_points--;
};





/* end of spline.c */

