/************************************************************************/
/*									*/
/*		chars.c 						*/
/*									*/
/*	This is an example of a BWE program that uses several of the	*/
/*	basic interaction facilities.  Its function is to provide a	*/
/*	package whereby the user can draw characters and output the	*/
/*	corresponding bitmaps.						*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include <DATATYPES.h>
#include <stdio.h>
#include <math.h>

#include <ash.h>
#include <stem.h>
#include <leaf.h>
#include <wind.h>
#include <rip.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_WIDTH		8
#define DEFAULT_HEIGHT		16

#define PEN_CURSOR		86

#define MAX_DIM 		256
#define MIN_DIM 		2

#define GRID_COLOR		"gray"

#define CURSOR_WIDTH		3




/************************************************************************/
/*									*/
/*	Type Definitions						*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		setup_windows();

static	int		handle_save_btn();
static	int		handle_resize_btn();
static	int		handle_cursor_btn();
static	int		handle_grid_btn();
static	int		handle_clear_btn();
static	int		handle_quit_btn();

static	void		chars_refresh();

static	int		handle_draw_hit();

static	int		chars_track();

static	void		add_point();




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	width;
static	Integer 	height;
static	Boolean 	showgrid;
static	String		basename;
static	Integer 	out_cnt;
static	Integer 	cursor_width;

static	ASH_WINDOW	root_win;
static	ASH_WINDOW	draw_win;
static	ASH_WINDOW	menu_win;

static	String		pixels;

static	Character	out_set;
static	Character	out_unset;

static	ASH_COLOR	grid_color;




/************************************************************************/
/*									*/
/*	Tables:  Window layout						*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define DRAW_WIN	2



static LEAF_DATA chars_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { DRAW_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT_LINE(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	Tables:  Pull down menus					*/
/*									*/
/************************************************************************/


static STEM_PDM_DATA chars_pdm[] = {
   { STEM_PSTATE_MENU,	"System",       NULL },
      { STEM_PSTATE_BTN,	"Save",         handle_save_btn },
      { STEM_PSTATE_BTN,	"Save As",      handle_save_btn },
      { STEM_PSTATE_BTN,	"Cursor",       handle_cursor_btn },
      { STEM_PSTATE_BTN,	"Resize",       handle_resize_btn },
      { STEM_PSTATE_BTN,	"Show Grid",    handle_grid_btn },
      { STEM_PSTATE_BTN,	"Clear",        handle_clear_btn },
      { STEM_PSTATE_BTN,	"Quit",         handle_quit_btn },
   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	main -- main program						*/
/*									*/
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   ASH_WINDOW w;
   Character buf[1024];
   Integer i;

   width = 0;
   height = 0;
   showgrid = TRUE;
   cursor_width = 0;
   basename = NULL;
   out_cnt = 0;
   pixels = NULL;
   out_set = '1';
   out_unset = '0';

   argc = ASHset_application(argc,argv);

   for (i = 1; i < argc; ++i) {
      if (argv[i][0] == '-') {
	 switch (argv[i][1]) {
	    case 'w' :
	       width = atol(argv[++i]);
	       break;
	    case 'h' :
	       height = atol(argv[++i]);
	       break;
	    case 'c' :
	       cursor_width = atol(argv[++i]);
	       break;
	    case 'g' :
	       showgrid = !showgrid;
	       break;
	    case 'o' :
	       if (argv[i][2] != 0) {
		  out_set = argv[i][2];
		  if (argv[i][3] != 0) out_unset = argv[i][3];
		  else out_unset = ' ';
		};
	       break;
	    default :
	       fprintf(stderr,"chars [-w width] [-h height] [-g] [-o10] [base]\n");
	       exit(1);
	  };
       }
      else {
	 if (basename == NULL) basename = argv[i];
	 else {
	    fprintf(stderr,"chars [-w width] [-h height] [-g] [-o10] [base]\n");
	    exit(1);
	  };
       };
    };

   if (width == 0) width = DEFAULT_WIDTH;
   if (height == 0) height = DEFAULT_HEIGHT;
   if (cursor_width == 0) cursor_width = CURSOR_WIDTH;
   if (basename == NULL) basename = "CHAROUT";

   out_cnt = 0;
   for (out_cnt = 0; ; ++out_cnt) {
      sprintf(buf,"%s.%d",basename,out_cnt);
      if (access(buf,0) < 0) break;
    };

   w = ASHinit(ASH_MODE_INQUIRE);
   grid_color = ASHlookup_color(w,GRID_COLOR);

   setup_windows(w);

   while (1) {
      RIPuser_pick(NULL);
    };
};





/************************************************************************/
/*									*/
/*	setup_windows -- setup windows					*/
/*									*/
/************************************************************************/


static void
setup_windows(top)
   ASH_WINDOW top;
{
   ASH_WINDOW w;

   root_win = top;

   LEAFsetup_window(top,chars_leaf,NULL);
   LEAFredraw(top);

   draw_win = LEAFinq_window(top,DRAW_WIN,0);
   if (draw_win == NULL) {
      printf("Window too small to run chars program\n");
      exit(1);
    };

   menu_win = LEAFinq_window(top,MENU_WIN,0);
   STEMpdm_define(menu_win,NULL,chars_pdm);
   STEMpdm_btn_select(menu_win,"System","Show Grid",showgrid);
   ASHvisible(menu_win,TRUE);

   ASHset_refresh(draw_win,chars_refresh);
   ASHvisible(draw_win,TRUE);
};





/************************************************************************/
/*									*/
/*	handle_save_btn -- save the current character			*/
/*	handle_resize_btn -- change grid size				*/
/*	handle_grid_btn -- turn grid on/off				*/
/*	handle_clear_btn -- clear current character			*/
/*	handle_quit_btn -- exit 					*/
/*									*/
/************************************************************************/


static int
handle_save_btn(data,mnm,btn)
   Integer data;
   String mnm;
   String btn;
{
   Character buf[1024],menu[1024];
   Integer ctr,use,clr;
   FILE * otf;
   Integer i,j;
   String s;

   if (pixels == NULL) return FALSE;

   clr = TRUE;

   if (out_cnt < 0 || STREQL(btn,"Save As")) {
      strcpy(menu,"%CFile to Save\n\n");
      strcat(menu,"Base Name: %0.48t\n");
      strcat(menu,"Counter:   %1d\n");
      strcat(menu,"   %2o Use counter\n");
      strcat(menu,"   %3o Clear after save\n\n");
      strcat(menu,"   %a%M   %c");
      strcpy(buf,basename);
      if (out_cnt < 0) {
	 use = FALSE;
	 ctr = 0;
       }
      else {
	 use = TRUE;
	 ctr = out_cnt;
       }
      if (!STEMdialog1(root_win,menu,buf,&ctr,&use,&clr)) return FALSE;
      if (buf[0] == 0) return FALSE;
      basename = SALLOC(buf);
      if (use) out_cnt = ctr;
      else out_cnt = -1;
    };

   if (out_cnt < 0) strcpy(buf,basename);
   else {
      sprintf(buf,"%s.%d",basename,out_cnt);
      out_cnt++;
    };

   otf = fopen(buf,"w");
   if (otf == NULL) {
      sprintf(menu,"Can't open file %s\n\n%%M   %%a",buf);
      STEMdialog1(root_win,menu);
      return FALSE;
    };

   s = pixels;
   for (i = 0; i < height; ++i) {
      for (j = 0; j < width; ++j) {
	 if (*s) putc(out_set,otf);
	 else putc(out_unset,otf);
	 ++s;
       };
      putc('\n',otf);
    };

   fclose(otf);

   printf("Image saved as %s\n",buf);

   if (clr) {
      bzero(pixels,width*height);
      chars_refresh(draw_win);
    };

   return TRUE;
};





static int
handle_cursor_btn(data,mnm,btn)
   Integer data;
   String mnm;
   String btn;
{
   Integer wid;
   Character menu[1024];

   strcpy(menu,"%CCursor Width (in Pixels)\n\n");
   strcat(menu,"Width:  %0d\n");
   strcat(menu,"   %a%M   %c");

   wid = cursor_width;

   if (!STEMdialog1(root_win,menu,&wid)) return FALSE;

   if (wid < 0) return FALSE;

   cursor_width = wid;

   return TRUE;
};




static int
handle_resize_btn(data,mnm,btn)
   Integer data;
   String mnm;
   String btn;
{
   Integer wid,ht;
   Character menu[1024];

   strcpy(menu,"%CNew Grid Size\n\n");
   strcat(menu,"Width:  %0d\n");
   strcat(menu,"Height: %1d\n\n");
   strcat(menu,"   %a%M   %c");

   wid = width;
   ht = height;

   if (!STEMdialog1(root_win,menu,&wid,&ht)) return FALSE;

   if (wid < MIN_DIM || wid > MAX_DIM) return FALSE;
   if (ht < MIN_DIM || ht > MAX_DIM) return FALSE;

   if (pixels != NULL) {
      free(pixels);
      pixels = NULL;
    };

   width = wid;
   height = ht;

   chars_refresh(draw_win);

   return TRUE;
};




static int
handle_grid_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   showgrid = !showgrid;

   STEMpdm_btn_select(menu_win,"System","Show Grid",showgrid);

   chars_refresh(draw_win);

   return TRUE;
};




static int
handle_clear_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   if (pixels != NULL) {
      bzero(pixels,width*height);
      chars_refresh(draw_win);
    };

   return TRUE;
};





static int
handle_quit_btn(data,menu,btn)
   Integer data;
   String menu;
   String btn;
{
   exit(0);
};





/************************************************************************/
/*									*/
/*	chars_refresh -- redraw current character			*/
/*									*/
/************************************************************************/



static void
chars_refresh(w)
   ASH_WINDOW w;
{
   Integer lx,by,rx,ty;
   Integer x0,y0,x1,y1;
   Integer i,j;
   Float sqw,sqh;
   String s;
   ASH_COLOR cl;

   if (pixels == NULL) {
      pixels = (char *) calloc(1,width*height);
    };

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   ASHclear_box(w,lx,by,rx,ty);

   RIPremove_window_regions(w);
   RIPdefine_region(w,0,0,0,0,RIP_NO_CHARS,RIP_BTN_ANY_EITHER,
		       handle_draw_hit,ASH_SENSE_NO_CHANGE);

   sqw = (rx-lx-1);
   sqw /= width;
   sqh = (by-ty-1);
   sqh /= height;

   if (showgrid) {
      cl = ASHcolor(w,grid_color);
      for (i = 0; i <= width; ++i) {
	 j = i*sqw + 0.5;
	 ASHline(w,j,by,j,ty);
       };
      for (i = 0; i <= height; ++i) {
	 j = i*sqh + 0.5;
	 ASHline(w,lx,j,rx,j);
       };
      ASHcolor(w,cl);
    };

   s = pixels;
   for (i = 0; i < height; ++i) {
      for (j = 0; j < width; ++j) {
	 if (*s) {
	    x0 = j*sqw + 0.5 + 1;
	    x1 = (j+1)*sqw + 0.5 - 1;
	    y0 = i*sqh + 0.5 + 1;
	    y1 = (i+1)*sqh + 0.5 - 1;
	    ASHrectangle(w,x0,y0,x1,y1);
	  };
	 ++s;
       };
    };
};





/************************************************************************/
/*									*/
/*	handle_draw_hit -- handle hits in draw window			*/
/*									*/
/************************************************************************/


static int
handle_draw_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   Integer i;
   Boolean fg;

   if (btn & RIP_BTN_NONE) return FALSE;

   fg = FALSE;

   ASHmap(ASHinq_top_window(draw_win),x,y,draw_win,&x,&y);

   if (btn & RIP_BTN_TAP) {
      add_point(x,y);
      fg = TRUE;
    }
   else if (btn & RIP_BTN_DOWN) {
      add_point(x,y);
      RIPinput_track_down(draw_win,"",chars_track,-1);
    }
   else if (btn & RIP_BTN_UP) return FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	chars_track -- handle tracking during mouse down		*/
/*									*/
/************************************************************************/


static int
chars_track(x,y,n,trans,max,w,btn,ch)
   Integer x,y;
   Integer n;
   RIP_TRANS trans;
   Integer max;
   ASH_WINDOW w;
   Integer btn;
   Integer ch;
{
   if (trans == RIP_TRANS_NONE) {
      return 1;
    };

   add_point(x,y);

   switch (trans) {
      case RIP_TRANS_MOVE :
	 break;
      case RIP_TRANS_DOWN :
      case RIP_TRANS_CHAR :
      case RIP_TRANS_UP :
	 return 0;
    };

   return 1;
};




/************************************************************************/
/*									*/
/*	add_point -- add a new point					*/
/*									*/
/************************************************************************/


static void
add_point(x,y)
   Integer x,y;
{
   Float sqw,sqh;
   Integer i0,i1,j0,j1,i,j;
   Integer lx,by,rx,ty;
   Integer x0,y0,x1,y1;

   if (pixels == NULL) return;

   ASHinq_size(draw_win,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   sqw = (rx-lx-1);
   sqw /= width;
   sqh = (by-ty-1);
   sqh /= height;

   i0 = (x-cursor_width) / sqw;
   i1 = (x+cursor_width) / sqw;
   j0 = (y-cursor_width) / sqh;
   j1 = (y+cursor_width) / sqh;

   if (i1 < 0 || i0 >= width) return;
   if (j1 < 0 || j0 >= height) return;

   if (i0 < 0) i0 = 0;
   if (i1 >= width) i1 = width-1;
   if (j0 < 0) j0 = 0;
   if (j1 >= height) j1 = height-1;

   for (i = i0; i <= i1; ++i) {
      for (j = j0; j <= j1; ++j) {
	 if (!pixels[j*width+i]) {
	    pixels[j*width+i] = TRUE;
	    x0 = i*sqw + 0.5 + 1;
	    x1 = (i+1)*sqw + 0.5 - 1;
	    y0 = j*sqh + 0.5 + 1;
	    y1 = (j+1)*sqh + 0.5 - 1;
	    ASHrectangle(draw_win,x0,y0,x1,y1);
	  };
       };
    };
 };





/* end of chars.c */

