/************************************************************************/
/*									*/
/*		edtwin.c						*/
/*									*/
/*	Window interface for base editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define SEPARATION_SPACE 3
#define PROMPT_CHAR	'\1'




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_edt_data;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	edt_ash_control();
static	Integer 	edt_view_control();
static	void		edt_window_refresh();
static	void		edt_status_refresh();
static	void		edt_free_editor();
static	void		check_edit();
static	void		setup_mode();
static	void		setup_initial_view();
static	void		fixup_sizes();
static	void		resize_views();
static	void		close_all_views();
static	void		remove_view();
static	void		check_bounds();
static	int		view_changer();
static	int		view_changer_track();
static	void		check_shell_input();
static	void		status_display();
static	Integer 	output_flag();
static	String		input_filter();
static	String		output_filter();
static	void		read_control_file();
static	void		write_control_file();
static	void		goto_shellpos();
static	void		get_end_shellpos();
static	void		get_cur_shellpos();






/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static struct _MODE_TBL {
   String when;
   String mode;
} mode_table[] = {
   { ".c.h.y.C.", "C" },
   { ".l.lisp.", "LISP" },
   { ".n.r.me.ms.tex.t.1.2.3.4.5.6.7.8.", "ROFF" },
   { ".f.", "FORTRAN" },
   { ".p.", "PASCAL" },
   { NULL, "TEXT" }
};





/************************************************************************/
/*									*/
/*	EDT_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_win_init()
{
   all_edt_data = NULL;
};





/************************************************************************/
/*									*/
/*	EDTdefine_editor -- setup editor in window			*/
/*									*/
/************************************************************************/


EDT_ID
EDTdefine_editor(w,file,ed,ud,menuwin,stswin)
   ASH_WINDOW w;
   String file;
   EDT_DATA ed;
   Universal ud;
   ASH_WINDOW menuwin;
   ASH_WINDOW stswin;
{
   EDT_ID ei;
   RIP_REGION rr;
   Integer sts;
   Character buf[256] ;
   Integer line,colm;
   FILE_POS npos,spos;

   if (w == NULL) return NULL;

   if (!ASHlock(w)) return NULL;

   EDT_init();

   EDT_ctbl_startup();

   line = 1;
   colm = 1;

   if (file == NULL) {
      buf[0] = 0;
      if (ed->savectl) {
	 read_control_file(buf,&line,&colm);
       };
      if (buf[0] == 0) {
	 strcpy(buf,"/tmp/EDTstringXXXXXX");
	 mktemp(buf);
       };
      file = buf;
    };

   ei = PALLOC(EDT_ID_INFO);
   ei->window = w;
   ei->lockwin = w;
   ei->menuwin = menuwin;
   ei->stswin = stswin;
   ei->filename = NULL;
   ei->data = ed;
   ei->udata = ud;
   ei->shell = (ei->data->input_rtn != NULL);
   ei->wait = FALSE;
   ei->in_command = FALSE;
   ei->views = NULL;
   ei->mode = NULL;
   ei->line_size = -1;
   ei->select.view = NULL;
   ei->select.valid = FALSE;
   ei->select.unit = EDT_SELECT_CHAR;
   ei->macro = NULL;
   ei->histct = DEFAULT_HISTORY;
   ei->histfirst = 0;
   ei->history = (Integer *) calloc(ei->histct,sizeof(Integer));
   ei->escape = 0;
   ei->marks = FILE_MARK_NEW;
   ei->cmdct = 0;
   ei->status_line = 0;
   ei->force_bounds = FALSE;
   ei->force_save = FALSE;
   ei->no_close = FALSE;
   ei->input_filter = ei->data->input_filter;
   ei->output_filter = ei->data->output_filter;
   ei->font = ASHinq_font(w);
   ei->shell_pos = NULL;

   EDT_ctbl_setup(ei);
   EDT_abbrev_setup(ei);

   ei->raw_mode = ei->data->raw;

   ASHset_control(w,edt_ash_control);
   ASHset_user_data(w,ei);
   ASHset_refresh(w,edt_window_refresh);
   ASHset_window_id(w,"edt");
   ASHset_window_defaults(w);
   ASHinq_window_resource(w,"font");

   if (stswin != NULL) {
      ASHset_refresh(stswin,edt_status_refresh);
      ASHset_user_data(stswin,ei);
    };

   PROTECT;
   all_edt_data = CONS(ei,all_edt_data);
   UNPROTECT;

   if (LPROTECT(ei)) {
      BIOnew_input_window(w);
      BIOset_cursor_standard(w,VIEW_CURSOR);

      rr = RIPdefine_region(w,0,0,0,0,RIP_NO_CHARS,RIP_BTN_ANY_DOWN,view_changer,
			       ASH_SENSE_NO_CHANGE);
      RIPset_data(rr,ei);

      for ( ; ; ) {
	 sts = EDT_open_file(ei,file);
	 if (sts > 0) break;
	 if (sts < 0) {
	    LUNPROTECT(ei);
	    ASHunlock(w);
	    return NULL;
	  };
	 file = NULL;
       };

      EDassoc(ei->edit_file,w);

      if (ei->shell) {
	 get_end_shellpos(ei);
	 FILEinq_position(ei->edit_file,FILE_MOVE_END,0,&spos);
	 FILEedit_limits(ei->edit_file,&spos,FILE_POS_END);
	 ei->auto_indent = FALSE;
       };

      if (!ei->shell && ei->data->disable) EDTenable(ei,FALSE);


      if (line !=1 || colm != 1) {
	 if (line != 1) {
	    FILEmove(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE,line-1,&npos);
	  };
	 if (colm != 1) {
	    FILEmove(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_CHAR,colm-1,&npos);
	  };
	 if (ei->views != NULL && ei->views->view != NULL) {
	    EDT_synch_view(ei->views->view);
	  };
	 EDT_save_command(ei,TRUE,FALSE,FALSE);
       };

      check_bounds(ei,TRUE);

      LUNPROTECT(ei);

      edt_window_refresh(w);
    };

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTrelease -- finish reading in input for editor		*/
/*									*/
/************************************************************************/


void
EDTrelease(ei)
   EDT_ID ei;
{
   if (ei->data->input_filter != NULL && ei->edit_file != NULL) {
      FILErelease(ei->edit_file);
    };
};





/************************************************************************/
/*									*/
/*	EDTinq_line_text -- return text for given line			*/
/*									*/
/************************************************************************/


String
EDTinq_line_text(ei,line,buf)
   EDT_ID ei;
   Integer line;
   Character buf[];
{
   FILE_POS p1,p2;
   String s;

   FILEinq_position(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE_START|
		       FILE_MOVE_LINE, line-1,&p1);
   FILEinq_position(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE_END|
		       FILE_MOVE_LINE, line-1,&p2);

   s = FILEcopy_text(ei->edit_file,&p1,&p2);

   if (buf != NULL) {
      strcpy(buf,s);
      free(s);
      s = buf;
    };

   return s;
};





/************************************************************************/
/*									*/
/*	EDTinq_contents -- return contents of editor			*/
/*									*/
/************************************************************************/


String
EDTinq_contents(ei)
   EDT_ID ei;
{
   String s;

   if (ei->edit_file == NULL) return NULL;

   s = FILEcopy_text(ei->edit_file,FILE_POS_START,FILE_POS_END);

   return s;
};





/************************************************************************/
/*									*/
/*	EDTcur_line -- return current line				*/
/*	EDTcur_char -- return current character 			*/
/*	EDTlast_line -- return last line				*/
/*									*/
/************************************************************************/


Integer
EDTcur_line(ei)
   EDT_ID ei;
{
   Integer ln;

   ln = FILEinq_currline(ei->edit_file);

   return ln;
};





Integer
EDTcur_char(ei)
   EDT_ID ei;
{
   Integer ch;

   ch = FILEinq_currchar(ei->edit_file);

   return ch;
};





Integer
EDTlast_line(ei)
   EDT_ID ei;
{
   Integer ln;

   if (ei->edit_file == NULL) return 0;

   ln = FILEinq_line(ei->edit_file,FILE_POS_END);

   return ln;
};





/************************************************************************/
/*									*/
/*	EDTinq_line_pos -- return position of line in editor		*/
/*	EDTcorrelate_line -- get line given y coordinate		*/
/*	EDTinq_bounds -- return top/bottom of window in lines		*/
/*									*/
/************************************************************************/


Integer
EDTinq_line_pos(ei,line,base,top,mx)
   EDT_ID ei;
   Integer line;
   Integer base[];
   Integer top[];
   Integer mx;
{
   register EDT_VINFO vi;
   register Integer ct;
   Integer bv,tv;

   ct = 0;
   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (EDT_view_line_pos(vi->view,line,&bv,&tv)) {
	 ASHmap(vi->window,0,bv,ei->window,NULL,&bv);
	 ASHmap(vi->window,0,tv,ei->window,NULL,&tv);
	 base[ct] = bv;
	 top[ct] = tv;
	 ++ct;
	 if (ct >= mx) break;
       };
    };

   return ct;
};





Integer
EDTcorrelate_line(ei,y)
   EDT_ID ei;
   Integer y;
{
   register EDT_VINFO vi;
   register Integer ln;

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if ((y-vi->by)*(y-vi->ty) <= 0) break;
    };

   if (vi == NULL) ln = -1;
   else {
      ASHmap(ei->window,0,y,vi->window,NULL,&y);
      ln = EDT_view_correlate_line(vi->view,y);
    };

   return ln;
};





Integer
EDTinq_bounds(ei,widp,tops,bots,mx)
   EDT_ID ei;
   Integer * widp;
   Integer tops[];
   Integer bots[];
   Integer mx;
{
   register EDT_VINFO vi;
   Integer top,bot;
   register Integer y,ct;

   ct = 0;
   for (vi = ei->views; vi != NULL; vi = vi->next) {
      y = EDT_view_inq_bounds(vi->view,&top,&bot);
      if (y == 0) continue;
      if (ct == 0 && widp != NULL) *widp = y;
      if (tops != NULL) tops[ct] = top;
      if (bots != NULL) bots[ct] = bot;
      ++ct;
      if (ct >= mx) break;
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	EDTmarker -- setup file marker					*/
/*	EDTmark_line -- return line for given marker			*/
/*	EDTmark_char -- return char for given marker			*/
/*									*/
/************************************************************************/


EDT_MARK
EDTmarker(ei,line,col)
   EDT_ID ei;
   Integer line;
   Integer col;
{
   register Integer fm;
   register EDT_MARK em;
   FILE_POS pos,opos,npos;

   if (ei->edit_file == NULL) return NULL;

   if (!LPROTECT(ei)) return NULL;

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEset_position(ei->edit_file,line,col,&pos);

   fm = FILEsave_pos(ei->edit_file);

   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&npos);

   em = PALLOC(EDT_MARK_INFO);
   em->file_pos = fm;
   em->valid = TRUE;

   em->next = ei->all_marks;
   ei->all_marks = em;

   LUNPROTECT(ei);

   return em;
};





Integer
EDTmark_line(ei,fm)
   EDT_ID ei;
   EDT_MARK fm;
{
   FILE_POS pos;

   if (!fm->valid || fm->file_pos == NULL) return -1;
   if (ei->edit_file == NULL) return -1;

   FILEinq_savepos(ei->edit_file,fm->file_pos,&pos);

   if (FILEpos_compare(ei->edit_file,&pos,FILE_POS_NONE) == 0) return -1;

   return FILEinq_line(ei->edit_file,&pos);
};





Integer
EDTmark_char(ei,fm)
   EDT_ID ei;
   EDT_MARK fm;
{
   FILE_POS pos;

   if (!fm->valid) return -1;
   if (ei->edit_file == NULL) return -1;

   FILEinq_savepos(ei->edit_file,fm->file_pos,&pos);

   if (FILEpos_compare(ei->edit_file,&pos,FILE_POS_NONE) == 0) return -1;

   return FILEinq_char(ei->edit_file,&pos);
};





/************************************************************************/
/*									*/
/*	EDTset_file -- change file					*/
/*									*/
/************************************************************************/


Boolean
EDTset_file(ei,file)
   EDT_ID ei;
   String file;
{
   if (ei == NULL || ei->in_command) return FALSE;

   if (!EDT_close_file(ei,TRUE,TRUE,TRUE,FALSE)) return FALSE;

   if (EDT_open_file(ei,file) <= 0) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDTclose -- close editor					*/
/*	EDTclose_save -- close editor and save result			*/
/*	EDTsecure -- save file						*/
/*	EDTforce_save -- force editor to save result on close		*/
/*	EDTprohibit_close -- don't allow user to close editor           */
/*	EDTsynch -- insure file is set in case of a crash		*/
/*									*/
/************************************************************************/


Boolean
EDTclose(ei)
   EDT_ID ei;
{
   Boolean fg;

   if (ei == NULL || ei->in_command) return FALSE;

   if (ei->window != NULL) {
      fg = EDT_close_file(ei,FALSE,FALSE,FALSE,FALSE);
    }
   else {
      edt_free_editor(ei);
      fg = TRUE;
    };

   return fg;
};





Boolean
EDTclose_save(ei)
   EDT_ID ei;
{
   Boolean fg;

   if (ei == NULL || ei->in_command) return FALSE;

   if (ei->window != NULL) {
      fg = EDT_close_file(ei,TRUE,FALSE,FALSE,FALSE);
    }
   else fg = TRUE;

   return fg;
};





Boolean
EDTclose_abort(ei)
   EDT_ID ei;
{
   Boolean fg;

   if (ei == NULL || ei->in_command) return FALSE;

   if (ei->window != NULL) {
      fg = EDT_close_file(ei,FALSE,TRUE,FALSE,TRUE);
    }
   else fg = TRUE;

   return fg;
};





Boolean
EDTsecure(ei,asname)
   EDT_ID ei;
   String asname;
{
   if (asname == NULL && !ei->force_save && !FILEmodified(ei->edit_file)) return TRUE;

   FILEsave_as(ei->edit_file,asname);

   if (ei->data->file_rtn != NULL) {
      (*ei->data->file_rtn)(ei->udata,EDT_FILE_SECURE,ei->filename);
    };
   ASHcontrol_msg(ei->window,"EDIT$SAVE");

   return TRUE;
};





void
EDTforce_save(ei)
   EDT_ID ei;
{
   ei->force_save = TRUE;
};





void
EDTprohibit_close(ei)
   EDT_ID ei;
{
   ei->no_close = TRUE;
};





void
EDTsynch(ei)
   EDT_ID ei;
{
   if (ei != NULL && ei->edit_file != NULL) {
      FILEsecure(ei->edit_file);
    };
};





Boolean
EDTinq_modified(ei)
   EDT_ID ei;
{
   if (ei == NULL || ei->edit_file == NULL) return FALSE;

   return FILEmodified(ei->edit_file);
};





/************************************************************************/
/*									*/
/*	EDTshell_output -- output text for shell			*/
/*	EDTshell_done_prompt -- indicate prompt has been output 	*/
/*									*/
/************************************************************************/


void
EDTshell_output(ei,txt)
   EDT_ID ei;
   String txt;
{
   FILE_POS p1,p2,p3,spos;
   Integer mark;
   register String s;
   Boolean newfg;
   Boolean prmfg;
   EDT_VIEW ev;
   Boolean forcefg;
   Integer linelen,pos;

   if (!LPROTECT(ei)) return;

   prmfg = FALSE;
   forcefg = FALSE;

   if (ei->edit_file == NULL) return;

   EDT_clear_selection(ei,NULL);

   mark = FILEsave_pos(ei->edit_file);
   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&p2);

   if (!ei->shell) get_end_shellpos(ei);

   goto_shellpos(ei);

   if (EDTinq_bounds(ei,&linelen,NULL,NULL,1) == 0) linelen = 256;

   for (s = txt; *s != 0; ++s) {
      if (*s == PROMPT_CHAR) prmfg = TRUE;
      else if (*s != '\r') {
	 if (ei->auto_linefeed && *s != '\n' && *s != '\b' &&
		FILEinq_currchar(ei->edit_file)-1 > linelen)
	    FILEtypein(ei->edit_file,'\n',TRUE);
	 if (*s == '\t') {
	    pos = FILEinq_currchar(ei->edit_file);
	    pos = (pos-1) % 8;
	    if (pos == 0) pos = 8;
	    while (pos-- > 0) FILEtypein(ei->edit_file,' ',TRUE);
	  }
	 else FILEtypein(ei->edit_file,*s,TRUE);
       };
    };

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&spos);
   get_cur_shellpos(ei);

   if (index(txt,'\n') != NULL) {
      newfg = TRUE;
      FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&p2);
      if (ei->shell) FILEedit_limits(ei->edit_file,&p2,FILE_POS_END);
    }
   else newfg = FALSE;

   if (!ei->hold_mode && ei->views != NULL && ei->views->view != NULL) {
      ev = ei->views->view;
      forcefg = TRUE;
    }
   else {
      FILEinq_savepos(ei->edit_file,mark,&p1);
      FILEmove(ei->edit_file,FILE_MOVE_ABS,&p1,&p3);
    };

   FILEdelete_pos(ei->edit_file,mark);

   if (newfg) check_bounds(ei,TRUE);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   if (forcefg) TEXTcursor(VIEW_TEXT(ev),TEXT_CURSOR_FORCE,&spos);

   if (prmfg) EDTshell_done_prompt(ei);

   LUNPROTECT(ei);
};





void
EDTshell_done_prompt(ei)
   EDT_ID ei;
{
   FILE_POS spos;

   if (ei->shell && ei->shell_pos != NULL) {
      FILEinq_savepos(ei->edit_file,ei->shell_pos,&spos);
      EDT_next_char(ei,&spos,&spos);
      FILEedit_limits(ei->edit_file,&spos,FILE_POS_END);
    };
};





/************************************************************************/
/*									*/
/*	EDTgoto -- make specified line visible				*/
/*	EDTgoto_alt -- make specified line visible in alternate view	*/
/*	EDTgoto_top -- make specified line first line in window 	*/
/*									*/
/************************************************************************/


void
EDTgoto(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos;
   EDT_VIEW ev;
   Integer t,b;

   if (ei->views == NULL || ei->in_command) return;

   ev = ei->views->view;
   if (ev == NULL || ei->edit_file == NULL) return;

   if (line <= 0) line = 1;

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEmove(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
	       line-1,&pos);

   t = TEXTinq_topline(VIEW_TEXT(ev));
   b = TEXTinq_bottomline(VIEW_TEXT(ev));
   if (line < t || line > b) {
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_NEAR_MIDDLE,&pos);
    };

   EDT_save_command(ei,TRUE,FALSE,FALSE);

   LUNPROTECT(ei);
};





void
EDTgoto_alt(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos;
   EDT_VIEW ev;
   Integer t,b;

   if (ei->views == NULL || ei->in_command) return;

   if (ei->views->next == NULL) {
      ev = ei->views->view;
      FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&pos);
      EDT_split_view(ev);
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_NEAR_MIDDLE,&pos);
    };

   ev = ei->views->next->view;
   if (ev == NULL || ei->edit_file == NULL) return;

   if (line <= 0) line = 1;

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEmove(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
	       line-1,&pos);

   t = TEXTinq_topline(VIEW_TEXT(ev));
   b = TEXTinq_bottomline(VIEW_TEXT(ev));
   if (line < t || line > b) {
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_NEAR_MIDDLE,&pos);
    };

   EDT_save_command(ei,TRUE,FALSE,FALSE);

   LUNPROTECT(ei);
};





void
EDTgoto_top(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos;
   EDT_VIEW ev;

   if (ei->views == NULL || ei->in_command) return;

   ev = ei->views->view;
   if (ev == NULL || ei->edit_file == NULL) return;

   if (line <= 0) line = 1;

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEmove(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
	       line-1,&pos);

   TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_TOP_LEFT,&pos);

   EDT_save_command(ei,TRUE,FALSE,FALSE);

   LUNPROTECT(ei);
};





/************************************************************************/
/*									*/
/*	EDTenable -- enable/disable file editing			*/
/*	EDTinsert_line -- insert line into file 			*/
/*	EDTdelete_line -- delete line from file 			*/
/*	EDTreplace_line -- replace line in file 			*/
/*	EDTtruncate_at -- truncate file at line 			*/
/*	EDTkeep_position -- force view to keep at current position	*/
/*	EDThilite_line -- highlight a given line			*/
/*	EDTjoin_views							*/
/*	EDTtypein -- handle input directed at editor			*/
/*									*/
/************************************************************************/


void
EDTenable(ei,fg)
   EDT_ID ei;
   Boolean fg;
{
   if (ei->edit_file == NULL) return;

   if (fg) {
      ei->readonly_mode = FALSE;
      FILEedit_limits(ei->edit_file,FILE_POS_START,FILE_POS_END);
    }
   else {
      ei->readonly_mode = TRUE;
      FILEedit_limits(ei->edit_file,FILE_POS_END,FILE_POS_START);
    };
};





void
EDTinsert_line(ei,line,text)
   EDT_ID ei;
   Integer line;
   String text;
{
   FILE_POS pos,opos;
   FILE_BUF buf;

   buf = FILEinq_buffer("BWEDIT$_LINE_EDIT");

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEset_position(ei->edit_file,line,1,&pos);
   FILEbuffer_define(buf,text);
   FILEbuffer_append(buf,"\n");
   FILEinsert(ei->edit_file,buf,1);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&pos);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   LUNPROTECT(ei);
};





void
EDTdelete_line(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos,opos;
   FILE_BUF buf;

   buf = FILEinq_buffer("BWEDIT$_LINE_EDIT");

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEset_position(ei->edit_file,line,1,&pos);
   FILEdelete(ei->edit_file,buf,FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_START,1);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&pos);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   LUNPROTECT(ei);
};





void
EDTreplace_line(ei,line,text)
   EDT_ID ei;
   Integer line;
   String text;
{
   FILE_POS pos,opos;
   FILE_BUF buf;

   buf = FILEinq_buffer("BWEDIT$_LINE_EDIT");

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEset_position(ei->edit_file,line,1,&pos);
   FILEdelete(ei->edit_file,buf,FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_END,0);
   FILEbuffer_define(buf,text);
   FILEinsert(ei->edit_file,buf,1);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&pos);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   LUNPROTECT(ei);
};





void
EDTtruncate_at(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos,opos;
   FILE_BUF buf;

   buf = FILEinq_buffer("BWEDIT$_LINE_EDIT");

   if (!LPROTECT(ei)) return;

   EDT_clear_selection(ei,NULL);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEset_position(ei->edit_file,line,1,&pos);
   FILEdelete(ei->edit_file,buf,FILE_MOVE_END,0);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&pos);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   LUNPROTECT(ei);
};





void
EDTkeep_position(ei,fg)
   EDT_ID ei;
   Boolean fg;
{
   EDT_VIEW ev;
   FILE_POS cpos;

   if (ei->views == NULL) return;

   if (!LPROTECT(ei)) return;

   ev = ei->views->view;
   if (ev == NULL || ei->edit_file == NULL) return;

   TEXTcursor_enable(VIEW_TEXT(ev),(fg ? 1 : 0));

   if (fg) {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&cpos);
      TEXTcursor(VIEW_TEXT(ev),TEXT_CURSOR_FORCE,&cpos);
    };

   LUNPROTECT(ei);
};






void
EDThilite_line(ei,line)
   EDT_ID ei;
   Integer line;
{
   FILE_POS pos;
   EDT_VIEW ev;

   if (!LPROTECT(ei)) return;

   if (ei->views == NULL) return;

   ev = ei->views->view;

   FILEset_position(ei->edit_file,line,1,&pos);
   EDT_start_selection(ev,&ei->select,&pos,FALSE);
   ei->select.unit = EDT_SELECT_LINE;
   EDT_extend_selection(ev,&ei->select,&pos,FALSE,FALSE);
   EDT_finish_selection(ev,&ei->select);

   EDT_save_command(ei,TRUE,FALSE,TRUE);

   LUNPROTECT(ei);
};





void
EDTjoin_views(ei)
   EDT_ID ei;
{
   if (!LPROTECT(ei)) return;

   if (ei->views == NULL) return;

   while (ei->views->next != NULL) {
      EDT_remove_view(ei->views->next->view);
    };

   LUNPROTECT(ei);
};





void
EDTtypein(ei,ch)
   EDT_ID ei;
   Integer ch;
{
   EDT_VIEW ev;

   if (!LPROTECT(ei)) return;

   if (ei->views == NULL) return;

   ev = ei->views->view;

   EDT_button_handler(0,0,ch,RIP_BTN_NONE,ev->region);

   LUNPROTECT(ei);
};





/************************************************************************/
/*									*/
/*	EDTset_user_data -- set user data field for editor		*/
/*	EDTset_mouse_rtn -- set mouse routine for editor		*/
/*	EDTauto_select_all -- set all as default replace selected	*/
/*	EDTauto_linefeed -- automatically do line feeds for transcripts */
/*									*/
/************************************************************************/


void
EDTset_user_data(ei,ud)
   EDT_ID ei;
   Universal ud;
{
   ei->udata = ud;
};




void
EDTset_mouse_rtn(ei,mr)
   EDT_ID ei;
   Function_Ptr mr;
{
   ei->data->mouse_rtn = mr;
};




void
EDTauto_select_all(ei,fg)
   EDT_ID ei;
   Boolean fg;
{
   ei->auto_select_all = fg;
};




void
EDTauto_linefeed(ei,fg)
   EDT_ID ei;
   Boolean fg;
{
   ei->auto_linefeed = fg;
};




/************************************************************************/
/*									*/
/*	EDTinq_selection -- return last global/local selection		*/
/*									*/
/************************************************************************/


String
EDTinq_selection(ei)
   EDT_ID ei;
{
   FILE_BUF fbf;
   String s;

   if (ei == NULL) {
      fbf = FILEinq_buffer(SELECT_BUFFER_NAME);
      s = FILEbuffer_contents(fbf);
      if (s != NULL && *s == 0) {
	 free(s);
	 s = NULL;
       };
    }
   else if (ei->select.valid) {
      s = FILEcopy_text(ei->edit_file,&ei->select.begin,&ei->select.end);
    }
   else s = NULL;

   return s;
};





/************************************************************************/
/*									*/
/*	EDT_info_by_window -- find EDT_ID given window			*/
/*	EDT_info_by_menu -- find EDT_ID given menu window		*/
/*	EDT_info_by_editor -- find EDT_ID given FILE_ID 		*/
/*									*/
/************************************************************************/


EDT_ID
EDT_info_by_window(w)
   ASH_WINDOW w;
{
   EDT_ID ei;

   ei = (EDT_ID) ASHinq_user_data(w);

   if (!MEMQ(ei,all_edt_data)) ei = NULL;

   return ei;
};






EDT_ID
EDT_info_by_menu(mw)
   ASH_WINDOW mw;
{
   EDT_ID ei;
   Sequence l;

   forin (ei,EDT_ID,l,all_edt_data) {
      if (ei->menuwin == mw) break;
    };

   return ei;
};






EDT_ID
EDT_info_by_editor(fid)
   FILE_ID fid;
{
   EDT_ID ei;
   Sequence l;

   forin (ei,EDT_ID,l,all_edt_data) {
      if (ei->edit_file == fid) break;
    };

   return ei;
};






/************************************************************************/
/*									*/
/*	EDT_open_file -- open a file for editing			*/
/*									*/
/************************************************************************/


Integer
EDT_open_file(ei,nm)
   EDT_ID ei;
   String nm;
{
   String menu[10];
   Integer ct,fg;
   Character fnm[64];

   close_all_views(ei);

   if (nm == NULL && ei->data->temporary) nm = "/tmp/EDITTEMP";

   if (nm == NULL) {
      ct = 0;
      if (ei->data->readonly) menu[ct++] = "%CFile To View (Readonly)\n";
      else menu[ct++] = "%CFile To Edit\n";
      menu[ct++] = "File Name: %0.32t\n";
      menu[ct++] = "   %a%M   %c";
      menu[ct] = 0;
      fnm[0] = 0;
      if (!STEMdialog(ei->window,menu,fnm)) return -1;
      nm = fnm;
    };

   fg = 0;
   if (ei->data->readonly) fg |= FILE_OPEN_READONLY;
   else fg |= FILE_OPEN_LOCK;
   if (ei->data->create) fg |= FILE_OPEN_CREATE;
   if (ei->data->temporary) fg |= FILE_OPEN_TEMPORARY;
   if (ei->data->input_filter != NULL) fg |= FILE_OPEN_HOLD;

   if (ei->data->file_rtn != NULL) {
      (*ei->data->file_rtn)(ei->udata,EDT_FILE_PREOPEN,nm);
    };

   ei->edit_file = FILEopen(nm,fg);
   if (ei->edit_file == FILE_ID_NONE) {
      ei->edit_file = NULL;
      return 0;
    };

   if (ei->data->output_filter != NULL)
      FILEfilter(ei->edit_file,FILE_FILTER_OUTPUT,output_filter);

   if (ei->data->input_filter != NULL) {
      FILEfilter(ei->edit_file,FILE_FILTER_INPUT,input_filter);
    };

   if (ei->previous_file != NULL) SFREE(ei->previous_file);
   if (ei->filename != NULL) ei->previous_file = (Universal) SALLOC(ei->filename);
   else ei->previous_file = NULL;
   ei->previous_line = ei->status_line;

   if (ei->filename != NULL) SFREE(ei->filename);
   ei->filename = SALLOC(nm);
   ei->status_line = -1;

   ASHset_window_name(ei->window,nm);

   setup_mode(ei);

   FILEregister_message(ei->edit_file,check_edit);

   if (ei->data->file_rtn != NULL) {
      (*ei->data->file_rtn)(ei->udata,EDT_FILE_OPEN,ei->filename);
    };

   EDT_ctbl_rebind(ei);

   setup_initial_view(ei);

   if (ei->readonly_mode) EDTenable(ei,FALSE);

   check_bounds(ei,TRUE);

   return 1;
};




/************************************************************************/
/*									*/
/*	EDT_close_file -- close current file				*/
/*									*/
/************************************************************************/


Boolean
EDT_close_file(ei,save,abort,cont,noq)
   EDT_ID ei;
   Boolean save;
   Boolean abort;
   Boolean cont;
   Boolean noq;
{
   register Boolean fg;
   register Boolean savefg;
   Integer ct,val;
   String menu[10];
   register EDT_MARK mk,nmk;
   Integer sts,i;

   if (ei->edit_file == NULL) return TRUE;

   if (!LPROTECT(ei)) return TRUE;

   if (ei->data->file_rtn != NULL) {
      if (!save) sts = (*ei->data->file_rtn)(ei->udata,EDT_FILE_INQ_ABORT,ei->filename);
      else sts = (*ei->data->file_rtn)(ei->udata,EDT_FILE_INQ_SAVE,ei->filename);
      if (!sts) return FALSE;
    };

   if (!save) sts = ASHcontrol_msg(ei->window,"EDIT$INQ_ABORT_CLOSE");
   else sts = ASHcontrol_msg(ei->window,"EDIT$INQ_SAVE_CLOSE");
   if (sts == ASH_CONTROL_BAD) return FALSE;

   fg = TRUE;
   if (!ei->force_save && !FILEmodified(ei->edit_file)) savefg = FALSE;
   else if (save) savefg = TRUE;
   else {
      ct = 0;
      menu[ct++] = "%CClose Current File\n";
      menu[ct++] = "%0.1o Save%M%0.0o Ignore Changes";
      menu[ct++] = "\n   %a%M   %c";
      menu[ct] = 0;
      val = (abort ? 0 : 1);
      if (!noq && !STEMdialog(ei->window,menu,&val)) fg = FALSE;
      else savefg = (val ? TRUE : FALSE);
    };

   if (fg && !cont && ei->data->savectl) write_control_file(ei);

   if (fg) fg = FILEclose(ei->edit_file,savefg);

   if (!fg) {
      LUNPROTECT(ei);
      return FALSE;
    };

   ei->edit_file = NULL;
   ei->select.view = NULL;
   ei->select.valid = FALSE;

   if (ei->data->file_rtn != NULL) {
      if (!save) (*ei->data->file_rtn)(ei->udata,EDT_FILE_ABORT,ei->filename);
      else (*ei->data->file_rtn)(ei->udata,EDT_FILE_SAVE,ei->filename);
    };

   if (!savefg) ASHcontrol_msg(ei->window,"EDIT$ABORT_CLOSE");
   else ASHcontrol_msg(ei->window,"EDIT$SAVE_CLOSE");

   close_all_views(ei);

   for (mk = ei->all_marks; mk != NULL; mk = nmk) {
      nmk = mk->next;
      mk->valid = FALSE;
    };

   ei->all_marks = NULL;

   ei->histfirst = 0;
   for (i = 0; i < ei->histct; ++i) ei->history[i] = 0;

   if (!cont) {
      ASHset_user_data(ei->window,NULL);
      ASHremove_control(ei->window,edt_ash_control);
      ASHcontrol_msg(ei->window,"EDIT$REMOVE");
      if (ei->data->file_rtn != NULL) {
	 (*ei->data->file_rtn)(ei->udata,EDT_FILE_REMOVE,ei->filename);
       };
      ei->window = NULL;
    };

   LUNPROTECT(ei);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_inq_current_view -- return current view for menu btns	*/
/*									*/
/************************************************************************/


EDT_VIEW
EDT_inq_current_view(ei)
   EDT_ID ei;
{
   register EDT_VINFO vi;
   register EDT_VIEW ev;
   Integer top,bot;
   Integer line;

   line = FILEinq_currline(ei->edit_file);

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      EDT_view_inq_bounds(vi->view,&top,&bot);
      if (line >= top && line <= bot) break;
    };

   if (vi != NULL) ev = vi->view;
   else if (ei->select.valid) ev = ei->select.view;
   else ev = ei->views->view;

   return ev;
};





/************************************************************************/
/*									*/
/*	EDT_rebind -- rebind keys in all active editors 		*/
/*									*/
/************************************************************************/


void
EDT_rebind()
{
   register Sequence l;
   register EDT_ID ei;

   forin (ei,EDT_ID,l,all_edt_data) {
      EDT_ctbl_rebind(ei);
    };
};





/************************************************************************/
/*									*/
/*	EDT_set_values -- set values for variables in editors		*/
/*									*/
/************************************************************************/


void
EDT_set_values(ev,val)
   EDT_VAR ev;
   Universal val;
{
   register Sequence l;
   register EDT_ID ei;

   forin (ei,EDT_ID,l,all_edt_data) {
      EDT_var_set_value(ei,ev,val);
    };
};





/************************************************************************/
/*									*/
/*	EDT_split_view -- split a given view				*/
/*	EDT_remove_view -- remove a given view				*/
/*	EDT_size_view -- size a given view				*/
/*									*/
/************************************************************************/


void
EDT_split_view(ev)
   EDT_VIEW ev;
{
   register EDT_VINFO vi,nvi;
   register EDT_ID ei;
   Integer lx,by,rx,ty,plx,pby;
   ASH_WINDOW w;

   ei = ev->edit;

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (vi->view == ev) break;
    };
   if (vi == NULL) return;

   if (!LPROTECT(ei)) return;

   ei->force_bounds = TRUE;

   nvi = PALLOC(EDT_VINFO_INFO);

   ASHinq_size(vi->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHmap(vi->window,lx,by,ei->window,&plx,&pby);
   w = ASHcreate(ei->window,plx,pby,lx,by,rx,ty,0,
		    ASH_WINDOW_NOSAVE|ASH_WINDOW_HIT_USE|ASH_WINDOW_INVISIBLE);
   ASHset_control(w,edt_view_control);

   nvi->fraction = vi->fraction/2;
   vi->fraction -= nvi->fraction;
   nvi->window = w;
   nvi->next = vi->next;
   vi->next = nvi;
   nvi->type = 0;
   nvi->ty = ty;
   nvi->by = by;

   nvi->view = EDT_view_create(ei,w,nvi->type);

   resize_views(ei);

   ASHvisible(w,TRUE);

   check_bounds(ei,TRUE);

   LUNPROTECT(ei);
};




void
EDT_remove_view(ev)
   EDT_VIEW ev;
{
   register EDT_VINFO vi;
   register EDT_ID ei;

   ei = ev->edit;

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (vi->view == ev) break;
    };

   if (vi == NULL) return;
   if (vi->next == NULL && ei->views == vi) return;

   if (!LPROTECT(ei)) return;

   ei->force_bounds = TRUE;

   ASHremove(vi->window);

   LUNPROTECT(ei);
};





void
EDT_size_view(ev,sz)
   EDT_VIEW ev;
   Integer sz;
{
   register EDT_VINFO vi;
   register EDT_ID ei;

   ei = ev->edit;

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (vi->view == ev) break;
    };

   if (vi == NULL) return;
   if (vi->next == NULL && ei->views == vi) return;

   if (!LPROTECT(ei)) return;

   vi->fraction = sz * (vi->fraction - 100) / (sz - 100);
   fixup_sizes(ei);

   resize_views(ei);

   ei->force_bounds = TRUE;

   LUNPROTECT(ei);
};





/************************************************************************/
/*									*/
/*	EDT_confirm -- issue confirmation of a command			*/
/*									*/
/************************************************************************/


void
EDT_confirm(ei,msg)
   EDT_ID ei;
   String msg;
{
   Integer lx,by,rx,ty;
   Integer x,y;

   if (ei->stswin == NULL) {
      EDassoc(ei->edit_file,ei->window);
      EDfile_display(ei->edit_file,msg);
    }
   else {
      ASHinq_size(ei->stswin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      ASHinq_text_offset(ei->stswin,msg,&x,&y);
      ASHclear_box(ei->stswin,lx,by,rx,ty);
      x += lx;
      if (by < ty) y = by+1+y;
      else y = by-1-y;
      ASHtext(ei->stswin,x,y,msg);
      ei->status_line = -1;
    };
};





/************************************************************************/
/*									*/
/*	edt_ash_control -- handle control messages for window		*/
/*									*/
/************************************************************************/


static int
edt_ash_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register EDT_ID ei;

   ei = EDT_info_by_window(w);
   if (ei == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$RESIZE")) {
      resize_views(ei);
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      edt_free_editor(ei);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	edt_view_control -- handle control messages for view windows	*/
/*									*/
/************************************************************************/


static int
edt_view_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register EDT_ID ei;
   register EDT_VINFO vi;

   ei = EDT_info_by_window(ASHinq_parent(w));
   if (ei == NULL) return ASH_CONTROL_REJECT;
   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (vi->window == w) break;
    };
   if (vi == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      remove_view(ei,vi);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	edt_window_refresh -- refresh underlying window 		*/
/*									*/
/************************************************************************/


static void
edt_window_refresh(w)
   ASH_WINDOW w;
{
   Integer rlx,rby,rrx,rty;
   EDT_ID ei;

   ASHinq_size(w,ASH_SIZE_WINDOW,&rlx,&rby,&rrx,&rty);

   ASHrectangle(w,rlx,rby,rrx,rty);

   ei = EDT_info_by_window(w);
   if (ei != NULL) {
      ei->status_line = -1;
      status_display(ei);
    };
};





/************************************************************************/
/*									*/
/*	edt_status_refresh -- refresh status window			*/
/*									*/
/************************************************************************/


static void
edt_status_refresh(w)
   ASH_WINDOW w;
{
   EDT_ID ei;

   ei = EDT_info_by_window(w);
   if (ei != NULL) {
      ei->status_line = -1;
      status_display(ei);
    };
};





/************************************************************************/
/*									*/
/*	edt_free_editor -- free edit window				*/
/*									*/
/************************************************************************/


static void
edt_free_editor(ei)
   EDT_ID ei;
{
   if (ei->window != NULL) {
      ASHset_user_data(ei->window,NULL);
      ei->window = NULL;
    };

   if (ei->wait) {
      ei->lockwin = NULL;
      return;
    };

   PROTECT;
   all_edt_data = REMOB(ei,all_edt_data);
   UNPROTECT;

   close_all_views(ei);

   SFREE(ei->filename);
   ei->filename = NULL;
   ei->status_line = -1;

   free(ei);
};





/************************************************************************/
/*									*/
/*	check_edit -- handle editor messages and take approp actions	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
check_edit(fi,msg,from,to,ch,insmd)
   FILE_ID fi;
   FILE_MSG msg;
   FILE_POS *from,*to;
   int ch;
   int insmd;
{
   register Integer i0,i1;
   FILE_POS p0;
   register EDT_MARK fm;
   register EDT_ID ei;
   register Sequence l;
   EDT_VINFO vi;

   if (msg == FILE_MSG_DELETE) {
      forin (ei,EDT_ID,l,all_edt_data) {
	 if (ei->edit_file == fi) break;
       };
      if (ei != NULL) ei->force_bounds = TRUE;
      i0 = FILEinq_line(fi,from);
      i1 = FILEinq_line(fi,to);
      if (i0 != i1) {
	 if (ei != NULL) {
	    for (fm = ei->all_marks; fm != NULL; fm = fm->next) {
	       if (fm->valid) {
		  FILEinq_savepos(fi,fm->file_pos,&p0);
		  if (FILEpos_compare(fi,&p0,from) >= 0 &&
			 FILEpos_compare(fi,&p0,to) <= 0)
		     fm->valid = FALSE;
		};
	     };
	  };
       };
    }
   else if (msg == FILE_MSG_INSERT || msg == FILE_MSG_CLOSE) {
      forin (ei,EDT_ID,l,all_edt_data) {
	 if (ei->edit_file == fi) break;
       };
      if (ei != NULL) {
	 if (msg == FILE_MSG_INSERT) ei->force_bounds = TRUE;
	 else if (msg == FILE_MSG_CLOSE) {
	    for (vi = ei->views; vi != NULL; vi = vi->next) {
	       if (vi->view != NULL) vi->view->edit_text = NULL;
	     };
	    ei->force_bounds = FALSE;
	  };
       };
    }
   else if (msg == FILE_MSG_COMMAND) {
      forin (ei,EDT_ID,l,all_edt_data) {
	 if (ei->edit_file == fi) break;
       };
      if (ei != NULL && (ei->window == NULL || !ASHinq_valid_window(ei->window))) ei= NULL;
      if (ei != NULL) {
	 check_bounds(ei,ei->force_bounds);
	 if (ei->shell) check_shell_input(ei);
	 ei->force_bounds = FALSE;
       };
    };
};





/************************************************************************/
/*									*/
/*	setup_mode -- define major mode for new editor			*/
/*									*/
/************************************************************************/


static void
setup_mode(ei)
   EDT_ID ei;
{
   register String s;
   register Integer i,ln;
   Character buf[256];

   s = EDT_ctbl_mode_select(ei);
   if (s == NULL) {
      s = rindex(ei->filename,'.');
      if (s != NULL && index(s,'/') != NULL) s = NULL;
      if (s == NULL) s = "";
      sprintf(buf,"%s.",s);
      ln = strlen(buf);
      for (i = 0; mode_table[i].when != NULL; ++i) {
	 for (s = mode_table[i].when; *s != 0; ++s) {
	    if (strncmp(s,buf,ln) == 0) break;
	  };
	 if (*s != 0) break;
       };
      s = mode_table[i].mode;
    };

   if (ei->data->mode != NULL) sprintf(buf,"%s,%s",ei->data->mode,s);
   else strcpy(buf,s);

   if (ei->data->readonly) strcat(buf,",READONLY");
   if (ei->shell) strcat(buf,",SCRIPT");

   s = ASHinq_window_resource(ei->window,"edt_mode");
   if (s != NULL && *s != 0) {
      strcat(buf,",");
      strcat(buf,s);
    };

   s = ASHinq_window_resource(ei->window,"simple");
   if (s != NULL && STRNEQ(s,"off")) strcat(buf,",SIMPLE");

   ei->mode = SALLOC(buf);
};





/************************************************************************/
/*									*/
/*	setup_initial_view -- handle setup of first edit view		*/
/*									*/
/************************************************************************/


static void
setup_initial_view(ei)
   EDT_ID ei;
{
   register ASH_WINDOW w;
   Integer lx,by,rx,ty;
   register EDT_VINFO vi;

   ASHinq_size(ei->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   w = ASHcreate(ei->window,lx,by,lx,by,rx,ty,0,ASH_WINDOW_NOSAVE|ASH_WINDOW_HIT_USE);
   ASHset_control(w,edt_view_control);

   vi = PALLOC(EDT_VINFO_INFO);
   vi->fraction = 100;
   vi->window = w;
   vi->next = NULL;
   vi->type = 0;
   vi->ty = ty;
   vi->by = by;

   vi->view = EDT_view_create(ei,w,vi->type);

   if (vi->view != NULL) ei->views = vi;
};





/************************************************************************/
/*									*/
/*	fixup_sizes -- fixup view sizes after some change		*/
/*									*/
/************************************************************************/


static void
fixup_sizes(ei)
   EDT_ID ei;
{
   register EDT_VINFO vi;
   register Integer tot,ntot,rnd;

   tot = 0;
   for (vi = ei->views; vi != NULL; vi = vi->next) {
      tot += vi->fraction;
    };

   if (tot == 100 || tot == 0) return;

   ntot = 0;
   rnd = tot/2;
   for (vi = ei->views; vi->next != NULL; vi = vi->next) {
      vi->fraction *= 100;
      vi->fraction += rnd;
      vi->fraction /= tot;
      ntot += vi->fraction;
    };

   vi->fraction = 100 - ntot;
};





/************************************************************************/
/*									*/
/*	resize_views -- resize views when window is resized		*/
/*									*/
/************************************************************************/


static void
resize_views(ei)
   EDT_ID ei;
{
   Integer rlx,rby,rrx,rty;
   register EDT_VINFO vi;
   Integer by,ty;

   ASHinq_size(ei->window,ASH_SIZE_WINDOW,&rlx,&rby,&rrx,&rty);

   ty = rty;

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      if (vi->next == NULL) by = rby;
      else by = ty + (rby-rty)*vi->fraction/100;
      ASHresize(vi->window,rlx,by,rlx,by,rrx,ty);
      vi->ty = ty;
      vi->by = by;
      if (rty < rby) ty = by + SEPARATION_SPACE;
      else ty = by - SEPARATION_SPACE;
    };
};





/************************************************************************/
/*									*/
/*	close_all_views -- close all views of an edit_id		*/
/*									*/
/************************************************************************/


static void
close_all_views(ei)
   EDT_ID ei;
{
   register EDT_VINFO vi;

   vi = NULL;
   while (ei->views != NULL) {
      if (vi == ei->views) break;
      vi = ei->views;
      if (ASHinq_valid_window(vi->window)) ASHremove(vi->window);
    };
};





/************************************************************************/
/*									*/
/*	remove_view -- remove a single view when its window goes away	*/
/*									*/
/************************************************************************/


static void
remove_view(ei,vi)
   EDT_ID ei;
   EDT_VINFO vi;
{
   register EDT_VINFO vx;

   vi->window = NULL;

   if (!LPROTECT(ei)) return;
   if (ei->views == vi) ei->views = vi->next;
   else {
      for (vx = ei->views; vx != NULL && vx->next != vi; vx = vx->next);
      if (vx != NULL) vx->next = vi->next;
    };
   if (ei->select.valid && ei->select.view == vi->view) {
      ei->select.view = NULL;
      ei->select.valid = FALSE;
    };
   LUNPROTECT(ei);

   free(vi);

   if (ei->edit_file != NULL) {
      fixup_sizes(ei);
      resize_views(ei);
      check_bounds(ei,TRUE);
    };
};





/************************************************************************/
/*									*/
/*	check_bounds -- handle bounds checking after command		*/
/*									*/
/************************************************************************/


static void
check_bounds(ei,upfg)
   EDT_ID ei;
   Boolean upfg;
{
   Integer last;
   FILE_POS fp;
   EDT_VINFO vi;
   Boolean fg;

   FILEinq_position(ei->edit_file,FILE_MOVE_END,0,&fp);
   last = FILEinq_line(ei->edit_file,&fp);

   fg = upfg;
   for (vi = ei->views; vi != NULL; vi = vi->next) {
      fg |= EDT_view_bounds(vi->view,last);
    };

   if (fg && ei->data->bounds_rtn != NULL) {
      (*ei->data->bounds_rtn)(ei->udata);
    };

   status_display(ei);
};




/************************************************************************/
/*									*/
/*	view_changer -- handle user request to move view boundaries	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
view_changer(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   register EDT_ID ei;
   register EDT_VINFO vi,va;
   Integer lx,by,rx,ty,dy;
   register Integer ny,p;
   register Integer i;
   ASH_WINDOW w;
   ASH_CURSOR cur;
   Universal ud;

   ei = (EDT_ID) RIPinq_data(rgn);
   if (ei == NULL || ei->window == NULL) return FALSE;
   if ((btn & RIP_BTN_LEFT) == 0) return FALSE;
   if ((btn & RIP_BTN_DOWN) == 0) return FALSE;

   ASHmap(ASHinq_top_window(ei->window),x,y,ei->window,&x,&y);
   ASHinq_size(ei->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dy = (ty < by ? 1 : -1);

   for (vi = ei->views; vi != NULL; vi = vi->next) {
      i = (y-vi->by)*dy;
      if (i >= 0 && i <= SEPARATION_SPACE) break;
    };
   if (vi == NULL || vi->next == NULL) return FALSE;

   if (!LPROTECT(ei)) return FALSE;

   ASHwindow_draw_thru(ei->window,TRUE);
   ASHdraw_hilite(ei->window,TRUE);
   w = RIPgrab_window(ei->window);
   BIOset_cursor(ei->window,TRUE);
   cur = BIOset_cursor_standard(ei->window,ASH_CURSOR_SMALL_XHAIRS);
   ud = ASHinq_user_data(ei->window);
   ASHset_user_data(ei->window,-1);
   RIPtrack(view_changer_track,1,ei->window,FALSE);
   ny = ASHinq_user_data(ei->window);
   ASHset_user_data(ei->window,ud);
   BIOset_cursor_pattern(ei->window,cur);
   RIPgrab_window(w);
   ASHdraw_hilite(ei->window,FALSE);
   ASHwindow_draw_thru(ei->window,FALSE);

   if (ny < 0) {
      LUNPROTECT(ei);
      return FALSE;
    };

   if ((ny-vi->by)*dy < 0) {
      va = vi->next;
      p = abs(ny-va->by);
    }
   else {
      va = vi;
      p = abs(va->ty-ny);
    };

   p = p*100/abs(by-ty);
   if (p >= 95) p = 95;

   va->fraction = p * (va->fraction - 100) / (p - 100);
   fixup_sizes(ei);

   resize_views(ei);

   LUNPROTECT(ei);

   return TRUE;
};




/*ARGSUSED*/

static int
view_changer_track(x,y,ct,act,mx,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer mx;
   ASH_WINDOW w;
{
   register Integer ny;
   Integer lx,by,rx,ty;

   ny = (Integer) ASHinq_user_data(w);
   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   switch (act) {
      case RIP_TRANS_NONE :
	 ny = y;
	 ASHline(w,lx,y,rx,y);
	 break;

      case RIP_TRANS_MOVE :
	 if (y != ny) {
	    ASHline(w,lx,ny,rx,ny);
	    ny = y;
	    ASHline(w,lx,y,rx,y);
	  };
	 break;

      case RIP_TRANS_UP :
	 ASHline(w,lx,ny,rx,ny);
	 ny = y;
	 break;

      default :
	 if (ny >= 0) ASHline(w,lx,ny,rx,ny);
	 ny = -1;
	 return FALSE;
    };

   ASHset_user_data(w,ny);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_shell_input -- check for input to shell			*/
/*									*/
/************************************************************************/


static void
check_shell_input(ei)
   EDT_ID ei;
{
   FILE_POS pos,npos,spos;
   FILE_BUF buf;
   Integer shln,cln,cch;
   String str;

   if (ei->hold_mode) return;

   cln = FILEinq_currline(ei->edit_file);
   cch = FILEinq_currchar(ei->edit_file);
   if (ei->shell_pos == NULL) FILEinq_position(ei->edit_file,FILE_MOVE_START,0,&spos);
   else {
      FILEinq_savepos(ei->edit_file,ei->shell_pos,&spos);
      EDT_next_char(ei,&spos,&spos);
    };

   if (ei->raw_mode) {
      if (FILEpos_compare(ei->edit_file,&spos,FILE_POS_END) >= 0)
	 return;
    }
   else {
      shln = FILEinq_line(ei->edit_file,&spos);
      if (shln >= cln) return;
    };

   if (ei->raw_mode) {
      FILEpos_copy(ei->edit_file,FILE_POS_END,&pos);
    }
   else {
      FILEnew_position(ei->edit_file,cln,1,&pos);
    };

   PROTECT;

   buf = FILEinq_buffer("BWEDIT$_SHELL");
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&spos,&npos);

   if (ei->data->echo) {
      FILEpick(ei->edit_file,buf,FILE_MOVE_ABS,&pos);
      FILEset_position(ei->edit_file,FILEinq_line(ei->edit_file,&pos),
			  FILEinq_char(ei->edit_file,&pos),
			  &spos);
      get_cur_shellpos(ei);
    }
   else {
      FILEdelete(ei->edit_file,buf,FILE_MOVE_ABS,&pos);
    };

   str = FILEbuffer_contents(buf);
   FILEset_position(ei->edit_file,cln,cch,&npos);

   UNPROTECT;

   (*ei->data->input_rtn)(ei->udata,0,str);

   free(str);
};





/************************************************************************/
/*									*/
/*	status_display -- handle status display for editor		*/
/*									*/
/************************************************************************/


static void
status_display(ei)
   EDT_ID ei;
{
   Integer x,y;
   Integer cx,cy;
   Integer lx,by,rx,ty;
   Character buf[32];
   Integer l,el;
   Boolean fg;

   if (ei->stswin == NULL || ei->edit_file == NULL) return;
   if (!ASHinq_valid_window(ei->stswin)) return;

   ASHinq_size(ei->stswin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHinq_text_next(ei->stswin,"W",&cx,&cy);
   ASHinq_text_offset(ei->stswin,"pXW_",&x,&y);

   if (ei->status_line <= 0) {
      ASHclear_box(ei->stswin,lx,by,rx,ty);
      ei->status = 0;
      ei->status_eof = 0;
    };

   x += lx;
   if (by < ty) y = by+1+y;
   else y = by-1-y;

   x = output_flag(ei,x,y,cx,ei->readonly_mode,EDT_STATUS_ENABLE,"ReadOnly",by,ty);
   x = output_flag(ei,x,y,cx,ei->open_mode,EDT_STATUS_OPEN,"Open",by,ty);
   x = output_flag(ei,x,y,cx,ei->insert_mode,EDT_STATUS_INSERT,"Insert",by,ty);
   x = output_flag(ei,x,y,cx,ei->auto_indent,EDT_STATUS_INDENT,"Indent",by,ty);
   x = output_flag(ei,x,y,cx,ei->hold_mode,EDT_STATUS_HOLD,"Hold",by,ty);
   x = output_flag(ei,x,y,cx,ei->get_args,EDT_STATUS_ARGS,"Args",by,ty);
   fg = (ei->macro != NULL && ei->macro->view != NULL);
   x = output_flag(ei,x,y,cx,fg,EDT_STATUS_MACRO,"Macro",by,ty);

   if (ei->filename != NULL && ei->status_line <= 0) {
      x += 2*cx;
      ASHtext(ei->stswin,x,y,ei->filename);
    };

   l = FILEinq_currline(ei->edit_file);
   el = FILEinq_line(ei->edit_file,FILE_POS_END);

   if (l != ei->status_line || el != ei->status_eof) {
      sprintf(buf,"%d/%d",l,el);
      ei->status_line = l;
      ei->status_eof = el;
      x = rx - 12*cx;
      ASHclear_box(ei->stswin,x,by,rx,ty);
      x = rx - strlen(buf)*cx;
      ASHtext(ei->stswin,x,y,buf);
    };
};






/************************************************************************/
/*									*/
/*	output_flag -- output status line flag				*/
/*									*/
/************************************************************************/


static Integer
output_flag(ei,x,y,cx,val,fg,txt,by,ty)
   EDT_ID ei;
   Integer x,y,cx;
   Boolean val;
   Integer fg;
   String txt;
   Integer by,ty;
{
   Integer ln;

   ln = strlen(txt);

   if (val && (ei->status & fg) == 0) {
      ASHtext(ei->stswin,x,y,txt);
      ei->status |= fg;
    }
   else if (!val && (ei->status & fg) != 0) {
      ASHclear_box(ei->stswin,x,by,x+(ln+1)*cx-1,ty);
      ei->status &= ~fg;
    };

   x += (ln+1)*cx;

   return x;
};






/************************************************************************/
/*									*/
/*	input_filter -- handle input filtering				*/
/*	output_filter -- handle output filtering			*/
/*									*/
/************************************************************************/


static String
input_filter(fid,txt)
   FILE_ID fid;
   String txt;
{
   EDT_ID ei;

   ei = EDT_info_by_editor(fid);

   if (ei != NULL && ei->input_filter != NULL) {
      txt = (String) (*(ei->input_filter))(ei->udata,txt);
    };

   return txt;
};





static String
output_filter(fid,txt)
   FILE_ID fid;
   String txt;
{
   EDT_ID ei;

   ei = EDT_info_by_editor(fid);

   if (ei != NULL && ei->output_filter != NULL) {
      txt = (String) (*(ei->output_filter))(ei->udata,txt);
    };

   return txt;
};





/************************************************************************/
/*									*/
/*	read_control_file -- read control information at startup	*/
/*	write_control_file -- write control information at end		*/
/*									*/
/************************************************************************/


static void
read_control_file(buf,line,col)
   String buf;
   Integer * line;
   Integer * col;
{
   FILE * inf;
   Integer i;

   inf = fopen(EDT_CONTROL_FILE,"r");

   if (inf != NULL) {
      i = fscanf(inf,EDT_CONTROL_TEXT,buf,line,col);
      fclose(inf);
    };

   if (inf == NULL || i != 3) {
      buf[0] = 0;
      *line = 1;
      *col = 1;
    };
};





static void
write_control_file(ei)
   EDT_ID ei;
{
   FILE * otf;

   if (ei->edit_file == NULL || !ei->data->savectl || ei->filename == NULL) return;

   otf = fopen(EDT_CONTROL_FILE,"w");
   if (otf == NULL) return;

   fprintf(otf,EDT_CONTROL_TEXT,ei->filename,FILEinq_currline(ei->edit_file),
	      FILEinq_currchar(ei->edit_file));
   fclose(otf);
};





/************************************************************************/
/*									*/
/*	goto_shellpos - move to shell position				*/
/*	get_end_shellpos -- set shell position to eof			*/
/*									*/
/************************************************************************/


static void
goto_shellpos(ei)
   EDT_ID ei;
{
   FILE_POS spos,npos;

   if (ei->shell_pos == NULL) {
      FILEmove(ei->edit_file,FILE_MOVE_START,0,&npos);
    }
   else {
      FILEinq_savepos(ei->edit_file,ei->shell_pos,&spos);
      EDT_next_char(ei,&spos,&spos);
      FILEmove(ei->edit_file,FILE_MOVE_ABS,&spos,&npos);
    };
};






static void
get_end_shellpos(ei)
   EDT_ID ei;
{
   FILE_POS pos,spos,npos;

   if (FILEinq_line(ei->edit_file,FILE_POS_END) == 1 &&
	  FILEinq_char(ei->edit_file,FILE_POS_END) == 1)
      ei->shell_pos = NULL;
   else {
      FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&pos);
      EDT_last_char(ei,FILE_POS_END,&npos);
      FILEmove(ei->edit_file,FILE_MOVE_ABS,&npos,&spos);
      if (ei->shell_pos != NULL) FILEdelete_pos(ei->edit_file,ei->shell_pos);
      ei->shell_pos = FILEsave_pos(ei->edit_file);
      FILEmove(ei->edit_file,FILE_MOVE_ABS,&pos,&npos);
    };
};





static void
get_cur_shellpos(ei)
   EDT_ID ei;
{
   FILE_POS pos,spos,npos;

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&pos);
   EDT_last_char(ei,&pos,&npos);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&npos,&spos);
   if (ei->shell_pos != NULL) FILEdelete_pos(ei->edit_file,ei->shell_pos);
   ei->shell_pos = FILEsave_pos(ei->edit_file);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&pos,&npos);
};






/* end of edtwin.c */


