/************************************************************************/
/*									*/
/*		edtview.c						*/
/*									*/
/*	View manipulation routines for editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "edt_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define INPUT_MASK	BIO_MASK_ASCII | BIO_MASK_META | BIO_MASK_FKEY | \
			BIO_MASK_SHIFT_FKEY | BIO_MASK_CTRL_FKEY | \
			BIO_MASK_META_FKEY | BIO_MASK_MOVE_DOWN | \
			BIO_MASK_BUTTON | BIO_MASK_BUTTON_UP | \
			BIO_MASK_TAP_BUTTON | BIO_MASK_SHIFT_BUTTON | \
			BIO_MASK_REDRAW | BIO_MASK_VISIBLE | BIO_MASK_VIEW

#define SCROLLSIZE	16





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	view_win_control();
static	void		view_win_refresh();
static	void		resize_view_win();
static	void		free_view_win();
static	void		find_coords();
static	int		edt_user_scroll();
static	void		edt_scroll_set();




/************************************************************************/
/*									*/
/*	EDT_view_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_view_init()
{
};





/************************************************************************/
/*									*/
/*	EDT_view_create -- create a new view				*/
/*									*/
/************************************************************************/


EDT_VIEW
EDT_view_create(ei,w,typ)
   EDT_ID ei;
   ASH_WINDOW w;
   EDT_VIEW_TYPE typ;
{
   register EDT_VIEW ev;
   register RIP_REGION rr;
   register Integer mask;
   Integer lx,by,rx,ty,x;
   FILE_POS pos;

   if (w == NULL) return NULL;

   if (!ASHlock(w)) return NULL;

   if (ei->shell) typ |= EDT_VIEW_TYPE_SHELL;

   ev = PALLOC(EDT_VIEW_INFO);
   ev->edit = ei;
   ev->type = typ;
   ev->window = w;
   ev->top_line = -1;
   ev->bottom_line = -1;
   ev->last_line = -1;

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (ei->data->scroll) {
      if (lx < rx) x = rx-SCROLLSIZE;
      else x = rx + SCROLLSIZE;
      ev->editwin = ASHcreate(w,lx,by,lx,by,x-2,ty,0,ASH_WINDOW_INVISIBLE|
				 ASH_WINDOW_NOSAVE|ASH_WINDOW_HIT_USE);
      ASHline(w,x-1,by,x-1,ty);
      ev->scrollwin = ASHcreate(w,x,by,x,by,rx,ty,0,ASH_WINDOW_INVISIBLE|
				   ASH_WINDOW_NOSAVE|ASH_WINDOW_HIT_USE);
    }
   else {
      ev->editwin = ASHcreate(w,lx,by,lx,by,rx,ty,0,ASH_WINDOW_INVISIBLE|
				 ASH_WINDOW_NOSAVE|ASH_WINDOW_HIT_USE);
      ev->scrollwin = NULL;
    };

   if (ev->editwin == NULL) {
      ASHunlock(w);
      return NULL;
    };

   ASHset_control(w,view_win_control);
   ASHset_region_refresh(w,view_win_refresh);
   ASHset_user_data(w,ev);

   rr = RIPdefine_region(ev->editwin,0,0,0,0,RIP_ALL_CHARS,
			    RIP_BTN_ANY_EXTRA|RIP_BTN_TAP,
			    EDT_button_handler,
			    ASH_SENSE_NO_CHANGE);
   RIPset_data(rr,ev);
   ev->region = rr;

   mask = INPUT_MASK;
   BIOnew_input_window(ev->editwin);
   BIOset_mask(ev->editwin,mask,NULL);
   BIOset_mapping(ev->editwin,FKEY(BPIO_KEY_HELP),BIO_MAP_DEFAULT,NULL,NULL);
   BIOset_cursor_standard(ev->editwin,ASH_CURSOR_BETWEEN_SMALL_L);

   HELPmouse_register(ev->editwin,"Character Select","Word Select",
			 "Extend Selection/Line Select");

   if (!ei->shell || ei->raw_mode) BIOset_cbreak(ev->editwin,TRUE);

   ASHfont(ev->editwin,ei->font);
   ev->edit_text = TEXTopen(ei->edit_file,ev->editwin);

   if (ev->scrollwin != NULL) {
      STEMscroll(ev->scrollwin,edt_user_scroll,ev);
    };

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&pos);

   TEXTcursor_enable(ev->edit_text,0);

   TEXTcursor(ev->edit_text,TEXT_CURSOR_CARET,&pos);

   if (ei->shell) TEXTset_shell_mode(ev->edit_text);

   find_coords(ev);

   ASHvisible(ev->editwin,TRUE);
   if (ev->scrollwin != NULL) ASHvisible(ev->scrollwin,TRUE);

   ASHunlock(w);

   return ev;
};





/************************************************************************/
/*									*/
/*	EDT_view_line_pos -- return position of line in view		*/
/*	EDT_view_correlate_line -- find line in view			*/
/*	EDT_view_inq_bounds -- return bounds for given view		*/
/*									*/
/************************************************************************/


Boolean
EDT_view_line_pos(ev,line,bp,tp)
   EDT_VIEW ev;
   Integer line;
   Integer * bp;
   Integer * tp;
{
   register Integer top,bot;
   register Integer y;

   if (VIEW_TEXT(ev) == NULL) return FALSE;

   top = TEXTinq_topline(VIEW_TEXT(ev));
   bot = TEXTinq_bottomline(VIEW_TEXT(ev));

   if (line < top || line > bot) return FALSE;
   else {
      y = ev->first_line + (line-top)*ev->edit->line_size;
      ASHmap(ev->editwin,0,y,ev->window,NULL,bp);
      ASHmap(ev->editwin,0,y-ev->edit->line_size+1,ev->window,NULL,tp);
    };

   return TRUE;
};





Integer
EDT_view_correlate_line(ev,y)
   EDT_VIEW ev;
   Integer y;
{
   FILE_POS pos;
   Integer ln;

   if (VIEW_TEXT(ev) == NULL) return 0;

   ASHmap(ev->window,0,y,ev->editwin,NULL,&y);

   TEXTcorrelate(VIEW_TEXT(ev),10,y,TRUE,&pos);
   ln = FILEinq_line(VIEW_FILE(ev),&pos);

   return ln;
};





Integer
EDT_view_inq_bounds(ev,topp,botp)
   EDT_VIEW ev;
   Integer * topp;
   Integer * botp;
{
   if (VIEW_TEXT(ev) == NULL) return 0;

   *topp = TEXTinq_topline(VIEW_TEXT(ev));
   *botp = TEXTinq_bottomline(VIEW_TEXT(ev));

   return TEXTinq_rightchar(VIEW_TEXT(ev)) - TEXTinq_leftchar(VIEW_TEXT(ev));
};





/************************************************************************/
/*									*/
/*	EDT_view_bounds -- handle bounds checking after command 	*/
/*									*/
/************************************************************************/


Boolean
EDT_view_bounds(ev,last)
   EDT_VIEW ev;
   Integer last;
{
   register Integer top,bot;
   register Boolean fg;

   if (VIEW_TEXT(ev) == NULL) return FALSE;

   top = TEXTinq_topline(VIEW_TEXT(ev));
   bot = TEXTinq_bottomline(VIEW_TEXT(ev));

   if (top != ev->top_line || bot != ev->bottom_line || last != ev->last_line) {
      ev->top_line = top;
      ev->bottom_line = bot;
      ev->last_line = last;
      edt_scroll_set(ev);
      fg = TRUE;
    }
   else fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	view_win_control -- handle control messages on view window	*/
/*	view_win_refresh -- handle refresh for view window		*/
/*									*/
/************************************************************************/


static Integer
view_win_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register EDT_ID ei;
   register EDT_VIEW ev;
   register Integer rs;

   ei = EDT_info_by_window(ASHinq_parent(w));
   ev = (EDT_VIEW) ASHinq_user_data(w);
   if (ei == NULL || ev == NULL) return ASH_CONTROL_REJECT;

   rs = ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$RESIZE")) {
      if (LPROTECT(ei)) {
	 resize_view_win(ev);
	 LUNPROTECT(ei);
       };
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      if (LPROTECT(ei)) {
	 if (ev->edit_text != NULL) TEXTclose(ev->edit_text);
	 ev->edit_text = NULL;
	 ASHset_user_data(ev->window,NULL);
	 free_view_win(ev);
	 LUNPROTECT(ei);
       };
    };

   return rs;
};






static void
view_win_refresh(w)
   ASH_WINDOW w;
{
   Integer lx,by,rx,ty;
   Integer x;
   EDT_ID ei;
   EDT_VIEW ev;

   ei = EDT_info_by_window(ASHinq_parent(w));
   ev = (EDT_VIEW) ASHinq_user_data(w);
   if (ei == NULL || ev == NULL) return;

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (ei->data->scroll) {
      if (lx < rx) x = rx-SCROLLSIZE;
      else x = rx + SCROLLSIZE;
      ASHline(w,x-1,by,x-1,ty);
    };
};






/************************************************************************/
/*									*/
/*	resize_view_win -- resize viewer				*/
/*									*/
/************************************************************************/


static void
resize_view_win(ev)
   EDT_VIEW ev;
{
   Integer lx,by,rx,ty,x;

   ASHinq_size(ev->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (ev->scrollwin == NULL) {
      ASHresize(ev->editwin,lx,by,lx,by,rx,ty);
    }
   else {
      if (lx < rx) x = rx-SCROLLSIZE;
      else x = rx + SCROLLSIZE;
      ASHresize(ev->editwin,lx,by,lx,by,x-2,ty);
      ASHline(ev->window,x-1,by,x-1,ty);
      ASHresize(ev->scrollwin,x,by,x,by,rx,ty);
    };

   TEXTresize(VIEW_TEXT(ev));

   find_coords(ev);

   edt_scroll_set(ev);
};




/************************************************************************/
/*									*/
/*	free_view_win -- free viewer when window is removed		*/
/*									*/
/************************************************************************/


static void
free_view_win(ev)
   EDT_VIEW ev;
{
   if (VIEW_TEXT(ev) != NULL) TEXTclose(VIEW_TEXT(ev));
   ev->edit_text = NULL;
   ASHset_user_data(ev->window,NULL);

   free(ev);
};





/************************************************************************/
/*									*/
/*	find_coords -- find first line and line size coords		*/
/*									*/
/************************************************************************/


static void
find_coords(ev)
   EDT_VIEW ev;
{
   FILE_POS pos;
   register Integer i,ln0,ln1;
   Integer lx,by,rx,ty;

   ASHinq_size(ev->editwin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   TEXTcorrelate(VIEW_TEXT(ev),10,ty,TRUE,&pos);
   ln0 = FILEinq_line(VIEW_FILE(ev),&pos);

   for (i = ty+1; i < by-1 ; ++i) {
      TEXTcorrelate(VIEW_TEXT(ev),10,i,TRUE,&pos);
      ln1 = FILEinq_line(VIEW_FILE(ev),&pos);
      if (ln1 != ln0) break;
    };
   ev->first_line = i-1;

   if (i < by-1) {
      ln0 = ln1;
      for (++i; i < by ; ++i) {
	 TEXTcorrelate(VIEW_TEXT(ev),10,i,TRUE,&pos);
	 ln1 = FILEinq_line(VIEW_FILE(ev),&pos);
	 if (ln1 != ln0) break;
       };
      ev->edit->line_size = i - ev->first_line - 1;
    }
   else {
      ev->edit->line_size = i - 2;
    };
};





/************************************************************************/
/*									*/
/*	edt_user_scroll -- handle scrolling by user			*/
/*									*/
/************************************************************************/


static int
edt_user_scroll(ev,dir,val)
   EDT_VIEW ev;
   Integer dir;
   Integer val;
{
   register Integer ln;

   if (VIEW_TEXT(ev) == NULL) return FALSE;

   ln = TEXTinq_topline(VIEW_TEXT(ev));

   if (dir == 0) ln = val;
   else if (dir < 0) ln -= val;
   else ln += val;
   if (ln < 1) ln = 1;

   EDT_scroll_to(ev,ln);

   return TRUE;
};




/************************************************************************/
/*									*/
/*	edt_scroll_set -- set scroll bounds				*/
/*									*/
/************************************************************************/


static void
edt_scroll_set(ev)
   EDT_VIEW ev;
{
   register Integer t,b,f,l;
   register EDT_ID ei;
   FILE_POS pos;

   if (ev->scrollwin == NULL) return;

   ei = ev->edit;

   if (VIEW_TEXT(ev) == NULL) {
      t = f = 0;
      b = l = 1;
    }
   else {
      if (LPROTECT(ev->edit)) {
	 FILEinq_position(ei->edit_file,FILE_MOVE_START,0,&pos);
	 t = FILEinq_line(ei->edit_file,&pos);
	 FILEinq_position(ei->edit_file,FILE_MOVE_END,0,&pos);
	 b = FILEinq_line(ei->edit_file,&pos);
	 f = TEXTinq_topline(VIEW_TEXT(ev));
	 l = TEXTinq_bottomline(VIEW_TEXT(ev));
	 if (b < l) b = l;
	 LUNPROTECT(ev->edit);
       };
    };

   STEMscroll_set(ev->scrollwin,t,b,f,l);
};





/* end of edtview.c */


