/************************************************************************/
/*									*/
/*		edtvars.c						*/
/*									*/
/*	Handle command arguments					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TEXT_DELIM		0




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	local_index;
static	Integer 	global_index;



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	String		scan_arg();
static	String		scan_string();




/************************************************************************/
/*									*/
/*	EDT_vars_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_vars_init()
{
   local_index = 1;
   global_index = 0;
};





/************************************************************************/
/*									*/
/*	EDT_var_reset -- reset variable indices 			*/
/*									*/
/************************************************************************/


void
EDT_var_reset()
{
   local_index = 1;
   global_index = 0;
};





/************************************************************************/
/*									*/
/*	EDT_var_setup -- setup variables for editor			*/
/*									*/
/************************************************************************/


void
EDT_var_setup(ei)
   EDT_ID ei;
{
   ei->varct = local_index+10;
   ei->vars = (Universal *) calloc(ei->varct,sizeof(Universal));

   ei->gvarct = global_index+10;
   ei->gvars = (Universal *) calloc(ei->gvarct,sizeof(Universal));
};




/************************************************************************/
/*									*/
/*	EDT_var_get_value -- get value of a variable			*/
/*	EDT_var_set_value -- set value of a variable			*/
/*	EDT_var_set_default -- set default value of a variable		*/
/*									*/
/************************************************************************/


Universal
EDT_var_get_value(ei,ev)
   EDT_ID ei;
   EDT_VAR ev;
{
   register Integer i;
   register Universal v;

   if (ev->index == 0) {
      PROTECT;
      if (ev->index == 0) ev->index = local_index++;
      EDT_set_values(ev,ev->dflt);
      UNPROTECT;
    };

   if (ev->index > 0) {
      if (ev->index >= ei->varct) {
	 i = ei->varct;
	 ei->varct = local_index+10;
	 ei->vars = (Universal * ) realloc(ei->vars,ei->varct*sizeof(Universal));
	 for ( ; i < ei->varct; ++i) ei->vars[i] = NULL;
       }
      v = ei->vars[ev->index];
      if (ev->temp) ei->vars[ev->index] = ev->dflt;
    }
   else {
      if (-ev->index >= global_index) global_index = -ev->index;
      if (-ev->index >= ei->gvarct) {
	 i = ei->gvarct;
	 ei->gvarct = global_index+10;
	 ei->gvars = (Universal *) realloc(ei->gvars,ei->gvarct*sizeof(Universal));
	 for ( ; i < ei->gvarct; ++i) ei->gvars[i] = NULL;
       };
      v = ei->gvars[-ev->index];
    };

   return v;
};






void
EDT_var_set_value(ei,ev,val)
   EDT_ID ei;
   EDT_VAR ev;
   Universal val;
{
   if (ev->index == 0 || ev->index >= ei->varct || -ev->index >= ei->gvarct) {
      EDT_var_get_value(ei,ev);
    };

   if (ev->index > 0) {
      ei->vars[ev->index] = val;
    }
   else {
      ei->gvars[-ev->index] = val;
    };
};






void
EDT_var_set_default(ev,val)
   EDT_VAR ev;
   Universal val;
{
   ev->dflt = val;
};





/************************************************************************/
/*									*/
/*	EDT_value_enum -- find enum value from type and string		*/
/*									*/
/************************************************************************/


Integer
EDT_value_enum(ty,str)
   EDT_VAR_TYPE ty;
   String str;
{
   register Integer ln;
   register String s;
   register Integer i;

   if (str == NULL) return 0;

   s = EDT_ctbl_inq_enums(ty);
   ln = strlen(str);

   i = 0;
   while (s != NULL && *s != 0) {
      if (strncmp(s,str,ln) == 0) break;
      s = index(s,',');
      if (s != NULL) ++s;
      ++i;
    };

   if (s == NULL || *s == 0) {
      EDT_error("Bad enumeration constant %s",str);
      i = 0;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	EDT_value_buffer -- create buffer value from name		*/
/*									*/
/************************************************************************/


EDT_BUF
EDT_value_buffer(nm)
   String nm;
{
   register EDT_BUF b;

   b = PALLOC(EDT_BUF_INFO);

   strcpy(b->name,nm);
   b->buffer = FILEinq_buffer(nm);

   return b;
};





/************************************************************************/
/*									*/
/*	EDT_value_position -- create positon value			*/
/*									*/
/************************************************************************/


EDT_POS
EDT_value_position(flags)
   Integer flags;
{
   register EDT_POS p;

   p = PALLOC(EDT_POS_INFO);

   p->flags = flags;

   return p;
};




/************************************************************************/
/*									*/
/*	EDT_value_copy -- copy value given type 			*/
/*									*/
/************************************************************************/


Universal
EDT_value_copy(ty,val)
   EDT_VAR_TYPE ty;
   Universal val;
{
   register EDT_BUF b;
   register EDT_POS p;

   switch (ty) {
      default :
	 break;
      case EDT_VAR_TYPE_STRING :
	 if (val != NULL) val = (Universal) SALLOC(val);
	 break;
      case EDT_VAR_TYPE_BUFFER :
	 if (val != NULL) {
	    b = (EDT_BUF) val;
	    val = (Universal) EDT_value_buffer(b->name);
	  };
	 break;
      case EDT_VAR_TYPE_POSITION :
	 if (val != NULL) {
	    p = (EDT_POS) val;
	    val = (Universal) EDT_value_position(p->flags);
	  };
	 break;
    };

   return val;
};





/************************************************************************/
/*									*/
/*	EDT_vars_request -- request arguments from user 		*/
/*									*/
/************************************************************************/


Boolean
EDT_vars_request(ew,name,act,vars)
   EDT_VIEW ew;
   String name;
   Integer act;
   EDT_VAR vars[];
{
   Character buf[256];
   String menu[64];
   Integer ct;
   EDT_VAR ev;
   Universal * aptr[MAX_ARGS];
   Integer vals[MAX_ARGS];
   Character tbuf[MAX_ARGS][128];
   String s;
   EDT_BUF eb;
   EDT_POS ep;
   Integer i,j,k,l;
   Universal eval;
   EDT_ID ei;
   Boolean fg;

   ei = ew->edit;

   ct = 0;
   sprintf(buf,"%%CParameters for %s\n",name);
   menu[ct++] = SALLOC(buf);

   i = 0;
   j = 0;
   for (k = 0; k < act; ++k) {
      ev = vars[k];
      if (ev->name[0] == 0 || ev->name[0] == '&' || ev->name[0] == '*') continue;
      eval = EDT_var_get_value(ei,ev);
      switch (ev->type) {
	 case EDT_VAR_TYPE_BOOL :
	    sprintf(buf,"%%%do  %s\n",i,ev->name);
	    menu[ct++] = SALLOC(buf);
	    vals[i] = (Integer) eval;
	    aptr[i] = (Universal *) &vals[i];
	    break;
	 case EDT_VAR_TYPE_INT :
	    sprintf(buf,"%s : %%%d.8t\n",ev->name,i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(tbuf[i],"%d",eval);
	    aptr[i] = (Universal *) tbuf[i];
	    break;
	 case EDT_VAR_TYPE_STRING :
	    sprintf(buf,"%s : %%%d.32t\n",ev->name,i);
	    menu[ct++] = SALLOC(buf);
	    s = (String) eval;
	    if (s == NULL) tbuf[i][0] = 0;
	    else strcpy(tbuf[i],s);
	    aptr[i] = (Universal *) tbuf[i];
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    sprintf(buf,"%s : %%%d.16t\n",ev->name,i);
	    menu[ct++] = SALLOC(buf);
	    eb = (EDT_BUF) eval;
	    if (eb == NULL) tbuf[i][0] = 0;
	    else strcpy(tbuf[i],eb->name);
	    aptr[i] = (Universal *) tbuf[i];
	    break;
	 case EDT_VAR_TYPE_POSITION :
	    ep = (EDT_POS) eval;
	    sprintf(buf,"%s:",ev->name);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.0o Current Position",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.1o Start of File",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.2o End of File",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.3o Start of Current Line",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.4o End of Current Line",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.5o Start of Selection",i);
	    menu[ct++] = SALLOC(buf);
	    sprintf(buf,"   %%%d.6o End of Selection\n",i);
	    menu[ct++] = SALLOC(buf);
	    vals[i] = 0;
	    if (ep == NULL) vals[i] = 0;
	    else if (ep->flags & EDT_POS_FLAG_START_FILE) vals[i] = 1;
	    else if (ep->flags & EDT_POS_FLAG_END_FILE) vals[i] = 2;
	    else if (ep->flags & EDT_POS_FLAG_LINE_START) vals[i] = 3;
	    else if (ep->flags & EDT_POS_FLAG_LINE_END) vals[i] = 4;
	    else if (ep->flags & EDT_POS_FLAG_SELECT_START) vals[i] = 5;
	    else if (ep->flags & EDT_POS_FLAG_SELECT_END) vals[i] = 6;
	    else vals[i] = 0;
	    aptr[i] = (Universal *) &vals[i];
	    break;
	 default :
	    l = 0;
	    sprintf(buf,"%s:",ev->name);
	    menu[ct++] = SALLOC(buf);
	    s = EDT_ctbl_inq_enums(ev->type);
	    while (*s != 0) {
	       sprintf(buf,"   %%%d.%do  ",i,l++);
	       j = strlen(buf);
	       while (*s != 0 && *s != ',') buf[j++] = *s++;
	       if (*s == 0) buf[j++] = '\n';
	       else ++s;
	       buf[j] = 0;
	       menu[ct++] = SALLOC(buf);
	     };
	    vals[i] = (Integer) eval;
	    aptr[i] = & vals[i];
	    break;
       };
      ++i;
    };

   menu[ct++] = SALLOC("   %a%M   %c");
   menu[ct] = 0;

   if (i == 0) fg = TRUE;
   else fg = STEMdialog(ei->window,menu,aptr[0],aptr[1],aptr[2],aptr[3],aptr[4],
			   aptr[5],aptr[6],aptr[7],aptr[8],aptr[9],aptr[10],aptr[11],
			   aptr[12],aptr[13],aptr[14],aptr[15]);

   for (j = 0; j < ct; ++j) {
      SFREE(menu[j]);
    };

   if (!fg || i == 0) return fg;

   i = 0;
   for (k = 0; k < act; ++k) {
      ev = vars[k];
      if (ev->name[0] == 0 || ev->name[0] == '&' || ev->name[0] == '*') continue;
      switch (ev->type) {
	 case EDT_VAR_TYPE_BOOL :
	    EDT_var_set_value(ei,ev,vals[i]);
	    break;
	 case EDT_VAR_TYPE_INT :
	    if (sscanf(tbuf[i],"%d",&j) == 1) EDT_var_set_value(ei,ev,j);
	    break;
	 case EDT_VAR_TYPE_STRING :
	    eval = EDT_var_get_value(ei,ev);
	    if (eval == NULL && tbuf[i][0] == 0) ;
	    else if (eval != NULL && STREQL(eval,tbuf[i])) ;
	    else {
	       if (eval != NULL) SFREE(eval);
	       if (tbuf[i][0] == 0) EDT_var_set_value(ei,ev,NULL);
	       else EDT_var_set_value(ei,ev,SALLOC(tbuf[i]));
	     };
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    if (tbuf[i][0] != 0) {
	       eb = (EDT_BUF) EDT_var_get_value(ei,ev);
	       if (eb == NULL) {
		  eb = EDT_value_buffer(tbuf[i]);
		  EDT_var_set_value(ei,ev,eb);
		}
	       else if (STRNEQ(eb->name,tbuf[i])) {
		  strcpy(eb->name,tbuf[i]);
		  eb->buffer = FILEinq_buffer(eb->name);
		};
	     };
	    break;
	 case EDT_VAR_TYPE_POSITION :
	    ep = (EDT_POS) EDT_var_get_value(ei,ev);
	    switch (vals[i]) {
	       case 0 :
		  j = EDT_POS_FLAG_CURRENT;
		  break;
	       case 1 :
		  j = EDT_POS_FLAG_START_FILE;
		  break;
	       case 2 :
		  j = EDT_POS_FLAG_END_FILE;
		  break;
	       case 3 :
		  j = EDT_POS_FLAG_LINE_START|EDT_POS_FLAG_CURRENT;
		  break;
	       case 4 :
		  j = EDT_POS_FLAG_LINE_END|EDT_POS_FLAG_CURRENT;
		  break;
	       case 5 :
		  j = EDT_POS_FLAG_SELECT_START;
		  break;
	       case 6 :
		  j = EDT_POS_FLAG_SELECT_END;
		  break;
	     };
	    if (ep == NULL) {
	       ep = EDT_value_position(j);
	       EDT_var_set_value(ei,ev,ep);
	     }
	    else {
	       ep->flags = j;
	     };
	    break;
	 default :
	    EDT_var_set_value(ei,ev,vals[i]);
	    break;
       };
      ++i;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_vars_scan -- scan arguments from string			*/
/*									*/
/************************************************************************/


Boolean
EDT_vars_scan(ew,act,vars,text)
   EDT_VIEW ew;
   Integer act;
   EDT_VAR vars[];
   String text;
{
   Character buf[256];
   EDT_VAR ev;
   Integer vals[MAX_ARGS];
   Character tbuf[MAX_ARGS][128];
   EDT_BUF eb;
   Integer i,j,k;
   Universal eval;
   EDT_ID ei;
   Boolean fg;

   if (text == NULL || act == 0) return TRUE;

   ei = ew->edit;

   fg = FALSE;
   for (k = 0; k < act; ++k) {
      ev = vars[k];
      if (ev->type == EDT_VAR_TYPE_STRING) {
	 fg = TRUE;
	 break;
       }
      else if (ev->type == EDT_VAR_TYPE_BOOL || ev->type == EDT_VAR_TYPE_INT ||
		  ev->type == EDT_VAR_TYPE_BUFFER)
	 break;
    };

   if (!fg) {
      while (isspace(*text)) ++text;
    };

   if (*text == 0) return TRUE;

   fg = TRUE;
   i = 0;
   for (k = 0; k < act; ++k) {
      ev = vars[k];
      if (ev->name[0] == 0 || ev->name[0] == '&' || ev->name[0] == '*') continue;
      switch (ev->type) {
	 case EDT_VAR_TYPE_BOOL :
	    text = scan_arg(text,buf,256);
	    switch (text[0]) {
	       case '1' :
	       case 't' :
	       case 'T' :
		  vals[i] = TRUE;
		  break;
	       case '0' :
	       case 'f' :
	       case 'F' :
		  vals[i] = FALSE;
		  break;
	       default :
		  fg = FALSE;
		  break;
	     };
	    break;
	 case EDT_VAR_TYPE_INT :
	    text = scan_arg(text,buf,256);
	    if (sscanf(buf,"%d",&j) == 1) vals[i] = j;
	    break;
	 case EDT_VAR_TYPE_STRING :
	    text = scan_string(text,tbuf[i],128);
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    text = scan_string(text,tbuf[i],128);
	    break;
	 default :
	    break;
       };
      ++i;

      while (isspace(*text)) ++text;
      if (!fg || *text == 0) break;
    };

   if (!fg) return FALSE;
   act = i;

   i = 0;
   for (k = 0; k < act; ++k) {
      ev = vars[k];
      if (ev->name[0] == 0 || ev->name[0] == '&' || ev->name[0] == '*') continue;
      switch (ev->type) {
	 case EDT_VAR_TYPE_BOOL :
	    EDT_var_set_value(ei,ev,vals[i]);
	    break;
	 case EDT_VAR_TYPE_INT :
	    EDT_var_set_value(ei,ev,vals[i]);
	    break;
	 case EDT_VAR_TYPE_STRING :
	    eval = EDT_var_get_value(ei,ev);
	    if (eval == NULL && tbuf[i][0] == 0) ;
	    else if (eval != NULL && STREQL(eval,tbuf[i])) ;
	    else {
	       if (eval != NULL) SFREE(eval);
	       if (tbuf[i][0] == 0) EDT_var_set_value(ei,ev,NULL);
	       else EDT_var_set_value(ei,ev,SALLOC(tbuf[i]));
	     };
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    if (tbuf[i][0] != 0) {
	       eb = (EDT_BUF) EDT_var_get_value(ei,ev);
	       if (eb == NULL) {
		  eb = EDT_value_buffer(tbuf[i]);
		  EDT_var_set_value(ei,ev,eb);
		}
	       else if (STRNEQ(eb->name,tbuf[i])) {
		  strcpy(eb->name,tbuf[i]);
		  eb->buffer = FILEinq_buffer(eb->name);
		};
	     };
	    break;
	 default :
	    break;
       };
      ++i;
    };

   return TRUE;
};






/************************************************************************/
/*									*/
/*	scan_arg -- scan argument off of text string			*/
/*	scan_string -- scan string off of text string			*/
/*									*/
/************************************************************************/


static String
scan_arg(txt,buf,len)
   String txt;
   String buf;
   Integer len;
{
   while (isspace(*txt)) ++txt;

   while (*txt != 0 && !isspace(*txt) && *txt != TEXT_DELIM) {
      if (len > 0) {
	 --len;
	 *buf++ = *txt;
       };
      ++txt;
    };

   *buf = 0;

   return txt;
};





static String
scan_string(txt,buf,len)
   String txt;
   String buf;
   Integer len;
{
   while (*txt != 0 && *txt != TEXT_DELIM) {
      if (len > 0) {
	 --len;
	 *buf++ = *txt;
       };
      ++txt;
    };

   *buf = 0;

   return txt;
};





/* end of edtvars.c */

