/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		edtsyn.y						*/
/*									*/
/*	Parsing definitions for editor language 			*/
/*									*/
/************************************************************************/


#include "edt_local.h"



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	EDT_VAR 	define_var();
static	EDT_VAR_TYPE	make_enum_type();
static	void		bind_keys();
static	Integer 	prop_value();
static	Integer 	position_option();
static	Integer 	find_hook();



%}



%union {
   Integer	intval;
   String	string;
   EDT_VAR	edtvar;
   EDT_VAR_TYPE vartyp;
   EDT_CMD_ID	cmd_id;
   EDT_POS	postyp;
   EDT_BUF	buftyp;
   Sequence	seqval;
}




%token			LX_COMMAND LX_BIND LX_WHEN LX_LOAD LX_SOURCE LX_DEFINE
%token			LX_HELP LX_TYPEDEF LX_HOOK
%token			LX_INT_TYPE LX_STR_TYPE LX_BOOL_TYPE
%token			LX_BUF_TYPE LX_POS_TYPE
%token			LX_TRUE LX_FALSE LX_IGNORE LX_UNDEFINED
%token	<intval>	LX_INT
%token	<string>	LX_STRING LX_ID LX_KEYDEF LX_HELPTEXT

%type	<seqval>	args arg_list copy_arg_list
%type	<edtvar>	arg_item arg_decl copy_arg_item
%type	<string>	name arg_name arg_desc
%type	<intval>	opt_int_default opt_bool_default
%type	<string>	opt_str_default opt_enum_default
%type	<vartyp>	type_name
%type	<seqval>	name_list enum_list
%type	<intval>	props prop_list
%type	<cmd_id>	cmd_name
%type	<string>	filename
%type	<postyp>	pos_const pos_type opt_pos_default
%type	<intval>	pos_opts pos_opt
%type	<buftyp>	buffer opt_buf_default
%type	<seqval>	bind_when mode_list
%type	<string>	mode helptext

%start	definitions



%%



definitions :	/* empty */
	|	definitions definition
	;


definition :	command
	|	bind
	|	hook
	|	load
	|	source
	|	define
	|	help
	|	typedef
	;


command :	LX_COMMAND name args props '=' name ';'
			{ EDT_ctbl_define($2,$3,$6,$4); }
	|	LX_COMMAND name props '=' name '(' copy_arg_list ')' ';'
			{ EDT_ctbl_subcommand($2,$7,$5,$3); }
	|	LX_COMMAND name props '=' name '(' ')' ';'
			{ EDT_ctbl_subcommand($2,NULL,$5,$3); }
	;


args	:	'(' arg_list ')'
			{ $$ = $2; }
	|	'(' ')'
			{ $$ = NULL; }
	;


copy_arg_list : copy_arg_item
			{ $$ = NEW1($1); }
	|	copy_arg_item ';' copy_arg_list
			{ $$ = CONS($1,$3); }
	;


copy_arg_item : arg_item
	|	'*'
			{ $$ = define_var(SALLOC("*"),NULL,EDT_VAR_TYPE_UDF,0); }
	|	'&'
			{ $$ = define_var(SALLOC("&"),NULL,EDT_VAR_TYPE_UDF,0); }
	|	LX_INT
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_INT,$1); }
	|	LX_STRING
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_STRING,$1); }
	|	LX_TRUE
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_BOOL,TRUE); }
	|	LX_FALSE
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_BOOL,FALSE); }
	|	name
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_ENUM,$1); }
	|	buffer
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_BUFFER,$1); }
	|	pos_const
			{ $$ = define_var(NULL,NULL,EDT_VAR_TYPE_POSITION,$1); }
	;


arg_list :	arg_item
			{ $$ = NEW1($1); }
	|	arg_item ';' arg_list
			{ $$ = CONS($1,$3); }
	;


arg_item :	arg_decl
	|	'*' name
			{ $$ = EDT_ctbl_find_global($2); }
	;


arg_decl :	arg_name arg_desc ':' LX_INT_TYPE opt_int_default
			{ $$ = define_var($1,$2,EDT_VAR_TYPE_INT,$5); }
	|	arg_name arg_desc ':' LX_STR_TYPE opt_str_default
			{ $$ = define_var($1,$2,EDT_VAR_TYPE_STRING,$5); }
	|	arg_name arg_desc ':' LX_BOOL_TYPE opt_bool_default
			{ $$ = define_var($1,$2,EDT_VAR_TYPE_BOOL,$5); }
	|	arg_name arg_desc ':' type_name opt_enum_default
			{ $$ = define_var($1,$2,$4,EDT_value_enum($4,$5)); }
	|	arg_name arg_desc ':' LX_BUF_TYPE opt_buf_default
			{ $$ = define_var($1,$2,EDT_VAR_TYPE_BUFFER,$5); }
	|	arg_name arg_desc ':' LX_POS_TYPE opt_pos_default
			{ $$ = define_var($1,$2,EDT_VAR_TYPE_POSITION,$5); }
	;


arg_name :	name
	|	'&'
			{ $$ = NULL; }
	|	'&' name
			{ SFREE($2);
			  $$ = NULL;
			}
	;


arg_desc :	/* empty */
			{ $$ = NULL; }
	|	LX_STRING
	;


opt_int_default :   /* empty */
			{ $$ = 0; }
	|	'=' LX_INT
			{ $$ = $2; }
	;


opt_str_default :   /* empty */
			{ $$ = NULL; }
	|	'=' LX_STRING
			{ $$ = $2; }
	;


opt_bool_default :  /* empty */
			{ $$ = FALSE; }
	|	'=' LX_FALSE
			{ $$ = FALSE; }
	|	'=' LX_TRUE
			{ $$ = TRUE; }
	;


opt_enum_default :  /* empty */
			{ $$ = NULL; }
	|	'=' LX_ID
			{ $$ = $2; }
	;


opt_buf_default :   /* empty */
			{ $$ = NULL; }
	|	'=' buffer
			{ $$ = $2; }
	;


opt_pos_default :   /* empty */
			{ $$ = NULL; }
	|	'=' pos_const
			{ $$ = $2; }
	;


type_name :	name
			{ $$ = EDT_ctbl_find_type($1); }
	;


name_list :	LX_ID
			{ $$ = CONS($1,NULL); }
	|	LX_ID ',' name_list
			{ $$ = CONS($1,$3); }
	;


props	:	/* empty */
			{ $$ = 0; }
	|	'[' prop_list ']'
			{ $$ = $2; }
	;


prop_list :	LX_ID
			{ $$ = prop_value($1); }
	|	prop_list ',' LX_ID
			{ $$ = ($1 | prop_value($3)); }
	;


bind	:	LX_BIND LX_KEYDEF '=' cmd_name bind_when ';'
			{ bind_keys($2,$4,$5); }
	;


cmd_name :	name
			{ $$ = EDT_ctbl_find_command($1); }
	|	LX_IGNORE
			{ $$ = EDT_CMD_IGNORE; }
	|	LX_UNDEFINED
			{ $$ = EDT_CMD_UNDEFINED; }
	;


bind_when :	/* empty */
			{ $$ = NULL; }
	|	LX_WHEN mode_list
			{ $$ = $2; }
	;


mode_list :	mode
			{ $$ = NEW1($1); }
	|	mode ',' mode_list
			{ $$ = CONS($1,$3); }
	;


mode	:	LX_ID
			{ $$ = $1; };
	|	'^' LX_ID
			{ Character buf[1024];
			  sprintf(buf,"^%s",$2);
			  $$ = SALLOC(buf);
			  SFREE($2);
			}
	;


hook	:	LX_HOOK name '=' name bind_when ';'
			{ int id;
			  id = find_hook($2);
			  if (EDT_CMD_IS_HOOK(id)) bind_keys($4,id,$5);
			}
	;


name	:	LX_ID
	;


load	:	LX_LOAD filename ';'
			{ EDT_ctbl_load_file($2); }
	;


source	:	LX_SOURCE filename ';'
			{ source($2); }
	;


filename :	name
	|	LX_STRING
	;


pos_const :	pos_type
	;


pos_type :	'[' pos_opts ']'
			{ $$ = EDT_value_position($2); }
	;


pos_opts :	pos_opt
			{ $$ = $1; }
	|	pos_opts ',' pos_opt
			{ $$ = $1 | $3; }
	;


pos_opt :	name
			{ $$ = position_option($1); }
	;


buffer	:	'#' name
			{ $$ = EDT_value_buffer($2); }
	;


define	:	LX_DEFINE arg_decl ';'
			{ EDT_ctbl_make_global($2); }
	|	LX_DEFINE '*' name opt_bool_default ';'
			{ EDT_var_set_default(EDT_ctbl_find_global($3),$4); }
	;


help	:	LX_HELP cmd_name '=' LX_STRING ';'
			{ EDT_ctbl_help($2,$4); }
	|	LX_HELP cmd_name '=' '{' helptext '}' ';'
			{ EDT_ctbl_help($2,$5); }
	;


helptext :	LX_HELPTEXT
	|	helptext LX_HELPTEXT
			{ Character buf[20480];
			  sprintf(buf,"%s%s",$1,$2);
			  SFREE($1);
			  SFREE($2);
			  $$ = SALLOC(buf);
			}
	;


typedef :	LX_TYPEDEF name '=' '<' enum_list '>' ';'
			{ make_enum_type($2,$5); }
	;


enum_list :	LX_ID
			{ $$ = CONS($1,NULL); }
	|	LX_ID ',' name_list
			{ $$ = CONS($1,$3); }
	|	LX_STRING
			{ $$ = CONS($1,NULL); }
	|	LX_STRING ',' name_list
			{ $$ = CONS($1,$3); }
	;


%%



#include "edtlex.c"




/************************************************************************/
/*									*/
/*	EDT_syn_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_syn_init()
{
};





/************************************************************************/
/*									*/
/*	EDTload_description -- load description file			*/
/*									*/
/************************************************************************/


Boolean
EDTload_description(filename)
   String filename;
{
   EDT_init();

   if (filename == NULL || *filename == 0) return FALSE;

   PROTECT;

   if (!first_file(filename)) {
      UNPROTECT;
      EDT_error("Couldn't open file %s",filename);
      return FALSE;
    };

   yyparse();

   EDT_rebind();

   UNPROTECT;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	define_var -- build an argument block				*/
/*									*/
/************************************************************************/


static EDT_VAR
define_var(nm,dsc,typ,val)
   String nm;
   String dsc;
   EDT_VAR_TYPE typ;
   Universal val;
{
   EDT_VAR ev;

   ev = EDT_ctbl_new_var(nm,dsc,typ,val,0);

   if (nm != NULL) SFREE(nm);
   if (dsc != NULL) SFREE(dsc);

   return ev;
};





/************************************************************************/
/*									*/
/*	make_enum_type -- build enum type from list			*/
/*									*/
/************************************************************************/


static EDT_VAR_TYPE
make_enum_type(nm,ens)
   String nm;
   Sequence ens;
{
   EDT_VAR_TYPE t;
   Character buf[1024];
   register String s;
   register Sequence l;

   buf[0] = 0;
   forin (s,String,l,ens) {
      if (buf[0] != 0) strcat(buf,",");
      strcat(buf,s);
    };

   t = EDT_ctbl_new_enum(nm,buf);

   if (nm != NULL) SFREE(nm);

   return t;
};





/************************************************************************/
/*									*/
/*	bind_keys -- bind command to key list				*/
/*									*/
/************************************************************************/


static void
bind_keys(keys,cmd,when)
   String keys;
   EDT_CMD_ID cmd;
   Sequence when;
{
   register Sequence l;
   register String s;

   if (when == NULL) {
      EDT_ctbl_new_bind(keys,cmd,NULL);
    }
   else {
      forin (s,String,l,when) {
	 EDT_ctbl_new_bind(keys,cmd,s);
	 SFREE(s);
       };
      LFREE(when);
    };

   SFREE(keys);
};





/************************************************************************/
/*									*/
/*	yyerror -- handle errors during parsing 			*/
/*									*/
/************************************************************************/


yyerror(msg)
   String msg;
{
   EDT_error("File %s Line %d: %s",cur_file,yylineno,msg);
};




/************************************************************************/
/*									*/
/*	prop_value -- return property value from name			*/
/*									*/
/************************************************************************/


static Integer
prop_value(name)
   String name;
{
   register Integer vl;

   if (STREQL(name,"GROUP_TYPE")) vl = EDT_CMD_FLAG_GROUP_TYPE;
   else if (STREQL(name,"GROUP_MOVE")) vl = EDT_CMD_FLAG_GROUP_MOVE;
   else if (STREQL(name,"GROUP_3")) vl = EDT_CMD_FLAG_GROUP_3;
   else if (STREQL(name,"GROUP_4")) vl = EDT_CMD_FLAG_GROUP_4;
   else if (STREQL(name,"NO_COMMAND")) vl = EDT_CMD_FLAG_NO_COMMAND;
   else if (STREQL(name,"SHOW_ARGS")) vl = EDT_CMD_FLAG_SHOW_ARGS;
   else if (STREQL(name,"SAME")) vl = EDT_CMD_FLAG_SAME;
   else if (STREQL(name,"CONFIRM")) vl = EDT_CMD_FLAG_CONFIRM;
   else if (STREQL(name,"CLOSE")) vl = EDT_CMD_FLAG_CLOSE;
   else if (STREQL(name,"MACRO_ABORT")) vl = EDT_CMD_FLAG_MACRO_ABORT;
   else if (STREQL(name,"NO_PREVIOUS")) vl = EDT_CMD_FLAG_NO_PSET;
   else {
      EDT_error("Bad command flag %s",name);
      vl = 0;
    };

   SFREE(name);

   return vl;
};





/************************************************************************/
/*									*/
/*	position_option -- convert position option name to flags	*/
/*									*/
/************************************************************************/


static Integer
position_option(txt)
   String txt;
{
   register Integer vl;

   if (STREQL(txt,"CURRENT")) vl = EDT_POS_FLAG_CURRENT;
   else if (STREQL(txt,"START_FILE")) vl = EDT_POS_FLAG_START_FILE;
   else if (STREQL(txt,"END_FILE")) vl = EDT_POS_FLAG_END_FILE;
   else if (STREQL(txt,"LEFT")) vl = EDT_POS_FLAG_LINE_START;
   else if (STREQL(txt,"RIGHT")) vl = EDT_POS_FLAG_LINE_END;
   else if (STREQL(txt,"START")) vl = EDT_POS_FLAG_COND_START;
   else if (STREQL(txt,"SELECT_START")) vl = EDT_POS_FLAG_SELECT_START;
   else if (STREQL(txt,"SELECT_END")) vl = EDT_POS_FLAG_SELECT_END;
   else if (STREQL(txt,"VIEW_TOP")) vl = EDT_POS_FLAG_VIEW_TOP;
   else if (STREQL(txt,"VIEW_BOTTOM")) vl = EDT_POS_FLAG_VIEW_BOTTOM;
   else if (STREQL(txt,"VIEW_MIDDLE")) vl = EDT_POS_FLAG_VIEW_MIDDLE;
   else if (STREQL(txt,"WORD")) vl = EDT_POS_FLAG_WORD;
   else if (STREQL(txt,"PARA")) vl = EDT_POS_FLAG_PARA;
   else if (STREQL(txt,"SENTENCE")) vl = EDT_POS_FLAG_SENTENCE;
   else if (STREQL(txt,"LINE")) vl = EDT_POS_FLAG_LINE;
   else if (STREQL(txt,"TAB")) vl = EDT_POS_FLAG_TAB;
   else if (STREQL(txt,"PREVIOUS")) vl = EDT_POS_FLAG_PREVIOUS;
   else if (STREQL(txt,"END")) vl = EDT_POS_FLAG_END;
   else if (STREQL(txt,"SEPARATOR")) vl = EDT_POS_FLAG_SEP;
   else if (STREQL(txt,"CURRENT_OK")) vl = EDT_POS_FLAG_CURROK;
   else if (STREQL(txt,"NEXT_LINE")) vl = EDT_POS_FLAG_NEXT_LINE;
   else if (STREQL(txt,"LAST_LINE")) vl = EDT_POS_FLAG_PREV_LINE;
   else {
      EDT_error("Bad position flag %s",txt);
      vl = 0;
    };

   return vl;
};





/************************************************************************/
/*									*/
/*	find_hook -- find hook by name					*/
/*									*/
/************************************************************************/


static Integer
find_hook(nm)
   String nm;
{
   register Integer id;

   if (STREQL(nm,"INDENT")) id = EDT_CMD_HOOK(EDT_HOOK_INDENT);
   else if (STREQL(nm,"WORD")) id = EDT_CMD_HOOK(EDT_HOOK_WORD_BOUNDS);
   else if (STREQL(nm,"SENTENCE")) id = EDT_CMD_HOOK(EDT_HOOK_SENT_BOUNDS);
   else if (STREQL(nm,"PARA")) id = EDT_CMD_HOOK(EDT_HOOK_PARA_BOUNDS);
   else if (STREQL(nm,"MODE")) id = EDT_CMD_HOOK(EDT_HOOK_MODE_SELECT);
   else if (STREQL(nm,"END_OF_LINE")) id = EDT_CMD_HOOK(EDT_HOOK_END_OF_LINE);
   else id = EDT_CMD_UNDEFINED;

   return id;
};





/* end of edtsyn.y */

