/************************************************************************/
/*									*/
/*		edtprim.c						*/
/*									*/
/*	Primitive command routines					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"
#include <sys/file.h>



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	delete_pick();
static	void		display_message();
static	Boolean 	current_position();
static	void		shell_position();
static	void		check_end_of_line();
static	void		indent_lines();
static	void		exec_unix_command();




/************************************************************************/
/*									*/
/*	EDT_prim_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_prim_init()
{
};





/************************************************************************/
/*									*/
/*	EDCMDNoop -- do nothing 					*/
/*	EDCMDEnter -- force args for next command			*/
/*	EDCMDInsertMode -- flip insert mode				*/
/*	EDCMDIndentMode -- flip indent mode				*/
/*	EDCMDOpenMode -- flip open mode 				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
EDCMDNoop(ev)
   EDT_VIEW ev;
{
   return TRUE;
};





Boolean
EDCMDEnter(ev)
   EDT_VIEW ev;
{
   ev->edit->get_args = TRUE;

   return TRUE;
};





Boolean
EDCMDInsertMode(ev)
   EDT_VIEW ev;
{
   ev->edit->insert_mode = ! ev->edit->insert_mode;

   return TRUE;
};





Boolean
EDCMDIndentMode(ev)
   EDT_VIEW ev;
{
   ev->edit->auto_indent = ! ev->edit->auto_indent;

   return TRUE;
};





Boolean
EDCMDOpenMode(ev)
   EDT_VIEW ev;
{
   Integer j,k;
   FILE_POS pos;

   if (ev->edit->open_mode) {
      ev->edit->open_mode = FALSE;
    }
   else {
      j = FILEinq_currchar(VIEW_FILE(ev))-1;
      k = FILEinq_linespaces(VIEW_FILE(ev));
      if (j <= k) {
	 FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&pos);
	 EDCMDInsertString(ev,"\n",1,NULL);
       };
      if (j <= k) EDT_indent_on_line(ev,TRUE,FALSE);
      ev->edit->open_mode = TRUE;
      EDT_synch_view(ev);
    };

   return TRUE;
};





Boolean
EDCMDReadOnly(ev,fg)
   EDT_VIEW ev;
   Boolean fg;
{
   if (ev->edit->shell) return FALSE;

   EDTenable(ev->edit,fg);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	EDCMDGenericMove -- generic move command			*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericMove(ev,ep,dline,dchar)
   EDT_VIEW ev;
   EDT_POS ep;
   Integer dline;
   Integer dchar;
{
   FILE_POS pos,npos;
   Integer fgs,efgs;
   Boolean new;

   if (ep == NULL) efgs = EDT_POS_FLAG_CURRENT;
   else efgs = ep->flags;

   if (efgs & EDT_POS_FLAG_LINE_START) fgs = FILE_MOVE_LINE_START;
   else if (efgs & EDT_POS_FLAG_LINE_END) fgs = FILE_MOVE_LINE_END;
   else fgs = 0;

   new = FALSE;

   if (!current_position(ep)) {
      EDT_file_position(ev,ep,&pos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
      new = TRUE;
    };

   if ((efgs & EDT_POS_FLAG_START_FILE) != 0 && dline > 0) --dline;

   if (dline != 0) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|fgs,dline,&npos);
      new = TRUE;
    };
   if (dchar != 0) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR|fgs,dchar,&npos);
      new = TRUE;
    };

   if (new) EDT_auto_extend(ev);

   EDT_synch_view(ev);

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	EDCMDTabN -- handle arbitrary tabs					*/
/*										*/
/********************************************************************************/


Boolean
EDCMDTabN(ev,size,fwd)
   EDT_VIEW ev;
   Integer size;
   Boolean fwd;
{
   Integer i,j,ln;
   FILE_POS pos;

   i = FILEinq_currchar(VIEW_FILE(ev));
   ln = FILEinq_currline(VIEW_FILE(ev));

   j = (i-1) / size;
   j *= size;
   j += 1;

   if (fwd) j += size;
   else if (j != 1 && i == j) j -= size;
   else if (j == 1 && i == j) return TRUE;

   FILEset_position(VIEW_FILE(ev),ln,j,&pos);
   EDT_auto_extend(ev);
   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDGenericSearch -- generic search command			*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericSearch(ev,str,bkwd,wrap,cas,re,regn)
   EDT_VIEW ev;
   String str;
   Boolean bkwd;
   Boolean wrap;
   Boolean cas;
   Boolean re;
   Integer regn;
{
   Integer style;
   FILE_POS spos,epos,pos;
   Boolean fg;
   EDT_SELECT sel;
   Character buf[256];

   style = 0;

   if (bkwd) style |= FILE_SEARCH_BACKWARD;
   else style |= FILE_SEARCH_FORWARD;

   if (wrap) style |= FILE_SEARCH_WRAP;

   if (cas) style |= FILE_SEARCH_NOCASE;

   if (re) style |= FILE_SEARCH_RE;

   if (str == NULL) fg = FALSE;
   else fg = FILEsearch_position(VIEW_FILE(ev),str,style,&spos,&epos);

   if (!fg) {
      if (str == NULL) sprintf(buf,"Nothing to search for");
      else sprintf(buf,"Search string \"%s\" not found",str);
      display_message(ev,buf);
    }
   else if (regn < 0) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&spos,&pos);
      EDT_auto_extend(ev);
    }
   else {
      sel = &ev->edit->select;
      sel->unit = EDT_SELECT_CHAR;
      EDT_start_selection(ev,sel,&spos,TRUE);
      EDT_extend_selection(ev,sel,&epos,FALSE,TRUE);
      EDT_finish_selection(ev,sel);
      EDT_next_char(ev->edit,&epos,&epos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&epos,&pos);
    };

   EDT_synch_view(ev);

   return fg;
};






/************************************************************************/
/*									*/
/*	EDCMDReplaceString -- replace string command			*/
/*									*/
/************************************************************************/


Boolean
EDCMDReplaceString(ev,cnt,str,rstr,bkwd,wrap,cas,re)
   EDT_VIEW ev;
   Integer cnt;
   String str;
   String rstr;
   Boolean bkwd;
   Boolean wrap;
   Boolean cas;
   Boolean re;
{
   Integer style;
   FILE_POS spos,epos,pos,npos;
   Boolean fg;
   Integer fnd;
   FILE_BUF buf,rbuf;
   Integer svpos;
   Character ebuf[256];

   fnd = 0;

   style = 0;
   if (bkwd) style |= FILE_SEARCH_BACKWARD;
   else style |= FILE_SEARCH_FORWARD;
   if (wrap) style |= FILE_SEARCH_WRAP;
   if (cas) style |= FILE_SEARCH_NOCASE;
   if (re) style |= FILE_SEARCH_RE;

   buf = FILEinq_buffer(DEFAULT_BUFFER_NAME);
   rbuf = FILEinq_buffer("BWEDIT$_SUBST");
   if (rstr == NULL) rstr = "";
   FILEbuffer_define(rbuf,rstr);

   do {
      fg = FILEsearch_position(VIEW_FILE(ev),str,style,&spos,&epos);

      if (!fg) {
	 if (fnd == 0) {
	    sprintf(ebuf,"Replace string \"%s\" not found",str);
	    display_message(ev,ebuf);
	  };
	 break;
       };

      ++fnd;

      style &= ~FILE_SEARCH_WRAP;

      EDT_next_char(ev->edit,&epos,&epos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&spos,&pos);
      FILEdelete(VIEW_FILE(ev),buf,FILE_MOVE_ABS,&epos);

      svpos = FILEsave_pos(VIEW_FILE(ev));
      FILEinsert(VIEW_FILE(ev),rbuf,1);
      FILEinq_savepos(VIEW_FILE(ev),svpos,&pos);
      FILEdelete_pos(VIEW_FILE(ev),svpos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    }
   while (--cnt != 0);

   EDT_synch_view(ev);

   return fg;
};






/************************************************************************/
/*									*/
/*	EDCMDGenericPosition -- window positioning commands		*/
/*	EDCMDScroll -- scrolling commands				*/
/*	EDCMDKeepPosition -- set flag for enabled cursor		*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericPosition(ev,pos,dline,dchar)
   EDT_VIEW ev;
   Integer pos; 	/* CURRENT, TOP, MIDDLE, BOTTOM, TOP_LEFT */
   Integer dline;
   Integer dchar;
{
   TEXT_FLAG fgs;
   FILE_POS fpos;
   Boolean top;

   top = TRUE;

   if (pos != 0) {
      switch (pos) {
	 case 1 :
	    fgs = TEXT_FLAG_NEAR_TOP;
	    break;
	 case 2 :
	    fgs = TEXT_FLAG_NEAR_MIDDLE;
	    break;
	 case 3 :
	    fgs = TEXT_FLAG_NEAR_BOTTOM;
	    break;
	 case 4 :
	    fgs = TEXT_FLAG_TOP_LEFT;
	    break;
       };
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&fpos);
      TEXTposition(VIEW_TEXT(ev),fgs,&fpos);
    };

   if (dline != 0) {
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_DELTA_LINES,dline);
      if (dline < 0) top = FALSE;
    };

   if (dchar != 0) {
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_DELTA_CHARS,dchar);
    };

   EDT_synch_position(ev,top);

   return TRUE;
};





Boolean
EDCMDScroll(ev,percent,bkwd)
   EDT_VIEW ev;
   Integer percent;
   Boolean bkwd;
{
   Integer tp,bt,dline;

   tp = TEXTinq_topline(VIEW_TEXT(ev));
   bt = TEXTinq_bottomline(VIEW_TEXT(ev));
   dline = (bt-tp)*percent/100;
   if (dline <= 0) dline = 1;
   if (bkwd) dline = -dline;

   TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_DELTA_LINES,dline);

   EDT_synch_position(ev,!bkwd);

   return TRUE;
};





Boolean
EDCMDKeepPosition(ev,fg)
   EDT_VIEW ev;
   Boolean fg;
{
   TEXTcursor_enable(VIEW_TEXT(ev),(fg ? 1 : 0));

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDInsertCharacter -- insert character into buffer		*/
/*	EDCMDSelfInsert -- insert last character typed into buffer	*/
/*	EDCMDNewLine -- insert new line (and indent)			*/
/*	EDCMDBackspace -- handle backspace character			*/
/*									*/
/************************************************************************/


Boolean
EDCMDInsertCharacter(ev,ch,insmode)
   EDT_VIEW ev;
   Character ch;
   Boolean insmode;
{
   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   shell_position(ev);
   check_end_of_line(ev,ch);

   FILEtypein(VIEW_FILE(ev),ch,insmode);

   return TRUE;
};




Boolean
EDCMDSelfInsert(ev)
   EDT_VIEW ev;
{
   EDT_ID ei;

   ei = ev->edit;

   if (ei->last_key_struck == '\b') {
      if (EDT_clear_selection(ei,DEFAULT_BUFFER_NAME)) return TRUE;
    }
   else {
      EDT_clear_selection(ei,DEFAULT_BUFFER_NAME);
    };

   if (ei->last_key_struck == ' ' && ei->abbrev_mode) {
      EDCMDAbbrevExpand(ev,FALSE);
    };

   shell_position(ev);
   check_end_of_line(ev,ei->last_key_struck);

   if (ei->last_key_struck == '\b' && !ei->insert_mode) {
      if (STRNEQ(EDT_ctbl_cmd_name(ei->previous_cmd),"SelfInsert")) {
	 return EDCMDGenericMove(ev,NULL,0,-1);
       };
    };

   FILEtypein(ei->edit_file,ei->last_key_struck,ei->insert_mode);

   return TRUE;
};





Boolean
EDCMDNewLine(ev,insmode,openmode,indmode)
   EDT_VIEW ev;
   Boolean insmode;
   Boolean openmode;
   Boolean indmode;
{
   FILE_POS pos;

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   shell_position(ev);

   if (indmode && ev->edit->cur_indent && !insmode) {
      EDCMDIndent(ev,NULL,NULL);
    };

   if (insmode) {
      FILEtypein(VIEW_FILE(ev),'\n',TRUE);
      if (indmode && ev->edit->cur_indent) {
	 if (FILEinq_currline(VIEW_FILE(ev)) != 1) {
	    FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE,-1,&pos);
	    EDCMDIndent(ev,NULL,NULL);
	    FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE,1,&pos);
	  };
       };
    }
   else if (openmode) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_END,0,&pos);
      FILEtypein(VIEW_FILE(ev),'\n',TRUE);
    }
   else {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
		1,&pos);
      EDT_auto_extend(ev);
    };

   if (indmode) {
      if (insmode) {
	 EDCMDIndent(ev,NULL,NULL);		/* force indentation	*/
       }
      else {
	 EDT_indent_on_line(ev,TRUE,FALSE);
       };
    };

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDGenericMap -- generic mapping function			*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericMap(ev,typ,from,to)
   EDT_VIEW ev;
   Integer typ; 	/* TO_UPPER, TO_LOWER, TO_CONTROL, INVERT_CASE, */
			/*   CAPITALIZE 				*/
   EDT_POS from;
   EDT_POS to;
{
   FILE_POS fpos,tpos,opos,pos;
   FILE_BUF buf;
   String cnts;
   String s;
   Boolean firstfg;

   EDT_file_position(ev,from,&fpos);
   EDT_file_position(ev,to,&tpos);

   EDT_clear_selection(ev->edit,NULL);

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&opos);

   buf = FILEinq_buffer("BWEDIT$_MAP");
   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&fpos,&pos);
   FILEdelete(VIEW_FILE(ev),buf,FILE_MOVE_ABS,&tpos);

   cnts = FILEbuffer_contents(buf);
   firstfg = TRUE;

   for (s = cnts; *s != 0; ++s) {
      switch (typ) {
	 case 0 :
	    if (islower(*s)) *s = *s - 'a' + 'A';
	    break;
	 case 1 :
	    if (isupper(*s)) *s = *s - 'A' + 'a';
	    break;
	 case 2 :
	    *s &= 0x1f;
	    break;
	 case 3:
	    if (islower(*s)) *s = *s - 'a' + 'A';
	    else if (isupper(*s)) *s = *s - 'A' + 'a';
	    break;
	 case 4 :
	    if (firstfg && islower(*s)) *s = *s - 'a' + 'A';
	    if (isalnum(*s) || *s == '-' || *s == '_') firstfg = FALSE;
	    else firstfg = TRUE;
	    break;
       };
    };

   FILEbuffer_define(buf,cnts);
   free(cnts);

   FILEinsert(VIEW_FILE(ev),buf,1);

   EDT_synch_view(ev);

   return TRUE;
};




/************************************************************************/
/*									*/
/*	EDCMDGenericDelete -- delete arbitrary text			*/
/*	EDCMDGenericPick -- pick arbitrary text 			*/
/*	EDCMDPatternDelete -- delete based on patterns			*/
/*									*/
/*	delete_pick -- code for generic delete and pick 		*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericDelete(ev,buf,fep,tep,dline,dchar)
   EDT_VIEW ev;
   EDT_BUF buf;
   EDT_POS fep;
   EDT_POS tep;
   Integer dline;
   Integer dchar;
{
   Boolean fg;

   fg = delete_pick(ev,buf,fep,tep,dline,dchar,TRUE);

   return fg;
};





Boolean
EDCMDGenericPick(ev,buf,fep,tep,dline,dchar)
   EDT_VIEW ev;
   EDT_BUF buf;
   EDT_POS fep;
   EDT_POS tep;
   Integer dline;
   Integer dchar;
{
   Boolean fg;

   fg = delete_pick(ev,buf,fep,tep,dline,dchar,FALSE);

   return fg;
};





Boolean
EDCMDPatternDelete(ev,buf,frompat,topat)
   EDT_VIEW ev;
   EDT_BUF buf;
   String frompat;
   String topat;
{
   FILE_POS from,to,pos;

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   if (frompat != NULL && frompat[0] != 0) {
      FILEsearch(VIEW_FILE(ev),frompat,FILE_SEARCH_RE|FILE_SEARCH_CURROK|
		    FILE_SEARCH_BACKWARD,&from);
    }
   else {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&from);
    };

   if (topat != NULL && topat[0] != 0) {
      FILEsearch(VIEW_FILE(ev),frompat,FILE_SEARCH_RE|FILE_SEARCH_CURROK|
		    FILE_SEARCH_FORWARD,&to);
    }
   else {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&to);
    };

   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&from,&pos);
   FILEdelete(VIEW_FILE(ev),buf->buffer,FILE_MOVE_ABS,&to);

   EDT_synch_view(ev);

   return TRUE;
};





static Boolean
delete_pick(ev,buf,fep,tep,dline,dchar,delfg)
   EDT_VIEW ev;
   EDT_BUF buf;
   EDT_POS fep;
   EDT_POS tep;
   Integer dline;
   Integer dchar;
   Boolean delfg;
{
   FILE_POS from,to,pos,opos;
   Integer fgs;
   Boolean indent;
   Boolean fg;

   if (current_position(fep) && current_position(tep)) {
      if (dline == 0 && dchar == 0) return TRUE;
      if (dline == 0 && dchar == -1 && delfg &&
	     (!ev->edit->backeol_mode || FILEinq_currchar(VIEW_FILE(ev)) != 1) ) {
	 fg = (ev->edit->select.replace && ev->edit->select.valid);
	 EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);
	 if (!fg) FILEtypein(VIEW_FILE(ev),'\b',TRUE);
	 return TRUE;
       };
    };

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&opos);

   if (ev->edit->usesel_mode && VIEW_SELECT_VALID(ev)) {
      FILEpos_copy(VIEW_FILE(ev),&VIEW_SELECT_START(ev),&from);
      FILEpos_copy(VIEW_FILE(ev),&VIEW_SELECT_END(ev),&to);
      EDT_replace_selection(&ev->edit->select,FALSE);
      dline = dchar = 0;
    }
   else {
      EDT_file_position(ev,fep,&from);
      EDT_file_position(ev,tep,&to);
    };

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   if (dline > 0 &&
	  FILEinq_currchar(VIEW_FILE(ev))-1 <= FILEinq_linespaces(VIEW_FILE(ev)))
      indent = TRUE;
   else indent = FALSE;

   if (dline != 0 || dchar != 0) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&to,&pos);
      if (tep == NULL) fgs = 0;
      else if (tep->flags & EDT_POS_FLAG_LINE_START) fgs = FILE_MOVE_LINE_START;
      else if (tep->flags & EDT_POS_FLAG_LINE_END) fgs = FILE_MOVE_LINE_END;
      else fgs = 0;
      if (dline != 0) {
	 FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|fgs,dline,&to);
       };
      if (dchar != 0) {
	 if (dchar < 0 && ev->edit->backeol_mode && fgs == 0 &&
		FILEinq_currchar(VIEW_FILE(ev)) == 1) {
	    if (FILEinq_currline(VIEW_FILE(ev)) > 1) {
	       FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_END,-1,&to);
	     };
	  }
	 else {
	    FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR|fgs,dchar,&to);
	  };
       };
    };

   if (FILEpos_compare(VIEW_FILE(ev),&from,&to) > 0) {
      FILEpos_copy(VIEW_FILE(ev),&from,&pos);
      FILEpos_copy(VIEW_FILE(ev),&to,&from);
      FILEpos_copy(VIEW_FILE(ev),&pos,&to);
    };

   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&from,&pos);

   if (delfg) FILEdelete(VIEW_FILE(ev),buf->buffer,FILE_MOVE_ABS,&to);
   else FILEpick(VIEW_FILE(ev),buf->buffer,FILE_MOVE_ABS,&to);

   if (indent) EDCMDIndent(ev,NULL,NULL);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDGenericInsert -- insert text at arbitrary position 	*/
/*	EDCMDInsertString -- insert string at position			*/
/*									*/
/************************************************************************/


Boolean
EDCMDGenericInsert(ev,buf,ct,ep,after)
   EDT_VIEW ev;
   EDT_BUF buf;
   Integer ct;
   EDT_POS ep;
   Boolean after;
{
   FILE_POS pos,npos,xpos;
   Integer svpos;
   Integer sln,eln;

   if (ct == 0) ct = 1;

   if (!current_position(ep)) {
      EDT_file_position(ev,ep,&pos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    }

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   shell_position(ev);

   svpos = FILEsave_pos(VIEW_FILE(ev));

   FILEinsert(VIEW_FILE(ev),buf->buffer,ct);

   if (ev->edit->cur_indent) {
      sln = FILEinq_currline(VIEW_FILE(ev));
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&npos);
      FILEinq_savepos(VIEW_FILE(ev),svpos,&pos);
      eln = FILEinq_line(VIEW_FILE(ev),&pos);
      if (sln < eln) indent_lines(ev,sln,eln);
      else if (after) FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&xpos);
      if (!after) FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&npos,&pos);
      else EDT_indent_on_line(ev,FALSE,FALSE);
    }
   else if (after) {
      FILEinq_savepos(VIEW_FILE(ev),svpos,&pos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    };

   FILEdelete_pos(VIEW_FILE(ev),svpos);

   EDT_synch_view(ev);

   return TRUE;
};





Boolean
EDCMDInsertString(ev,txt,ct,ep)
   EDT_VIEW ev;
   String txt;
   Integer ct;
   EDT_POS ep;
{
   FILE_BUF buf;
   FILE_POS pos,npos;

   if (ct == 0) ct = 1;
   if (txt == NULL) return TRUE;

   buf = FILEinq_buffer("BWEDIT$_INSSTR");
   FILEbuffer_define(buf,txt);

   if (!current_position(ep)) {
      EDT_file_position(ev,ep,&pos);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    };

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   shell_position(ev);

   FILEinsert(VIEW_FILE(ev),buf,ct);

   EDT_indent_on_line(ev,FALSE,FALSE);

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDSavePosition -- save current file position 		*/
/*	EDCMDGotoPosition -- goto saved file position			*/
/*									*/
/************************************************************************/


Boolean
EDCMDSavePosition(ev)
   EDT_VIEW ev;
{
   EDT_ID ei;

   ei = ev->edit;

   if (ei->marks == FILE_MARK_NEW) {
      ei->marks = FILEset_mark(ei->edit_file,FILE_MARK_NEW);
    };

   FILEset_mark(ei->edit_file,ei->marks);

   return TRUE;
};




Boolean
EDCMDGotoPosition(ev)
   EDT_VIEW ev;
{
   FILE_POS pos,npos;

   if (ev->edit->marks == FILE_MARK_NEW) {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_START|FILE_MOVE_LINE_START,
			  0,&pos);
    }
   else {
      FILEinq_mark(VIEW_FILE(ev),ev->edit->marks,&pos);
      FILEunset_mark(VIEW_FILE(ev),ev->edit->marks);
    };

   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);

   EDT_auto_extend(ev);

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDCommand -- evaluate command by name			*/
/*									*/
/************************************************************************/


Boolean
EDCMDCommand(ev,name,argfg)
   EDT_VIEW ev;
   String name;
   Boolean argfg;
{
   EDT_CMD_ID cmd;
   Boolean fg;
   Character buf[256];

   if (name == NULL || *name == 0) return FALSE;

   cmd = EDT_ctbl_find_command(name);

   if (cmd == EDT_CMD_UNDEFINED) {
      sprintf(buf,"Command %s not found",name);
      display_message(ev,buf);
      return FALSE;
    };

   if (argfg) ev->edit->get_args = TRUE;

   fg = EDT_eval_cmd(ev,cmd);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDCMDUndo -- handle UNDO					*/
/*	EDCMDRedo -- handle REDO					*/
/*									*/
/************************************************************************/


Boolean
EDCMDUndo(ev)
   EDT_VIEW ev;
{
   Boolean fg;
   FILE_POS pos;

   EDT_clear_selection(ev->edit,NULL);

   fg = EDT_undo(ev->edit);

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
   TEXTcursor(VIEW_TEXT(ev),TEXT_CURSOR_FORCE,&pos);

   return fg;
}




Boolean
EDCMDRedo(ev)
   EDT_VIEW ev;
{
   Boolean fg;
   FILE_POS pos;

   EDT_clear_selection(ev->edit,NULL);

   fg = EDT_redo(ev->edit);

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
   TEXTcursor(VIEW_TEXT(ev),TEXT_CURSOR_FORCE,&pos);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDCMDCloseFile -- close current file				*/
/*	EDCMDOpenFile -- replace current file				*/
/*	EDCMDSaveFile -- save current file				*/
/*	EDCMDAlterAccess -- alter r/o access				*/
/*									*/
/************************************************************************/


Boolean
EDCMDCloseFile(ev,abort)
   EDT_VIEW ev;
   Boolean abort;
{
   Boolean fg;
   Boolean savefg;

   if (ev->edit->no_close) {
      if (!abort) fg = EDCMDSaveFile(ev,NULL);
      else {
	 ASHremove(ev->edit->window);
	 fg = TRUE;
       };
      return fg;
    };

   EDT_clear_selection(ev->edit,NULL);

   savefg = (abort ? FALSE : TRUE);

   fg = EDT_close_file(ev->edit,savefg,abort,FALSE,FALSE);

   return fg;
};





Boolean
EDCMDOpenFile(ev,name,abort,line)
   EDT_VIEW ev;
   String name;
   Boolean abort;
   Integer line;
{
   Boolean fg;
   Boolean savefg;
   EDT_ID ei;

   if (ev->edit->no_close) return FALSE;

   EDT_clear_selection(ev->edit,NULL);

   ei = ev->edit;
   savefg = (abort ? FALSE : TRUE);

   fg = EDT_close_file(ei,savefg,abort,TRUE,FALSE);

   if (fg) fg = (EDT_open_file(ei,name) > 0);

   if (fg) {
      if (line > 1) {
	 EDTgoto(ei,line);
/*	 FILEmove(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
		     line-1,&pos);
 */
       };
    };

   return fg;
};





Boolean
EDCMDSaveFile(ev,name)
   EDT_VIEW ev;
   String name;
{
   Boolean fg;

   EDT_clear_selection(ev->edit,NULL);

   fg = EDTsecure(ev->edit,name);

   return fg;
};





Boolean
EDCMDAlterAccess(ev)
   EDT_VIEW ev;
{
   Boolean fg;

   EDT_clear_selection(ev->edit,NULL);

   fg = FILEalter_access(VIEW_FILE(ev));

   return fg;
};





/************************************************************************/
/*									*/
/*	EDCMDSplitView -- split current view				*/
/*	EDCMDRemoveView -- remove current view				*/
/*	EDCMDSizeView -- set view size					*/
/*									*/
/************************************************************************/


Boolean
EDCMDSplitView(ev)
   EDT_VIEW ev;
{
   EDT_clear_selection(ev->edit,NULL);

   EDT_split_view(ev);

   return TRUE;
};




Boolean
EDCMDRemoveView(ev)
   EDT_VIEW ev;
{
   EDT_clear_selection(ev->edit,NULL);

   EDT_remove_view(ev);

   return TRUE;
};





Boolean
EDCMDSizeView(ev,sz)
   EDT_VIEW ev;
   Integer sz;
{
   EDT_clear_selection(ev->edit,NULL);

   if (sz <= 0 || sz >= 100) return FALSE;

   EDT_size_view(ev,sz);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDLoadCommands -- load command file				*/
/*	EDCMDSaveCommandTable -- store command table			*/
/*	EDCMDLoadCommandTable -- restore command table			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
EDCMDLoadCommands(ev,file)
   EDT_VIEW ev;
   String file;
{
   return EDTload_description(file);
};




Boolean
EDCMDSaveCommandTable(ev,file)
   EDT_VIEW ev;
   String file;
{
   return EDTctbl_save(ev->edit,file);
};





Boolean
EDCMDLoadCommandTable(ev,file)
   EDT_VIEW ev;
   String file;
{
   EDTctbl_load(file);

   EDT_ctbl_rebind(ev->edit);
};





/************************************************************************/
/*									*/
/*	EDCMDSelectBegin -- start selection				*/
/*	EDCMDSelectExtend -- extend selection				*/
/*	EDCMDSelectReplace -- replace selection 			*/
/*	EDCMDSelectAbort -- abort selection				*/
/*									*/
/************************************************************************/


Boolean
EDCMDSelectBegin(ev,ep,unit,replace)
   EDT_VIEW ev;
   EDT_POS ep;
   EDT_SELECT_TYPE unit;
   Boolean replace;
{
   FILE_POS pos;

   EDT_file_position(ev,ep,&pos);

   if (unit != EDT_SELECT_VOID) ev->edit->select.unit = unit;

   EDT_start_selection(ev,&ev->edit->select,&pos,replace);
   EDT_finish_selection(ev,&ev->edit->select);

   return TRUE;
};




Boolean
EDCMDSelectExtend(ev,ep,unit,stopfg)
   EDT_VIEW ev;
   EDT_POS ep;
   EDT_SELECT_TYPE unit;
   Boolean stopfg;
{
   FILE_POS pos;

   if (!ev->edit->select.valid) {
      return EDCMDSelectBegin(ev,ep,unit,FALSE);
    };

   if (stopfg && (unit == EDT_SELECT_VOID || unit == ev->edit->select.unit)) {
      return EDCMDSelectAbort(ev);
    };

   if (unit != EDT_SELECT_VOID) ev->edit->select.unit = unit;

   EDT_file_position(ev,ep,&pos);
   EDT_extend_selection(ev,&ev->edit->select,&pos,TRUE,TRUE);
   EDT_finish_selection(ev,&ev->edit->select);

   return TRUE;
};




Boolean
EDCMDSelectReplace(ev)
   EDT_VIEW ev;
{
   if (!ev->edit->select.valid) return FALSE;

   EDT_replace_selection(&ev->edit->select,!ev->edit->select.replace);

   return TRUE;
};





Boolean
EDCMDSelectAbort(ev)
   EDT_VIEW ev;
{
   EDT_clear_selection(ev->edit,NULL);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDCopyFromFile -- copy from named file to location		*/
/*	EDCMDWriteToFile -- copy selection to a file			*/
/*									*/
/************************************************************************/


Boolean
EDCMDCopyFromFile(ev,ep,file)
   EDT_VIEW ev;
   EDT_POS ep;
   String file;
{
   FILE_POS pos;
   FILE_BUF fbf;
   Character buf[1024+1];
   Integer fid,ln;

   if (file == NULL || *file == 0) return FALSE;

   EDT_clear_selection(ev->edit,NULL);

   fbf = FILEinq_buffer("BWEDIT$_FILE");

   fid = open(file,O_RDONLY,0);
   if (fid < 0) return FALSE;

   if ((ln = read(fid,buf,1024)) > 0) {
      buf[ln] = 0;
      FILEbuffer_define(fbf,buf);
      while ((ln = read(fid,buf,1024)) > 0) {
	 buf[ln] = 0;
	 FILEbuffer_append(fbf,buf);
       };
    };

   close(fid);

   EDT_file_position(ev,ep,&pos);

   FILEinsert(VIEW_FILE(ev),fbf,1);

   return TRUE;
};





/*ARGSUSED*/

Boolean
EDCMDWriteToFile(ev,file,append)
   EDT_VIEW ev;
   String file;
   Boolean append;
{
   FILE_BUF fbf;
   String cnts;
   FILE * otf;

   if (file == NULL || *file == 0) return FALSE;
   otf = fopen(file,(append ? "a" : "w"));
   if (otf == NULL) return FALSE;

   fbf = FILEinq_buffer(SELECT_BUFFER_NAME);
   cnts = FILEbuffer_contents(fbf);

   fputs(cnts,otf);

   fclose(otf);

   free(cnts);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDUnixCommand -- run a unix command				*/
/*									*/
/************************************************************************/


Boolean
EDCMDUnixCommand(ev,cmd,inp,ep)
   EDT_VIEW ev;
   String cmd;
   Boolean inp;
   EDT_POS ep;
{
   Character infile[32],outfile[32],ctxt[256];

   if (cmd == NULL || *cmd == 0) return FALSE;

   sprintf(ctxt,"csh -c %s",cmd);

   if (inp) {
      strcpy(infile,"/tmp/EDTinXXXXXX");
      mktemp(infile);
      EDCMDWriteToFile(ev,infile,FALSE);
      strcat(ctxt," < ");
      strcat(ctxt,infile);
    };

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   strcpy(outfile,"/tmp/EDToutXXXXXX");
   mktemp(outfile);
   strcat(ctxt," > ");
   strcat(ctxt,outfile);

   exec_unix_command(ctxt);

   EDCMDCopyFromFile(ev,ep,outfile);

   unlink(infile);
   unlink(outfile);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDPrint -- print portion of file				*/
/*									*/
/************************************************************************/


Boolean
EDCMDPrint(ev,fep,tep,printer,title,format)
   EDT_VIEW ev;
   EDT_POS fep;
   EDT_POS tep;
   String printer;
   String title;
   Boolean format;
{
   FILE_POS pos1,pos2;
   String s;
   Character file[32],cmd[1024];
   FILE * otf;

   if (!VIEW_SELECT_VALID(ev) && fep->flags & EDT_POS_FLAG_SELECT_START) {
      FILEpos_copy(VIEW_FILE(ev),FILE_POS_START,&pos1);
    }
   else {
      EDT_file_position(ev,fep,&pos1);
    };

   if (!VIEW_SELECT_VALID(ev) && tep->flags & EDT_POS_FLAG_SELECT_END) {
      FILEpos_copy(VIEW_FILE(ev),FILE_POS_END,&pos2);
    }
   else {
      EDT_file_position(ev,tep,&pos2);
    };

   s = FILEcopy_text(VIEW_FILE(ev),&pos1,&pos2);
   if (s == NULL || *s == 0) {
      display_message(ev,"Nothing to print");
      return FALSE;
    };

   strcpy(file,"/tmp/EDTprtXXXXXX");
   mktemp(file);

   otf = fopen(file,"w");
   fputs(s,otf);
   fclose(otf);
   free(s);

   if (title == NULL || *title == 0) {
      if (ev->edit->filename != NULL) {
	 title = rindex(ev->edit->filename,'/');
	 if (title != NULL) ++title;
	 else title = ev->edit->filename;
       };
    };

   s = getenv("LPR");
   if (s != NULL) strcpy(cmd,s);
   else strcpy(cmd,"lpr");
   if (format) strcat(cmd," -p");
   if (printer != NULL && *printer != 0) {
      strcat(cmd," -P");
      strcat(cmd,printer);
    };
   if (title != NULL && *title != 0) {
      strcat(cmd," -J");
      strcat(cmd,title);
    };
   strcat(cmd," ");
   strcat(cmd,file);

   exec_unix_command(cmd);

   unlink(file);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDIndent -- do indentation of a region of code		*/
/*	EDCMDJoin -- join current line with next			*/
/*									*/
/************************************************************************/


Boolean
EDCMDIndent(ev,fep,tep)
   EDT_VIEW ev;
   EDT_POS fep;
   EDT_POS tep;
{
   Integer sln,eln;
   FILE_POS pos;

   EDT_file_position(ev,fep,&pos);
   sln = FILEinq_line(VIEW_FILE(ev),&pos);
   EDT_file_position(ev,tep,&pos);
   eln = FILEinq_line(VIEW_FILE(ev),&pos);

   EDT_clear_selection(ev->edit,NULL);

   indent_lines(ev,sln,eln);

   EDT_indent_on_line(ev,TRUE,TRUE);

   EDT_synch_view(ev);

   return TRUE;
};





Boolean
EDCMDJoin(ev,numspace)
   EDT_VIEW ev;
   Integer numspace;
{
   FILE_POS p1,p2,p3;
   Integer ct;

   EDT_clear_selection(ev->edit,NULL);

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&p1);

   FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_START,1,&p2);
   ct = FILEinq_linespaces(VIEW_FILE(ev));
   if (ct > 0)
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,ct,&p2);

   FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_END,
	       -1,&p3);
   FILEdelete(VIEW_FILE(ev),FILEinq_buffer("BWEDIT$_JOIN"),
		 FILE_MOVE_ABS,&p2);

   for (ct = 0; ct < numspace; ++ct) {
      FILEtypein(VIEW_FILE(ev),' ',TRUE);
    };

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDMacroCompile -- start/finish macro compilation		*/
/*	EDCMDMacroDo -- evaluate current macro				*/
/*									*/
/************************************************************************/


Boolean
EDCMDMacroCompile(ev)
   EDT_VIEW ev;
{
   EDT_macro_begin(ev);

   return TRUE;
};





Boolean
EDCMDMacroDo(ev,ct)
   EDT_VIEW ev;
   Integer ct;
{
   if (ct == 0) ct = 1;

   return EDT_macro_do(ev,ct);
};





/************************************************************************/
/*									*/
/*	EDCMDBindKeys -- add a new binding				*/
/*									*/
/************************************************************************/


Boolean
EDCMDBindKeys(ev,keys,cmd,when)
   EDT_VIEW ev;
   String keys;
   String cmd;
   String when;
{
   EDT_CMD_ID cmdid;

   if (keys == NULL || *keys == 0) return FALSE;
   if (cmd == NULL || *cmd == 0) return FALSE;

   cmdid = EDT_ctbl_find_command(cmd);
   if (cmdid < 0) return FALSE;

   EDT_ctbl_new_bind(keys,cmdid,when);
   EDT_ctbl_rebind(ev->edit);

   return TRUE;
};




/************************************************************************/
/*									*/
/*	EDCMDTranspose -- transpose characters around point		*/
/*									*/
/************************************************************************/


Boolean
EDCMDTranspose(ev)
   EDT_VIEW ev;
{
   Integer ch1,ch2;
   FILE_POS pos,lpos,npos;

   ch2 = FILEget_char(VIEW_FILE(ev),
			 FILEinq_currline(VIEW_FILE(ev)),
			 FILEinq_currchar(VIEW_FILE(ev)));
   if (ch2 == 0) ch2 = ' ';

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
   EDT_last_char(ev->edit,&pos,&lpos);
   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&lpos,&npos);

   ch1 = FILEget_char(VIEW_FILE(ev),
			 FILEinq_currline(VIEW_FILE(ev)),
			 FILEinq_currchar(VIEW_FILE(ev)));
   if (ch1 == 0) return FALSE;

   if (ch1 == '\n') {
      FILEtypein(VIEW_FILE(ev),ch2,TRUE);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
      FILEdelete(VIEW_FILE(ev),FILEinq_buffer("BWEDIT$_TRANSPOSE"),
		    FILE_MOVE_REL|FILE_MOVE_CHAR,1);
    }
   else if (ch2 == '\n') {
      FILEdelete(VIEW_FILE(ev),FILEinq_buffer("BWEDIT$_TRANSPOSE"),
		    FILE_MOVE_REL|FILE_MOVE_CHAR,1);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
      FILEtypein(VIEW_FILE(ev),ch1,TRUE);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    }
   else {
      FILEtypein(VIEW_FILE(ev),ch2,FALSE);
      FILEtypein(VIEW_FILE(ev),ch1,FALSE);
      FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&pos,&npos);
    };

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	EDCMDUserCall -- handle calls to user functions 			*/
/*										*/
/********************************************************************************/


Boolean
EDCMDUserCall(ev,rtn,arg)
   EDT_VIEW ev;
   String rtn;
   String arg;
{
   Function_Ptr rp;
   Boolean fg;

   rp = DLlookup(rtn);

   if (rp == NULL) {
      if (STREQL(rtn,"STEMdialog_edt_cmd")) rp = (Function_Ptr) STEMdialog_edt_cmd;
    };

   if (rp == NULL) return FALSE;

   fg = (*rp)(ev->edit->udata,arg);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDHOOKword_wrap -- do word wrap at eol				*/
/*									*/
/************************************************************************/


void
EDHOOKword_wrap(fid)
   FILE_ID fid;
{
   FILE_POS spos,epos,pos;
   String s;
   Integer i;
   Integer svpos;

   FILEinq_position(fid,FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&spos);
   FILEinq_position(fid,FILE_MOVE_REL|FILE_MOVE_LINE_END,0,&epos);

   s = FILEcopy_text(fid,&spos,&epos);
   i = strlen(s)-1;
   while (i > 0 && isspace(s[i])) --i;
   while (i > 0 && !isspace(s[i])) --i;
   free(s);

   if (i > 0) {
      svpos = FILEsave_pos(fid);
      FILEset_position(fid,FILEinq_currline(fid),i+2,&pos);
      FILEtypein(fid,'\n',TRUE);
      FILEinq_savepos(fid,svpos,&pos);
      FILEdelete_pos(fid,svpos);
      FILEmove(fid,FILE_MOVE_ABS,&pos,&epos);
    };
};




/************************************************************************/
/*									*/
/*	EDT_indent_on_line -- indent for current line			*/
/*									*/
/************************************************************************/


void
EDT_indent_on_line(ev,sol,force)
   EDT_VIEW ev;
   Boolean sol;
   Boolean force;
{
   FILE_POS pos;
   Integer i,j,ch;

   if (!force && !ev->edit->auto_indent) return;

   if (sol) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&pos);
    }
   else {
      i = FILEinq_currchar(VIEW_FILE(ev));
      j = FILEinq_linespaces(VIEW_FILE(ev));
      if (i-1 > j) return;
      if (i != 1) {
	 FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&pos);
       };
    };

   i = FILEinq_currline(VIEW_FILE(ev));
   ch = FILEget_char(VIEW_FILE(ev),i,1);
   if (ch != 0 && ch != '\n') i = FILEinq_linespaces(VIEW_FILE(ev));
   else i = EDT_ctbl_indent(ev->edit);

   if (i != 0) FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,i,&pos);
};





/************************************************************************/
/*									*/
/*	display_message -- display message				*/
/*									*/
/************************************************************************/


static void
display_message(ev,msg)
   EDT_VIEW ev;
   String msg;
{
   EDassoc(VIEW_FILE(ev),ev->editwin);
   EDfile_display(VIEW_FILE(ev),msg);
   EDassoc(VIEW_FILE(ev),ev->edit->window);
};





/************************************************************************/
/*									*/
/*	current_position -- check if given EDT_POS is current position	*/
/*									*/
/************************************************************************/


static Boolean
current_position(ep)
   EDT_POS ep;
{
   Boolean fg;

   if (ep == NULL) fg = TRUE;
   else if (ep->flags == 0 || ep->flags == EDT_POS_FLAG_CURRENT) fg = TRUE;
   else fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	shell_position -- automatic positioning for shell typein	*/
/*									*/
/************************************************************************/


static void
shell_position(ev)
   EDT_VIEW ev;
{
   FILE_POS pos,spos;

   if (!ev->edit->shell || ev->edit->shell_pos == NULL) return;

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
   FILEinq_savepos(VIEW_FILE(ev),ev->edit->shell_pos,&spos);

   if (FILEpos_compare(VIEW_FILE(ev),&spos,&pos) >= 0) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_END,0,&pos);
      EDT_synch_view(ev);
    };
};





/************************************************************************/
/*									*/
/*	check_end_of_line -- check if beyond eol and handle if so	*/
/*									*/
/************************************************************************/


static void
check_end_of_line(ev,ch)
   EDT_VIEW ev;
   Integer ch;
{
   if (ch < ' ') return;

   if (FILEinq_currchar(VIEW_FILE(ev)) < TEXTinq_rightchar(VIEW_TEXT(ev)))
      return;

   EDT_ctbl_end_of_line(ev->edit);
};





/************************************************************************/
/*									*/
/*	indent_lines -- indent lines of a file				*/
/*									*/
/************************************************************************/


static void
indent_lines(ev,sln,eln)
   EDT_VIEW ev;
   Integer sln;
   Integer eln;
{
   Integer ln,cl;
   Integer i,j;
   FILE_POS pos;
   FILE_BUF fbf;

   fbf = FILEinq_buffer("BWEDIT$_INDENT");

   for (ln = sln; ln <= eln; ++ln) {
      cl = FILEinq_currline(VIEW_FILE(ev));
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE|
		  FILE_MOVE_LINE_START,ln-cl,&pos);
      i = EDT_ctbl_indent(ev->edit);
      j = FILEinq_linespaces(VIEW_FILE(ev));
      if (j > i) {
	 FILEdelete(VIEW_FILE(ev),fbf,FILE_MOVE_REL|FILE_MOVE_CHAR,j-i);
       }
      else if (i > j) {
	 FILEbuffer_define(fbf," ");
	 FILEinsert(VIEW_FILE(ev),fbf,i-j);
       };
    };
};





/********************************************************************************/
/*										*/
/*	exec_unix_command -- handle a shell command				*/
/*										*/
/********************************************************************************/


static void
exec_unix_command(txt)
   String txt;
{
   String sh;
   Integer pid;

   sh = getenv("SHELL");
   if (sh == NULL) sh = "/bin/csh";

   if ((pid = vfork()) == 0) {
      execl(sh,"sh","-c",txt,0);
      _exit(127);
    };

   while (getpgrp(pid) > 0) {
      CMPXselect(0);
    };
};





/* end of edtprim.c */

