/************************************************************************/
/*									*/
/*		edtpascal.c						*/
/*									*/
/*	Pascal specific routines for EDT editor 			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define FCTSRCHSTR "^[A-Za-z_][A-Za-z0-9_]*[ ]*("

#define INDENT_A_DEFAULT 3
#define INDENT_B_DEFAULT 3
#define INDENT_C_DEFAULT 1
#define INDENT_D_DEFAULT 2

#define BLOCK_START	"(************************************************************************)\n"
#define BLOCK_MID	"(*                                                                      *)\n"
#define BLOCK_END	"(************************************************************************)\n"
#define BLOCK_SIZE	5

#define CMMT_SEARCH_LINES	75





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	indent_A = INDENT_A_DEFAULT;
static	Integer 	indent_B = INDENT_B_DEFAULT;
static	Integer 	indent_C = INDENT_C_DEFAULT;
static	Integer 	indent_D = INDENT_D_DEFAULT;
static	Boolean 	setup = FALSE;



static	String		stmtkeys[] = {
   "begin",
   "case",
   "else",
   "for",
   "goto",
   "if",
   "repeat",
   "while",
   "with",
   "else if",
   0
};





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		get_line();
static	Boolean 	key_match();
static	Boolean 	rkey_match();
static	void		define_indents();
static	void		set_indent();





/************************************************************************/
/*									*/
/*	EDTPAS_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDTPAS_init()
{
   FILE_BUF fbf;
   Character buf[10240];
   Integer i;

   fbf = FILEinq_buffer("PASCAL_BLOCK_CMMT");
   if (FILEbuffer_size(fbf) == 0) {
      strcpy(buf,BLOCK_START);
      for (i = 0; i < BLOCK_SIZE-2; ++i) strcat(buf,BLOCK_MID);
      strcat(buf,BLOCK_END);
      FILEbuffer_define(fbf,buf);
    };

   indent_A = INDENT_A_DEFAULT;
   indent_B = INDENT_B_DEFAULT;
   indent_C = INDENT_C_DEFAULT;
   indent_D = INDENT_D_DEFAULT;
   setup = FALSE;
};





/************************************************************************/
/*									*/
/*	EDTPASindent -- compute pascal indentation			*/
/*									*/
/************************************************************************/


Integer
EDTPASindent(fid)
   FILE_ID fid;
{
   Integer y;
   Integer x0,x,lastsig,nestlevel;
   Integer ch,firstch,i;
   Integer strflg,cmmtfg,eosfirst,nestfg,colfirst;
   Integer fctfg,casefg,elsefg,lblfg,rbrfg,cmafg,lbrfg,dclfg;
   Character line_buffer[1024];

   if (!setup) define_indents();

   y = FILEinq_currline(fid);
   lastsig = -1;
   nestlevel = 0;
   strflg = 0;
   cmmtfg = 0;
   eosfirst = FALSE;
   colfirst = FALSE;
   nestfg = -1;
   fctfg = FALSE;
   casefg = FALSE;
   elsefg = FALSE;
   dclfg = FALSE;
   lblfg = FALSE;
   rbrfg = FALSE;
   cmafg = -1;
   lbrfg = FALSE;

   get_line(fid,y,line_buffer);
   for (x = 0; line_buffer[x] == ' '; ++x);
   i = strlen(line_buffer)-1;
   while (i >= 0 && isspace(line_buffer[i])) --i;
   x0 = x;

   /* Check for initial sequences */

   if (line_buffer[x] == '#') return(0);
   else if (key_match(line_buffer,x,"end") ||
	       key_match(line_buffer,x,"until")) {
      nestlevel = 1;
      eosfirst = TRUE;
      rbrfg = TRUE;
    }
   else if (key_match(line_buffer,x,"begin")) lbrfg = TRUE;
   else if (key_match(line_buffer,x,"type") ||
	       key_match(line_buffer,x,"var") ||
	       key_match(line_buffer,x,"const") ||
	       key_match(line_buffer,x,"label") ||
	       key_match(line_buffer,x,"procedure") ||
	       key_match(line_buffer,x,"function")) dclfg = 1;
   else if (key_match(line_buffer,x,"otherwise")) casefg = 1;
   else if (key_match(line_buffer,x,"else")) elsefg = 1;
   else if (line_buffer[x] == '{' ||
	       (line_buffer[x] == '(' && line_buffer[x+1] == '*')) {
      return x0;
    }
   else if (i >= 0 && line_buffer[i] == ':') casefg = 1;

#ifdef CMMT_SEARCH_LINES
   for (i = 1; i  < CMMT_SEARCH_LINES; ++i) {
      if (y-i <= 0) break;
      get_line(fid,y-i,line_buffer);
      for (x = strlen(line_buffer)-1; x >= 0; --x) {
	 ch = line_buffer[x];
	 if (strflg != 0) {
	    if (ch == strflg && (x == 0 || line_buffer[x-1] != strflg)) strflg = 0;
	    else if (ch == strflg && x > 0) line_buffer[x-1] = ' ';
	  }
	 else if ((ch == '(' && line_buffer[x+1] == '*') || ch == '{') return x0;
	 else if (ch == '}' || (ch == ')' && x > 0 && line_buffer[x-1] == '*')) {
	    cmmtfg = 1;
	    break;
	  }
	 else if (ch == '\'' || ch == '"') strflg = ch;
       };
      strflg = 0;
      if (cmmtfg != 0) break;
    };
   cmmtfg = 0;
#endif

   x = 0;

   for ( ; ; ) {
      while (x <= 0) {
	 --y;
	 fctfg = FALSE;
	 if (y <= 0) goto found;
	 strflg = 0;
	 get_line(fid,y,line_buffer);
	 if (line_buffer[0] == '#') x = 0;
	 else x = strlen(line_buffer);
	 for (firstch = 0; firstch < x; firstch++) {
	    if (!isspace(line_buffer[firstch])) break;
	  };
	 while (x > 0 && isspace(line_buffer[x-1])) --x;
	 if (line_buffer[x-1] == ':' && lastsig > 0 && casefg) x = 0;
       };
      ch = line_buffer[--x];

      if (cmmtfg != 0) {
	 if (cmmtfg == 1 && ch == '(' && line_buffer[x+1] == '*') cmmtfg = 0;
	 else if (ch == '{') cmmtfg = 0;
	 continue;
       };
      if (strflg != 0) {
	 if (ch == strflg && (x == 0 || line_buffer[x-1] != strflg)) {
	    strflg = 0;
	    ch = 'X';
	  }
	 else if (ch == strflg && x > 0) line_buffer[x-1] = ' ';
	 else continue;
       };

      if ((ch == '(' && line_buffer[x+1] == '*') || ch == '{') return x0;

      if (ch == ')' && x > 0 && line_buffer[x-1] == '*') {
	 cmmtfg = 1;
	 --x;
	 continue;
       };

      if (isspace(ch)) ;
      else if (ch == '\'' || ch == '"') strflg = ch;
      else if (ch == '(' || rkey_match(line_buffer,&x,"begin") ||
		  rkey_match(line_buffer,&x,"repeat") ||
		  rkey_match(line_buffer,&x,"record")) {
	 if (lastsig < 0) casefg = 0;
	 if (nestlevel == 0 && dclfg) dclfg = FALSE;
	 if (nestlevel > 0) {
	      --nestlevel;
	      if (nestlevel == 0) {
		 lastsig = x;
		 if (rbrfg) goto found;
		 cmafg = -1;
		 continue;
	       };
	  }
	 else if (lastsig < 0 && x > 0) nestfg = x;
	 else if (x == 0 && lastsig < 0) { nestfg = 0; goto found; }
	 else goto found;
	 cmafg = -1;
       }
      else if (rkey_match(line_buffer,&x,"case")) {
	 if (nestlevel > 0) {
	      --nestlevel;
	      if (nestlevel == 0) lastsig = x;
	  }
	 else {
	    casefg = FALSE;
	    lastsig = x;
	    nestfg = x;
	    cmafg = -1;
	  };
       }
      else if (rkey_match(line_buffer,&x,"type") ||
		  rkey_match(line_buffer,&x,"var") ||
		  rkey_match(line_buffer,&x,"const") ||
		  rkey_match(line_buffer,&x,"label")) {
	 if (lastsig < 0) {
	    casefg = 0;
	    nestfg = x;
	    goto found;
	  }
	 else if (cmafg == y && !eosfirst) cmafg = y+1;
	 else if (x >= 2*indent_B) lastsig = x;
       }
      else if (rkey_match(line_buffer,&x,"procedure") ||
		  rkey_match(line_buffer,&x,"function")) {
	 lastsig = -1;
	 if (x == 0) nestfg = 0;
	 goto found;
       }
      else if (ch == '}') cmmtfg = 2;
      else if (rkey_match(line_buffer,&x,"end")) ++nestlevel;
      else if (rkey_match(line_buffer,&x,"until")) ++nestlevel;
      else if (ch == ')') ++nestlevel;
      else if (ch == ';') {
	 if (nestlevel <= 0) {
	    if (lastsig >= 0) goto found;
	    eosfirst = TRUE;
	  };
       }
      else if (ch == ':') {
	 if (nestlevel <= 0) {
	    if (lastsig <= 0) colfirst = TRUE;
	  };
       }
      else if (ch == ',') {
	 if (nestlevel == 0) cmafg = y;
       }
      else if (nestlevel <= 0) {
	 lastsig = x;
	 if (x == 0) {
	    if (fctfg && indent_D != 0) nestfg = 0;
	    lastsig = -1;
	    goto found;
	  };
       };

      if (x == firstch && nestlevel == 0) {
	 if (colfirst) goto found;
	 if (!eosfirst) {
	    for (i = 0; stmtkeys[i] != NULL; ++i) {
	       if (key_match(line_buffer,x,stmtkeys[i])) {
		  lastsig = x;
		  goto found;
		};
	     };
	  };
       };
    };

found:
   if (lastsig < 0 && nestfg < 0) x = 0;
   else if (lastsig < 0) x = indent_B;
   else if (nestfg >= lastsig+indent_B) x = nestfg+indent_B;
   else if (nestfg >= 0) x = lastsig+indent_B;
   else if (!eosfirst && (cmafg < 0 || cmafg == y)) x = lastsig+indent_A;
   else if (colfirst) x = lastsig + indent_A;
   else x = lastsig;

   if (x > 8 && dclfg) dclfg = FALSE;

   if (lbrfg && x == indent_B) x = 0;
   if (rbrfg && x != 0) x += indent_C;
   else if (casefg || elsefg) x -= indent_B;
   else if (lblfg) x -= 2*indent_B;
   else if (dclfg) x = 0;
   if (x < 0) x = 0;

   return x;
};





/************************************************************************/
/*									*/
/*	get_line -- get given line					*/
/*									*/
/************************************************************************/


static void
get_line(fid,l,buf)
   FILE_ID fid;
   Integer l;
   String buf;
{
   FILE_POS p1,p2;
   String s;
   Integer cl;

   cl = FILEinq_currline(fid);

   FILEinq_position(fid,FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_START,
		       l-cl,&p1);
   FILEinq_position(fid,FILE_MOVE_REL|FILE_MOVE_LINE|FILE_MOVE_LINE_END,
		       l-cl,&p2);

   s = FILEcopy_text(fid,&p1,&p2);

   if (s != NULL) {
      strcpy(buf,s);
      free(s);
    }
   else buf[0] = 0;
};





/************************************************************************/
/*									*/
/*	key_match -- check for keyword					*/
/*	rkey_mathc -- check for keyword at tail 			*/
/*									*/
/************************************************************************/


static Boolean
key_match(buf,k,key)
   String buf;
   Integer k;
   String key;
{
   Integer i;

   if (k > 0 && (isalnum(buf[k-1]) || buf[k-1] == '_')) return FALSE;

   for (i = 0; key[i] != 0; ++i) {
      if (buf[k+i] != key[i] && tolower(buf[k+i]) != key[i]) return FALSE;
    };

   if (isalnum(buf[k+i]) || buf[k+i] == '_') return FALSE;

   return TRUE;
};





static Boolean
rkey_match(buf,kp,key)
   String buf;
   Integer * kp;
   String key;
{
   Integer i,ln;
   Integer k;

   k = *kp;

   if (isalnum(buf[k+1]) || buf[k+1] == '_') return FALSE;

   ln = strlen(key);

   for (i = 0; i < ln; ++i) {
      if (buf[k-i] != key[ln-i-1] && tolower(buf[k-i]) != key[ln-i-1]) return FALSE;
    };

   if (k >= ln && (isalnum(buf[k-ln]) || buf[k-ln] == '_')) return FALSE;

   *kp = k-ln+1;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	define_indents -- get user indentation definitions if any	*/
/*									*/
/************************************************************************/


static void
define_indents()
{
   setup = TRUE;

   set_indent("indent_A",&indent_A);
   set_indent("indent_B",&indent_B);
   set_indent("indent_C",&indent_C);
   set_indent("indent_D",&indent_D);
};





static void
set_indent(name,varp)
   String name;
   Integer * varp;
{
   EDT_VAR ev;
   Integer v;

   ev = EDT_ctbl_find_global(name);

   if (ev != NULL && ev->type == EDT_VAR_TYPE_INT) {
      v = (Integer) ev->dflt;
      if (v >= 0 && v <= 16) *varp = v;
    };
};





/* end of edtpascal.c */



