/************************************************************************/
/*									*/
/*		edtmain.c						*/
/*									*/
/*	General routines for editor interface package			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		visible_windows();
static	int		edit_subwin_control();



/************************************************************************/
/*									*/
/*	Local variables 						*/
/*									*/
/************************************************************************/


static	Boolean 	edt_inited = FALSE;
	PROT_DECL;




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define EDIT_WIN	2
#define MSG_WIN 	3


static LEAF_DATA file_edit_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { EDIT_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(MSG_WIN),
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   { MSG_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY, LEAF_COORD_RX, LEAF_COORD_TEXT_LINE },
	NULL, NULL },
   LEAF_END
};



static STEM_PDM_DATA empty_menu[] = {
   { STEM_PSTATE_END, 0, 0 }
};





/************************************************************************/
/*									*/
/*	EDT_init -- package initialization				*/
/*									*/
/************************************************************************/


void
EDT_init()
{
   FILE_BUF bf;

   PROT_INIT;
   PROTECT;
   if (!edt_inited) {
      edt_inited = TRUE;
      EDT_win_init();
      EDT_view_init();
      EDT_input_init();
      EDT_eval_init();
      EDT_ctbl_init();
      EDT_vars_init();
      EDT_prim_init();
      EDT_help_init();
      EDT_syn_init();
      EDT_exec_init();
      EDT_abbrev_init();
      EDTCC_init();
      EDTPAS_init();

      bf = FILEinq_buffer(X_BUFFER_NAME);
      FILEbuffer_window(bf,ASHinq_top());
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	EDTstartup -- insure editor is fully initialized		*/
/*									*/
/************************************************************************/


void
EDTstartup()
{
   EDT_init();

   EDT_ctbl_startup();
};






/************************************************************************/
/*									*/
/*	EDTedit_file -- put up default editor on a file 		*/
/*									*/
/************************************************************************/


EDT_ID
EDTedit_file(w,name)
   ASH_WINDOW w;
   String name;
{
   register EDT_DATA ed;
   register EDT_ID ei;

   if (!ASHlock(w)) return NULL;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = TRUE;
   ed->temporary = FALSE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = TRUE;
   ed->mode = NULL;
   ed->bounds_rtn = NULL;
   ed->input_rtn = NULL;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   LEAFsetup_window(w,file_edit_leaf,ed);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTdefine_editor(LEAFinq_window(w,EDIT_WIN,0),name,ed,NULL,
			    LEAFinq_window(w,MENU_WIN,0),
			    LEAFinq_window(w,MSG_WIN,0));

   if (ei != NULL) visible_windows(w);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTview_file -- put up default editor on a file (ro)		*/
/*									*/
/************************************************************************/


EDT_ID
EDTview_file(w,name)
   ASH_WINDOW w;
   String name;
{
   register EDT_DATA ed;
   register EDT_ID ei;

   if (!ASHlock(w)) return NULL;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = TRUE;
   ed->create = TRUE;
   ed->temporary = FALSE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = TRUE;
   ed->mode = NULL;
   ed->bounds_rtn = NULL;
   ed->input_rtn = NULL;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   LEAFsetup_window(w,file_edit_leaf,ed);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTdefine_editor(LEAFinq_window(w,EDIT_WIN,0),name,ed,NULL,
			    LEAFinq_window(w,MENU_WIN,0),
			    LEAFinq_window(w,MSG_WIN,0));

   if (ei != NULL) visible_windows(w);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTedit_string -- put up editor on string in window		*/
/*									*/
/************************************************************************/


EDT_ID
EDTedit_string(w,txt)
   ASH_WINDOW w;
   String txt;
{
   register EDT_DATA ed;
   register EDT_ID ei;
   register FILE_BUF bf;

   if (!ASHlock(w)) return NULL;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = FALSE;
   ed->temporary = TRUE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = FALSE;
   ed->mode = "TEXT";
   ed->bounds_rtn = NULL;
   ed->input_rtn = NULL;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   ei = EDTdefine_editor(w,NULL,ed,NULL,NULL,NULL);

   if (ei != NULL && txt != NULL) {
      PROTECT;
      if (LPROTECT(ei)) {
	 bf = FILEinq_buffer("BWEDIT$_edit_string");
	 FILEbuffer_define(bf,txt);
	 FILEinsert(ei->edit_file,bf,1);
	 EDT_save_command(ei,TRUE,FALSE,FALSE);
	 LUNPROTECT(ei);
       };
      UNPROTECT;
    };

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTedit_interactive -- put up an interactive editor window	*/
/*									*/
/************************************************************************/


EDT_ID
EDTedit_interactive(w,name,fct)
   ASH_WINDOW w;
   String name;
   Function_Ptr fct;
{
   register EDT_DATA ed;
   register EDT_ID ei;

   if (!ASHlock(w)) return NULL;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = TRUE;
   ed->temporary = TRUE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = TRUE;
   ed->savectl = FALSE;
   ed->mode = NULL;
   ed->bounds_rtn = NULL;
   ed->input_rtn = fct;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   LEAFsetup_window(w,file_edit_leaf);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTdefine_editor(LEAFinq_window(w,EDIT_WIN,0),name,ed,NULL,
			    LEAFinq_window(w,MENU_WIN,0),
			    LEAFinq_window(w,MSG_WIN,0));

   if (ei != NULL) visible_windows(w);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTedit_shell -- put up a csh window				*/
/*									*/
/************************************************************************/


EDT_EXEC
EDTedit_shell(w)
   ASH_WINDOW w;
{
   register EDT_EXEC ei;
   String sh;

   sh = (String) getenv("SHELL");
   if (sh == NULL) sh = "/bin/csh";

   if (!ASHlock(w)) return NULL;

   LEAFsetup_window(w,file_edit_leaf);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTexec(LEAFinq_window(w,EDIT_WIN,0),
		   LEAFinq_window(w,MENU_WIN,0),
		   LEAFinq_window(w,MSG_WIN,0),
		   sh,
		   sh,
		   "-i",
		   0);

   if (ei != NULL) visible_windows(w);

   EDTexec_prohibit_close(ei);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTedit_pty -- put up a pty window				*/
/*									*/
/************************************************************************/


EDT_EXEC
EDTedit_pty(w,buf)
   ASH_WINDOW w;
   String buf;
{
   register EDT_EXEC ei;

   if (!ASHlock(w)) return NULL;

   LEAFsetup_window(w,file_edit_leaf);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTptty(LEAFinq_window(w,EDIT_WIN,0),
		   LEAFinq_window(w,MENU_WIN,0),
		   LEAFinq_window(w,MSG_WIN,0),
		   buf);

   if (ei != NULL) visible_windows(w);

   EDTexec_prohibit_close(ei);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTedit_output -- set up output viewer				*/
/*									*/
/************************************************************************/


EDT_ID
EDTedit_output(w,ud)
   ASH_WINDOW w;
   Universal ud;
{
   register EDT_DATA ed;
   register EDT_ID ei;

   if (!ASHlock(w)) return NULL;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = TRUE;
   ed->temporary = FALSE;
   ed->scroll = TRUE;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = FALSE;
   ed->mode = NULL;
   ed->bounds_rtn = NULL;
   ed->input_rtn = NULL;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   LEAFsetup_window(w,file_edit_leaf,ed);
   LEAFredraw(w);

   STEMpdm_define(LEAFinq_window(w,MENU_WIN,0),NULL,empty_menu);

   ei = EDTdefine_editor(LEAFinq_window(w,EDIT_WIN,0),NULL,ed,ud,
			    LEAFinq_window(w,MENU_WIN,0),
			    LEAFinq_window(w,MSG_WIN,0));

   if (ei != NULL) visible_windows(w);

   ASHunlock(w);

   return ei;
};





/************************************************************************/
/*									*/
/*	EDTwait -- wait for editor to complete				*/
/*									*/
/************************************************************************/


void
EDTwait(ei)
   EDT_ID ei;
{
   if (ei == NULL) return;

   ei->wait = TRUE;

   while (ei->window != NULL) {
      RIPuser_pick(NULL);
    };

   ei->wait = FALSE;

   EDTclose(ei);
};





/************************************************************************/
/*									*/
/*	EDT_error -- handle error conditions				*/
/*									*/
/************************************************************************/


void
EDT_error(msg,p1,p2,p3)
   String msg;
   String p1,p2,p3;
{
   Character buf[1024];

   sprintf(buf,msg,p1,p2,p3);

   fprintf(stderr,"EDT: %s\n",buf);
   fflush(stderr);
};





/********************************************************************************/
/*										*/
/*	visible_windows -- make interior windows visible			*/
/*										*/
/********************************************************************************/


static void
visible_windows(w)
   ASH_WINDOW w;
{
   ASHvisible(LEAFinq_window(w,EDIT_WIN,0),TRUE);
   ASHvisible(LEAFinq_window(w,MENU_WIN,0),TRUE);
   ASHvisible(LEAFinq_window(w,MSG_WIN,0),TRUE);

   ASHset_control(LEAFinq_window(w,EDIT_WIN,0),edit_subwin_control);
};





/************************************************************************/
/*									*/
/*	edit_subwin_control -- handle messages for edit subwindow	*/
/*									*/
/************************************************************************/


static int
edit_subwin_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   ASH_WINDOW p;

   if (STREQL(msg,"ASH$REMOVE")) {
      p = ASHinq_parent(w);
      if (ASHinq_valid_window(p)) ASHremove(p);
    };

   return ASH_CONTROL_REJECT;
};





/* end of edtmain.c */
