/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

%{

/************************************************************************/
/*									*/
/*		edtlex.l						*/
/*									*/
/*	Lexical definitions for editor language 			*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Local storage and definitions					*/
/*									*/
/************************************************************************/


#define MAX_NEST	16
static	Integer 	nest_level = 0;
static	FILE *		nest_files[MAX_NEST];
static	String		nest_names[MAX_NEST];
static	Integer 	nest_lines[MAX_NEST];
static	String		cur_file;

static	void		source();
static	String		fix_string();


%}


%START	HELP

WHITE	([ \t\r\n])+
WHITE0	([ \t\r\n])*

INT	(\-?[0-9]+)

STRING	(\"([^\n"\\]|(\\.))*\")

ID	([@a-zA-Z_$][-a-zA-Z_$0-9]*)
QID	(\'([^\n'\\]|(\\.))*\')

SINGLE	([():=;,<>*&\[\]#])

CMMT1	("--".*)
CMMT2	("/*"([^\n*]|(\*+[^/]))*"*/")
CMMT	({CMMT1}|{CMMT2})

KEYTPFX (A|C|M|CM|MC|E|CE|EC)
KEYTTXT ((([^\n\]])|"\\]")+)
KEYTDEF ({KEYTPFX}\[{KEYTTXT}\])
KEYFPFX (F|SF|FS|CF|FC|UF|FU|MF|FM)
KEYFTXT ({INT}((","{INT})*))
KEYFDEF ({KEYFPFX}\[{KEYFTXT}\])
KEYMDEF (P\[{KEYTTXT}\])
KEYDEF	({KEYTDEF}|{KEYFDEF}|{KEYMDEF})

HELPTEXT ([^}\n]+)


%%

<HELP>{HELPTEXT}	{ yylval.string = SALLOC(yytext);
			  return LX_HELPTEXT;
			}
<HELP>"\n"{HELPTEXT}    { yylval.string = SALLOC(yytext);
			  return LX_HELPTEXT;
			}
<HELP>"\n"              { yylval.string = SALLOC("\n");
			  return LX_HELPTEXT;
			}
<HELP>"}"               { BEGIN 0;
			  return yytext[0];
			}

"COMMAND"               { return LX_COMMAND; }
"BIND"                  { return LX_BIND; }
"HOOK"                  { return LX_HOOK; }
"LOAD"                  { return LX_LOAD; }
"SOURCE"                { return LX_SOURCE; }
"DEFINE"                { return LX_DEFINE; }
"WHEN"                  { return LX_WHEN; }
"HELP"                  { return LX_HELP; }
"TYPEDEF"               { return LX_TYPEDEF; }

"TRUE"                  { return LX_TRUE; }
"FALSE"                 { return LX_FALSE; }
"IGNORE"                { return LX_IGNORE; }
"UNDEFINED"             { return LX_UNDEFINED; }

"Integer"               { return LX_INT_TYPE; }
"String"                { return LX_STR_TYPE; }
"Boolean"               { return LX_BOOL_TYPE; }
"Buffer"                { return LX_BUF_TYPE; }
"Position"              { return LX_POS_TYPE; }

{INT}			{ yylval.intval = atol(yytext);
			  return LX_INT;
			}
{STRING}		{ yytext[yyleng-1] = 0;
			  yylval.string = fix_string(&yytext[1]);
			  return LX_STRING;
			}

{ID}			{ yylval.string = SALLOC(yytext);
			  return LX_ID;
			}
{QID}			{ yytext[yyleng-1] = 0;
			  yylval.string = SALLOC(&yytext[1]);
			  return LX_ID;
			}

{KEYDEF}		{ yylval.string = SALLOC(yytext);
			  return LX_KEYDEF;
			}

"{"                     { BEGIN HELP;
			  return yytext[0];
			}

{SINGLE}		{ return yytext[0]; }

{CMMT}			;
{WHITE} 		;

.			{ return yytext[0]; }


%%



/************************************************************************/
/*									*/
/*	first_file -- set to read the first file			*/
/*	source -- handle nested source files				*/
/*									*/
/************************************************************************/


static Boolean
first_file(filename)
   String filename;
{
   nest_level = 0;
   cur_file = SALLOC(filename);
   yyin = fopen(filename,"r");
   yylineno = 1;

   return (yyin != NULL);
};





static void
source(filename)
   String filename;
{
   register FILE * inf;

   inf = fopen(filename,"r");
   if (inf == NULL) {
      EDT_error("Couldn't open file %s",filename);
      return;
    };

   nest_names[nest_level] = cur_file;
   cur_file = SALLOC(filename);
   nest_lines[nest_level] = yylineno;
   yylineno = 1;
   nest_files[nest_level++] = yyin;
   yyin = inf;
};




yywrap()
{
   fclose(yyin);

   SFREE(cur_file);

   if (nest_level == 0) return 1;

   yyin = nest_files[--nest_level];
   yylineno = nest_lines[nest_level];
   cur_file = nest_names[nest_level];

   return 0;
};





static String
fix_string(txt)
   String txt;
{
   register String s,t;
   register Integer i;

   s = txt;
   t = txt;

   while (*s != 0) {
      if (*s == '"' && *(s+1) == '&') {
	 ++s;
	 ++s;
	 while (*s++ != '"');
       }
      else {
	 if (*s != '\\') i = *s++;
	 else {
	    ++s;
	    if (isdigit(*s)) {
	       i = *s++ - '0';
	       while (isdigit(*s)) i = i*8 + *s++ - '0';
	     }
	    else switch (i = *s++) {
	       case 'n' :
		  i = '\n';
		  break;
	       case 'r' :
		  i = '\r';
		  break;
	       case 't' :
		  i = '\t';
		  break;
	       case 'b' :
		  i = '\t';
		  break;
	       case 'f' :
		  i = '\f';
		  break;
	       default :
		  break;
	     };
	  };
	 *t++ = i;
       };
    };

   *t = 0;

   return SALLOC(txt);
};





/* end of edtlex.l */



















