/************************************************************************/
/*									*/
/*		edtinput.c						*/
/*									*/
/*	Editor input processing routines				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "edt_local.h"



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define PARA_SEARCH_STRING	"^$"
#define WORD_SEARCH_STRING	"[A-Za-z0-9_$]+"




/************************************************************************/
/*									*/
/*	Local structures						*/
/*									*/
/************************************************************************/


typedef struct _TRACK_INFO {
   EDT_SELECT	select;
   Integer	btn;
   Integer	count;
   Boolean	valid;
} TRACK_INFO;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	deselect_on_tap;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	edit_track();
static	void		real_position();
static	void		closer_position();
static	void		raw_input();




/************************************************************************/
/*									*/
/*	EDT_input_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
EDT_input_init()
{
   deselect_on_tap = -1;
};





/************************************************************************/
/*									*/
/*	EDT_button_handler -- handle user hits in edit window		*/
/*									*/
/************************************************************************/


Integer
EDT_button_handler(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   Boolean fg;
   EDT_ID ei;
   EDT_VIEW ev;
   EDT_SELECT_TYPE unit;
   EDT_SELECT sel;
   FILE_POS pos,npos,epos,spos;
   TRACK_INFO tin;
   Integer esc,ln;
   EDT_VAR var;
   Boolean mfg;

   ev = (EDT_VIEW) RIPinq_data(rgn);
   if (ev == NULL) return FALSE;
   ei = ev->edit;
   sel = &ei->select;
   esc = ei->escape;
   ei->escape = 0;

   if (deselect_on_tap < 0) {
      var = EDT_ctbl_find_global("deselect_on_tap");
      if (var != NULL && var->type == EDT_VAR_TYPE_BOOL) {
	 deselect_on_tap = var->dflt;
       }
      else deselect_on_tap = FALSE;
    };

   ASHmap(ASHinq_top_window(ev->editwin),x,y,ev->editwin,&x,&y);

   if (!LPROTECT(ei)) return FALSE;

   mfg = FALSE;

   if (btn & RIP_BTN_NONE) {
      if (ei->raw_mode && !ei->hold_mode && ch < 128) {
	 raw_input(ei,ch);
	 fg = TRUE;
       }
      else {
	 if (ei->raw_mode && !ei->hold_mode) {
	    if (FKEY_TEST_META(ch)) ch -= FKEY_META_FIRST;
	  };
	 if (esc == 0 && ch == '\33') {
	    ei->escape = 1;
	    fg = TRUE;
	  }
	 else if (esc == 1 && ch == '[') {
	    ei->escape = 2;
	    fg = TRUE;
	  }
	 else {
	    if (esc != 0 && ch < 128) ch += EDT_ESCAPE_FIRST;
	    ei->last_key_struck = (Universal) ch;
	    fg = EDT_eval_cmd(ev,EDT_ctbl_map_key(ei,ch));
	  };
       };
    }
   else if ((btn & RIP_BTN_UP) != 0 && (btn & RIP_BTN_TAP) == 0) {
      EDT_clear_selection(ei,NULL);
      fg = TRUE;
    }
   else if ((btn & (RIP_BTN_SHIFT|RIP_BTN_META)) != 0) {
      if (deselect_on_tap) EDT_clear_selection(ei,NULL);
      TEXTcorrelate(VIEW_TEXT(ev),x,y,TRUE,&pos);
      mfg = TRUE;
      fg = TRUE;
    }
   else {
      mfg = TRUE;
      TEXTcorrelate(VIEW_TEXT(ev),x,y,TRUE,&pos);
      tin.select = sel;
      tin.btn = btn & (RIP_BTN_LEFT|RIP_BTN_MID|RIP_BTN_RIGHT);
      tin.count = 0;
      tin.valid = TRUE;
      if ((btn & (RIP_BTN_LEFT|RIP_BTN_MID)) || !sel->valid) {
	 if (deselect_on_tap) EDT_clear_selection(ei,NULL);
	 if (btn & RIP_BTN_MID) unit = EDT_SELECT_WORD;
	 else if (btn & RIP_BTN_RIGHT) unit = EDT_SELECT_LINE;
	 else unit = EDT_SELECT_CHAR;

	 if (btn & RIP_BTN_TAP) {
	    real_position(ei,unit,&pos,&spos,&epos);
	    ln = FILEinq_line(ei->edit_file,&spos);
	    FILEinq_position(ei->edit_file,FILE_MOVE_START|FILE_MOVE_LINE|FILE_MOVE_LINE_END,
				ln-1,&npos);
	    if (FILEinq_char(ei->edit_file,&npos) < FILEinq_char(ei->edit_file,&spos)) {
	       FILEpos_copy(ei->edit_file,&npos,&spos);
	     };
	    EDT_next_char(ei,&epos,&epos);
	    closer_position(ei,&pos,&spos,&epos,&npos);
	    FILEmove(ei->edit_file,FILE_MOVE_ABS,&npos,&pos);
	    fg = TRUE;
	  }
	 else {
	    sel->unit = unit;
	    fg = ((btn & RIP_BTN_CTRL) != 0);
	    if (!fg && unit == EDT_SELECT_CHAR) {
	       FILEmove(ei->edit_file,FILE_MOVE_ABS,&pos,&npos);
	       tin.valid = FALSE;
	     }
	    else {
	       EDT_start_selection(ev,sel,&pos,fg);
	     };
	    ei->temp = (Universal) &tin;
	    RIPtrack(edit_track,-1,ev->editwin,FALSE);
	    fg = TRUE;
	  };
       }
      else if ((btn & (RIP_BTN_RIGHT)) && sel->valid) {
	 if (!sel->replace && (btn & RIP_BTN_CTRL) != 0)
	    EDT_replace_selection(sel,TRUE);
	 sel->anchor = EDT_ANCHOR_SET;
	 fg = EDT_extend_selection(ev,sel,&pos,FALSE,FALSE);
	 if (fg && (btn&RIP_BTN_TAP) == 0) {
	    ei->temp = (Universal) &tin;
	    RIPtrack(edit_track,-1,ev->editwin,FALSE);
	  }
	 else {
	    EDT_finish_selection(ev,sel);
	  };
       }
      else fg = FALSE;
      if (fg) EDT_save_command(ei,TRUE,TRUE,FALSE);
    };

   if (mfg && ei->data->mouse_rtn != NULL) {
      if (ei->select.valid) {
	 (*ei->data->mouse_rtn)(ei->udata,btn,
				   FILEinq_line(ei->edit_file,&pos),
				   FILEinq_char(ei->edit_file,&pos),
				   FILEinq_line(ei->edit_file,&ei->select.begin),
				   FILEinq_char(ei->edit_file,&ei->select.begin),
				   FILEinq_line(ei->edit_file,&ei->select.end),
				   FILEinq_char(ei->edit_file,&ei->select.end));
       }
      else {
	 (*ei->data->mouse_rtn)(ei->udata,btn,
				   FILEinq_line(ei->edit_file,&pos),
				   FILEinq_char(ei->edit_file,&pos),
				   0,0,0,0);
       };
    };

   LUNPROTECT(ei);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDT_clear_selection -- remove current selection 		*/
/*									*/
/************************************************************************/


Boolean
EDT_clear_selection(ei,bufname)
   EDT_ID ei;
   String bufname;
{
   EDT_SELECT sel;
   FILE_POS pos;
   Boolean fg;

   sel = &ei->select;

   fg = FALSE;
   if (sel->view != NULL && sel->view->edit != NULL) {
      if (sel->view->edit != ei) abort();
      if (sel->replace) TEXTmark_end(VIEW_TEXT(sel->view));
      else TEXTmark2_end(VIEW_TEXT(sel->view));
      if (bufname != NULL && sel->replace && sel->valid) {
	 if (LPROTECT(ei)) {
	    FILEmove(ei->edit_file,FILE_MOVE_ABS,&sel->begin,&pos);
	    FILEdelete(ei->edit_file,FILEinq_buffer(bufname),
			  FILE_MOVE_ABS,&sel->end);
	    LUNPROTECT(ei);
	  };
       };
      sel->valid = FALSE;
      sel->view = NULL;
      fg = TRUE;
    }
   else if (ei->auto_select_all && bufname != NULL && ei->edit_file != NULL) {
      ei->auto_select_all = FALSE;
      if (LPROTECT(ei)) {
	 FILEmove(ei->edit_file,FILE_MOVE_START,0,&pos);
	 FILEdelete(ei->edit_file,FILEinq_buffer("BWEDIT$_AUTOSELECT"),FILE_MOVE_END,0);
	 LUNPROTECT(ei);
       };
      fg = TRUE;
    };

   ei->auto_select_all = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	EDT_start_selection -- begin selection				*/
/*									*/
/************************************************************************/


void
EDT_start_selection(ev,sel,fp,repfg)
   EDT_VIEW ev;
   EDT_SELECT sel;
   FILE_POS * fp;
   Boolean repfg;
{
   FILE_POS spos,epos,pos,npos;
   EDT_ID ei;

   ei = ev->edit;

   if (ei->always_replace) repfg = TRUE;

   EDT_clear_selection(ei,NULL);

   sel->view = ev;

   real_position(ei,sel->unit,fp,&spos,&epos);
   EDT_next_char(ei,&epos,&epos);

   sel->begin = spos;
   sel->end = epos;
   sel->valid = FALSE;
   sel->replace = repfg;
   sel->anchor = EDT_ANCHOR_START;

   if (sel->replace) {
      TEXTmark_begin(VIEW_TEXT(ev),&spos);
      TEXTmark_until(VIEW_TEXT(ev),&epos);
    }
   else {
      TEXTmark2_begin(VIEW_TEXT(ev),&spos);
      TEXTmark2_until(VIEW_TEXT(ev),&epos);
    };

   closer_position(ei,fp,&spos,&epos,&npos);

   FILEmove(ei->edit_file,FILE_MOVE_ABS,&npos,&pos);
};





/************************************************************************/
/*									*/
/*	EDT_extend_selection -- extend current selection		*/
/*									*/
/************************************************************************/


Boolean
EDT_extend_selection(ev,sel,fp,both,real)
   EDT_VIEW ev;
   EDT_SELECT sel;
   FILE_POS * fp;
   Boolean both;
   Boolean real;
{
   FILE_POS epos,spos,pos,npos;
   EDT_ID ei;
   Boolean bfg,efg,force;
   Integer sts;

   ei = ev->edit;
   if (sel->view == NULL || sel->view != ev) return FALSE;

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&pos);
   if (FILEpos_compare(ei->edit_file,&pos,&sel->begin) <= 0) sts = 1;
   else if (FILEpos_compare(ei->edit_file,&pos,&sel->end) >= 0) sts = 2;
   else sts = 0;

   real_position(ei,sel->unit,fp,&spos,&epos);

   force = FALSE;

   if (sel->anchor == EDT_ANCHOR_START) {
      if (FILEpos_compare(ei->edit_file,fp,&sel->begin) < 0) {
	 FILEpos_copy(ei->edit_file,&sel->begin,&sel->end);
	 sel->anchor = EDT_ANCHOR_END;
	 force = TRUE;
       }
    }
   else if (sel->anchor == EDT_ANCHOR_END) {
      if (FILEpos_compare(ei->edit_file,fp,&sel->end) > 0) {
	 FILEpos_copy(ei->edit_file,&sel->end,&sel->begin);
	 sel->anchor = EDT_ANCHOR_START;
	 force = TRUE;
       };
    };

   if (!both && sel->anchor == EDT_ANCHOR_START) {
      bfg = FALSE;
      efg = TRUE;
    }
   else if (!both && sel->anchor == EDT_ANCHOR_END) {
      bfg = TRUE;
      efg = FALSE;
    }
   else {
      if (FILEpos_compare(ei->edit_file,&spos,&sel->begin) < 0) bfg = TRUE;
      else bfg = FALSE;
      if (FILEpos_compare(ei->edit_file,&epos,&sel->end) > 0) efg = TRUE;
      else efg = FALSE;
      if (!bfg && !efg) {
	 if (FILEpos_compare(ei->edit_file,&spos,&sel->begin) > 0 &&
		FILEpos_compare(ei->edit_file,&spos,&sel->end) < 0) {
	    if (!real && sts == 1) bfg = TRUE;
	    else efg = TRUE;
	  };
       };
    };

   if (sel->anchor == EDT_ANCHOR_SET) {
      if (efg) sel->anchor = EDT_ANCHOR_START;
      else if (bfg) sel->anchor = EDT_ANCHOR_END;
    };

   if (bfg && FILEpos_compare(ei->edit_file,&spos,&sel->begin) == 0) bfg = FALSE;

   if (force) {
      if (bfg) FILEpos_copy(ei->edit_file,&spos,&sel->begin);
      else FILEpos_copy(ei->edit_file,&sel->begin,&spos);
      if (!efg) FILEpos_copy(ei->edit_file,&sel->end,&epos);
      if (sel->replace) {
	 TEXTmark_end(VIEW_TEXT(ev));
	 TEXTmark_begin(VIEW_TEXT(ev),&spos);
	 if (!efg) TEXTmark_until(VIEW_TEXT(ev),&epos);
       }
      else {
	 TEXTmark2_end(VIEW_TEXT(ev));
	 TEXTmark2_begin(VIEW_TEXT(ev),&spos);
	 if (!efg) TEXTmark2_until(VIEW_TEXT(ev),&epos);
       };
    }
   else if (bfg) {
      FILEpos_copy(ei->edit_file,&spos,&sel->begin);
      if (!efg) FILEpos_copy(ei->edit_file,&sel->end,&epos);
      if (sel->replace) TEXTmark_from(VIEW_TEXT(ev),&spos);
      else TEXTmark2_from(VIEW_TEXT(ev),&spos);
    };

   if (efg) {
      if (!real) EDT_next_char(ei,&epos,&epos);
      if (FILEpos_compare(ei->edit_file,&epos,&sel->end) != 0) {
	 FILEpos_copy(ei->edit_file,&epos,&sel->end);
	 if (sel->replace) TEXTmark_until(VIEW_TEXT(ev),&epos);
	 else TEXTmark2_until(VIEW_TEXT(ev),&epos);
       };
    };

   if (sts == 0) closer_position(ei,fp,&spos,&epos,&npos);
   else if (sts == 1) FILEpos_copy(ei->edit_file,&spos,&npos);
   else FILEpos_copy(ei->edit_file,&epos,&npos);

   if (!real) FILEmove(ei->edit_file,FILE_MOVE_ABS,&npos,&pos);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_finish_selection -- finish extending selection		*/
/*									*/
/************************************************************************/


void
EDT_finish_selection(ev,sel)
   EDT_VIEW ev;
   EDT_SELECT sel;
{
   FILE_POS pos,opos;

   if (sel->view == NULL) return;
   if (sel->view->edit != ev->edit) abort();

   sel->valid = TRUE;
   sel->anchor = EDT_ANCHOR_NONE;

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&opos);
   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&sel->begin,&pos);
   FILEpick(VIEW_FILE(ev),FILEinq_buffer(SELECT_BUFFER_NAME),
	       FILE_MOVE_ABS,&sel->end);
   FILEmove(VIEW_FILE(ev),FILE_MOVE_ABS,&opos,&pos);
};





/************************************************************************/
/*									*/
/*	EDT_replace_selection -- handle replace flag in a selection	*/
/*									*/
/************************************************************************/


void
EDT_replace_selection(sel,fg)
   EDT_SELECT sel;
   Boolean fg;
{
   FILE_POS epos;

   if (!sel->valid) return;

   if (fg && sel->view->edit->data->readonly) fg = FALSE;

   FILEpos_copy(VIEW_FILE(sel->view),&sel->end,&epos);
   if (sel->replace && !fg) {
      TEXTmark_end(VIEW_TEXT(sel->view));
      TEXTmark2_begin(VIEW_TEXT(sel->view),&sel->begin);
      TEXTmark2_until(VIEW_TEXT(sel->view),&epos);
    }
   else if (!sel->replace && fg) {
      TEXTmark2_end(VIEW_TEXT(sel->view));
      TEXTmark_begin(VIEW_TEXT(sel->view),&sel->begin);
      TEXTmark_until(VIEW_TEXT(sel->view),&epos);
    };

   sel->replace = fg;
};





/************************************************************************/
/*									*/
/*	edit_track -- mouse tracking function for marking		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Boolean
edit_track(x,y,ct,act,max,w,btn,ch)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
   Integer btn;
   Integer ch;
{
   EDT_VIEW ev;
   EDT_SELECT sel;
   TRACK_INFO * tin;
   FILE_POS pos,opos;
   Integer dir;

   ev = (EDT_VIEW) ASHinq_user_data(ASHinq_parent(w));
   tin = (TRACK_INFO *) ev->edit->temp;
   sel = tin->select;

   if (!tin->valid) {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&opos);
      if (act == RIP_TRANS_MOVE) {
	 TEXTcorrelate(VIEW_TEXT(ev),x,y,TRUE,&pos);
	 if (FILEpos_compare(VIEW_FILE(ev),&pos,&opos) == 0) return TRUE;
       }
      else if (act == RIP_TRANS_NONE) return TRUE;
      else if (act == RIP_TRANS_UP) return FALSE;
      tin->valid = TRUE;
      EDT_start_selection(ev,sel,&opos,FALSE);
    };

   if (sel->view != ev) return FALSE;

   switch (act) {
      case RIP_TRANS_NONE :
	 break;

      case RIP_TRANS_MOVE :
	 TEXTcorrelate(VIEW_TEXT(ev),x,y,FALSE,&pos);
	 EDT_extend_selection(ev,sel,&pos,FALSE,FALSE);
	 break;

      case RIP_TRANS_UP :
	 if (tin->count-- == 0 || btn & tin->btn) {
	    TEXTcorrelate(VIEW_TEXT(ev),x,y,FALSE,&pos);
	    EDT_extend_selection(ev,sel,&pos,FALSE,FALSE);
	    EDT_finish_selection(ev,sel);
	    return FALSE;			/* finish */
	  };
	 break;

      case RIP_TRANS_DOWN :
      case RIP_TRANS_TAP :
	 if (tin->btn & RIP_BTN_LEFT) {
	    dir =  (btn & RIP_BTN_MID) ? -1 : 1;
	  }
	 else {
	    dir = (btn & RIP_BTN_LEFT) ? -1 : 1;
	  };
	 if (dir < 0 && sel->unit != EDT_SELECT_CHAR) {
	    sel->unit = (EDT_SELECT_TYPE) (((Integer)(sel->unit)) - 1);
	  }
	 else if (dir > 0 && sel->unit != EDT_SELECT_FILE) {
	    sel->unit = (EDT_SELECT_TYPE) (((Integer)(sel->unit)) + 1);
	  };
	 TEXTcorrelate(VIEW_TEXT(ev),x,y,FALSE,&pos);
	 EDT_extend_selection(ev,sel,&pos,TRUE,FALSE);
	 if (act != RIP_TRANS_TAP) ++tin->count;
	 break;

      default :
	 EDT_clear_selection(ev->edit,NULL);
	 break;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	real_position -- modify position according to selection unit	*/
/*									*/
/************************************************************************/


static void
real_position(ei,sty,fp,sp,ep)
   EDT_ID ei;
   EDT_SELECT_TYPE sty;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   switch (sty) {
      case EDT_SELECT_CHAR :
	 FILEpos_copy(ei->edit_file,fp,sp);
	 FILEpos_copy(ei->edit_file,fp,ep);
	 break;

      case EDT_SELECT_WORD :
	 EDT_ctbl_word_bounds(ei,fp,sp,ep);
	 break;

      case EDT_SELECT_LINE :
	 EDT_ctbl_sent_bounds(ei,fp,sp,ep);
	 break;

      case EDT_SELECT_PARA :
	 EDT_ctbl_para_bounds(ei,fp,sp,ep);
	 break;

      case EDT_SELECT_FILE :
	 FILEinq_position(ei->edit_file,FILE_MOVE_START,0,sp);
	 FILEinq_position(ei->edit_file,FILE_MOVE_END,0,ep);
	 break;

      default :
	 FILEpos_copy(ei->edit_file,FILE_POS_NONE,sp);
	 FILEpos_copy(ei->edit_file,FILE_POS_NONE,ep);
	 break;
    };
};





/************************************************************************/
/*									*/
/*	closer_position -- find position that is closest to given one	*/
/*									*/
/************************************************************************/


static void
closer_position(ei,fp,sp,ep,rp)
   EDT_ID ei;
   FILE_POS *fp;
   FILE_POS *sp;
   FILE_POS *ep;
   FILE_POS *rp;
{
   Integer l0,l1,l2,c0,c1,c2;
   Integer i,j,v;

   l0 = FILEinq_line(ei->edit_file,fp);
   l1 = FILEinq_line(ei->edit_file,sp);
   l2 = FILEinq_line(ei->edit_file,ep);

   i = abs(l0-l1) - abs(l0-l2);
   if (i < 0) v = 1;
   else if (i > 0) v = 2;
   else {
      c0 = FILEinq_char(ei->edit_file,fp);
      c1 = FILEinq_char(ei->edit_file,sp);
      c2 = FILEinq_char(ei->edit_file,ep);
      j = abs(c0-c1) - abs(c0-c2);
      if (j < 0) v = 1;
      else v = 2;
    };

   if (v == 1) FILEpos_copy(ei->edit_file,sp,rp);
   else FILEpos_copy(ei->edit_file,ep,rp);
};





/************************************************************************/
/*									*/
/*	raw_input -- handle input to be sent to shell			*/
/*									*/
/************************************************************************/


static void
raw_input(ei,ch)
   EDT_ID ei;
   Integer ch;
{
   if (ei->shell) {
      (*ei->data->input_rtn)(ei->udata,ch,NULL);
    };
};





/* end of edtinput.c */



