/************************************************************************/
/*									*/
/*		edtexec.c						*/
/*									*/
/*	Editor interface for running a program				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "edt_local.h"

#include <signal.h>
#include <sgtty.h>
#include <fcntl.h>
#include <varargs.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/wait.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/






/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


typedef struct _EDT_EXEC {
   ASH_WINDOW window;
   ASH_WINDOW menu_win;
   ASH_WINDOW status_win;
   Integer pty;
   Integer process;
   EDT_ID edt_info;
   String ptyname;
} EDT_EXEC_INFO;



static	Sequence	all_execs;

static	Integer 	last_pid;
static	Integer 	last_status;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		exec_window_setup();
static	Boolean 	exec_startup();
static	void		exec_run();
static	Integer 	exec_tty_set();
static	Integer 	get_ptty();
static	Integer 	edt_exec_edt_input();
static	Integer 	edt_exec_pgm_input();
static	Integer 	edt_exec_control();
static	void		exec_status_handler();
static	void		set_pty_size();





/************************************************************************/
/*									*/
/*	EDT_exec_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_exec_init()
{
   struct sigvec hdlr;

   all_execs = NULL;
   last_pid = 0;

   hdlr.sv_handler = exec_status_handler;
   hdlr.sv_mask = 0;
   hdlr.sv_onstack = 0;
   sigvec(SIGCHLD,&hdlr,NULL);
};





/************************************************************************/
/*									*/
/*	EDTexec -- exec a program in given window			*/
/*									*/
/************************************************************************/


EDT_EXEC
EDTexec(va_alist)
   va_dcl
{
   va_list ap;
   ASH_WINDOW w,mw,sw;
   String sys;
   String args[64];
   Integer argct;
   EDT_EXEC ee;

   va_start(ap);

   w = va_arg(ap,ASH_WINDOW);
   mw = va_arg(ap,ASH_WINDOW);
   sw = va_arg(ap,ASH_WINDOW);
   sys = va_arg(ap,String);

   for (argct = 0; ; ++argct) {
      args[argct] = va_arg(ap,String);
      if (args[argct] == 0) break;
    };

   if (argct < 1 || sys == NULL) {
      EDT_error("Bad argument list for EDTexec");
      return NULL;
    };

   ee = (EDT_EXEC) malloc(sizeof(EDT_EXEC_INFO));
   ee->window = w;
   ee->menu_win = mw;
   ee->status_win = sw;
   ee->pty = -1;
   ee->process = 0;
   ee->ptyname = NULL;

   ASHset_control(w,edt_exec_control);

   exec_window_setup(ee);

   exec_startup(ee,sys,args);

   PROTECT;
   all_execs = CONS(ee,all_execs);
   UNPROTECT;

   set_pty_size(ee);

   return ee;
};





/************************************************************************/
/*									*/
/*	EDTptty -- create a pseudo tty in a window			*/
/*									*/
/************************************************************************/


EDT_EXEC
EDTptty(w,mw,sw,buf)
   ASH_WINDOW w;
   ASH_WINDOW mw;
   ASH_WINDOW sw;
   Character buf[];
{
   EDT_EXEC ee;
   Integer msk[2];
   Character ttynm[64];

   ee = (EDT_EXEC) malloc(sizeof(EDT_EXEC_INFO));
   ee->window = w;
   ee->menu_win = mw;
   ee->status_win = sw;
   ee->pty = -1;
   ee->process = 0;

   ASHset_control(w,edt_exec_control);

   exec_window_setup(ee);

   ee->pty = get_ptty(ttynm);
   ee->ptyname = SALLOC(ttynm);
   if (buf != NULL) {
      strcpy(buf,ttynm);
      buf[strlen("/dev/")] = 't';
    };

   exec_tty_set(ttynm,FALSE);

   fcntl(ee->pty,F_SETFL,FNDELAY);

   msk[0] = 1 << ee->pty;
   CMPXregister_add(ee->pty+1,msk,NULL,NULL,edt_exec_pgm_input);

   PROTECT;
   all_execs = CONS(ee,all_execs);
   UNPROTECT;

   return ee;
};





/************************************************************************/
/*									*/
/*	EDTexec_stop -- stop the exec'd process                         */
/*									*/
/************************************************************************/


void
EDTexec_stop(ee)
   EDT_EXEC ee;
{
   EDTexec_kill(ee,SIGKILL);

   ee->process = 0;

   if (ee->pty >= 0) {
      CMPXclose(ee->pty);
      close(ee->pty);
      ee->pty = -1;
    };
};





/************************************************************************/
/*									*/
/*	EDTexec_kill -- signal exec'd process                           */
/*									*/
/************************************************************************/


void
EDTexec_kill(ee,sig)
   EDT_EXEC ee;
   Integer sig;
{
   if (ee->process > 0) {
      kill(ee->process,sig);
    };
};




/************************************************************************/
/*									*/
/*	EDTexec_input -- fake input from user for process		*/
/*									*/
/************************************************************************/


void
EDTexec_input(ee,txt)
   EDT_EXEC ee;
   String txt;
{
   if (txt[0] == 0) return;
   else edt_exec_edt_input(ee,0,txt);
};





/************************************************************************/
/*									*/
/*	EDTexec_wait -- wait for exec window editor to terminate	*/
/*									*/
/************************************************************************/


void
EDTexec_wait(ee)
   EDT_EXEC ee;
{
   if (ee == NULL) return;

   EDTwait(ee->edt_info);
};





/********************************************************************************/
/*										*/
/*	EDTexec_inq_status -- get status returned for pid			*/
/*										*/
/********************************************************************************/


int
EDTexec_inq_status(pid)
   Integer pid;
{
   if (pid == last_pid) return last_status;

   return 0;
};





/************************************************************************/
/*									*/
/*	EDTexec_prohibit_close -- disallow close of exec window 	*/
/*									*/
/************************************************************************/


void
EDTexec_prohibit_close(ee)
   EDT_EXEC ee;
{
   EDTprohibit_close(ee->edt_info);
};





/************************************************************************/
/*									*/
/*	exec_window_setup -- setup shell editor in window		*/
/*									*/
/************************************************************************/


static void
exec_window_setup(ee)
   EDT_EXEC ee;
{
   EDT_DATA ed;
   Character tnm[32];

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = TRUE;
   ed->temporary = FALSE;
   ed->scroll = TRUE;
   ed->raw = TRUE;
   ed->echo = FALSE;
   ed->savectl = FALSE;
   ed->mode = NULL;
   ed->bounds_rtn = NULL;
   ed->input_rtn = edt_exec_edt_input;
   ed->mouse_rtn = NULL;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   strcpy(tnm,"/tmp/edtXXXXXX");
   mktemp(tnm);

   ee->edt_info = EDTdefine_editor(ee->window,tnm,ed,ee,ee->menu_win,
				      ee->status_win);
   ee->edt_info->auto_linefeed = TRUE;
};





/************************************************************************/
/*									*/
/*	exec_startup -- start up exec'd process                         */
/*									*/
/************************************************************************/


static Boolean
exec_startup(ee,sys,argv)
   EDT_EXEC ee;
   String sys;
   String argv[];
{
   Character ttynm[64];
   Integer child;
   Integer msk[2];

   ee->pty = get_ptty(ttynm);
   ee->ptyname = SALLOC(ttynm);

   fcntl(ee->pty,F_SETFL,FNDELAY);

   msk[0] = 1 << ee->pty;
   CMPXregister_add(ee->pty+1,msk,NULL,NULL,edt_exec_pgm_input);

   fflush(stdout);
   fflush(stderr);

   child = vfork();
   if (child < 0) return FALSE;

   if (child == 0) {
      exec_run(ee,ttynm,sys,argv);

      EDT_error("Exec failed in EDTexec\n");
      _exit(0);
    };

   ee->process = child;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	exec_run -- start running the child process			*/
/*									*/
/************************************************************************/


static void
exec_run(ee,ttynm,sys,argv)
   EDT_EXEC ee;
   Character ttynm[];
   String sys;
   String argv[];
{
   register Integer slv;
   int	      i,j;
   int	      pgrp;
   extern int errno;

   slv = exec_tty_set(ttynm,TRUE);

   if (slv < 0) return;

   pgrp = getpid();
   setpgrp(0,pgrp);
   ioctl(slv,TIOCSPGRP,&pgrp);

   dup2(slv,0);
   dup2(slv,1);
   dup2(slv,2);

   j = getdtablesize();
   for (i = 3; i < j; ++i) {
      close(i);
    };

   execv(sys,argv);
};





/************************************************************************/
/*									*/
/*	exec_tty_set -- setup tty slave 				*/
/*									*/
/************************************************************************/


static Integer
exec_tty_set(ttynm,runfg)
   Character ttynm[];
   Boolean runfg;
{
   register Integer slv;
   struct  sgttyb b;
   struct  tchars tc;
   struct  ltchars lc;
   int	      lb;
   int	      l;
   Integer tty;

   tty = open("/dev/tty",2);
   if (tty < 0) tty = 1;

   ioctl(tty, TIOCGETP, (char *)&b);
   ioctl(tty, TIOCGETC, (char *)&tc);
   ioctl(tty, TIOCGETD, (char *)&l);
   ioctl(tty, TIOCGLTC, (char *)&lc);
   ioctl(tty, TIOCLGET, (char *)&lb);

   if (tty > 2) {
      if (runfg) ioctl(tty,TIOCNOTTY,0);
      close(tty);
    };

   ttynm[strlen("/dev/")] = 't';
   slv = open(ttynm, 2);
   if (slv < 0) {
      printf("failed to open debug slave tty %s\n",ttynm);
      return -1;
    };

   /* b.sg_flags &= ~CRMOD;	   */
   b.sg_flags |= ECHO;
   b.sg_flags |= XTABS;
   b.sg_flags &= ~(CBREAK|RAW);

   lc.t_suspc = -1;
   lc.t_dsuspc = -1;

   ioctl(slv, TIOCSETP, (char *)&b);
   ioctl(slv, TIOCSETC, (char *)&tc);
   ioctl(slv, TIOCSLTC, (char *)&lc);
   ioctl(slv, TIOCLSET, (char *)&lb);
   ioctl(slv, TIOCSETD, (char *)&l);

   return slv;
};





/************************************************************************/
/*									*/
/*	get_ptty -- get pseudo tty					*/
/*									*/
/************************************************************************/


static Integer
get_ptty(ttynm)
   Character ttynm[];
{
   register Character c;
   struct stat stb;
   register Integer i,master,slave;

   strcpy(ttynm,"/dev/ptyXX");

   for (c = 'p'; c <= 's'; c++) {
      ttynm[strlen("/dev/pty")] = c;
      ttynm[strlen("/dev/ptyp")] = '0';
      if (stat(ttynm, &stb) < 0)
	 break;
      for (i = 0; i < 16; i++) {
	 ttynm[strlen("/dev/ptyp")] = "0123456789abcdef"[i];
	 master = open(ttynm, 2);
	 if (master < 0) continue;
	 ttynm[strlen("/dev/")] = 't';
	 slave = open(ttynm,2);
	 close(slave);
	 ttynm[strlen("/dev/")] = 'p';
	 if (slave >= 0) return master;
	 close(master);
       };
    };

   printf("Out of pty's\n");
   exit(1);
   /*NOTREACHED*/
};





/************************************************************************/
/*									*/
/*	edt_exec_edt_input -- input from editor to running process	*/
/*									*/
/************************************************************************/


static Integer
edt_exec_edt_input(ee,ch,st)
   EDT_EXEC ee;
   Character ch;
   String st;
{
   register Integer ln;

   if (ee->pty < 0) return FALSE;

   if (st != NULL) {
      ln = strlen(st);
      write(ee->pty,st,ln);
    }
   else if (ch != 0) {
      if (ch == '\r') ch = '\n';
      write(ee->pty,&ch,1);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	edt_exec_pgm_input -- handle output on tty from exec'd program  */
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
edt_exec_pgm_input(rfg,wfg,xfg)
   Integer rfg[];
   Integer wfg[];
   Integer xfg[];
{
   EDT_EXEC ee;
   Integer ct,i;
   Character buf[1025];
   Sequence l;
   String s;

   forin (ee,EDT_EXEC,l,all_execs) {
      if (ee->pty >= 0 && ((1<<(ee->pty))&rfg[0]) != 0) {
	 if ((ct = read(ee->pty,buf,1024)) > 0) {
	    s = buf;
	    for (i = 0; i < ct; ++i) {
	       if (buf[i] != 0) *s++ = buf[i];
	     };
	    *s = 0;
	    EDTshell_output(ee->edt_info,buf);
	  }
	 if (ct < 0 && errno == 5) {
	    if (ee->window != NULL) ASHremove(ee->window);
	  };
	 break;
       };
    };

   return 0;
};





/************************************************************************/
/*									*/
/*	edt_exec_control -- handle control messages from window 	*/
/*									*/
/************************************************************************/


static Integer
edt_exec_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   Sequence l;
   EDT_EXEC ee;

   if (STREQL(msg,"EDT$SAVE_CLOSE") || STREQL(msg,"EDT$ABORT_CLOSE") ||
	  STREQL(msg,"ASH$REMOVE")) {
      forin (ee,EDT_EXEC,l,all_execs) {
	 if (ee->window == w) break;
       };
      if (ee != NULL) EDTexec_stop(ee);
    }
   else if (STREQL(msg,"ASH$RESIZE")) {
      forin (ee,EDT_EXEC,l,all_execs) {
	 if (ee->window == w) break;
       };
      if (ee != NULL) set_pty_size(ee);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	exec_status_handler -- handle changes in child status		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
exec_status_handler(sig,code,scp)
   Integer sig;
   Integer code;
   struct sigcontext * scp;
{
   register Integer pid;
   union wait sts;
   register EDT_EXEC ee;
   register Sequence l;

   pid = wait3(&sts,WNOHANG,NULL);

   if (WIFSTOPPED(sts) || pid == 0) return;

   forin (ee,EDT_EXEC,l,all_execs) {
      if (ee->process == pid) break;
    };

   if (ee == NULL) {
      last_pid = pid;
      last_status = sts.w_status;
      return;
    };

   if (sts.w_termsig != 0) {
      printf("Program aborted.  Sorry.\n");
      exit(sts.w_termsig);
    };

   ee->process = 0;

   if (ee->window == NULL) exit(0);
};





/************************************************************************/
/*									*/
/*	set_pty_size -- set size for pty				*/
/*									*/
/************************************************************************/


static void
set_pty_size(ee)
   EDT_EXEC ee;
{
#ifdef TIOCSWINSZ

   struct winsize wsz;
   Integer lx,by,rx,ty;
   Integer tops[2],bots[2],wid;

   if (ee->pty >= 0 && ee->edt_info != NULL) {
      ASHinq_size(ee->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      EDTinq_bounds(ee->edt_info,&wid,tops,bots,1);
      wsz.ws_row = bots[0]-tops[0]+1;
      wsz.ws_col = wid;
      wsz.ws_xpixel = abs(rx-lx)+1;
      wsz.ws_ypixel = abs(by-ty)+1;
      ioctl(ee->pty,TIOCSWINSZ,&wsz);
    };

#endif
};





/* end of edtexec.c */
