/************************************************************************/
/*									*/
/*		edteval.c						*/
/*									*/
/*	Command evaluation routines for EDT				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		free_macro();
static	void		free_macro_block();
static	Boolean 	begin_command();
static	void		end_command();
static	Boolean 	eval_command();
static	void		move_position();
static	void		tab_position();





/************************************************************************/
/*									*/
/*	EDT_eval_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_eval_init()
{
};





/************************************************************************/
/*									*/
/*	EDT_eval_cmd -- evaluate a command				*/
/*									*/
/************************************************************************/


Boolean
EDT_eval_cmd(ev,cmd)
   EDT_VIEW ev;
   EDT_CMD_ID cmd;
{
   register Boolean fg;
   EDT_VAR args[MAX_ARGS];
   register Integer ct;
   register EDT_ID ei;
   Character buf[256];
   String s;

   ei = ev->edit;

   if (begin_command(ev,cmd)) {
      fg = TRUE;
      if (ei->scan_args != NULL) {
	 ct = EDT_ctbl_arg_list(cmd,args);
	 if (!EDT_vars_scan(ev,ct,args,ei->scan_args)) fg = FALSE;
	 if (ei->scan_args != NULL) {
	    SFREE(ei->scan_args);
	    ei->scan_args = NULL;
	  };
       };
      if (ei->get_args || EDT_ctbl_show_args(cmd)) {
	 ct = EDT_ctbl_arg_list(cmd,args);
	 if (ct > 0) {
	    ei->in_command = TRUE;
	    if (!EDT_vars_request(ev,EDT_ctbl_cmd_name(cmd),ct,args)) fg = FALSE;
	    ei->in_command = FALSE;
	  };
       };
      if (ev->edit == ei && ASHinq_valid_window(ev->window) && ASHinq_valid_window(ev->editwin)) {
	 ei->get_args = FALSE;
	 if (fg) fg = eval_command(ev,cmd);
	 if (EDT_ctbl_cmd_close(cmd)) ev = NULL;
       }
      else ev = NULL;
      end_command(ei,ev,cmd);
      if (fg) {
	 s = EDT_ctbl_confirm(cmd,buf,fg);
	 if (s != NULL) EDT_confirm(ei,s);
       };
    }
   else fg = FALSE;

   return fg;
};






/************************************************************************/
/*									*/
/*	EDT_eval_menu_btn -- handle commands through menu buttons	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Boolean
EDT_eval_menu_btn(md,menu,btn,w)
   Universal md;
   String menu;
   String btn;
   ASH_WINDOW w;
{
   EDT_ID ei;
   register EDT_VIEW ev;
   register EDT_CMD_ID cmd;
   register Boolean fg;

   ei = EDT_info_by_menu(w);
   if (ei == NULL) return FALSE;
   if (ei->edit_file == NULL) {
      fg = (EDT_open_file(ei,NULL) > 0);
      return fg;
    };

   ev = EDT_inq_current_view(ei);
   cmd = EDT_ctbl_find_command(btn);

   if (ev == NULL) fg = FALSE;
   else if (ev->edit != ei) abort();
   else fg = EDT_eval_cmd(ev,cmd);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDT_save_command -- save command in history stack		*/
/*									*/
/************************************************************************/


void
EDT_save_command(ei,upd,hist,remhist)
   EDT_ID ei;
   Boolean upd;
   Boolean hist;
   Boolean remhist;
{
   Integer vl,i;
   Boolean noupd;

   if (ei->edit_file == NULL) return;

   noupd = !upd;

   vl = FILEend_command(ei->edit_file,noupd);

   if (hist) {
      if (ei->history[ei->histfirst] != 0)
	 FILEfree_history(ei->history[ei->histfirst]);
      ei->history[ei->histfirst] = vl;
      ei->histfirst = (ei->histfirst + 1) % ei->histct;
    }
   else if (remhist) {
      FILEfree_history(vl);
      for (i = 1; i < ei->histct; ++i) ei->history[i] = 0;
      ei->histfirst = 1;
      ei->history[0] = vl;
    };

   ei->cmdct++;
   if ((ei->cmdct % SECURE_COUNTER) == 0) {
      FILEsecure(ei->edit_file);
    };
};






/************************************************************************/
/*									*/
/*	EDT_undo -- undo last (ct) commands				*/
/*	EDT_redo -- redo last undo					*/
/*									*/
/************************************************************************/


Boolean
EDT_undo(ei)
   EDT_ID ei;
{
   register Integer i;
   register Boolean fg;

   i = (ei->histfirst + ei->histct - 1) % ei->histct;

   if (ei->history[i] != 0) {
      fg = FILEundo_command(ei->edit_file,ei->history[i]);
      ei->history[i] = 0;
      ei->histfirst = i;
    }
   else fg = FALSE;

   return fg;
};




Boolean
EDT_redo(ei)
   EDT_ID ei;
{
   register Boolean fg;

   fg = FILEredo_command(ei->edit_file);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDT_scroll_to -- scroll to given line				*/
/*									*/
/************************************************************************/


Boolean
EDT_scroll_to(ev,ln)
   EDT_VIEW ev;
   Integer ln;
{
   register Integer cln;

   cln = TEXTinq_topline(VIEW_TEXT(ev));

   ln -= cln;

   if (ln != 0) {
      ev->edit->ingroup = 0;
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_DELTA_LINES,ln);
      EDT_synch_position(ev,TRUE);
      EDT_save_command(ev->edit,TRUE,TRUE,FALSE);
    };


   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_file_position -- get position from EDT_POS			*/
/*									*/
/************************************************************************/


void
EDT_file_position(ev,ep,fp)
   EDT_VIEW ev;
   EDT_POS ep;
   FILE_POS * fp;
{
   Integer fg;
   Integer val;
   FILE_POS sp;

   if (ep == NULL) {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,fp);
      return;
    };

   val = 0;

   if (ep->flags & EDT_POS_FLAG_START_FILE) fg = FILE_MOVE_START;
   else if (ep->flags & EDT_POS_FLAG_END_FILE) fg = FILE_MOVE_END;
   else if (ep->flags & EDT_POS_FLAG_SELECT_START) {
      if (!VIEW_SELECT_VALID(ev))
	 FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,-1,&sp);
      else
	 FILEpos_copy(VIEW_FILE(ev),&VIEW_SELECT_START(ev),&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_SELECT_END) {
      if (!VIEW_SELECT_VALID(ev))
	 FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,-1,&sp);
      else
	 FILEpos_copy(VIEW_FILE(ev),&VIEW_SELECT_END(ev),&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_VIEW_TOP) {
      TEXTinq_top(VIEW_TEXT(ev),&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_VIEW_BOTTOM) {
      TEXTinq_bottom(VIEW_TEXT(ev),&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_VIEW_MIDDLE) {
      val = TEXTinq_topline(VIEW_TEXT(ev)) + TEXTinq_bottomline(VIEW_TEXT(ev));
      FILEnew_position(VIEW_FILE(ev),val/2,1,&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_TAB) {
      tab_position(ev->edit,ep,&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_MOVEMENT) {
      move_position(ev->edit,ep,&sp);
      fg = FILE_MOVE_ABS;
    }
   else if (ep->flags & EDT_POS_FLAG_NEXT_LINE) {
      fg = FILE_MOVE_REL|FILE_MOVE_LINE;
      val = 1;
    }
   else if (ep->flags & EDT_POS_FLAG_PREV_LINE) {
      fg = FILE_MOVE_REL|FILE_MOVE_LINE;
      val = -1;
    }
   else fg = FILE_MOVE_REL;

   if (ep->flags & EDT_POS_FLAG_LINE_START) fg |= FILE_MOVE_LINE_START;
   else if (ep->flags & EDT_POS_FLAG_LINE_END) fg |= FILE_MOVE_LINE_END;
   else if (ep->flags & EDT_POS_FLAG_COND_START) fg |= FILE_MOVE_COND_START;

   if (fg & FILE_MOVE_ABS) val = (Integer) &sp;

   FILEinq_position(VIEW_FILE(ev),fg,val,fp);
};





/************************************************************************/
/*									*/
/*	EDT_word_bounds -- get bounds for word at position		*/
/*									*/
/************************************************************************/


Integer
EDT_word_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   Integer lin,chr,ch;
   FILE_POS pos;
   Integer typ;
   Integer och,oln;

   oln = FILEinq_currline(ei->edit_file);
   och = FILEinq_currchar(ei->edit_file);

   FILEmove(ei->edit_file,FILE_MOVE_ABS,fp,&pos);

   lin = FILEinq_currline(ei->edit_file);
   chr = FILEinq_currchar(ei->edit_file);
   ch = FILEget_char(ei->edit_file,lin,chr);

   typ = 0;
   if (ch == ' ' || ch == '\t' || ch == '\n' || ch == 0) {
      typ = 1;
      FILEsearch(ei->edit_file,"[^ \t]",
		    FILE_SEARCH_RE|FILE_SEARCH_BACKWARD,sp);
      if (FILEpos_compare(ei->edit_file,sp,FILE_POS_NONE) == 0)
	 FILEinq_position(ei->edit_file,FILE_MOVE_START,0,sp);
      else {
	 EDT_next_char(ei,sp,sp);
       };
      if (ch == '\n') {
	 FILEpos_copy(ei->edit_file,fp,ep);
	 typ = 2;
	 if (lin == 1 && chr == 1) typ = 0;
       }
      else {
	 FILEsearch(ei->edit_file,"[^ \t]",FILE_SEARCH_RE,ep);
	 if (FILEpos_compare(ei->edit_file,ep,FILE_POS_NONE) == 0)
	    FILEinq_position(ei->edit_file,FILE_MOVE_END,0,ep);
	 chr = FILEinq_char(ei->edit_file,ep);
	 ch = FILEget_char(ei->edit_file,lin,chr);
	 if (ch == '\n')  typ = 2;
	 else if (ch != 0) EDT_last_char(ei,ep,ep);
       };
      if (ch == 0) typ = 0;
    }
   else if (isalpha(ch) || isdigit(ch) || ch == '_' || ch == '$') {
      FILEsearch(ei->edit_file,"[^A-Za-z0-9_$]",
		    FILE_SEARCH_RE|FILE_SEARCH_BACKWARD,sp);
      if (FILEpos_compare(ei->edit_file,sp,FILE_POS_NONE) == 0)
	 FILEinq_position(ei->edit_file,FILE_MOVE_START,0,sp);
      else
	 EDT_next_char(ei,sp,sp);
      FILEsearch(ei->edit_file,"[^A-Za-z0-9_$]",FILE_SEARCH_RE,ep);
      if (FILEpos_compare(ei->edit_file,ep,FILE_POS_NONE) == 0)
	 FILEinq_position(ei->edit_file,FILE_MOVE_END,0,ep);
      else
	 EDT_last_char(ei,ep,ep);
    }
   else {
      FILEpos_copy(ei->edit_file,fp,ep);
      FILEpos_copy(ei->edit_file,fp,sp);
    };

   FILEset_position(ei->edit_file,oln,och,&pos);

   return typ;
};





/************************************************************************/
/*									*/
/*	EDT_para_bounds -- get bounds of paragraph at point		*/
/*									*/
/************************************************************************/


Integer
EDT_para_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   FILE_POS opos,pos,npos;
   Integer typ;

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&opos);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,fp,&pos);

   EDT_last_char(ei,&pos,&pos);
   FILEmove(ei->edit_file,FILE_MOVE_ABS,&pos,&npos);

   typ = 0;

   FILEsearch(ei->edit_file,"\n\n",FILE_SEARCH_BACKWARD,sp);
   if (FILEpos_compare(ei->edit_file,sp,FILE_POS_NONE) == 0) {
      FILEmove(ei->edit_file,FILE_MOVE_START,0,sp);
      FILEpos_copy(ei->edit_file,sp,&pos);
    }
   else {
      FILEmove(ei->edit_file,FILE_MOVE_ABS,sp,&pos);
      EDT_next_char(ei,sp,sp);
      EDT_next_char(ei,sp,sp);
    };

   FILEsearch(ei->edit_file,"\n\n",0,ep);
   if (FILEpos_compare(ei->edit_file,ep,FILE_POS_NONE) == 0) {
      FILEinq_position(ei->edit_file,FILE_MOVE_END,0,ep);
    }
   else {
      EDT_next_char(ei,ep,ep);
    };

   FILEmove(ei->edit_file,FILE_MOVE_ABS,&opos,&pos);

   return typ;
};




/************************************************************************/
/*									*/
/*	EDT_line_bounds -- return bounds for given line 		*/
/*									*/
/************************************************************************/


Integer
EDT_line_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   register Integer i;
   Integer cl;

   i = FILEinq_line(ei->edit_file,fp);
   cl = FILEinq_currline(ei->edit_file);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
		       FILE_MOVE_LINE_START,i-cl,sp);
   FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
		       FILE_MOVE_LINE_END,i-cl,ep);

   return 0;
};





/************************************************************************/
/*									*/
/*	EDT_last_char -- get previous character from a FILE_POS 	*/
/*	EDT_next_char -- get next character from a FILE_POS		*/
/*									*/
/************************************************************************/


void
EDT_last_char(ei,fp,rp)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * rp;
{
   register Integer ln,ch;
   Integer cl;

   ln = FILEinq_line(ei->edit_file,fp);
   ch = FILEinq_char(ei->edit_file,fp);
   cl = FILEinq_currline(ei->edit_file);

   if (FILEget_char(ei->edit_file,ln,ch) == 0) {
      FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
			  FILE_MOVE_LINE_END,ln-cl,rp);
      ch = FILEinq_char(ei->edit_file,rp);
    };

   if (ch == 1 && ln == 1) {
      if (fp != rp) FILEpos_copy(ei->edit_file,fp,rp);
    }
   else if (ch == 1) {
      FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
			  FILE_MOVE_LINE_END,ln-cl-1,rp);
    }
   else {
      FILEnew_position(ei->edit_file,ln,ch-1,rp);
    };
};





void
EDT_next_char(ei,fp,rp)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * rp;
{
   register Integer ln,ch;
   Integer cl;
   FILE_POS pos,posa,posb;

   ln = FILEinq_line(ei->edit_file,fp);
   ch = FILEinq_char(ei->edit_file,fp);
   cl = FILEinq_currline(ei->edit_file);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&posa);

   FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
		       FILE_MOVE_LINE_END,ln-cl,&pos);

   if (FILEpos_compare(ei->edit_file,fp,&pos) >= 0) {
      FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,&posb);
      FILEinq_position(ei->edit_file,FILE_MOVE_REL|FILE_MOVE_LINE|
			  FILE_MOVE_LINE_START,ln-cl+1,rp);
      if (FILEpos_compare(ei->edit_file,rp,FILE_POS_NONE) == 0)
	 FILEinq_position(ei->edit_file,FILE_MOVE_END,0,rp);
    }
   else {
      FILEnew_position(ei->edit_file,ln,ch+1,rp);
    };
};





/************************************************************************/
/*									*/
/*	EDT_macro_begin -- start a macro definition			*/
/*	EDT_macro_add -- add to a macro definition			*/
/*	EDT_macro_end -- finish a macro definition			*/
/*									*/
/************************************************************************/


void
EDT_macro_begin(ev)
   EDT_VIEW ev;
{
   register EDT_ID ei;

   ei = ev->edit;

   if (ei->macro != NULL && ei->macro->view != NULL) {
      EDT_macro_end(ev);
      return;
    };

   free_macro(ei);

   if (ei->macro == NULL) {
      ei->macro = PALLOC(EDT_MACRO_INFO);
    };

   ei->macro->view = ev;
   ei->macro->first = NULL;
   ei->macro->last = NULL;
};





void
EDT_macro_add(ev,cmd,ct,vals,args)
   EDT_VIEW ev;
   EDT_CMD_ID cmd;
   Integer ct;
   Universal vals[];
   EDT_VAR args[];
{
   register EDT_MACRO_BLK mb;
   register EDT_ID ei;
   register Integer i;

   ei = ev->edit;

   if (ei->macro == NULL || ei->macro->view != ev) return;

   mb = PALLOC(EDT_MACRO_BLK_INFO);
   mb->next = NULL;
   mb->cmd = cmd;
   mb->key = ei->last_key_struck;
   for (i = 0; i < ct; ++i) {
      mb->vals[i] = EDT_value_copy(args[i]->type,vals[i]);
    };

   if (ei->macro->first == NULL) ei->macro->first = mb;
   if (ei->macro->last != NULL) ei->macro->last->next = mb;
   ei->macro->last = mb;
};





void
EDT_macro_end(ev)
   EDT_VIEW ev;
{
   register EDT_ID ei;
   register EDT_MACRO_BLK eb;

   ei = ev->edit;

   if (ei->macro == NULL || ei->macro->view == NULL) return;

   ei->macro->view = NULL;

   if (ei->macro->first != NULL) {
      free_macro_block(ei->macro->last);
      if (ei->macro->first == ei->macro->last) {
	 ei->macro->first = NULL;
	 ei->macro->last = NULL;
       }
      else {
	 for (eb = ei->macro->first;
	      eb->next != ei->macro->last;
	      eb = eb->next);
	 ei->macro->last = eb;
	 eb->next = NULL;
       };
    };
};





static void
free_macro(ei)
   EDT_ID ei;
{
   register EDT_MACRO_BLK eb,neb;

   if (ei->macro == NULL || ei->macro->first == NULL) return;

   for (eb = ei->macro->first; eb != NULL; eb = neb) {
      neb = eb->next;
      free_macro_block(eb);
    };

   ei->macro->first = NULL;
   ei->macro->last = NULL;
   ei->macro->view = NULL;
};





static void
free_macro_block(eb)
   EDT_MACRO_BLK eb;
{
   Integer ct;
   EDT_VAR args[MAX_ARGS];
   register Integer i;

   ct = EDT_ctbl_arg_list(eb->cmd,args);
   for (i = 0; i < ct; ++i) {
      switch (args[i]->type) {
	 case EDT_VAR_TYPE_STRING :
	 case EDT_VAR_TYPE_BUFFER :
	 case EDT_VAR_TYPE_POSITION :
	    free(eb->vals[i]);
	    break;
	 default :
	    break;
       };
    };
   free(eb);
};





/************************************************************************/
/*									*/
/*	EDT_macro_do -- evaluate the current macro			*/
/*									*/
/************************************************************************/


Boolean
EDT_macro_do(ev,rpt)
   EDT_VIEW ev;
   Integer rpt;
{
   register Boolean fg;
   register Integer i;
   register EDT_ID ei;
   register Function_Ptr rtn;
   register EDT_MACRO_BLK eb;

   ei = ev->edit;
   if (ei->macro == NULL) return FALSE;
   if (ei->macro->view != NULL) {
      EDT_macro_end(ev);
      return TRUE;
    };
   if (ei->macro->first == NULL) return TRUE;

   if (!LPROTECT(ei)) return FALSE;

   ei->ingroup = 0;

   for (i = 0; i < rpt; ++i) {
      for (eb = ei->macro->first; eb != NULL; eb = eb->next) {
	 rtn = EDT_ctbl_routine(eb->cmd);
	 ei->last_key_struck = eb->key;
	 fg = (*rtn)(ev,eb->vals[0],eb->vals[1],eb->vals[2],eb->vals[3],
			eb->vals[4],eb->vals[5],eb->vals[6],eb->vals[7]);
	 if (!fg && EDT_ctbl_macro_abort(eb->cmd)) break;
	 fg = TRUE;
       };
      if (!fg) break;
    };

   EDT_save_command(ei,TRUE,TRUE,FALSE);

   LUNPROTECT(ei);

   return fg;
};






/************************************************************************/
/*									*/
/*	EDT_synch_view -- synch view with current position		*/
/*	EDT_synch_position -- synch position with current view		*/
/*									*/
/************************************************************************/


void
EDT_synch_view(ev)
   EDT_VIEW ev;
{
   FILE_POS pos;
   Integer cln,tln,bln,lch,rch,cch;

   cln = FILEinq_currline(VIEW_FILE(ev));
   cch = FILEinq_currchar(VIEW_FILE(ev));
   tln = TEXTinq_topline(VIEW_TEXT(ev));
   bln = TEXTinq_bottomline(VIEW_TEXT(ev));
   lch = TEXTinq_leftchar(VIEW_TEXT(ev));
   rch = TEXTinq_rightchar(VIEW_TEXT(ev));

   if (cln < tln || cln > bln || cch < lch || cch > rch) {
      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
      TEXTposition(VIEW_TEXT(ev),TEXT_FLAG_NEAR_MIDDLE,&pos);
    };
};




void
EDT_synch_position(ev,top)
   EDT_VIEW ev;
   Boolean top;
{
   FILE_POS pos;
   Integer cln,tln,bln,lch,rch,cch;
   Integer nln;

   cln = FILEinq_currline(VIEW_FILE(ev));
   cch = FILEinq_currchar(VIEW_FILE(ev));
   tln = TEXTinq_topline(VIEW_TEXT(ev));
   bln = TEXTinq_bottomline(VIEW_TEXT(ev));
   lch = TEXTinq_leftchar(VIEW_TEXT(ev));
   rch = TEXTinq_rightchar(VIEW_TEXT(ev));

   if (cln < tln || cln > bln || cch < lch || cch > rch) {
      nln = (top ? tln : bln);
      FILEset_position(VIEW_FILE(ev),nln,lch,&pos);
      EDT_auto_extend(ev);
    };
};





/************************************************************************/
/*									*/
/*	EDT_auto_extend -- extend selection on file movement		*/
/*									*/
/************************************************************************/


void
EDT_auto_extend(ev)
   EDT_VIEW ev;
{
   FILE_POS pos;

   if (!ev->edit->extend_mode || ev->edit->select.view != ev) return;

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&pos);
   EDT_extend_selection(ev,&ev->edit->select,&pos,TRUE,TRUE);
};





/************************************************************************/
/*									*/
/*	begin_command -- start a command				*/
/*	end_command -- finish a command 				*/
/*									*/
/************************************************************************/


static Boolean
begin_command(ev,cmd)
   EDT_VIEW ev;
   EDT_CMD_ID cmd;
{
   Integer grp;

   if (cmd < 0) return FALSE;

   if (!LPROTECT(ev->edit)) return FALSE;

   grp = EDT_ctbl_cmd_group(cmd);
   if (ev->edit->raw_mode) grp = 0;

   if (ev->edit->ingroup != 0 && ev->edit->ingroup != grp) {
      ev->edit->ingroup = 0;
    };

   return TRUE;
};





static void
end_command(ei,ev,cmd)
   EDT_ID ei;
   EDT_VIEW ev;
   EDT_CMD_ID cmd;
{
   register Boolean fg;
   Integer grp;
   FILE_POS cpos;

   if (ev != NULL) {
      if (!EDT_ctbl_no_prev_set(cmd)) ev->edit->previous_cmd = cmd;
      grp = EDT_ctbl_cmd_group(cmd);
      if (ev->edit->raw_mode) grp = 0;

      if (grp != 0) {
	 fg = (grp != ev->edit->ingroup);
	 ev->edit->ingroup = grp;
       }
      else if (EDT_ctbl_ignore_cmd(cmd)) fg = FALSE;
      else {
	 fg = TRUE;
	 ev->edit->ingroup = 0;
       };

      EDT_save_command(ev->edit,TRUE,fg,FALSE);

      FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL,0,&cpos);
      TEXTcursor(VIEW_TEXT(ev),TEXT_CURSOR_FORCE,&cpos);
    }
   else if (ei != NULL && ei->force_bounds) {
      FILEend_command(ei->edit_file,TRUE);
    };

   LUNPROTECT(ei);
};





/************************************************************************/
/*									*/
/*	eval_command -- evaluate a command				*/
/*									*/
/************************************************************************/


static Boolean
eval_command(ev,cmd)
   EDT_VIEW ev;
   EDT_CMD_ID cmd;
{
   Universal args[MAX_ARGS];
   EDT_VAR vars[MAX_ARGS];
   register Boolean fg;
   register Integer i,ct;
   register Function_Ptr rtn;
   register EDT_ID ei;

   ei = ev->edit;

   ct = EDT_ctbl_arg_list(cmd,vars);
   for (i = 0; i < ct; ++i) {
      args[i] = EDT_var_get_value(ei,vars[i]);
    };
   while (i < MAX_ARGS) args[i++] = NULL;

   rtn = EDT_ctbl_routine(cmd);
   if (rtn != NULL) {
      EDT_macro_add(ev,cmd,ct,args,vars);
      fg = (*rtn)(ev,args[0],args[1],args[2],args[3],args[4],
		     args[5],args[6],args[7]);
    }
   else fg = FALSE;

   return fg;
};





/************************************************************************/
/*									*/
/*	move_position -- get a position via unit searching		*/
/*									*/
/************************************************************************/


static void
move_position(ei,ep,fp)
   EDT_ID ei;
   EDT_POS ep;
   FILE_POS * fp;
{
   Function_Ptr rtn;
   Integer fg,ofg,afg;
   FILE_POS spos,epos,lspos,lepos;

   FILEinq_position(ei->edit_file,FILE_MOVE_REL,0,fp);

   if (ep->flags & EDT_POS_FLAG_WORD) rtn = EDT_ctbl_word_bounds;
   else if (ep->flags & EDT_POS_FLAG_PARA) rtn = EDT_ctbl_para_bounds;
   else if (ep->flags & EDT_POS_FLAG_LINE) rtn = EDT_line_bounds;
   else if (ep->flags & EDT_POS_FLAG_SENTENCE) rtn = EDT_ctbl_sent_bounds;
   else return;

   fg = (*rtn)(ei,fp,&spos,&epos);

   if ((ep->flags & EDT_POS_FLAG_PREVIOUS) &&
	  (ep->flags & EDT_POS_FLAG_CURROK) == 0 &&
	  FILEpos_compare(ei->edit_file,&spos,fp) == 0) {
      EDT_last_char(ei,&spos,fp);
      fg = (*rtn)(ei,fp,&spos,&epos);
    };

   if ((ep->flags & EDT_POS_FLAG_PREVIOUS) == 0) {
      EDT_next_char(ei,&epos,fp);
      afg = fg;
      fg = (*rtn)(ei,fp,&spos,&epos);
    };

   if ((ep->flags & EDT_POS_FLAG_SEP) == 0 && fg != 0) {
      ofg = 0;
      if ((ep->flags & EDT_POS_FLAG_PREVIOUS) == 0 && afg == 0 && fg == 2) {
	 FILEpos_copy(ei->edit_file,&spos,&lspos);
	 FILEpos_copy(ei->edit_file,&epos,&lepos);
	 ofg = 2;
       };
      while (fg != 0) {
	 if ((ep->flags & EDT_POS_FLAG_PREVIOUS) != 0) {
	    FILEpos_copy(ei->edit_file,&spos,&lspos);
	    FILEpos_copy(ei->edit_file,&epos,&lepos);
	    if (fg == 2) ofg = 2;
	  };
	 if (ep->flags & EDT_POS_FLAG_PREVIOUS)
	    EDT_last_char(ei,&spos,fp);
	 else
	    EDT_next_char(ei,&epos,fp);
	 fg = (*rtn)(ei,fp,&spos,&epos);
       };
      if (ofg == 2) {
	 FILEpos_copy(ei->edit_file,&lspos,&spos);
	 FILEpos_copy(ei->edit_file,&lepos,&epos);
       };
    };

   if (ep->flags & EDT_POS_FLAG_END)
      FILEpos_copy(ei->edit_file,&epos,fp);
   else
      FILEpos_copy(ei->edit_file,&spos,fp);
};




/************************************************************************/
/*									*/
/*	tab_position -- get position for tabbing			*/
/*									*/
/************************************************************************/


static void
tab_position(ei,ep,fp)
   EDT_ID ei;
   EDT_POS ep;
   FILE_POS * fp;
{
   register Integer i,j,ln;

   i = FILEinq_currchar(ei->edit_file);
   ln = FILEinq_currline(ei->edit_file);

   j = ((i-1)&(~7)) + 1;

   if ((ep->flags & EDT_POS_FLAG_PREVIOUS) &&
	  (ep->flags & EDT_POS_FLAG_CURROK) == 0 &&
	  i == j && i != 1) {
      j -= 8;
    };

   if ((ep->flags & EDT_POS_FLAG_PREVIOUS) == 0) {
      j += 8;
    };

   FILEnew_position(ei->edit_file,ln,j,fp);
};





/* end of edteval.c */
