/************************************************************************/
/*									*/
/*		edtctbl.c						*/
/*									*/
/*	Command table routines for EDT					*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "edt_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define STRING_TBL_SIZE 	20480

#define BASE_CMD_TABLE_SIZE	64
#define BASE_VAR_TABLE_SIZE	128
#define BASE_BIND_TABLE_SIZE	64
#define BASE_TYPE_TABLE_SIZE	16





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	command_table_size;
static	Integer 	num_commands;
static	EDT_CMD_INFO *	command_table;

static	Integer 	var_table_size;
static	Integer 	num_vars;
static	EDT_VAR_INFO *	var_table;

static	Integer 	bind_table_size;
static	Integer 	num_binds;
static	EDT_BIND_INFO * bind_table;

static	Integer 	type_table_size;
static	Integer 	num_types;
static	EDT_ENUM_TYPE_INFO *	type_table;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	add_string();
static	EDT_VAR 	copy_var();
static	void		do_binding();
static	Sequence	bind_menu_names();
static	void		add_to_menu();
static	void		remove_menu();
static	void		bind_key();
static	void		bind_hook();
static	EDT_VAR 	var_enter();
static	Function_Ptr	image_address();
static	void		define_system_vars();





/********************************************************************************/
/*										*/
/*	Tables									*/
/*										*/
/********************************************************************************/


typedef struct _LINK_TBL {
   String name;
   Boolean (*rtn)();
} LINK_TBL;

static LINK_TBL local_cmds[] = {
   { "EDCMDNoop", EDCMDNoop },
   { "EDCMDEnter", EDCMDEnter },
   { "EDCMDInsertMode", EDCMDInsertMode },
   { "EDCMDIndentMode", EDCMDIndentMode },
   { "EDCMDOpenMode", EDCMDOpenMode },
   { "EDCMDReadOnly", EDCMDReadOnly },
   { "EDCMDGenericMove", EDCMDGenericMove },
   { "EDCMDTabN", EDCMDTabN },
   { "EDCMDGenericSearch", EDCMDGenericSearch },
   { "EDCMDReplaceString", EDCMDReplaceString },
   { "EDCMDGenericPosition", EDCMDGenericPosition },
   { "EDCMDScroll", EDCMDScroll },
   { "EDCMDKeepPosition", EDCMDKeepPosition },
   { "EDCMDInsertCharacter", EDCMDInsertCharacter },
   { "EDCMDSelfInsert", EDCMDSelfInsert },
   { "EDCMDNewLine", EDCMDNewLine },
   { "EDCMDGenericMap", EDCMDGenericMap },
   { "EDCMDGenericDelete", EDCMDGenericDelete },
   { "EDCMDGenericPick", EDCMDGenericPick },
   { "EDCMDPatternDelete", EDCMDPatternDelete },
   { "EDCMDGenericInsert", EDCMDGenericInsert },
   { "EDCMDInsertString", EDCMDInsertString },
   { "EDCMDSavePosition", EDCMDSavePosition },
   { "EDCMDGotoPosition", EDCMDGotoPosition },
   { "EDCMDCommand", EDCMDCommand },
   { "EDCMDUndo", EDCMDUndo },
   { "EDCMDRedo", EDCMDRedo },
   { "EDCMDCloseFile", EDCMDCloseFile },
   { "EDCMDOpenFile", EDCMDOpenFile },
   { "EDCMDSaveFile", EDCMDSaveFile },
   { "EDCMDAlterAccess", EDCMDAlterAccess },
   { "EDCMDSplitView", EDCMDSplitView },
   { "EDCMDRemoveView", EDCMDRemoveView },
   { "EDCMDSizeView", EDCMDSizeView },
   { "EDCMDLoadCommands", EDCMDLoadCommands },
   { "EDCMDSaveCommandTable", EDCMDSaveCommandTable },
   { "EDCMDLoadCommandTable", EDCMDLoadCommandTable },
   { "EDCMDSelectBegin", EDCMDSelectBegin },
   { "EDCMDSelectExtend", EDCMDSelectExtend },
   { "EDCMDSelectReplace", EDCMDSelectReplace },
   { "EDCMDSelectAbort", EDCMDSelectAbort },
   { "EDCMDCopyFromFile", EDCMDCopyFromFile },
   { "EDCMDWriteToFile", EDCMDWriteToFile },
   { "EDCMDUnixCommand", EDCMDUnixCommand },
   { "EDCMDIndent", EDCMDIndent },
   { "EDCMDPrint", EDCMDPrint },
   { "EDCMDJoin", EDCMDJoin },
   { "EDCMDMacroCompile", EDCMDMacroCompile },
   { "EDCMDMacroDo", EDCMDMacroDo },
   { "EDCMDBindKeys", EDCMDBindKeys },
   { "EDCMDTranspose", EDCMDTranspose },
   { "EDCMDUserCall", EDCMDUserCall },
   { "EDCMDHelp", EDCMDHelp },
   { "EDCMDAbbrevDefine", EDCMDAbbrevDefine },
   { "EDCMDAbbrevExpand", EDCMDAbbrevExpand },
   { "EDCMDAbbrevLoad", EDCMDAbbrevLoad },
   { "EDCMDAbbrevStore", EDCMDAbbrevStore },
   { "EDCMDAbbrevClear", EDCMDAbbrevClear },
   { "EDCMDAbbrevComplete", EDCMDAbbrevComplete },
   { "EDHOOKword_wrap", (Boolean (*)()) EDHOOKword_wrap },
   { "EDTCCindent", (Boolean (*)()) EDTCCindent },
   { "EDTPASindent", (Boolean (*)()) EDTPASindent },
   { 0, 0 }
};






/************************************************************************/
/*									*/
/*	EDT_ctbl_init -- module initialization				*/
/*									*/
/************************************************************************/


void
EDT_ctbl_init()
{
   command_table_size = BASE_CMD_TABLE_SIZE;
   command_table = (EDT_CMD_INFO *) calloc(command_table_size,sizeof(EDT_CMD_INFO));

   var_table_size = BASE_VAR_TABLE_SIZE;
   var_table = (EDT_VAR_INFO *) calloc(var_table_size,sizeof(EDT_VAR_INFO));

   bind_table_size = BASE_BIND_TABLE_SIZE;
   bind_table = (EDT_BIND_INFO *) calloc(bind_table_size,sizeof(EDT_BIND_INFO));

   type_table_size = BASE_TYPE_TABLE_SIZE;
   type_table = (EDT_ENUM_TYPE_INFO *) calloc(type_table_size,sizeof(EDT_ENUM_TYPE_INFO));

   num_commands = 0;
   num_vars = 0;
   num_binds = 0;
   num_types = 1;

   define_system_vars();
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_startup -- insure we have some commands		*/
/*									*/
/************************************************************************/


void
EDT_ctbl_startup()
{
   Character buf[128];
   String s;

   if (num_commands != 0) return;

   s = (String) getenv("EDT_BASIS_CTBL");
   if (s == NULL) {
      sprintf(buf,EDT_BASIS_FILE,BWEbwe_project(),BWEarch());
      if (access(buf,4) < 0) {
	 sprintf(buf,EDT_BASIS_FILE0,BWEbwe_project(),BWEarch());
       };
      s = buf;
    };
   EDTctbl_load(s);

   s = (String) getenv("EDT_BASIS_CMD");
   if (s != NULL) {
      EDTload_description(s);
    }
   else {
      s = getenv("HOME");
      if (s != NULL) {
	 sprintf(buf,EDT_BASIS_CMD_FILE,s);
	 if (access(buf,4) >= 0) EDTload_description(buf);
       }
    };
};




/************************************************************************/
/*									*/
/*	EDTctbl_save -- save command table into a file			*/
/*									*/
/************************************************************************/


#define PUTITEM(var)	fwrite(&var,sizeof(var),1,otf)
#define PUTSTR(txt)	PUTITEM(sbufp); sbufp = add_string(txt,sbuf,sbufp)
#define SAVESTR(txt)	((nval = sbufp),(sbufp = add_string(txt,sbuf,sbufp)),nval)
#define DOSTR(txt)	((txt) == NULL ? (-1) : SAVESTR(txt))


Boolean
EDTctbl_save(ei,file)
   EDT_ID ei;
   String file;
{
   register FILE * otf;
   Character sbuf[STRING_TBL_SIZE];
   Integer sbufp;
   Integer i;
   EDT_VAR_INFO evi;
   Universal val,nval;
   EDT_BUF bf;
   EDT_POS ep;

   if (file == NULL || *file == 0) return FALSE;

   sbufp = 0;
   otf = fopen(file,"w");
   if (otf == NULL) {
      EDT_error("Couldn't open file %s for write",file);
      return FALSE;
    };

   PROTECT;

   /* write out version number */
   i = (VERSION << 16) | SUBVERSION;
   PUTITEM(i);

   /* write out commands */
   PUTITEM(num_commands);
   fwrite(command_table,sizeof(EDT_CMD_INFO),num_commands,otf);

   /* write out bindings */
   PUTITEM(num_binds);
   fwrite(bind_table,sizeof(EDT_BIND_INFO),num_binds,otf);

   /* write out enumerations */
   PUTITEM(num_types);
   fwrite(type_table,sizeof(EDT_ENUM_TYPE_INFO),num_types,otf);

   /* write out variable values */
   PUTITEM(num_vars);
   for (i = 0; i < num_vars; ++i) {
      evi = var_table[i];
      val = EDT_var_get_value(ei,&var_table[i]);
      switch (evi.type) {
	 default :
	    break;
	 case EDT_VAR_TYPE_STRING :
	    evi.dflt = DOSTR(evi.dflt);
	    val = DOSTR(val);
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    if (evi.dflt != NULL) {
	       bf = (EDT_BUF) evi.dflt;
	       evi.dflt = DOSTR(bf->name);
	     };
	    if (val != NULL) {
	       bf = (EDT_BUF) val;
	       val = DOSTR(bf->name);
	     };
	    break;
	 case EDT_VAR_TYPE_POSITION :
	    if (evi.dflt != NULL) {
	       ep = (EDT_POS) evi.dflt;
	       evi.dflt = ep->flags;
	     };
	    if (val != NULL) {
	       ep = (EDT_POS) val;
	       val = ep->flags;
	     };
	    break;
       };
      PUTITEM(evi);
      PUTITEM(val);
    };

   /* write out help information */
   for (i = 0; i < num_commands; ++i) {
      val = (Universal) command_table[i].help;
      val = DOSTR(val);
      PUTITEM(val);
    };

   /* write out string table */
   PUTITEM(sbufp);
   fwrite(sbuf,1,sbufp,otf);

   UNPROTECT;

   fclose(otf);

   return TRUE;
};





static Integer
add_string(txt,buf,ct)
   String txt;
   String buf;
   Integer ct;
{
   while (*txt != 0) buf[ct++] = *txt++;
   buf[ct++] = 0;

   return ct;
};






/************************************************************************/
/*									*/
/*	EDTctbl_load -- load command table file 			*/
/*									*/
/************************************************************************/


#define GETITEM(var)	fread(&var,sizeof(var),1,inf)
#define GETSTR(where)	fread(&var,sizeof(Integer),1,inf)
#define FIXSTR(where)	((where == -1) ? NULL : SALLOC(&sbuf[((int) where)]))

Boolean
EDTctbl_load(file)
   String file;
{
   register FILE * inf;
   Character sbuf[STRING_TBL_SIZE];
   Integer sbufp;
   Integer i;
   Universal values[1024],val;
   register EDT_VAR ev;
   String * help;

   EDT_init();

   if (file == NULL || *file == 0) return FALSE;

   inf = fopen(file,"r");
   if (inf == NULL) {
      EDT_error("Couldn't open command file %s",file);
      return FALSE;
    };

   EDT_var_reset();

   PROTECT;

   /* read in version */
   GETITEM(i);
   if ((i >> 16) != VERSION || (i & 0xffff) != SUBVERSION) {
      EDT_error("Command file is an old version.");
      fclose(inf);
      return FALSE;
    };

   /* read in commands */
   GETITEM(num_commands);
   if (command_table_size <= num_commands) {
      command_table_size = num_commands+10;
      command_table = (EDT_CMD_INFO *)
	 realloc(command_table,command_table_size*sizeof(EDT_CMD_INFO));
    };
   fread(command_table,sizeof(EDT_CMD_INFO),num_commands,inf);

   /* read in bindings */
   GETITEM(num_binds);
   if (bind_table_size <= num_binds) {
      bind_table_size = num_binds+10;
      bind_table = (EDT_BIND_INFO *)
	 realloc(bind_table,bind_table_size*sizeof(EDT_BIND_INFO));
    };
   fread(bind_table,sizeof(EDT_BIND_INFO),num_binds,inf);

   /* read in enumerations */
   GETITEM(num_types);
   if (type_table_size <= num_types) {
      type_table_size = num_types+10;
      type_table = (EDT_ENUM_TYPE_INFO *)
	 realloc(type_table,type_table_size*sizeof(EDT_ENUM_TYPE_INFO));
    };
   fread(type_table,sizeof(EDT_ENUM_TYPE_INFO),num_types,inf);

   /* read in variables */
   GETITEM(num_vars);
   if (var_table_size <= num_vars) {
      var_table_size = num_vars+10;
      var_table = (EDT_VAR_INFO *)
	 realloc(var_table,var_table_size*sizeof(EDT_VAR_INFO));
    };
   for (i = 0; i < num_vars; ++i) {
      fread(&var_table[i],sizeof(EDT_VAR_INFO),1,inf);
      fread(&values[i],sizeof(Universal),1,inf);
    };

   /* read in help information */
   help = (String *) alloca(sizeof(String)*num_commands);
   fread(help,sizeof(String),num_commands,inf);

   /* read in string table */
   GETITEM(sbufp);
   fread(sbuf,1,sbufp,inf);

   /* fix up everything */
   for (i = 0; i < num_commands; ++i) {
      command_table[i].rtn = image_address(command_table[i].rtn_name);
      val = (Universal) help[i];
      command_table[i].help = FIXSTR(val);
    };

   for (i = 0; i < num_vars; ++i) {
      ev = &var_table[i];
      if (ev->index >= 0) ev->index = 0;
      ev->loc = i;
      switch (ev->type) {
	 default :
	    break;
	 case EDT_VAR_TYPE_STRING :
	    ev->dflt = (Universal) FIXSTR(ev->dflt);
	    values[i] = (Universal) FIXSTR(values[i]);
	    break;
	 case EDT_VAR_TYPE_BUFFER :
	    ev->dflt = (Universal) FIXSTR(ev->dflt);
	    values[i] = (Universal) FIXSTR(values[i]);
	    ev->dflt = (Universal) EDT_value_buffer(ev->dflt);
	    values[i] = (Universal) EDT_value_buffer(values[i]);
	    break;
	 case EDT_VAR_TYPE_POSITION :
	    ev->dflt = (Universal) EDT_value_position(ev->dflt);
	    values[i] = (Universal) EDT_value_position(values[i]);
	    break;
       };
      EDT_set_values(ev,values[i]);
    };

   UNPROTECT;

   fclose(inf);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_show_args -- check if command requires arg display	*/
/*	EDT_ctbl_ignore_cmd -- check if command is ignored in counting	*/
/*	EDT_ctbl_cmd_group -- return command group of command		*/
/*	EDT_ctbl_normalize -- return command normalization		*/
/*	EDT_ctbl_routine -- return command routine			*/
/*	EDT_ctbl_cmd_close -- check if command closes view		*/
/*	EDT_ctbl_macro_abort -- check if abort macro if cmd fails	*/
/*	EDT_ctbl_confirm -- get confirmation message if any		*/
/*									*/
/************************************************************************/


Boolean
EDT_ctbl_show_args(cmd)
   EDT_CMD_ID cmd;
{
   return ((command_table[cmd].flags & EDT_CMD_FLAG_SHOW_ARGS) != 0);
};






Boolean
EDT_ctbl_ignore_cmd(cmd)
   EDT_CMD_ID cmd;
{
   return ((command_table[cmd].flags & EDT_CMD_FLAG_NO_COMMAND) != 0);
};






Integer
EDT_ctbl_cmd_group(cmd)
   EDT_CMD_ID cmd;
{
   return (command_table[cmd].flags & EDT_CMD_FLAG_GROUP_NUM);
};






Function_Ptr
EDT_ctbl_routine(cmd)
   EDT_CMD_ID cmd;
{
   return command_table[cmd].rtn;
};






String
EDT_ctbl_cmd_name(cmd)
   EDT_CMD_ID cmd;
{
   return command_table[cmd].name;
};






Boolean
EDT_ctbl_cmd_close(cmd)
   EDT_CMD_ID cmd;
{
   return ((command_table[cmd].flags & EDT_CMD_FLAG_CLOSE) != 0);
};






Boolean
EDT_ctbl_macro_abort(cmd)
   EDT_CMD_ID cmd;
{
   return ((command_table[cmd].flags & EDT_CMD_FLAG_MACRO_ABORT) != 0);
};






Boolean
EDT_ctbl_no_prev_set(cmd)
   EDT_CMD_ID cmd;
{
   return ((command_table[cmd].flags & EDT_CMD_FLAG_NO_PSET) != 0);
};






String
EDT_ctbl_confirm(cmd,buf,fg)
   EDT_CMD_ID cmd;
   String buf;
   Boolean fg;
{

   if ((command_table[cmd].flags & EDT_CMD_FLAG_CONFIRM) == 0) return NULL;

   if (fg) sprintf(buf,"%s completed successfully",command_table[cmd].name);
   else sprintf(buf,"%s failed",command_table[cmd].name);

   return buf;
};






/************************************************************************/
/*									*/
/*	EDT_ctbl_map_key -- do key mapping				*/
/*									*/
/************************************************************************/


EDT_CMD_ID
EDT_ctbl_map_key(ei,ch)
   EDT_ID ei;
   Integer ch;
{
   return ei->keymap.key[ch];
};






/************************************************************************/
/*									*/
/*	EDT_ctbl_mode_select -- get mode from command table		*/
/*	EDT_ctbl_indent -- compute indentation for current line 	*/
/*	EDT_ctbl_word_bounds -- compute word limits			*/
/*	EDT_ctbl_sent_bounds -- compute sentence limits 		*/
/*	EDT_ctbl_para_bounds -- compuate paragraph limits		*/
/*	EDT_ctbl_end_of_line -- handle insert at end of line		*/
/*									*/
/************************************************************************/


String
EDT_ctbl_mode_select(ei)
   EDT_ID ei;
{
   register String s;

   if (ei->hooks[EDT_HOOK_MODE_SELECT] != NULL) {
      s = (String) (*ei->hooks[EDT_HOOK_MODE_SELECT])(ei->filename,ei->edit_file);
    }
   else s = NULL;

   return s;
};





Integer
EDT_ctbl_indent(ei)
   EDT_ID ei;
{
   register Integer i;

   if (ei->hooks[EDT_HOOK_INDENT] != NULL) {
      i = (*ei->hooks[EDT_HOOK_INDENT])(ei->edit_file);
    }
   else {
      i = FILEinq_autoindent(ei->edit_file);
    };

   return i;
};





Integer
EDT_ctbl_word_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   register Integer i;

   if (ei->hooks[EDT_HOOK_WORD_BOUNDS] != NULL) {
      i = (*ei->hooks[EDT_HOOK_WORD_BOUNDS])(ei->edit_file,fp,sp,ep);
    }
   else {
      i = EDT_word_bounds(ei,fp,sp,ep);
    };

   return i;
};





Integer
EDT_ctbl_sent_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   register Integer i;

   if (ei->hooks[EDT_HOOK_SENT_BOUNDS] != NULL) {
      i = (*ei->hooks[EDT_HOOK_SENT_BOUNDS])(ei->edit_file,fp,sp,ep);
    }
   else {
      i = EDT_line_bounds(ei,fp,sp,ep);
    };

   return i;
};





Integer
EDT_ctbl_para_bounds(ei,fp,sp,ep)
   EDT_ID ei;
   FILE_POS * fp;
   FILE_POS * sp;
   FILE_POS * ep;
{
   register Integer i;

   if (ei->hooks[EDT_HOOK_PARA_BOUNDS] != NULL) {
      i = (*ei->hooks[EDT_HOOK_PARA_BOUNDS])(ei->edit_file,fp,sp,ep);
    }
   else {
      i = EDT_para_bounds(ei,fp,sp,ep);
    };

   return i;
};





void
EDT_ctbl_end_of_line(ei)
   EDT_ID ei;
{
   if (ei->hooks[EDT_HOOK_END_OF_LINE] != NULL) {
      (*ei->hooks[EDT_HOOK_END_OF_LINE])(ei->edit_file);
    };
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_find_command -- lookup command by name 		*/
/*									*/
/************************************************************************/


EDT_CMD_ID
EDT_ctbl_find_command(nm)
   String nm;
{
   register Integer i;

   for (i = 0; i < num_commands; ++i) {
      if (STREQL(nm,command_table[i].name)) break;
    };

   if (i >= num_commands) return EDT_CMD_UNDEFINED;

   return (EDT_CMD_ID) i;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_arg_list -- return set of arguments			*/
/*									*/
/************************************************************************/


Integer
EDT_ctbl_arg_list(cmd,args)
   EDT_CMD_ID cmd;
   EDT_VAR args[];
{
   register Integer i;

   for (i = 0; i < command_table[cmd].nvars; ++i) {
      args[i] = &var_table[command_table[cmd].vars[i]];
    };

   return command_table[cmd].nvars;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_setup -- setup new EDT_ID for commands 		*/
/*									*/
/************************************************************************/


void
EDT_ctbl_setup(ei)
   EDT_ID ei;
{
   register Integer i;
   register EDT_VAR ev;

   ei->menus = NULL;

   for (i = 0; i < MAX_EDT_KEYS; ++i) {
      ei->keymap.key[i] = EDT_CMD_IGNORE;
    };

   EDT_var_setup(ei);

   for (i = 0; i < num_vars; ++i) {
      ev = &var_table[i];
      EDT_var_set_value(ei,ev,ev->dflt);
    };

   for (i = 0; i < EDT_NUM_HOOKS; ++i) {
      ei->hooks[i] = NULL;
    };
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_define -- define new command				*/
/*									*/
/************************************************************************/


void
EDT_ctbl_define(nm,args,rtn,flags)
   String nm;
   Sequence args;
   String rtn;
   Integer flags;
{
   register Integer i,j;
   register Sequence l;
   register EDT_VAR ev;

   PROTECT;

   for (i = 0; i < num_commands; ++i) {
      if (STREQL(nm,command_table[i].name)) break;
    };

   if (i >= num_commands) i = num_commands++;

   if (command_table_size <= num_commands) {
      command_table_size *= 2;
      command_table = (EDT_CMD_INFO *)
	 realloc(command_table,command_table_size*sizeof(EDT_CMD_INFO));
    };


   strcpy(command_table[i].name,nm);
   strcpy(command_table[i].rtn_name,rtn);
   command_table[i].rtn = image_address(rtn);
   command_table[i].flags = flags;
   command_table[i].help = NULL;

   j = 0;
   forin (ev,EDT_VAR,l,args) {
      ev = var_enter(ev);
      if (ev->loc < 0) {
	 ev->loc = num_vars++;
	 if (num_vars >= var_table_size) {
	    var_table_size *= 2;
	    var_table = (EDT_VAR_INFO *) realloc(var_table,var_table_size*sizeof(EDT_VAR_INFO));
	  };
	 var_table[ev->loc] = *ev;
	 free(ev);
	 ev = &var_table[ev->loc];
       };
      command_table[i].vars[j++] = ev->loc;
    };
   command_table[i].nvars = j;

   SFREE(nm);
   SFREE(rtn);
   LFREE(args);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_subcommand -- define a subcommand of a command 	*/
/*									*/
/************************************************************************/


void
EDT_ctbl_subcommand(nm,args,cmd,flags)
   String nm;
   Sequence args;
   String cmd;
   Integer flags;
{
   register Integer i;
   register Sequence nargs;
   register EDT_VAR ev,nev,tev;
   register EDT_CMD_ID eci;
   register Universal val;

   PROTECT;

   eci = EDT_ctbl_find_command(cmd);

   nargs = NULL;
   for (i = 0; i < command_table[eci].nvars; ++i) {
      ev = &var_table[command_table[eci].vars[i]];
      if (EMPTY(args)) nev = NULL;
      else {
	 nev = CAR(EDT_VAR,args);
	 args = CDRF(args);
       };

      if (nev == NULL || STREQL(nev->name,"*")) {
	 tev = ev;
       }
      else if (STREQL(nev->name,"&")) {
	 tev = copy_var(ev);
       }
      else if (nev->name[0] == 0) {		/* constant */
	 if (nev->type == EDT_VAR_TYPE_ENUM) {
	    val = (Universal) EDT_value_enum(ev->type,nev->dflt);
	    SFREE(nev->dflt);
	    nev->dflt = val;
	  };
	 tev = copy_var(ev);
	 tev->dflt = EDT_value_copy(tev->type,nev->dflt);
	 tev->name[0] = 0;
       }
      else {
	 tev = nev;
	 nev = NULL;
       };

      nargs = APPEND(tev,nargs);
      if (nev != NULL) free(nev);
    };

   if (flags & EDT_CMD_FLAG_SAME) {
      flags &= ~EDT_CMD_FLAG_SAME;
      flags |= command_table[eci].flags;
    };

   EDT_ctbl_define(nm,nargs,SALLOC(command_table[eci].rtn_name),flags);
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_new_var -- create new variable 			*/
/*	EDT_ctbl_make_global -- make variable globally known		*/
/*	EDT_ctbl_find_global -- find global variable			*/
/*									*/
/************************************************************************/


EDT_VAR
EDT_ctbl_new_var(nam,dsc,typ,dflt,idx)
   String nam;
   String dsc;
   EDT_VAR_TYPE typ;
   Universal dflt;
   Integer idx;
{
   register EDT_VAR ev;

   ev = PALLOC(EDT_VAR_INFO);

   ev->type = typ;
   ev->dflt = dflt;
   ev->index = idx;
   ev->loc = -1;
   ev->global = FALSE;
   ev->temp = FALSE;

   if (nam != NULL) {
      if (nam[0] == '@') {
	 ev->temp = TRUE;
	 strcpy(ev->name,&nam[1]);
       }
      else strcpy(ev->name,nam);
    }
   else ev->name[0] = 0;

   if (dsc != NULL) {
      strcpy(ev->description,dsc);
    }
   else ev->description[0] = 0;

   if (ev->index < 0) ev = EDT_ctbl_make_global(ev);

   return ev;
};





EDT_VAR
EDT_ctbl_make_global(ev)
   EDT_VAR ev;
{
   ev = var_enter(ev);
   ev->global = TRUE;

   return ev;
};




EDT_VAR
EDT_ctbl_find_global(nm)
   String nm;
{
   register Integer i;
   register EDT_VAR ev;

   for (i = 0; i < num_vars; ++i) {
      ev = &var_table[i];
      if (ev->global && STREQL(ev->name,nm)) break;
    };

   if (i >= num_vars) ev = NULL;

   return ev;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_new_enum -- define new enumeration type		*/
/*	EDT_ctbl_find_type -- find type given name			*/
/*	EDT_ctbl_enum_value -- get value for enumeratioon constant	*/
/*									*/
/************************************************************************/


EDT_VAR_TYPE
EDT_ctbl_new_enum(nam,txt)
   String nam;
   String txt;
{
   if (num_types >= type_table_size) {
      type_table_size *= 2;
      type_table = (EDT_ENUM_TYPE_INFO *)
	 realloc(type_table,type_table_size*sizeof(EDT_ENUM_TYPE_INFO));
    };

   if (nam == NULL) type_table[num_types].name[0] = 0;
   else strcpy(type_table[num_types].name,nam);

   strcpy(type_table[num_types].text,txt);

   return (EDT_VAR_TYPE) (num_types++);
};





EDT_VAR_TYPE
EDT_ctbl_find_type(nam)
   String nam;
{
   register Integer i;

   for (i = 1; i < num_types; ++i) {
      if (STREQL(nam,type_table[i].name)) break;
    };

   if (i >= num_types) i = EDT_VAR_TYPE_UNDEF;

   return (EDT_VAR_TYPE) i;
};





String
EDT_ctbl_inq_enums(ty)
   EDT_VAR_TYPE ty;
{
   return type_table[ty].text;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_new_bind -- define new binding 			*/
/*									*/
/************************************************************************/


void
EDT_ctbl_new_bind(keys,cmd,when)
   String keys;
   EDT_CMD_ID cmd;
   String when;
{
   PROTECT;

   if (num_binds >= bind_table_size) {
      bind_table_size *= 2;
      bind_table = (EDT_BIND_INFO *)
	 realloc(bind_table,bind_table_size*sizeof(EDT_BIND_INFO));
    };

   strcpy(bind_table[num_binds].keys,keys);

   bind_table[num_binds].cmd = cmd;

   if (when == NULL) bind_table[num_binds].when[0] = 0;
   else {
      strcpy(bind_table[num_binds].when,when);
    };

   ++num_binds;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_rebind -- rebind all keys and menus			*/
/*									*/
/************************************************************************/


void
EDT_ctbl_rebind(ei)
   EDT_ID ei;
{
   register Sequence l,nl,new;
   register Integer i;

   PROTECT;

   new = bind_menu_names(ei);

   nl = new;
   l = ei->menus;
   while (nl != NULL && l != NULL && STREQL(CAR(String,l),CAR(String,nl))) {
      STEMpdm_menu_remove_btns(ei->menuwin,CAR(String,l));
      nl = CDR(nl);
      l = CDR(l);
    };

   while (l != NULL) {
      STEMpdm_menu_remove(ei->menuwin,CAR(String,l));
      l = CDR(l);
    };

   while (nl != NULL) {
      STEMpdm_menu_add(ei->menuwin,CAR(String,nl));
      nl = CDR(nl);
    };

   LFREE(ei->menus);
   ei->menus = new;

   for (i = 0; i < MAX_EDT_KEYS; ++i) {
      ei->keymap.key[i] = EDT_CMD_IGNORE;
    };

   for (i = 0; i < num_binds; ++i) {
      do_binding(ei,&bind_table[i]);
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_help -- associate help information with command	*/
/*	EDT_ctbl_inq_help -- return help information for command	*/
/*	EDT_ctbl_inq_name -- return name for command			*/
/*									*/
/************************************************************************/


void
EDT_ctbl_help(cmd,value)
   EDT_CMD_ID cmd;
   String value;
{
   if (cmd >= 0) {
      command_table[cmd].help = value;
    };
};






String
EDT_ctbl_inq_help(cmd)
   EDT_CMD_ID cmd;
{
   if (cmd >= 0) return command_table[cmd].help;

   return NULL;
};






String
EDT_ctbl_inq_name(cmd)
   EDT_CMD_ID cmd;
{
   if (cmd >= 0) return command_table[cmd].name;

   return NULL;
};






/********************************************************************************/
/*										*/
/*	EDT_ctbl_inq_cmds -- return list of all command names			*/
/*										*/
/********************************************************************************/


Integer
EDT_ctbl_inq_cmds(help,mx,nms)
   Boolean help;
   Integer mx;
   String nms[];
{
   Integer i,j;

   j = 0;
   for (i = 0; i < num_commands; ++i) {
      if (!help || command_table[i].help != NULL)
	 nms[j++] = command_table[i].name;
      if (j >= mx) break;
    };

   return j;
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_load_file -- specify file to load dynamically		*/
/*									*/
/************************************************************************/


void
EDT_ctbl_load_file(nm)
   String nm;
{
   DLload(nm);
};





/************************************************************************/
/*									*/
/*	EDT_ctbl_check_mode -- check for a mode match			*/
/*									*/
/************************************************************************/


Boolean
EDT_ctbl_check_mode(cur,mat)
   String cur;
   String mat;
{
   String s,ns,t,nt;
   Character buf[256],tbf[256];
   Boolean fg;

   if (mat == NULL || mat[0] == 0) return TRUE;

   fg = TRUE;
   for (s = cur; s != NULL; s = ns) {
      ns = index(s,',');
      if (ns != NULL) {
	 ++ns;
	 strcpy(buf,s);
	 s = index(buf,',');
	 *s = 0;
	 s = buf;
       };
      for (t = mat; t != NULL; t = nt) {
	 nt = index(t,',');
	 if (nt != NULL) {
	    ++nt;
	    strcpy(tbf,t);
	    t = index(tbf,',');
	    *t = 0;
	    t = tbf;
	  };
	 if (t[0] != '^' && STREQL(s,t)) return TRUE;
	 if (t[0] != '^') fg = FALSE;
       };
    };

   for (s = cur; s != NULL; s = ns) {
      ns = index(s,',');
      if (ns != NULL) {
	 ++ns;
	 strcpy(buf,s);
	 s = index(buf,',');
	 *s = 0;
	 s = buf;
       };
      for (t = mat; t != NULL; t = nt) {
	 nt = index(t,',');
	 if (nt != NULL) {
	    ++nt;
	    strcpy(tbf,t);
	    t = index(tbf,',');
	    *t = 0;
	    t = tbf;
	  };
	 if (t[0] == '^' && STREQL(s,&t[1])) return FALSE;
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	copy_var -- copy a variable block				*/
/*									*/
/************************************************************************/


static EDT_VAR
copy_var(ev)
   EDT_VAR ev;
{
   register EDT_VAR nev;

   nev = PALLOC(EDT_VAR_INFO);

   *nev = *ev;

   nev->dflt = EDT_value_copy(nev->type,nev->dflt);
   nev->loc = -1;
   nev->index = 0;

   return nev;
};





/************************************************************************/
/*									*/
/*	do_binding -- setup a binding request				*/
/*									*/
/************************************************************************/


static void
do_binding(ei,bd)
   EDT_ID ei;
   EDT_BIND bd;
{
   Integer delta;
   Boolean ctrlfg,fctfg,menufg;
   Integer i,j;
   register String keys;
   Character menu[32];
   EDT_CMD_ID cmd;

   if (!EDT_ctbl_check_mode(ei->mode,bd->when)) return;

   if (EDT_CMD_IS_HOOK(bd->cmd)) {
      bind_hook(ei,bd);
      return;
    };

   delta = 0;
   ctrlfg = FALSE;
   fctfg = FALSE;
   menufg = FALSE;
   keys = bd->keys;
   cmd = bd->cmd;

   while (*keys != '[') {
      if (*keys == 0) return;
      switch (*keys++) {
	 case 'A' :
	    break;
	 case 'C' :
	    ctrlfg = TRUE;
	    break;
	 case 'M' :
	    delta = FKEY_META_FIRST;
	    break;
	 case 'S' :
	    delta = FKEY_FIRST_SHIFT;
	    fctfg = TRUE;
	    break;
	 case 'U' :
	    delta = FKEY_FIRST_UP;
	    fctfg = TRUE;
	    break;
	 case 'E' :
	    delta = EDT_ESCAPE_FIRST;
	    break;
	 case 'F' :
	    fctfg = TRUE;
	    break;
	 case 'P' :
	    menufg = TRUE;
	    break;
       };
    };
   ++keys;

   if (fctfg) {
      if (ctrlfg) {
	 delta = FKEY_FIRST_CTRL;
	 ctrlfg = FALSE;
       }
      else if (delta == FKEY_META_FIRST) delta = FKEY_FIRST_CTRL;
      else if (delta == 0) delta = FKEY_FIRST;
    };

   if (menufg) {
      j = 0;
      while (*keys != ']') menu[j++] = *keys++;
      menu[j] = 0;
      if (cmd >= 0) {
	 add_to_menu(ei,menu,command_table[cmd].name);
       }
      else if (cmd == EDT_CMD_IGNORE) {
	 remove_menu(ei,menu);
       };
    }
   else if (!fctfg) {
      j = -1;
      while (*keys != ']') {
	 i = *keys++;
	 if (i != '-') j = -1;
	 else if (j >= 0) i = *keys++;
	 if (i == '\\') {
	    i = *keys++;
	    if (isdigit(i)) {
	       i -= '0';
	       if (isdigit(*keys)) i = i*8 + *keys++ -'0';
	       if (isdigit(*keys)) i = i*8 + *keys++ -'0';
	     };
	  };
	 if (ctrlfg) {
	    if (i == '?') i = '\177';
	    else i &= 0x1f;
	  };
	 if (j < 0) {
	    bind_key(ei,i+delta,cmd);
	    j = i;
	  }
	 else {
	    for (j = j+1; j <= i; ++j) {
	       bind_key(ei,j+delta,cmd);
	     };
	    j = -1;
	  };
       };
    }
   else {
      if (ctrlfg) delta = FKEY_FIRST_CTRL;
      else if (delta == 0) delta = FKEY_FIRST;
      while (*keys != ']') {
	 i = *keys++ - '0';
	 while (isdigit(*keys)) i = i*10 + *keys++ - '0';
	 if (*keys == ',') ++keys;
	 bind_key(ei,i+delta,cmd);
       };
    };
};





/************************************************************************/
/*									*/
/*	bind_menu_names -- return list of menu names for this binding	*/
/*									*/
/************************************************************************/


static Sequence
bind_menu_names(ei)
   EDT_ID ei;
{
   Boolean menufg;
   Integer j,k;
   String keys;
   Character menu[32];
   Sequence menus,l;
   String s;
   EDT_BIND bd;

   menus = NULL;

   for (k = 0; k < num_binds; ++k) {
      bd = &bind_table[k];
      if (!EDT_ctbl_check_mode(ei->mode,bd->when)) continue;
      if (EDT_CMD_IS_HOOK(bd->cmd)) continue;

      menufg = FALSE;
      for (keys = bd->keys; *keys != 0 && *keys != '['; ++keys) {
	 if (*keys == 'P') menufg = TRUE;
       };
      if (!menufg || *keys++ == 0) continue;

      j = 0;
      while (*keys != ']') menu[j++] = *keys++;
      menu[j] = 0;

      if (bd->cmd >= 0) {
	 forin (s,String,l,menus) {
	    if (STREQL(menu,s)) break;
	  };
	 if (EMPTY(l)) menus = APPEND(SALLOC(menu),menus);
       }
      else if (bd->cmd == EDT_CMD_IGNORE) {
	 forin (s,String,l,menus) {
	    if (STREQL(menu,s)) break;
	  };
	 if (!EMPTY(l)) menus = REMOB(s,menus);
       };
    };

   return menus;
};





/************************************************************************/
/*									*/
/*	add_to_menu -- add button to pull down menu			*/
/*	remove_menu -- remove a menu					*/
/*									*/
/************************************************************************/


static void
add_to_menu(ei,menu,btn)
   EDT_ID ei;
   String menu;
   String btn;
{
   register String s;
   register Sequence l;

   if (ei->menuwin == NULL) return;

   forin (s,String,l,ei->menus) {
      if (STREQL(s,menu)) break;
    };

   if (!EMPTY(l)) {
      STEMpdm_btn_add(ei->menuwin,menu,btn,EDT_eval_menu_btn);
    };
};





static void
remove_menu(ei,menu)
   EDT_ID ei;
   String menu;
{
   String s;
   Sequence l;

   if (ei->menuwin == NULL) return;

   forin (s,String,l,ei->menus) {
      if (STREQL(s,menu)) break;
    };

   if (!EMPTY(l)) {
      STEMpdm_menu_remove_btns(ei->menuwin,menu);
    };
};





/************************************************************************/
/*									*/
/*	bind_key -- bind individual key to command			*/
/*									*/
/************************************************************************/


static void
bind_key(ei,key,cmd)
   EDT_ID ei;
   Integer key;
   EDT_CMD_ID cmd;
{
   ei->keymap.key[key] = cmd;
};





/************************************************************************/
/*									*/
/*	bind_hook -- bind a hook					*/
/*									*/
/************************************************************************/


static void
bind_hook(ei,bd)
   EDT_ID ei;
   EDT_BIND bd;
{
   Integer id;
   Function_Ptr rtn;

   id = EDT_CMD_HOOK_ID(bd->cmd);
   rtn = image_address(bd->keys);

   ei->hooks[id] = rtn;
};





/************************************************************************/
/*									*/
/*	var_enter -- enter variable in var table			*/
/*									*/
/************************************************************************/


static EDT_VAR
var_enter(ev)
   EDT_VAR ev;
{
   if (ev->loc < 0) {
      ev->loc = num_vars++;
      if (num_vars >= var_table_size) {
	 var_table_size *= 2;
	 var_table = (EDT_VAR_INFO *) realloc(var_table,var_table_size*sizeof(EDT_VAR_INFO));
       };
      var_table[ev->loc] = *ev;
      free(ev);
      ev = &var_table[ev->loc];
    };

   return ev;
};





/************************************************************************/
/*									*/
/*	image_address -- get address of routine 			*/
/*									*/
/************************************************************************/


static Function_Ptr
image_address(rtn)
   String rtn;
{
   register Function_Ptr rp;
   Integer i;

   for (i = 0; local_cmds[i].name != NULL; ++i) {
      if (STREQL(rtn,local_cmds[i].name)) return (Function_Ptr) local_cmds[i].rtn;
    };

   rp = DLlookup(rtn);

   return rp;
};





/************************************************************************/
/*									*/
/*	define_system_vars -- define system variables			*/
/*									*/
/************************************************************************/


static void
define_system_vars()
{
   EDT_ctbl_new_var("last_key_struck",NULL,EDT_VAR_TYPE_INT,0,
		       -last_key_struck_INDEX);
   EDT_ctbl_new_var("get_args",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -get_args_INDEX);
   EDT_ctbl_new_var("insert_mode",NULL,EDT_VAR_TYPE_BOOL,TRUE,
		       -insert_mode_INDEX);
   EDT_ctbl_new_var("auto_indent",NULL,EDT_VAR_TYPE_BOOL,TRUE,
		       -auto_indent_INDEX);
   EDT_ctbl_new_var("raw_mode",NULL,EDT_VAR_TYPE_BOOL,TRUE,
		       -raw_mode_INDEX);
   EDT_ctbl_new_var("hold_mode",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -hold_mode_INDEX);
   EDT_ctbl_new_var("open_mode",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -open_mode_INDEX);
   EDT_ctbl_new_var("abbrev_mode",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -abbrev_mode_INDEX);
   EDT_ctbl_new_var("extend_mode",NULL,EDT_VAR_TYPE_BOOL,TRUE,
		       -extend_mode_INDEX);
   EDT_ctbl_new_var("use_selection_mode",NULL,EDT_VAR_TYPE_BOOL,TRUE,
		       -usesel_mode_INDEX);
   EDT_ctbl_new_var("back_over_eol_mode",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -backeol_mode_INDEX);
   EDT_ctbl_new_var("previous_file",NULL,EDT_VAR_TYPE_STRING,NULL,
		       -previous_file_INDEX);
   EDT_ctbl_new_var("previous_line",NULL,EDT_VAR_TYPE_INT,0,
		       -previous_line_INDEX);
   EDT_ctbl_new_var("always_replace",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -always_replace_INDEX);
   EDT_ctbl_new_var("previous_command",NULL,EDT_VAR_TYPE_INT,0,
		       -previous_cmd_INDEX);
   EDT_ctbl_new_var("auto_linefeed",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -auto_linefeed_INDEX);
   EDT_ctbl_new_var("current_indent",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -cur_indent_INDEX);
   EDT_ctbl_new_var("auto_select_all",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -auto_select_all_INDEX);
   EDT_ctbl_new_var("arguments",NULL,EDT_VAR_TYPE_STRING,NULL,
		       -scan_args_INDEX);
   EDT_ctbl_new_var("readonly",NULL,EDT_VAR_TYPE_BOOL,FALSE,
		       -readonly_mode_INDEX);
};





/* end of edtctbl.c */


