/************************************************************************/
/*									*/
/*		edtabbrev.c						*/
/*									*/
/*	Abbreviation package for edt editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

/*
 * Copyright (c) 1980 Regents of the University of California.
 * All rights reserved.  The Berkeley Software License Agreement
 * specifies the terms and conditions for redistribution.
 */


#include "edt_local.h"

#undef MIN			/* these are defined in param.h 	*/
#undef MAX
#include <sys/param.h>

#include <sys/types.h>
#include <sys/dir.h>
#include <pwd.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_FILE_EXTEND 	16





/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _ABBREV *	ABBREV;
typedef struct _ABBREV_INST *	ABBREV_INST;
typedef struct _ABBREV_ROOT *	ABBREV_ROOT;


typedef struct _ABBREV {
   String source;
   ABBREV_INST use;
   ABBREV lson;
   ABBREV rson;
} ABBREV_INFO;



typedef struct _ABBREV_INST {
   String mode;
   String target;
   ABBREV_INST next;
} ABBREV_INST_INFO;



typedef struct _ABBREV_ROOT {
   ABBREV root;
   EDT_ID editor;
   ABBREV_ROOT next;
} ABBREV_ROOT_INFO;




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	ABBREV_ROOT		all_roots;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		delete_all();
static	void		save_abbrev();
static	ABBREV_ROOT	find_root();
static	ABBREV		find_abbrev();

static	Integer 	complete_search();
static	void		extract_dir_and_name();
static	String		tilde();
static	String		getentry();
static	Boolean 	ignored();
static	Boolean 	recognize();
static	Boolean 	is_prefix();
static	void		copyn();
static	void		catn();
static	void		ask_for_item();
static	int		sortscmp();






/************************************************************************/
/*									*/
/*	EDT_abbrev_init -- module initialization			*/
/*									*/
/************************************************************************/


void
EDT_abbrev_init()
{
   all_roots = NULL;
};





/************************************************************************/
/*									*/
/*	EDCMDAbbrevDefine -- define an abbreviation			*/
/*									*/
/************************************************************************/


Boolean
EDCMDAbbrevDefine(ev,abb,buf,txt,mode)
   EDT_VIEW ev;
   String abb;
   EDT_BUF buf;
   String txt;
   Boolean mode;
{
   Boolean fre;
   String md;

   EDT_clear_selection(ev->edit,DEFAULT_BUFFER_NAME);

   if (abb == NULL || *abb == 0) return FALSE;

   fre = FALSE;

   if (txt == NULL || *txt == 0) {
      if (buf != NULL) {
	 txt = FILEbuffer_contents(buf->buffer);
	 fre = TRUE;
       };
    };

   if (txt == NULL || *txt == 0) return FALSE;

   md = (mode ? ev->edit->mode : NULL);

   EDT_abbrev_define(ev->edit,abb,txt,md);

   if (fre && txt != NULL) free(txt);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	EDCMDAbbrevExpand -- expand abbreviation before position	*/
/*									*/
/************************************************************************/


Boolean
EDCMDAbbrevExpand(ev,pfx)
   EDT_VIEW ev;
   Boolean pfx;
{
   String line;
   FILE_POS p1,p2;
   Integer col,col1,col0;
   String s;
   Integer ch;

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&p1);
   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_END,0,&p2);

   line = FILEcopy_text(VIEW_FILE(ev),&p1,&p2);

   if (line == NULL) return FALSE;

   col = FILEinq_currchar(VIEW_FILE(ev)) - 1;
   col1 = col-1;

   line[col--] = 0;
   while (col >= 0 && isspace(line[col])) line[col--] = 0;
   col0 = col;
   while (col >= 0 && isalnum(line[col])) --col;
   ++col;

   if (line[col] == 0) s = NULL;
   else s = EDT_abbrev_find(ev->edit,&line[col],pfx);

   if (s == NULL) return FALSE;

   FILEset_position(VIEW_FILE(ev),FILEinq_currline(VIEW_FILE(ev)),col+1,&p1);
   FILEdelete(VIEW_FILE(ev),FILEinq_buffer("BWEDIT$_ABBREV"),
		 FILE_MOVE_REL|FILE_MOVE_CHAR,col0-col+1);

   while (*s != NULL) {
      ch = *s++;
      FILEtypein(VIEW_FILE(ev),ch,TRUE);
      if (ch == '\n') EDT_indent_on_line(ev,TRUE,FALSE);
    };

   if (col0 != col1) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,col1-col0,&p1);
    };

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDAbbrevLoad -- load abbreviations from file 		*/
/*	EDCMDAbbrevStore -- save abbreviations into file		*/
/*	EDCMDAbbrevClear -- clear all current abbreviations		*/
/*									*/
/************************************************************************/


Boolean
EDCMDAbbrevLoad(ev,file)
   EDT_VIEW ev;
   String file;
{
   return EDT_abbrev_load(ev->edit,file);
};





Boolean
EDCMDAbbrevStore(ev,file)
   EDT_VIEW ev;
   String file;
{
   return EDT_abbrev_save(ev->edit,file);
};





Boolean
EDCMDAbbrevClear(ev)
   EDT_VIEW ev;
{
   EDT_abbrev_clear(ev->edit);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDCMDAbbrevComplete -- handle filename completion		*/
/*									*/
/************************************************************************/


Boolean
EDCMDAbbrevComplete(ev,ask)
   EDT_VIEW ev;
   Boolean ask;
{
   String line;
   FILE_POS p1,p2;
   Integer col,col1,col0;
   String s;
   Integer ch,numitm;
   Character buf[MAXNAMLEN];

   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_START,0,&p1);
   FILEinq_position(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_LINE_END,0,&p2);

   line = FILEcopy_text(VIEW_FILE(ev),&p1,&p2);

   if (line == NULL) return FALSE;

   col = FILEinq_currchar(VIEW_FILE(ev)) - 1;
   col1 = col-1;

   while (line[col] != 0 && !isspace(line[col])) ++col;
   line[col--] = 0;
   while (col >= 0 && isspace(line[col])) line[col--] = 0;
   col0 = col;
   while (col >= 0 && !isspace(line[col])) --col;
   ++col;

   if (line[col] == 0) s = NULL;
   else {
      numitm = complete_search(ev,&line[col],buf,MAXNAMLEN,ask);
      if (numitm != 1 && !ask) ASHbell();
      if (numitm == 0) s = NULL;
      else s = buf;
    };

   if (s == NULL) return FALSE;

   FILEset_position(VIEW_FILE(ev),FILEinq_currline(VIEW_FILE(ev)),col+1,&p1);
   FILEdelete(VIEW_FILE(ev),FILEinq_buffer("BWEDIT$_ABBREV"),
		 FILE_MOVE_REL|FILE_MOVE_CHAR,col0-col+1);

   while (*s != NULL) {
      ch = *s++;
      FILEtypein(VIEW_FILE(ev),ch,TRUE);
    };

   if (col0 != col1) {
      FILEmove(VIEW_FILE(ev),FILE_MOVE_REL|FILE_MOVE_CHAR,col1-col0,&p1);
    };

   EDT_synch_view(ev);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_setup -- setup new editor abbreviations		*/
/*									*/
/************************************************************************/


void
EDT_abbrev_setup(ei)
   EDT_ID ei;
{
   Character buf[1024];

   sprintf(buf,EDT_ABBREV_FILE,BWEbwe_project(),BWEarch());
   if (access(buf,4) < 0) sprintf(buf,EDT_ABBREV_FILE0,BWEbwe_project(),BWEarch());

   EDT_abbrev_load(ei,buf);

   sprintf(buf,EDT_LOCAL_ABBREV,getenv("HOME"));
   EDT_abbrev_load(ei,buf);
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_define -- define an abbreviation			*/
/*									*/
/************************************************************************/


void
EDT_abbrev_define(ei,from,to,mode)
   EDT_ID ei;
   String from;
   String to;
   String mode;
{
   ABBREV_ROOT r;
   ABBREV a;
   ABBREV_INST ai;

   if (mode != NULL && *mode == 0) mode = NULL;

   r = find_root(ei);

   a = find_abbrev(&r->root,from,TRUE,FALSE);

   if (a == NULL) ai = NULL;
   else for (ai = a->use; ai != NULL; ai = ai->next) {
      if (ai->mode == NULL && mode == NULL) break;
      if (ai->mode != NULL && mode != NULL && STREQL(ai->mode,mode)) break;
    };

   if (ai == NULL) {
      ai = PALLOC(ABBREV_INST_INFO);
      ai->target = NULL;
      if (mode == NULL) ai->mode = NULL;
      else ai->mode = SALLOC(mode);
      ai->next = a->use;
      a->use = ai;
    };

   if (ai->target == NULL || STRNEQ(ai->target,to)) {
      if (ai->target != NULL) SFREE(ai->target);
      ai->target = SALLOC(to);
    };
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_clear -- remove all abbreviations			*/
/*	delete_all -- delete whole tree 				*/
/*									*/
/************************************************************************/


void
EDT_abbrev_clear(ei)
   EDT_ID ei;
{
   ABBREV_ROOT r;

   r = find_root(ei);

   delete_all(r->root);

   r->root = NULL;
};





static void
delete_all(a)
   ABBREV a;
{
   ABBREV_INST ai,ain;

   if (a == NULL) return;
   delete_all(a->lson);
   delete_all(a->rson);
   if (a->source != NULL) SFREE(a->source);

   for (ai = a->use; ai != NULL; ai = ain) {
      ain = ai->next;
      if (ai->target != NULL) SFREE(ai->target);
      if (ai->mode != NULL) SFREE(ai->mode);
      free(ai);
    };

   free(a);
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_save -- save abbreviations in a file 		*/
/*									*/
/************************************************************************/


Boolean
EDT_abbrev_save(ei,file)
   EDT_ID ei;
   String file;
{
   ABBREV_ROOT r;
   FILE * otf;

   otf = fopen(file,"w");
   if (otf == NULL) return FALSE;

   r = find_root(ei);

   save_abbrev(r->root,otf);

   fclose(otf);

   return TRUE;
};




static void
save_abbrev(a,otf)
   ABBREV a;
   FILE * otf;
{
   ABBREV_INST ai;
   String s;

   if (a == NULL) return;

   for (ai = a->use; ai != NULL; ai = ai->next) {
      if (ai->target != NULL) {
	 s = (ai->mode == NULL ? "" : ai->mode);
	 fprintf(otf,"\"%s\" \"%s\" \"%s\"\n",a->source,s,ai->target);
       };
    };

   save_abbrev(a->lson,otf);
   save_abbrev(a->rson,otf);
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_load -- load abbreviation file			*/
/*									*/
/************************************************************************/


Boolean
EDT_abbrev_load(ei,file)
   EDT_ID ei;
   String file;
{
   FILE * inf;
   Character sbuf[1024],mbuf[1024],tbuf[10240];
   String s;
   Integer c;
   Boolean fg;

   inf = fopen(file,"r");
   if (inf == NULL) return FALSE;

   fg = FALSE;

   for ( ; ; ) {
      for (c = getc(inf); isspace(c); c = getc(inf));
      if (c == EOF) {
	 fg = TRUE;
	 break;
       };
      if (c != '"') break;
      s = sbuf;
      for (c = getc(inf); c != '"' && c != EOF; c = getc(inf)) *s++ = c;
      *s = 0;
      if (c == EOF) break;
      for (c = getc(inf); isspace(c); c = getc(inf));
      if (c == EOF) break;
      if (c != '"') break;
      s = mbuf;
      for (c = getc(inf); c != '"' && c != EOF; c = getc(inf)) *s++ = c;
      *s = 0;
      if (c == EOF) break;
      for (c = getc(inf); isspace(c); c = getc(inf));
      if (c == EOF) break;
      if (c != '"') break;
      s = tbuf;
      for (c = getc(inf); c != '"' && c != EOF; c = getc(inf)) *s++ = c;
      *s = 0;
      if (c == EOF) break;
      EDT_abbrev_define(ei,sbuf,tbuf,mbuf);
    };

   fclose(inf);

   return fg;
};





/************************************************************************/
/*									*/
/*	EDT_abbrev_find -- find abbreviation				*/
/*									*/
/************************************************************************/


String
EDT_abbrev_find(ei,src,pfx)
   EDT_ID ei;
   String src;
   Boolean pfx;
{
   ABBREV_ROOT r;
   ABBREV a;
   ABBREV_INST ai,dai;

   r = find_root(ei);

   a = find_abbrev(&r->root,src,FALSE,pfx);

   if (a == NULL) return NULL;

   if (pfx && (find_abbrev(&a->lson,src,FALSE,TRUE) != NULL ||
		  find_abbrev(&a->rson,src,FALSE,TRUE) != NULL))
      return NULL;

   dai = NULL;
   for (ai = a->use; ai != NULL; ai = ai->next) {
      if (ai->mode == NULL) dai = ai;
      else if (EDT_ctbl_check_mode(ei->mode,ai->mode)) break;
    };
   if (ai == NULL) ai = dai;

   if (ai == NULL) return NULL;

   return ai->target;
};





/************************************************************************/
/*									*/
/*	find_root -- find the proper root for editor			*/
/*									*/
/************************************************************************/


static ABBREV_ROOT
find_root(ei)
   EDT_ID ei;
{
   ABBREV_ROOT r;

   PROTECT;

   for (r = all_roots; r != NULL; r = r->next) {
      if (r->editor == ei) break;
    };

   if (r == NULL) {
      r = PALLOC(ABBREV_ROOT_INFO);
      r->root = NULL;
      r->editor = ei;
      r->next = all_roots;
      all_roots = r;
    };

   UNPROTECT;

   return r;
};





/************************************************************************/
/*									*/
/*	find_abbrev -- find abbreviation entry				*/
/*									*/
/************************************************************************/


static ABBREV
find_abbrev(fp,id,new,pfx)
   ABBREV * fp;
   String id;
   Boolean new;
   Boolean pfx;
{
   Integer i,ln;
   ABBREV a;

   ln = strlen(id);

   while ((a = *fp) != NULL) {
      if (pfx) i = strncmp(id,a->source,ln);
      else i = strcmp(id,a->source);
      if (i == 0) break;
      else if (i < 0) fp = &a->lson;
      else fp = &a->rson;
    };

   if (a == NULL && new) {
      a = PALLOC(ABBREV_INFO);
      a->source = SALLOC(id);
      a->use = NULL;
      a->lson = NULL;
      a->rson = NULL;
      *fp = a;
    };

   return a;
};





/************************************************************************/
/*									*/
/*	complete_search -- handle search for name completion		*/
/*									*/
/************************************************************************/


static Integer
complete_search(ev,word,buf,maxlen,ask)
   EDT_VIEW ev;
   String word;
   String buf;
   Integer maxlen;
   Boolean ask;
{
   EDT_ID ei;
   DIR * dir_fd;
   Integer numitems;
   Boolean ignoring;
   Integer nignored;
   Integer name_length;
   Integer looking_for_lognames;
   Character tilded_dir[MAXPATHLEN + 1], dir[MAXPATHLEN + 1];
   Character name[MAXNAMLEN + 1], extended_name[MAXNAMLEN + 1];
   String entry;
   String items[MAX_FILE_EXTEND];
   Boolean extended;

   ei = ev->edit;

   if (!LPROTECT(ei)) return FALSE;

   looking_for_lognames = (*word == '~') && (index(word,'/') == NULL);

   if (looking_for_lognames) {
      setpwent();
      copyn(name, & word[1], MAXNAMLEN);
    }
   else {
      extract_dir_and_name(word,dir,name);
      if (tilde(tilded_dir,dir) == 0) return 0;
      dir_fd = opendir(*tilded_dir ? tilded_dir : ".");
      if (dir_fd == NULL) return 0;
    };

   ignoring = FALSE;		/* MAKE TRUE if doing ignoring **********/
   extended = TRUE;

   for ( ; ; ) {
      nignored = 0;
      name_length = strlen(name);
      for (numitems = 0; entry = getentry(dir_fd,looking_for_lognames); ) {
	 if (!is_prefix(name,entry)) continue;
	 if (name_length == 0 && entry[0] == '.' && !looking_for_lognames) continue;
	 if (ignoring && ignored(entry)) nignored++;
	 else {
	    if (ask && numitems < MAX_FILE_EXTEND) {
	       items[numitems] = SALLOC(entry);
	     };
	    if (recognize(extended_name,entry,name_length,++numitems)) {
	       extended = FALSE;
	       if (!ask) break;
	     };
	  };
       };

      if (!ignoring || numitems != 0 || nignored == 0) break;

      ignoring = FALSE;
      if (looking_for_lognames) setpwent();
      else rewinddir(dir_fd);
    }

   if (looking_for_lognames) endpwent();
   else closedir(dir_fd);

   if (numitems == 0) return 0;

   if (looking_for_lognames) copyn(buf,"~",1);
   else copyn(buf,dir,maxlen);

   if (ask && numitems > 1) {
      ask_for_item(ev,dir,numitems,items,extended,extended_name);
      if (extended_name[0] == 0) return 0;
      numitems = 1;
    };

   catn(buf,extended_name,maxlen);

   return numitems;
};






/************************************************************************/
/*									*/
/*	extract_dir_and_name -- separate name from directory		*/
/*									*/
/************************************************************************/


static void
extract_dir_and_name(path,dir,name)
   String path;
   String dir;
   String name;
{
   String p;

   p = rindex(path,'/');
   if (p == NULL) {
      copyn(name,path,MAXNAMLEN);
      dir[0] = 0;
    }
   else {
      copyn(name,++p,MAXNAMLEN);
      copyn(dir,path,p-path);
    };
};





/************************************************************************/
/*									*/
/*	tilde -- expand file name for possible tilde usage		*/
/*									*/
/************************************************************************/


static String
tilde(new,old)
   String new;
   String old;
{
   String o,p;
   struct passwd *pw;
   Character person[64];

   if (old[0] != '~') {
      strcpy(new,old);
      return new;
    };

   for (p = person, o = &old[1]; *o != 0 && *o != '/'; *p++ = *o++);
   *p = 0;

   if (person[0] == 0) {
      strcpy(new,getenv("HOME"));
    }
   else {
      pw = getpwnam(person);
      if (pw == NULL) return NULL;
      strcpy(new,pw->pw_dir);
    };

   strcat(new,o);

   return new;
};





/************************************************************************/
/*									*/
/*	getentry -- return next directory entry 			*/
/*									*/
/************************************************************************/


static String
getentry(dir_fd, looking_for_lognames)
   DIR *dir_fd;
   Boolean looking_for_lognames;
{
   struct passwd *pw;
   struct direct *dirp;

   if (looking_for_lognames) {
      if ((pw = getpwent()) == NULL) return (NULL);
      return (pw->pw_name);
    }

   if (dirp = readdir(dir_fd)) return (dirp->d_name);

   return (NULL);
}





/************************************************************************/
/*									*/
/*	ignored -- check if file should be ignored			*/
/*									*/
/************************************************************************/


static Boolean
ignored(entry)
   String entry;
{
   return FALSE;
};





/************************************************************************/
/*									*/
/*	recognize -- extend name to ambiguity				*/
/*									*/
/************************************************************************/


static Boolean
recognize(extended_name,entry,name_length,numitems)
   String extended_name;
   String entry;
   Integer name_length;
   Integer numitems;
{
   String x,ent;
   Integer len;

   if (numitems == 1) copyn(extended_name,entry,MAXNAMLEN);
   else {
      len = 0;
      x = extended_name;
      for (ent = entry; *x != 0 && *x == *ent++; x++, len++);
      *x = 0;
      if (len == name_length) return TRUE;
    }

   return FALSE;
};





/************************************************************************/
/*									*/
/*	is_prefix -- test if check matches initial part of template	*/
/*									*/
/************************************************************************/


static Boolean
is_prefix(check,template)
   String check;
   String template;
{
   do {
      if (*check == 0) return TRUE;
    }
   while (*check++ == *template++);

   return FALSE;
};





/************************************************************************/
/*									*/
/*	copyn -- copy up to count characters				*/
/*									*/
/************************************************************************/


static void
copyn(des,src,count)
   String des;
   String src;
   Integer count;
{
   while (--count >= 0) {
      if ((*des++ = *src++) == 0) return;
    };

   *des = 0;
};





/************************************************************************/
/*									*/
/*	catn -- concatenate up to a total of count chars		*/
/*									*/
/************************************************************************/


static void
catn(des,src,count)
   String des;
   String src;
   Integer count;
{
   while (--count >= 0 && *des != 0) ++des;

   while (--count >= 0) {
      if ((*des++ = *src++) == 0) return;
    };

   *des = 0;
};





/************************************************************************/
/*									*/
/*	ask_for_item -- ask user to choose proper file			*/
/*	sortscmp -- string comparison for qsort 			*/
/*									*/
/************************************************************************/


static void
ask_for_item(ev,dir,cnt,items,usedflt,dflt)
   EDT_VIEW ev;
   String dir;
   Integer cnt;
   String * items;
   Boolean usedflt;
   String dflt;
{
   String menu[10240],buf[MAXPATHLEN+1];
   Integer i,opt;

   strcpy(menu,"%CFilename Completion\n\n");

   if (dir != NULL && dir[0] != 0) {
      sprintf(buf,"%s ...\n",dir);
    };

   qsort(items,cnt,sizeof(String),sortscmp);

   for (i = 0; i < cnt; ++i) {
      sprintf(buf,"   %%0.%do %s\n",i,items[i]);
      strcat(menu,buf);
    };

   if (usedflt && dflt[0] != 0) {
      sprintf(buf,"\n%%0.%do Extend to %s\n",cnt,dflt);
      strcat(menu,buf);
    };

   strcat(menu,"\n   %a%M   %c ");

   opt = cnt;
   if (!STEMdialog1(ev->window,menu,&opt)) {
      dflt[0] = 0;
      return;
    };

   if (opt != cnt) {
      strcpy(dflt,items[opt]);
    };
};





static int
sortscmp(n1,n2)
   String * n1;
   String * n2;
{
   return strcmp(*n1,*n2);
};






/* end of edtabbrev.c */
