/************************************************************************/
/*									*/
/*		textlocal.c						*/
/*									*/
/*	Local routines for editor display				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		update_display();
static	void		normal_view_typein();
static	void		duplicate_view_typein();
static	void		normal_view_delete();
static	void		duplicate_view_delete();
static	void		normal_view_insert();
static	void		duplicate_view_insert();
static	void		multiple_command_update();
static	void		move_cursor_to();
static	void		highlight_coord();
static	void		update_highlight();





/************************************************************************/
/*									*/
/*   TEXT_move_on_screen - alter the display so that the given file pos */
/*	  will be on the screen.  Try to place it in the middle of the	*/
/*	  screen, but make sure not to show any lines past the eof.	*/
/*									*/
/************************************************************************/


void
TEXT_move_on_screen(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   Integer fp_line,fp_char;	  /* line and character of the file pos 	      */
   Integer top_line,bottom_line;  /* line numbers (file) of top & bottom lines of screen */
   Integer offset;	       /* lines to change if screen can be scrolled	      */

   fp_line = FILEinq_line(tid->file,fp);
   fp_char = FILEinq_char(tid->file,fp);
   top_line = tid->line;
   bottom_line = top_line + tid->numlines - 1;

   ASHbatch_mode(TRUE);

   PROTECT;
   if ((top_line > fp_line) || (fp_line > bottom_line)) { /* above or below the screen	*/
      if ((tid->shell_mode && fp_line > bottom_line) ||
	     fp_line <= bottom_line + 3) {		/* do small scrolls directly */
	 fp_line = top_line + (fp_line - bottom_line);
       }
      else {
	 fp_line = fp_line - (tid->numlines/2); 	/* try to place in the middle */
       };
    }
   else 			 /* left or right of screen */
      fp_line = top_line;	 /* just do a slide	    */

   if (fp_line < 1) fp_line = 1;

   if ((tid->ch <= fp_char) && (fp_char < (tid->ch + tid->numchars))) {  /* try to scroll screen */
      offset = fp_line - top_line;
      if ((offset > 0) && (offset < tid->numlines-1)) { /* positive delta less than a screen so do smart update */
	 TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	 TULIPline_delete(tid->tulip,offset);
	 tid->line += offset;
	 TEXT_new_display(tid,tid->numlines-offset,tid->line + tid->numlines - offset,tid->ch,FULL_SCREEN);
       }
      else if ((offset < 0) && ( (-offset) < tid->numlines-1)) {
	 TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	 TULIPline_insert(tid->tulip,-offset);		      /* negative delta less than a screen so do smart update */
	 tid->line += offset;
	 TEXT_new_display(tid,TULIP_START_LINE,tid->line,tid->ch,-offset);
       }
      else if (offset != 0)			  /* may as well redisplay whole screen */
	 TEXT_new_display(tid,TULIP_START_LINE,fp_line,tid->ch,FULL_SCREEN);
    }
   else {
      if (fp_char < tid->numchars)
	 TEXT_new_display(tid,TULIP_START_LINE,fp_line,1,FULL_SCREEN);
      else
	 TEXT_new_display(tid,TULIP_START_LINE,fp_line,fp_char - (tid->numchars/3),FULL_SCREEN);
    }
   UNPROTECT;

   ASHbatch_mode(FALSE);
}




/************************************************************************/
/*									*/
/*    TEXT_on_screen - return TRUE if the given file pos is currently	*/
/*	  being displayed on the screen.				*/
/*									*/
/************************************************************************/


Boolean
TEXT_on_screen(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   Integer line,ch;
   Integer bottom_line,right_char;

   line = FILEinq_line(tid->file,fp);
   ch = FILEinq_char(tid->file,fp);
   bottom_line = tid->line + tid->numlines - 1;
   right_char = tid->ch + tid->numchars - 1;

   return ((line >= tid->line) && (line <= bottom_line)
	      && (ch >= tid->ch) && (ch <= right_char));
}





/************************************************************************/
/*									*/
/*    TEXT_new_display - rewrite given display window starting at the	*/
/*	  given tulip line, with the given (line,char) position, for	*/
/*	  numlines amount of lines.					*/
/*									*/
/************************************************************************/


void
TEXT_new_display(tid,start_print_line,startline,startchar,numlines)
   TEXT_ID tid;
   Integer start_print_line;	  /* what tulip line (0-numlines) to start redisplay from */
   Integer startline;	      /* actual file's line and char to start from            */
   Integer startchar;
   Integer numlines;	      /* the number of lines to be redisplayed		      */
{
   Integer line;
   Integer endchar,endline;  /* last possible line and char to print */
   FILE_POS startpos,endpos;	   /* FILE_POS' surrounding display region */
   String text,text_start;	   /* text in total region to display	   */
   String printer,printer_start;   /* buffer filled to be printed (slid)   */
   Boolean eol; 		   /* set to TRUE when \n is encountered   */
   Integer i;
   Character ch;

   ASHbatch_mode(TRUE);

   PROTECT;
   if (start_print_line == TULIP_START_LINE) {
      if (startline <= 0) startline = 1;
      tid->line = startline;   /* reset window start coordinates */
      tid->ch = startchar;     /* if necessary		      */
    }

   endchar = startchar + tid->numchars - 1;    /* get bottom-right pos	 */
   if (numlines == FULL_SCREEN)
      endline = tid->numlines - start_print_line;
   else
      endline = MIN(numlines,tid->numlines - start_print_line);

   /* get all text in the region */
   FILEnew_position(tid->file,startline,1,&startpos);
   FILEnew_position(tid->file,startline + endline - 1,endchar + 1,&endpos);
   text = FILEcopy_text(tid->file,&startpos,&endpos);

   TULIPinsertmode(tid->tulip,FALSE);
   VT_NLCLEAR(tid->tulip);
   VT_DISCARD(tid->tulip);	/* very important! */
   TULIPmove(tid->tulip,start_print_line,TULIP_START_CHAR);

   if (startchar == 1) {		    /* display not shifted over */
      if (numlines == FULL_SCREEN)
	 TULIPerase_screen(tid->tulip);
      else
	 TULIPerase_line(tid->tulip);
      TULIPwrite(tid->tulip,text);
      if (text != NULL) free(text);
      if (tid->insert_mode) TULIPinsertmode(tid->tulip,TRUE);
      VT_NO_NLCLEAR(tid->tulip);
      update_highlight(tid);
      ASHbatch_mode(FALSE);
      UNPROTECT;
      return;
    }

   /* otherwise, display has been slid right.  We must always throw away
   some text at the beginning of each line.			      */

   text_start = text;	 /* "start" variables used to free at end */
   printer = printer_start = (String) malloc(strlen(text) + 1);

   /* the array "printer" is filled up then printed */

   for (line=1; line <= endline; line++) {
      eol = FALSE;
      for (i=1; i<startchar; ++i) { /* throw away text to left of screen */
	 if ((ch = *(text++)) == NEWLINE) {
	    *printer++ = NEWLINE;
	    eol = TRUE;
	    break;
	  }
	 else if (ch == 0) {
	    *printer = 0;
	    if (numlines == FULL_SCREEN) TULIPerase_screen(tid->tulip);
	    TULIPwrite(tid->tulip,printer_start);
	    if (text_start != NULL) free(text_start);
	    if (printer_start != NULL) free(printer_start);
	    if (tid->insert_mode) TULIPinsertmode(tid->tulip,TRUE);
	    VT_NO_NLCLEAR(tid->tulip);
	    update_highlight(tid);
	    ASHbatch_mode(FALSE);
	    UNPROTECT;
	    return;
	  }
       };

      if (!eol) { /* text starting at the left screen edge */
	 while ((ch = *(text++)) != NEWLINE) {
	    *printer++ = ch;  /* add char to print buffer */
	    if (ch == 0) {
	       if (numlines == FULL_SCREEN) TULIPerase_screen(tid->tulip);
	       TULIPwrite(tid->tulip,printer_start);
	       if (text_start != NULL) free(text_start);
	       if (printer_start != NULL) free(printer_start);
	       if (tid->insert_mode) TULIPinsertmode(tid->tulip,TRUE);
	       VT_NO_NLCLEAR(tid->tulip);
	       update_highlight(tid);
	       ASHbatch_mode(FALSE);
	       UNPROTECT;
	       return;
	     }
	  } /* while */
	 *printer++ = NEWLINE;
       } /* if */
    } /* for */

   *printer = 0;
   TULIPwrite(tid->tulip,printer_start);
   if (text_start != NULL) free(text_start);
   if (printer_start != NULL) free(printer_start);
   if (tid->insert_mode) TULIPinsertmode(tid->tulip,TRUE);
   VT_NO_NLCLEAR(tid->tulip);
   update_highlight(tid);
   ASHbatch_mode(FALSE);
   UNPROTECT;
}





/************************************************************************/
/*									*/
/*    TEXT_highlight - highlight the region between the given file pos	*/
/*	 (assume x precedes y in the file).				*/
/*									*/
/************************************************************************/


void
TEXT_highlight(tid,x,y,mode)
   TEXT_ID tid;
   FILE_POS *x;
   FILE_POS *y;
   Integer  mode;
{
   Integer xline,xchar;
   Integer yline,ychar;

   highlight_coord(tid,x,&xline,&xchar);  /* get correct line,ch on screen for highlight */
   highlight_coord(tid,y,&yline,&ychar);

   if ((xline == yline) && (xchar == ychar)) return;

   if (ychar == TULIP_START_CHAR) {
      yline--;			/* only want to highlight to previous character */
      ychar = tid->numchars;
    }
   else ychar--;

   switch (mode) {
      case HIGHLIGHT_OFF :
	 TULIPhighlight(tid->tulip,xline,xchar,yline,ychar,tid->basefont,
			   FALSE,HILITE_1);
	 break;
      case HIGHLIGHT_ON :
	 TULIPhighlight(tid->tulip,xline,xchar,yline,ychar,tid->basefont,
			   TRUE,HILITE_1);
	 break;
      case SECONDARY_OFF :
	 TULIPhighlight(tid->tulip,xline,xchar,yline,ychar,tid->basefont,
			   FALSE,HILITE_2);
	 break;
      case SECONDARY_ON :
	 TULIPhighlight(tid->tulip,xline,xchar,yline,ychar,tid->basefont,
			   TRUE,HILITE_2);
	 break;
    };
}





/************************************************************************/
/*									*/
/*    TEXT_message_handler - get messages from file module and update	*/
/*	  a display correspondingly.					*/
/*									*/
/************************************************************************/


void
TEXT_message_handler(file,msg,from,to,ins_ch,ins_mode)
   FILE_ID   file;
   FILE_MSG  msg;
   FILE_POS *from;
   FILE_POS *to;
   Character ins_ch;
   Boolean   ins_mode;
{
   TEXT_ID     tid;
   TEXT_CHANGE change_node;

   for (tid = TEXT__list; tid != NULL; tid = tid->next) {
      if (tid->file != file) continue;
      switch(msg) {
	 case FILE_MSG_TYPEIN :
	 case FILE_MSG_DELETE :
	 case FILE_MSG_INSERT :
	    PROTECT;
	    change_node = PALLOC(TEXT_CHANGE_INFO);
	    change_node->msg_type = msg;
	    FILEpos_copy(file,from,&(change_node->from));
	    if (msg == FILE_MSG_TYPEIN)
	       change_node->ch = ins_ch;     /* add a text change node */
	    else
	       FILEpos_copy(file,to,&(change_node->to));
	    change_node->ins_mode = ins_mode;
	    change_node->next = NULL;

	    if (tid->update_list == NULL)
	       tid->update_list = tid->update_listend = change_node;
	    else {
	       tid->update_listend->next = change_node;
	       tid->update_listend = change_node;
	     }
	    UNPROTECT;
	    break;

	 case FILE_MSG_CHANGED :
	    break;

	 case FILE_MSG_COMMAND :	  /* on this message receipt, update */
	    update_display(tid);
	    break;

	 case FILE_MSG_CLOSE :		  /* close up a text tid */
	    TEXT_close_local(tid);
	    break;
       }
    };
}






/************************************************************************/
/*									*/
/*    update_display - scan down the given display's update list and    */
/*	     perform an intelligent update of the text.  If there is	*/
/*	     only one update on the list call the appropriate handler.	*/
/*	     If there is more than one change, call the multiple	*/
/*	     command update handler.					*/
/*									*/
/************************************************************************/


static void
update_display(tid)
   TEXT_ID   tid;
{
   FILE_MSG  msg;
   FILE_POS *from;
   FILE_POS *to;
   FILE_POS  fp;
   Boolean   ins_mode;

   ASHbatch_mode(TRUE);
   PROTECT;

   if (tid->update_list == NULL) ;
   else if (tid->update_list == tid->update_listend) {
      msg = tid->update_list->msg_type;
      from = &(tid->update_list->from);
      to = &(tid->update_list->to);

      switch(msg) {		 /* if there's only one change on the list, */
				 /*  do an intelligent update		    */
	 case FILE_MSG_TYPEIN :
	    ins_mode = tid->update_list->ins_mode;
	    if (tid->cursor_on)
	       normal_view_typein(tid,from,tid->update_list->ch,ins_mode);
	    else
	       duplicate_view_typein(tid,from,tid->update_list->ch,ins_mode);
	    break;

	 case FILE_MSG_DELETE:
	    if (tid->cursor_on)
	       normal_view_delete(tid,from,to);
	    else
	       duplicate_view_delete(tid,from,to);
	    break;

	 case FILE_MSG_INSERT:

	    if (tid->cursor_on)
	       normal_view_insert(tid,from,to);
	    else
	       duplicate_view_insert(tid,from,to);
	    break;
       }
      free(tid->update_list);
    }
   else {	/* multiple command update */
      if (tid->cursor_on) {
	 multiple_command_update(tid);
       }
      else {			 /* updating a duplicate file copy */
/*	 FILEset_position(tid->file,tid->line,tid->ch,&fp);		*/
	 multiple_command_update(tid);
       }
    };

   tid->update_list = tid->update_listend = NULL;

   UNPROTECT;
   ASHbatch_mode(FALSE);

   FILEinq_position(tid->file,FILE_MOVE_REL,0,&fp);
   TEXTcursor(tid,TEXT_CURSOR_SAME,&fp);
}





/************************************************************************/
/*									*/
/*    normal_view_typein - intelligently update the given		*/
/*	    display for typein of given character at given position.	*/
/*									*/
/************************************************************************/

static void
normal_view_typein(tid,from,ins_ch,ins_mode)
   TEXT_ID   tid;
   FILE_POS  *from;
   Character ins_ch;
   Boolean   ins_mode;
{
   Character	  s[2];
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   FILE_POS	  fp1,fp2;
   String	  text;

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);

   if (ins_mode) {
      if (!tid->insert_mode) {
	 TULIPinsertmode(tid->tulip,TRUE);
	 tid->insert_mode = TRUE;
       }
      if (ins_ch == BACKSPACE) {
	 if (from_char == left_char) {
	    TEXTposition(tid,TEXT_FLAG_DELTA_CHARS,-tulipchars/2);	/* if left edge of screen, scroll */
	  }
	 else {
	    move_cursor_to(tid,from);
	    TULIPmove_rel(tid->tulip,0,-1);	/* left */
	    TULIPchar_delete(tid->tulip,1);
	    if ( s[0] = FILEget_char(tid->file,from_line,right_char-1) ) {
	       TULIPmove(tid->tulip,from_line-top_line,tulipchars-1);
	       s[1] = '\0';
	       TULIPwrite(tid->tulip,s);
	     }
	  }
       }
      else if (ins_ch == NEWLINE) {
	 if (left_char == 1) {
	    move_cursor_to(tid,from);
	    TULIPerase_line(tid->tulip);
	    if (from_line != bottom_line - 1) { /* not bottom line of screen */
	       TULIPwrite(tid->tulip,LINE_FEEDs);
	       TULIPline_insert(tid->tulip,1);
	     }
	    else {
	       TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	       TULIPline_delete(tid->tulip,1);
	       TULIPmove(tid->tulip,tuliplines - 1,TULIP_START_CHAR);
	       ++(tid->line);
	     };
	    FILEinq_position(tid->file,FILE_MOVE_REL,0,&fp1);
	    FILEinq_position(tid->file,FILE_MOVE_REL|FILE_MOVE_CHAR,tulipchars,&fp2);
	    text = FILEcopy_text(tid->file,&fp1,&fp2);
	    TULIPwrite(tid->tulip,text);
	    free(text);
	    TULIPwrite(tid->tulip,CARRIAGE_RETURN);
	  }
	 else { /* left_char != 1) */
	    if ((from_line-top_line) < (tuliplines/2)) { /* top half of screen */
	       TEXT_new_display(tid,TULIP_START_LINE,top_line,1,FULL_SCREEN);
	     }
	    else {
	       TEXT_new_display(tid,TULIP_START_LINE,top_line + 1,1,FULL_SCREEN);
	     }
	  }
       }
      else { /* textual character */
	 if (from_char == right_char - 1) {  /* typing at right edge */
	    ASHbell();
	    TEXTposition(tid,TEXT_FLAG_DELTA_CHARS,tulipchars/4);
	  }
	 else {
	    move_cursor_to(tid,from);
	    s[0] = ins_ch;
	    s[1] = '\0';
	    TULIPwrite(tid->tulip,s);
	  }
       }
    }
   else {  /* overstrike mode */
      if (tid->insert_mode) {
	 TULIPinsertmode(tid->tulip,FALSE);
	 tid->insert_mode = FALSE;
       }
      if (ins_ch == BACKSPACE) {
	 if (FILEinq_char(tid->file,from) == left_char) {      /* if left edge of screen, scroll */
	    TEXTposition(tid,TEXT_FLAG_DELTA_CHARS,-tulipchars/2);
	  }
	 else {
	    move_cursor_to(tid,from);
	    TULIPmove_rel(tid->tulip,0,-1);	/* left */
	    TULIPwrite(tid->tulip," ");
	    TULIPmove_rel(tid->tulip,0,-1);	/* left */
	    /*	VT_BSCLEAR;  really should be
				TULIPwrite(tid->tulip,BACK_SPACE); */
	  }
       }
      else if (ins_ch == NEWLINE) {
	 if (from_line != bottom_line - 1) { /* not bottom line of screen */
	    if (left_char == 1) {
	       move_cursor_to(tid,from);
	       TULIPwrite(tid->tulip,LINE_FEEDs);
	     }
	    else {
	       TEXT_new_display(tid,TULIP_START_LINE,top_line,1,FULL_SCREEN);
	     }
	  }
	 else { /* bottom line of screen, so scroll the screen */
	    if (left_char == 1) {
	       move_cursor_to(tid,from);
	       TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	       TULIPline_delete(tid->tulip,1);
	       TULIPmove(tid->tulip,tuliplines - 1,TULIP_START_CHAR);
	       FILEinq_position(tid->file,FILE_MOVE_REL,0,&fp1);
	       FILEinq_position(tid->file,FILE_MOVE_REL|FILE_MOVE_CHAR,tulipchars,&fp2);
	       text = FILEcopy_text(tid->file,&fp1,&fp2);
	       TULIPwrite(tid->tulip,text);
	       free(text);
	       TULIPwrite(tid->tulip,CARRIAGE_RETURN);
	       ++(tid->line);
	     }
	    else {
	       TEXT_new_display(tid,TULIP_START_LINE,top_line + 1,1,FULL_SCREEN);
	     }
	  }
       }
      else { /* regular text character */
	 if (from_char == right_char - 1) { /* typing at right edge of screen */
	    ASHbell();
	    TEXTposition(tid,TEXT_FLAG_DELTA_CHARS,2*tulipchars/3);
	  }
	 else {
	    move_cursor_to(tid,from);
	    s[0] = ins_ch;
	    s[1] = '\0';
	    TULIPwrite(tid->tulip,s);
	  }
       }
    }
};






/************************************************************************/
/*									*/
/*    duplicate_view_typein - intelligently update a duplicate display	*/
/*	    of the currently active file for typein of given character	*/
/*	    at given position.						*/
/*									*/
/************************************************************************/

static void
duplicate_view_typein(tid,from,ins_ch,ins_mode)
   TEXT_ID   tid;
   FILE_POS *from;
   Character ins_ch;
   Boolean   ins_mode;
{
   Character	  s[2];
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   Integer  numchars;

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);

   if (from_line < top_line) {
      if ((ins_mode) && (ins_ch == NEWLINE)) (tid->line)++;
    }
   else if (from_line < bottom_line) {
      move_cursor_to(tid,from);
      /*    FILEset_position(tid->file,tid->line,tid->ch,&fp);	      */
      if (ins_mode) {
	 if (!tid->insert_mode) {
	    TULIPinsertmode(tid->tulip,TRUE);
	    tid->insert_mode = TRUE;
	  }
	 if (ins_ch == NEWLINE) {
	    TULIPmove(tid->tulip,from_line-top_line,0);
	    if (from_char < left_char)
	       TULIPerase_line(tid->tulip);
	    else if (from_char < right_char) {
	       if (numchars = from_char-left_char)
		  TULIPmove_rel(tid->tulip,0,numchars);  /* right */
	       TULIPerase_line(tid->tulip);
	     }
	    if (from_line != bottom_line-1) {
	       TULIPmove(tid->tulip,from_line-top_line+1,0);
	       TULIPline_insert(tid->tulip,1);
	       TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,1);
	     }
	  }
	 else if (ins_ch == BACKSPACE) {
	    if (from_char <= right_char) {
	       if (from_char > left_char)
		  TULIPmove(tid->tulip,from_line-top_line,from_char-left_char-1);
	       else
		  TULIPmove(tid->tulip,from_line-top_line,0);
	       TULIPchar_delete(tid->tulip,1);
	       if ( s[0] = FILEget_char(tid->file,from_line,right_char-1) ) {
		  TULIPmove(tid->tulip,from_line-top_line,tulipchars-1);
		  s[1] = '\0';
		  TULIPwrite(tid->tulip,s);
		}
	     }
	  }
	 else { /* any other char */
	    if (from_char < left_char) {
	       if ( s[0] = FILEget_char(tid->file,from_line,left_char) ) {
		  TULIPmove(tid->tulip,from_line-top_line,0);
		  s[1] = '\0';
		  TULIPwrite(tid->tulip,s);
		}
	     }
	    else if (from_char < right_char) {
	       TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	       s[0] = ins_ch;
	       s[1] = '\0';
	       TULIPwrite(tid->tulip,s);
	     }
	  }
       }
      else { /* overstrike mode */
	 if (tid->insert_mode) {
	    TULIPinsertmode(tid->tulip,FALSE);
	    tid->insert_mode = FALSE;
	  }
	 if (ins_ch == BACKSPACE) {
	    if ((left_char <= from_char-1) && (from_char-1 < right_char)) {
	       TULIPmove(tid->tulip,from_line-top_line,from_char-left_char-1);
	       s[0] = SPACE;
	       s[1] = '\0';
	       TULIPwrite(tid->tulip,s);
	     }
	  }
	 else if (ins_ch != NEWLINE) {
	    if (TEXT_on_screen(tid,from)) {
	       TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	       s[0] = ins_ch;
	       s[1] = '\0';
	       TULIPwrite(tid->tulip,s);
	     }
	  }
       }
    }
}






/************************************************************************/
/*									*/
/*    normal_view_delete - intelligently update the given		*/
/*	    display for a deletion over the given positions.		*/
/*									*/
/************************************************************************/


static void
normal_view_delete(tid,from,to)
   TEXT_ID   tid;
   FILE_POS  *from;
   FILE_POS  *to;
{
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   Integer  to_line,to_char;
   Integer  numlines,numchars;
   FILE_POS	  fp1,fp2;
   String	  text;

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);
   to_line = FILEinq_line(tid->file,to);
   to_char = FILEinq_char(tid->file,to);
   numlines = to_line-from_line;

   if ((top_line <= from_line) && (from_line < bottom_line)) {
	if ((left_char <= from_char) && (from_char < right_char)) {
	   if (from_line == to_line) {
	      if (to_char < right_char) {
		 numchars = to_char - from_char;
		 TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
		 TULIPchar_delete(tid->tulip,numchars);
		 TULIPmove(tid->tulip,from_line-top_line,tulipchars-numchars);
		 TULIPerase_line(tid->tulip);
		 FILEnew_position(tid->file,from_line,right_char-numchars,&fp1);
		 FILEnew_position(tid->file,from_line,right_char,&fp2);
		 text = FILEcopy_text(tid->file,&fp1,&fp2);
	       }
	      else {
		 TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
		 TULIPerase_line(tid->tulip);
		 FILEnew_position(tid->file,from_line,right_char,&fp1);
		 text = FILEcopy_text(tid->file,from,&fp1);
	       }
	      TULIPwrite(tid->tulip,text);
	      free(text);
	    }
	   else {
	      TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	      TULIPerase_line(tid->tulip);
	      FILEnew_position(tid->file,from_line,right_char,&fp1);
	      text = FILEcopy_text(tid->file,from,&fp1);
	      TULIPwrite(tid->tulip,text);
	      free(text);
	      if (to_line < bottom_line) {
		 TULIPmove(tid->tulip,from_line-top_line+1,0);
		 TULIPline_delete(tid->tulip,numlines);
		 TEXT_new_display(tid,tuliplines-numlines,bottom_line-numlines,left_char,FULL_SCREEN);
	       }
	      else if (from_line+1 < bottom_line)
		 TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,FULL_SCREEN);
	    }
	 }
	else if (from_char <= tulipchars - 3)
	   TEXT_new_display(tid,TULIP_START_LINE,top_line,1,FULL_SCREEN);
	else
	   TEXT_new_display(tid,TULIP_START_LINE,top_line,from_char-(tulipchars/3),FULL_SCREEN);
    }
   else TEXTposition(tid,TEXT_FLAG_NEAR_MIDDLE,from);
}


/************************************************************************/
/*									*/
/*    duplicate_view_delete - intelligently update a duplicate display	*/
/*	    of the currently active file for a deletion over the given	*/
/*	    positions.							*/
/*									*/
/************************************************************************/


static void
duplicate_view_delete(tid,from,to)
   TEXT_ID   tid;
   FILE_POS  *from;
   FILE_POS  *to;
{
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   Integer  to_line,to_char;
   Integer  numlines,numchars;
   Integer  deltalines;
   FILE_POS	  fp1,fp2;
   String	  text;

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);
   to_line = FILEinq_line(tid->file,to);
   to_char = FILEinq_char(tid->file,to);
   numlines = to_line-from_line;

   if (from_line < top_line) {
      if (to_line < top_line)
	 tid->line -= (to_line - from_line);
      else {
	 TEXT_new_display(tid,TULIP_START_LINE,from_line,left_char,FULL_SCREEN);
	 if ((deltalines = FILEinq_line(tid->file,FILE_POS_END) - from_line - tuliplines + 1) < 0)
	    TEXTposition(tid,TEXT_FLAG_DELTA_LINES,deltalines);
       }
    }
   else if (from_line < bottom_line) {
      if (from_line == to_line) {
	 if (from_char < left_char) {
	    numchars = to_char - from_char;
	    TULIPmove(tid->tulip,from_line-top_line,0);
	    TULIPchar_delete(tid->tulip,numchars);
	    TULIPmove(tid->tulip,from_line-top_line,tulipchars-numchars);
	    TULIPerase_line(tid->tulip);
	    FILEnew_position(tid->file,from_line,right_char-numchars,&fp1);
	    FILEnew_position(tid->file,from_line,right_char,&fp2);
	    text = FILEcopy_text(tid->file,&fp1,&fp2);
	  }
	 else if (from_char < right_char) {
	    if (to_char < right_char) {
	       numchars = to_char - from_char;
	       TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	       TULIPchar_delete(tid->tulip,numchars);
	       TULIPmove(tid->tulip,from_line-top_line,tulipchars-numchars);
	       TULIPerase_line(tid->tulip);
	       FILEnew_position(tid->file,from_line,right_char-numchars,&fp1);
	       FILEnew_position(tid->file,from_line,right_char,&fp2);
	       text = FILEcopy_text(tid->file,&fp1,&fp2);
	     }
	    else {
	       TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	       TULIPerase_line(tid->tulip);
	       FILEnew_position(tid->file,from_line,right_char,&fp1);
	       text = FILEcopy_text(tid->file,from,&fp1);
	     }
	  }
	 TULIPwrite(tid->tulip,text);
	 free(text);
       }
      else {
	 if (from_char < right_char) {
	    TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	    TULIPerase_line(tid->tulip);
	    FILEnew_position(tid->file,from_line,right_char,&fp1);
	    text = FILEcopy_text(tid->file,from,&fp1);
	    TULIPwrite(tid->tulip,text);
	    free(text);
	  };
	 if (to_line < bottom_line) {
	    TULIPmove(tid->tulip,from_line-top_line+1,0);
	    TULIPline_delete(tid->tulip,numlines);
	    TEXT_new_display(tid,tuliplines-numlines,bottom_line-numlines,left_char,FULL_SCREEN);
	  }
	 else if (from_line+1 < bottom_line)
	    TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,FULL_SCREEN);
	 if ((deltalines = FILEinq_line(tid->file,FILE_POS_END) - bottom_line + 1) < 0)
	    TEXTposition(tid,TEXT_FLAG_DELTA_LINES,deltalines);
       }
    }
}


/************************************************************************/
/*									*/
/*    normal_view_insert - intelligently update the given		*/
/*	    display for an insertion over the given positions.		*/
/*									*/
/************************************************************************/


static void
normal_view_insert(tid,from,to)
   TEXT_ID   tid;
   FILE_POS  *from;
   FILE_POS  *to;
{
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   Integer  to_line;
   Integer  numlines;
   FILE_POS	  fp;
   String	  text;

   if (!tid->insert_mode) {
      TULIPinsertmode(tid->tulip,TRUE);
      tid->insert_mode = TRUE;
    }

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);
   FILEnew_position(tid->file,from_line,from_char,from);
   to_line = FILEinq_line(tid->file,to);
   numlines = to_line-from_line;

   if (TEXT_on_screen(tid,from)) {
      if (from_line == to_line) {
	 TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	 if (FILEinq_char(tid->file,to) < right_char) {
	    text = FILEcopy_text(tid->file,from,to);
	    TULIPwrite(tid->tulip,text);
	    free(text);
	  }
	 else {
	    FILEnew_position(tid->file,from_line,right_char,&fp);
	    text = FILEcopy_text(tid->file,from,&fp);
	    TULIPwrite(tid->tulip,text);
	    free(text);
	  }
       }
      else {
	 TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	 TULIPerase_line(tid->tulip);
	 FILEnew_position(tid->file,from_line,right_char,&fp);
	 text = FILEcopy_text(tid->file,from,&fp);
	 TULIPwrite(tid->tulip,text);
	 free(text);
	 if (to_line < bottom_line) {
	    TULIPmove(tid->tulip,from_line-top_line+1,0);
	    TULIPline_insert(tid->tulip,numlines);
	    TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,numlines);
	  }
	 else if (from_line+1 < bottom_line)
	    TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,FULL_SCREEN);
       }
    }
/* else TEXTposition(tid,TEXT_FLAG_NEAR_MIDDLE,from);	*/
}





/************************************************************************/
/*									*/
/*    duplicate_view_insert - intelligently update a duplicate display	*/
/*	    of the currently active file for an insertion over the	*/
/*	    given positions.						*/
/*									*/
/************************************************************************/


static void
duplicate_view_insert(tid,from,to)
   TEXT_ID   tid;
   FILE_POS  *from;
   FILE_POS  *to;
{
   Integer tuliplines,tulipchars;
   Integer  top_line,bottom_line;
   Integer  left_char,right_char;
   Integer  from_line,from_char;
   Integer  to_line,to_char;
   Integer  numlines,numchars;
   FILE_POS	  fp1,fp2;
   String	  text;

   if (!tid->insert_mode) {
      TULIPinsertmode(tid->tulip,TRUE);
      tid->insert_mode = TRUE;
    }

   tuliplines = tid->numlines;
   tulipchars = tid->numchars;
   top_line = tid->line;
   left_char = tid->ch;
   bottom_line = top_line + tuliplines;
   right_char = left_char + tulipchars;
   from_line = FILEinq_line(tid->file,from);
   from_char = FILEinq_char(tid->file,from);
   to_line = FILEinq_line(tid->file,to);
   to_char = FILEinq_char(tid->file,to);
   numlines = to_line-from_line;

   if (from_line < top_line) tid->line += (to_line - from_line);
   else if (from_line < bottom_line) {
      if (from_line == to_line) {
	 if (from_char < left_char) {
	    numchars = to_char - from_char;
	    TULIPmove(tid->tulip,from_line-top_line,0);
	    FILEnew_position(tid->file,from_line,left_char,&fp1);
	    FILEnew_position(tid->file,from_line,left_char+numchars,&fp2);
	    text = FILEcopy_text(tid->file,&fp1,&fp2);
	  }
	 else if (from_char < right_char) {
	    TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	    if (to_char < right_char)
	       text = FILEcopy_text(tid->file,from,to);
	    else {
	       FILEnew_position(tid->file,from_line,right_char,&fp1);
	       text = FILEcopy_text(tid->file,from,&fp1);
	     }
	  }
	 else return;

	 TULIPwrite(tid->tulip,text);
	 free(text);
       }
      else {
	 if (from_char < right_char) {
	    TULIPmove(tid->tulip,from_line-top_line,from_char-left_char);
	    TULIPerase_line(tid->tulip);
	    FILEnew_position(tid->file,from_line,right_char,&fp1);
	    text = FILEcopy_text(tid->file,from,&fp1);
	    TULIPwrite(tid->tulip,text);
	    free(text);
	  };
	 if (to_line < bottom_line) {
	    TULIPmove(tid->tulip,from_line-top_line+1,0);
	    TULIPline_insert(tid->tulip,numlines);
	    TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,numlines);
	  }
	 else if (from_line+1 < bottom_line)
	    TEXT_new_display(tid,from_line-top_line+1,from_line+1,left_char,FULL_SCREEN);
       }
    };
}


/************************************************************************/
/*									*/
/*    multiple_command_update - scans through the given display's       */
/*	       update list and calculates how much of the text being	*/
/*	       displayed must be changed (if any).			*/
/*									*/
/************************************************************************/

static void
multiple_command_update(tid)
   TEXT_ID   tid;
{
   Character  ins_char;
   Boolean  ins_mode;
   Integer  top_line,left_char;
   Integer  from_line,to_line;
   Integer  from_tline;
   Integer  changed_tline;     /* the tulip line from which screen must be redisplayed */
   Boolean  display_changed;   /* indicates if the displayed text must be updated	    */
   Boolean  same_line;	    /* indicates if all changes are on the same line	    */
   TEXT_CHANGE	  change,completed;

   top_line = tid->line;
   left_char = tid->ch;
   display_changed = FALSE;
   same_line = TRUE;

   change = tid->update_list;
   while ( change != NULL) {
      from_line = FILEinq_line(tid->file,&(change->from));
      switch(change->msg_type) {
	 case FILE_MSG_TYPEIN :
	    ins_mode = change->ins_mode;
	    ins_char = change->ch;
	    if (from_line < top_line) {
	       if ((ins_char == NEWLINE) && (ins_mode)) top_line++;
	     }
	    else if (from_line < top_line + tid->numlines) {
	       from_tline = from_line - top_line;
	       if (!display_changed) {
		  display_changed = TRUE;
		  changed_tline = from_tline;
		  if ((ins_char == NEWLINE) && (ins_mode)) same_line = FALSE;
		}
	       else {
		  if (((ins_char == NEWLINE) && (ins_mode))
			 || (from_tline != changed_tline))
		     same_line = FALSE;
		  changed_tline = MIN(changed_tline,from_tline);
		}
	     }
	    break;

	 case FILE_MSG_DELETE :
	    to_line = FILEinq_line(tid->file,&(change->to));
	    if (from_line < top_line) {
	       if (to_line < top_line)
		  top_line -= to_line - from_line;
	       else {
		  if (!display_changed) {
		     display_changed = TRUE;
		     if (to_line != top_line) same_line = FALSE;
		   }
		  else if (same_line) {
		     if ((to_line != top_line) || (changed_tline != 0)) same_line = FALSE;
		   }
		  changed_tline = 0;
		  top_line = from_line;
		}
	     }
	    else if (from_line < top_line + tid->numlines) {
	       from_tline = from_line - top_line;
	       if (!display_changed) {
		  display_changed = TRUE;
		  changed_tline = from_tline;
		  if (from_line != to_line) same_line = FALSE;
		}
	       else {
		  if ((from_line != to_line) || (from_tline != changed_tline))
		     same_line = FALSE;
		  changed_tline = MIN(changed_tline,from_tline);
		}
	     }
	    break;

	 case FILE_MSG_INSERT :
	    to_line = FILEinq_line(tid->file,&(change->to));
	    if (from_line < top_line) top_line += to_line - from_line;
	    else if (from_line < top_line + tid->numlines) {
	       from_tline = from_line - top_line;
	       if (!display_changed) {
		  display_changed = TRUE;
		  changed_tline = from_tline;
		  if (from_line != to_line) same_line = FALSE;
		}
	       else {
		  if ((from_line != to_line) || (from_tline != changed_tline))
		     same_line = FALSE;
		  changed_tline = MIN(changed_tline,from_tline);
		}
	     }
	    break;
       } /* end switch */
      completed = change;
      change = change->next;
      free(completed);
    }

   if (display_changed) {
      if (same_line)
	 TEXT_new_display(tid,changed_tline, top_line + changed_tline, left_char, 1);
      else
	 TEXT_new_display(tid,changed_tline, top_line + changed_tline, left_char, FULL_SCREEN);
    };

   tid->line = top_line;
}





/************************************************************************/
/*									*/
/*    move_cursor_to - move the TULIP cursor to the given file position */
/*									*/
/************************************************************************/


static void
move_cursor_to(tid,filepos)
   TEXT_ID tid;
   FILE_POS *filepos;
{
   Integer line_offset,char_offset;

   if (! TEXT_on_screen(tid,filepos) )
      TEXT_move_on_screen(tid,filepos);    /* do an intelligent placement on the screen */

   line_offset = FILEinq_line(tid->file,filepos) - tid->line;
   char_offset = FILEinq_char(tid->file,filepos) - tid->ch;

   /* error check?? */

   TULIPmove(tid->tulip,line_offset,char_offset);
}






/************************************************************************/
/*									*/
/*    highlight_coord - determine proper line and character coordinates */
/*	 of screen to highlight for given file position.		*/
/*									*/
/************************************************************************/


static void
highlight_coord(tid,pos,linep,charp)
   TEXT_ID tid;
   FILE_POS *pos;
   Integer  *linep;
   Integer  *charp;
{
   Integer topline,leftchar;
   Integer pos_line,pos_char;

   topline = tid->line;
   leftchar = tid->ch;
   pos_line = FILEinq_line(tid->file,pos);
   pos_char = FILEinq_char(tid->file,pos);

   if (pos_line < topline) {
      *linep = TULIP_START_LINE;
      *charp = TULIP_START_CHAR;
    }
   else if (pos_line < topline + tid->numlines) {
      *linep = pos_line - topline;
      if (pos_char < leftchar)
	 *charp = TULIP_START_CHAR;
      else if (pos_char < leftchar + tid->numchars)
	 *charp = pos_char - leftchar;
      else {
	 if (pos_line == topline + tid->numlines - 1)
	    *charp = tid->numchars;
	 else {
	    (*linep)++;
	    *charp = TULIP_START_CHAR;
	  }
       }
    }
   else {
      *linep = tid->numlines - 1;
      *charp = tid->numchars;
    }
}





/************************************************************************/
/*									*/
/*	update_highlight -- redraw highlighting 			*/
/*									*/
/************************************************************************/


static void
update_highlight(tid)
   TEXT_ID tid;
{
   if (tid->marked) {
      TEXT_highlight(tid,&tid->mstartpos,&tid->muntilpos,HIGHLIGHT_ON);
    };

   if (tid->marked2) {
      TEXT_highlight(tid,&tid->m2startpos,&tid->m2untilpos,SECONDARY_ON);
    };
};





/* end of textlocal.c */
