/************************************************************************/
/*									*/
/*		text.c							*/
/*									*/
/*	External entries for editor text manipulation			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"






/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


#define  TEXT_INIT  if (!Text_Initialized) TEXT_init()

static	Boolean 	Text_Initialized = FALSE;

	TEXT_ID 	TEXT__list = NULL;	/* list of active TEXT_IDs		*/





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


#ifdef LIB
static	void		text_cleanup();
#endif

static	void		mark_until();





/************************************************************************/
/*									*/
/*	TEXT_init - initializes text output package for editor		*/
/*	text_cleanup -- cleanup handler 				*/
/*									*/
/************************************************************************/


void
TEXT_init()
{
   PROT_INIT;

   PROTECT;
#ifdef LIB
   BROWNregister_cleanup(text_cleanup);
#endif
   TEXT__list = NULL;
   Text_Initialized = TRUE;
   UNPROTECT;
}




#ifdef LIB

static void
text_cleanup()
{
   Text_Initialized = FALSE;
   TEXT__list = NULL;
}

#endif



/************************************************************************/
/*									*/
/*    TEXTopen - open a simple text display in a tulip window.		*/
/*	 Recieves a FILE_ID and an ASH_WINDOW and returns a TEXT_ID.	*/
/*									*/
/************************************************************************/


TEXT_ID
TEXTopen(file,window)
   FILE_ID    file;
   ASH_WINDOW window;
{
   TEXT_ID newtext;

   TEXT_INIT;

   if (file == FILE_ID_NONE)
      EDabort("Attempt to open a text display without a file");

   if (!ASHlock(window)) return NULL;

   newtext = PALLOC(TEXT_ID_INFO);
   newtext->file = file;
   newtext->window = window;
   newtext->tulip = TULIPopen(window);
   TULIPinq_size(newtext->tulip,&(newtext->numlines),&(newtext->numchars));
   TULIPinq_max_size(newtext->tulip,&(newtext->numlines),&(newtext->numchars));
   newtext->line = 0;
   newtext->ch = 0;
   newtext->up = FALSE;
   newtext->marked = FALSE;
   newtext->marked2 = FALSE;
   newtext->update_list = NULL;
   newtext->update_listend = NULL;
   newtext->insert_mode = TRUE;
   newtext->cursor_on = TRUE;
   newtext->cursor = TEXT_CURSOR_CHAR;
   newtext->cursor_force = CURSOR_FORCE_VIEW;
   newtext->basefont = TULIPinq_font(newtext->tulip);

   PROTECT;
   newtext->next = TEXT__list;
   TEXT__list = newtext;
   UNPROTECT;

   EDassoc(file,window);

   FILEregister_message(file,TEXT_message_handler);

   TEXTposition(newtext,TEXT_FLAG_TOP_LEFT,FILE_POS_START);
   TEXTcursor(newtext,TEXT_CURSOR_CHAR,FILE_POS_START);
   TEXTcursor_enable(newtext,1);
   TULIPinsertmode(newtext->tulip,TRUE);      /* doing the initial screen display */

   ASHunlock(window);

   return(newtext);
}




/************************************************************************/
/*									*/
/*    TEXTposition - insure that the designated file position will be	*/
/*	  in the proper place on the screen.				*/
/*									*/
/************************************************************************/


void
TEXTposition(tid,flag,location)
   TEXT_ID tid;
   TEXT_FLAG flag;
   Integer location;
{
   FILE_POS fp; 		     /* cast into a file pos			       */
   Integer numlines,numchars;	/* number of lines and charcters on the screen	  */
   Integer offset;	       /* lines to change on a DELTA_LINES call 	 */
   Integer line,ch;			/* line and char position of given file pos	  */
   Integer topline;			/* line number (file) of top line on screen	  */
   Integer mtiddle;			 /* line to put in the mtiddle of the screen	   */
   Integer leftchar;			/* character number (file) in left col of screen  */

   TEXT_INIT;

   numlines = tid->numlines;
   numchars = tid->numchars;

   switch (flag) {
      case  TEXT_FLAG_TOP_LEFT	 :
	 FILEpos_copy(tid->file,(FILE_POS *)location,&fp);
	 leftchar = FILEinq_char(tid->file,&fp);
	 topline = FILEinq_line(tid->file,&fp);
	 if (topline < 1) topline = 1;
	 TEXT_new_display(tid,TULIP_START_LINE,topline,leftchar,FULL_SCREEN);
	 break;

      case  TEXT_FLAG_NEAR_TOP	 :
	 FILEpos_copy(tid->file,(FILE_POS *)location,&fp);
	 line = FILEinq_line(tid->file,&fp);
	 ch = FILEinq_char(tid->file,&fp);
	 if (numlines <= 2) topline = line;
	 else if (numlines <= 4) topline = line - 1;
	 else topline = line - 2;
	 if (topline < 1) topline = 1;
	 if (ch < (numchars - 2))    /* near right edge of screen */
	    TEXT_new_display( tid,TULIP_START_LINE,topline,1,FULL_SCREEN );
	 else
	    TEXT_new_display( tid,TULIP_START_LINE,topline,ch - (numchars/3),FULL_SCREEN );
	 break;

      case  TEXT_FLAG_NEAR_BOTTOM   :
	 FILEpos_copy(tid->file,(FILE_POS *)location,&fp);
	 line = FILEinq_line(tid->file,&fp);
	 ch = FILEinq_char(tid->file,&fp);
	 if (numlines <= 2) topline = line;
	 else if (numlines == 3) topline = line - 1;
	 else if (numlines == 4) topline = line - 2;
	 else topline = line - numlines + 3;
	 if (topline < 1) topline = 1;
	 if (ch < (numchars - 3))  /* near right edge of screen */
	    TEXT_new_display( tid,TULIP_START_LINE,topline,1,FULL_SCREEN );
	 else
	    TEXT_new_display( tid,TULIP_START_LINE,topline,ch,FULL_SCREEN );
	 break;

      case  TEXT_FLAG_NEAR_MIDDLE:
	 FILEpos_copy(tid->file,(FILE_POS *)location,&fp);
	 line = FILEinq_line(tid->file,&fp);
	 ch = FILEinq_char(tid->file,&fp);
	 mtiddle = (numlines - TULIP_START_LINE) / 2;
	 topline = line - mtiddle;
	 if (topline < 1) topline = 1;
	 if (ch < (numchars - 2))
	    TEXT_new_display( tid,TULIP_START_LINE,topline,1,FULL_SCREEN );
	 else
	    TEXT_new_display( tid,TULIP_START_LINE,topline,ch - (3*numchars/4),FULL_SCREEN );
	 break;

      case  TEXT_FLAG_DELTA_LINES:
	 if (location == 0) break;
	 topline = tid->line + location;
	 if (topline < 1) topline = 1;
	 offset = topline - tid->line;
	 if ((offset > 0) && (offset < numlines-1)) { /* positive delta less than a screen so do smart update */
	    TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	    TULIPline_delete(tid->tulip,offset);
	    tid->line += offset;
	    TEXT_new_display(tid,numlines-offset,tid->line + numlines - offset,tid->ch,FULL_SCREEN);
	  }
	 else if ((offset < 0) && ( (-offset) < numlines-1)) {
	    TULIPmove(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR);
	    TULIPline_insert(tid->tulip,-offset); /* negative delta less than a screen so do smart update */
	    tid->line += offset;
	    TEXT_new_display(tid,TULIP_START_LINE,tid->line,tid->ch,-offset);
	  }
	 else if (offset == 0) /* do nothing */;
	 else					    /* may as well redisplay whole screen */
	    TEXT_new_display(tid,TULIP_START_LINE,topline,tid->ch,FULL_SCREEN);
	 break;

      case  TEXT_FLAG_DELTA_CHARS:
	 if (location == 0) break;
	 if ((leftchar = (tid->ch + location)) < 1) leftchar = 1;
	 TEXT_new_display(tid,TULIP_START_LINE, tid->line,leftchar,FULL_SCREEN);
	 break;
    }
}





/************************************************************************/
/*									*/
/*    TEXTinq_top - return the FILE_POS of the top left corner of the	*/
/*	  text display. 						*/
/*									*/
/************************************************************************/


void
TEXTinq_top(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   TEXT_INIT;

   FILEnew_position(tid->file,tid->line,tid->ch,fp);
}


/************************************************************************/
/*									*/
/*    TEXTinq_topline - return the line number (in the file) of the	*/
/*	  character at the top left corner of the given display.	*/
/*									*/
/************************************************************************/


int
TEXTinq_topline(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   return(tid->line);
}



/************************************************************************/
/*									*/
/*    TEXTinq_leftchar - return the character number (relative to its	*/
/*	  line) of the char at the top left of the given window.	*/
/*									*/
/************************************************************************/


int
TEXTinq_leftchar(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   return(tid->ch);
}


/************************************************************************/
/*									*/
/*    TEXTinq_bottom - return the FILE_POS of the bottom right corner	*/
/*	  of the text display.						*/
/*									*/
/************************************************************************/


void
TEXTinq_bottom(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   TEXT_INIT;

   FILEnew_position(tid->file,
		       tid->line + tid->numlines - 1,
		       tid->ch + tid->numchars - 1,
		       fp);
}





/************************************************************************/
/*									*/
/*    TEXTinq_bottomline -  return the line number (in the file) of the */
/*	  character at the bottom right corner of the given display.	*/
/*									*/
/************************************************************************/


int
TEXTinq_bottomline(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   return(tid->line + tid->numlines - 1);
}





/************************************************************************/
/*									*/
/*    TEXTinq_rightchar - return the character number (relative to its	*/
/*	  line) of the char at the bottom right of the given window.	*/
/*									*/
/************************************************************************/


int
TEXTinq_rightchar(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   return(tid->ch + tid->numchars - 1);
}





/************************************************************************/
/*									*/
/*    TEXTcorrelate - return the FILE_POS of the given x,y position	*/
/*	  on the window.						*/
/*									*/
/************************************************************************/


void
TEXTcorrelate(tid,x,y,leftfg,fp)
   TEXT_ID tid;
   Integer x,y;
   Boolean leftfg;
   FILE_POS *fp;
{
   Integer tulipline,tulipchar;

   TEXT_INIT;

   TULIPcorrelate(tid->tulip,x,y,&tulipline,&tulipchar); /* returns the tulip line and char of x,y */

   if (leftfg && tulipchar < 0) tulipchar = 0;

   if (tulipchar >= 0) {
      FILEnew_position(tid->file,tid->line + tulipline,tid->ch + tulipchar,fp);
    }
   else if (tulipline == 0 && tid->line == 1) {
      FILEinq_position(tid->file,FILE_MOVE_START,0,fp);
    }
   else {
      FILEinq_position(tid->file,FILE_MOVE_START|
			  FILE_MOVE_LINE_END|FILE_MOVE_LINE,
			  tid->line+tulipline-1-1,fp);
    };
}





/************************************************************************/
/*									*/
/*    TEXTmark_begin - start the marking process (highlighting) at the	*/
/*	  given file position.						*/
/*    TEXTmark2_begin -- begin secondary mark				*/
/*									*/
/************************************************************************/


void
TEXTmark_begin(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   TEXT_INIT;

   tid->marked = TRUE;
   FILEpos_copy(tid->file,fp,&(tid->mstartpos));
   FILEpos_copy(tid->file,fp,&(tid->muntilpos));
}





void
TEXTmark2_begin(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   TEXT_INIT;

   tid->marked2 = TRUE;
   FILEpos_copy(tid->file,fp,&(tid->m2startpos));
   FILEpos_copy(tid->file,fp,&(tid->m2untilpos));
}





/************************************************************************/
/*									*/
/*    TEXTinq_mark_begin - return the file position representing the	*
/*	  start of the marking process (highlighting).			*/
/*									*/
/************************************************************************/


void
TEXTinq_mark_begin(tid,fp)
   TEXT_ID tid;
   FILE_POS *fp;
{
   TEXT_INIT;

   if (tid->marked)
      FILEpos_copy(tid->file,&(tid->mstartpos),fp);
   else
      FILEpos_copy(tid->file,FILE_POS_NONE,fp);
}






/************************************************************************/
/*									*/
/*    TEXTmark_until - continue marking just updating the difference	*/
/*	  from the last mark_until call.				*/
/*    TEXTmark2_until -- continue marking for secondary mark		*/
/*    TEXTmark_from -- replace start of mark				*/
/*    TEXTmark2_from -- replace start of secondary mark 		*/
/*									*/
/************************************************************************/


void
TEXTmark_until(tid,filepos)
   TEXT_ID tid;
   FILE_POS *filepos;
{
   TEXT_INIT;

   if (!tid->marked) {
      EDabort("Attempting to mark without any start position");
    };

   mark_until(tid,filepos,&(tid->mstartpos),&(tid->muntilpos),HILITE_1);

   FILEpos_copy(tid->file,filepos,&(tid->muntilpos));
}





void
TEXTmark_from(tid,filepos)
   TEXT_ID tid;
   FILE_POS *filepos;
{
   TEXT_INIT;

   if (!tid->marked) {
      EDabort("Attempting to mark without any start position");
    };

   mark_until(tid,filepos,&(tid->muntilpos),&(tid->mstartpos),HILITE_1);

   FILEpos_copy(tid->file,filepos,&(tid->mstartpos));
}





void
TEXTmark2_until(tid,filepos)
   TEXT_ID tid;
   FILE_POS *filepos;
{
   TEXT_INIT;

   if (!tid->marked2) {
      EDabort("Attempting to secondary mark without any start position");
    };

   mark_until(tid,filepos,&(tid->m2startpos),&(tid->m2untilpos),HILITE_2);

   FILEpos_copy(tid->file,filepos,&(tid->m2untilpos));
}





void
TEXTmark2_from(tid,filepos)
   TEXT_ID tid;
   FILE_POS *filepos;
{
   TEXT_INIT;

   if (!tid->marked2) {
      EDabort("Attempting to secondary mark without any start position");
    };

   mark_until(tid,filepos,&(tid->m2untilpos),&(tid->m2startpos),HILITE_2);

   FILEpos_copy(tid->file,filepos,&(tid->m2startpos));
}





static void
mark_until(tid,filepos,startpos,untilpos,mode)
   TEXT_ID tid;
   FILE_POS *filepos;
   FILE_POS *startpos;
   FILE_POS *untilpos;
{
   Integer on,off;

   if (mode == HILITE_1) {
      on = HIGHLIGHT_ON;
      off = HIGHLIGHT_OFF;
    }
   else {
      on = SECONDARY_ON;
      off = SECONDARY_OFF;
    };

   if (FILEpos_compare(tid->file,startpos,untilpos) <= 0) {
      if (FILEpos_compare(tid->file,filepos,startpos) <= 0) {
	 TEXT_highlight(tid,startpos,untilpos,off);
	 TEXT_highlight(tid,filepos,startpos,on);
       }
      else if (FILEpos_compare(tid->file,filepos,untilpos) <= 0) {
	 TEXT_highlight(tid,startpos,filepos,on); /* ensure old area still marked   */
	 TEXT_highlight(tid,filepos,untilpos,off);
       }
      else TEXT_highlight(tid,startpos,filepos,on);  /* i.e. highlight untilpos to filepos */
    }
   else if (FILEpos_compare(tid->file,filepos,untilpos) <= 0)
      TEXT_highlight(tid,filepos,startpos,on); /* i.e. highlight filepos to untilpos */
   else if (FILEpos_compare(tid->file,filepos,startpos) <= 0) {
      TEXT_highlight(tid,untilpos,filepos,off);
      TEXT_highlight(tid,filepos,startpos,on); /* ensure old area still marked	 */
    }
   else {
      TEXT_highlight(tid,untilpos,startpos,off);
      TEXT_highlight(tid,startpos,filepos,on);
    }
};





/************************************************************************/
/*									*/
/*    TEXTmark_end - remove the currently marked region.		*/
/*    TEXTmark2_end -- remove secondary marked region			*/
/*									*/
/************************************************************************/


void
TEXTmark_end(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   TULIPhighlight(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR,
		     tid->numlines,tid->numchars,tid->basefont,FALSE,HILITE_1);
   tid->marked = FALSE;
}




void
TEXTmark2_end(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   TULIPhighlight(tid->tulip,TULIP_START_LINE,TULIP_START_CHAR,
		     tid->numlines,tid->numchars,tid->basefont,FALSE,HILITE_2);
   tid->marked2 = FALSE;
}



/************************************************************************/
/*									*/
/*    TEXTcursor_enable - turn the TULIPcursor on (!0) or off (0).	*/
/*									*/
/************************************************************************/


void
TEXTcursor_enable(tid,flag)
   TEXT_ID tid;
   Integer flag;
{
   TEXT_INIT;

   tid->cursor_force = flag;

   if (flag == CURSOR_FORCE_OFF) {
      TULIPcursor(tid->tulip,FALSE);
      tid->cursor_on = FALSE;
    };
}



/************************************************************************/
/*									*/
/*    TEXTcursor - move the cursor to the given file position and make	*/
/*	  the cursor into the given cursor type.			*/
/*									*/
/************************************************************************/


void
TEXTcursor(tid,type,filepos)
   TEXT_ID tid;
   TEXT_CURSOR type;
   FILE_POS *filepos;
{
   Integer line_offset,char_offset;
   FILE_POS npos;

   TEXT_INIT;

   if (FILEpos_compare(tid->file,FILE_POS_NONE,filepos) == 0) return;

   if (! TEXT_on_screen(tid,filepos) ) {
      if (tid->cursor_force == CURSOR_FORCE_OFF && type == TEXT_CURSOR_SAME) {
	 if (tid->cursor_on) {
	    TULIPcursor(tid->tulip,FALSE);
	    tid->cursor_on = FALSE;
	  };
	 return;
       };
      if (tid->cursor_force == CURSOR_FORCE_FILE && type == TEXT_CURSOR_SAME) {
	 FILEset_position(tid->file,tid->line,tid->ch,&npos);
	 filepos = &npos;
       }
      else TEXT_move_on_screen(tid,filepos);	/* do an intelligent placement on the screen */
    };

   if (type == TEXT_CURSOR_SAME || type == TEXT_CURSOR_FORCE) /* do nothing */;
   else if (type == TEXT_CURSOR_CHAR) {
      tid->cursor = TEXT_CURSOR_CHAR;
      TULIPwrite(tid->tulip,"\033[>4l");    /* makes a block cursor */
    }
   else if (type == TEXT_CURSOR_UNDERLINE) {
      tid->cursor = TEXT_CURSOR_UNDERLINE;   /* makes an underline cursor */
      TULIPwrite(tid->tulip,"\033[>4h");
    }
   else if (type == TEXT_CURSOR_CARET) {
      tid->cursor = TEXT_CURSOR_CARET;	     /* makes a caret cursor */
      TULIPwrite(tid->tulip,"\033[>12h");
    }

   line_offset = FILEinq_line(tid->file,filepos) - tid->line;
   char_offset = FILEinq_char(tid->file,filepos) - tid->ch;

   /* error check */

   TULIPmove(tid->tulip,line_offset, char_offset);

   if (!tid->cursor_on) {
      TULIPcursor(tid->tulip,TRUE);
      tid->cursor_on = TRUE;
    };
}





/************************************************************************/
/*									*/
/*    TEXTclose - close the given text display. 			*/
/*	TEXT_close_local -- close text window w/o freeing		*/
/*									*/
/************************************************************************/


void
TEXTclose(tid)
   TEXT_ID tid;
{
   TEXT_INIT;

   TEXT_close_local(tid);

   free(tid);
}




void
TEXT_close_local(tid)
   TEXT_ID tid;
{
   TEXT_ID  textp;

   if (tid->tulip != NULL) {
      TULIPclose(tid->tulip);
      tid->tulip = NULL;
      EDassoc(tid->file,NULL);
    };

   PROTECT;
   if (tid == TEXT__list) {    /* remove tid from TEXT__list */
      TEXT__list = TEXT__list->next;
    }					   /* free up the space */
   else {
      for (textp = TEXT__list; textp != NULL; textp = textp->next) {
	 if (textp->next == tid) break;
       }
      if (textp != NULL) {
	 textp->next = textp->next->next;
       };
    }
   UNPROTECT;
}




/************************************************************************/
/*									*/
/*    TEXTinq_screensize - return the width (in chars) and height (in	*/
/*	lines) of the window for the given TEXT_ID.			*/
/*									*/
/************************************************************************/


void
TEXTinq_screensize(tid,width,height)
   TEXT_ID  tid;
   Integer     *width;
   Integer     *height;
{
   TEXT_INIT;

   *width  = tid->numchars;
   *height = tid->numlines;
}





/************************************************************************/
/*									*/
/*    TEXTresize - handles resize messages of the given display.	*/
/*									*/
/************************************************************************/


void
TEXTresize(tid)
   TEXT_ID tid;
{
   FILE_POS fp;

   TEXT_INIT;

   TULIPinq_size(tid->tulip,&(tid->numlines),&(tid->numchars)); /* get the new dimensions */
   TULIPinq_max_size(tid->tulip,&(tid->numlines),&(tid->numchars)); /* get the new dimensions */

   if (tid->line + tid->numlines - 1 > FILEinq_line(tid->file,FILE_POS_END))
      tid->line = FILEinq_line(tid->file,FILE_POS_END) - tid->numlines + 1;
   if (tid->line < 1)	/* don't want redisplay */
      tid->line = 1;	 /* past end of file */
   TEXT_new_display(tid,TULIP_START_LINE,tid->line,tid->ch,FULL_SCREEN);

   FILEinq_position(tid->file,FILE_MOVE_REL,0,&fp);
   if ( !(TEXT_on_screen(tid,&fp)) )
      FILEset_position(tid->file,tid->line,tid->ch,&fp);

   tid->cursor_on = FALSE;
   TEXTcursor(tid,tid->cursor,&fp);
   TULIPinsertmode(tid->tulip,tid->insert_mode);
}





/************************************************************************/
/*									*/
/*	TEXTset_shell_mode -- mark editor as shell-type 		*/
/*									*/
/************************************************************************/


void
TEXTset_shell_mode(tid)
   TEXT_ID tid;
{
   tid->shell_mode = TRUE;
};






/* end of text.c */
