/************************************************************************/
/*									*/
/*		filesearch.c						*/
/*									*/
/*	Search routines for BWE base editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "editor_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define INIT	    char *sp = instring 	/* set up info for regex */
#define GETC()	    (*sp++)
#define PEEKC()     (*sp)
#define UNGETC(c)   (--sp)
#define RETURN(c)   return;
#define ERROR(c)    { regerr(c,fid); return;}


#define CBRA	2
#define CCHR	4
#define CDOT	8
#define CCL	12
#define CDOL	20
#define CCEOF	22
#define CKET	24
#define CBACK	36

#define STAR	01
#define RNGE	03

#define NBRA	9

#define PLACE(c)	ep[c >> 3] |= bittab[c & 07]
#define ISTHERE(c)	(ep[c >> 3] & bittab[c & 07])




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/

static	char	*braslist[NBRA];
static	char	*braelist[NBRA];
static	int	nbra, ebra;
static	char	*loc1, *loc2, *locs;   /* loc1 is placed at start of matching text */
static	int	sed;		       /* on a regular expression match 	   */
static	int	nodelim;

static	int	circf;
static	int	low;
static	int	size;

static	Boolean reg_compile_error;     /* true if an expression compilation error  */

static char    bittab[] = {
   1,
   2,
   4,
   8,
   16,
   32,
   64,
   128
};




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/

static	FILE_INDEX   regex_search_forward();
static	FILE_INDEX   regex_search_backward();
static	FILE_INDEX   normal_search_forward();
static	FILE_INDEX   normal_search_backward();
static	FILE_INDEX   nocase_search_forward();
static	FILE_INDEX   nocase_search_backward();
static	Boolean      normal_forward_match();
static	Boolean      normal_backward_match();
static	Boolean      nocase_forward_match();
static	Boolean      nocase_backward_match();
static	Boolean      regex_backward_match();
static	String	     compile();
static	int	     step();
static	int	     advance();
static	void	     getrnge();
static	int	     ecmp();
static	void	     regerr();




/******************************************************************/
/*								  */
/*    FILEsearch - search for the string "text" with the given    */
/*	  search parameter options.				  */
/*								  */
/******************************************************************/


void
FILEsearch(fid,text,style,fp)
   FILE_ID fid;
   String text;
   Integer style;
   FILE_POS *fp;
{
   FILE_POS epos;
   Boolean fg;

   fg = FILEsearch_position(fid,text,style,fp,&epos);

   if (!fg) {
      CopyPos(fid,FILE_POS_NONE,fp);
    }
}





/******************************************************************/
/*								  */
/*    FILEsearch_position -- return start/end of re match	  */
/*								  */
/******************************************************************/


int
FILEsearch_position(fid,text,style,sp,ep)
   FILE_ID fid;
   String text;
   Integer style;
   FILE_POS *sp;
   FILE_POS *ep;
{
   FILE_INDEX result;	   /* found position	     */
   Character expbuf[200];	   /* holds compiled reg exp */

   FILE_INIT;

   if (text == NULL) return FALSE;

   PROTECT;

   if (style & FILE_SEARCH_RE) {
      reg_compile_error = FALSE;
      compile(text,expbuf,&expbuf[200],0,fid);	/* compile the reg exp */
      if (reg_compile_error) {
	 UNPROTECT;
	 return FALSE;
       };

      FILE_movegapto(fid,fid->FILE_INDEX_END);
      *(fid->TEXTPTR + fid->GAPSTART) = '\0';        /* pad with a null byte */

      if (style & FILE_SEARCH_BACKWARD)
	 result = regex_search_backward(fid,expbuf,style);
      else
	 result = regex_search_forward(fid,expbuf,style);
    }
   else if (style & FILE_SEARCH_NOCASE) {
      FILE_movegapto(fid,fid->FILE_INDEX_END);		 /* must always get the gap */
      if (style & FILE_SEARCH_BACKWARD) 		 /* out of the way	    */
	 result = nocase_search_backward(fid,text,style);
      else
	 result = nocase_search_forward(fid,text,style);
    }
   else {
      FILE_movegapto(fid,fid->FILE_INDEX_END);
      if (style & FILE_SEARCH_BACKWARD)
	 result = normal_search_backward(fid,text,style);
      else
	 result = normal_search_forward(fid,text,style);
    };

   UNPROTECT;

   if (result == FILE_INDEX_NONE) return FALSE;

   FILE_convert(fid,result,sp);
   result = (loc2 - fid->TEXTPTR);
   FILE_convert(fid,result,ep);

   return TRUE;
}



/******************************************************************/
/*								  */
/*    FILEsearch_word - search (plus or minus) for the next word. */
/*	  This routine returns the file_pos of that word.	  */
/*								  */
/******************************************************************/


void
FILEsearch_word(fid,style,file_pos)
   FILE_ID   fid;
   Integer   style;
   FILE_POS *file_pos;
{
   FILE_INDEX idx;	/* index into file buffer */

   FILE_INIT;

   if (!style) { /* backwards word search */
      idx = fid->POINT;
      if (idx <= FILE_INDEX_START+1) {
	 CopyPos(fid,FILE_POS_START,file_pos);
	 return;
       }

      --idx;
      if (charAt(fid,idx) == NEWLINE) {
	 FILE_convert(fid,idx,file_pos);
	 return;
       }
      else if (charAt(fid,idx) == SPACE) {
	 do {
	    --idx;
	    if (idx == FILE_INDEX_START) {
	       CopyPos(fid,FILE_POS_START,file_pos);
	       return;
	     }
	  }
	 while (charAt(fid,idx) == SPACE);
	 if (charAt(fid,idx) == NEWLINE) {
	    FILE_convert(fid,idx,file_pos);
	    return;
	  }
       }

      /* now we're in some text */
      for (;;) {
	 if ((charAt(fid,idx-1) < '!') || (charAt(fid,idx-1) > '~')) {
	    FILE_convert(fid,idx,file_pos);
	    return;
	  }
	 --idx;
	 if (idx == FILE_INDEX_START) {
	    CopyPos(fid,FILE_POS_START,file_pos);
	    return;
	  }
       }
    }
   else { /* forward word search */
      idx = fid->POINT;
      if (idx >= fid->FILE_INDEX_END) {
	 CopyPos(fid,FILE_POS_END,file_pos);
	 return;
       }
      if ((charAt(fid,idx) >= '!') && (charAt(fid,idx) <= '~')) { /* text chars */
	 for (;;) {
	    idx++;
	    if (idx >= fid->FILE_INDEX_END) break;
	    if ((charAt(fid,idx) < '!') || (charAt(fid,idx) > '~'))  /* non-text */
	       break;
	  };
       }
      else /* char is newline or space */
	 ++idx;

      if (idx >= fid->FILE_INDEX_END) {
	 CopyPos(fid,FILE_POS_END,file_pos);
	 return;
       }

      for (;;) {
	 if (charAt(fid,idx) != SPACE) break;
	 else {
	    ++idx;
	    if (idx >= fid->FILE_INDEX_END) break;
	  }
       }
      FILE_convert(fid,idx,file_pos);
    }
}





/******************************************************************/
/*								  */
/*    regex_search_forward - perform a regular expression search  */
/*	in the forward direction.				  */
/*								  */
/******************************************************************/


static FILE_INDEX
regex_search_forward(fid,expbuf,style)
   FILE_ID fid;
   String expbuf;
   Integer style;
{
   Character *pos;
   Character *badpos = NULL;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else if (style & FILE_SEARCH_CURROK) pos = fid->TEXTPTR + fid->POINT;
   else {
      pos = fid->TEXTPTR + fid->POINT + 1;
      badpos = fid->TEXTPTR + fid->POINT;
    }

   if (fid->POINT >= fid->FILE_INDEX_END) return FILE_INDEX_NONE;

   if (step(pos,expbuf,fid->TEXTPTR))		/* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 && step(fid->TEXTPTR,expbuf,fid->TEXTPTR) &&
	       loc1 != badpos)
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}





/******************************************************************/
/*								  */
/*    regex_search_backward - perform a regular expression search */
/*	in the backward direction.				  */
/*								  */
/******************************************************************/


static FILE_INDEX
regex_search_backward(fid,expbuf,style)
   FILE_ID fid;
   String expbuf;
   Integer  style;
{
   Character *pos;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else if (style & FILE_SEARCH_CURROK) pos = fid->TEXTPTR + fid->POINT;
   else pos = fid->TEXTPTR + fid->POINT - 1;

   if (regex_backward_match(fid,expbuf,pos,pos + 1))	 /* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 &&
	       regex_backward_match(fid,expbuf,fid->TEXTPTR + fid->FILE_INDEX_END - 1,
				       fid->TEXTPTR + fid->FILE_INDEX_END))
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}





/******************************************************************/
/*								  */
/*    normal_search_forward - search in the forward direction	  */
/*	for the string "text."                                    */
/*								  */
/******************************************************************/


static FILE_INDEX
normal_search_forward(fid,text,style)
   FILE_ID fid;
   String   text;
   Integer  style;
{
   Character *pos;
   Character *badpos = NULL;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else {
      pos = fid->TEXTPTR + fid->POINT + 1;
      badpos = fid->TEXTPTR + fid->POINT;
    }

   if (normal_forward_match(fid,pos,text))    /* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 &&
	       normal_forward_match(fid,fid->TEXTPTR,text) && loc1 != badpos)
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}




/******************************************************************/
/*								  */
/*    normal_search_backward - search in the forward direction	  */
/*	for the string "text."                                    */
/*								  */
/******************************************************************/


static FILE_INDEX
normal_search_backward(fid,text,style)
   FILE_ID  fid;
   String   text;
   Integer  style;
{
   Character *pos;
   Character *badpos = NULL;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else {
      pos = fid->TEXTPTR + fid->POINT - 1;
      badpos = fid->TEXTPTR + fid->POINT;
    }

   if (normal_backward_match(fid,pos,text))   /* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 &&
	       normal_backward_match(fid,fid->TEXTPTR + fid->FILE_INDEX_END - 1,text) &&
	       loc1 != badpos)
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}





/******************************************************************/
/*								  */
/*    nocase_search_forward - search in the forward direction	  */
/*	for the string "text" ignoring case.                      */
/*								  */
/******************************************************************/


static FILE_INDEX
nocase_search_forward(fid,text,style)
   FILE_ID fid;
   String   text;
   Integer  style;
{
   Character *pos;
   Character *badpos = NULL;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else {
      pos = fid->TEXTPTR + fid->POINT + 1;
      badpos = fid->TEXTPTR + fid->POINT;
    }

   if (nocase_forward_match(fid,pos,text))    /* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 &&
	       nocase_forward_match(fid,fid->TEXTPTR,text) && loc1 != badpos)
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}





/******************************************************************/
/*								  */
/*    nocase_search_backward - search in the backward direction   */
/*	for the string "text" ignoring case.                      */
/*								  */
/******************************************************************/


static FILE_INDEX
nocase_search_backward(fid,text,style)
   FILE_ID  fid;
   String   text;
   Integer  style;
{
   Character *pos;
   Character *badpos = NULL;

   if (style & FILE_SEARCH_START) pos = fid->TEXTPTR;
   else if (style & FILE_SEARCH_END) pos = fid->TEXTPTR + fid->FILE_INDEX_END - 1;
   else {
      pos = fid->TEXTPTR + fid->POINT - 1;
      badpos = fid->TEXTPTR + fid->POINT;
    }

   if (nocase_backward_match(fid,pos,text))   /* if found */
      return(loc1 - fid->TEXTPTR);
   else if ((style & FILE_SEARCH_WRAP) != 0 &&
	       nocase_backward_match(fid,fid->TEXTPTR + fid->FILE_INDEX_END - 1,text) &&
	       loc1 != badpos)
      return(loc1 - fid->TEXTPTR);

   return(FILE_INDEX_NONE);
}





/******************************************************************/
/*								  */
/*    normal_forward_match - search in the forward direction	  */
/*	for the string "text." Return true if found.              */
/*								  */
/******************************************************************/


static Boolean
normal_forward_match(fid,startpos,text)
   FILE_ID fid;
   String startpos;
   String text;
{
   Character   *pos = startpos;
   String target = text;
   Character   *ender;
   Integer length;
   Integer i;

   length = strlen(target);
   ender = fid->TEXTPTR + fid->FILE_INDEX_END - length;
   while (pos <= ender) {
      if (*target == *pos) {
	 for (i=1; i<length; ++i)
	    if (*(target + i) != *(pos + i)) break;
	 if (i == length) {
	    loc1 = pos;
	    loc2 = pos+length-1;
	    return(TRUE);
	  }
       }
      ++pos;
    }

   return(FALSE);
}


/******************************************************************/
/*								  */
/*    normal_backward_match - search in the backward direction	  */
/*	for the string "text." Return true if found.              */
/*								  */
/******************************************************************/


static Boolean
normal_backward_match(fid,startpos,text)
   FILE_ID fid;
   String startpos;
   String text;
{
   Character   *pos = startpos;
   String target = text;
   Character   *ender;
   Integer length;
   Integer i;

   length = strlen(target);
   if (pos > (fid->TEXTPTR + fid->FILE_INDEX_END - length))   /* can't run past ending boundary */
      pos = fid->TEXTPTR + fid->FILE_INDEX_END - length;
   ender = fid->TEXTPTR ;

   while (pos >= ender) {
      if (*target == *pos) {
	 for (i=1; i<length; ++i)
	    if (*(target + i) != *(pos + i)) break;
	 if (i == length) {
	    loc1 = pos;
	    loc2 = pos+length-1;
	    return(TRUE);
	  }
       }
      --pos;
    }

   return(FALSE);
}




/******************************************************************/
/*								  */
/*    nocase_forward_match - search in the forward direction	  */
/*	ignoring case for the string "text." Return true if found.*/
/*								  */
/******************************************************************/


static Boolean
nocase_forward_match(fid,startpos,text)
   FILE_ID fid;
   String startpos;
   String text;
{
   Character   *pos = startpos;
   String target = text;
   Character   *ender;
   Integer length;
   Character   x,y;
   Integer i;

   length = strlen(target);
   ender = fid->TEXTPTR + fid->FILE_INDEX_END - length;
   while (pos <= ender) {
      for (i=0; i<length; ++i) {
	 if ((x = *(target + i)) == (y = *(pos + i)))
	    /* do nothing */;
	 else {
	    if (isupper(x)) x = tolower(x);
	    if (isupper(y)) y = tolower(y);
	    if (x != y) break;
	  };
       }
      if (i == length) {
	 loc1 = pos;
	 loc2 = pos+length-1;
	 return(TRUE);
       }
      ++pos;
    }

   return(FALSE);
}





/******************************************************************/
/*								  */
/*    nocase_backward_match - search in the backward direction	  */
/*	ignoring case for the string "text."Return true if found. */
/*	Return true if found.					  */
/*								  */
/******************************************************************/


static Boolean
nocase_backward_match(fid,startpos,text)
   FILE_ID fid;
   String startpos;
   String text;
{
   Character   *pos = startpos;
   String target = text;
   Character   *ender;
   Integer length;
   Integer i;
   Character   x,y;

   length = strlen(target);
   if (pos > (fid->TEXTPTR + fid->FILE_INDEX_END - length))   /* can't run past ending boundary */
      pos = fid->TEXTPTR + fid->FILE_INDEX_END - length;
   ender = fid->TEXTPTR ;

   while (pos >= ender) {
      for (i=0; i<length; ++i) {
	 if ((x = *(target + i)) == (y = *(pos + i)))
	    /* do nothing */;
	 else if (x != (islower(y) ? toupper(y) : tolower(y)))
	    break;
       }
      if (i == length) {
	 loc1 = pos;
	 loc2 = pos+length-1;
	 return(TRUE);
       }
      --pos;
    }

   return(FALSE);
}


/******************************************************************/
/*								  */
/*    regex_backward_match - search in the backward direction	  */
/*	from position pos for the compiled reg.exp. in expbuf.	  */
/*	Return true if found.					  */
/*								  */
/******************************************************************/


static Boolean
regex_backward_match(fid,expbuf,startpos,badpos)
   FILE_ID fid;
   String  expbuf;
   String  startpos;
   String  badpos;
{
   Boolean found = FALSE;
   Character *newspot = startpos;   /* go farther back and try to match from there  */
   Character *qtext;	       /* quick access for TEXTPTR		       */
   Integer value = 500;        /* how far back to go			       */
   Character *best,*best2;	       /* nearest matching position preceding startpos */

   qtext = fid->TEXTPTR;
   while ((!found) && (newspot != qtext)) {
      newspot = newspot - value;
      if (newspot < qtext)	       /* initial hunt from earlier position */
	 newspot = qtext;
      if ((step(newspot,expbuf,fid->TEXTPTR)) && (loc1 < badpos))
	 found = TRUE;
      else
	 value *= 2;
    }

   if (found) {
      best = loc1;	       /* if it was found there could be a closer one */
      best2 = loc2;
      do {
	 found = FALSE;
	 if ((step(best + 1,expbuf,fid->TEXTPTR)) && (loc1 < badpos)) {
	    found = TRUE;
	    best = loc1;
	    best2 = loc2;
	  }
       }
      while (found == TRUE);
      loc1 = best;
      loc2 = best2;
      return(TRUE);
    }

   return(FALSE);
}





/******************************************************************/
/*								  */
/*	compile - compile the regular expression instring and	  */
/*	   place the result in ep.				  */
/*								  */
/******************************************************************/


static String
compile(instring, ep, endbuf, seof, fid)
   char *ep;
   char *instring, *endbuf;
   FILE_ID fid;
{
   INIT;   /* Dependent declarations and initializations */
   Integer  c;
   Integer eof = seof;
   char *lastep = instring;
   int cclcnt;
   char bracket[NBRA], *bracketp;
   int closed;
   char neg;
   int lc;
   int i, cflg;

   lastep = 0;
   if((c = GETC()) == eof || c == '\n') {
      if(c == '\n') {
	 UNGETC(c);
	 nodelim = 1;
       }
      if(*ep == 0 && !sed)
	 ERROR(41);
      RETURN(ep);
    }
   bracketp = bracket;
   circf = closed = nbra = ebra = 0;
   if(c == '^')
      circf++;
   else
      UNGETC(c);
   while(1) {
      if(ep >= endbuf)
	 ERROR(50);
      c = GETC();
      if(c != '*' && ((c != '\\') || (PEEKC() != '{')))
	 lastep = ep;
      if(c == eof) {
	 *ep++ = CCEOF;
	 RETURN(ep);
       }
      switch(c) {

	 case '.':
	    *ep++ = CDOT;
	    continue;

	 case '\n':
	    if(!sed) {
	    UNGETC(c);
	    *ep++ = CCEOF;
	    nodelim = 1;
	    RETURN(ep);
	     }
	    else ERROR(36);
	 case '*':
	    if(lastep==0 || *lastep==CBRA || *lastep==CKET)
	    goto defchar;
	    *lastep |= STAR;
	    continue;

	 case '$':
	    if(PEEKC() != eof && PEEKC() != '\n')
	    goto defchar;
	    *ep++ = CDOL;
	    continue;

	 case '[':
	    if(&ep[17] >= endbuf)
	    ERROR(50);

	    *ep++ = CCL;
	    lc = 0;
	    for(i = 0; i < 16; i++)
	       ep[i] = 0;

	    neg = 0;
	    if((c = GETC()) == '^') {
	       neg = 1;
	       c = GETC();
	     }

	    do {
	       if(c == '\0' || c == '\n')
		  ERROR(49);
	       if(c == '-' && lc != 0) {
		  if((c = GETC()) == ']') {
		     PLACE('-');
		     break;
		   }
		  while(lc < c) {
		     PLACE(lc);
		     lc++;
		   }
		}
	       lc = c;
	       PLACE(c);
	     }
	    while((c = GETC()) != ']');
	    if(neg) {
	       for(cclcnt = 0; cclcnt < 16; cclcnt++)
		  ep[cclcnt] ^= -1;
	       ep[0] &= 0376;
	     }

	    ep += 16;

	    continue;

	 case '\\':
	    switch(c = GETC()) {

	       case '(':
		  if(nbra >= NBRA)
		  ERROR(43);
		  *bracketp++ = nbra;
		  *ep++ = CBRA;
		  *ep++ = nbra++;
		  continue;

	       case ')':
		  if(bracketp <= bracket || ++ebra != nbra)
		  ERROR(42);
		  *ep++ = CKET;
		  *ep++ = *--bracketp;
		  closed++;
		  continue;

	       case '{':
		  if(lastep == (char *) (0))
		  goto defchar;
		  *lastep |= RNGE;
		  cflg = 0;
	    nlim:
		  c = GETC();
		  i = 0;
		  do {
		     if('0' <= c && c <= '9')
			i = 10 * i + c - '0';
		     else
			ERROR(16);
		   }
		  while(((c = GETC()) != '\\') && (c != ','));
		  if(i > 255)
		     ERROR(11);
		  *ep++ = i;
		  if(c == ',') {
		     if(cflg++)
			ERROR(44);
		     if((c = GETC()) == '\\')
			*ep++ = 255;
		     else {
			UNGETC(c);
			goto nlim;
			/* get 2'nd number */
		      }
			 }
		  if(GETC() != '}')
		     ERROR(45);
		  if(!cflg)	  /* one number */
		     *ep++ = i;
		  else if((ep[-1] & 0377) < (ep[-2] & 0377))
		     ERROR(46);
		  continue;

	       case '\n':
		  ERROR(36);

	       case 'n':
		  c = '\n';
		  goto defchar;

	       default:
		  if(c >= '1' && c <= '9') {
		  if((c -= '1') >= closed)
		     ERROR(25);
		  *ep++ = CBACK;
		  *ep++ = c;
		  continue;
		   }
	     }
	    /* Drop through to default to use \ to turn off special chars */

      defchar:
	 default:
	    lastep = ep;
	    *ep++ = CCHR;
	    *ep++ = c;
       }
    }
}






/******************************************************************/
/*								  */
/*	step - check for any matches of p2 starting at the	  */
/*	   beginning of string p1.				  */
/*								  */
/******************************************************************/


static int
step(p1, p2, beg)
   char *p1, *p2, *beg;
{
   Integer c;

   /* fast check for first character */
   if(*p2==CCHR && !circf) {
      c = p2[1];
      do {
	 if(*p1 != c)
	    continue;
	 if(advance(p1, p2)) {
	    loc1 = p1;
	    return(1);
	  }
       }
      while(*p1++);
      return(0);
    }
   /* regular algorithm */
   do {
      if (circf && p1 != beg && *(p1-1) != '\n') continue;
      if(advance(p1, p2)) {
	 loc1 = p1;
	 return(1);
       }
    }
   while(*p1++);
   return(0);
}




/******************************************************************/
/*								  */
/*	advance - check for a match of the next character	  */
/*								  */
/******************************************************************/


static int
advance(lp, ep)
   char *lp, *ep;
{
   char *curlp;
   char c;
   char *bbeg;
   int ct;

   while(1)
      switch(*ep++) {

	 case CCHR:
	    if(*ep++ == *lp++) continue;
	    return(0);

	 case CDOT:
	    if(*lp++) continue;
	    return(0);

	 case CDOL:
	    if(*lp==0 || *lp == '\n') continue;
	    return(0);

	 case CCEOF:
	    loc2 = lp;
	    return(1);

	 case CCL:
	    c = *lp++ & 0177;
	    if(ISTHERE(c)) {
	       ep += 16;
	       continue;
	     }
	    return(0);
	 case CBRA:
	    braslist[*ep++] = lp;
	    continue;

	 case CKET:
	    braelist[*ep++] = lp;
	    continue;

	 case CCHR|RNGE:
	    c = *ep++;
	    getrnge(ep);
	    while(low--)
	       if(*lp++ != c)
	       return(0);
	    curlp = lp;
	    while(size--)
	       if(*lp++ != c)
	       break;
	    if(size < 0)
	       lp++;
	    ep += 2;
	    goto star;

	 case CDOT|RNGE:
	    getrnge(ep);
	    while(low--)
	       if(*lp++ == '\0')
	       return(0);
	    curlp = lp;
	    while(size--)
	       if(*lp++ == '\0')
	       break;
	    if(size < 0)
	       lp++;
	    ep += 2;
	    goto star;

	 case CCL|RNGE:
	    getrnge(ep + 16);
	    while(low--) {
	       c = *lp++ & 0177;
	       if(!ISTHERE(c))
		  return(0);
	     }
	    curlp = lp;
	    while(size--) {
	       c = *lp++ & 0177;
	       if(!ISTHERE(c))
		  break;
	     }
	    if(size < 0)
	       lp++;
	    ep += 18;		    /* 16 + 2 */
	    goto star;

	 case CBACK:
	    bbeg = braslist[*ep];
	    ct = braelist[*ep++] - bbeg;

	    if(ecmp(bbeg, lp, ct)) {
	       lp += ct;
	       continue;
	     }
	    return(0);

	 case CBACK|STAR:
	    bbeg = braslist[*ep];
	    ct = braelist[*ep++] - bbeg;
	    curlp = lp;
	    while(ecmp(bbeg, lp, ct))
	       lp += ct;

	    while(lp >= curlp) {
	       if(advance(lp, ep))     return(1);
	       lp -= ct;
	     }
	    return(0);

	 case CDOT|STAR:
	    curlp = lp;
	    while(*lp++);
	    goto star;

	 case CCHR|STAR:
	    curlp = lp;
	    while(*lp++ == *ep);
	    ep++;
	    goto star;

	 case CCL|STAR:
	    curlp = lp;
	    do {
	       c = *lp++ & 0177;
	     }
	    while(ISTHERE(c));
	    ep += 16;
	    goto star;

      star:
	    do {
	       if(--lp == locs)
		  break;
	       if(advance(lp, ep))
		  return(1);
	     }
	    while(lp > curlp);
	    return(0);

    }
}





/******************************************************************/
/*								  */
/*	getrnge -						  */
/*								  */
/******************************************************************/


static void
getrnge(str)
   unsigned char *str;
{
   low = *str++ & 0377;
   size = *str == 255 ? 20000 : (*str &0377) - low;
}




/******************************************************************/
/*								  */
/*	ecmp - compare two strings to check for a match 	  */
/*								  */
/******************************************************************/


static int
ecmp(a, b, count)
   char   *a, *b;
   int count;
{
   while(count--)
      if(*a++ != *b++)
	 return(0);

   return(1);
}



/******************************************************************/
/*								  */
/*	regerr - print out an error message when a regular	  */
/*	    expression fails to compile.			  */
/*								  */
/******************************************************************/


static void
regerr(x,fid)
   Integer x;
   FILE_ID fid;
{
   Character reason[100];
   Character msg[150];

   switch (x) {
      case  11	:  sprintf(reason,"Range endpoint too large.");
	 break;
      case  16	:  sprintf(reason,"Bad number.");
	 break;
      case  25	:  sprintf(reason,"``\\digit'' out of range.");
	 break;
      case  36	:  sprintf(reason,"Illegal or missing delimiter.");
	 break;
      case  41	:  sprintf(reason,"No remembered search string.");
	 break;
      case  42	:  sprintf(reason,"\\( \\) imbalance.");
	 break;
      case  43	:  sprintf(reason,"Too many \\(.");
	 break;
      case  44	:  sprintf(reason,"More than 2 numbers given in \\{ \\}.");
	 break;
      case  45	:  sprintf(reason,"} expected after \\.");
	 break;
      case  46	:  sprintf(reason,"First number exceeds second in \\{ \\}.");
	 break;
      case  49	:  sprintf(reason,"[ ] imbalance.");
	 break;
      case  50	:  sprintf(reason,"Regular expression overflow.");
	 break;
      default	:  sprintf(reason," ");
	 break;
    }
   sprintf(msg,"Reg. exp. is invalid.  %s",reason);
   EDfile_display(fid,msg);
   reg_compile_error = TRUE;
}





/* end of filesearch.c */
