/************************************************************************/
/*									*/
/*		filemarks.c						*/
/*									*/
/*	Marking routines for BWE base editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/




/******************************************************************/
/*								  */
/*    FILEset_mark - push the current position (in given file)	  */
/*	  onto the top of the given mark stack. If called with	  */
/*	  FILE_MARK_NEW, the identity of a new mark stack is	  */
/*	  returned.						  */
/*								  */
/******************************************************************/


FILE_MARK
FILEset_mark(fid,mark_id)
   FILE_ID   fid;
   FILE_MARK mark_id;
{
   FILE_MARK filemarkp;  /* the new mark stack if mark_id is FILE_MARK_NEW */
   MARK markp;		 /* the new mark to be added to stack and list	   */
   MARK prev;		 /* the mark previous to position of the new mark  */
   MARK next;		 /* the mark following the position of new mark    */

   FILE_INIT;

   /* create new markstack if FILE_MARK_NEW	 */
   if (mark_id == FILE_MARK_NEW) {
      filemarkp = PALLOC(FILE_MARK_INFO);
      filemarkp->file = fid;
      filemarkp->stacktop = NULL;
      return(filemarkp);
    }

   /* create a new mark to add to stack and list */
   if (mark_id->file == fid) {
      PROTECT;
      markp = PALLOC(MARK_INFO);
      markp->pos = absolute(fid,fid->POINT);   /* NOTE:  marks in files are sticky marks     */
      markp->active = TRUE;
      markp->nxtstackmark = mark_id->stacktop; /* add the new mark to the top of given stack */
      mark_id->stacktop = markp;
      prev = NULL;			       /* add mark to files sorted list of marks     */
      next = fid->filedata->marklist;
      while ((next != NULL) && (next->pos < markp->pos)) {
	 prev = next;
	 next = next->nxtfilemark;
       }
      if (prev == NULL) {		       /* added to the front of the list	     */
	 markp->nxtfilemark = fid->filedata->marklist;
	 fid->filedata->marklist = markp;
       }
      else {				       /* added to proper place within list	     */
	 markp->nxtfilemark = prev->nxtfilemark;
	 prev->nxtfilemark = markp;
       }
      UNPROTECT;
    }
   else EDfile_display(fid,"FILEset_mark: Markstack not associated with given file");

   return mark_id;
}





/******************************************************************/
/*								  */
/*    FILEunset_mark - pop the top position from the indicated	  */
/*	  mark stack.						  */
/*								  */
/******************************************************************/


void
FILEunset_mark(fid,mark_id)
   FILE_ID   fid;
   FILE_MARK mark_id;
{
   MARK markp;
   MARK prev;
   MARK deadmark;

   FILE_INIT;

   if (mark_id != FILE_MARK_NEW) {
      if (mark_id->file == fid) {
	 PROTECT;
	 markp = mark_id-> stacktop;		/* pop from stack until active mark at top    */
	 while ((markp != NULL) && (!(markp->active))) {
	    deadmark = markp;
	    markp = markp->nxtstackmark;
	    free(deadmark);
	  }
	 if (markp != NULL) {			/* pop active mark and remove from the list   */
	    mark_id->stacktop = markp->nxtstackmark;
	    if (markp == fid->filedata->marklist)
	       fid->filedata->marklist = markp->nxtfilemark;
	    else {
	       prev = fid->filedata->marklist;
	       while (prev->nxtfilemark != markp) prev = prev->nxtfilemark;
	       prev->nxtfilemark = markp->nxtfilemark;
	     }
	    free(markp);
	  }
	 else {
	    mark_id->stacktop = NULL;
	  }
	 UNPROTECT;
       }
      else
	 EDfile_display(fid,"FILEunset_mark: Markstack not associated with given file");
    }
   else
      EDfile_display(fid,"FILEunset_mark: Markstack not defined");
}



/******************************************************************/
/*								  */
/*    FILEinq_mark - return the top mark in the designated	  */
/*	  mark stack.						  */
/*								  */
/******************************************************************/


void
FILEinq_mark(fid,mark_id,fp)
   FILE_ID fid;
   FILE_MARK mark_id;
   FILE_POS *fp;
{
   MARK markp;
   MARK deadmark;

   FILE_INIT;

   if (mark_id != FILE_MARK_NEW) {
      if (mark_id->file == fid) {
	 PROTECT;
	 if ((mark_id->stacktop) == NULL) {
	    CopyPos(fid,FILE_POS_START,fp);    /* if stack is empty return FILE_POS_START    */
	  }
	 else {
	    markp = mark_id-> stacktop; 	 /* pop from stack until active mark at top    */
	    while ((markp != NULL) && (!(markp->active))) {
	       deadmark = markp;
	       markp = markp->nxtstackmark;
	       free(deadmark);
	     }
	    mark_id->stacktop = markp;
	    if (markp != NULL)
	       FILE_convert(fid,FILE_relative(fid,markp->pos),fp);
	    else {
	       CopyPos(fid,FILE_POS_START,fp);
	     };
	  };
	 UNPROTECT;
       }
      else
	 EDfile_display(fid,"FILEinq_mark: Markstack not associated with given file");
    }
   else
      EDfile_display(fid,"FILEinq_mark: Markstack not defined");
}





/******************************************************************/
/*								  */
/*    FILEsave_pos - return an integer id representing a sticky   */
/*	  mark into the given files text at the current pos.	  */
/*	 (The user should use these ids only when the file in	  */
/*	  in which they were created is active).		  */
/*								  */
/******************************************************************/


int
FILEsave_pos(fid)
   FILE_ID fid;
{
   MARK markp;		 /* the new mark to be added to list		   */
   MARK prev;		 /* the mark previous to position of the new mark  */
   MARK next;		 /* the mark following the position of new mark    */

   FILE_INIT;

   PROTECT;
   markp = PALLOC(MARK_INFO);
   markp->pos = absolute(fid,fid->POINT);      /* NOTE:  marks in files are sticky marks     */
   markp->active = TRUE;
   markp->nxtstackmark = NULL;
   prev = NULL; 			       /* add mark to files sorted list of marks     */
   next = fid->filedata->marklist;
   while ((next != NULL) && (next->pos < markp->pos)) {
      prev = next;
      next = next->nxtfilemark;
    }
   if (prev == NULL) {			       /* added to the front of the list	     */
      markp->nxtfilemark = fid->filedata->marklist;
      fid->filedata->marklist = markp;
    }
   else {				       /* added to proper place within list	     */
      markp->nxtfilemark = next;
      prev->nxtfilemark = markp;
    }
   UNPROTECT;

   return( (int) markp );
}





/******************************************************************/
/*								  */
/*   FILEdelete_pos -  delete the sticky mark associated with the */
/*	 given id (previously returned by FILEsave_pos).	  */
/*								  */
/******************************************************************/


void
FILEdelete_pos(fid,id)
   FILE_ID fid;
   MARK id;
{
   MARK prev;

   FILE_INIT;

   if (id->active) {
      if (fid->filedata->marklist == NULL)
	 EDabort("Mark not associated with given file - FILEdelete_pos");
      PROTECT;
      if (id == fid->filedata->marklist)
	 fid->filedata->marklist = id->nxtfilemark;
      else {
	 prev = fid->filedata->marklist;
	 while ((prev != NULL) && (prev->nxtfilemark != id)) prev = prev->nxtfilemark;
	 if (prev != NULL)
	    prev->nxtfilemark = id->nxtfilemark;
	 else
	    EDabort("Mark not associated with given file - FILEdelete_pos");
       }
      UNPROTECT;
    }

   free(id);
}





/******************************************************************/
/*								  */
/*   FILEinq_savepos -	return the FILE_POS associated with an	  */
/*	 id of sticky mark previously returned by FILEsave_pos.   */
/*								  */
/******************************************************************/


void
FILEinq_savepos(fid,id,fp)
   FILE_ID fid;
   MARK id;
   FILE_POS *fp;
{
   FILE_INDEX idx;

   FILE_INIT;

   if (id != NULL && id->active) {
      idx = FILE_relative(fid,id->pos);
      if (idx >= 0) FILE_convert(fid,idx,fp );
      else CopyPos(fid,FILE_POS_NONE,fp);
    }
   else {
      CopyPos(fid,FILE_POS_NONE,fp);
    };
}





/******************************************************************/
/*								  */
/*  FILE_adjust_marks - adjust the sticky marks of the file	  */
/*	within the region [from,to) by the displacement delta.	  */
/*	The fixed mark representing the start of the edit region  */
/*	and the sticky mark repres. The end of the edit region	  */
/*	are adjusted to maintain the correct editing limits. The  */
/*	last positions of all other files associated with the	  */
/*	given file are alo adjusted.				  */
/*								  */
/******************************************************************/


void
FILE_adjust_marks(fid,from,to,delta)
   FILE_ID     fid;
   ABSOL_INDEX from;
   ABSOL_INDEX to;
   Integer   delta;
{
   MARK markp;
   FILE_NODE flistp;

   PROTECT;
   fid->filedata->lastvalid = FALSE;

   markp = fid->filedata->marklist;
   while((markp != NULL) && (markp->pos < from)) markp = markp->nxtfilemark;
   while((markp != NULL) && (markp->pos < to)) {
      markp->pos += delta;
      markp = markp->nxtfilemark;
    }

   if ((from < fid->filedata->startregion) && (fid->filedata->startregion <= to))
      fid->filedata->startregion += delta;

   if ((from <= fid->filedata->endregion) && (fid->filedata->endregion < to))
      fid->filedata->endregion += delta;

   flistp = fid->filedata->filelist;
   while (flistp != NULL) {
      if ((from <= flistp->id->pos) && (flistp->id->pos < to))
	 flistp->id->pos += delta;
      flistp = flistp->nxtfile;
    }
   UNPROTECT;
}





/******************************************************************/
/*								  */
/*   FILE_remove_marks - removes sticky marks in the specified	  */
/*	region and adjust former positions in duplicate files.	  */
/*								  */
/******************************************************************/


void
FILE_remove_marks(fid,from,to)
   FILE_ID fid;
   ABSOL_INDEX from;
   ABSOL_INDEX to;
{
   MARK prev;
   MARK next;
   FILE_NODE flistp;

   PROTECT;
   fid->filedata->lastvalid = FALSE;

   prev = NULL; 				      /* remove sticky marks from  */
   next = fid->filedata->marklist;		      /* filelist that are now in  */
   while ((next != NULL) && (next->pos < from)) {     /* the region [from,to)	   */
      prev = next;
      next = next->nxtfilemark;
    }
   while ((next != NULL) && (next->pos < to)) {       /* make mark inactive so it  */
      next->active = FALSE;			     /* can later be removed from */
      next = next->nxtfilemark; 		     /* file mark stack 	  */
    }

   if (prev == NULL)				      /* remove marks from list    */
      fid->filedata->marklist = next;
   else
      prev->nxtfilemark = next;

   flistp = fid->filedata->filelist;
   while (flistp != NULL) {
      if ((from <= flistp->id->pos) && (flistp->id->pos < to))
	 flistp->id->pos = absolute(fid,from);
      flistp = flistp->nxtfile;
    }
   UNPROTECT;
}





/******************************************************************/
/*								  */
/*  FILE_relative - return the corresponding FILE_INDEX to the	  */
/*	absolute character position 'abspos' in the given file.   */
/*	(i.e. change a fixed/sticky mark to a FILE_INDEX)	  */
/*								  */
/******************************************************************/


FILE_INDEX
FILE_relative(fid,abspos)
   FILE_ID     fid;
   ABSOL_INDEX abspos;
{
   if (abspos <= fid->GAPSTART) return(abspos);
   else if (abspos >= fid->GAPEND) return(abspos - (fid->GAPEND - fid->GAPSTART));
   else return(FILE_INDEX_NONE);
}





/* end of filemarks.c */
