/************************************************************************/
/*									*/
/*		filelocation.c						*/
/*									*/
/*	Routines to manipulate FILE_POS's in the BWE editor             */
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	int		inq_line();
static	void		movelines();
static	void		movechars();
static	void		line_start();
static	void		line_end();
static	void		reserved_position();





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


	FILE_POS *	FILE_POS_START;
	FILE_POS *	FILE_POS_END;
	FILE_POS *	FILE_POS_NONE;

static	FILE_POS	f_p_s_node;
static	FILE_POS	f_p_e_node;
static	FILE_POS	f_p_n_node;





/******************************************************************/
/*								  */
/*    FILE_location_init - initalize the file module.		  */
/*								  */
/******************************************************************/


void
FILE_location_init()
{
   f_p_s_node.type = START;
   f_p_e_node.type = END;
   f_p_n_node.type = NONE;

   FILE_POS_START = &(f_p_s_node);
   FILE_POS_END = &(f_p_e_node);
   FILE_POS_NONE = &(f_p_n_node);
}





/******************************************************************/
/*								  */
/*    FILEmove - change the current position in the file via	  */
/*	  the two argument values.  The new position is copied	  */
/*	  into FILE_POS pointed to be "newpos".                   */
/*								  */
/******************************************************************/


void
FILEmove(fid,delta,value,newpos)
   FILE_ID fid;
   Integer  delta;
   Integer value;
   FILE_POS *newpos;
{
   FILEinq_position(fid,delta,value,newpos);

   FILE_set_history(fid,HIST_POSITION,&(fid->CURRPOS),newpos,0,0,0,0);

   FILE_position(fid,newpos);

   CopyPos(fid,newpos,&(fid->CURRPOS));
}





/******************************************************************/
/*								  */
/*    FILEset_position - move to the given (line,char) position   */
/*	  in the file.	Return result in fp's FILE_POS.           */
/*								  */
/******************************************************************/


void
FILEset_position(fid,line,ch,fp)
   FILE_ID fid;
   Integer line;
   Integer ch;
   FILE_POS *fp;
{
   FILEnew_position(fid,line,ch,fp);

   FILE_set_history(fid,HIST_POSITION,&(fid->CURRPOS),fp,0,1,0,0);

   CopyPos(fid,fp,&(fid->CURRPOS));
}





/******************************************************************/
/*								  */
/*    FILEinq_position - return the file position determined by   */
/*	  the given (delta,value) arguments to "newpos".          */
/*								  */
/******************************************************************/


void
FILEinq_position(fid,delta,value,newpos)
   FILE_ID fid;
   Integer delta;
   Integer value;
   FILE_POS *newpos;
{
   FILE_POS * cur;

   FILE_INIT;

   if (delta & FILE_MOVE_ABS) { 		/* determine the inquired position	      */
      CopyPos(fid,(FILE_POS *)value,newpos);
    }
   else {
      if (delta & FILE_MOVE_REL) cur = & fid->CURRPOS;
      else if (delta & FILE_MOVE_START) cur = FILE_POS_START;
      else if (delta & FILE_MOVE_END) cur = FILE_POS_END;
      else cur = & fid->CURRPOS;
      if (delta & FILE_MOVE_LINE)
	 movelines(fid,cur,delta,value,newpos);
      else if (delta & FILE_MOVE_CHAR)
	 movechars(fid,cur,value,newpos);
      else if (delta & FILE_MOVE_LINE_START)
	 line_start(fid,cur,newpos);
      else if (delta & FILE_MOVE_LINE_END)
	 line_end(fid,cur,newpos);
      else if (delta & FILE_MOVE_COND_START)
	 movelines(fid,cur,delta,0,newpos);
      else {
	 CopyPos(fid,cur,newpos);
       };
   };
}





/******************************************************************/
/*								  */
/*    FILEnew_position - given a line number and character	  */
/*	  number, return the corresponding file position.	  */
/*								  */
/******************************************************************/


void
FILEnew_position(fid,linenum,charnum,newpos)
   FILE_ID fid;
   Integer linenum;
   Integer charnum;
   FILE_POS *newpos;
{
   Integer	rel_disp;
   Integer	end_disp;

   FILE_INIT;

   if ((linenum <= 0) || (charnum <= 0)) {
      CopyPos(fid,FILE_POS_NONE,newpos);
      return;
    }

   rel_disp = linenum - fid->CURRLINE;
   end_disp = linenum - fid->ENDLINE;

   if (!fid->CURRPOS.virt && ABS(rel_disp) <= linenum) {
      if (ABS(rel_disp) <= ABS(end_disp))
	 movelines(fid,&(fid->CURRPOS),FILE_MOVE_LINE_START,rel_disp,newpos);
      else
	 movelines(fid,FILE_POS_END,FILE_MOVE_LINE_START,end_disp,newpos);
    }
   else {
      if (linenum <= ABS(end_disp))
	 movelines(fid,FILE_POS_START,FILE_MOVE_LINE_START,--linenum,newpos);
      else
	 movelines(fid,FILE_POS_END,FILE_MOVE_LINE_START,end_disp,newpos);
    };

   movechars(fid,newpos,--charnum,newpos);
}





/******************************************************************/
/*								  */
/*    FILEinq_line - decode a file position into its line number  */
/*    FILEinq_char -- decode a file position into char number	  */
/*								  */
/******************************************************************/


int
FILEinq_line(fid,pos)
   FILE_ID   fid;
   FILE_POS *pos;
{
   FILE_INIT;

   switch (pos->type) {
      case NONE  :
	 return 0;
      case START :
	 return 1;
      case END	 :
	 return fid->ENDLINE;
      case REG	 :
	 return pos->line;
    }

   /*NOTREACHED*/
}





int
FILEinq_char(fid,pos)
   FILE_ID   fid;
   FILE_POS *pos;
{
   FILE_INIT;

   switch (pos->type) {
      case NONE  :
	 return 0;
      case START :
	 return 1;
      case END	 :
	 return FILE_inq_char(fid,fid->FILE_INDEX_END);
      case REG	 :
	 return pos->ch;
    }

   /*NOTREACHED*/
}





/******************************************************************/
/*								  */
/*    FILEinq_currline - return current line number of given file */
/*    FILEinq_currchar - return current char number of given file */
/*								  */
/******************************************************************/


int
FILEinq_currline(fid)
   FILE_ID fid;
{
   FILE_INIT;

   return fid->CURRLINE;
}





int
FILEinq_currchar(fid)
   FILE_ID fid;
{
   FILE_INIT;

   return(fid->CURRCHAR);
}





/******************************************************************/
/*								  */
/*    FILEinq_autoindent  - return the number of spaces on the	  */
/*	  previous non-blank line.				  */
/*								  */
/******************************************************************/


int
FILEinq_autoindent(fid)
   FILE_ID fid;
{
   Integer numspaces = 0;
   FILE_INDEX pos;

   FILE_INIT;

   pos = fid->POINT - 1;
   while ((pos >= FILE_INDEX_START) && (charAt(fid,pos) == NEWLINE)) --pos;
   while ((pos >= FILE_INDEX_START) && (charAt(fid,pos) != NEWLINE)) --pos;

   ++pos;
   while (charAt(fid,pos) == SPACE) {
      ++pos;
      ++numspaces;
    }

   return(numspaces);
}





/******************************************************************/
/*								  */
/*    FILEinq_linespaces  - return the number of spaces at the	  */
/*	  start of the current line.				  */
/*								  */
/******************************************************************/


int
FILEinq_linespaces(fid)
   FILE_ID fid;
{
   Integer numspaces = 0;
   FILE_INDEX pos;

   FILE_INIT;

   pos = fid->POINT - 1;
   while ((pos >= FILE_INDEX_START) && (charAt(fid,pos) != NEWLINE)) --pos;

   ++pos;
   while ((pos < fid->FILE_INDEX_END) && (charAt(fid,pos) == SPACE)) {
      ++numspaces;
      ++pos;
    }

   return(numspaces);
}



/******************************************************************/
/*								  */
/*    FILEpos_compare - return a negative value if x precedes y,  */
/*	  0 if x and y are equivalent, and a positive value	  */
/*	  if y precedes x in the file.				  */
/*								  */
/******************************************************************/


int
FILEpos_compare(fid,x,y)
   FILE_ID fid;
   FILE_POS* x;
   FILE_POS* y;
{
   long diff;

   diff = FILEinq_line(fid,x) - FILEinq_line(fid,y);
   if (diff == 0) diff = FILEinq_char(fid,x) - FILEinq_char(fid,y);

   return diff;
}





/******************************************************************/
/*								  */
/*    FILEpos_copy - copy the first FILE_POS "into" the second    */
/*	  FILE_POS.						  */
/*								  */
/******************************************************************/


void
FILEpos_copy(fid,from,to)
   FILE_ID   fid;
   FILE_POS *from;
   FILE_POS *to;
{
   FILE_INIT;

   CopyPos(fid,from,to);
}





/******************************************************************/
/*								  */
/*   FILE_inq_char - given a file index return the corresponding  */
/*	character position within the line.			  */
/*								  */
/******************************************************************/


Integer
FILE_inq_char(fid,idx)
   FILE_ID    fid;
   FILE_INDEX idx;
{
   Integer charnum;

   if ((idx >= FILE_INDEX_START) && (idx <= fid->FILE_INDEX_END)) {
      charnum = 1;
      while ((idx > FILE_INDEX_START) && (charAt(fid,idx-1) != '\n')) {
	 idx--;
	 charnum++;
       }
      return(charnum);
    }
   else EDabort("Bad argument to FILE_inq_char");

   /*NOTREACHED*/
}






/******************************************************************/
/*								  */
/*   FILE_convert - receive a file index and return the 	  */
/*	corresponding file position.				  */
/*								  */
/******************************************************************/


void
FILE_convert(fid,loc,ptr)
   FILE_ID fid;
   FILE_INDEX loc;
   FILE_POS *ptr;
{
   reserved_position(ptr);

   ptr->type = REG;
   ptr->index = loc;
   ptr->virt = FALSE;
   ptr->line = inq_line(fid,loc);
   ptr->ch = FILE_inq_char(fid,loc);
   ptr->modcnt = fid->filedata->modified;
}





/******************************************************************/
/*								  */
/*   FILE_position - given a FILE_POS, correct its fields if it is*/
/*	one of the special values.				  */
/*								  */
/******************************************************************/


void
FILE_position(fid,fp)
   FILE_ID   fid;
   FILE_POS *fp;
{
   if (fp->type == START) {
      fp->index = FILE_INDEX_START;
      fp->virt = FALSE;
      fp->line = 1;
      fp->ch = 1;
      fp->modcnt = fid->filedata->modified;
    }
   else if (fp->type == END) {
      fp->index = fid->FILE_INDEX_END;
      fp->virt = FALSE;
      fp->line = fid->ENDLINE;
      fp->ch = FILE_inq_char(fid,fid->FILE_INDEX_END);
      fp->modcnt = fid->filedata->modified;
    }
   else if (fp->type == NONE) {
      fp->index = FILE_INDEX_NONE;
      fp->virt = FALSE;
      fp->line = 0;
      fp->ch = 0;
      fp->modcnt = fid->filedata->modified;
    }
   else if (fp->modcnt != fid->filedata->modified) {
      if (fp == &fid->filedata->currpos) {
	 fp->modcnt = fid->filedata->modified;
       }
      else {
	 FILEnew_position(fid,fp->line,fp->ch,fp);
       };
    };
}





/******************************************************************/
/*								  */
/*   FILE_copy_position - copy the given FILE_POS fields into a   */
/*		new FILE_POS_PTR's.                               */
/*								  */
/******************************************************************/


void
FILE_copy_position(fid,from,to)
   FILE_ID fid;
   FILE_POS *from;
   FILE_POS *to;
{
   reserved_position(to);

   FILE_position(fid,from);

   if (from->type == NONE) to->type = NONE;
   else to->type = REG;

   to->index = from->index;
   to->virt = from->virt;
   to->line = from->line;
   to->ch = from->ch;
   to->modcnt = from->modcnt;
}





/******************************************************************/
/*								  */
/*    inq_line - given a file index return the corresponding	  */
/*	line position within the file.				  */
/*								  */
/******************************************************************/


static int
inq_line(fid,idx)
   FILE_ID    fid;
   FILE_INDEX idx;
{
   Integer linenum;
   Integer rel_disp;
   Integer end_disp;
   Integer last_disp;
   FILE_INDEX i;

   if ((idx >= FILE_INDEX_START) && (idx <= fid->FILE_INDEX_END)) {
      rel_disp = idx - fid->POINT;		 /* calculate relative displacements from start, end,  */
      end_disp = fid->FILE_INDEX_END - idx;	 /* and current position and use smallest to calculate */
						   /* the line number of the given index.		 */
      if (fid->filedata->lastvalid) last_disp = idx - fid->filedata->lastindex;
      else last_disp = end_disp+1;
      if (fid->CURRPOS.virt) rel_disp = end_disp+1;

      if ((ABS(rel_disp) < idx) && (ABS(rel_disp) < end_disp) &&
	     (ABS(rel_disp) <= ABS(last_disp))) {
	 linenum = fid->CURRLINE;
	 if (rel_disp >= 0) {
	    for (i = fid->POINT; i != idx; i++)
	       if (charAt(fid,i) == NEWLINE) linenum++;
	  }
	 else {
	    for (i = fid->POINT; i != idx; i--)
	       if (charAt(fid,i-1) == NEWLINE) linenum--;
	  }
       }
      else if ((ABS(last_disp) < idx) && (ABS(last_disp) < end_disp)) {
	 linenum = fid->filedata->lastline;
	 if (last_disp >= 0) {
	    for (i = fid->filedata->lastindex; i != idx; i++)
	       if (charAt(fid,i) == NEWLINE) linenum++;
	  }
	 else {
	    for (i = fid->filedata->lastindex; i != idx; i--)
	       if (charAt(fid,i-1) == NEWLINE) linenum--;
	  }
	}
      else if (idx <= end_disp) {
	 linenum = 1;
	 for (i = FILE_INDEX_START; i != idx; i++)
	    if (charAt(fid,i) == NEWLINE) linenum++;
       }
      else {
	 linenum = fid->ENDLINE;
	 for (i = fid->FILE_INDEX_END; i != idx; i--)
	    if (charAt(fid,i-1) == NEWLINE) linenum--;
       }

      fid->filedata->lastindex = idx;
      fid->filedata->lastline = linenum;
      fid->filedata->lastvalid = TRUE;
      if (idx >= fid->FILE_INDEX_END-1) fid->filedata->lastvalid = FALSE;

      return(linenum);
    }
   else EDabort("Bad argument to inq_line");

   /*NOTREACHED*/
}





/******************************************************************/
/*								  */
/*    movelines - return the file position 'numlines' above or    */
/*	below the current position. It returns the pos at the	  */
/*	line start or line end of the new pos if specified by	  */
/*	delta.							  */
/*								  */
/******************************************************************/


static void
movelines(fid,pos,delta,numlines,newpos)
   FILE_ID fid;
   FILE_POS *pos;
   Integer  delta;
   Integer numlines;
   FILE_POS *newpos;
{
   register FILE_INDEX	 index;
   register Integer	 charnum;
   register Integer	 linenum;
   FILE_POS	fp;
   Integer i,ch;
   FILE_INDEX idx;

   reserved_position(newpos);

   FILE_position(fid,pos);	     /* make sure fields are ok eg.FILE_POS_END */
   linenum = pos->line + numlines;   /* calculate line number for new position	*/
   charnum = pos->ch;

   PROTECT;

   if ((linenum > 1) && (linenum <= fid->ENDLINE)) {
/*    FILE_position_real(fid,pos);	*/
      index = pos->index;
      if (numlines > 0) {	    /* scan forward from pos' file index       */
	 while (numlines != 0) {
	    if (charAt(fid,index) == NEWLINE) numlines--;
	    index++;
	  }
       }
      else {			    /* scan backwards from pos's file index    */
	 if (pos->line > fid->ENDLINE) numlines = linenum - fid->ENDLINE;
	 numlines--;
	 while (numlines != 0) {
	    if (charAt(fid,index-1) == NEWLINE) numlines++;
	    index--;
	    if (index < 0) abort();
	  }
	 index++;
       }
      newpos->type = REG;
      newpos->index = index;
      newpos->virt = FALSE;
      newpos->line = linenum;
      newpos->ch = 1;
    }
   else if (linenum > fid->ENDLINE) {	     /* newpos must be in virtual space 	*/
      newpos->type = REG;
      newpos->index = fid->FILE_INDEX_END;
      newpos->virt = TRUE;
      newpos->line = linenum;
      newpos->ch = 1;
    }
   else { /* linenum <= 1 */
      newpos->type = REG;
      newpos->index = FILE_INDEX_START;
      newpos->virt = FALSE;
      newpos->line = 1;
      newpos->ch = 1;
    }

   newpos->modcnt = fid->filedata->modified;

   UNPROTECT;

   if (delta & FILE_MOVE_COND_START) {
      if (newpos->virt) charnum = 1;
      else {
	 idx = index;
	 i = charnum-1;
	 while ((idx < fid->FILE_INDEX_END) && i > 0) {
	    ch = charAt(fid,idx);
	    if (ch == NEWLINE) i = 0;
	    else if (ch != ' ') break;
	    idx++;
	    i--;
	  };
	 if (i <= 0) charnum = 1;
       };
    };

   /* determine newpos' character number      */
   if (delta & FILE_MOVE_LINE_START) return;
   else if (delta & FILE_MOVE_LINE_END) {
      CopyPos(fid,newpos,&fp);
      line_end(fid,&fp,newpos);
    }
   else {
      newpos->ch = charnum;
      if (newpos->virt) return;
      charnum--;
      index = newpos->index;
      while ((index < fid->FILE_INDEX_END) &&
		(charAt(fid,index) != NEWLINE) && (charnum != 0)) {
	 index++;
	 charnum--;
       }
      newpos->index = index;
      if (charnum == 0) newpos->virt = FALSE;
      else newpos->virt = TRUE;
    }
}





/******************************************************************/
/*								  */
/*    movechars - returns the file position 'numchars' away       */
/*	from the given position on the same line. It cannot go	  */
/*	before the start of the line but will go 'virtual' spaces */
/*	past the end of the line.				  */
/*								  */
/******************************************************************/


static void
movechars(fid,pos,numchars,newpos)
   FILE_ID fid;
   FILE_POS *pos;
   Integer numchars;
   FILE_POS *newpos;
{
   FILE_POS   ln_start,ln_end;
   Integer    charnum;

   reserved_position(newpos);

   if (numchars == 0) {
      FILE_position(fid,pos);
      CopyPos(fid,pos,newpos);
      return;
    }

   PROTECT;

   FILE_position(fid,pos);

   line_start(fid,pos,&ln_start);
   line_end(fid,pos,&ln_end);

   charnum = pos->ch + numchars;     /* calculate char number for new position	*/

   if (charnum <= 1) {
      CopyPos(fid,&ln_start,newpos);
    }
   else if (charnum <= ln_end.ch) {
      CopyPos(fid,&ln_start,newpos);
      newpos->index += (charnum-1);
      newpos->ch = charnum;
    }
   else {
      CopyPos(fid,&ln_end,newpos);
      newpos->virt = TRUE;
      newpos->ch = charnum;
    }

   newpos->type = REG;

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    line_start - return file position at the start of the line  */
/*	in which the given position is located. 		  */
/*								  */
/******************************************************************/

static void
line_start(fid,pos,newpos)
   FILE_ID   fid;
   FILE_POS *pos;
   FILE_POS *newpos;
{
   register FILE_INDEX	 idx;

   reserved_position(newpos);

   PROTECT;

   FILE_position(fid,pos);
   newpos->line = pos->line;
   newpos->ch = 1;
   if (newpos->line > fid->ENDLINE) {
      newpos->index = fid->FILE_INDEX_END;
      newpos->virt = TRUE;
    }
   else {
      idx = pos->index;
      while ((idx > FILE_INDEX_START) && (charAt(fid,idx-1) != NEWLINE)) idx--;
      newpos->index = idx;
      newpos->virt = FALSE;
    }

   newpos->type = REG;
   newpos->modcnt = fid->filedata->modified;

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    line_end - return file position at the end of the line	  */
/*	in which the given position is located. 		  */
/*								  */
/******************************************************************/


static void
line_end(fid,pos,newpos)
   FILE_ID   fid;
   FILE_POS *pos;
   FILE_POS *newpos;
{
   register FILE_INDEX	 idx;

   reserved_position(newpos);

   PROTECT;

   FILE_position(fid,pos);

   if ((newpos->line = pos->line) > fid->ENDLINE) {
      newpos->index = fid->FILE_INDEX_END;
      newpos->virt = TRUE;
      newpos->ch = 1;
    }
   else {
      idx = pos->index;
      while ((idx < fid->FILE_INDEX_END) && (charAt(fid,idx) != NEWLINE)) idx++;
      newpos->index = idx;
      newpos->virt = FALSE;
      newpos->ch = FILE_inq_char(fid,idx);
    }

   newpos->type = REG;
   newpos->modcnt = fid->filedata->modified;

   UNPROTECT;
}






/******************************************************************/
/*								  */
/*    reserved_position - insure that modifiable FILE_POS is not  */
/*	   the reserved FILE_POS' START,END, or NONE.             */
/*								  */
/******************************************************************/


static void
reserved_position(fp)
   FILE_POS *fp;
{
   if ((fp == FILE_POS_START) || (fp == FILE_POS_END) || (fp == FILE_POS_NONE))
      EDabort("Attempt to modify reserved FILE_POS");
}





/* end of filelocation.c */
