/******************************************************************/
/*								  */
/*		filehistory.c					  */
/*								  */
/*	History routines for the BWE editor			  */
/*								  */
/******************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"





/******************************************************************/
/*								  */
/*	Forward definitions					  */
/*								  */
/******************************************************************/


static	void		undo_action();





/******************************************************************/
/*								  */
/*    FILEend_command - indicates a user level command has ended. */
/*	  Returns an integer marker for later undoing.		  */
/*								  */
/******************************************************************/


int
FILEend_command(fid,noupdate)
   FILE_ID fid;
   Set noupdate;
{
   FILE_HIST HSitem;

   FILE_INIT;

   fid->filedata->ignerror = FALSE;

   if (!noupdate) FILE_regmsg(fid,FILE_MSG_COMMAND,0,0,0,0);

   PROTECT;
   HSitem = fid->filedata->undo_history;
   while (HSitem != NULL && HSitem->action != HIST_COMMAND)
      HSitem = HSitem->nxt_history;
   if (HSitem != fid->filedata->undo_history)
      FILE_set_history(fid,HIST_COMMAND,0,0,0,0,0,0);
   UNPROTECT;

   return( (int) HSitem );
}




/******************************************************************/
/*								  */
/*    FILEundo_command - causes the editing of the given file	  */
/*	  to be undone to back before the command identified by   */
/*	  the given mark.					  */
/*								  */
/******************************************************************/


int
FILEundo_command(fid,histmark)
   FILE_ID   fid;
   FILE_HIST histmark;
{
   register FILE_HIST HSitem;
   register FILE_HIST undone;

   FILE_INIT;

   fid->filedata->ignerror = FALSE;

   if (fid != histmark->file) {
      EDfile_display(fid,"History mark not associated with given file in FILEundo_command");
      fid->filedata->ignerror = FALSE;
      return(FALSE);
    }
   if ((HSitem = fid->filedata->undo_history) == NULL) {
      EDfile_display(fid,"Nothing left to UNDO");
      fid->filedata->ignerror = FALSE;
      return(FALSE);
    }

   fid->filedata->hist_mode = HIST_MODE_UNDO;

   FILE_set_history(fid,HIST_COMMAND,0,0,0,0,0,0);  /* add command node to redo stack */

   while ((HSitem != histmark) && (HSitem != NULL)) {
      undo_action(fid,HSitem);
      undone = HSitem;
      HSitem = HSitem->nxt_history;
      if (undone->oldtext != NULL) free(undone->oldtext);
      free(undone);
      fid->filedata->undo_history = HSitem;
    }

   fid->filedata->hist_mode = HIST_MODE_NORMAL;

   if (HSitem == histmark) {
      fid->filedata->undo_history = histmark;
      return(TRUE);
    }
   else {
      FILE_set_history(fid,HIST_COMMAND,0,0,0,0,0,0);
      return(FALSE);
    }
}





/******************************************************************/
/*								  */
/*    FILEredo_command - causes the editing of the given file	  */
/*	  to be redone back to before the last undo.		  */
/*								  */
/******************************************************************/


int
FILEredo_command(fid)
   FILE_ID fid;
{
   register FILE_HIST HSitem;
   register FILE_HIST redone;

   FILE_INIT;

   fid->filedata->ignerror = FALSE;

   if ((HSitem = fid->filedata->redo_history) == NULL) {
      EDfile_display(fid,"Nothing left to REDO");
      fid->filedata->ignerror = FALSE;
      return(FALSE);
    }

   fid->filedata->hist_mode = HIST_MODE_REDO;

   while (HSitem->action != HIST_COMMAND) {
      undo_action(fid,HSitem);
      redone = HSitem;
      HSitem = HSitem->nxt_history;
      if (redone->oldtext != NULL) free(redone->oldtext);
      free(redone);
    }

   fid->filedata->redo_history = HSitem->nxt_history;
   free(HSitem);

   fid->filedata->hist_mode = HIST_MODE_NORMAL;

   return(TRUE);
}





/******************************************************************/
/*								  */
/*    FILEfree_history -- frees all history below given mark in   */
/*	  a history list for the file.	(NOTE:	The node that is  */
/*	  associated with the mark is not removed from the list.) */
/*								  */
/******************************************************************/


void
FILEfree_history(histmark)
   FILE_HIST histmark;
{
   FILE_HIST q;

   FILE_INIT;

   if (histmark == NULL) return;

   PROTECT;
   while ((q = histmark->nxt_history) != NULL) {
      histmark->nxt_history = q->nxt_history;
      if ( ((q->action == HIST_DELETE) || (q->action == HIST_BUFFER))
	      && (q->oldtext != NULL) )
	 free(q->oldtext);
      q->nxt_history = NULL;
      free(q);
    }
   UNPROTECT;
}





/******************************************************************/
/*								  */
/*   FILE_set_history - adds a history node with the given action */
/*     onto the history list of the given file. 		  */
/*								  */
/******************************************************************/


void
FILE_set_history(fid,hs_action,pos1,pos2,buf,oldchar,oldtext,ins_mode)
   FILE_ID	fid;
   HIST_ACTION	hs_action;
   FILE_POS    *pos1;
   FILE_POS    *pos2;
   FILE_BUF	buf;
   Character	oldchar;
   String	oldtext;
   Boolean	ins_mode;
{
   FILE_HIST HSitem;
   String buf_contents;

   HSitem = PALLOC(FILE_HIST_INFO);
   HSitem->file = fid;
   HSitem->action = hs_action;
   HSitem->oldtext = NULL;

   switch (hs_action) {
      case HIST_INSERT:
      case HIST_LIMITS:
	 CopyPos(fid,pos1,&(HSitem->pos1));
	 CopyPos(fid,pos2,&(HSitem->pos2));
	 break;

      case HIST_DELETE:
	 CopyPos(fid,pos1,&(HSitem->pos1));
	 HSitem->oldtext = (String) malloc( strlen(oldtext) + 1);
	 strcpy(HSitem->oldtext,oldtext);
	 break;

      case HIST_TYPEIN:
      case HIST_BACKSPACE:
      case HIST_NEWLINE:
	 CopyPos(fid,pos1,&(HSitem->pos1));
	 CopyPos(fid,pos2,&(HSitem->pos2));
	 HSitem->oldchar = oldchar;
	 HSitem->ins_mode = ins_mode;
	 break;

      case HIST_BUFFER:
	 HSitem->buffer = buf;
	 HSitem->oldtext = (String) malloc( FILEbuffer_size(buf) + 1);
	 buf_contents = FILEbuffer_contents(buf);
	 if (buf_contents) {
	    strcpy(HSitem->oldtext,buf_contents);
	    free(buf_contents);
	  }
	 else *(HSitem->oldtext) = 0;
	 break;

      case HIST_POSITION:
	 CopyPos(fid,pos1,&HSitem->pos1);
	 CopyPos(fid,pos2,&(HSitem->pos2));
	 HSitem->oldchar = oldchar;
	 break;

      case HIST_COMMAND:
	 CopyPos(fid,&fid->CURRPOS,&HSitem->pos1);
	 HSitem->pos2.index = fid->filedata->size;
	 HSitem->pos2.line = fid->filedata->endline;
	 break;
    }

   PROTECT;
   if (fid->filedata->hist_mode == HIST_MODE_UNDO) {
      HSitem->nxt_history = fid->filedata->redo_history;
      fid->filedata->redo_history = HSitem;
    }
   else {
      if ((fid->filedata->hist_mode != HIST_MODE_REDO) &&
	     (hs_action != HIST_COMMAND) &&  /* clear redo stack */
	     (fid->filedata->redo_history != NULL)) {
	 FILEfree_history(fid->filedata->redo_history);
	 free(fid->filedata->redo_history);
	 fid->filedata->redo_history = NULL;
       }

      if (fid->filedata->undo_history != NULL)
	 fid->filedata->undo_history->last_history = HSitem;
      HSitem->last_history = NULL;

      HSitem->nxt_history = fid->filedata->undo_history;
      fid->filedata->undo_history = HSitem;
    }
   UNPROTECT;
}





/******************************************************************/
/*								  */
/*   undo_action -- undo the action specified in the given hist-  */
/*	  ory node pointer.  The node is not freed.		  */
/*								  */
/******************************************************************/


static void
undo_action(fid,HSitem)
   FILE_ID fid;
   FILE_HIST HSitem;
{
   FILE_BUF buf_id;
   Character ch_str[2];

   switch (HSitem->action) {
      case HIST_INSERT:
	 FILE_position(fid,&(HSitem->pos1));
	 CopyPos(fid,&(HSitem->pos1),&(fid->CURRPOS));
	 buf_id = FILEinq_buffer("BWEDIT$_undobuf");
	 FILEdelete(fid,buf_id,FILE_MOVE_ABS,&(HSitem->pos2));
	 break;

      case HIST_DELETE:
	 FILE_position(fid,&(HSitem->pos1));
	 CopyPos(fid,&(HSitem->pos1),&(fid->CURRPOS));
	 buf_id = FILEinq_buffer("BWEDIT$_undobuf");
	 FILE_set_history(fid,HIST_BUFFER,0,0,buf_id,0,0,0);
	 FILEbuffer_define(buf_id,HSitem->oldtext);
	 FILEinsert(fid,buf_id,1);
	 break;

      case HIST_TYPEIN:
	 FILE_position(fid,&(HSitem->pos2));
	 CopyPos(fid,&(HSitem->pos2),&(fid->CURRPOS));
	 FILEtypein(fid,BACKSPACE,1);
	 if (! (HSitem->ins_mode) ) {
	    sprintf(ch_str,"%c",HSitem->oldchar);
	    buf_id = FILEinq_buffer("BWEDIT$_undobuf");
	    FILE_set_history(fid,HIST_BUFFER,0,0,buf_id,0,0,0);
	    FILEbuffer_define(buf_id,ch_str);
	    FILEinsert(fid,buf_id,1);
	  }
	 break;

      case HIST_BACKSPACE:
	 FILE_position(fid,&(HSitem->pos2));
	 CopyPos(fid,&(HSitem->pos2),&(fid->CURRPOS));
	 FILEtypein(fid,HSitem->oldchar, HSitem->ins_mode);
	 break;

      case HIST_NEWLINE:
	 FILE_position(fid,&(HSitem->pos1));
	 CopyPos(fid,&(HSitem->pos1),&(fid->CURRPOS));
	 buf_id = FILEinq_buffer("BWEDIT$_undobuf");
	 FILE_position(fid,&(HSitem->pos2));
	 FILEdelete(fid,buf_id,FILE_MOVE_ABS,&(HSitem->pos2));
	 break;

      case HIST_LIMITS:
	 FILE_position(fid,&(HSitem->pos1));
	 FILE_position(fid,&(HSitem->pos2));
	 FILEedit_limits(fid,&(HSitem->pos1),&(HSitem->pos2));
	 break;

      case HIST_BUFFER:
	 FILE_set_history(fid,HIST_BUFFER,0,0,HSitem->buffer,0,0,0);
	 FILEbuffer_define(HSitem->buffer,HSitem->oldtext);
	 break;

      case HIST_POSITION:
	 FILE_position(fid,&(HSitem->pos1));
	 CopyPos(fid,&(HSitem->pos1),&(fid->CURRPOS));
	 break;

      case HIST_COMMAND:
	 break;
    }
}






/* end of filehistory.c */
