/************************************************************************/
/*									*/
/*		filefiles.c						*/
/*									*/
/*	File manipulation routines for editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "editor_local.h"

#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define INPUT	0
#define OUTPUT	1





/************************************************************************/
/*									*/
/*	Local storage definitions					*/
/*									*/
/************************************************************************/


	PROT_DECL;
	Boolean 	FILE__Initialized = FALSE;

static	FILE_ID 	File_list = FILE_ID_NONE;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


#ifdef LIB
static	void		cleanup();
#endif

static	FILE_ID 	duplicate();
static	FILE_ID 	filedecide();
static	FILE_ID 	createfileid();
static	Boolean 	expand_tabs();
static	Integer 	tabify();
static	void		cancel_file();
static	void		free_filedata();
static	void		free_fileid();
static	void		filter_io();
static	Boolean 	savebackups();
static	void		dir();
static	void		remove_name();
static	void		filetail();





/******************************************************************/
/*								  */
/*	FILE_init - intialize the file package. 		  */
/*	cleanup - reinitialize global file variables (lib)	  */
/*								  */
/******************************************************************/


void
FILE_init()
{
   PROT_INIT;
   PROTECT;

   if (!FILE__Initialized) {
#ifdef LIB
      BROWNregister_cleanup(cleanup);
#endif

      FILE__Initialized = TRUE;
      File_list = FILE_ID_NONE;

      FILE_location_init();
      FILE_buffer_init();
    };

   UNPROTECT;
}


#ifdef LIB
static void
cleanup()
{
   PROTECT;
   File_Initialized = FALSE;
   File_list = FILE_ID_NONE;
   UNPROTECT;
}
#endif





/****************************************************************/
/*								*/
/*    FILEopen - open the given file under the open		*/
/*	  flag parameters. Set up the internal structure	*/
/*	  (file header) to maintain the in-core file.		*/
/*								*/
/****************************************************************/


FILE_ID
FILEopen(file,open_flag)
   String file;
   Integer open_flag;
{
   FILE_ID file_id;		      /* returned FILE_ID		      */
   FILE_ID fp;			      /* used for finding previous file_ids   */
   FL_STATUS status = FL_FAIL;	      /* status after the open try	      */
   Integer openfd = -1; 	      /* file descriiptor from the open       */
   struct stat sbuf;		      /* info. stat buffer		      */
   Integer size = 0;		      /* size of the opened file	      */
   Integer device = 0;		      /* file's device type                   */
   Integer inode = 0;		      /* file's inode number                  */
   Integer mode = 0666;

   FILE_INIT;

   if (file == NULL) {
      EDfile_display(NULL,"No file name given");
      return(FILE_ID_NONE);
    }

   if (open_flag & FILE_OPEN_TEMPORARY) status = FL_TEMP;
   else if (access(file,F_OK) == 0) {
      stat(file,&sbuf);
      device = sbuf.st_dev;
      inode = sbuf.st_ino;
      size = sbuf.st_size;
      mode = sbuf.st_mode;

      /* traverse the file_id list to see if this is a duplicate */
      for (fp=File_list; fp!=FILE_ID_NONE; fp=fp->nxtfile) {
	 if ((device == fp->device) && (inode == fp->inode)) {
	    file_id = duplicate(fp,open_flag);
	    return(file_id);
	  }
       };

      switch(sbuf.st_mode & S_IFMT) {
	 case S_IFBLK:
	    status = FL_BAD;
	    break;
	 case S_IFCHR:
	    status = FL_BAD;
	    break;
	 case S_IFDIR:
	    status = FL_DIR;
	    break;
	 default:
	    status = FL_OK;
	    break;
       }

      if (open_flag & FILE_OPEN_READONLY) {			 /* READONLY */
	 if ((openfd  = open(file,O_RDONLY)) >= 0) {
	    if (status == FL_OK) status = FL_READ;
	  }
	 else status = FL_FAIL;
       }
      else if (open_flag & FILE_OPEN_NEW) {			 /* NEW */
	 if ((openfd = open(file,O_WRONLY)) >= 0) {
	    if (status == FL_OK) status = FL_TRUNC;
	  }
	 else status = FL_NOWRITE;
       }
      else if (status == FL_OK && (openfd = open(file,O_CREAT|O_RDWR,0666)) >= 0) ;
      else if (status == FL_OK && (openfd  = open(file,O_RDONLY)) >= 0) {
	 status = FL_READ;
       }
      else status = FL_FAIL;
    }
   else { /* the file does not exist */
      if (open_flag & FILE_OPEN_READONLY)			 /* READONLY */
	 status = FL_NOREAD;
      else {
	 if ((openfd = open(file,O_CREAT|O_RDWR,0666)) >= 0) {
	    status = FL_NEW;
	    fstat(openfd,&sbuf);
	    device = sbuf.st_dev;
	    inode = sbuf.st_ino;
	    mode = sbuf.st_mode;
	  }
	 else status = FL_FAIL;
       }
    }

   file_id = filedecide(file,status,open_flag,openfd,size,device,inode,mode);

   return(file_id);
}





/******************************************************************/
/*								  */
/*    FILEinq_filename - return the given path of the given file. */
/*								  */
/******************************************************************/


char *
FILEinq_filename(fid)
   FILE_ID fid;
{
   FILE_INIT;

   return (fid->pathname == NULL ? NULL : SALLOC(fid->pathname));
}



/******************************************************************/
/*								  */
/*    FILEfilter - add the filter "fct" to the list of filters    */
/*	  associated with the given file.			  */
/*								  */
/******************************************************************/


void
FILEfilter(fid,when,fct)
   FILE_ID fid;
   FILE_FILTER_WHEN when;
   Function_Ptr  fct;
{
   FILTER fltr,last;
   FILTER newfilter;

   FILE_INIT;

   PROTECT;

   if (when & FILE_FILTER_INPUT) {
      last = NULL;
      for (fltr = fid->filedata->infilters;
	   (fltr != NULL) && (fltr->fct != fct);
	   fltr = fltr->nxtfilter) {
	 last = fltr;
       }
      if (fltr == NULL) {
	 newfilter = PALLOC(FILTER_INFO);
	 newfilter->fct = fct;
	 newfilter->nxtfilter = NULL;
	 if (last == NULL) fid->filedata->infilters = newfilter;
	 else last->nxtfilter = newfilter;
       };
    }

   if (when & FILE_FILTER_OUTPUT) {
      last = NULL;
      for (fltr = fid->filedata->outfilters;
	   (fltr != NULL) && (fltr->fct != fct);
	   fltr = fltr->nxtfilter) {
	 last = fltr;
       }
      if (fltr == NULL) {
	 newfilter = PALLOC(FILTER_INFO);
	 newfilter->fct = fct;
	 newfilter->nxtfilter = NULL;
	 if (last == NULL) fid->filedata->outfilters = newfilter;
	 else last->nxtfilter = newfilter;
       };
    }

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    FILErelease - read in a file, but pass the text through	  */
/*	  the current set of input filters.  Called when the	  */
/*	  open_flag includes the HOLD option.			  */
/*								  */
/******************************************************************/


void
FILErelease(fid)
   FILE_ID fid;
{
   FILE_INIT;

   PROTECT;
   filter_io(fid,INPUT);
   if (expand_tabs(&(fid->filedata->filetext),&(fid->filedata->size),
		      &(fid->filedata->allocsize),
		      &(fid->filedata->endline))) {
      fid->filedata->startregion = FILE_INDEX_START;
      fid->filedata->endregion = fid->filedata->allocsize;
      fid->filedata->nullregion = FALSE;
      fid->filedata->gapstart = fid->filedata->size;
      fid->filedata->gapend = fid->filedata->allocsize;
      fid->filedata->filetext[fid->filedata->allocsize] = 0;
      UNPROTECT;
      FILE_regmsg(fid,FILE_MSG_INSERT,FILE_POS_START,FILE_POS_END,0,0);
      FILEend_command(fid,FALSE);
    }
   else {  /* wasn't enough space in expanding tabs */
      UNPROTECT;
      EDfile_display(NULL,"Not enough core to open file");
    }
}






/******************************************************************/
/*								  */
/*    FILEmodified - return true if the given FILE_ID has been	  */
/*	modified during the current edit session and it is of	  */
/*	type OK, NEW, or TRUNC.  (i.e., possible changes won't    */
/*	be saved in READ, TEMP, or COPY.)			  */
/*								  */
/******************************************************************/


int
FILEmodified(fid)
   FILE_ID fid;
{
   if (fid != FILE_ID_NONE)
      return( (fid->filedata->modified > 0) &&
		 ((fid->filetype == FL_NEW) ||
		     (fid->filetype == FL_OK) ||
		     (fid->filetype == FL_TEMP) ||
		     (fid->filetype == FL_TRUNC)));
   else
      return(FALSE);
}





/******************************************************************/
/*								  */
/*    FILEclose - close the given file.  If save_flag is true	  */
/*	  then save the new version, else restore the old one.	  */
/*								  */
/******************************************************************/


int
FILEclose(file_id,save_flag)
   FILE_ID file_id;
   Boolean save_flag;
{
   Character	  justfile[MAXSTRINGLENGTH];	/* only the filename of the abs path		  */
   Character	  fileptr[MAXSTRINGLENGTH];	/* used to remove secured version		  */
   FILE_DATA_PTR  dptr; 			/* used for quicker access to Currfile->filedata  */
   Integer	  writeval;			/* number of bytes written to file		  */

   FILE_INIT;

   if (file_id == FILE_ID_NONE)
      EDabort("Attempt to close FILE_ID_NONE");

   if (--file_id->filedata->usecount > 0) {
      cancel_file(file_id);
      return TRUE;
    };

   PROTECT;

   switch (file_id->filetype) {
      case FL_OK	:
      case FL_NEW	:
      case FL_TRUNC	:
	 if (save_flag) {
	    FILE_movegapto(file_id,file_id->FILE_INDEX_END);
	    if (file_id->filetype != FL_NEW) { /* if not new, save backups and remove old contents */
	       close(file_id->filedesc);
	       if (!savebackups(file_id,file_id->pathname)) unlink(file_id->pathname);
	       /* ^^backup didn't remove old version, so do it here^^ */
	       if ((file_id->filedesc = open(file_id->pathname,O_CREAT|O_WRONLY,file_id->rights)) < 0) {
		  EDfile_display(file_id,"Unable to save file changes.Check /tmp and backups.");
		  return FALSE;
		}
	     }
	    dptr = file_id->filedata;
	    if (dptr->outfilters == NULL) { /* if no outfilters, just write new text */
	       *(file_id->TEXTPTR + file_id->GAPSTART) = 0;
	       file_id->SIZE = tabify(file_id->TEXTPTR);
	       writeval = write(file_id->filedesc,dptr->filetext,dptr->size);
	       if (writeval != dptr->size) {
		  EDfile_display(file_id,"Error in write to file. Check /tmp and backups.");
		  return FALSE;
		}
	     }
	    else    /* run the text through output filters */
	       filter_io(file_id,OUTPUT);

	    close(file_id->filedesc);
	  }
	 else {  /* save_flag == FALSE */
	    close(file_id->filedesc);
	    if (file_id->filetype == FL_NEW) unlink(file_id->pathname);
	  }
	 break;

      case FL_READ	:
	 close(file_id->filedesc);
	 break;

      case FL_TEMP	:
      case FL_COPY	:
	 break;
    }

   UNPROTECT;

   filetail(file_id->pathname,justfile);
   sprintf(fileptr,"/tmp/%s",justfile);
   unlink(fileptr);

   cancel_file(file_id);

   return(TRUE);
}





/******************************************************************/
/*								  */
/*    FILEsave_as -- save current file given name		  */
/*								  */
/******************************************************************/


int
FILEsave_as(file_id,name)
   FILE_ID file_id;
   String name;
{
   FILE_DATA_PTR  dptr;
   Integer ln,otf;
   String txt;
   Boolean fg;

   FILE_INIT;

   if (file_id == FILE_ID_NONE)
      EDabort("Attempt to close FILE_ID_NONE");

   PROTECT;

   if (name != NULL && *name == 0) name = NULL;

   if (name == NULL) {
      if (FILEmodified(file_id) &&
	     (file_id->filetype == FL_OK || file_id->filetype == FL_TRUNC)) {
	 FILE_movegapto(file_id,file_id->FILE_INDEX_END);
	 *(file_id->TEXTPTR + file_id->GAPSTART) = 0;
	 if (file_id->filetype != FL_NEW) { /* if not new, save backups and remove old contents */
	    close(file_id->filedesc);
	    if (!savebackups(file_id,file_id->pathname)) unlink(file_id->pathname);
	    /* ^^backup didn't remove old version, so do it here^^ */
	    if ((file_id->filedesc = open(file_id->pathname,O_CREAT|O_WRONLY,file_id->rights)) < 0) {
	       UNPROTECT;
	       EDfile_display(file_id,"Unable to save file changes.Check /tmp and backups.");
	       return FALSE;
	     }
	  }
       };
      file_id->filedata->modified = 0;
      name = file_id->pathname;
    };

   otf = file_id->filedesc;
   file_id->filedesc = creat(name,file_id->rights);
   if (file_id->filedesc < 0) {
      file_id->filedesc = otf;
      UNPROTECT;
      EDfile_display(file_id,"Error in save to file");
      return FALSE;
    };

   fg = TRUE;

   FILE_movegapto(file_id,file_id->FILE_INDEX_END);
   *(file_id->TEXTPTR + file_id->GAPSTART) = 0;

   dptr = file_id->filedata;
   if (dptr->outfilters == NULL) {
      txt = malloc(strlen(file_id->TEXTPTR)+4);
      strcpy(txt,file_id->TEXTPTR);
      ln = tabify(txt);
      write(file_id->filedesc,txt,ln);
      SFREE(txt);
    }
   else {
      filter_io(file_id,OUTPUT);
    };

   close(file_id->filedesc);
   file_id->filedesc = otf;

   UNPROTECT;

   return fg;
}





/******************************************************************/
/*								  */
/*    FILEclose_allfiles - close all the files (save=TRUE) that   */
/*	 are currently open in the editor.			  */
/*								  */
/******************************************************************/


void
FILEclose_allfiles()
{
   FILE_ID  new,old;

   FILE_INIT;

   if (File_list == FILE_ID_NONE) return;

   old = File_list;
   for (old = File_list; old != FILE_ID_NONE && old != NULL; old = new) {
      new = old->nxtfile;
      FILEclose(old,TRUE);
    }
}





/******************************************************************/
/*								  */
/*    FILEset_backups - set the var. numbackups to the		  */
/*	  given value.						  */
/*								  */
/******************************************************************/


void
FILEset_backups(fid,count)
   FILE_ID fid;
   Integer count;
{
   FILE_INIT;

   fid->filedata->numbackups = count;
}





/******************************************************************/
/*								  */
/*    FILEalter_access - change the given file from readonly	  */
/*	  to read/write access, if possible.			  */
/*								  */
/******************************************************************/


int
FILEalter_access(fid)
   FILE_ID fid;
{
   Integer newfd;  /* file descriptor from new rw open */
   Boolean fg;

   FILE_INIT;

   if (fid->filetype != FL_READ) {
      EDfile_display(fid,"File is not read-only access. Unable to modify.");
      return(FALSE);
    }

   PROTECT;
   close(fid->filedesc);
   if ((newfd = open(fid->pathname,O_RDWR)) < 0) {
      EDfile_display(fid,"Unable to alter access. No write access.");
      if ((fid->filedesc = open(fid->pathname,O_RDONLY)) <= 0)
	 EDabort("Editing lost--unable to reopen.  Check /tmp");
      fg = FALSE;
    }
   else {
      EDfile_display(fid,"Access has been modified to r/w.");
      fid->filedesc = newfd;
      fid->filetype = FL_OK;
      fg = TRUE;
    }
   UNPROTECT;

   return fg;
}





/******************************************************************/
/*								  */
/*    FILEsecure - save a version of the given file in the	  */
/*	  /tmp directory in case the system crashes.		  */
/*								  */
/******************************************************************/


void
FILEsecure(fid)
   FILE_ID fid;
{
   Character fileptr[MAXSTRINGLENGTH];	    /* holds temporary path		     */
   Integer   file_desc; 		    /* file desc from temporary open	     */
   Character justfile[MAXSTRINGLENGTH];     /* only the file name from the abs path  */

   FILE_INIT;

   PROTECT;

   if (fid != FILE_ID_NONE) {
      if ((fid->filetype == FL_OK) || (fid->filetype == FL_NEW) ||
	     (fid->filetype == FL_TRUNC)) {
	 if (fid->filedata->modified != fid->filedata->lastsecure) {
	    filetail(fid->pathname,justfile);
	    sprintf(fileptr,"/tmp/BWE.%s",justfile);
	    if ((file_desc = open(fileptr,O_CREAT|O_TRUNC|O_RDWR,0666)) >= 0) { /* OK */
	       write(file_desc,fid->TEXTPTR,fid->GAPSTART);
	       write(file_desc,fid->TEXTPTR+fid->GAPEND,fid->ALLOCSIZE-fid->GAPEND);   /* get around the buffer gap */
	       close(file_desc);
	     }
	    else /* unable to secure the file, tough */;
	    fid->filedata->lastsecure = fid->filedata->modified;
	  };
       }
    }

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*     duplicate - take an existing file_id and "link" to it.     */
/*	  Utilize the same text core information, but this	  */
/*	  version will only be a copy (read-only).		  */
/*								  */
/******************************************************************/


static FILE_ID
duplicate(copiedfile,openflag)
   FILE_ID copiedfile;
   Integer openflag;
{
   FILE_ID file_id;	 /* new file_id */
   FILE_NODE temp;
   FILE_NODE p;

   PROTECT;
   file_id = PALLOC(FILE_ID_INFO);
   file_id->pathname = SALLOC(copiedfile->pathname);
   file_id->device = copiedfile->device;
   file_id->inode = copiedfile->inode;
   file_id->filetype = copiedfile->filetype;
   file_id->filedesc = -1;
   file_id->rights = 0666;
   file_id->pos = FILE_INDEX_START;

   file_id->filedata = copiedfile->filedata;		     /* copied data */
   ++file_id->filedata->usecount;

   temp = PALLOC(FILE_NODE_INFO);
   temp->id = file_id;
   temp->nxtfile = NULL;
   for (p=file_id->filedata->filelist; p->nxtfile != NULL; p = p->nxtfile) ;
   p->nxtfile = temp;		     /* add to the list of associated files */

   file_id->nxtfile = File_list;
   File_list = file_id;

   if (openflag & FILE_OPEN_READONLY) {
      if (file_id->filetype == FL_OK ||
	     file_id->filetype == FL_NEW ||
	     file_id->filetype == FL_TRUNC ||
	     file_id->filetype == FL_TEMP)
	 file_id->filetype = FL_READ;
    };

   UNPROTECT;

   return(file_id);
}



/******************************************************************/
/*								  */
/*     filedecide - given a file status, display an error message */
/*	   if the open failed, or call createfileid on a	  */
/*	   successful open.					  */
/*								  */
/******************************************************************/


static FILE_ID
filedecide(pathname,filestatus,open_flag,filedesc,filesize,device,inode,mode)
   String pathname;
   FL_STATUS filestatus;
   Integer open_flag;
   Integer filedesc;
   Integer filesize;
   Integer device;
   Integer inode;
   Integer mode;
{
   Integer allocsize;	  /* size allocated for file		     */
   Integer numblocks;	  /* number of file blocks currently in file */

   switch (filestatus) {
      case FL_OK      :
      case FL_READ    :
	 numblocks = filesize / FILEBLOCKSIZE;
	 allocsize = (numblocks + 5) * FILEBLOCKSIZE;
	 return createfileid(pathname,filedesc,filesize,allocsize,open_flag,
				filestatus,device,inode,mode);

      case FL_TRUNC   :
      case FL_NEW     :
      case FL_TEMP    :
	 allocsize = 5 * FILEBLOCKSIZE;
	 return createfileid(pathname,filedesc,0,allocsize,open_flag,
				filestatus,device,inode,mode);

      case FL_FAIL    :
	 EDfile_display(NULL,"Unable to open file\n");
	 return FILE_ID_NONE;

      case FL_BAD     :
	 EDfile_display(NULL,"Unable to edit block or char special file\n");
	 return FILE_ID_NONE;

      case FL_NOWRITE :
	 EDfile_display(NULL,"Unable to open. No write access for new file\n");
	 return FILE_ID_NONE;

      case FL_NOREAD  :
	 EDfile_display(NULL,"Unable to create readonly file which does not exist\n");
	 return FILE_ID_NONE;

      case FL_DIR     :
	 EDfile_display(NULL,"Unable to edit. Given file is a directory\n");
	 return FILE_ID_NONE;

      case FL_LOCK    :
	 EDfile_display(NULL,"File is locked for editing. Unable to open\n");
	 return FILE_ID_NONE;
    }

   /*NOTREACHED*/
}





/******************************************************************/
/*								  */
/*     createfileid - allocated space for a file header, and	  */
/*	  initialize all the fields appropriately.		  */
/*								  */
/******************************************************************/


static FILE_ID
createfileid(path,filedesc,size,allocsize,open_flag,filestatus,device,inode,mode)
   String path;
   Integer filedesc;
   Integer size;
   Integer allocsize;
   Integer open_flag;
   FL_STATUS filestatus;
   Integer device;
   Integer inode;
   Integer mode;
{
   FILE_ID file_id;		/* newly created file_id		    */
   Integer readsize;		/* size read from previous version of file  */
   Boolean noread;			/* true if no read from disk should occur   */
   String b;			/* result of ENVIRONMENT var. BACKUPS	    */
   String getenv();

   file_id = PALLOC(FILE_ID_INFO);
   file_id->pathname = SALLOC(path);
   file_id->filetype = filestatus;
   file_id->device = device;
   file_id->inode = inode;
   file_id->filedesc = filedesc;
   file_id->rights = mode;
   file_id->pos = FILE_INDEX_START;

   PROTECT;
   file_id->nxtfile = File_list;
   File_list = file_id;
   UNPROTECT;

   file_id->filedata = PALLOC(FILE_DATA_INFO);

   file_id->filedata->filelist = PALLOC(FILE_NODE_INFO);
   file_id->filedata->filelist->id = file_id;
   file_id->filedata->filelist->nxtfile = NULL;

   file_id->filedata->modified = 0;
   file_id->filedata->lastsecure = 0;
   if ((b = getenv("BACKUPS")) == NULL)
      file_id->filedata->numbackups = 1;
   else if ((file_id->filedata->numbackups = atoi(b)) < 0)
      file_id->filedata->numbackups = 1;
   file_id->filedata->size = size;
   file_id->filedata->allocsize = allocsize;
   file_id->filedata->currpos.type = REG;
   file_id->filedata->currpos.line = 1;
   file_id->filedata->currpos.ch = 1;
   file_id->filedata->currpos.virt = FALSE;
   file_id->filedata->currpos.index = FILE_INDEX_START;
   file_id->filedata->currpos.modcnt = 0;
   file_id->filedata->marklist = NULL;
   file_id->filedata->infilters = NULL;
   file_id->filedata->outfilters = NULL;
   file_id->filedata->regmsg = NULL;
   file_id->filedata->lastvalid = FALSE;
   file_id->filedata->ignerror = FALSE;

   file_id->filedata->hist_mode = HIST_MODE_NORMAL;
   file_id->filedata->undo_history = PALLOC(FILE_HIST_INFO);
   file_id->filedata->undo_history->file = file_id;
   file_id->filedata->undo_history->action = HIST_COMMAND;
   file_id->filedata->undo_history->nxt_history = NULL;
   file_id->filedata->redo_history = NULL;
   file_id->filedata->usecount = 1;

   if ((file_id->filedata->filetext = (String) malloc(allocsize+1)) == NULL) {	/* allocate working space      */
      EDfile_display(NULL,"Not enough core to open file");
      PROTECT;
      File_list = File_list->nxtfile;
      UNPROTECT;
      free_filedata(file_id->filedata);
      free_fileid(file_id);
      return(FILE_ID_NONE);
    }
   file_id->filedata->filetext[allocsize] = 0;

   noread = ( (open_flag & FILE_OPEN_HOLD) || (filestatus == FL_NEW) || (filestatus == FL_TRUNC) || (filestatus == FL_TEMP) );

   if (!noread) {		    /* read file from disk     */
      readsize = read(filedesc,(file_id->filedata->filetext)+(allocsize-size),size);
      if (readsize != size) {
	 EDfile_display(NULL,"Error in open/read--existing file was not read in");
	 PROTECT;
	 File_list = File_list->nxtfile;
	 UNPROTECT;
	 free_filedata(file_id->filedata);
	 free_fileid(file_id);
	 return(FILE_ID_NONE);
       }
      if (expand_tabs(&(file_id->filedata->filetext),&(file_id->filedata->size),
			   &(file_id->filedata->allocsize),
			 &(file_id->filedata->endline))) {
	 file_id->filedata->startregion = FILE_INDEX_START;
	 file_id->filedata->endregion = file_id->filedata->allocsize;
	 file_id->filedata->nullregion = FALSE;
	 file_id->filedata->gapstart = file_id->filedata->size;
	 file_id->filedata->gapend = file_id->filedata->allocsize;
	 file_id->filedata->filetext[file_id->filedata->allocsize] = 0;
       }
      else {  /* wasn't enough space in expanding tabs */
	 EDfile_display(NULL,"Not enough core to open file");
	 PROTECT;
	 File_list = File_list->nxtfile;
	 UNPROTECT;
	 free_filedata(file_id->filedata);
	 free_fileid(file_id);
	 return(FILE_ID_NONE);
       }
    }
   else {     /* there was no read from disk */
      file_id->filedata->size = size;
      file_id->filedata->allocsize = allocsize;
      file_id->filedata->endline = 1;
      file_id->filedata->startregion = FILE_INDEX_START;
      file_id->filedata->endregion = allocsize;
      file_id->filedata->nullregion = FALSE;
      file_id->filedata->gapstart = 0;
      file_id->filedata->gapend = allocsize;
      file_id->filedata->filetext[0] = 0;
    }

   return(file_id);
}



/******************************************************************/
/*								  */
/*    expand_tabs - this routine takes the malloced core area with*/
/*	the file at the rear and the gap at the front, and it	  */
/*	expands tab (\t) characters into 8 (or less) spaces.  It  */
/*	counts the number of lines in the file.  Basically, chars */
/*	are moved back to front in this core area.  If the gap	  */
/*	shrinks too small, a larger region is reallocated.  The   */
/*	new text ptr, size, allocsize and numlines are returned.  */
/*								  */
/******************************************************************/


static Boolean
expand_tabs(file_text,file_size,file_alloc_size,file_lines)
   char **file_text;
   int *file_size,*file_alloc_size,*file_lines;
{
   Character *text;		   /* ptr to start of core area 		 */
   Character *end_text;    /* ptr to end of core area			 */
   Integer   lines = 1;        /* number of lines in the file		     */
   Character  ch;		   /* character being examined			 */
   Integer   col = 0;	       /* column of line currently at		     */
   Integer   padding;	       /* # spaces that the tab filled		     */
   Character *gap_start;	   /* ptr to area where expanded char(s) placed  */
   Character *gap_end;	   /* ptr to area where new char is checked	 */
   Integer   alloc_size;       /* allocated size of core area		     */
   Integer   new_alloc_size;   /* reallocated area size 		     */
   Character *new_text;    /* ptr to reallocated area			 */
   Character *new_tail;    /* these two used in moving chars during a	 */
   Character *old_tail;    /* reallocation (coalesce two gaps)		 */


   alloc_size = *file_alloc_size;		  /* file comes in with gap at start */
   text = gap_start = *file_text;		  /*  |<  gap  >sometext|	     */
   gap_end = text + (alloc_size - *file_size);
   end_text = text + alloc_size;

   while (gap_end < end_text) {       /* work your way through text, moving chars up to */
      if ((ch = *gap_end) == '\n') { /* beginning of allocated area */
	 ++lines;
	 col = 0;
	 *(gap_start++) = '\n';
       }
      else if (ch == '\t') {
	 padding = 8 - (col & 07);  /* # of blanks to insert for tab */
	 while (padding-- > 0) *gap_start++ = ' ';
	 col += 8;		    /* increment the column by the spaces */
	 col &= ~07;		    /* in the tab			  */
       }
      else {
	 *(gap_start++) = ch;
	 col++;
       }
      ++gap_end;     /* taken one char away from end text of file */
      if (gap_end < gap_start + 8) {  /* gap has shrunk too small */
	 new_alloc_size = alloc_size+(5*FILEBLOCKSIZE);
	 new_text = (String) realloc(text,new_alloc_size+1);
	 if (new_text) {				     /* reallocation */
	    new_tail = new_text + new_alloc_size - 1;
	    old_tail = text + alloc_size - 1;

	    while (old_tail >= gap_end) 	/* move chars behind the gap back into this */
	       *(new_tail--) = *(old_tail--);	/* new extra space */

	    gap_start = gap_start - text + new_text;
	    gap_end = new_tail + 1;
	    text = new_text;			/* readjust values to new allocate area */
	    end_text = text + new_alloc_size;
	    alloc_size = new_alloc_size;
	  }
	 else
	    return(FALSE);
       }
    }

   *file_text = text;
   *file_lines = lines;
   *file_size = gap_start - text;
   *file_alloc_size = alloc_size;

   return(TRUE);
}




/******************************************************************/
/*								  */
/*    tabify - this routine prepares a file to be written to disk.*/
/*	 Spaces at the beginnings of lines are converted into	  */
/*	 tab characters.					  */
/*								  */
/******************************************************************/


static Integer
tabify(txt)
   String txt;
{
   Character *oldtext;	  /* ptr that moves (before tabification    */
   Character *newtext;	  /* ptr that moves (after tabification     */
   Integer   spaces;	 /* count on # spaces at beginning of line */
   Boolean  line_end;	/* when TRUE, stops looking at that line  */

   oldtext = newtext = txt;

   while (oldtext != NULL && *oldtext != 0) {
      line_end = FALSE;
      spaces = 0;
      while (oldtext != NULL && *oldtext != 0) {
	 switch (*oldtext) {	/* check old character */
	    case ' ':
	       spaces++;	 /* increment spaces at beg of line */
	       oldtext++;
	       break;
	    case '\n':               /* line is all spaces, so just */
	       *newtext++ = '\n';    /* remove them and put in a    */
	       oldtext++;	     /* newline character	    */
	       line_end = TRUE;
	       break;
	    case 0 :
	       *newtext++ = '\n';    /* remove them and put in a    */
	       oldtext = NULL;
	       line_end = TRUE;
	       break;
	    default:
	       while (spaces >= 8) {	/* for every batch of 8 spaces */
		  *newtext++ = '\t';   /* put in a tab character      */
		  spaces -= 8;
		}
	       while (spaces-- > 0) {	/* add the straggler spaces */
		  *newtext++ = ' ';
		}
	       *newtext++ = *oldtext++;   /* finish off the rest of the line */
	       while (*oldtext != '\n' && *oldtext != 0) {
		  *newtext++ = *oldtext++;
		}
	       if (*oldtext == '\n') *newtext++ = *oldtext++;
	       else {
		  *newtext++ = '\n';
		  oldtext = NULL;
		};
	       line_end = TRUE;
	       break;
	  } /* switch */
	 if (line_end) break;
       } /* for */
      if (oldtext == NULL || *oldtext == 0) break;
    } /* while */
   *newtext = 0;

   return (newtext - txt);
}




/******************************************************************/
/*								  */
/*    cancelfile - changes all occurrences of given file within   */
/*	 the file stack to FILE_ID_NONE, and remove the file	  */
/*	 from the list of files associated with its text.	  */
/*	 Free up as much space as possible.			  */
/*								  */
/******************************************************************/


static void
cancel_file(file_id)
   FILE_ID file_id;
{
   FILE_NODE temp;
   FILE_NODE old,new;
   FILE_ID temp_id;

   FILE_regmsg(file_id,FILE_MSG_CLOSE,0,0,0,0);       /* close up the text display */

   PROTECT;

   if (file_id->filedata->filelist->id == file_id) {
      temp = file_id->filedata->filelist;
      file_id->filedata->filelist = file_id->filedata->filelist->nxtfile;
    }
   else {
      old = file_id->filedata->filelist;	     /* remove this file from the filelist on this filedata */
      new = old->nxtfile;
      while (new->id != file_id) {
	 if (new->id == NULL) EDabort("Error in closing up a file in cancel_file");
	 old = new;
	 new = new->nxtfile;
       }
      temp = new;
      old->nxtfile = new->nxtfile;
    }
   free(temp);

   if (file_id->filedata->filelist == NULL)    /* if no other files on this filedata, free it up */
      free_filedata(file_id->filedata);

   if (File_list == file_id)
      File_list = file_id->nxtfile;	      /* remove this file_id from the active file_id list */
   else {
      for (temp_id = File_list; temp_id->nxtfile != file_id; temp_id = temp_id->nxtfile) ;
      temp_id->nxtfile = temp_id->nxtfile->nxtfile;
    }

   UNPROTECT;

   free_fileid(file_id);
}



/******************************************************************/
/*								  */
/*    free_filedata - free a FILE_DATA structure, including all   */
/*	  the subfields which themselves may be lists or text.	  */
/*								  */
/******************************************************************/


static void
free_filedata(fdp)
   FILE_DATA_PTR fdp;
{
   FILTER filter;
   REG_MSG msg_ptr;
   MARK mark_ptr;

   if (fdp->filetext)
      free(fdp->filetext);   /* actual text (buffer) of a file */

   if (fdp->undo_history != NULL) {
      FILEfree_history(fdp->undo_history);   /* undo history nodes */
      free(fdp->undo_history);
    }

   if (fdp->redo_history != NULL) {
      FILEfree_history(fdp->redo_history);   /* redo history nodes */
      free(fdp->redo_history);
    }

   while ((filter = fdp->infilters) != NULL) {
      fdp->infilters = filter->nxtfilter;
      free(filter);
    }

   while ((filter = fdp->outfilters) != NULL) {
      fdp->outfilters = filter->nxtfilter;
      free(filter);
    }

   while ((msg_ptr = fdp->regmsg) != NULL) {   /* associated msg routines to call */
      fdp->regmsg = msg_ptr->nxtregmsg;
      free(msg_ptr);
    }

   while ((mark_ptr = fdp->marklist) != NULL) {
      fdp->marklist = mark_ptr->nxtfilemark;
      free(mark_ptr);
    }

   free(fdp);	/* free the structure itself */
}




/******************************************************************/
/*								  */
/*    free_fileid - free up a FILE_ID structure, including all	  */
/*	  the subfields which themselves may be lists or text.	  */
/*								  */
/******************************************************************/


static void
free_fileid(fid)
   FILE_ID fid;
{
   free(fid->pathname);
   free(fid);
}





/******************************************************************/
/*								  */
/*    filter_io - utilize filters for either input or output to   */
/*	  a file.						  */
/*								  */
/******************************************************************/


static void
filter_io(fid,ioflag)
   FILE_ID fid;
   Set ioflag;
{
   String  oldstring;	  /* string passed to the filters		    */
   String    oldtext;	  /* the old file text				    */
   String  newstring;	  /* string returned from the filters		    */
   Integer    newidx = 0;    /* index into the new text 			    */
   Integer    allocsize;
   Integer    size;			/* quick access to Currfile->filedata->size	  */
   FILTER ptr;			    /* is Currfile->filedata-> (in or out)filters     */
   FILTER p;			    /* traverses list of filters		      */

   size = fid->filedata->size;

   if (ioflag == INPUT) {
      if (fid->filedesc <= 0) return;
      oldtext = (String) malloc(size + 1);
      if (read(fid->filedesc,oldtext,size) != size)
	 EDabort("Error in file read in filter_io. Unable to open.");
      oldtext[size] = 0;
    }
   else {
      oldtext = fid->TEXTPTR;
      oldtext[size] = 0;
    };

   if (ioflag == INPUT)
      ptr = fid->filedata->infilters;
   else
      ptr = fid->filedata->outfilters;

   oldstring = oldtext;
   for (p = ptr; p != NULL; p = p->nxtfilter) {
      newstring = (String) (*(p->fct))(fid,oldstring);
      if (newstring != oldstring && oldstring != oldtext) free(oldstring);
      oldstring = newstring;
    };

   newidx = strlen(oldstring);
   if (ioflag == INPUT) {
      allocsize = newidx + 5*FILEBLOCKSIZE;
      newstring = (String) malloc(allocsize+1);
      newstring[allocsize] = 0;
      strcpy(&newstring[5*FILEBLOCKSIZE],oldstring);
      free(oldstring);
      if (fid->TEXTPTR != NULL) free(fid->TEXTPTR);
      fid->TEXTPTR = newstring;
      fid->SIZE = newidx;
      fid->GAPSTART = 0;
      fid->GAPEND = 5*FILEBLOCKSIZE;
      fid->ALLOCSIZE = fid->filedata->endregion = allocsize;
    }
   else {
      if (oldstring == oldtext) {
	 oldstring = SALLOC(oldtext);
       };
      newidx = tabify(oldstring);
      if (write(fid->filedesc,oldstring,newidx) != newidx)
	 EDfile_display(fid,"Filter error in file write. Changes have been lost. Check /tmp");
      if (oldstring != oldtext) free(oldstring);
    };
};




/******************************************************************/
/*								  */
/*    savebackups - save a backup of the given file.  The number  */
/*	 of backups maintained may be altered by FILEset_backups. */
/*								  */
/******************************************************************/


static Boolean
savebackups(fid,path)
   FILE_ID fid;
   String path;
{
   Character BUdir[MAXSTRINGLENGTH];	 /* back up directory			  */
   Character pathdir[MAXSTRINGLENGTH];	 /* given path's directory                     */
   Character no_name[MAXSTRINGLENGTH];	 /* given path with no filename 	  */
   Character BUname[MAXSTRINGLENGTH];	 /* file name of backup 		  */
   Character justfile[MAXSTRINGLENGTH];  /* just the file name of the given path	  */
   Integer i;
   String getenv();			     /* number got from call to detect environ var */
   Character file[MAXSTRINGLENGTH];	    /* temporary names for accessing backup files */
   Character newfile[MAXSTRINGLENGTH];

   remove_name(path,no_name);
   sprintf(BUdir,"%sbBACKUP",no_name);

   if (access(BUdir,F_OK) < 0)	/* dir doesn't exist */
      mkdir(BUdir,0776);	 /* so try to make it */
   if (access(BUdir,X_OK|W_OK|R_OK) >= 0)    /* success */
      /* do nothing */ ;
   else {			/* Backup dir failed, try current dir */
      EDfile_display(fid,"Can't access Backup directory; trying file's directory");
      sleep(2);
      dir(path,pathdir);
      if (access(pathdir,W_OK|R_OK) >= 0) { /* success */
	 sprintf(BUdir,"%s",pathdir);
       }
      else { /* can't save in file's directory */
	 EDfile_display(fid,"Can't save there; trying home Backup");
	 sleep(2);
	 sprintf(BUdir,"%s/bBACKUP",getenv("HOME"));
	 if (access(BUdir,F_OK) < 0)  /* dir doesn't exist */
	    mkdir(BUdir,0776);	       /* so try to make it */
	 if (access(BUdir,X_OK|W_OK|R_OK) >= 0)    /* success */
	    /* do nothing */ ;
	 else { /* all failed */
	    EDfile_display(fid,"Can't access home Backup directory");
	    sleep(2);
	    EDfile_display(fid,"No backup saved");
	    return(FALSE);
	  }
       }
    }

   filetail(path,justfile);
   sprintf(BUname,"%s/%s",BUdir,justfile);

   for (i=fid->filedata->numbackups; ; ++i) {	      /* remove old versions >= numbackups */
      sprintf(file,"%s.%d",BUname,i);
      if (access(file,F_OK) == 0)     /* the file exists */
	 unlink(file);
      else
	 break;
    }

   for (i=fid->filedata->numbackups - 1; i>0; --i) {	/* move a backup to 1 older version */
      sprintf(file,"%s.%d",BUname,i);
      if (access(file,F_OK) == 0) {   /* the file exists */
	 sprintf(newfile,"%s.%d",BUname,i+1);
	 rename(file,newfile);
       }
    }
   sprintf(newfile,"%s.1",BUname);   /* save most recent version */
   rename(path,newfile);

   return(TRUE);
}





/******************************************************************/
/*								  */
/*    dir - return the directory of the given pathname. 	  */
/*								  */
/******************************************************************/


static void
dir(path,dirr)
   String path; 	 /* given path		 */
   Character dirr[];	 /* place result in here */
{
   Integer i,j;

   for (i=strlen(path)-1; (i >= 0) && (*(path + i) != '/') ; --i);

   if (i == -1) {
      dirr[0] = '.';     /* current directory */
      dirr[1] = '\0';
      return;
    }

   if (i == 0) {	  /* path looks like `/somename' */
      dirr[0] = '/';
      dirr[1] = '\0';
      return;
    }

   for (j=i-1; j>=0; --j) dirr[j] = *(path + j);
   dirr[i] = '\0';
}





/******************************************************************/
/*								  */
/*    remove_name - strip off the trailing filename at the end of */
/*	the given pathname.					  */
/*								  */
/******************************************************************/


static void
remove_name(path,dirr)
   String path;      /* given path	     */
   Character dirr[];	 /* place result in here */
{
   Integer i,j;

   for (i=strlen(path)-1; (i >= 0) && (*(path + i) != '/') ; --i);

   if (i == -1) {
      dirr[0] = '\0';    /* current directory */
      return;
    }

   for (j=i; j>=0; --j) dirr[j] = *(path + j);
   dirr[i+1] = '\0';
}





/******************************************************************/
/*								  */
/*    filetail - return the file name from the end of a path	  */
/*								  */
/******************************************************************/


static void
filetail(path,name)
   String path;     /* given path		      */
   Character name[];   /* place to deposit returned name */
{
   Integer i,j;

   for (i=strlen(path)-1; (i >= 0) && (*(path + i) != '/'); --i);

   for (j=i+1; j<=strlen(path); ++j)
      name[j - i - 1] = *(path + j);
}






/* end of filefiles.c */
