/******************************************************************/
/*								  */
/*		fileedit.c					  */
/*								  */
/*	Editing routines for the bwe editor			  */
/*								  */
/******************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"





/******************************************************************/
/*								  */
/*	Forward definitions					  */
/*								  */
/******************************************************************/


static	void		make_currpos_real();
static	void		roomfor();
static	Boolean 	withinregion();
static	Boolean 	withinlimits();
static	String		copy_text();
static	void		limits_error();





/******************************************************************/
/*								  */
/*    FILEinsert - insert the contents of the given buffer at	  */
/*	  the current position of the given file count times.	  */
/*								  */
/******************************************************************/


void
FILEinsert(fid,buf,count)
   FILE_ID fid;
   FILE_BUF buf;
   Integer count;
{
   Integer buf_size;	 /* the number of characters in the given buffer */
   Integer added_chars;  /* the total number of characters inserted	 */
   Integer ins_lines;	 /* the number of lines added to the file	      */
   FILE_POS pos2;     /* the pos at the end of the inserted text      */
   String buf_start,buf_ptr;
   Character ch;
   String text_ptr;
   ABSOL_INDEX gap_start;

   FILE_INIT;

   if (!(withinlimits(fid,fid->POINT))) {
      limits_error(fid);
      return;
    }

   if (buf == NULL) {
      EDfile_display(fid,"Attempting to insert with bad buffer");
      return;
    }

   if (count <= 0) return;
   if (fid->filetype == FL_READ) {
      EDfile_display(fid,"File is read-only");
      return;
    };


   PROTECT;

   make_currpos_real(fid);

   buf_start = FILEbuffer_contents(buf);
   if (buf_start == NULL) buf_size = 0;
   buf_size = strlen(buf_start);

   roomfor(fid,added_chars = count * buf_size);
   FILE_movegapto(fid,fid->POINT);
   text_ptr = fid->TEXTPTR;
   gap_start = fid->GAPSTART;
   ins_lines = 0;
   if (buf_start != NULL) {
      while (count != 0) {
	 buf_ptr = buf_start;
	 while ( ch = *buf_ptr++ ) {	  /* insert by single char input   */
	    *(text_ptr + gap_start) = ch;
	    if (ch == NEWLINE) ins_lines++;
	    gap_start++;
	  }
	 count--;
       }
      free(buf_start);
    }

   fid->ENDLINE += ins_lines;
   fid->SIZE += added_chars;
   fid->GAPSTART = gap_start;
   fid->filedata->lastvalid = FALSE;

   pos2.type = REG;
   pos2.index = fid->GAPSTART;
   pos2.virt = FALSE;
   pos2.line = fid->CURRLINE + ins_lines;
   pos2.ch = FILE_inq_char(fid,fid->GAPSTART);
   pos2.modcnt = fid->filedata->modified;

   FILE_set_history(fid,HIST_INSERT,&(fid->CURRPOS),&pos2,0,0,0,0);
   fid->filedata->modified++;
   pos2.modcnt = fid->filedata->modified;
   fid->CURRMOD = fid->filedata->modified;

   UNPROTECT;

   FILE_regmsg(fid,FILE_MSG_INSERT,&(fid->CURRPOS),&pos2,0,0);	/* register insertion message */
}




/******************************************************************/
/*								  */
/*    FILEdelete - delete the text from the current position	  */
/*	  to the given relative position (as in FILEmove).	  */
/*	  Place the deleted text in the designated buffer.	  */
/*								  */
/******************************************************************/


void
FILEdelete(fid,buf,delta,value)
   FILE_ID fid;
   FILE_BUF buf;
   Integer delta;
   Integer value;
{
   FILE_POS pos2;	  /* position specified by (delta,value) */
   FILE_POS temp;	  /* used to exchange pos2 and CURRPOS	 */
   String   del_text;	  /* the deleted text			 */
   ABSOL_INDEX start;	  /* the start of the edit region	 */
   ABSOL_INDEX end;	  /* the end of the edit region 	 */
   Boolean fg;

   FILE_INIT;

   if (buf == NULL) {
      EDfile_display(fid,"Attempting to delete with bad buffer");
      return;
    }
   if (fid->filetype == FL_READ) {
      EDfile_display(fid,"File is read-only");
      return;
    };

   if (fid->filedata->nullregion) {
      limits_error(fid);
      return;
    };

   FILEinq_position(fid,delta,value,&pos2);

   PROTECT;

   FILE_position_real(fid,&pos2);

   start = fid->filedata->startregion;			      /* make CURRPOS and pos2	   */
   end = fid->filedata->endregion;			      /* within edit limits with   */
   if (fixed_absolute(fid,fid->POINT) < start) {	      /* CURRPOS coming first	   */
      if (fixed_absolute(fid,pos2.index) < start) {
	 UNPROTECT;
	 limits_error(fid);
	 return;
       }
      else if (end < absolute(fid,pos2.index))
	 FILE_convert(fid,FILE_relative(fid,end),&pos2);
      FILE_convert(fid,FILE_relative(fid,start),&(fid->CURRPOS));
    }
   else if (end < absolute(fid,fid->POINT)) {
      if (fixed_absolute(fid,pos2.index) < start)
	 FILE_convert(fid,FILE_relative(fid,start), &(fid->CURRPOS));
      else if (end < absolute(fid,pos2.index)) {
	 UNPROTECT;
	 limits_error(fid);
	 return;
       }
      else CopyPos(fid,&pos2,&(fid->CURRPOS));
      FILE_convert(fid,FILE_relative(fid,end),&pos2);
    }
   else {
      if (fixed_absolute(fid,pos2.index) < start) {
	 CopyPos(fid,&(fid->CURRPOS),&pos2);
	 FILE_convert(fid,FILE_relative(fid,start),&(fid->CURRPOS));
       }
      else if (end < absolute(fid,pos2.index))
	 FILE_convert(fid,FILE_relative(fid,end),&pos2);
      else if (pos2.index <= fid->POINT) {
	 CopyPos(fid,&pos2,&temp);
	 CopyPos(fid,&(fid->CURRPOS),&pos2);
	 CopyPos(fid,&temp,&(fid->CURRPOS));
       }
    }

   if (fid->CURRPOS.virt)			     /* if CURRPOS is virtual	  */
      FILE_convert(fid,fid->POINT,&(fid->CURRPOS));   /*   move to last real char  */

   del_text = copy_text(fid,fid->POINT,pos2.index);

   FILE_set_history(fid,HIST_BUFFER,0,0,buf,0,0,0);

   if ( FILEbuffer_define(buf,del_text) ) {	     /* copy the text to buffer   */
      FILE_movegapto(fid,pos2.index);
      fid->GAPSTART = fid->POINT;
      fid->SIZE -= (pos2.index - fid->POINT);	    /* swallow text into the gap */
      if (pos2.line < fid->ENDLINE)
	 fid->ENDLINE -= (pos2.line - fid->CURRLINE);
      else
	 fid->ENDLINE = fid->CURRLINE;
      FILE_remove_marks(fid,fid->GAPSTART,fid->GAPEND);   /* remove swallowed marks    */

      fid->filedata->lastvalid = FALSE;
      FILE_set_history(fid,HIST_DELETE,&(fid->CURRPOS),0,0,0,del_text,0);
      fid->filedata->modified++;
      fid->CURRMOD = fid->filedata->modified;
      pos2.modcnt = fid->filedata->modified;
      fg = TRUE;
    }
   else fg = FALSE;

   UNPROTECT;

   if (fg) {
      FILE_regmsg(fid,FILE_MSG_DELETE,&(fid->CURRPOS),&pos2,0,0);   /* register deletion message */
    };

   free(del_text);
}





/******************************************************************/
/*								  */
/*    FILEpick - place the given text in the given buffer	  */
/*	  (no deletion).					  */
/*								  */
/******************************************************************/


void
FILEpick(fid,buf,delta,value)
   FILE_ID fid;
   FILE_BUF buf;
   Integer delta;
   Integer value;
{
   FILE_POS inq_pos;   /* position specified by (delta,value) */
   FILE_INDEX idx;

   FILE_INIT;

   if (buf == NULL) {
      EDfile_display(fid,"Attempting to pick with bad buffer");
      return;
    }

   FILEinq_position(fid,delta,value,&inq_pos);

   PROTECT;

   FILE_position_real(fid,&inq_pos);
   idx = inq_pos.index;

   if (idx != fid->POINT) {
      FILE_set_history(fid,HIST_BUFFER,0,0,buf,0,0,0);
      if (fid->POINT < idx)
	 FILEinq_text(fid,&(fid->CURRPOS),&inq_pos,buf);
      else  /* idx < POINT */
	 FILEinq_text(fid,&inq_pos,&(fid->CURRPOS),buf);
    };

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    FILEtypein - place the given character at the current	  */
/*	  file position.					  */
/*								  */
/******************************************************************/


void
FILEtypein(fid,ch,insert_mode)
   FILE_ID fid;
   Character ch;
   Set insert_mode;
{
   FILE_POS	oldpos;        /* position at start of typein */
   FILE_POS	currpos;       /* contains currpos after move */
   HIST_ACTION	hs_action;     /* used to save undo history   */
   Character	hs_char = 0;

   FILE_INIT;

   if (!withinlimits(fid,fid->POINT)) {
      limits_error(fid);
      return;
    }
   if (fid->filetype == FL_READ) {
      EDfile_display(fid,"File is read-only");
      return;
    };

   PROTECT;

   make_currpos_real(fid);
   CopyPos(fid,&(fid->CURRPOS),&oldpos);

   if (ch == BACKSPACE) {
      if ((fid->POINT == FILE_INDEX_START) || (charAt(fid,fid->POINT-1) == NEWLINE) ||
	     !withinlimits(fid,fid->POINT-1)) {
	 UNPROTECT;
	 return;
       };
    };

   if (insert_mode) {
      if (ch == BACKSPACE) {
	 FILE_movegapto(fid,fid->POINT);
	 fid->CURRCHAR--;
	 fid->POINT--;
	 hs_action = HIST_BACKSPACE;
	 hs_char = charAt(fid,fid->POINT);
	 fid->GAPSTART--;
	 fid->SIZE--;
	 FILE_remove_marks(fid,fid->GAPSTART,fid->GAPEND);
       }
      else { /* any other char */
	 roomfor(fid,1);
	 hs_action = HIST_TYPEIN;
	 FILE_movegapto(fid,fid->POINT);
	 *(fid->TEXTPTR + fid->GAPSTART) = ch;
	 if (ch == NEWLINE) {
	    fid->ENDLINE++;
	    fid->CURRLINE++;
	    fid->CURRCHAR = 0;
	    hs_action = HIST_NEWLINE;
	  }
	 fid->CURRCHAR++;
	 fid->POINT++;
	 fid->SIZE++;
	 fid->GAPSTART++;
       }
    }
   else { /*  non-insert mode */
      if (ch == BACKSPACE) {
	 fid->CURRCHAR--;
	 fid->POINT--;
	 hs_action = HIST_BACKSPACE;
	 hs_char = charAt(fid,fid->POINT);
	 *(fid->TEXTPTR + absolute(fid,fid->POINT)) = SPACE;
       }
      else if (ch == NEWLINE) {
	 if (fid->POINT == fid->FILE_INDEX_END) {
	    roomfor(fid,1);
	    hs_action = HIST_NEWLINE;
	    insert_mode = TRUE;
	    FILE_movegapto(fid,fid->POINT);
	    *(fid->TEXTPTR + fid->GAPSTART) = ch;
	    fid->ENDLINE++;
	    fid->CURRLINE++;
	    fid->CURRCHAR = 1;
	    fid->POINT++;
	    fid->SIZE++;
	    fid->GAPSTART++;
	  }
	 else {
	    FILEmove(fid,FILE_MOVE_LINE | FILE_MOVE_LINE_START,1,&currpos);
	    FILE_regmsg(fid,FILE_MSG_TYPEIN,&oldpos,0,ch,insert_mode);
	    return;
	  };
       }
      else { /* any other char */
	 hs_action = HIST_TYPEIN;
	 if ((fid->POINT == fid->FILE_INDEX_END) || (charAt(fid,fid->POINT) == NEWLINE)) {
	    roomfor(fid,1);
	    insert_mode = TRUE;
	    FILE_movegapto(fid,fid->POINT);
	    *(fid->TEXTPTR + fid->GAPSTART) = ch;
	    fid->GAPSTART++;
	    fid->SIZE++;
	  }
	 else {
	    hs_char = charAt(fid,fid->POINT);
	    *(fid->TEXTPTR + absolute(fid,fid->POINT)) = ch;	      /* set char at POINT to ch       */
	  }
	 fid->CURRCHAR++;
	 fid->POINT++;
       }
    };

   FILE_set_history(fid,hs_action,&oldpos,&(fid->CURRPOS),0,hs_char,0,insert_mode);
   fid->filedata->modified++;
   fid->CURRMOD = fid->filedata->modified;
   oldpos.modcnt = fid->filedata->modified;
   fid->filedata->lastvalid = FALSE;
   UNPROTECT;

   FILE_regmsg(fid,FILE_MSG_TYPEIN,&oldpos,0,ch,insert_mode);
}



/******************************************************************/
/*								  */
/*    FILEedit_limits - define the region in which editing can	  */
/*	  be done in the given file.				  */
/*								  */
/******************************************************************/


void
FILEedit_limits(fid,from,to)
   FILE_ID fid;
   FILE_POS  *from,*to;
{
   FILE_POS   old_from,old_to;
   FILE_INDEX from_index,to_index;

   FILE_INIT;

   PROTECT;

   FILE_convert(fid,FILE_relative(fid,fid->filedata->startregion),&old_from);
   FILE_convert(fid,FILE_relative(fid,fid->filedata->endregion),&old_to);

   FILE_position(fid,from);
   FILE_position(fid,to);

   from_index = from->index;
   to_index = to->index;
   fid->filedata->nullregion = FALSE;

   if (from_index == FILE_INDEX_NONE) from_index = FILE_INDEX_START;
   if (to_index == FILE_INDEX_NONE) to_index = fid->FILE_INDEX_END;

   if (from == FILE_POS_END && to == FILE_POS_START) {
      fid->filedata->nullregion = TRUE;
    }
   else if (from_index <= to_index) {
      fid->filedata->startregion = fixed_absolute(fid,from_index);
      fid->filedata->endregion = absolute(fid,to_index);
      FILE_set_history(fid,HIST_LIMITS,&old_from,&old_to,0,0,0,0);
    }
   else {
      fid->filedata->nullregion = TRUE;
    };

   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    FILE_position_real - if position is virtual, make it real   */
/*								  */
/******************************************************************/


void
FILE_position_real(fid,pos)
   FILE_ID fid;
   FILE_POS *pos;
{
   FILE_POS fp2;
   Integer ln,cl;

   FILE_position(fid,pos);

   if (!(pos->virt)) return;

   PROTECT;
   ln = FILEinq_currline(fid);
   cl = FILEinq_currchar(fid);

   FILEmove(fid,FILE_MOVE_ABS,pos,&fp2);
   make_currpos_real(fid);
   FILEinq_position(fid,FILE_MOVE_REL,0,pos);

   FILEset_position(fid,ln,cl,&fp2);
   UNPROTECT;
}





/******************************************************************/
/*								  */
/*    FILE_movegapto - move the gap to the given position with a  */
/*	possible shuffle of characters. 			  */
/*								  */
/******************************************************************/


void
FILE_movegapto(fid,pos)
   FILE_ID fid;
   FILE_INDEX pos;
{
   ABSOL_INDEX idx;
   ABSOL_INDEX abs_pos;
   ABSOL_INDEX old_gs,old_ge;
   String  text_ptr = fid->TEXTPTR;
   Integer  gap_size = fid->GAPEND - fid->GAPSTART;

   if (pos == fid->GAPSTART) return;	      /* gap already there	*/

   old_gs = fid->GAPSTART;
   old_ge = fid->GAPEND;
   if (pos < fid->GAPSTART) {
      idx = fid->GAPSTART;		   /*  |   P  GS-------GE  | */
      abs_pos = absolute(fid,pos);
      while (idx != abs_pos) {
	 idx--;
	 *(text_ptr + idx + gap_size) = *(text_ptr + idx);
       }
    }
   else { /* pos > GAPSTART */		 /*  |	GS------GE   P	 | */
      idx = fid->GAPEND;
      abs_pos = absolute(fid,pos);
      while (idx != abs_pos) {
	 *(text_ptr + idx - gap_size) = *(text_ptr + idx);
	 idx++;
       }
    }
   fid->GAPEND = pos + fid->GAPEND - fid->GAPSTART;
   fid->GAPSTART = pos;

   if (fid->GAPSTART < old_gs)
      FILE_adjust_marks(fid,fid->GAPSTART,old_gs,fid->GAPEND - fid->GAPSTART);
   else /* olg_ge < GAPEND */
      FILE_adjust_marks(fid,old_ge,fid->GAPEND,fid->GAPSTART - fid->GAPEND /* -GAPSIZE */);
}





/******************************************************************/
/*								  */
/*   FILE_regmsg - registers the given message to the appropriate */
/*     files over the given limits.				  */
/*								  */
/******************************************************************/


void
FILE_regmsg(file,msg,from,to,ins_ch,ins_mode)
   FILE_ID	file;
   FILE_MSG	msg;
   FILE_POS    *from;
   FILE_POS    *to;
   Character	ins_ch;
   Boolean	ins_mode;
{
   REG_MSG  rmsg;
   FILE_ID  file_id;
   FILE_NODE flistp;

   switch (msg) {
      case FILE_MSG_TYPEIN:
	 flistp = file->filedata->filelist;
	 while (flistp != NULL) {
	    file_id = flistp->id;
	    rmsg = file->filedata->regmsg;
	    while (rmsg != NULL) {
	       if (rmsg->limitset) {
		  if ( withinregion(from,&(rmsg->from),&(rmsg->to)) )
		     (*(rmsg->fct))(file_id,msg,from,to,ins_ch,ins_mode);
		}
	       else (*(rmsg->fct))(file_id,msg,from,to,ins_ch,ins_mode);
	       rmsg = rmsg->nxtregmsg;
	     }
	    flistp = flistp->nxtfile;
	  }
	 break;

      case FILE_MSG_DELETE:
      case FILE_MSG_INSERT:
	 flistp = file->filedata->filelist;
	 while (flistp != NULL) {
	    file_id = flistp->id;
	    rmsg = file->filedata->regmsg;
	    while (rmsg != NULL) {
	       if (rmsg->limitset) {
		  if ( withinregion(from,&(rmsg->from),&(rmsg->to)) ||
			  withinregion(&(rmsg->from),from,to) )
		     (*(rmsg->fct))(file_id,msg,from,to,ins_ch,ins_mode);
		}
	       else (*(rmsg->fct))(file_id,msg,from,to,ins_ch,ins_mode);
	       rmsg = rmsg->nxtregmsg;
	     }
	    flistp = flistp->nxtfile;
	  }
	 break;

      case FILE_MSG_CHANGED:
      case FILE_MSG_COMMAND:
	 flistp = file->filedata->filelist;
	 while (flistp != NULL) {
	    file_id = flistp->id;
	    rmsg = file->filedata->regmsg;
	    while (rmsg != NULL) {
	       (*(rmsg->fct))(file_id,msg,from,to,ins_ch,ins_mode);
	       rmsg = rmsg->nxtregmsg;
	     }
	    flistp = flistp->nxtfile;
	  }
	 break;

      case FILE_MSG_CLOSE:
	 rmsg = file->filedata->regmsg;
	 while (rmsg != NULL) {
	    (*(rmsg->fct))(file,msg,from,to,ins_ch,ins_mode);
	    rmsg = rmsg->nxtregmsg;
	  }
	 break;
    }
}




/******************************************************************/
/*								  */
/*    make_currpos_real - if CURRPOS is at virtual position,	  */
/*	newlines and spaces are added to make it real.		  */
/*								  */
/******************************************************************/


static void
make_currpos_real(fid)
   FILE_ID fid;
{
   FILE_POS fp1,fp2;
   FILE_INDEX old_pt;
   Integer added_lines;
   Integer added_chars;
   Integer i;

   if (!(fid->CURRPOS.virt)) return;

   FILEset_position(fid,fid->CURRLINE,fid->CURRCHAR,&fp1);

   old_pt = fid->POINT;
   if (fid->CURRLINE > fid->ENDLINE) {
      roomfor(fid,added_lines = fid->CURRLINE - fid->ENDLINE);
      FILE_movegapto(fid,fid->POINT);
      for (i = added_lines; i != 0; i--) {
	 *(fid->TEXTPTR + fid->GAPSTART) = NEWLINE;
	 fid->GAPSTART++;
       }
      fid->POINT = fid->GAPSTART;
      fid->SIZE += added_lines;
      fid->ENDLINE = fid->CURRLINE;
    }
   else added_lines = 0;

   roomfor(fid,added_chars = fid->CURRCHAR - FILE_inq_char(fid,fid->POINT));
   FILE_movegapto(fid,fid->POINT);
   for (i = added_chars; i > 0; i--) {
      *(fid->TEXTPTR + fid->GAPSTART) = SPACE;
      fid->GAPSTART++;
    }
   fid->POINT = fid->GAPSTART;
   if (added_chars > 0) fid->SIZE += added_chars;

   FILE_convert(fid,old_pt,&fp1);
   FILE_convert(fid,fid->POINT,&fp2);
   FILE_set_history(fid,HIST_INSERT,&fp1,&fp2,0,0,0,0);

   fid->CURRPOS.virt = FALSE;

   fid->filedata->lastvalid = FALSE;

   ++fid->filedata->modified;
   fid->CURRMOD = fid->filedata->modified;
}




/******************************************************************/
/*								  */
/*    roomfor - if there is room for k characters in the	  */
/*	file (gap) return true, else allocate a larger file	  */
/*	size to accommodate the addition.			  */
/*								  */
/******************************************************************/


static void
roomfor(fid,k)
   FILE_ID fid;
   Integer k;
{
   Integer room;
   Integer addedspace;
   String tempptr;

   room = fid->ALLOCSIZE - fid->SIZE;
   if (room < k) {
      addedspace = FILEBLOCKSIZE;
      while ((addedspace + room) < k) addedspace += FILEBLOCKSIZE;
      FILE_movegapto(fid,fid->FILE_INDEX_END);
      tempptr = (String) realloc (fid->TEXTPTR,fid->ALLOCSIZE + addedspace + 1);
      if (tempptr) {
	 fid->ALLOCSIZE += addedspace;
	 FILE_adjust_marks(fid,fid->GAPEND,fid->ALLOCSIZE,addedspace);
	 fid->GAPEND = fid->ALLOCSIZE;
	 fid->TEXTPTR = tempptr;
	 fid->TEXTPTR[fid->ALLOCSIZE] = 0;
       }
      else EDabort("Out of space expanding file");
    }
}





/******************************************************************/
/*								  */
/*   withinregion - returns bool indicating whether or not	  */
/*     the given file pos is between the two given limits.	  */
/*								  */
/******************************************************************/


static Boolean
withinregion(pos,from,to)
   FILE_POS *pos;
   FILE_POS *from;
   FILE_POS *to;
{
   if (from->line == to->line)
      return( (from->ch >= pos->ch) && (pos->ch < to->ch) );
   else
      return( ((from->line == pos->line) && (from->ch <= pos->ch)) ||
		 ((from->line < pos->line) && (pos->line < to->line)) ||
		 ((pos->line == to->line) && (pos->ch < to->ch)) );
}





/******************************************************************/
/*								  */
/*    withinlimits - return true if the given FILE_INDEX is	  */
/*	within the current editing region of the given file.	  */
/*								  */
/******************************************************************/


static Boolean
withinlimits(fid,pos)
   FILE_ID fid;
   FILE_INDEX pos;
{
   if (fid->filedata->nullregion) return FALSE;

   return( ((fid->filedata->startregion) <= fixed_absolute(fid,pos)) &&
	      (absolute(fid,pos) <= (fid->filedata->endregion)) );
}





/******************************************************************/
/*								  */
/*   copy_text - returns a string containing the text		  */
/*     between the given positions of the given file.		  */
/*								  */
/******************************************************************/


static String
copy_text(fid,from,to)
   FILE_ID fid;
   FILE_INDEX  from,to;
{
   String  text;
   Character *idx;

   if (from > to) EDabort("from > to in copy_text");

   text = (String) malloc(to-from+1);

   for (idx = text; from != to; from++, idx++)
      *idx = charAt(fid,from);
   *idx = 0;

   return(text);
}





/************************************************************************/
/*									*/
/*	limits_error -- handle error of editing outside limits		*/
/*									*/
/************************************************************************/


static void
limits_error(fid)
   FILE_ID fid;
{
   if (fid->filedata->nullregion) {
      EDfile_display(fid,"Attempting to edit readonly view");
    }
   else {
      EDfile_display(fid,"Attempting to edit outside limits");
    };
};





/* end of fileedit.c */
