/********************************************************************/
/*								    */
/*		filebuffers.c					    */
/*								    */
/*	Buffer routines for editor				    */
/*								    */
/********************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "editor_local.h"

#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>




/********************************************************************/
/*								    */
/*	     constants and parameters				    */
/*								    */
/********************************************************************/

#define  BFC_EOF   '\0'



/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


typedef struct _INCORE_BUF *	INCORE_BUF_PTR;
typedef struct _BUF_LOOKUP *	BUF_LOOKUP;



typedef enum _BF_STATUS {	/* two types of buffers 	    */
  INCORE,			/* local to a process		*/
  ONFILE			/* global to the user id	*/
} BF_STATUS;


typedef struct _INCORE_BUF {	/*  buffer data in core      */
    Integer  allocsize; 	/* allocated text size	   */
    Integer  txtidx;		/* current position	   */
    String  curtxt;		/* ptr to current position */
    String  txtptr;		/* ptr to start of text    */
} INCORE_BUF_INFO;


typedef struct _FILE_BUF {	/*   info about a buffer     */
    BF_STATUS	  status;	/*   local or global	   */
    String	  name; 	/*   buffer's textual name */
    String	  path; 	/*   if file, the abs path */
    INCORE_BUF_PTR  data;	/*   text info above	   */
    ASH_WINDOW x_assoc; 	/* x buffer assoc window   */
} FILE_BUF_INFO;



typedef struct _BUF_LOOKUP {	/* process info on which buffers are active */
    String   name;		/* textual name */
    FILE_BUF id;		/* buffer id	*/
    BUF_LOOKUP nxt;		/* ptr to next	*/
} BUF_LOOKUP_INFO;


typedef struct _STORAGE_BLK {	/* used when all buffers are stored in a file */
    Integer   size;		/* current length of the text	 */
    Character	name[MAXSTRINGLENGTH];	 /* textual name of buffer	     */
} STORAGE_BLK;





/********************************************************************/
/*								    */
/*	Local storage definitions				    */
/*								    */
/********************************************************************/


static	BUF_LOOKUP	all_buffers = NULL;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	local_buffer();
static	void		bufstore();
static	void		checkbuf();





/********************************************************************/
/*								    */
/*	FILE_buffer_init -- module initialization		    */
/*	FILE_buffer_cleanup -- cleanup at end			    */
/*								    */
/********************************************************************/


void
FILE_buffer_init()
{
  Character buf_dir[MAXSTRINGLENGTH];
  Integer i;
  extern int errno;

  all_buffers = NULL;

  FILEbuffer_define(FILEinq_buffer("BWEDIT$_space")," ");
  FILEbuffer_define(FILEinq_buffer("BWEDIT$_newline"),"\n");

  sprintf(buf_dir,"%s/.Buffers",getenv("HOME"));
  if ((i = access(buf_dir,F_OK)) < 0) {
     if (mkdir(buf_dir,0775) < 0) {
	EDfile_display(NULL,"Unable to create buffer directory");
	printf("Unable to create bufer directory %s -- %d %d\n",
		  buf_dir,i,errno);
	ASHbell();
	sleep(2);
	EDfile_display(NULL,"You will not be able to use named buffers");
      }
    }

  if (access(buf_dir,X_OK) < 0) {
     EDfile_display(NULL,"You will not be able to use named buffers");
     ASHbell();
   }
}




/********************************************************************/
/*								    */
/*    FILEinq_buffer_name - given a buf_id, return the		    */
/*	  corresponding textual name of that buffer.		    */
/*								    */
/********************************************************************/


char *
FILEinq_buffer_name(buf_id)
  FILE_BUF buf_id;
{
  FILE_INIT;

  return (buf_id == NULL ? NULL : buf_id->name);
};





/********************************************************************/
/*								    */
/*    FILEinq_buffer - given a name, create a buffer identity for   */
/*	  it (if it already doesn't exist) and return the buf_id.   */
/*								    */
/********************************************************************/


FILE_BUF
FILEinq_buffer(text)
  String text;
{
  BUF_LOOKUP l; 		/* used to check if buffer already exists    */
  FILE_BUF fb;			   /* newly created buffer id			*/
  Character bufpath[MAXSTRINGLENGTH];	/* if a file, its abs path			*/
  Integer fdesc;			   /* fdesc from a new file open		*/
  BUF_LOOKUP new;		/* used to add to the list of active buffers */

  FILE_INIT;

  if (text == NULL) return(NULL);

  PROTECT;

  for (l = all_buffers; l != NULL; l = l->nxt) {
     if (STREQL(l->name,text)) break;
   };

  if (l != NULL) {
     UNPROTECT;
     return l->id;
   };

  fb = PALLOC(FILE_BUF_INFO);

  fb->name = SALLOC(text);
  fb->data = PALLOC(INCORE_BUF_INFO);
  fb->data->txtptr = NULL;
  fb->data->allocsize = 0;
  fb->x_assoc = NULL;

  if (local_buffer(text)) fb->status = INCORE;
  else {
     sprintf(bufpath,"%s/.Buffers",getenv("HOME"));
     if (access(bufpath,X_OK|W_OK|R_OK) < 0) {
	UNPROTECT;
	EDfile_display(NULL,"Unable to access buffer");
	free(fb);
	return(NULL);
      }
     fb->status = ONFILE;
     sprintf(bufpath,"%s/.Buffers/%s",getenv("HOME"),text);
     fb->path = SALLOC(bufpath);
     if (access(bufpath,F_OK) < 0) {
	fdesc = open(bufpath,O_CREAT|O_RDWR,0666);
	if (fdesc <= 0) {
	   UNPROTECT;
	   EDfile_display(NULL,"Unable to access buffer");
	   free(fb);
	   return(NULL);
	 }
	close(fdesc);
      }
   }

  new = PALLOC(BUF_LOOKUP_INFO);
  new->name = SALLOC(text);
  new->id = fb;
  new->nxt = all_buffers;
  all_buffers = new;

  UNPROTECT;

  return(fb);
}



/******************************************************************/
/*								  */
/*    FILEbuffer_size - return the size (in chars.) of the buffer.*/
/*								  */
/******************************************************************/


int
FILEbuffer_size(buf_id)
  FILE_BUF buf_id;
{
  struct stat sbuf;
  Integer i;

  FILE_INIT;

  checkbuf(buf_id);
  if (buf_id->status == ONFILE) {
     if (stat(buf_id->path,&sbuf) < 0) i = 0;
     else i = sbuf.st_size;
   }
  else if (buf_id->data->txtptr == NULL) i = 0;
  else i = strlen(buf_id->data->txtptr);

  return i;
}



/********************************************************************/
/*								    */
/*     FILEbuffer_define - set the contents of the given buffer     */
/*	   equal to the given text string.			    */
/*								    */
/********************************************************************/


int
FILEbuffer_define(buf_id,text)
  FILE_BUF buf_id;
  String text;
{
  Integer length;	 /* length of incoming text */
  Integer fdesc;	 /* file desc from open     */
  Character msg[MAXSTRINGLENGTH];
  Boolean fg;
  extern int errno;

  FILE_INIT;

  checkbuf(buf_id);

  length = strlen(text);

  PROTECT;

  fg = TRUE;
  if (buf_id->status == ONFILE) {
     if ((fdesc = open(buf_id->path,O_TRUNC|O_WRONLY)) < 0) {
	sprintf(msg,"Unable to access buffer %s in FILEbuffer_define (%d)",
		   buf_id->name,errno);
	EDfile_display(NULL,msg);
	fg = FALSE;
      }
     if (fg && write(fdesc,text,length) != length) {
	sprintf(msg,"Error in write to buffer %s in FILEbuffer_define",buf_id->name);
	EDabort(msg);
	fg = FALSE;
      }
     close(fdesc);
   }
  else {
     if (buf_id->data->txtptr != NULL) free(buf_id->data->txtptr);
     buf_id->data->txtptr = SALLOC(text);
     buf_id->data->allocsize = length + 1;
   }

  if (buf_id->x_assoc != NULL) ASHstore_buffer_text(buf_id->x_assoc,text);

  UNPROTECT;

  return fg;
}





/********************************************************************/
/*								    */
/*     FILEbuffer_append - add given string to the given buffer     */
/*								    */
/********************************************************************/


int
FILEbuffer_append(buf_id,text)
  FILE_BUF buf_id;
  String text;
{
  Integer length;	 /* length of incoming text */
  Integer fdesc;	 /* file desc from open     */
  Character msg[MAXSTRINGLENGTH];
  String cbf;
  Boolean fg;

  FILE_INIT;

  checkbuf(buf_id);

  length = strlen(text);

  PROTECT;

  fg = TRUE;
  if (buf_id->status == ONFILE) {
     if ((fdesc = open(buf_id->path,O_APPEND|O_WRONLY)) < 0) {
	sprintf(msg,"Unable to access buffer %s in FILEbuffer_define",buf_id->name);
	EDfile_display(NULL,msg);
	fg = FALSE;
      }
     if (fg && write(fdesc,text,length) != length) {
	sprintf(msg,"Error in write to buffer %s in FILEbuffer_define",buf_id->name);
	fg = FALSE;
      }
     close(fdesc);
   }
  else {
     cbf = buf_id->data->txtptr;
     if (cbf != NULL) {
	buf_id->data->allocsize = length+strlen(cbf)+1;
	buf_id->data->txtptr = (String) realloc(cbf,buf_id->data->allocsize);
	strcat(buf_id->data->txtptr,text);
      }
     else {
	buf_id->data->txtptr = SALLOC(text);
	buf_id->data->allocsize = length + 1;
      };
    };

  UNPROTECT;

  return fg;
}



/********************************************************************/
/*								    */
/*     FILEbuffer_contents - return the text string currently	    */
/*	   residing in the given buffer.			    */
/*								    */
/********************************************************************/


char *
FILEbuffer_contents(buf_id)
  FILE_BUF buf_id;
{
  String contents;
  Integer filedesc;	 /* if a file, file descriptor from open    */
  struct stat sbuf;	 /* info from an fstat			    */

  FILE_INIT;

  PROTECT;

  checkbuf(buf_id);

  if (buf_id->x_assoc != NULL) {
     contents = ASHload_buffer_text(buf_id->x_assoc);
     if (contents == NULL) contents = SALLOC("");
   }
  else if (buf_id->status == INCORE) {
     if (buf_id->data->txtptr != NULL) contents = SALLOC(buf_id->data->txtptr);
     else contents = SALLOC("");
   }
  else {
     if ((filedesc = open(buf_id->path,O_RDONLY)) < 0) contents = SALLOC("");
     else {
	fstat(filedesc,&sbuf);
	contents = (String) malloc(sbuf.st_size + 1);
	if (read(filedesc,contents,sbuf.st_size) != sbuf.st_size)
	   EDabort("Error in buffer read size FILEbuffer_contents");
	contents[sbuf.st_size] = '\0';
	close(filedesc);
      };
   }

  UNPROTECT;

  return(contents);
}



/********************************************************************/
/*								    */
/*   FILEbuffer_store - store all the current buffers in this	    */
/*     process into the file "file."                                */
/*								    */
/********************************************************************/


void
FILEbuffer_store(file)
  String file;
{
  Integer	 fd;			  /* file descriptor from open	      */
  BUF_LOOKUP  p;			  /* used to walk active buffer list  */
  Character	 reply; 		  /* response to question	      */
  Character	 msg[MAXSTRINGLENGTH];	  /* used to make inquiry	      */

  FILE_INIT;

  if (access(file,F_OK) == 0) {     /* the file exists */
     sprintf(msg,"Buffer storage file %s already exists. Clobber? ",file);
     reply = *(EDfile_boolean(NULL,msg));
     if (reply != 'y' && reply != 'Y') return;
   }

  if ((fd = open(file,O_CREAT|O_TRUNC|O_WRONLY,0666)) < 0) {
     sprintf(msg,"Unable to open buffer file %s. No buffer save.",file);
     EDfile_display(NULL,msg);
     return;
   }

  for (p = all_buffers; p != NULL; p = p->nxt)
    bufstore(fd,p->id);

  close(fd);
};






/********************************************************************/
/*								    */
/*   FILEbuffer_load - load from the given file all the buffers     */
/*     that it contains.  Will overwrite current contents (if any)  */
/*     of each buffer name.					    */
/*								    */
/********************************************************************/


void
FILEbuffer_load(file)
  String file;
{
  Integer fd;		      /* file descriptor from open  */
  String text;		      /* file's text storage area   */
  STORAGE_BLK block;	      /* buffer info read into this */
  Character msg[MAXSTRINGLENGTH];  /* used to display a message  */

  FILE_INIT;

  if ((fd = open(file,O_RDONLY)) < 0) {
     sprintf(msg,"Unable to open buffer file %s. No buffer load.",file);
     EDfile_display(NULL,msg);
     return;
   }

  while (read(fd,&block,sizeof(block)) == sizeof(block))
    { text = (String) malloc(block.size + 1);
      if (read(fd,text,block.size) == block.size) {
	 text[block.size] = '\0';
	 FILEbuffer_define(FILEinq_buffer(block.name),text);
       }
      else break;
      free(text);
    }

  close(fd);
}




/************************************************************************/
/*									*/
/*	FILEbuffer_window -- set X-associated window for buffer 	*/
/*									*/
/************************************************************************/


void
FILEbuffer_window(bf,w)
   FILE_BUF bf;
   ASH_WINDOW w;
{
   checkbuf(bf);
   bf->x_assoc = w;
};





/******************************************************************/
/*								  */
/*    local_buffer - return TRUE if the given buffer name is a	  */
/*	  (in-core) buffer.					  */
/*								  */
/******************************************************************/


static Boolean
local_buffer(name)
  String name;
{
   if (strncmp(name,"BWEDIT$_",8) == 0) return TRUE;

   return FALSE;
}


/******************************************************************/
/*								  */
/*    bufstore - store a buffer in the given file		  */
/*								  */
/******************************************************************/


static void
bufstore(fd,buf_id)
  Integer fd;	       /* file descriptor of storage file */
  FILE_BUF buf_id;     /* buffer to be stored		  */
{
  STORAGE_BLK block;   /* header block of info that goes into the file for each buffer */
  Integer bfd;	       /* if a file, the file desc from the open */
  Integer size;        /* size of a file buffer */
  String text;	       /* storage for the text	*/
  struct stat sbuf;    /* info from an fstat	*/

  strcpy(block.name,buf_id->name);
  if (buf_id->status == ONFILE) {
     bfd = open(buf_id->path,O_RDONLY);
     if (bfd < 0) return;
     fstat(bfd,&sbuf);
     size = sbuf.st_size;
     block.size = size;
     text = (String) malloc(size);
     if (read(bfd,text,size) == size) {
	write(fd,&block,sizeof(block));   /* write the header info */
	write(fd,text,size);		  /* write the text	   */
      };
     close(bfd);
     free(text);
   }
  else {
     block.size = buf_id->data->txtidx;
     write(fd,&block,sizeof(block));
     write(fd,buf_id->data->txtptr,buf_id->data->txtidx);
   };
}





/******************************************************************/
/*								  */
/*    checkbuf - insure that the given buffer is active.	  */
/*								  */
/******************************************************************/


static void
checkbuf(buf_id)
  FILE_BUF buf_id;
{
  BUF_LOOKUP l;

  for (l = all_buffers; l != NULL; l = l->nxt) {
     if (l->id == buf_id) break;
    }

  if (l == NULL) EDabort("Buffer module has encountered an illegal FILE_BUF");
}






/* end of filebuffers.c */
