/****************************************************************************/
/*									    */
/*		editor_local.h						    */
/*									    */
/*	Local definitions for using the editor				    */
/*									    */
/****************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include <DATATYPES.h>

#define PROT_NAME EDITOR__sema
#include <bwethread.h>

#include <ash.h>
#include <rip.h>
#include <tulip.h>

#define EDITOR_LOCAL
#include "editor.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define   FILE_INDEX_START    0
#define   FILE_INDEX_END      SIZE
#define   FILE_INDEX_NONE     -1

#define  FILEBLOCKSIZE	  1024


#define  TULIP_START_LINE  0
#define  TULIP_START_CHAR  0

#define  NEWLINE	'\n'
#define  BACKSPACE	'\b'
#define  SPACE		' '

#define  LINE_FEEDc	   '\012'
#define  LINE_FEEDs	   "\012"
#define  BACK_SPACE	   "\010"
#define  CARRIAGE_RETURN   "\015"

#define  HIGHLIGHT_OFF	   0		/* modes for text	*/
#define  HIGHLIGHT_ON	   1
#define  SECONDARY_OFF	   2
#define  SECONDARY_ON	   3

#define HILITE_1		0	/* modes for tulip	*/
#define HILITE_2		1


#define  FULL_SCREEN	   -1

#define _TURN_CURSOR_ON    /*TULIPwrite(tid->tulip,"\033[>5l")*/
#define _TURN_CURSOR_OFF   /*TULIPwrite(tid->tulip,"\033[>5h")*/

#define CURSOR_FORCE_OFF	0
#define CURSOR_FORCE_VIEW	1
#define CURSOR_FORCE_FILE	2

#define MAXSTRINGLENGTH 	1024





/****************************************************************************/
/*									    */
/*	Data type definitions						    */
/*									    */
/****************************************************************************/


typedef Integer ABSOL_INDEX;
typedef Integer FILE_INDEX;

typedef struct _FILE_NODE *		FILE_NODE;
typedef struct _FILE_HIST_NODE *	FILE_HIST;
typedef struct _MARK_NODE *		MARK;
typedef struct _FILTER_NODE *		FILTER;
typedef struct _REG_MSG_NODE *		REG_MSG;
typedef struct _FILE_DATA *		FILE_DATA_PTR;
typedef struct _TEXT_NODE *		TEXT_NODE;
typedef struct _TEXT_CHANGE_NODE *	TEXT_CHANGE;





typedef enum _FILE_POS_TYPE {	  /* enumeration of FILE_POS types	    */
   REG, 			 /*  specifies regular FILE_POS node	   */
   START,			 /*  specifies FILE_POS_START node	   */
   END, 			 /*  specifies FILE_POS_END node	   */
   NONE 			 /*  specifies FILE_POS_NONE node	   */
} FILE_POS_TYPE;





typedef struct _FILE_POS_NODE {        /* what a file pos actually is  */
   FILE_POS_TYPE type;		       /*  start,end,none, or reg.     */
   FILE_INDEX index;		       /*  absolute index into the file*/
   Integer line;		       /*  the line number	       */
   Integer ch;			       /*  the character number on that*/
   Boolean virt;		       /*  TRUE if position is past \n */
   Integer modcnt;		       /*  Modified counter		*/
} FILE_POS_INFO;





typedef enum _FL_STATUS {	    /* types of files attempted on an open  */
   FL_OK,			  /* already there, rw privilege	  */
   FL_NEW,			  /* a new file 			  */
   FL_TRUNC,			  /* overwrite the old file		  */
   FL_READ,			  /* a read-only file			  */
   FL_TEMP,			  /* a temporary file (uncatalogued)	  */
   FL_BAD,			  /* special block or character file	  */
   FL_FAIL,			  /* the open failed			  */
   FL_DIR,			  /* a directory			  */
   FL_NOWRITE,			  /* unable to write over existing file   */
   FL_NOREAD,			  /* try edit non-existing readonly file  */
   FL_LOCK,			  /* file is locked			  */
   FL_COPY			  /* file is non-editable duplicate	  */
} FL_STATUS;


typedef enum _HIST_ACTION {	    /* enumeration of history actions	    */
   HIST_COMMAND,		   /*  specifies the end of a command	   */
   HIST_INSERT, 		   /*  specifies an insertion		   */
   HIST_DELETE, 		   /*  specifies a deletion		   */
   HIST_TYPEIN, 		   /*  specifies "normal" character typein */
   HIST_BACKSPACE,		   /*  specifies typein of a backspace	   */
   HIST_NEWLINE,		   /*  specifies typein of a newline	   */
   HIST_LIMITS, 		   /*  specifies a change in edit limits   */
   HIST_BUFFER, 		   /*  specifies a change in buffer's text */
   HIST_POSITION,		   /*  specifies a change in position	   */
} HIST_ACTION;


typedef enum _HIST_MODE {	    /* enumeration of history modes	    */
   HIST_MODE_NORMAL,		   /* not undoing or redoing		   */
   HIST_MODE_UNDO,		   /* currently undoing 		   */
   HIST_MODE_REDO		   /* currently redoing 		   */
} HIST_MODE;



typedef struct _FILE_NODE {	    /* node on the list of associated files */
   FILE_ID   id  ;		   /*  assoc. file			   */
   FILE_NODE nxtfile;		   /*  ptr to next node 		   */
} FILE_NODE_INFO;


typedef struct _FILE_HIST_NODE {    /* node in a history stack (undo/redo)  */
   HIST_ACTION	action; 	   /*  the type of action		   */
   FILE_ID	file;		   /*  the file in which action took place */
   FILE_POS	pos1;		   /*  the first position of action	   */
   FILE_POS	pos2;		   /*  the second position of action	   */
   FILE_BUF	buffer; 	   /*  the buffer involved in action	   */
   Set	ins_mode;		   /*  insertion mode at time of action    */
   Character	oldchar;	   /*  the old charcter (for undoing)	   */
   String	oldtext;	   /*  the old text (for undoing)	   */
   FILE_HIST	nxt_history;	   /*  pointer to next node in stack  */
   FILE_HIST	last_history;	   /*  FOR DEBUGGING ONLY		*/
} FILE_HIST_INFO;


typedef struct _MARK_NODE {	    /* actual mark in a file		    */
   ABSOL_INDEX pos;		   /*  marked position			   */
   MARK nxtstackmark;		   /*  ptr to next mark (stack) 	   */
   MARK nxtfilemark;		   /*  ptr to next mark (#'ed list)        */
   Boolean     active;		   /*  false if mark is destroyed	   */
} MARK_INFO;


typedef struct _FILE_MARK_NODE {    /* header item on a mark stack	    */
   FILE_ID   file;		   /*  in which file			   */
   MARK  stacktop;		   /*  ptr to stack top 		   */
} FILE_MARK_INFO;



typedef struct _FILTER_NODE {	    /* node on list of filters		    */
   Function_Ptr  fct;		   /*  actual filter function		   */
   FILTER nxtfilter;		   /*  pointer to next node		   */
} FILTER_INFO;


typedef struct _REG_MSG_NODE {	    /* register message node		    */
   Function_Ptr fct;		   /*  fct to call with message 	   */
   Set	    limitset;		   /*  true if limits on message region    */
   FILE_POS  from;		   /*  region start			   */
   FILE_POS  to;		   /*  region end			   */
   REG_MSG nxtregmsg;		   /*  pointer to next message node	   */
} REG_MSG_INFO;


typedef struct _FILE_DATA {	    /* everything about an edit file (core) */
   FILE_NODE	   filelist;	   /*	list of associated file_id's       */
   Integer	   size;	   /*	virtual file size		   */
   Integer	   allocsize;	   /*	allocated file size		   */
   Integer	   modified;	   /*	ctr of changed to file		   */
   Integer	   lastsecure;	   /*	last time file was secured	   */
   Integer	   numbackups;	   /*	number of backups to be saved	   */
   ABSOL_INDEX	   startregion;    /*	start of editing region 	   */
   ABSOL_INDEX	   endregion;	   /*	end of editing region		   */
   Boolean	   nullregion;	   /*	no editing region at all	   */
   FILE_POS	   currpos;	   /*	current working file position	   */
   Integer	   endline;	   /*	last valid line # in the file	   */
   ABSOL_INDEX	   gapstart;	   /*	start of space gap		   */
   ABSOL_INDEX	   gapend;	   /*	end of space gap		   */
   MARK 	   marklist;	   /*	list of marks within this file	   */
   FILTER	   infilters;	   /*	list of input filters		   */
   FILTER	   outfilters;	   /*	list of output filters		   */
   REG_MSG	   regmsg;	   /*	list of fcts to call with messages */
   HIST_MODE	   hist_mode;	   /*	undoing, redoing, or normal	   */
   FILE_HIST	   undo_history;   /*	undo history of actions on file    */
   FILE_HIST	   redo_history;   /*	redo history of actions on file    */
   String	   filetext;	   /*	ptr to actual file text 	   */
   Integer	   usecount;	   /*	count of FILE_IDs using this	   */
   Boolean	   lastvalid;	   /*	last inq line position valid	   */
   Boolean	   ignerror;	   /*	ignore error messages		   */
   FILE_INDEX	   lastindex;	   /*	last inq line index		   */
   Integer	   lastline;	   /*	last line number		   */
} FILE_DATA_INFO;



typedef struct _FILE_HEADER {	    /* info about edit window onto a file   */
   String	  pathname;	   /*	file's abs path name               */
   FL_STATUS	  filetype;	   /*	open parameters given on the open  */
   Integer	  filedesc;	   /*	file descriptor from a read	   */
   Integer	  device;	   /*	file's device number               */
   Integer	  inode;	   /*	file's inode number                */
   Integer	  rights;	   /*	file's permissions                 */
   ABSOL_INDEX	  pos;		   /*	current working file position mark */
   FILE_DATA_PTR  filedata;	   /*	in core text information	   */
   FILE_ID	  nxtfile;	   /*	pointer to next file header	   */
} FILE_ID_INFO;




typedef struct _TEXT_NODE {	    /* nodes on text_list		   */
   TEXT_ID   id;		  /*	its String_ID			 */
   Boolean	active; 	  /*	true if display is still active  */
   TEXT_NODE	next;		  /*	pointer to next node		 */
} TEXT_INFO;



typedef struct _TEXT_CHANGE_NODE {/* node describing pending text change */
   FILE_MSG   msg_type; 	/*   insert,delete,typein,close        */
   FILE_POS   from;		/*   region of change		       */
   FILE_POS   to;		/*				       */
   Character  ch;		/*   if typein, the character	       */
   Boolean    ins_mode; 	/*   typein's insert mode              */
   TEXT_CHANGE next;
} TEXT_CHANGE_INFO;



typedef struct _TEXT_DISPLAY {	   /*	 an actual TEXT_ID		      */
   FILE_ID	 file;		   /*	 associated FILE_ID		      */
   ASH_WINDOW	 window;	   /*	 its ASH_WINDOW 		      */
   TULIP_ID	 tulip; 	   /*	 its TULIP_ID			      */
   Integer	 line;		   /*	 line # (file) at top left	      */
   Integer	 ch;		   /*	 char # (file) at top left	      */
   Integer	 numlines;	   /*	 number of tulip_lines on this screen */
   Integer	 numchars;	   /*	 number of tulip_chars on this screen */
   Boolean	 up;		   /*	 true if visual display for screen    */
   Boolean	 insert_mode;	   /*	 true if ins_mode,false if overstrike */
   Boolean	 cursor_on;	   /*	 true if cursor is visible	      */
   Boolean	 shell_mode;	   /*	 true if shell type display	      */
   TEXT_CURSOR	 cursor;	   /*	 type of cursor in display	      */
   Integer	 cursor_force;	   /*	 force cursor to be displayed */
   Boolean	 marked;	   /*	 TRUE if marking is going on	      */
   FILE_POS	 mstartpos;	   /*	 file pos where mark originates       */
   FILE_POS	 muntilpos;	   /*	 last file pos of a mark until	      */
   Boolean	 marked2;
   FILE_POS	 m2startpos;
   FILE_POS	 m2untilpos;
   TEXT_CHANGE	 update_list;	   /*	 list of pending text changes	    */
   TEXT_CHANGE	 update_listend;   /*	 ptr to last change on the list     */
   TEXT_ID	 next;		   /*	 ptr to the next String_ID	    */
   Integer	 basefont;
} TEXT_ID_INFO;




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


extern	Boolean 	FILE__Initialized;

extern	TEXT_ID 	TEXT__list;





/************************************************************************/
/*									*/
/*	Macros for accessing FILE information				*/
/*									*/
/************************************************************************/



#define FILE_INIT if (!FILE__Initialized) FILE_init()


#define   CURRPOS      filedata->currpos   /* MACROS for ease of readability */
#define   POINT        filedata->currpos.index
#define   CURRLINE     filedata->currpos.line
#define   CURRCHAR     filedata->currpos.ch
#define   CURRMOD      filedata->currpos.modcnt
#define   ENDLINE      filedata->endline
#define   TEXTPTR      filedata->filetext
#define   GAPSTART     filedata->gapstart
#define   GAPEND       filedata->gapend
#define   SIZE	       filedata->size
#define   ALLOCSIZE    filedata->allocsize





/************************************************************************/
/*									*/
/*	Macro extensions						*/
/*									*/
/************************************************************************/


#define charAt(fid,pos) (*(fid->TEXTPTR + (( pos < fid->GAPSTART)\
			 ? pos : (pos + (fid->GAPEND - fid->GAPSTART)))))


#define absolute(fid,pos) ((pos < fid->GAPSTART) \
			? pos : (pos + fid->GAPEND - fid->GAPSTART))


#define fixed_absolute(fid,pos) ((pos <= fid->GAPSTART) \
			? pos : (pos + fid->GAPEND - fid->GAPSTART))


#define CopyPos(fid,from,to) {						\
		if ((from)->type != REG) FILE_copy_position((fid),(from),(to)); 	\
		else *(to) = *(from); }





/************************************************************************/
/*									*/
/*	External function definitions for files 			*/
/*									*/
/************************************************************************/


extern	void		FILE_init();		/*  filefiles.c 	*/

extern	void		FILE_location_init();	/* filelocation.c	*/
extern	Integer 	FILE_inq_char();
extern	void		FILE_convert();
extern	void		FILE_position();
extern	void		FILE_copy_position();

extern	void		FILE_position_real();	/* fileedit.c		*/
extern	void		FILE_movegapto();
extern	void		FILE_regmsg();

extern	void		FILE_buffer_init();	/* filebuffers.c	*/

extern	void		FILE_adjust_marks();	/* filemarks.c		*/
extern	void		FILE_remove_marks();
extern	FILE_INDEX	FILE_relative();

extern	void		FILE_set_history();	/* filehistory.c	*/






/************************************************************************/
/*									*/
/*	External function definitions for text				*/
/*									*/
/************************************************************************/

extern	void		TEXT_init();
extern	void		TEXT_close_local();

extern	void		TEXT_move_on_screen();
extern	Boolean 	TEXT_on_screen();
extern	void		TEXT_new_display();
extern	void		TEXT_highlight();
extern	void		TEXT_message_handler();





/************************************************************************/
/*									*/
/*	External functions defintions for I/O				*/
/*									*/
/************************************************************************/


extern	void		ED_init();





/* end of editor_local.h */
