/************************************************************************/
/*									*/
/*		cmpxmain.c						*/
/*									*/
/*	Communications multiplexing package				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "cmpx_local.h"




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


	Integer 	CMPX__tracelvl = 0;

static	CMPX_DATA	first_data = NULL;
static	Boolean 	initfg = FALSE;
static	Integer 	tick_time = CMPX_TICK_TIME;
static	Integer 	read_fgs[CMPX_MASK_SIZE];
static	Integer 	write_fgs[CMPX_MASK_SIZE];
static	Integer 	except_fgs[CMPX_MASK_SIZE];
static	THREAD_MONITOR	thread_mon;
static	Integer 	numtimers;

	PROT_DECL;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


#ifdef LIB
static	void		cmpx_cleanup();
#endif

static	CMPX_DATA	find_data();
static	void		recompute_flags();





/************************************************************************/
/*									*/
/*	CMPXinit -- module initialization				*/
/*									*/
/************************************************************************/


void
CMPXinit()
{
   PROT_INIT;

   PROTECT;
   if (!initfg) {
      initfg = TRUE;
      first_data = NULL;
      tick_time = CMPX_TICK_TIME;
      thread_mon = NULL;
      numtimers = 0;
      recompute_flags();
#ifdef LIB
      BROWNregister_cleanup(cmpx_cleanup);
#endif
    };
   UNPROTECT;
};





#ifdef LIB

static void
cmpx_cleanup()
{
   CMPX__tracelvl = 0;
   initfg = FALSE;
};

#endif




/************************************************************************/
/*									*/
/*	CMPXtrace -- set trace flag					*/
/*									*/
/************************************************************************/


void
CMPXtrace(lvl)
   Integer lvl;
{
   CMPX__tracelvl = lvl;
};






/************************************************************************/
/*									*/
/*	CMPXregister -- register a potential event			*/
/*									*/
/************************************************************************/


void
CMPXregister(ct,rfgs,wfgs,xfgs,rtn)
   Integer ct;
   Integer *rfgs;
   Integer *wfgs;
   Integer *xfgs;
   Function_Ptr rtn;
{
   register CMPX_DATA d;
   register Integer i;

   TRACE("CMPXregister %d 0x%x 0x%x",ct,
	    (rfgs == 0 ? 0 : *rfgs),(wfgs == 0 ? 0 : *wfgs));

   CMPXinit();

   PROTECT;

   d = find_data(rtn,TRUE);

   ITRACE("\t=> 0x%x",d);

   for (i = 0; i < CMPX_MASK_SIZE; ++i) {
      d->rfgs[i] = 0;
      d->wfgs[i] = 0;
      d->xfgs[i] = 0;
    };

   for (i = 0; i < (ct+31)/32; ++i) {
      if (rfgs != NULL) d->rfgs[i] = *rfgs++;
      if (wfgs != NULL) d->wfgs[i] = *wfgs++;
      if (xfgs != NULL) d->xfgs[i] = *xfgs++;
    };

   UNPROTECT;

   recompute_flags();
};





/************************************************************************/
/*									*/
/*	CMPXregister_timeout -- set timeout event			*/
/*									*/
/************************************************************************/


void
CMPXregister_timeout(time,rtn)
   Integer time;
   Function_Ptr rtn;
{
   register CMPX_DATA d;

   TRACE("CMPXregister_timeout %d");

   CMPXinit();

   PROTECT;

   d = find_data(rtn,TRUE);

   ITRACE("\t=> 0x%x",d);

   d->timeout = time;
   d->curtime = time;
   ++numtimers;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	CMPXregister_add -- add to existing event			*/
/*									*/
/************************************************************************/


void
CMPXregister_add(ct,rfgs,wfgs,xfgs,rtn)
   Integer ct;
   Integer *rfgs;
   Integer *wfgs;
   Integer *xfgs;
   Function_Ptr rtn;
{
   register CMPX_DATA d;
   register Integer i;

   TRACE("CMPXregister_add %d 0x%x 0x%x",ct,
	    (rfgs == 0 ? 0 : *rfgs),(wfgs == 0 ? 0 : *wfgs));

   PROTECT;

   d = find_data(rtn,TRUE);

   ITRACE("\t=> 0x%x",d);

   if (d != NULL) {
      for (i = 0; i < (ct+31)/32; ++i) {
	 if (rfgs != NULL) d->rfgs[i] |= *rfgs++;
	 if (wfgs != NULL) d->wfgs[i] |= *wfgs++;
	 if (xfgs != NULL) d->xfgs[i] |= *xfgs++;
       };
    };

   UNPROTECT;

   recompute_flags();
};





/************************************************************************/
/*									*/
/*	CMPXset_tick -- set tick interval				*/
/*									*/
/************************************************************************/


int
CMPXset_tick(time)
   Integer time;
{
   TRACE("CMPXset_tick %d",time);

   if (time > 0) tick_time = time;

   return tick_time;
};





/************************************************************************/
/*									*/
/*	CMPXset_monitor -- set monitor for thread waits 		*/
/*									*/
/************************************************************************/


void
CMPXset_monitor(mon)
   THREAD_MONITOR mon;
{
   TRACE("CMPXset_monitor 0x%x",mon);

   thread_mon = mon;
};





/************************************************************************/
/*									*/
/*	CMPXclose -- close a file descriptor				*/
/*									*/
/************************************************************************/


void
CMPXclose(fd)
   Integer fd;
{
   CMPX_DATA cd;

   read_fgs[fd/32] &= ~(1<<(fd%32));
   write_fgs[fd/32] &= ~(1<<(fd%32));
   except_fgs[fd/32] &= ~(1<<(fd%32));

   for (cd = first_data; cd != NULL; cd = cd->next) {
      cd->rfgs[fd/32] &= ~(1<<(fd%32));
      cd->wfgs[fd/32] &= ~(1<<(fd%32));
      cd->xfgs[fd/32] &= ~(1<<(fd%32));
    };
};





/************************************************************************/
/*									*/
/*	CMPXselect -- find next event					*/
/*									*/
/************************************************************************/


void
CMPXselect(time)
   Integer time;
{
   register Integer i;
   register CMPX_DATA d;
   struct { int tv_sec, tv_usec; } tv, *tvp;
   Integer rfgs[CMPX_MASK_SIZE];
   Integer wfgs[CMPX_MASK_SIZE];
   Integer xfgs[CMPX_MASK_SIZE];
   THREAD_MANAGER_BLOCK tmblk;

   TRACE("CMPXselect %d",time);

   if (time > 0 && numtimers == 0) {
      tv.tv_sec = tick_time / 1000;
      tv.tv_usec = (tick_time % 1000)*1000;
      tvp = &tv;
    }
   else if (numtimers == 0 || time < 0) {
      tv.tv_sec = 0;
      tv.tv_usec = 0;
      if (time == -2) tvp = &tv;
      else tvp = NULL;
      time = -1;
    }
   else {
      tv.tv_sec = tick_time / 1000;
      tv.tv_usec = (tick_time % 1000)*1000;
      tvp = &tv;
    };

   for ( ; ; ) {
      if (thread_mon != NULL) THREADmonitorentry(thread_mon,&tmblk);

      for (i = 0; i < CMPX_MASK_SIZE; ++i) {
	 rfgs[i] = read_fgs[i];
	 wfgs[i] = write_fgs[i];
	 xfgs[i] = except_fgs[i];
       };

      if (thread_mon != NULL) {
	 i = THREADmonitorwaitevent(thread_mon,0,
				       CMPX_MASK_SIZE*32,rfgs,wfgs,xfgs,tvp);
	 THREADmonitorexit(thread_mon);
       }
      else {
	 i = select(CMPX_MASK_SIZE*32,rfgs,wfgs,xfgs,tvp);
       };

      if (i <= 0) {
	 ITRACE("Timeout %d %d",i,time);
	 for (d = first_data; d != NULL; d = d->next) {
	    if (d->timeout > 0 && --(d->curtime) <= 0) {
	       (*(d->routine))(NULL,NULL,NULL);
	       d->curtime = d->timeout;
	     };
	  };
	 if (time != 0 && --time == 0) break;
	 if (time <= 0) break;
       }
      else {
	 ITRACE("Event %d 0x%x 0x%x",i,rfgs[0],wfgs[0]);
	 for (d = first_data; d != NULL; d = d->next) {
	    for (i = 0; i < CMPX_MASK_SIZE; ++i) {
	       if ((d->rfgs[i]&rfgs[i]) != 0 ||
		      (d->wfgs[i]&wfgs[i]) != 0 ||
		      (d->xfgs[i]&xfgs[i]) != 0) {
		  ITRACE("\tCall 0x%x",d);
		  (*(d->routine))(rfgs,wfgs,xfgs);
		  d->curtime = d->timeout;
		};
	     };
	  };
	 if (time == 0) {
	    tv.tv_sec = 0;
	    tv.tv_usec = 0;
	    time = -1;
	  }
	 else if (time < 0) break;
	 else --time;
       };
    };

   return;
};





/************************************************************************/
/*									*/
/*	CMPX_trace -- output trace information				*/
/*									*/
/************************************************************************/


void
CMPX_trace(msg,a1,a2,a3,a4,a5,a6,a7,a8,a9)
   String msg;
   Integer a1,a2,a3,a4,a5,a6,a7,a8,a9;
{
   Character mbf[1024];

   sprintf(mbf,msg,a1,a2,a3,a4,a5,a6,a7,a8,a9);

   printf("CMPX: %s\n",mbf);
};






/************************************************************************/
/*									*/
/*	find_data -- find data entry for given routine			*/
/*									*/
/************************************************************************/


static CMPX_DATA
find_data(rtn,fg)
   Function_Ptr rtn;
   Boolean fg;
{
   register CMPX_DATA d,ld;
   register Integer i;

   DTRACE("find_data 0x%x %d",rtn,fg);

   ld = NULL;
   for (d = first_data; d != NULL; d = d->next) {
      if (d->routine == rtn) break;
      ld = d;
    };

   if (d == NULL && fg) {
      d = (CMPX_DATA) malloc(sizeof(CMPX_DATA_B));
      d->next = NULL;
      d->routine = rtn;
      d->curtime = 0;
      d->timeout = 0;
      for (i = 0; i < CMPX_MASK_SIZE; ++i) {
	 d->rfgs[i] = 0;
	 d->wfgs[i] = 0;
	 d->xfgs[i] = 0;
       };
      if (ld == NULL) first_data = d;
      else ld->next = d;
    };

   return d;
};





/************************************************************************/
/*									*/
/*	recompute_flags -- recompute total mask 			*/
/*									*/
/************************************************************************/


static void
recompute_flags()
{
   register CMPX_DATA d;
   register Integer i;
   THREAD_MANAGER_BLOCK tmb;

   DTRACE("recompute_flags");

   if (thread_mon != NULL) THREADmonitorentry(thread_mon,&tmb);

   for (i = 0; i < CMPX_MASK_SIZE; ++i) {
      read_fgs[i] = 0;
      write_fgs[i] = 0;
      except_fgs[i] = 0;
    };

   for (d = first_data; d != NULL; d = d->next) {
      for (i = 0; i < CMPX_MASK_SIZE; ++i) {
	 read_fgs[i] |= d->rfgs[i];
	 write_fgs[i] |= d->wfgs[i];
	 except_fgs[i] |= d->xfgs[i];
       };
    };

   if (thread_mon != NULL) THREADmonitorexit(thread_mon);
};





/* end of cmpxmain.c */
