/************************************************************************/
/*									*/
/*		biowork.c						*/
/*									*/
/*	Main routines for input handler 				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#define BIO_MAIN

#include "bio_local.h"
#include <sgtty.h>
#include <sys/time.h>





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TRACK_STACK_SIZE	65536
#define TRACK_PRIORITY		16
#define SIG_PRIORITY		16




/************************************************************************/
/*									*/
/*	Local variables 						*/
/*									*/
/************************************************************************/


static	BIO_WINDATA	outerwindow = NULL;
static	BIO_WINDATA	all_windata = NULL;
static	BIO_QUEUE	input_queue = NULL;
static	ASH_WINDOW	last_window = NULL;
static	Integer 	last_x = 0;
static	Integer 	last_y = 0;
static	BPIO_TIME	last_time = 0;
static	Boolean 	movemap = FALSE;
static	Boolean 	usethreads = FALSE;
static	THREAD		track_thread;
static	Integer 	thread_data_offset;
static	Boolean 	tracking = FALSE;

static	FILE *		transcript = NULL;
static	FILE *		intranscript = NULL;
static	BPIO_EVENT_B	trans_event;
static	Boolean 	have_trans_event;
static	Integer 	trans_time;
static	Integer 	trans_time0;
static	ASH_WINDOW	trans_root;
static	ASH_WINDOW	trans_grab;

	Integer 	BIO__debugfg = 0;
	Boolean 	BIO__inited_flag = FALSE;

static	Character	intr_ch;
static	Character	quit_ch;

/*	PROT_DECL;	*/	/* we use the ASH semaphore for now */

static	Integer 	inhibit_flag;
static	Integer 	inhibit_count;
static	BIO_WINDATA	inhibit_window;
static	Integer 	inhibit_sig;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		bio_thread_initer();
static	void		bio_thread_finish();
static	Boolean 	next_event();
static	BIO_WINDATA	find_window();
static	void		send_message();
static	void		send_signal();
static	Boolean 	test_relevant();
static	Boolean 	click_time();
static	BIO_WINDATA	find_windata();
static	Integer 	fkey_map();
static	void		default_maps();
static	void		copy_maps();
static	void		flush();
static	Boolean 	same_event();
static	void		register_bio_mpx();
static	void		track_fct();
static	void		get_transcript();
static	Boolean 	nontrans_event();
static	void		setup_thread();
static	int		bio_control();
static	Boolean 	check_ignore();

#ifdef LIB
static	void		bio_cleanup();
#endif




/************************************************************************/
/*									*/
/*	Macro definitions						*/
/*									*/
/************************************************************************/


#define THREADinq_input_data(th) (*((BIO_THREAD *) THREADdynamic(th,thread_data_offset)))
#define thread_data(th) 	THREADinq_input_data(th)
#define find_thread_thread()	thread_data(THREADinq_current())
#define find_window_queue(wd)	(usethreads ? (wd)->BIO_thread->BIO_queue : input_queue)
#define find_thread_queue()	(usethreads ? thread_data(THREADinq_current())->BIO_queue : input_queue)





/************************************************************************/
/*									*/
/*	BIOthread_init -- initialize for threads			*/
/*									*/
/************************************************************************/


void
BIOthread_init()
{
   thread_data_offset = THREADtype1_register(4,bio_thread_initer,
						bio_thread_finish,NULL,NULL);
};




static void
bio_thread_initer(th)
   THREAD th;
{
   THREADinq_input_data(th) = NULL;
};





static void
bio_thread_finish(th)
   THREAD th;
{
   register BIO_WINDATA wd,wd1;
   register BIO_THREAD bth;

   ITRACE("bio_thread_finish 0x%x",th);

   if (th == NULL) return;

   bth = THREADinq_input_data(th);
   if (bth == NULL) return;

   PROTECT;
   for (wd = all_windata; wd != NULL; wd = wd1) {
      if (wd->BIO_thread == bth) {
	 ASHremove(wd->BIO_window);
	 wd1 = all_windata;
       }
      else wd1 = wd->BIO_next;
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	BIOinit -- module initialization				*/
/*									*/
/************************************************************************/


void
BIOinit()
{
   register THREAD_MONITOR mon;
   struct tchars tchar;

   TRACE("BIOinit");

   PROT_INIT;

   PROTECT;
   if (!BIO__inited_flag) {
      outerwindow = NULL;
      all_windata = NULL;
      last_window = NULL;
      last_x = 0;
      last_y = 0;
      last_time = 0;
      movemap = FALSE;
      transcript = NULL;
      intranscript = NULL;
      trans_root = NULL;
      trans_grab = NULL;
      track_thread = NULL;
      tracking = FALSE;
      usethreads = USE_THREADS;
      inhibit_flag = 0;
      inhibit_count = 0;
      inhibit_window = NULL;
      inhibit_sig = 0;
      if (!usethreads) {
	 input_queue = PALLOC(BIO_QUEUE_B);
	 input_queue->BIO_qhead = 0;
	 input_queue->BIO_qtail = 0;
	 input_queue->BIO_string = NULL;
       };
#ifdef LIB
      BROWNregister_cleanup(bio_cleanup);
#endif
      BIO__inited_flag = TRUE;

      if (ioctl(0,TIOCGETC,&tchar) >= 0) {
	 intr_ch = tchar.t_intrc;
	 quit_ch = tchar.t_quitc;
       }
      else {
	 intr_ch = 0x7f;
	 quit_ch = 0x1c;
       };

      BIOnew_input_window(ASHinq_top());

      register_bio_mpx();

      if (usethreads) {
	 track_thread = THREADcreate(track_fct,NULL,0,TRUE,
					TRACK_STACK_SIZE,TRACK_PRIORITY);
	 mon = THREADmonitorinit(1,NULL);
	 CMPXset_monitor(mon);
       };

      BIO_move_init();
    };
   UNPROTECT;
};




#ifdef LIB

static void
bio_cleanup()
{
   BIO__inited_flag = FALSE;
   BIO__debugfg = 0;
};

#endif




/************************************************************************/
/*									*/
/*	BIOtrace -- set up for tracing					*/
/*									*/
/************************************************************************/


void
BIOtrace(vl)
   Integer vl;
{
   BIO__debugfg = vl;
};





/************************************************************************/
/*									*/
/*	BIOnext_wait -- get next input, waiting for it (via cmpx)	*/
/*									*/
/************************************************************************/


BIO_EVENT
BIOnext_wait()
{
   register BIO_EVENT i;
   register BIO_QUEUE q;

   ENTER("BIOnext_wait");

   q = find_thread_queue();

   for ( ; ; ) {
      i = BIOnext();
      if (i != FKEY_EVENT_NONE) break;
      if (usethreads) {
	 PROTECT;
	 if (q->BIO_string == NULL && q->BIO_qhead == q->BIO_qtail) {
	    find_thread_thread()->BIO_waiting = TRUE;
	    UNPROTECT;
	    THREADpsem(find_thread_thread()->BIO_sema);
	  }
	 else {
	    UNPROTECT;
	  };
       }
      else {
	 if (intranscript != NULL) CMPXselect(-2);
	 else CMPXselect(0);
       };
    };

   return i;
};





/************************************************************************/
/*									*/
/*	BIOnext -- get next input from input queue for current thread	*/
/*									*/
/************************************************************************/


BIO_EVENT
BIOnext()
{
   BPIO_EVENT_B e;
   register Integer i,j;
   register BIO_QUEUE q;
   register BIO_WINDATA w;
   register BIO_THREAD th;
   Integer x,y;
   register ASH_WINDOW aw;

   ENTER("BIOnext");

   q = find_thread_queue();

   j = q->BIO_qhead -1;
   if (j < 0) j = MAX_QUEUE_SIZE-1;

   if (q->BIO_string != NULL) {
      i = *q->BIO_string++;
      if (i != 0) return i;
      q->BIO_string = NULL;
    };

   if (q->BIO_qhead == q->BIO_qtail) {
      BIOtrack();
    };

   if (q->BIO_qhead == q->BIO_qtail) return FKEY_EVENT_NONE;

   do {
      e = q->BIO_events[q->BIO_qhead];
      w = q->BIO_windata[q->BIO_qhead];

      i = q->BIO_qhead+1;
      if (i == MAX_QUEUE_SIZE) i = 0;
      q->BIO_qhead = i;

      if ((w->BIO_mask.BIO_mask_flags & BIO_MASK_TAP_BUTTON) != 0 &&
	     (e.BPIO_class == BPIO_CLASS_MOUSE)) {
	 if (q->BIO_qhead == q->BIO_qtail) {
	    BPIOwait(TAP_CLICK_INTERVAL);
	    BIOtrack();
	  };
	 if (q->BIO_qhead != q->BIO_qtail &&
		(q->BIO_events[q->BIO_qhead].BPIO_class == BPIO_CLASS_UP_MOUSE) &&
		click_time(&e,&q->BIO_events[q->BIO_qhead])) {
	    i = q->BIO_qhead+1;
	    if (i == MAX_QUEUE_SIZE) i = 0;
	    q->BIO_qhead = i;
	    e.BPIO_class = BPIO_CLASS_TAP_MOUSE;
	  };
       };

      i = fkey_map(&e,w);

      if (i != FKEY_EVENT_NONE) break;

      if (q->BIO_qhead == q->BIO_qtail) BIOtrack();
    }
   while (q->BIO_qhead != q->BIO_qtail);

   if (i == FKEY_EVENT_NONE) w = NULL;

   if (e.BPIO_window != NULL) {
      aw = ASHinq_top_window(e.BPIO_window);
      if (aw != e.BPIO_window) {
	 if (!ASHmap(e.BPIO_window,e.BPIO_xpos,e.BPIO_ypos,aw,&x,&y)) {
	    i = FKEY_EVENT_NONE;
	    w = NULL;
	  };
       }
      else {
	 x = e.BPIO_xpos;
	 y = e.BPIO_ypos;
       };
      aw = e.BPIO_window;
    }
   else {
      x = e.BPIO_xpos;
      y = e.BPIO_ypos;
      if (w == NULL) aw = NULL;
      else aw = w->BIO_window;
    };

   if (usethreads) {
      th = find_thread_thread();
      th->BIO_last_x = x;
      th->BIO_last_y = y;
      th->BIO_last_time = e.BPIO_time;
      th->BIO_last_window = aw;
    }
   else {
      last_x = x;
      last_y = y;
      last_time = e.BPIO_time;
      last_window = aw;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	BIOpeek -- peek at next event for current thread	       */
/*									*/
/************************************************************************/


BIO_EVENT
BIOpeek()
{
   register BIO_QUEUE q;
   BPIO_EVENT_B e;
   BIO_WINDATA w;
   register BIO_EVENT i;

   ENTER("BIOpeek");

   q = find_thread_queue();

   if (q->BIO_string != NULL) return *q->BIO_string;

   if (q->BIO_qhead == q->BIO_qtail) BIOtrack();

   while (q->BIO_qhead != q->BIO_qtail) {
      e = q->BIO_events[q->BIO_qhead];
      w = q->BIO_windata[q->BIO_qhead];
      i = fkey_map(&e,w);
      if (i != FKEY_EVENT_NONE) return i;
      i = q->BIO_qhead+1;
      if (i == MAX_QUEUE_SIZE) i = 0;
      q->BIO_qhead = i;
    };

   return FKEY_EVENT_NONE;
};





/************************************************************************/
/*									*/
/*	BIOinq_window -- return last window				*/
/*	BIOinq_position -- return last position 			*/
/*	BIOinq_time -- return last time 				*/
/*									*/
/************************************************************************/


ASH_WINDOW
BIOinq_window()
{
   ENTER("BIOinq_window");

   return (usethreads ? find_thread_thread()->BIO_last_window : last_window);
};





void
BIOinq_position(xp,yp)
   Integer *xp,*yp;
{
   ENTER("BIOinq_position");

   if (xp != NULL) *xp = (usethreads ? find_thread_thread()->BIO_last_x : last_x);
   if (yp != NULL) *yp = (usethreads ? find_thread_thread()->BIO_last_y : last_y);
};





void
BIOinq_time(sec,usec)
   Integer *sec, *usec;
{
   register BPIO_TIME t;

   ENTER("BIOinq_time");

   t = (usethreads ? (find_thread_thread()->BIO_last_time) : last_time);

   if (sec != NULL) *sec = t /1000;
   if (usec != NULL) *usec = (t % 1000) * 1000;
};





/************************************************************************/
/*									*/
/*	BIOpoll -- test if input is pending for current thread		*/
/*									*/
/************************************************************************/


int
BIOpoll()
{
   register BIO_QUEUE q;

   ENTER("BIOpoll");

   q = find_thread_queue();

   if (q->BIO_string != NULL) return TRUE;

   if (q->BIO_qhead == q->BIO_qtail) BIOtrack();

   return (q->BIO_qhead != q->BIO_qtail);
};





/************************************************************************/
/*									*/
/*	BIOtrack -- track input 					*/
/*									*/
/*	This routine should be called periodically and independently	*/
/*	of the rest of the program to handle tracking.	It queues up	*/
/*	events and as long as the queue routines are interruptable,	*/
/*	should be completely asynchronous				*/
/*									*/
/************************************************************************/


void
BIOtrack()
{
   BPIO_EVENT_B eb,es;
   register BIO_WINDATA w;
   Boolean esok;

   if (!BIO__inited_flag) {
      while (ASHnext_event(NULL));
      return;
    };

   TRACE("BIOtrack");

   if (usethreads && THREADinq_current() != track_thread) {
      THREADreschedule();
      return;
    };

   PROTECT;
   esok = tracking;
   tracking = TRUE;
   UNPROTECT;

   if (esok) return;

   esok = FALSE;
   while (next_event(&eb)) {
      if (esok && (eb.BPIO_class != BPIO_CLASS_EVENT ||
		      eb.BPIO_data != BPIO_EVENT_MOVE)) {
	 w = find_window(&es);
	 send_message(w,&es);
       };
      esok = FALSE;

      if (eb.BPIO_class == BPIO_CLASS_EVENT && eb.BPIO_data == BPIO_EVENT_MOVE) {
	 es = eb;
	 esok = TRUE;
	 continue;
       };

      DTRACE("EVENT: %d %d (%d,%d) %d",
		eb.BPIO_class,eb.BPIO_data,eb.BPIO_xpos,eb.BPIO_ypos,
		eb.BPIO_time);

      w = find_window(&eb);

      send_message(w,&eb);
    };

   if (esok) {
      w = find_window(&es);
      send_message(w,&es);
    };

   PROTECT;
   tracking = FALSE;
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	BIOsend_event -- add event to queue explicitly			*/
/*									*/
/************************************************************************/


void
BIOsend_event(ev)
   BPIO_EVENT ev;
{
   BIO_WINDATA w;

   w = find_window(ev);
   send_message(w,ev);
};





/************************************************************************/
/*									*/
/*	BIOnew_input_window -- set up new mapping window		*/
/*									*/
/************************************************************************/


void
BIOnew_input_window(w)
   ASH_WINDOW w;
{
   register BIO_WINDATA wd;
   register THREAD th;

   ENTER("BIOnew_input_window 0x%x",w);

   wd = find_windata(w);
   if (wd != NULL && wd->BIO_window == w) return;

   wd = PALLOC(BIO_WINDATA_B);

   wd->BIO_window = w;

   default_maps(wd);

   wd->BIO_cbreak = FALSE;

   PROTECT;
   if (usethreads) {
      th = THREADinq_current();
      wd->BIO_thread = thread_data(th);
      if (wd->BIO_thread == NULL) setup_thread(wd,th);
      if (wd->BIO_thread->BIO_windata == NULL) wd->BIO_thread->BIO_windata = wd;
    }
   else wd->BIO_thread = NULL;

   ASHset_control(w,bio_control);
   wd->BIO_next = all_windata;
   all_windata = wd;
   if (outerwindow == NULL) outerwindow = wd;
   UNPROTECT;

   ASHinput_window(w,wd);
};





/************************************************************************/
/*									*/
/*	BIOremove_window -- remove input window 			*/
/*									*/
/************************************************************************/


void
BIOremove_window(w)
   ASH_WINDOW w;
{
   register BIO_WINDATA wd,wd1;
   register BIO_THREAD bt;
   register BIO_QUEUE q;
   register Integer i;

   ENTER("BIOremove_window 0x%x",w);

   wd = find_windata(w);
   if (wd == outerwindow) {
      if (all_windata == NULL || all_windata->BIO_next == NULL) return;
      outerwindow = NULL;
    };

   PROTECT;

   q = find_window_queue(wd);
   for (i = 0; i < MAX_QUEUE_SIZE; ++i) {
      if (q->BIO_events[i].BPIO_window == w) {
	 q->BIO_events[i].BPIO_window = NULL;
	 q->BIO_events[i].BPIO_class = BPIO_CLASS_NONE;
       };
    };

   if (all_windata == wd) all_windata = wd->BIO_next;
   else {
      for (wd1 = all_windata;
	   wd1->BIO_next != wd && wd1 != NULL;
	   wd1 = wd1->BIO_next);
      if (wd1 != NULL) wd1->BIO_next = wd->BIO_next;
    };

   if (usethreads) {
      bt = wd->BIO_thread;
      if (bt != NULL && bt->BIO_windata == wd) bt->BIO_windata = NULL;
    };

   if (outerwindow == NULL) {
      for (outerwindow = all_windata;
	   outerwindow != NULL && outerwindow->BIO_next != NULL;
	   outerwindow = outerwindow->BIO_next);
    };

   UNPROTECT;

   ASHremove_control(wd->BIO_window,bio_control);
   ASHinput_window(wd->BIO_window,NULL);

   free(wd);
};





/************************************************************************/
/*									*/
/*	BIOset_mapping -- set a single mapping				*/
/*									*/
/************************************************************************/


void
BIOset_mapping(w,n,type,v1,v2)
   ASH_WINDOW w;
   Integer n;
   BIO_MAP_TYPE type;
   Universal v1,v2;
{
   register BIO_MAP m;
   register BIO_WINDATA wd;

   ENTER("BIOset_mapping 0x%x %d %d %d %d",w,n,type,v1,v2);

   wd = find_windata(w);

   if (FKEY_TEST(n)) {
      if (FKEY_TEST_PLAIN(n))
	 m = &wd->BIO_fkey_map[FKEY_BASE(n)];
      else if (FKEY_TEST_SHIFT(n))
	 m = &wd->BIO_shift_fkey_map[FKEY_BASE(n)];
      else if (FKEY_TEST_CTRL(n))
	 m = &wd->BIO_ctrl_fkey_map[FKEY_BASE(n)];
      else if (FKEY_TEST_META(n))
	 m = &wd->BIO_meta_fkey_map[FKEY_BASE(n)];
      else if (FKEY_TEST_UP(n))
	 m = &wd->BIO_up_fkey_map[FKEY_BASE(n)];
    }
   else if (FKEY_TEST_ASCII(n)) {
      m = &wd->BIO_ascii_map[FKEY_ASCII_BASE(n)];
    }
   else if (FKEY_TEST_META(n)) {
      m = &wd->BIO_meta_ascii_map[FKEY_ASCII_BASE(n)];
    }
   else if (FKEY_TEST_EVENT(n)) {
      m = &wd->BIO_event_map[n-FKEY_EVENT_FIRST];
    }
   else {
      fprintf(stderr,"Bad event %d for mapping\n",n);
      return;
    };

   if (type == BIO_MAP_MOVE) movemap = TRUE;

   m->BIO_maptype = type;
   m->BIO_mapdata.map_misc.v1 = v1;
   m->BIO_mapdata.map_misc.v2 = v2;
};





/************************************************************************/
/*									*/
/*	BIOset_mask -- set window processing mask			*/
/*									*/
/************************************************************************/


void
BIOset_mask(w,m,fkeys)
   ASH_WINDOW w;
   Integer m;
   Integer *fkeys;
{
   register BIO_WINDATA wd;
   register Integer i;

   ENTER("BIOset_mask 0x%x 0x%x",w,m);

   wd = find_windata(w);

   wd->BIO_mask.BIO_mask_flags = m;

   if (fkeys != NULL) {
      for (i = 0; i < FKEY_COUNT/(8*sizeof(Integer)); ++i)
	 wd->BIO_mask.BIO_fkeys[i] = *fkeys++;
    };
};





/************************************************************************/
/*									*/
/*	BIOset_cbreak -- set cbreak mode for window			*/
/*									*/
/************************************************************************/


void
BIOset_cbreak(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   register BIO_WINDATA wd;

   ENTER("BIOset_cbreak 0x%x %d",w,fg);

   wd = find_windata(w);

   wd->BIO_cbreak = fg;
};





/************************************************************************/
/*									*/
/*	BIOset_cursor -- set cursor for window				*/
/*	BIOset_cursor_pattern -- set cursor type for window		*/
/*	BIOset_cursor_window -- set cursor type by window for window	*/
/*									*/
/************************************************************************/


int
BIOset_cursor(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   register BIO_WINDATA wd;
   register Boolean ofg;

   ENTER("BIOset_cursor 0x%x %d",w,fg);

   wd = find_windata(w);
   w = wd->BIO_window;

   ofg = ASHcursor_enable(w,fg);

   return ofg;
};





ASH_CURSOR
BIOset_cursor_pattern(w,type)
   ASH_WINDOW w;
   ASH_CURSOR type;
{
   register BIO_WINDATA wd;
   register ASH_CURSOR oty;

   ENTER("BIOset_cursor_pattern 0x%x 0x%x",w,type);

   wd = find_windata(w);
   w = wd->BIO_window;

   oty = ASHwindow_cursor(w,type);

   return oty;
};





ASH_CURSOR
BIOset_cursor_standard(w,type)
   ASH_WINDOW w;
   ASH_CURSOR_ID type;
{
   ENTER("BIOset_cursor_standard 0x%x %d",w,type);

   return BIOset_cursor_pattern(w,ASHcursor_standard(type));
};





/************************************************************************/
/*									*/
/*	BIOset_window_thread -- set thread for window			*/
/*									*/
/************************************************************************/


int
BIOset_window_thread(w,th)
   ASH_WINDOW w;
   THREAD th;
{
   register BIO_WINDATA wd;
   THREAD oldth;

   ENTER("BIOset_window_thread 0x%x 0x%x",w,th);

   wd = find_windata(w);

   if (wd->BIO_thread == NULL) oldth = NULL;
   else oldth = wd->BIO_thread->BIO_thread_id;

   if (usethreads && th != NULL && th != oldth) {
      wd->BIO_thread = thread_data(th);
      if (wd->BIO_thread == NULL) setup_thread(wd,th);
      if (wd->BIO_thread->BIO_windata == NULL) wd->BIO_thread->BIO_windata = wd;
    };

   return (int) oldth;
};





/************************************************************************/
/*									*/
/*	BIOinq_top_window -- return top window for current thread	*/
/*									*/
/************************************************************************/


ASH_WINDOW
BIOinq_top_window()
{
   register BIO_THREAD bt;
   register ASH_WINDOW w;

   ENTER("BIOinq_top_window");

   w = NULL;

   if (usethreads) {
      bt = thread_data(THREADinq_current());
      if (bt != NULL && bt->BIO_windata != NULL) {
	 w = bt->BIO_windata->BIO_window;
       };
    };

   if (w == NULL) w = ASHinq_top();

   return w;
};





/************************************************************************/
/*									*/
/*	BIOgrab_io -- grab all io for given window			*/
/*									*/
/************************************************************************/


ASH_WINDOW
BIOgrab_io(w)
   ASH_WINDOW w;
{
   register BIO_WINDATA wd;
   register ASH_WINDOW ow;

   ENTER("BIOgrab_io 0x%x",w);

   if (w != NULL) {
      wd = find_windata(w);
      w = wd->BIO_window;
    };

   ow = ASHgrab_input(w);

   return ow;
};





/************************************************************************/
/*									*/
/*	BIOflush -- flush events not of given masked types		*/
/*	BIOflush_same -- flush events not of mask or same consecutive	*/
/*									*/
/************************************************************************/


void
BIOflush(mask)
   Integer mask;
{
   ENTER("BIOflush 0x%x",mask);

   flush(mask,0);
};





void
BIOflush_same(mask,samemask)
   Integer mask;
   Integer samemask;
{
   ENTER("BIOflush_same 0x%x 0x%x",mask,samemask);

   flush(mask,samemask);
};





/************************************************************************/
/*									*/
/*	BIOtranscript -- set transcript file				*/
/*									*/
/************************************************************************/


void
BIOtranscript(file)
   String file;
{
   ENTER("BIOtranscript %s",(file == NULL ? "0x0" : file));

   if (transcript != NULL) {
      fclose(transcript);
    };

   if (file != NULL) {
      transcript = fopen(file,"w");
      if (trans_root == NULL) trans_root = ASHroot_X_window(ASHinq_top());
    };
};





/************************************************************************/
/*									*/
/*	BIOinput_transcript -- set up input transcript			*/
/*									*/
/************************************************************************/


void
BIOinput_transcript(file)
   String file;
{
   ENTER("BIOinput_transcript %s",file);

   intranscript = fopen(file,"r");

   have_trans_event = FALSE;
   trans_time = 0;
   trans_time0 = 0;
   trans_grab = NULL;
   if (trans_root == NULL) trans_root = ASHroot_X_window(ASHinq_top());
};





/************************************************************************/
/*									*/
/*	BIOinhibit -- temporarily inhibit interrupts			*/
/*									*/
/************************************************************************/


void
BIOinhibit(fg)
   Boolean fg;
{
   BIO_WINDATA win;
   Integer sig;

   win = NULL;
   sig = 0;

   if (fg) {				/* start			*/
      if (inhibit_flag++ == 0) {
	 inhibit_count = 0;
	 inhibit_window = NULL;
	 inhibit_sig = 0;
       }
    }
   else {				/* finish			*/
      if (--inhibit_flag == 0) {
	 if (inhibit_count > 0) {
	    win = inhibit_window;
	    sig = inhibit_sig;
	  };
       }
      else if (inhibit_flag < 0) inhibit_flag = 0;
    };

   if (inhibit_flag == 0 && sig > 0) {
      send_signal(win,sig);
    };
};





/************************************************************************/
/*									*/
/*	BIO_error -- handle ash error message			       */
/*									*/
/************************************************************************/


void
BIO_error(msg)
   String msg;
{
   printf("BIO: ERROR: %s\n",msg);
   fflush(stdout);
};






/************************************************************************/
/*									*/
/*	BIO_trace -- handle tracing output			       */
/*									*/
/************************************************************************/


void
BIO_trace(f,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9)
   String f;
   Integer a0,a1,a2,a3,a4,a5,a6,a7,a8,a9;
{
   Character buf[1024];

   sprintf(buf,f,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9);

   printf("BIO: %s\n",buf);
   fflush(stdout);
};





/************************************************************************/
/*									*/
/*	next_event -- return next event 				*/
/*									*/
/************************************************************************/


static Boolean
next_event(e)
   BPIO_EVENT e;
{
   register Boolean fg;
   Integer x,y;

   DTRACE("next_event");

   fg = ASHnext_event(e);

   if (fg == FALSE) e->BPIO_class = BPIO_CLASS_NONE;
   else if (e->BPIO_class == BPIO_CLASS_NONE) fg = FALSE;

   if (intranscript != NULL) {
      get_transcript(e);
      fg = (e->BPIO_class != BPIO_CLASS_NONE);
    };

   if (transcript != NULL) {
      if (e->BPIO_class == BPIO_CLASS_NONE) {
	 x = y = 0;
	 e->BPIO_time = 0;
	 e->BPIO_data = 0;
       }
      else if (ASHmap(e->BPIO_window,e->BPIO_xpos,e->BPIO_ypos,trans_root,&x,&y)) {
	 fprintf(transcript,TRANSCRIPT_FORMAT,
		    e->BPIO_class,e->BPIO_data,x,y,e->BPIO_time);
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	find_window -- find current window				*/
/*									*/
/************************************************************************/


static BIO_WINDATA
find_window(e)
   BPIO_EVENT e;
{
   BIO_WINDATA d;

   DTRACE("find_window 0x%x",e);

   d = (BIO_WINDATA) ASHinq_input_data(e->BPIO_window);

   if (d == NULL) d = outerwindow;

   return d;
};





/************************************************************************/
/*									*/
/*	send_message -- handle message switching			*/
/*									*/
/************************************************************************/


static void
send_message(w,e)
   BIO_WINDATA w;
   BPIO_EVENT e;
{
   register Integer i;
   register BIO_QUEUE q;

   DTRACE("send_message 0x%x 0x%x",w,e);

   if (movemap) BIO_move_check(e,w);

   if (e->BPIO_class == BPIO_CLASS_ASCII && !w->BIO_cbreak) {
      if (e->BPIO_data == intr_ch) {
	 e->BPIO_class = BPIO_CLASS_ABORT;
	 e->BPIO_data = SIGINT;
       }
      else if (e->BPIO_data == quit_ch) {
	 e->BPIO_class = BPIO_CLASS_ABORT;
	 e->BPIO_data = SIGQUIT;
       }
    };

   if (e->BPIO_class == BPIO_CLASS_ABORT) {
      send_signal(w,e->BPIO_data);
    }
   else if (test_relevant(w,e) && !check_ignore(w,e)) {
      q = find_window_queue(w);
      i = q->BIO_qtail-1;
      if (i < 0) i = MAX_QUEUE_SIZE-1;
      if (q->BIO_qhead != q->BIO_qtail &&
	     e->BPIO_class == BPIO_CLASS_EVENT &&
	     e->BPIO_data == BPIO_EVENT_MOVE &&
	     q->BIO_events[i].BPIO_class == BPIO_CLASS_EVENT &&
	     q->BIO_events[i].BPIO_data == BPIO_EVENT_MOVE) {
	 q->BIO_events[i] = *e;
       }
      else {
	 i = (q->BIO_qtail+1)%MAX_QUEUE_SIZE;
	 if (i != q->BIO_qhead) {
	    q->BIO_events[q->BIO_qtail] = *e;
	    q->BIO_windata[q->BIO_qtail] = w;
	    q->BIO_qtail = i;
	  };
       };
      if (usethreads) {
	 PROTECT;
	 if (w->BIO_thread->BIO_waiting) {
	    w->BIO_thread->BIO_waiting = FALSE;
	    THREADvsem(w->BIO_thread->BIO_sema);
	  };
	 UNPROTECT;
       };
    };
};





/************************************************************************/
/*									*/
/*	send_signal -- send signal to window				*/
/*									*/
/************************************************************************/


static void
send_signal(win,sig)
   BIO_WINDATA win;
   Integer sig;
{
   DTRACE("send_signal 0x%x %d",win,sig);

   if (inhibit_flag > 0) {
      if (inhibit_count++ < 5) {
	 inhibit_window = win;
	 inhibit_sig = sig;
	 return;
       };
    };

   if (win->BIO_thread != NULL && usethreads &&
	       win->BIO_thread->BIO_thread_id != THREADinq_current()) {
      struct sigvec ovec;
      SIGHANDLER thvec;
      Function_Ptr rtn;
      THREADregistersignal(sig,NULL,&thvec);
      if (thvec.type != 0) rtn = (Function_Ptr) thvec.func;
      else {
	 sigvec(sig,NULL,&ovec);
	 rtn = (Function_Ptr) ovec.sv_handler;
       };
      if (rtn == (Function_Ptr) SIG_IGN) ;
      else if (rtn == (Function_Ptr) SIG_DFL) {
	 kill(getpid(),sig);
       }
      else {
	 THREADfakesignal(win->BIO_thread->BIO_thread_id,rtn,16,sig,0);
       };
      THREADreschedule();
    }
   else {
      kill(getpid(),sig);
    };
};






/************************************************************************/
/*									*/
/*	test_relevant -- check if message should be ignore or not	*/
/*									*/
/************************************************************************/


static Boolean
test_relevant(w,e)
   BIO_WINDATA w;
   BPIO_EVENT e;
{
   register Boolean fg;
   register Integer mask,i,j;

   DTRACE("test_relevant 0x%x",e);

   mask = w->BIO_mask.BIO_mask_flags;

   switch (e->BPIO_class) {
      case BPIO_CLASS_NONE :
	 fg = FALSE;
	 break;

      case BPIO_CLASS_ASCII :
	 fg = ((mask&BIO_MASK_ASCII) != 0);
	 break;

      case BPIO_CLASS_META_ASCII :
	 fg = ((mask&BIO_MASK_META) != 0);
	 if (!fg) {
	    fg = ((mask&BIO_MASK_ASCII) != 0);
	    if (fg) e->BPIO_class = BPIO_CLASS_ASCII;
	  };
	 break;

      case BPIO_CLASS_FKEY :
      case BPIO_CLASS_SHIFT_FKEY :
      case BPIO_CLASS_CTRL_FKEY :
      case BPIO_CLASS_META_FKEY :
      case BPIO_CLASS_UP_FKEY :
	 fg = ((mask&BIO_MASK_FKEY) != 0);
	 if (!fg) break;
	 i = e->BPIO_data%(8*sizeof(Integer));
	 j = e->BPIO_data/(8*sizeof(Integer));
	 if ((w->BIO_mask.BIO_fkeys[j]&(1<<i)) == 0) fg = FALSE;
	 else {
	    switch (e->BPIO_class) {
	       case BPIO_CLASS_SHIFT_FKEY :
		  if ((mask&BIO_MASK_SHIFT_FKEY) == 0)
		     e->BPIO_class = BPIO_CLASS_FKEY;
		  break;
	       case BPIO_CLASS_CTRL_FKEY :
		  if ((mask&BIO_MASK_CTRL_FKEY) == 0)
		     e->BPIO_class = BPIO_CLASS_FKEY;
		  break;
	       case BPIO_CLASS_META_FKEY :
		  if ((mask&BIO_MASK_META_FKEY) == 0)
		     e->BPIO_class = BPIO_CLASS_FKEY;
		  break;
	       case BPIO_CLASS_UP_FKEY :
		  fg = ((mask&BIO_MASK_UP_FKEY) != 0);
		  break;
	     };
	  };
	 break;

      case BPIO_CLASS_UP_MOUSE :
      case BPIO_CLASS_MOUSE :
      case BPIO_CLASS_SHIFT_MOUSE :
      case BPIO_CLASS_CTRL_MOUSE :
      case BPIO_CLASS_META_MOUSE :
	 if ((mask&BIO_MASK_OR_BUTTON) != 0) {
	    e->BPIO_data = BPIO_MOUSE_STATE(e->BPIO_data);
	    e->BPIO_class = BPIO_CLASS_MOUSE_DATA;
	    fg = TRUE;
	  }
	 else {
	    if ((mask&BIO_MASK_SHIFT_BUTTON) == 0 && e->BPIO_class != BPIO_CLASS_UP_MOUSE)
	       e->BPIO_class = BPIO_CLASS_MOUSE;
	    e->BPIO_data = BPIO_MOUSE_BTN(e->BPIO_data);
	    if (e->BPIO_class == BPIO_CLASS_UP_MOUSE)
	       fg = ((mask&BIO_MASK_BUTTON_UP) != 0);
	    else
	       fg = ((mask&BIO_MASK_BUTTON) != 0);
	  };
	 break;

      case BPIO_CLASS_ABORT :
	 fg = TRUE;
	 break;

      case BPIO_CLASS_EVENT :
	 switch (e->BPIO_data) {
	    case BPIO_EVENT_MOVE :
	       i = BIO_MASK_MOVE;
	       break;
	    case BPIO_EVENT_MOVE_DOWN :
	       if ((mask&BIO_MASK_MOVE_DOWN) == 0)
		  e->BPIO_data = BPIO_EVENT_MOVE;
	       i = BIO_MASK_MOVE_DOWN|BIO_MASK_MOVE;
	       break;
	    case BPIO_EVENT_ENTER :
	       i = BIO_MASK_WINDOW_ENTER;
	       break;
	    case BPIO_EVENT_EXIT :
	       i = BIO_MASK_WINDOW_EXIT;
	       break;
	    case BPIO_EVENT_STILL :
	       i = BIO_MASK_STILL;
	       break;
	    case BPIO_EVENT_STILL_DOWN :
	       if ((mask&BIO_MASK_STILL_DOWN) == 0)
		  e->BPIO_data = BPIO_EVENT_STILL;
	       i = BIO_MASK_STILL_DOWN|BIO_MASK_STILL;
	       break;
	    case BPIO_EVENT_REDRAW :
	       i = BIO_MASK_REDRAW;
	       break;
	    case BPIO_EVENT_VISIBLE :
	    case BPIO_EVENT_INVISIBLE :
	       i = BIO_MASK_VISIBLE;
	       break;
	    case BPIO_EVENT_RESIZE :
	    case BPIO_EVENT_VIEW :
	       i = BIO_MASK_VIEW;
	       break;
	  };
	 fg = ((mask&i) != 0);
	 break;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	click_time -- check if two clicks are close enough to double	*/
/*									*/
/************************************************************************/


static Boolean
click_time(e1,e2)
   BPIO_EVENT e1,e2;
{
   register Integer dt;

   DTRACE("click_time 0x%x 0x%x",e1,e2);

   dt = e2->BPIO_time-e1->BPIO_time;
   dt *= 1000;

   return (dt <= TAP_CLICK_INTERVAL);
};





/************************************************************************/
/*									*/
/*	find_windata -- find window data block for window		*/
/*									*/
/************************************************************************/


static BIO_WINDATA
find_windata(w)
   ASH_WINDOW w;
{
   register BIO_WINDATA wd;
   register BIO_THREAD th;

   DTRACE("find_windata 0x%x",w);

   if (w == NULL) {
      if (usethreads) {
	 th = find_thread_thread();
	 if (th == NULL) wd = outerwindow;
	 else wd = th->BIO_windata;
       }
      else wd = outerwindow;
    }
   else {
      wd = (BIO_WINDATA) ASHinq_input_data(w);
    };

   if (wd == NULL) wd = outerwindow;

   return wd;
};





/************************************************************************/
/*									*/
/*	fkey_map -- map bpio event into a fkey value			*/
/*									*/
/************************************************************************/


static Integer
fkey_map(e,w)
   BPIO_EVENT e;
   BIO_WINDATA w;
{
   register Integer v;
   register BIO_MAP m;
   register BIO_QUEUE q;

   DTRACE("fkey_map 0x%x 0x%x",e,w);

   m = NULL;
   v = FKEY_EVENT_NONE;

   switch (e->BPIO_class) {
      case BPIO_CLASS_NONE :
	 v = FKEY_EVENT_NONE;
	 break;
      case BPIO_CLASS_FKEY :
	 v = FKEY(e->BPIO_data);
	 m = &w->BIO_fkey_map[e->BPIO_data];
	 break;
      case BPIO_CLASS_SHIFT_FKEY :
	 m = &w->BIO_shift_fkey_map[e->BPIO_data];
	 v = FKEY_SHIFT(e->BPIO_data);
	 break;
      case BPIO_CLASS_CTRL_FKEY :
	 m = &w->BIO_ctrl_fkey_map[e->BPIO_data];
	 v = FKEY_CTRL(e->BPIO_data);
	 break;
      case BPIO_CLASS_META_FKEY :
	 m = &w->BIO_meta_fkey_map[e->BPIO_data];
	 v = FKEY_CTRL(e->BPIO_data);
	 break;
      case BPIO_CLASS_UP_FKEY :
	 m = &w->BIO_up_fkey_map[e->BPIO_data];
	 v = FKEY_UP(e->BPIO_data);
	 break;
      case BPIO_CLASS_MOUSE :
	 v = FKEY_MOUSE(e->BPIO_data);
	 break;
      case BPIO_CLASS_SHIFT_MOUSE :
	 v = FKEY_MOUSE_SHIFT(e->BPIO_data);
	 break;
      case BPIO_CLASS_CTRL_MOUSE :
	 v = FKEY_MOUSE_CTRL(e->BPIO_data);
	 break;
      case BPIO_CLASS_META_MOUSE :
	 v = FKEY_MOUSE_META(e->BPIO_data);
	 break;
      case BPIO_CLASS_UP_MOUSE :
	 v = FKEY_MOUSE_UP(e->BPIO_data);
	 break;
      case BPIO_CLASS_TAP_MOUSE :
	 v = FKEY_MOUSE_TAP(e->BPIO_data);
	 break;
      case BPIO_CLASS_MOUSE_DATA :
	 v = FKEY_MOUSE_COMBO(e->BPIO_data);
	 break;
      case BPIO_CLASS_ASCII :
	 m = &w->BIO_ascii_map[e->BPIO_data];
	 v = FKEY_ASCII(e->BPIO_data);
	 break;
      case BPIO_CLASS_META_ASCII :
	 m = &w->BIO_meta_ascii_map[e->BPIO_data];
	 v = FKEY_META(e->BPIO_data);
	 break;
      case BPIO_CLASS_EVENT :
	 switch (e->BPIO_data) {
	    case BPIO_EVENT_MOVE :
	       v = FKEY_EVENT_MOVE;
	       break;
	    case BPIO_EVENT_MOVE_DOWN :
	       v = FKEY_EVENT_MOVE;
	       break;
	    case BPIO_EVENT_ENTER :
	       v = FKEY_EVENT_ENTER;
	       break;
	    case BPIO_EVENT_EXIT :
	       v = FKEY_EVENT_EXIT;
	       break;
	    case BPIO_EVENT_STILL :
	       v = FKEY_EVENT_STILL;
	       break;
	    case BPIO_EVENT_STILL_DOWN :
	       v = FKEY_EVENT_STILL;
	       break;
	    case BPIO_EVENT_REDRAW :
	       v = FKEY_EVENT_REDRAW;
	       break;
	    case BPIO_EVENT_VISIBLE :
	       v = FKEY_EVENT_VISIBLE;
	       break;
	    case BPIO_EVENT_INVISIBLE :
	       v = FKEY_EVENT_INVISIBLE;
	       break;
	    case BPIO_EVENT_VIEW :
	       v = FKEY_EVENT_VIEW;
	       break;
	    case BPIO_EVENT_RESIZE :
	       v = FKEY_EVENT_RESIZE;
	       break;
	  };
	 m = &w->BIO_event_map[v-FKEY_EVENT_FIRST];
	 break;
      default :
	 break;
    };

   if (m != NULL) {
      switch (m->BIO_maptype) {
	 case BIO_MAP_DEFAULT :
	    break;
	 case BIO_MAP_STRING :
	    q = find_window_queue(w);
	    q->BIO_string = m->BIO_mapdata.map_string;
	    v = *q->BIO_string++;
	    if (v == 0) {
	       q->BIO_string = NULL;
	       v = FKEY_EVENT_NONE;
	     };
	    break;
	 case BIO_MAP_ACTION :
	     { Integer xpos,ypos;
/*	       ASHmap(ASHinq_top_window(w->BIO_window),e->BPIO_xpos,e->BPIO_ypos,	*/
/*			 w->BIO_window,&xpos,&ypos);					*/
	       xpos = e->BPIO_xpos;
	       ypos = e->BPIO_ypos;
	       (*(m->BIO_mapdata.map_call.map_rtn))(m->BIO_mapdata.map_call.map_data,
						       w->BIO_window,xpos,ypos);
	       v = FKEY_EVENT_NONE;
	     };
	    break;
	 case BIO_MAP_VALUE :
	    v = m->BIO_mapdata.map_value;
	    break;
	 case BIO_MAP_MOVE :
	 case BIO_MAP_NONE :
	    v= FKEY_EVENT_NONE;
	    break;
	 case BIO_MAP_WINACT :
	    (*(m->BIO_mapdata.map_call.map_rtn))(e->BPIO_window);
	    break;
       };
    };

   return v;
};






/************************************************************************/
/*									*/
/*	default_maps -- set up default mappings for window		*/
/*									*/
/************************************************************************/


static void
default_maps(wd)
   BIO_WINDATA wd;
{
   register Integer i;

   DTRACE("default_maps 0x%x",wd);

   if (outerwindow == NULL) {
      for (i = 0; i < 127; ++i) {
	 wd->BIO_ascii_map[i].BIO_maptype = BIO_MAP_DEFAULT;
	 wd->BIO_meta_ascii_map[i].BIO_maptype = BIO_MAP_DEFAULT;
       };
      for (i = 0; i < 64; ++i) {
	 wd->BIO_fkey_map[i].BIO_maptype = BIO_MAP_DEFAULT;
	 wd->BIO_shift_fkey_map[i].BIO_maptype = BIO_MAP_DEFAULT;
	 wd->BIO_ctrl_fkey_map[i].BIO_maptype = BIO_MAP_DEFAULT;
	 wd->BIO_meta_fkey_map[i].BIO_maptype = BIO_MAP_DEFAULT;
	 wd->BIO_up_fkey_map[i].BIO_maptype = BIO_MAP_DEFAULT;
       };
      for (i = 0; i < 16; ++i) {
	 wd->BIO_event_map[i].BIO_maptype = BIO_MAP_DEFAULT;
       };
      wd->BIO_mask.BIO_mask_flags = BIO_MASK_DEFAULT;
      for (i = 0; i < 64/32; ++i) {
	 wd->BIO_mask.BIO_fkeys[i] = -1;
       };
      i = FKEY_EVENT_REDRAW - FKEY_EVENT_FIRST;
      wd->BIO_event_map[i].BIO_maptype = BIO_MAP_WINACT;
      wd->BIO_event_map[i].BIO_mapdata.map_call.map_rtn = (Function_Ptr) ASHfix_damage;
      i = FKEY_EVENT_VISIBLE - FKEY_EVENT_FIRST;
      wd->BIO_event_map[i].BIO_maptype = BIO_MAP_WINACT;
      wd->BIO_event_map[i].BIO_mapdata.map_call.map_rtn = (Function_Ptr) ASHfix_visible;
      i = FKEY_EVENT_INVISIBLE - FKEY_EVENT_FIRST;
      wd->BIO_event_map[i].BIO_maptype = BIO_MAP_WINACT;
      wd->BIO_event_map[i].BIO_mapdata.map_call.map_rtn = (Function_Ptr) ASHfix_invisible;
      i = FKEY_EVENT_VIEW - FKEY_EVENT_FIRST;
      wd->BIO_event_map[i].BIO_maptype = BIO_MAP_WINACT;
      wd->BIO_event_map[i].BIO_mapdata.map_call.map_rtn = (Function_Ptr) ASHfix_configure;
      i = FKEY_EVENT_RESIZE - FKEY_EVENT_FIRST;
      wd->BIO_event_map[i].BIO_maptype = BIO_MAP_WINACT;
      wd->BIO_event_map[i].BIO_mapdata.map_call.map_rtn = (Function_Ptr) ASHfix_configure;
    }
   else {
      for (i = 0; i < 127; ++i) {
	 wd->BIO_ascii_map[i] = outerwindow->BIO_ascii_map[i];
	 wd->BIO_meta_ascii_map[i] = outerwindow->BIO_meta_ascii_map[i];
       };
      for (i = 0; i < 64; ++i) {
	 wd->BIO_fkey_map[i] = outerwindow->BIO_fkey_map[i];
	 wd->BIO_shift_fkey_map[i] = outerwindow->BIO_shift_fkey_map[i];
	 wd->BIO_ctrl_fkey_map[i] = outerwindow->BIO_ctrl_fkey_map[i];
	 wd->BIO_meta_fkey_map[i] = outerwindow->BIO_meta_fkey_map[i];
	 wd->BIO_up_fkey_map[i] = outerwindow->BIO_up_fkey_map[i];
       };
      for (i = 0; i < 16; ++i) {
	 wd->BIO_event_map[i] = outerwindow->BIO_event_map[i];
       };
      wd->BIO_mask = outerwindow->BIO_mask;
    };
};





/************************************************************************/
/*									*/
/*	flush -- flush input until event occurs 			*/
/*									*/
/************************************************************************/


static void
flush(mask,smask)
   Integer mask;
   Integer smask;
{
   register Integer i,j;
   register BIO_QUEUE q;

   DTRACE("flush 0x%x 0x%x",mask,smask);

   q = find_thread_queue();

   if (q->BIO_qhead == q->BIO_qtail) BIOtrack();

   while (q->BIO_qhead != q->BIO_qtail) {
      switch (q->BIO_events[q->BIO_qhead].BPIO_class) {
	 case BPIO_CLASS_NONE :
	    i = 0;
	    break;
	 case BPIO_CLASS_FKEY :
	    i = BIO_MASK_FKEY;
	    break;
	 case BPIO_CLASS_SHIFT_FKEY :
	    i = BIO_MASK_SHIFT_FKEY;
	    break;
	 case BPIO_CLASS_CTRL_FKEY :
	    i = BIO_MASK_CTRL_FKEY;
	    break;
	 case BPIO_CLASS_META_FKEY :
	    i = BIO_MASK_META_FKEY;
	    break;
	 case BPIO_CLASS_UP_FKEY :
	    i = BIO_MASK_UP_FKEY;
	    break;
	 case BPIO_CLASS_MOUSE :
	 case BPIO_CLASS_MOUSE_DATA :
	    i = BIO_MASK_BUTTON;
	    break;
	 case BPIO_CLASS_SHIFT_MOUSE :
	 case BPIO_CLASS_CTRL_MOUSE :
	 case BPIO_CLASS_META_MOUSE :
	    i = BIO_MASK_SHIFT_BUTTON;
	    break;
	 case BPIO_CLASS_UP_MOUSE :
	    i = BIO_MASK_BUTTON_UP;
	    break;
	 case BPIO_CLASS_ASCII :
	    i = BIO_MASK_ASCII;
	    break;
	 case BPIO_CLASS_META_ASCII :
	    i = BIO_MASK_META;
	    break;
	 case BPIO_CLASS_EVENT :
	    switch (q->BIO_events[q->BIO_qhead].BPIO_data) {
	       case BPIO_EVENT_MOVE :
		  i = BIO_MASK_MOVE;
		  break;
	       case BPIO_EVENT_MOVE_DOWN :
		  i = BIO_MASK_MOVE_DOWN;
		  break;
	       case BPIO_EVENT_ENTER :
		  i = BIO_MASK_WINDOW_ENTER;
		  break;
	       case BPIO_EVENT_EXIT :
		  i = BIO_MASK_WINDOW_EXIT;
		  break;
	       case BPIO_EVENT_STILL :
		  i = BIO_MASK_STILL;
		  break;
	       case BPIO_EVENT_STILL_DOWN :
		  i = BIO_MASK_STILL_DOWN;
		  break;
	       case BPIO_EVENT_REDRAW :
		  i = BIO_MASK_REDRAW;
		  break;
	       case BPIO_EVENT_VISIBLE :
	       case BPIO_EVENT_INVISIBLE :
		  i = BIO_MASK_VISIBLE;
		  break;
	       case BPIO_EVENT_RESIZE :
	       case BPIO_EVENT_VIEW :
		  i = BIO_MASK_VIEW;
		  break;
	       default :
		  i = 0;
		  break;
	     };
	    break;
	 default :
	    i = 0;
	    break;
       };

      j = q->BIO_qhead+1;
      if (j >= MAX_QUEUE_SIZE) j = 0;

      if ((i&mask) != 0) {
	 if ((i&smask) == 0) break;
	 if (j == q->BIO_qtail) break;
	 if (!same_event(&q->BIO_events[q->BIO_qhead],
			    &q->BIO_events[j]))
	    break;
       };

      q->BIO_qhead = j;
    };
};





/************************************************************************/
/*									*/
/*	same_event -- check if two events are about the same		*/
/*									*/
/************************************************************************/


static Boolean
same_event(e1,e2)
   BPIO_EVENT e1,e2;
{
   DTRACE("same_event 0x%x 0x%x",e1,e2);

   return (e1->BPIO_class == e2->BPIO_class &&
	      e1->BPIO_data == e2->BPIO_data);
};





/************************************************************************/
/*									*/
/*	register_bio_mpx -- register in multiplexor			*/
/*									*/
/************************************************************************/


static void
register_bio_mpx()
{
   Integer r[32];

   DTRACE("register_bio_mpx");

   ASHinq_read_mask(r);
   CMPXregister(CMPX_MASK_COUNT,r,NULL,NULL,BIOtrack);
};





/************************************************************************/
/*									*/
/*	track_fct -- mainloop for tracking function			*/
/*									*/
/************************************************************************/


static void
track_fct()
{
   for ( ; ; ) {
      if (intranscript != NULL) BIOtrack();
      else CMPXselect(0);
      THREADreschedule();
    };
};





/************************************************************************/
/*									*/
/*	get_transcript -- get input from transcript file		*/
/*	nontrans_event -- check if event shouldn't come from transcript */
/*									*/
/************************************************************************/


static void
get_transcript(e)
   BPIO_EVENT e;
{
   Integer i;
   Integer c,d,x,y,t0;
   struct timeval time;
   Integer mt;
   ASH_WINDOW w;

   DTRACE("get_transcript");

   if (nontrans_event(e)) return;
   if (e->BPIO_class == BPIO_CLASS_NONE && trans_time == 0) return;

   while (!have_trans_event) {
      i = fscanf(intranscript,TRANSCRIPT_FORMAT,&c,&d,&x,&y,&t0);
      if (i != 5) {
	 PROTECT;
	 fclose(intranscript);
	 intranscript = NULL;
	 UNPROTECT;
	 return;
       };
      have_trans_event = TRUE;
      trans_event.BPIO_class = (BPIO_CLASS) c;
      trans_event.BPIO_data = d;
      trans_event.BPIO_xpos = x;
      trans_event.BPIO_ypos = y;
      trans_event.BPIO_time = t0;
      if (trans_event.BPIO_class == BPIO_CLASS_NONE) have_trans_event = FALSE;
      if (nontrans_event(&trans_event)) have_trans_event = FALSE;
      if (trans_time == 0 && t0 == 0) have_trans_event = FALSE;
    };

   e->BPIO_class = BPIO_CLASS_NONE;

   gettimeofday(&time,NULL);

   if (trans_time == 0) {
      trans_time0 = time.tv_sec;
    };

   mt = (time.tv_sec - trans_time0)*1000 + time.tv_usec/1000;

   if (trans_time == 0) {
      trans_time = trans_event.BPIO_time - mt;
    };

   if (trans_event.BPIO_time - trans_time <= mt) {
      if (!ASHinq_valid_window(trans_root)) return;
      w = ASHfind_child_window(trans_root,trans_event.BPIO_xpos,trans_event.BPIO_ypos);
      if (w == NULL) return;
      *e = trans_event;
      have_trans_event = FALSE;
      e->BPIO_window = w;
      ASHmap(trans_root,e->BPIO_xpos,e->BPIO_ypos,w,&x,&y);
      if (trans_grab != NULL) w = trans_grab;
      else w = ASHinput_script(w,x,y);
      if (w == NULL) w = e->BPIO_window;
      if (w != e->BPIO_window) ASHmap(e->BPIO_window,x,y,w,&x,&y);
      e->BPIO_window = w;
      e->BPIO_xpos = x;
      e->BPIO_ypos = y;
      if (e->BPIO_class == BPIO_CLASS_MOUSE) {
	 trans_grab = e->BPIO_window;
       }
      else if (e->BPIO_class == BPIO_CLASS_UP_MOUSE) {
	 trans_grab = NULL;
       };
    };
};




static Boolean
nontrans_event(e)
   BPIO_EVENT e;
{
   if (e->BPIO_class == BPIO_CLASS_EVENT) {
      switch (e->BPIO_data) {
	 case BPIO_EVENT_REDRAW :
	 case BPIO_EVENT_VISIBLE :
	 case BPIO_EVENT_INVISIBLE :
	 case BPIO_EVENT_VIEW :
	 case BPIO_EVENT_RESIZE :
	    return TRUE;
       };
    };

   return FALSE;
};






/************************************************************************/
/*									*/
/*	setup_thread -- setup a thread control block for bio		*/
/*									*/
/************************************************************************/


static void
setup_thread(wd,th)
   BIO_WINDATA wd;
   THREAD th;
{
   register BIO_THREAD bt;
   register BIO_QUEUE q;

   DTRACE("setup_thread 0x%x 0x%x",wd,th);

   bt = PALLOC(BIO_THREAD_B);

   q = PALLOC(BIO_QUEUE_B);
   q->BIO_qhead = 0;
   q->BIO_qtail = 0;
   q->BIO_string = NULL;

   bt->BIO_queue = q;
   bt->BIO_thread_id = th;
   bt->BIO_windata = wd;
   bt->BIO_last_x = 0;
   bt->BIO_last_y = 0;
   bt->BIO_last_window = NULL;
   bt->BIO_last_time = 0;
   bt->BIO_waiting = FALSE;
   bt->BIO_sema = THREADseminit(0);

   THREADinq_input_data(th) = bt;

   wd->BIO_thread = bt;
};





/************************************************************************/
/*									*/
/*	bio_control -- control routine for BIO windows			*/
/*									*/
/************************************************************************/


static int
bio_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register BIO_WINDATA wd;

   DTRACE("bio_control %s 0x%x",msg,w);

   if (STREQL(msg,"ASH$REMOVE")) {
      wd = (BIO_WINDATA) ASHinq_input_data(w);
      if (wd != NULL) BIOremove_window(w);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	check_ignore -- check if we should ignore duplicate event	*/
/*									*/
/************************************************************************/


static Boolean
check_ignore(w,e)
   BIO_WINDATA w;
   BPIO_EVENT e;
{
   Boolean fg;
   BIO_QUEUE q;
   Integer i,j;

   DTRACE("check_ignore 0x%x 0x%x",w,e);

   if (e->BPIO_class != BPIO_CLASS_EVENT) return FALSE;

   switch (e->BPIO_data) {
      case BPIO_EVENT_REDRAW :
	 q = find_window_queue(w);
	 fg = FALSE;
	 if (q->BIO_qhead != q->BIO_qtail) {
	    for (i = 0; i < MAX_QUEUE_SIZE; ++i) {
	       j = (q->BIO_qhead+i)%MAX_QUEUE_SIZE;
	       if (j == q->BIO_qtail) break;
	       if (q->BIO_events[j].BPIO_class == BPIO_CLASS_EVENT &&
		      q->BIO_events[j].BPIO_data == BPIO_EVENT_REDRAW &&
		      q->BIO_events[j].BPIO_window == e->BPIO_window) fg = TRUE;
	       if (fg) break;
	     };
	  };
	 break;
      default :
	 fg = FALSE;
	 break;
    };

   return fg;
};





/* end of biowork.c */


