/************************************************************************/
/*									*/
/*		auxdsyn.c						*/
/*									*/
/*	Parser for auxilliary definitions module			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "auxd_local.h"




/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef enum _PARSE_TOKEN {
   TOKEN_UNKNOWN,
   TOKEN_EOF,
   TOKEN_ID,
   TOKEN_COLON,
   TOKEN_PLUS,
   TOKEN_SEMI,
   TOKEN_DOT,
   TOKEN_EQUALS,
   TOKEN_LPAREN,
   TOKEN_RPAREN,
   TOKEN_STRING,
   TOKEN_INT,
   TOKEN_REAL,
   TOKEN_AT,
} PARSE_TOKEN;




typedef union _PARSE_VALUE {
   Integer intval;
   String strval;
   Float * fltval;
   AUXD_FIELD fldval;
   Sequence seqval;
} PARSE_VALUE;



typedef struct _PARSE_DATA {
   FILE * file;
   Integer lineno;
   PARSE_TOKEN token;
   PARSE_VALUE value;
   AUXD location;
   String filename;
} PARSE_DATA_INFO, *PARSE_DATA;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	parse();
static	Boolean 	parse_def();
static	Boolean 	parse_value();
static	Boolean 	parse_list_value();
static	Boolean 	check_token();
static	void		next_token();
static	String		scan_string();
static	AUXD_FIELD	new_int();
static	AUXD_FIELD	new_float();
static	AUXD_FIELD	new_string();
static	AUXD_FIELD	new_seq();
static	AUXD_FIELD	new_field();





/************************************************************************/
/*									*/
/*	AUXD_syn_init -- module initialization				*/
/*									*/
/************************************************************************/


void
AUXD_syn_init()
{
};




/************************************************************************/
/*									*/
/*	AUXDuse -- use a file						*/
/*									*/
/************************************************************************/


void
AUXDuse(file)
   String file;
{
   FILE * inf;
   PARSE_DATA_INFO pdi;

   AUXD_init();

   inf = fopen(file,"r");

   if (inf == NULL) return;

   pdi.file = inf;
   pdi.lineno = 1;
   pdi.location = NULL;
   pdi.filename = file;

   next_token(&pdi);
   parse(&pdi);

   fclose(inf);
};





/************************************************************************/
/*									*/
/*	parse -- parse a auxilliary definition				*/
/*									*/
/************************************************************************/


static Boolean
parse(pdi)
   PARSE_DATA pdi;
{
   while (pdi->token != TOKEN_EOF) {
      if (!parse_def(pdi)) return FALSE;
    };

   return TRUE;
};





static Boolean
parse_def(pdi)
   PARSE_DATA pdi;
{
   PARSE_VALUE id,idh;
   AUXD_FIELD fld;
   AUXD new,old,inh;

   for ( ; ; ) {
      if (check_token(pdi,TOKEN_SEMI,NULL,FALSE)) break;
      if (check_token(pdi,TOKEN_EOF,NULL,FALSE)) break;

      if (!check_token(pdi,TOKEN_ID,&id,TRUE)) return FALSE;

      if (pdi->location != NULL && check_token(pdi,TOKEN_EQUALS,NULL,FALSE)) {
	 if (!parse_value(pdi,&fld)) return FALSE;
	 fld->name = id.strval;
	 fld = AUXD_add_field(pdi->location,fld);
       }
      else {
	 old = pdi->location;
	 if (check_token(pdi,TOKEN_COLON,NULL,FALSE))
	    new = AUXD_new_auxd(id.strval,old,FALSE);
	 else if (check_token(pdi,TOKEN_PLUS,NULL,TRUE))
	    new = AUXD_new_auxd(id.strval,old,TRUE);
	 else return FALSE;

	 if (check_token(pdi,TOKEN_AT,NULL,FALSE)) {
	    if (!check_token(pdi,TOKEN_ID,&idh,TRUE)) return FALSE;
	    inh = AUXDget_handle(old,idh.strval);
	    if (inh == NULL) inh = AUXDget_handle(NULL,idh.strval);
	    new->inherit = inh;
	  };

	 pdi->location = new;
	 parse_def(pdi);
	 pdi->location = old;
       };
    };

   return TRUE;
};




static Boolean
parse_value(pdi,fldp)
   PARSE_DATA pdi;
   AUXD_FIELD * fldp;
{
   PARSE_VALUE id;

   if (check_token(pdi,TOKEN_LPAREN,NULL,FALSE)) {
      return parse_list_value(pdi,fldp);
    }
   else if (check_token(pdi,TOKEN_ID,&id,FALSE)) {
      *fldp = new_string(id.strval);
    }
   else if (check_token(pdi,TOKEN_STRING,&id,FALSE)) {
      *fldp = new_string(id.strval);
    }
   else if (check_token(pdi,TOKEN_INT,&id,FALSE)) {
      *fldp = new_int(id.intval);
    }
   else if (check_token(pdi,TOKEN_REAL,&id,FALSE)) {
      *fldp = new_float(id.fltval);
    }
   else if (check_token(pdi,TOKEN_DOT,NULL,TRUE)) {
      *fldp = new_field(TYPE_UNDEFINED,FALSE);
    };

   return TRUE;
};





static Boolean
parse_list_value(pdi,fldp)
   PARSE_DATA pdi;
   AUXD_FIELD * fldp;
{
   AUXD_TYPE ty;
   Sequence l;
   PARSE_VALUE id;

   ty = TYPE_UNDEFINED;
   l = NULL;

   for ( ; ; ) {
      if (check_token(pdi,TOKEN_ID,&id,FALSE) ||
	     check_token(pdi,TOKEN_STRING,&id,FALSE)) {
	 if (ty == TYPE_UNDEFINED) ty = TYPE_STRING;
	 if (ty != TYPE_STRING) {
	    AUXD_error("Can't mix types in a list");
	    return FALSE;
	  };
	 l = APPEND(id.strval,l);
       }
      else if (check_token(pdi,TOKEN_INT,&id,FALSE)) {
	 if (ty == TYPE_UNDEFINED) ty = TYPE_INTEGER;
	 if (ty != TYPE_INTEGER) {
	    AUXD_error("Can't mix types in a list");
	    return FALSE;
	  };
	 l = APPEND(id.intval,l);
       }
      else if (check_token(pdi,TOKEN_REAL,&id,FALSE)) {
	 if (ty == TYPE_UNDEFINED) ty = TYPE_FLOAT;
	 if (ty != TYPE_FLOAT) {
	    AUXD_error("Can't mix types in a list");
	    return FALSE;
	  };
	 l = APPEND(id.fltval,l);
       }
      else break;
    };

   if (!check_token(pdi,TOKEN_RPAREN,NULL,TRUE)) return FALSE;

   if (ty == TYPE_UNDEFINED) ty = TYPE_STRING;

   *fldp = new_seq(l,ty);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_token -- check for current token				*/
/*									*/
/************************************************************************/


static Boolean
check_token(pdi,tok,valp,fg)
   PARSE_DATA pdi;
   PARSE_TOKEN tok;
   PARSE_VALUE * valp;
   Boolean fg;
{
   Boolean rfg;

   rfg = (pdi->token == tok);

   if (rfg) {
      if (valp != NULL) *valp = pdi->value;
      next_token(pdi);
    }
   else if (fg) AUXD_error("Syntax error in input definitions at line %d of %s",
			      pdi->lineno,pdi->filename);

   return rfg;
};




/************************************************************************/
/*									*/
/*	next_token -- get next token					*/
/*									*/
/************************************************************************/


static void
next_token(pdi)
   PARSE_DATA pdi;
{
   Integer ch,i;
   Integer val,sgn,dct,dct1;
   Float fval,x,y;
   Character buf[1024];

   pdi->value.strval = NULL;

   for ( ; ; ) {
      ch = getc(pdi->file);
      if (ch == EOF) {
	 pdi->token = TOKEN_EOF;
	 return;
       };
      if (isspace(ch)) {
	 if (ch == '\n') ++ pdi->lineno;
	 continue;
       }
      if (ch == '-') {
	 ch = getc(pdi->file);
	 if (ch != '-') {
	    ungetc(ch,pdi->file);
	    ch = '-';
	  }
	 else {
	    do {
	       ch = getc(pdi->file);
	     }
	    while (ch != '\n' && ch != EOF);
	    if (ch == EOF) {
	       pdi->token = TOKEN_EOF;
	       return;
	     };
	    ungetc(ch,pdi->file);
	    continue;
	  };
       };
      break;
    };

   if (index("=+:();.@",ch) != NULL) {
      switch (ch) {
	 case '=' :
	    pdi->token = TOKEN_EQUALS;
	    break;
	 case '+' :
	    pdi->token = TOKEN_PLUS;
	    break;
	 case ':' :
	    pdi->token = TOKEN_COLON;
	    break;
	 case '(' :
	    pdi->token = TOKEN_LPAREN;
	    break;
	 case ')' :
	    pdi->token = TOKEN_RPAREN;
	    break;
	 case ';' :
	    pdi->token = TOKEN_SEMI;
	    break;
	 case '.' :
	    pdi->token = TOKEN_DOT;
	    break;
	 case '@' :
	    pdi->token = TOKEN_AT;
	    break;
       };
    }
   else if (isdigit(ch) || ch == '-') {
      if (ch == '-') {
	 sgn = -1;
	 ch = getc(pdi->file);
       }
      else sgn = 1;

      val = 0;
      fval = 0.0;
      while (isdigit(ch)) {
	 val = val*10 + ch - '0';
	 fval = fval*10.0 + (ch - '0');
	 ch = getc(pdi->file);
       };

      if (ch == 'x' && val == 0) {
	 ch = getc(pdi->file);
	 while (isalnum(ch)) {
	    if (isdigit(ch)) val = val*16 + ch - '0';
	    else if (ch >= 'a' && ch <= 'f') val = val*16 + 10 + ch - 'a';
	    else if (ch >= 'A' && ch <= 'F') val = val*16 + 10 + ch - 'A';
	    else break;
	    ch = getc(pdi->file);
	  };
	 ungetc(ch,pdi->file);
	 pdi->value.intval = val*sgn;
	 pdi->token = TOKEN_INT;
       }
      else if (ch != '.' && ch != 'e' && ch != 'E') {
	 ungetc(ch,pdi->file);
	 pdi->value.intval = val*sgn;
	 pdi->token = TOKEN_INT;
       }
      else {
	 dct = 0;
	 if (ch == '.') {
	    ch = getc(pdi->file);
	    while (isdigit(ch)) {
	       fval = fval*10.0 + (ch - '0');
	       --dct;
	       ch = getc(pdi->file);
	     };
	  };
	 if (ch == 'E' || ch == 'e') {
	    ch = getc(pdi->file);
	    val = 0;
	    while (isdigit(ch)) {
	       val = val*10 + ch - '0';
	       ch = getc(pdi->file);
	     };
	    dct += val;
	  };

	 if (dct != 0) {
	    dct1 = abs(dct);
	    x = 1.0;
	    y = 10.0;
	    while (dct1 > 0) {
	       if (dct1 & 1) x *= y;
	       y *= y;
	       dct1 >>= 1;
	     };
	    if (dct > 0) fval *= x;
	    else fval /= x;
	  };

	 ungetc(ch,pdi->file);

	 pdi->value.fltval = PALLOC(Float);
	 *pdi->value.fltval = fval*sgn;
	 pdi->token = TOKEN_REAL;
       };
    }
   else if (ch == '"') {
      scan_string(pdi,buf);
      pdi->value.strval = SALLOC(buf);
      pdi->token = TOKEN_STRING;
    }
   else if (isalpha(ch) || index("_@$?",ch) != NULL) {
      i = 0;
      while (isalpha(ch) || isdigit(ch) || index("_@$?",ch) != NULL) {
	 buf[i++] = ch;
	 ch = getc(pdi->file);
       };
      buf[i] =0;
      ungetc(ch,pdi->file);
      pdi->value.strval = SALLOC(buf);
      pdi->token = TOKEN_ID;
    }
   else {
      pdi->token = TOKEN_UNKNOWN;
    };
};





/************************************************************************/
/*									*/
/*	scan_string -- scan and edit a string				*/
/*									*/
/************************************************************************/


static String
scan_string(pdi,txt)
   PARSE_DATA pdi;
   String txt;
{
   String t;
   Integer i;
   Integer ch;

   t = txt;
   ch = getc(pdi->file);

   while (ch != EOF && ch != '"') {
      if (ch != '\\') {
	 i = ch;
	 ch = getc(pdi->file);
       }
      else {
	 ch = getc(pdi->file);
	 if (isdigit(ch)) {
	    i = ch - '0';
	    ch = getc(pdi->file);
	    if (isdigit(ch)) {
	       i = i*8 + ch - '0';
	       ch = getc(pdi->file);
	     };
	    if (isdigit(ch)) {
	       i = i*8 + ch - '0';
	       ch = getc(pdi->file);
	     };
	  }
	 else {
	    i = ch;
	    ch = getc(pdi->file);
	    switch (i) {
	       case 'n' :
		  i = '\n';
		  break;
	       case 'r' :
		  i = '\r';
		  break;
	       case 't' :
		  i = '\t';
		  break;
	       case 'b' :
		  i = '\t';
		  break;
	       case 'f' :
		  i = '\f';
		  break;
	       default :
		  break;
	     };
	  };
       };
      *t++ = i;
    };

   *t = 0;

   return txt;
};





/************************************************************************/
/*									*/
/*	new_int -- create integer field 				*/
/*	new_float -- create real field					*/
/*	new_string -- create string field				*/
/*	new_seq -- create list field					*/
/*									*/
/************************************************************************/


static AUXD_FIELD
new_int(v)
   Integer v;
{
   AUXD_FIELD ff;

   ff = new_field(TYPE_INTEGER,FALSE);

   ff->value.intval = v;

   return ff;
};






static AUXD_FIELD
new_float(v)
   Float* v;
{
   AUXD_FIELD ff;

   ff = new_field(TYPE_FLOAT,FALSE);

   ff->value.fltval = v;

   return ff;
};






static AUXD_FIELD
new_string(v)
   String v;
{
   AUXD_FIELD ff;

   ff = new_field(TYPE_STRING,FALSE);

   ff->value.strval = v;

   return ff;
};






static AUXD_FIELD
new_seq(v,typ)
   Sequence v;
   AUXD_TYPE typ;
{
   AUXD_FIELD ff;

   ff = new_field(typ,TRUE);

   ff->value.seqval = v;

   return ff;
};






/************************************************************************/
/*									*/
/*	new_field -- create a generic field				*/
/*									*/
/************************************************************************/


static AUXD_FIELD
new_field(typ,lstfg)
   AUXD_TYPE typ;
   Boolean lstfg;
{
   AUXD_FIELD ff;

   ff = PALLOC(AUXD_FIELD_INFO);

   ff->name = NULL;
   ff->type = typ;
   ff->listfg = lstfg;
   ff->next = NULL;
   ff->value.intval = 0;

   return ff;
};






/* end of auxdsyn.c */


