/************************************************************************/
/*									*/
/*		auxdinfo.c						*/
/*									*/
/*	Internal code for auxilliary definitions package		*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "auxd_local.h"



/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	AUXD		root = NULL;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	AUXD_FIELD		find_field();





/************************************************************************/
/*									*/
/*	AUXD_info_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
AUXD_info_init()
{
   root = NULL;
};





/************************************************************************/
/*									*/
/*	AUXDget_handle -- return handle for string			*/
/*	AUXDnext_handle -- return next handle for handle		*/
/*									*/
/************************************************************************/


AUXD
AUXDget_handle(base,s)
   AUXD base;
   String s;
{
   register AUXD auxd,first;

   AUXD_init();

   first = (base == NULL) ? root : base->son;

   for (auxd = first; auxd != NULL; auxd = auxd->next) {
      if (STREQL(auxd->name,s)) break;
    };

   return auxd;
};





AUXD
AUXDnext_handle(f)
   AUXD f;
{
   return f->nextalt;
};





/************************************************************************/
/*									*/
/*	AUXDget_info -- return string value for handle			*/
/*	AUXDget_info_int -- return integer value for handle		*/
/*	AUXDget_info_real -- return real value for handle		*/
/*	AUXDget_defined -- check if field is defined			*/
/*									*/
/************************************************************************/


char *
AUXDget_info(flx,name)
   AUXD flx;
   String name;
{
   register AUXD_FIELD fld;

   fld = find_field(flx,name,TYPE_STRING,FALSE);

   if (fld == NULL) return NULL;

   return fld->str_value;
};






int
AUXDget_info_int(flx,name)
   AUXD flx;
   String name;
{
   register AUXD_FIELD fld;

   fld = find_field(flx,name,TYPE_INTEGER,FALSE);

   if (fld == NULL) return 0;

   return fld->int_value;
};






double
AUXDget_info_real(flx,name)
   AUXD flx;
   String name;
{
   register AUXD_FIELD fld;
   double d;

   fld = find_field(flx,name,TYPE_UNDEFINED,FALSE);

   if (fld == NULL) d = 0.0;
   else if (fld->type == TYPE_FLOAT) return d = *fld->flt_value;
   else if (fld->type == TYPE_INTEGER) d = fld->int_value;
   else {
      AUXD_error("Bad type expected for field %s",name);
      d = 0.0;
    };

   return d;
};





int
AUXDget_defined(flx,name)
   AUXD flx;
   String name;
{
   register AUXD_FIELD fld;

   fld = find_field(flx,name,TYPE_UNDEFINED,FALSE);

   return (fld != NULL);
};







/************************************************************************/
/*									*/
/*	AUXDget_info_list -- return string list for handle		*/
/*	AUXDget_info_list_int -- return integer list for handle 	*/
/*	AUXDget_info_list_real -- return real list for handle		*/
/*									*/
/************************************************************************/


int
AUXDget_info_list(flx,name,max,data)
   AUXD flx;
   String name;
   Integer max;
   String data[];
{
   register AUXD_FIELD fld;
   register Sequence l;
   register Integer ct;
   register String s;

   fld = find_field(flx,name,TYPE_STRING,TRUE);

   if (fld == NULL) return 0;

   ct = 0;
   if (fld->listfg) {
      forin (s,String,l,fld->seq_value) {
	 if (ct < max) data[ct++] = s;
       };
    }
   else if (ct < max) data[ct++] = fld->str_value;

   return ct;
};






int
AUXDget_info_list_int(flx,name,max,data)
   AUXD flx;
   String name;
   Integer max;
   Integer data[];
{
   register AUXD_FIELD fld;
   register Sequence l;
   register Integer ct;
   register Integer v;

   fld = find_field(flx,name,TYPE_INTEGER,TRUE);

   if (fld == NULL) return 0;

   ct = 0;
   if (fld->listfg) {
      forin (v,Integer,l,fld->seq_value) {
	 if (ct < max) data[ct++] = v;
       };
    }
   else if (ct < max) data[ct++] = fld->int_value;

   return ct;
};






int
AUXDget_info_list_real(flx,name,max,data)
   AUXD flx;
   String name;
   Integer max;
   Float data[];
{
   register AUXD_FIELD fld;
   register Sequence l;
   register Integer ct;
   register Float* v;

   fld = find_field(flx,name,TYPE_FLOAT,TRUE);

   if (fld == NULL) return 0;

   ct = 0;
   if (fld->listfg) {
      forin (v,Float*,l,fld->seq_value) {
	 if (ct < max) data[ct++] = *v;
       };
    }
   else if (ct < max) data[ct++] = *fld->flt_value;

   return ct;
};






/************************************************************************/
/*									*/
/*	AUXDdecode -- decode a string, doing AUXD replacements		*/
/*									*/
/************************************************************************/


char *
AUXDdecode(src,ln,obuf)
   String src;
   Integer ln;
   String obuf;
{
   String p,q,s;
   Character tbuf[10240],nbuf[256];
   AUXD hdl;

   if (src[0] != AUXD_STRING_CHAR) return src;

   if (src[1] == AUXD_STRING_CHAR) s = &src[2];
   else s = src;

   if (obuf == NULL) {
      p = tbuf;
      ln = 10240;
    }
   else p = obuf;

   while (*s != 0) {
      if (*s != AUXD_STRING_CHAR) {
	 if (--ln > 0) *p++ = *s++;
       }
      else {
	 hdl = NULL;
	 for ( ; ; ) {
	    ++s;
	    q = nbuf;
	    while (*s != 0 && *s != AUXD_STRING_CHAR && *s != '.') *q++ = *s++;
	    *q = 0;
	    if (*s != '.') break;
	    hdl = AUXDget_handle(hdl,nbuf);
	    ++s;
	  };
	 if (*s == AUXD_STRING_CHAR) ++s;
	 q = AUXDget_info(hdl,nbuf);
	 if (q == NULL) {
	    AUXD_error("Item not found in decoding <%s>",src);
	  }
	 else {
	    while (*q != 0) if (--ln > 0) *p++ = *q++;
	  };
       };
    };

   *p = 0;

   if (obuf == NULL) obuf = SALLOC(tbuf);

   return obuf;
};





/************************************************************************/
/*									*/
/*	AUXD_new_auxd -- create new AUXD_INFO block			*/
/*									*/
/************************************************************************/


AUXD
AUXD_new_auxd(nm,par,addfg)
   String nm;
   AUXD par;
   Boolean addfg;
{
   register AUXD fi,fnd,lst;

   PROTECT;

   lst = NULL;
   for (fi = (par == NULL ? root : par->son); fi != NULL; fi = fi->next) {
      if (STREQL(nm,fi->name)) break;
      lst = fi;
    };

   fnd = fi;

   if (fi == NULL || addfg) {
      fi = PALLOC(AUXD_INFO);
      fi->name = nm;
      fi->listfg = TRUE;
      fi->next = NULL;
      fi->nextalt = NULL;
      fi->son = NULL;
      fi->field = NULL;
      fi->inherit = NULL;
    };

   if (fnd == NULL) {
      if (lst != NULL) {
	 fi->next = lst->next;
	 lst->next = fi;
       }
      else if (par == NULL) {
	 fi->next = root;
	 root = fi;
       }
      else {
	 fi->next = par->son;
	 par->son = fi;
       };
    }
   else if (addfg) {
      while (fnd->nextalt != NULL) fnd = fnd->nextalt;
      fnd->nextalt = fi;
    };

   UNPROTECT;

   return fi;
};




/************************************************************************/
/*									*/
/*	AUXD_add_field -- add field to AUXD definition			*/
/*									*/
/************************************************************************/


AUXD_FIELD
AUXD_add_field(ad,fld)
   AUXD ad;
   AUXD_FIELD fld;
{
   register AUXD_FIELD ff,lf;

   PROTECT;

   lf = NULL;
   for (ff = ad->field; ff != NULL; ff = ff->next) {
      if (STREQL(ff->name,fld->name)) break;
      lf = ff;
    };

   if (ff != NULL) {
      ff->listfg = fld->listfg;
      ff->type = fld->type;
      ff->value = fld->value;
      free(fld);
    }
   else if (lf != NULL) {
      ff = fld;
      fld->next = lf->next;
      lf->next = fld;
    }
   else {
      ff = fld;
      fld->next = ad->field;
      ad->field = fld;
    };

   UNPROTECT;

   return ff;
};





/************************************************************************/
/*									*/
/*	find_field -- find a field in given AUXD_INFO			*/
/*									*/
/************************************************************************/


static AUXD_FIELD
find_field(fi,nm,typ,lstfg)
   AUXD fi;
   String nm;
   AUXD_TYPE typ;
   Boolean lstfg;
{
   register AUXD_FIELD ff;

   if (fi == NULL) return NULL;

   for (ff = fi->field; ff != NULL; ff = ff->next) {
      if (STREQL(ff->name,nm)) break;
    };

   if (ff == NULL && fi->inherit != NULL && fi->inherit != fi) {
      ff = find_field(fi->inherit,nm,typ,lstfg);
    };

   if (ff != NULL && typ != TYPE_UNDEFINED) {
      if (ff->type != typ) {
	 AUXD_error("Bad type expected for field %s",nm);
       };
      if (ff->listfg && !lstfg) {
	 AUXD_error("List field %s used in non-list context",nm);
       };
    };

   return ff;
};






/* end of auxdinfo.c */
