/************************************************************************/
/*									*/
/*		ashwindraw.c						*/
/*									*/
/*	Window support for drawing					*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Integer 	batch_level = 0;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	ASH_windraw_init -- initialize window drawing module		*/
/*	ASH_windraw_disp_init -- initialize new display 		*/
/*									*/
/************************************************************************/


void
ASH_windraw_init()
{
   batch_level = 0;
};






/*ARGSUSED*/

void
ASH_windraw_disp_init(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   XGCValues vals;
   Integer mask;

   PROTECT;

   vals.foreground = d->foreg;
   vals.background = d->backg;
   mask = GCForeground|GCBackground;
   d->data->refresh_context = XCreateGC(d->display,
					   RootWindow(d->display,d->screen),
					   mask,&vals);

   vals.foreground = d->backg;
   vals.background = d->foreg;
   vals.subwindow_mode = IncludeInferiors;
   mask = GCSubwindowMode|GCForeground|GCBackground;
   d->data->save_context = XCreateGC(d->display,
					   RootWindow(d->display,d->screen),
					   mask,&vals);

   vals.foreground = d->backg;
   vals.background = d->foreg;
   mask = GCForeground|GCBackground;
   d->data->clear_context = XCreateGC(d->display,
					   RootWindow(d->display,d->screen),
					   mask,&vals);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHbatch_mode -- begin/end batch mode				*/
/*									*/
/************************************************************************/


void
ASHbatch_mode(fg)
   Boolean fg;
{
   TRACE("ASHbatch_mode %d",fg);

   if (fg) {
      PROTECT;
      ++batch_level;
      UNPROTECT;
    }
   else if (batch_level > 0) {
      PROTECT;
      --batch_level;
      UNPROTECT;
      if (batch_level == 0) ASH_flush_output(NULL);
    };
};





/************************************************************************/
/*									*/
/*	ASHsync -- synchronize with X windows				*/
/*									*/
/************************************************************************/


void
ASHsync(w)
   ASH_WINDOW w;
{
   if (w == NULL) w = ASHinq_top();

   PROTECT;
   XSync(DISPLAYOF(w),0);
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHclear_damage -- remove damage indicators for window		*/
/*	ASHfix_damage -- fix damaged portions of window 		*/
/*	ASHregister_damage -- note a damanged rectangle 		*/
/*									*/
/************************************************************************/


void
ASHclear_damage(w)
   ASH_WINDOW w;
{
   ASH_flush_output(w);

   if (ASH__debugf != NULL) {
      fprintf(ASH__debugf,"ASH: clear 0x%x",w);
    };

   w->damage = NULL;
};





void
ASHfix_damage(w)
   ASH_WINDOW w;
{
   CLIP_RECT fcr,cr,ncr;
   Integer lx,by,rx,ty;
   Integer wd,ht;
   Integer flx,fty;
   Boolean retfg;

   if (w == NULL || w->safety != SAFETY || w->removed) return;

   if (ASH__debugf != NULL) {
      fprintf(ASH__debugf,"ASH: fix 0x%x\n",w);
      for (cr = w->damage; cr != NULL; cr = cr->next) {
	 fprintf(ASH__debugf,"\t(%d,%d)->(%d,%d)\n",
		    cr->rect.lx,cr->rect.by,cr->rect.rx,cr->rect.ty);
       };
    };

   if (w->damage == NULL) return;


   if (w->refresh == NULL && w->noclear && w->full_win == NULL) return;

   if (!ASHlock(w)) return;
   if (!ASH_lock_refresh(w)) {
      ASHunlock(w);
      return;
    };

   fcr = w->damage;
   w->damage = NULL;

   ASHbatch_mode(TRUE);

   retfg = (!w->frame && w->full_win != NULL && !w->retained);

   if (!retfg) ASHpush_drawinfo(w,NULL);

   for (cr = fcr; cr != NULL; cr = ncr) {
      ncr = cr->next;
      if (retfg) {
	 lx = XMAPVIEW(w,cr->rect.lx);
	 by = YMAPVIEW(w,cr->rect.by);
	 rx = XMAPVIEW(w,cr->rect.rx);
	 ty = YMAPVIEW(w,cr->rect.ty);
	 wd = rx-lx+1;
	 ht = by-ty+1;
	 flx = XMAPVFULL(w,lx);
	 fty = YMAPVFULL(w,ty);
	 PROTECT;
	 XCopyArea(DISPLAYOF(w),w->full_win,w->view_win,w->display->data->refresh_context,
		      flx,fty,wd,ht,lx,ty);
	 UNPROTECT;
       }
      else if (!w->region_refresh) ;
      else if (cr == fcr) {
	 ASHouter_clip_region(w,cr->rect.lx,cr->rect.by,cr->rect.rx,cr->rect.ty);
       }
      else {
	 ASHouter_clip_add_region(w,cr->rect.lx,cr->rect.by,cr->rect.rx,cr->rect.ty);
       };
      free(cr);
    };

   if (!retfg) {
      if (w->region_refresh) ASHouter_clip(w,TRUE);
      if (!w->noclear) {
	 ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
	 ASHclear_box(w,lx,by,rx,ty);
       };
      if (w->refresh != NULL) (*w->refresh)(w,lx,by,rx,ty);
      if (ASHinq_valid_window(w)) ASHpop_drawinfo(w);
    };

   ASHbatch_mode(FALSE);

   ASHunlock(w);
};





int
ASHregister_damage(w,more,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer more;
   Integer lx,by,rx,ty;
{
   CLIP_RECT cr;
   ASH_WINDOW p;
   Boolean fg;

   if (ASH__debugf != NULL) {
      fprintf(ASH__debugf,"ASH: register 0x%x (%d,%d)->(%d,%d) [%d]\n",
		 w,lx,by,rx,ty,more);
    };

   if (w == NULL || w->safety != SAFETY || w->removed) return FALSE;

   cr = PALLOC(CLIP_RECT_B);
   cr->rect.lx = lx;
   cr->rect.by = by;
   cr->rect.rx = rx;
   cr->rect.ty = ty;
   PROTECT;
   cr->next = w->damage;
   w->damage = cr;
   UNPROTECT;

   fg = TRUE;
   if (more == 0) {
      for (p = w; p != NULL; p = p->parent) {
	 if (p->inputdata != NULL) break;
       };
      if (p == NULL) ASHfix_damage(w);
      else fg = FALSE;
    };

   return fg;
};






/************************************************************************/
/*									*/
/*	ASHfix_visible -- handle window becoming visible		*/
/*	ASHfix_invisible -- handle window becoming invisible		*/
/*	ASHfix_configure -- handle window configuration changes 	*/
/*									*/
/************************************************************************/


void
ASHfix_visible(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY || w->removed) return;

   if (w->parent == NULL && !w->visible) ASHvisible(w,TRUE);
};





void
ASHfix_invisible(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY || w->removed) return;

   if (w->parent == NULL && w->visible) ASHvisible(w,FALSE);
};





void
ASHfix_configure(w)
   ASH_WINDOW w;
{
   Integer x,y,wd,ht,bw,dp;
   Window root;
   Integer vfg;
   Boolean rsfg;

   if (w == NULL || w->safety != SAFETY || w->removed) return;

   if (!w->resize && w->parent == NULL) {
      PROTECT;
      XGetGeometry(DISPLAYOF(w),w->x_win,&root,&x,&y,&wd,&ht,&bw,&dp);
      UNPROTECT;
      if (wd != abs(w->view.rx - w->view.lx)+1 ||
	     ht != abs(w->view.by - w->view.ty)+1 ||
	     x != w->base.x ||
	     y+ht-1+2*bw != w->base.y) {
	 rsfg = FALSE;
	 if (wd != abs(w->view.rx - w->view.lx)+1 ||
		ht != abs(w->view.by - w->view.ty)+1) rsfg = TRUE;
	 w->base.x = x;
	 w->base.y = y+ht-1+2*bw;
	 w->coords.lx = 0;
	 w->coords.by = ht-1;
	 w->coords.rx = wd-1;
	 w->coords.ty = 0;
	 w->view = w->coords;
	 w->inv_x = FALSE;
	 w->inv_y = FALSE;
	 vfg = w->visible;

	 ASH_setup_maps(w,rsfg,vfg,FALSE);
       };
    };
};






/************************************************************************/
/*									*/
/*	ASH_inq_batched -- check if inside batch mode			*/
/*									*/
/************************************************************************/


Boolean
ASH_inq_batched()
{
   return (batch_level > 0);
};





/************************************************************************/
/*									*/
/*	ASH_draw_begin -- start a drawing operation			*/
/*	ASH_draw_end -- finish a drawing operation			*/
/*									*/
/************************************************************************/


void
ASH_draw_begin(w,ifg)
   ASH_WINDOW w;
   Boolean ifg;
{
   if (ifg && !w->inited && w->safety != PROP_SAFETY) ASH_setup_window(w);

   PROTECT;

   if (ifg && w->safety != PROP_SAFETY) ASH_desensitize(w);
};




void
ASH_draw_end(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   UNPROTECT;

   if (fg && batch_level == 0) ASH_flush_output(w);
};





/************************************************************************/
/*									*/
/*	ASH_window_to_pixmap -- get a pixmap from an ash window 	*/
/*									*/
/************************************************************************/


Pixmap
ASH_window_to_pixmap(w,wd,ht,disp)
   ASH_WINDOW w;
   Integer wd,ht;
   DISPLAY disp;
{
   Pixmap p,p1;
   Drawable d;
   Integer lx,ty;

   if (wd == 0) wd = abs(w->coords.rx - w->coords.lx) +1;
   if (ht == 0) ht = abs(w->coords.by - w->coords.ty) +1;

   if (w->full_win != NULL) d = (Drawable) w->full_win;
   else d = (Drawable) w->view_win;

   PROTECT;

   p = XCreatePixmap(DISPLAYOF(w),w->x_win,wd,ht,w->depth);

   lx = XMAPVIEW(w,w->coords.lx);
   ty = YMAPVIEW(w,w->coords.ty);

   XCopyArea(DISPLAYOF(w),d,p,w->display->data->refresh_context,lx,ty,wd,ht,0,0);

   if (disp != NULL && w->display->display != disp->display) {
      p1 = ASH_copy_pixmap(w->display,p,disp);
      XFreePixmap(DISPLAYOF(w),p);
      p = p1;
    };

   UNPROTECT;

   return p;
};





/************************************************************************/
/*									*/
/*	ASH_make_pixmap -- make 1-plane pixmap from array of Shorts	*/
/*									*/
/************************************************************************/


Pixmap
ASH_make_pixmap(d,pat)
   DISPLAY d;
   unsigned short pat[16];
{
   XImage * im;
   Pixmap p;
   GC gc;
   XGCValues vals;
   unsigned int npat[32];
   Integer i;

   PROTECT;

   for (i = 0; i < 16; ++i) {
      npat[i] = (pat[i]<<16) | pat[i];
      npat[16+i] = npat[i];
    };

   im = XCreateImage(d->display,
			d->default_visual,
			1,
			XYPixmap,
			0,
			npat,
			32, 32, 32, 0);

   im->byte_order = BYTE_ORDER;
   im->bitmap_bit_order = BIT_ORDER;

   p = XCreatePixmap(d->display,RootWindow(d->display,d->screen),32,32,1);
   gc = XCreateGC(d->display,p,0,&vals);

   XPutImage(d->display,p,gc,im,0,0,0,0,32,32);
   XFreeGC(d->display,gc);

   XFree(im);

   UNPROTECT;

   return p;
};





/************************************************************************/
/*									*/
/*	ASH_window_refresh -- refresh a window				*/
/*									*/
/************************************************************************/


void
ASH_window_refresh(w)
   ASH_WINDOW w;
{
   Integer lx,ty,wd,ht;

   if (!w->visible || !w->inited) return;

   ASHbatch_mode(TRUE);

   if (!w->saved) {
      if (w->refresh != NULL) {
	 PROTECT;
	 XClearWindow(DISPLAYOF(w),w->x_win);
	 UNPROTECT;
	 ASHpush_drawinfo(w,NULL);
	 (*w->refresh)(w,w->view.lx,w->view.by,w->view.rx,w->view.ty);
	 ASHpop_drawinfo(w);
       }
      else if (w->noclear) ;
      else if (!w->frame) {
	 PROTECT;
	 XClearWindow(DISPLAYOF(w),w->x_win);
	 UNPROTECT;
       }
      else {
	 ASHclear(w);
       };
    }
   else if (!w->frame && w->full_win != NULL && !w->retained) {
      lx = XMAPVFULL(w,0);
      ty = YMAPVFULL(w,0);
      wd = abs(w->view.lx - w->view.rx) +1;
      ht = abs(w->view.ty - w->view.by) +1;
      PROTECT;
      XCopyArea(DISPLAYOF(w),w->full_win,w->view_win,w->display->data->refresh_context,
		   lx,ty,wd,ht,0,0);
      UNPROTECT;
    }

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	ASH_region_refresh -- refresh a window region			*/
/*									*/
/************************************************************************/


void
ASH_region_refresh(w,rlx,rty,rwd,rht)
   ASH_WINDOW w;
   Integer rlx,rty;
   Integer rwd,rht;
{
   Integer lx,by,rx,ty;

   if (!w->visible || !w->inited) return;

   ASHbatch_mode(TRUE);

   if (!w->saved) {
      lx = XUNMAPVIEW(w,rlx);
      rx = XUNMAPVIEW(w,rlx+rwd-1);
      ty = YUNMAPVIEW(w,rty);
      by = YUNMAPVIEW(w,rty+rht-1);
      ASHpush_drawinfo(w,NULL);
      ASHouter_clip_region(w,lx,by,rx,ty);
      ASHouter_clip(w,TRUE);
      if (!w->noclear) ASHclear_box(w,lx,by,rx,ty);
      if (w->refresh != NULL) (*w->refresh)(w,lx,by,rx,ty);
      ASHpop_drawinfo(w);
    }
   else if (!w->frame && w->full_win != NULL && !w->retained) {
      lx = XMAPVFULL(w,rlx);
      ty = YMAPVFULL(w,rty);
      PROTECT;
      XCopyArea(DISPLAYOF(w),w->full_win,w->view_win,w->display->data->refresh_context,
		   lx,ty,rwd,rht,rlx,rty);
      UNPROTECT;
    }

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	ASH_fix_clip -- fix up clipping rectangles			*/
/*									*/
/************************************************************************/


void
ASH_fix_clip(w)
   ASH_WINDOW w;
{
   XRectangle rect;
   Integer ct;
   CLIP_RECT cr;
   Region r,r1,rf,rf1;

   PROTECT;

   if (!w->draw->use_clip && ! w->draw->out_clip) ct = -1;
   else if (w->draw->clip.type == CLIP_MAP || w->draw->outer_clip.type == CLIP_MAP) {
      ct = -1;
      ERROR("Can't do window clipping yet");
    }
   else {
      if (w->draw->out_clip) {
	 r1 = XCreateRegion();
	 if (w->full_win != NULL) rf1 = XCreateRegion();
	 for (cr = w->draw->outer_clip.rect; cr != NULL; cr = cr->next) {
	    rect.x = XMAPVIEW(w,cr->rect.lx);
	    rect.y = YMAPVIEW(w,cr->rect.ty);
	    rect.width = XMAPVIEW(w,cr->rect.rx) - rect.x + 1;
	    rect.height = YMAPVIEW(w,cr->rect.by) - rect.y + 1;
	    XUnionRectWithRegion(&rect,r1,r1);
	    if (w->full_win != NULL) {
	       rect.x = XMAPVFULL(w,rect.x);
	       rect.y = YMAPVFULL(w,rect.y);
	       XUnionRectWithRegion(&rect,rf1,rf1);
	     };
	  };
	 if (!w->draw->use_clip) {
	    r = r1;
	    rf = rf1;
	  };
       };

      if (w->draw->use_clip) {
	 r = XCreateRegion();
	 if (w->full_win != NULL) rf = XCreateRegion();
	 for (cr = w->draw->clip.rect; cr != NULL; cr = cr->next) {
	    rect.x = XMAPVIEW(w,cr->rect.lx);
	    rect.y = YMAPVIEW(w,cr->rect.ty);
	    rect.width = XMAPVIEW(w,cr->rect.rx) - rect.x + 1;
	    rect.height = YMAPVIEW(w,cr->rect.by) - rect.y + 1;
	    XUnionRectWithRegion(&rect,r,r);
	    if (w->full_win != NULL) {
	       rect.x = XMAPVFULL(w,rect.x);
	       rect.y = YMAPVFULL(w,rect.y);
	       XUnionRectWithRegion(&rect,rf,rf);
	      };
	  };
	 if (w->draw->out_clip) {
	    XIntersectRegion(r1,r,r);
	    XDestroyRegion(r1);
	    if (w->full_win) {
	       XIntersectRegion(rf1,rf,rf);
	       XDestroyRegion(rf1);
	     };
	  };
       };

      XSetRegion(DISPLAYOF(w),w->draw->context,r);
      XSetRegion(DISPLAYOF(w),w->draw->text_context,r);
      XDestroyRegion(r);

      if (w->full_win != NULL) {
	 XSetRegion(DISPLAYOF(w),w->draw->full_context,rf);
	 XSetRegion(DISPLAYOF(w),w->draw->full_text_context,rf);
	 XDestroyRegion(rf);
       };

      ct = 1;
    };


   if (ct < 0) {
      XSetClipMask(DISPLAYOF(w),w->draw->context,None);
      XSetClipMask(DISPLAYOF(w),w->draw->text_context,None);
      if (w->draw->full_context != NULL) {
	 XSetClipMask(DISPLAYOF(w),w->draw->full_context,None);
	 XSetClipMask(DISPLAYOF(w),w->draw->full_text_context,None);
       };
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_copy_pixmap -- copy pixmap between displays 		*/
/*									*/
/************************************************************************/


Pixmap
ASH_copy_pixmap(fromd,pix,tod)
   DISPLAY fromd;
   Pixmap pix;
   DISPLAY tod;
{
   Pixmap npix;
   Integer x,y,wd,ht,bw,dp;
   Window root;
   XImage * im;
   GC gc;
   XGCValues vals;

   PROTECT;
   XGetGeometry(fromd->display,pix,&root,&x,&y,&wd,&ht,&bw,&dp);
   root = RootWindow(tod->display,tod->screen);
   npix = XCreatePixmap(tod->display,root,wd,ht,dp);
   im = XGetImage(fromd->display,pix,0,0,wd,ht,~0,XYPixmap);
   gc = XCreateGC(tod->display,npix,0,&vals);
   XPutImage(tod->display,npix,gc,im,0,0,0,0,wd,ht);
   XFreeGC(tod->display,gc);
   free(im);
   UNPROTECT;


   return npix;
};





/************************************************************************/
/*									*/
/*	ASH_save_under_image -- save image of this window		*/
/*	ASH_restore_under_image -- save image of this window		*/
/*									*/
/************************************************************************/


void
ASH_save_under_image(w)
   ASH_WINDOW w;
{
   XWindowAttributes attrs;
   Pixmap pm;
   int wd,ht;
   int lx,ty;

   PROTECT;
   XGetWindowAttributes(DISPLAYOF(w),w->view_win,&attrs);
   UNPROTECT;

   if (attrs.map_state != IsViewable) return;

   PROTECT;
   if (w->under_image != NULL) {
      UNPROTECT;
      return;
    };

   wd = abs(w->view.rx - w->view.lx)+1;
   ht = abs(w->view.by - w->view.ty)+1;
   lx = XMAPVIEW(w,w->view.lx);
   ty = YMAPVIEW(w,w->view.ty);
   if (ty < 0) {
      ht += ty;
      ty = 0;
    };
   if (lx < 0) {
      wd += lx;
      lx = 0;
    };

   pm = XCreatePixmap(DISPLAYOF(w),w->view_win,wd,ht,w->depth);
   w->under_image = pm;
   XCopyArea(DISPLAYOF(w),w->view_win,pm,w->display->data->save_context,
		lx,ty,wd,ht,0,0);

   XFillRectangle(DISPLAYOF(w),w->view_win,w->display->data->save_context,
		     lx,ty,wd,ht);

   UNPROTECT;
};





void
ASH_restore_under_image(w)
   ASH_WINDOW w;
{
   XWindowAttributes attrs;
   Pixmap pm;
   int wd,ht;
   int lx,ty;

   PROTECT;
   XGetWindowAttributes(DISPLAYOF(w),w->view_win,&attrs);
   UNPROTECT;

   if (attrs.map_state != IsViewable) return;

   PROTECT;
   if (w->under_image == NULL) {
      UNPROTECT;
      return;
    };

   wd = abs(w->view.rx - w->view.lx)+1;
   ht = abs(w->view.by - w->view.ty)+1;
   lx = XMAPVIEW(w,w->view.lx);
   ty = YMAPVIEW(w,w->view.ty);
   if (ty < 0) {
      ht += ty;
      ty = 0;
    };
   if (lx < 0) {
      wd += lx;
      lx = 0;
    };

   pm = w->under_image;
   w->under_image = NULL;

   XCopyArea(DISPLAYOF(w),pm,w->view_win,w->display->data->save_context,
		0,0,wd,ht,lx,ty);

   XFreePixmap(DISPLAYOF(w),pm);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_flush_output -- force all output, read input, etc.		*/
/*									*/
/************************************************************************/


void
ASH_flush_output(w)
   ASH_WINDOW w;
{
   if (USE_THREADS) {
      PROTECT;
      XFlush(ASHinq_display(w));
      UNPROTECT;
    };

   BIOtrack();
};





/* end of ashwindraw.c */
