/************************************************************************/
/*									*/
/*		ashwindow.c						*/
/*									*/
/*	Main entries for window management				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


#define MAX_ASSOC	256

	Boolean 	ASH__exiting;

static	ASH_WINDOW	freewinlist = NULL;
static	ASH_WINDOW	rootwinlist = NULL;

typedef struct _BORDER_INFO {
   Integer width;
   Integer pixel;
   Boolean tiled;
} BORDER_INFO;

static	BORDER_INFO border_table[BORDER_TABLE_SIZE];






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		pre_remove();
static	void		sig_remove();
static	void		real_remove();
static	void		call_for_tree();
static	void		border_tile_fixup();
static	void		set_border();
static	ASH_WINDOW	create_window();
static	void		initial_window();
static	void		free_full_win();
static	void		initial_borders();
static	void		copy_borders();




/************************************************************************/
/*									*/
/*	ASH_window_init -- initialize window module			*/
/*	ASH_window_disp_init -- initialize new display for module	*/
/*									*/
/************************************************************************/


void
ASH_window_init()
{
   freewinlist = NULL;
   rootwinlist = NULL;
   ASH__exiting = FALSE;

   initial_borders();
};





void
ASH_window_disp_init(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   PROTECT;
   d->data->ash_assoc = XUniqueContext();
   d->data->menu_assoc = XUniqueContext();
   d->data->user_assoc = XUniqueContext();
   d->data->id_assoc = XUniqueContext();
   copy_borders(d,ffg);
   UNPROTECT;
};




/************************************************************************/
/*									*/
/*	ASHcreate -- create a window					*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHcreate(p,plx,pby,lx,by,rx,ty,brdrid,flags)
   ASH_WINDOW p;
   Integer plx,pby;
   Integer lx,by;
   Integer rx,ty;
   ASH_BORDERID brdrid;
   Integer flags;
{
   register ASH_WINDOW w;

   ENTER("ASHcreate 0x%x (%d,%d) @ (%d,%d) (%d,%d) %d 0x%x",
	    p,plx,pby,lx,by,rx,ty,brdrid,flags);

   CHECKWIN(p,ASHcreate);

   if (p->removed) w = NULL;
   else w = create_window(p,NULL,NULL,plx,pby,lx,by,rx,ty,brdrid,flags);

   TRACE(" ==> 0x%x",w);

   return w;
};




/************************************************************************/
/*									*/
/*	ASHremove -- remove given window				*/
/*									*/
/************************************************************************/

static	ASH_WINDOW	newfree;




void
ASHremove(w)
   ASH_WINDOW w;
{
   DISPLAY d;
   ASH_WINDOW r;

   TRACE("ASHremove 0x%x",w);

   CHECKWIN(w,ASHremove);

   if (w->removed) return;
   if (!L_PROTECT(w)) return;
   if (!ASH_lock_remove(w)) {
      L_UNPROTECT(w);
      return;
    };

   if (w->parent == NULL) {
      if (rootwinlist == w) rootwinlist = w->nextroot;
      else {
	 for (r = rootwinlist; r != NULL && r->nextroot != w; r = r->nextroot);
	 if (r != NULL) r->nextroot = w->nextroot;
       };
      w->nextroot = NULL;
    };

   ASHenter_window(NULL);
   if (w->visible && w->our_court) ASHvisible(w,FALSE);

   PROTECT;
   ASH_call_for_tree(w,pre_remove,TRUE);
   UNPROTECT;

   ASH_call_for_tree(w,sig_remove,TRUE);

   if (w->safety != SAFETY) return;

   WINDRAW_BEGIN(w);

   if (w->parent == NULL) d = w->display;
   else d = NULL;

   PROTECT;
   newfree = NULL;
   ASH_call_for_tree(w,real_remove,TRUE);
   UNPROTECT;

   if (w->inited) {
      free_full_win(w);
      if (w->x_win != NULL) {
	 PROTECT;
	 XDestroyWindow(DISPLAYOF(w),w->x_win);
	 XFlush(DISPLAYOF(w));
	 UNPROTECT;
	 w->x_win = NULL;
	 w->view_win = NULL;
	 ASH_free_drawinfo(w);
       };
    };

   WINDRAW_END(w);

   PROTECT;
   if (USE_THREADS) {
      w = (ASH_WINDOW) realloc(w,1);		/* don't reuse windows          */
    }
   else {
      if (freewinlist == NULL) freewinlist = newfree;
      else {
	 for (w = freewinlist; w->parent != NULL; w = w->parent);
	 w->parent = newfree;
       };
    };
   UNPROTECT;

   if (d != NULL) ASH_close_display(d);
};





static void
pre_remove(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY) return;
   if (w->removed) return;

   ASH_remove_enter(w);
   ASHsensitive_remove_all(w);

   w->removed = TRUE;
};





static void
sig_remove(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY || !w->removed) return;

   ASHcontrol_msg(w,"ASH$REMOVE");
   if (w->safety != SAFETY) return;

   ASH_remove_enter(w);
   ASHsensitive_remove_all(w);
};





static void
real_remove(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY || !w->removed) return;

   XDeleteContext(DISPLAYOF(w),w->x_win,w->display->data->ash_assoc);
   XDeleteContext(DISPLAYOF(w),w->x_win,w->display->data->user_assoc);
   XDeleteContext(DISPLAYOF(w),w->x_win,w->display->data->id_assoc);
   XDeleteContext(DISPLAYOF(w),w->x_win,w->display->data->menu_assoc);
   BWEthread_sema_remove(w->sema);

   if (w->full_win != NULL && w->retained) {
      XDeleteContext(DISPLAYOF(w),w->full_win,w->display->data->ash_assoc);
    };

   w->parent = newfree;
   newfree = w;
   w->safety = 0;
};





/************************************************************************/
/*									*/
/*	ASHwindow_background -- set background for window		*/
/*									*/
/************************************************************************/


void
ASHwindow_background(w,pixel,fs,pat)
   ASH_WINDOW w;
   Integer pixel;
   Integer fs;
   ASH_WINDOW pat;
{
   Pixmap pix;
   Integer f;
   Integer c;

   TRACE("ASHwindow_background 0x%x %d %d 0x%x",w,pixel,fs,pat);

   CHECKWIN(w,ASHwindow_background);

   if (w->frame || w->our_court) {
      ERROR("Can't change background of a frame or courteous window");
      return;
    };

   PROTECT;
   if (w->bkg_pixmap != NULL) XFreePixmap(DISPLAYOF(w),w->bkg_pixmap);
   w->bkg_pixmap = NULL;
   UNPROTECT;

   if (pat != NULL) {
      pix = ASH_window_to_pixmap(pat,0,0,w->display);
      pixel = 0;
    }
   else if (fs > 0) {
      PROTECT;
      f = ASHfill(w,fs);
      c = ASHcombination_rule(w,3);
      pix = XCreatePixmap(DISPLAYOF(w),w->view_win,32,32,w->depth);
      XFillRectangle(DISPLAYOF(w),pix,w->draw->context,0,0,32,32);
      ASHfill(w,f);
      ASHcombination_rule(w,c);
      UNPROTECT;
      pixel = 0;
    }
   else pix = None;

   WINDRAW_BEGIN(w);

   w->bkg_pixel = pixel;
   w->bkg_pixmap = pix;

   if (pix == None) XSetWindowBackground(DISPLAYOF(w),w->view_win,pixel);
   else XSetWindowBackgroundPixmap(DISPLAYOF(w),w->view_win,pix);

   if (w->full_win != NULL && w->retained) {
      if (pix != None) XSetWindowBackground(DISPLAYOF(w),w->full_win,pixel);
      XSetWindowBackgroundPixmap(DISPLAYOF(w),w->full_win,pix);
    };

   if (pix != None) XFreePixmap(DISPLAYOF(w),pix);

   WINDRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHwindow_border_id -- set standard border for window		*/
/*	ASHwindow_border -- set border for window			*/
/*									*/
/*	set_border -- do the work					*/
/*									*/
/************************************************************************/


void
ASHwindow_border_id(w,id)
   ASH_WINDOW w;
   Integer id;
{
   Pixmap pix;

   TRACE("ASHwindow_border_id 0x%x %d",w,id);

   CHECKWIN(w,ASHwindow_border_id);

   if (id < 0 || id >= BORDER_TABLE_SIZE) {
      ERROR("Invalid border id");
      return;
    };

   if (border_table[id].tiled) pix = w->display->data->border_tile[id];
   else pix = NULL;

   set_border(w,border_table[id].width,border_table[id].pixel,pix);
};





void
ASHwindow_border(w,width,pixel,pat)
   ASH_WINDOW w;
   Integer width;
   Integer pixel;
   ASH_WINDOW pat;
{
   Pixmap pix;
   Integer fill;
   ASH_WINDOW remwin;

   TRACE("ASHwindow_border 0x%x %d %d 0x%x",w,width,pixel,pat);

   CHECKWIN(w,ASHwindow_border);

   remwin = NULL;

   if (pat != NULL) {
      fill = (Integer) pat;
      if (fill > 0 && fill < 64) {
	 pat = ASHcreate(w,0,0,0,15,15,0,0,ASH_WINDOW_OFF_SCREEN);
	 ASHfill(pat,fill);
	 ASHrectangle(pat,0,15,15,0);
	 remwin = pat;
       };
      pix = ASH_window_to_pixmap(pat,0,0,w->display);
    }
   else pix = NULL;

   set_border(w,width,pixel,pix);

   if (pix != NULL) {
      PROTECT;
      XFreePixmap(DISPLAYOF(w),pix);
      UNPROTECT;
    };
   if (remwin != NULL) {
      ASHremove(remwin);
    };
};





static void
set_border(w,width,pixel,pix)
   ASH_WINDOW w;
   Integer width;
   Integer pixel;
   Pixmap pix;
{
   XWindowAttributes attrs;

   WINDRAW_BEGIN(w);

   XGetWindowAttributes(DISPLAYOF(w),w->x_win,&attrs);
   if (width != attrs.border_width)
      XSetWindowBorderWidth(DISPLAYOF(w),w->x_win,width);
   w->border_width = width;

   if (pix == NULL) XSetWindowBorder(DISPLAYOF(w),w->x_win,pixel);
   else XSetWindowBorderPixmap(DISPLAYOF(w),w->x_win,pix);

   if (w->inited) ASH_setup_maps(w,FALSE,FALSE,TRUE);

   WINDRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHwindow_gravity -- set gravitys for window			*/
/*									*/
/************************************************************************/


void
ASHwindow_gravity(w,bit,win)
   ASH_WINDOW w;
   Integer bit;
   Integer win;
{
   XSetWindowAttributes attrs;

   TRACE("ASHwindow_gravity 0x%x %d %d",w,bit,win);

   CHECKWIN(w,ASHwindow_gravity);

   attrs.bit_gravity = bit;
   attrs.win_gravity = win;

   PROTECT;

   XChangeWindowAttributes(DISPLAYOF(w),w->x_win,CWBitGravity|CWWinGravity,&attrs);

   UNPROTECT;
};






/************************************************************************/
/*									*/
/*	ASHinq_size -- get coords of window				*/
/*									*/
/************************************************************************/


void
ASHinq_size(w,type,lxp,byp,rxp,typ)
   ASH_WINDOW w;
   ASH_SIZE_TYPE type;
   Integer *lxp,*byp;
   Integer *rxp,*typ;
{
   BOX *b;
   BOX bb;
   Integer lx,by,rx,ty;

   TRACE("ASHinq_size 0x%x %d",w,type);

   CHECKWIN(w,ASHinq_size);

   switch (type) {
      case ASH_SIZE_WINDOW :
	 b = &w->coords;
	 break;

      case ASH_SIZE_VIEW :
	 b = &w->view;
	 break;

      case ASH_SIZE_SCREEN :
	 b = &w->view;
	 ASHmap(w,b->lx,b->by,ASHinq_top_window(w),&lx,&by);
	 ASHmap(w,b->rx,b->ty,ASHinq_top_window(w),&rx,&ty);
	 bb.lx = lx;
	 bb.rx = rx;
	 bb.by = by;
	 bb.ty = ty;
	 b = &bb;
	 break;

      case ASH_SIZE_BORDER :
	 bb.lx = bb.rx = bb.by = bb.ty = w->border_width;
	 b = &bb;
	 break;

      default :
	 ERROR("BAD SIZE TYPE");
	 b = &w->coords;
	 break;
    };

   *lxp = b->lx;
   *byp = b->by;
   *rxp = b->rx;
   *typ = b->ty;
};





/************************************************************************/
/*									*/
/*	ASHinq_parent -- return parent of current window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHinq_parent(w)
   ASH_WINDOW w;
{
   TRACE("ASHinq_parent 0x%x",w);

   if (w == NULL) return NULL;

   CHECKWIN(w,ASHinq_parent);

   return w->parent;
};






/************************************************************************/
/*									*/
/*	ASHinq_top_window -- return top window relative to given one	*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHinq_top_window(w)
   ASH_WINDOW w;
{
   ENTER("ASHinq_top_window 0x%x",w);

   if (w != NULL && w->safety != SAFETY) w = NULL;

   if (w == NULL) return ASHinq_top();

   while (w->parent != NULL) w = w->parent;

   return w;
};





/************************************************************************/
/*									*/
/*	ASHinq_X_window -- return X window for ASH window		*/
/*									*/
/************************************************************************/


Window
ASHinq_X_window(w)
   ASH_WINDOW w;
{
   TRACE("ASHinq_X_window 0x%x",w);

   CHECKWIN(w,ASHinq_X_window);

   return w->x_win;
};





/************************************************************************/
/*									*/
/*	ASHset_refresh -- set refresh routine in lieu of bitmap 	*/
/*	ASHset_region_refresh -- set region refresh routine		*/
/*	ASHinq_saved -- test if window is saved by ash			*/
/*									*/
/************************************************************************/


void
ASHset_refresh(w,fct)
   ASH_WINDOW w;
   Function_Ptr fct;
{
   TRACE("ASHset_refresh 0x%x 0x%x",w,fct);

   CHECKWIN(w,ASHset_refresh);

   if (fct == ASH_REFRESH_WINDOW && w->inited && !w->saved) {
      w->inited = FALSE;
    };

   if (w->saved && w->inited && fct != ASH_REFRESH_WINDOW) {
      free_full_win(w);
      w->use_view = TRUE;
    };

   if (fct == ASH_REFRESH_WINDOW) {
      w->saved = TRUE;
      w->region_refresh = FALSE;
      w->refresh = NULL;
    }
   else if (fct == ASH_REFRESH_IGNORE) {
      w->saved = FALSE;
      w->refresh = NULL;
      w->region_refresh = FALSE;
      w->noclear = TRUE;
    }
   else {
      if (fct == NULL) w->region_refresh = FALSE;
      w->saved = FALSE;
      w->refresh = fct;
    };
};




void
ASHset_region_refresh(w,fct)
   ASH_WINDOW w;
   Function_Ptr fct;
{
   TRACE("ASHset_region_refresh 0x%x 0x%x",w,fct);

   CHECKWIN(w,ASHset_region_refresh);

   w->refresh = NULL;
   w->region_refresh = TRUE;
   ASHset_refresh(w,fct);
};





int
ASHinq_saved(w)
   ASH_WINDOW w;
{
   TRACE("ASHinq_saved 0x%x",w);

   CHECKWIN(w,ASHinq_saved);

   return w->saved;
};





/************************************************************************/
/*									*/
/*	ASHset_menu_data, ASHinq_menu_data -- menu data for window	*/
/*	ASHset_user_data, ASHinq_user_data -- user data for window	*/
/*									*/
/************************************************************************/


void
ASHset_menu_data(w,data)
   ASH_WINDOW w;
   Universal data;
{
   TRACE("ASHset_menu_data 0x%x 0x%x",w,data);

   CHECKWIN(w,ASHset_menu_data);

   PROTECT;
   XSaveContext(DISPLAYOF(w),w->x_win,w->display->data->menu_assoc,data);
   UNPROTECT;
};





Universal
ASHinq_menu_data(w)
   ASH_WINDOW w;
{
   Universal v;

   TRACE("ASHinq_menu_data 0x%x",w);

   if (w == NULL || w->safety != SAFETY) return NULL;

   PROTECT;
   if (XFindContext(DISPLAYOF(w),w->x_win,w->display->data->menu_assoc,&v) != 0)
      v = NULL;
   UNPROTECT;

   return v;
};






void
ASHset_user_data(w,data)
   ASH_WINDOW w;
   Universal data;
{
   TRACE("ASHset_user_data 0x%x 0x%x",w,data);

   CHECKWIN(w,ASHset_user_data);

   PROTECT;
   XSaveContext(DISPLAYOF(w),w->x_win,w->display->data->user_assoc,data);
   UNPROTECT;
};





Universal
ASHinq_user_data(w)
   ASH_WINDOW w;
{
   Universal v;

   TRACE("ASHinq_user_data 0x%x",w);

   if (w == NULL || w->safety != SAFETY) return NULL;

   PROTECT;
   if (XFindContext(DISPLAYOF(w),w->x_win,w->display->data->user_assoc,&v) != 0)
      v = NULL;
   UNPROTECT;

   return v;
};





void
ASHset_window_id(w,data)
   ASH_WINDOW w;
   String data;
{
   TRACE("ASHset_window_id 0x%x %s",w,data);

   CHECKWIN(w,ASHset_window_id);

   PROTECT;
   XSaveContext(DISPLAYOF(w),w->x_win,w->display->data->id_assoc,data);
   UNPROTECT;
};





char *
ASHinq_window_id(w)
   ASH_WINDOW w;
{
   String v;

   TRACE("ASHinq_window_id 0x%x",w);

   if (w == NULL || w->safety != SAFETY) return NULL;

   PROTECT;
   if (XFindContext(DISPLAYOF(w),w->x_win,w->display->data->id_assoc,&v) != 0)
      v = NULL;
   UNPROTECT;

   return v;
};





/************************************************************************/
/*									*/
/*	ASHset_window_name, ASHinq_window_name -- set/get window name	*/
/*									*/
/************************************************************************/


void
ASHset_window_name(w,name)
   ASH_WINDOW w;
   String name;
{
   TRACE("ASHset_window_name 0x%x %s",w,(name == NULL ? "" : name));

   CHECKWIN(w,ASHset_window_name);

   if (name == NULL) name = "";

   PROTECT;
   XStoreName(DISPLAYOF(w),w->x_win,name);
   UNPROTECT;

   ASHcontrol_msg(w,"ASH$TITLE");
};






String
ASHinq_window_name(w)
   ASH_WINDOW w;
{
   String nm;

   TRACE("ASHinq_window_name 0x%x",w);

   CHECKWIN(w,ASHinq_window_name);

   PROTECT;
   XFetchName(DISPLAYOF(w),w->x_win,&nm);
   UNPROTECT;

   if (nm != NULL && *nm == 0) nm = NULL;

   return nm;
};






/************************************************************************/
/*									*/
/*	ASHregister_border -- define border pattern			*/
/*									*/
/************************************************************************/


void
ASHregister_border(id,width,pixel,pat)
   Integer id;
   Integer width;
   Integer pixel;
   ASH_WINDOW pat;
{
   Pixmap pix;

   TRACE("ASHregister_border %d %d %d 0x%x",id,width,pixel,pat);

   if (pat != NULL) {
      pix = ASH_window_to_pixmap(pat,0,0,NULL);
    }
   else pix = NULL;

   border_table[id].width = width;
   border_table[id].pixel = pixel;
   border_table[id].tiled = (pix != NULL);

   if (pix != NULL) {
      ASH_for_all_displays(border_tile_fixup,pat->display,pix,id);
    };
};





static void
border_tile_fixup(d,od,pix,id)
   DISPLAY d;
   DISPLAY od;
   Pixmap pix;
{
   if (d != od) pix = ASH_copy_pixmap(od,pix,d);

   if (d->data->border_tile[id] != NULL) {
      PROTECT;
      XFreePixmap(d->display,d->data->border_tile[id]);
      UNPROTECT;
    };

   d->data->border_tile[id] = pix;
};





/************************************************************************/
/*									*/
/*	ASHinq_valid_window -- check if window is valid 		*/
/*									*/
/************************************************************************/


int
ASHinq_valid_window(w)
   ASH_WINDOW w;
{
   if (w == NULL || w->safety != SAFETY || w->removed) return FALSE;

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	ASHexit -- handle exit							*/
/*										*/
/********************************************************************************/


void
ASHexit()
{
   ASH__exiting = TRUE;

   while (rootwinlist != NULL) {
      rootwinlist->lock = NULL;
      rootwinlist->lock_by = NULL;
      if (!ASHinq_valid_window(rootwinlist)) break;
      ASHremove(rootwinlist);
    };
};




/************************************************************************/
/*									*/
/*	ASH_setup_window -- initialize ASH window			*/
/*									*/
/************************************************************************/


void
ASH_setup_window(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p;
   Boolean fg;
   Integer vlx,vty;
   XSetWindowAttributes attrs;

   if (w->inited) return;

   p = w->parent;
   if (p != NULL && !p->inited) ASH_setup_window(p);

   if (!L_PROTECT(w)) return;

   PROTECT;

   if (w->inited) {
      UNPROTECT;
      L_UNPROTECT(w);
      return;
    };

   if (w->frame) {
      w->full_win = p->full_win;
      if (w->full_win != NULL) {
	 ASH_full_drawinfo(w);
       };
      w->use_view = TRUE;
    }
   else {
      if (w->saved) {
	 if (w->display->backing_store && !w->use_pixmap) {
	    vlx = w->coords.lx - w->view.lx;
	    if (w->inv_x) vlx = -vlx;
	    vty = w->coords.ty - w->view.ty;
	    if (w->inv_y) vty = -vty;
	    attrs.backing_store = Always;
	    attrs.background_pixmap = ParentRelative;
	    w->full_win = XCreateWindow(DISPLAYOF(w),w->view_win,vlx,vty,
					   abs(w->coords.lx - w->coords.rx) + 1,
					   abs(w->coords.by - w->coords.ty) + 1,
					   0,
					   0,
					   InputOutput,
					   CopyFromParent,
					   CWBackingStore|CWBackPixmap,
					   &attrs);
	    w->use_view = FALSE;
	    w->retained = TRUE;
	    XMapWindow(DISPLAYOF(w),w->full_win);
	    XSaveContext(DISPLAYOF(w),w->full_win,w->display->data->ash_assoc,w);
	  }
	 else {
	    w->full_win = XCreatePixmap(DISPLAYOF(w),w->x_win,
					   abs(w->coords.lx - w->coords.rx) + 1,
					   abs(w->coords.by - w->coords.ty) + 1,
					   w->depth);
	    w->use_view = TRUE;
	  };
	 ASH_full_drawinfo(w);
       }
      else {
	 w->full_win = NULL;
	 w->use_view = TRUE;
       };
    };

   ASH_setup_maps(w,FALSE,FALSE,FALSE);
   w->inited = TRUE;

   UNPROTECT;

   if (!w->noclear) {
      PROTECT;
      fg = ASHclip(w,FALSE);
      ASHclear(w);
      ASHclip(w,fg);
      UNPROTECT;
    };

   L_UNPROTECT(w);
};





/************************************************************************/
/*									*/
/*	ASH_call_for_tree -- call routine for each window in ASH tree	*/
/*									*/
/************************************************************************/


void
ASH_call_for_tree(w,rtn,thisfg)
   ASH_WINDOW w;
   void_Function_Ptr rtn;
   Boolean thisfg;
{
   call_for_tree(w,w->display,w->x_win,rtn,thisfg);
};





static void
call_for_tree(aw,d,w,rtn,thisfg)
   ASH_WINDOW aw;
   DISPLAY d;
   Window w;
   void_Function_Ptr rtn;
   Boolean thisfg;
{
   Window root,par;
   Window * chld;
   Integer nchld,i;
   ASH_WINDOW s;

   PROTECT;
   i = XQueryTree(d->display,w,&root,&par,&chld,&nchld);
   UNPROTECT;

   if (i == 0) return;

   for (i = 0; i < nchld; ++i) {
      PROTECT;
      if (XFindContext(d->display,chld[i],d->data->ash_assoc,&s) != 0) s = NULL;
      UNPROTECT;
      if (s != NULL && s->parent != aw) continue;
      if (s == NULL && aw != NULL && aw->parent == NULL &&
	     aw->x_win == XRootWindowOfScreen(aw->display->screen_id))
	 continue;
      call_for_tree(s,d,chld[i],rtn,(s != NULL && s->x_win == chld[i]));
    };

   PROTECT;
   if (chld != NULL) XFree(chld);
   UNPROTECT;

   if (thisfg && aw != NULL) (*rtn)(aw);
};





/************************************************************************/
/*									*/
/*	ASH_find_ash_window -- find ASH window for X window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASH_find_ash_window(xd,xw1)
   Display * xd;
   Window xw1;
{
   register DISPLAY d;
   register Window xw;
   ASH_WINDOW w;

   xw = xw1;
   d = ASH_find_display(xd);
   if (d == NULL) return NULL;

   PROTECT;

   w = NULL;
   if (XFindContext(xd,xw,d->data->ash_assoc,&w) != 0) w = NULL;

#ifdef USE_HIERARCHY_FOR_SEARCH
   if (w == NULL && xw != NULL) {
      Window par, *chld, root;
      Integer nchld;

      XQueryTree(xd,xw,&root,&par,&chld,&nchld);
      free(chld);
      xw = par;
      if (xw != NULL && XFindContext(xd,xw,d->data->ash_assoc,&w) != 0) w = NULL;
    };
#endif

   UNPROTECT;

   if (w != NULL) {
      if (w->x_win != xw && w->full_win != xw) w = NULL;
      else if (w->safety != SAFETY) w = NULL;
      else if (w->removed) w = NULL;
    };

   return w;
};





/************************************************************************/
/*									*/
/*	ASH_define_window -- define ASH window for X window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASH_define_window(d,xw)
   DISPLAY d;
   Window xw;
{
   XWindowAttributes attrs;
   ASH_WINDOW w;
   Integer flags;

   PROTECT;
   XGetWindowAttributes(d->display,xw,&attrs);
   UNPROTECT;

   flags = (attrs.map_state == IsUnmapped ? ASH_WINDOW_INVISIBLE :
	       ASH_WINDOW_VISIBLE);
   flags |= ASH_WINDOW_NOSAVE;
   flags |= ASH_WINDOW_DEPTH(attrs.depth);

   w = create_window(NULL,d,xw,attrs.x /* -attrs.border_width */ ,
			attrs.y+attrs.height-1+2*attrs.border_width,
			0,attrs.height-1,attrs.width-1,0, 0,
			flags);

   return w;
};





/************************************************************************/
/*									*/
/*	create_window -- create ASH window				*/
/*									*/
/************************************************************************/


static ASH_WINDOW
create_window(p,d,xw,plx,pby,lx,by,rx,ty,brdrid,flags)
   ASH_WINDOW p;
   DISPLAY d;
   Window xw;
   Integer plx,pby;
   Integer lx,by;
   Integer rx,ty;
   ASH_BORDERID brdrid;
   Integer flags;
{
   register ASH_WINDOW w;
   Integer mask;

   PROTECT;
   w = freewinlist;
   if (w != NULL) freewinlist = w->parent;
   else w = (ASH_WINDOW) calloc(1,sizeof(__WINDOW));
   UNPROTECT;

   w->safety = SAFETY;

   w->coords.lx = lx;
   w->coords.by = by;
   w->coords.rx = rx;
   w->coords.ty = ty;
   w->inv_x = (rx < lx);
   w->inv_y = (by < ty);
   w->view = w->coords;

   w->base.x = plx;
   w->base.y = pby;

   w->parent = p;
   w->rootmap_valid = FALSE;

   if (p == NULL) {
      w->display = d;
      d->root_cnt++;
    }
   else w->display = p->display;

   w->full_win = NULL;
   w->view_win = NULL;
   w->x_win = NULL;
   w->use_view = FALSE;
   w->retained = FALSE;
   w->inited = FALSE;
   w->redraw_input = FALSE;

   w->frame = (flags & ASH_WINDOW_FRAME) != 0;
   w->courteous = (flags & ASH_WINDOW_COURTEOUS) != 0;
   w->our_court = FALSE;
   w->under_image = NULL;
   w->saved = (flags & ASH_WINDOW_NOSAVE) == 0;
   w->independent = (flags & ASH_WINDOW_INDEPENDENT) != 0;
   if (w->frame) w->saved = FALSE;
   w->noclear = (flags & ASH_WINDOW_NOCLEAR) != 0;
   w->use_pixmap = (flags & ASH_WINDOW_OFF_SCREEN) != 0;
   w->noredirect = (flags & ASH_WINDOW_NOREDIRECT) != 0;

   w->removed = FALSE;
   w->resize = FALSE;
   w->many_regions = FALSE;
   w->visible = FALSE;
   w->refresh = NULL;
   w->region_refresh = TRUE;
   w->sensitive = NULL;
   w->control = NULL;
   w->inputdata = NULL;
   w->cursor = NULL;
   w->cursoron = TRUE;
   w->bkg_pixmap = NULL;
   w->bkg_pixel = w->display->backg;
   w->border_width = 0;
   w->colormap = w->display->colormap;
   w->thread = NULL;
   w->threadct = 0;
   w->sema = NULL;
   w->grab_by = NULL;
   w->grab_text_by = NULL;
   w->damage = NULL;
   w->lock = NULL;
   w->lock_by = NULL;

   if (w->parent == NULL) w->defaults = NULL;
   else w->defaults = w->parent->defaults;

   w->depth = (flags & ASH_WINDOW__DEPTH_MASK) / ASH_WINDOW__DEPTH_BIT;
   if (w->depth == 0) {
      if (p != NULL) w->depth = p->depth;
      else w->depth = DisplayPlanes(DISPLAYOF(w),SCREENOF(w));
    };

   if (w->parent != NULL) w->nextroot = NULL;
   else {
      w->nextroot = rootwinlist;
      rootwinlist = w;
    };

   PROTECT;
   if (xw == NULL) initial_window(w,brdrid);
   else {
      w->view_win = w->x_win = xw;
      if (w->parent != NULL) mask = WINDOW_EVENT_MASK;
      else if (xw == XRootWindowOfScreen(w->display->screen_id)) mask = XROOT_EVENT_MASK;
      else mask = ROOT_EVENT_MASK;
      XSelectInput(DISPLAYOF(w),w->x_win,mask);
    };
   XSaveContext(DISPLAYOF(w),w->x_win,w->display->data->ash_assoc,w);
   UNPROTECT;

   w->draw = NULL;
   ASH_new_drawinfo(w);

   if (w->courteous && w->frame) {
      ERROR("Cannot have courteous frame");
      w->frame = FALSE;
    };

   if (! (flags & ASH_WINDOW_INVISIBLE)) ASHvisible(w,TRUE);
   if (flags & ASH_WINDOW_INPUT) BIOnew_input_window(w);

   TRACE(" ==> 0x%x",w);

   return w;
};




/************************************************************************/
/*									*/
/*	initial_window -- create X window for ASH window		*/
/*									*/
/************************************************************************/


static void
initial_window(w,brdrid)
   ASH_WINDOW w;
   Integer brdrid;
{
   ASH_WINDOW p;
   Integer plx,pty;
   XSetWindowAttributes attrs;
   Integer mask;
   Integer bwid;
   Visual * vis;

   p = w->parent;
   if (p != NULL && !p->inited) ASH_setup_window(p);

   if (w->inited) {
      return;
    };

   plx = XMAPVIEW(p,w->base.x);
   pty = YMAPVIEW(p,w->base.y);
   pty -= abs(w->view.by - w->view.ty);

   mask = 0;

   mask |= CWBorderPixel|CWBorderPixmap;
   if (border_table[brdrid].pixel >= 0)
      attrs.border_pixel = border_table[brdrid].pixel;
   else
      attrs.border_pixel = w->display->foreg;
   if (border_table[brdrid].tiled)
      attrs.border_pixmap = w->display->data->border_tile[brdrid];
   else attrs.border_pixmap = NULL;
   bwid = border_table[brdrid].width;
   pty -= 2*bwid;

   if (!w->frame && !w->noclear) {
      mask |= CWBackPixel;
      attrs.background_pixel = w->bkg_pixel;
    }
   else {
      mask |= CWBackPixmap;
      attrs.background_pixmap = None;
    };

   if (w->courteous) {
      if (w->display->save_under) {
	 mask |= CWSaveUnder;
	 attrs.save_under = TRUE;
       }
      else if (w->parent != NULL) {
	 mask = 0;
	 w->our_court = TRUE;
	 bwid = 0;
       };
    };

   mask |= CWWinGravity;
   attrs.win_gravity = SouthWestGravity;
   attrs.win_gravity = ForgetGravity;
   attrs.event_mask = (w->parent == NULL ? ROOT_EVENT_MASK : WINDOW_EVENT_MASK);
   mask |= CWEventMask;

   PROTECT;

   if (!w->our_court) {
      vis = (Visual *) CopyFromParent;
      if (p->x_win == XRootWindowOfScreen(w->display->screen_id) &&
	     w->display->default_visual != w->display->color_visual &&
	     w->colormap == w->display->top_colormap &&
	     w->display->top_colormap != NULL) {
	 vis = w->display->color_visual;
	 mask |= CWColormap;
	 attrs.colormap = w->colormap;
       };

      if (w->noredirect) {
	 mask |= CWOverrideRedirect;
	 attrs.override_redirect = TRUE;
       };

      w->view_win = XCreateWindow(DISPLAYOF(w),p->view_win,plx,pty,
				     abs(w->view.lx - w->view.rx) + 1,
				     abs(w->view.by - w->view.ty) + 1,
				     bwid,
				     CopyFromParent,
				     InputOutput,
				     vis,
				     mask,&attrs);
      w->x_win = w->view_win;
    }
   else {
      w->view_win = w->parent->view_win;
      w->x_win = XCreateWindow(DISPLAYOF(w),p->view_win,plx,pty,
				     abs(w->view.lx - w->view.rx) + 1,
				     abs(w->view.by - w->view.ty) + 1,
				     bwid,
				     CopyFromParent,
				     InputOnly,
				     CopyFromParent,
				     mask,&attrs);
    };

   w->border_width = bwid;
   mask = (w->parent == NULL ? ROOT_EVENT_MASK : WINDOW_EVENT_MASK);
/* XSelectInput(DISPLAYOF(w),w->x_win,mask);		*/

   if (w->parent != NULL &&
	  w->parent->x_win == XRootWindowOfScreen(w->display->screen_id)) {
      if (!w->noredirect) ASH_set_properties(w->display,w->x_win,TRUE,0);
      ASH_set_wm_properties(w->display,w->x_win);
    };

   if (!w->our_court) XSetWindowColormap(DISPLAYOF(w),w->x_win,w->colormap);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	free_full_win -- free resources for full bitmap 		*/
/*									*/
/************************************************************************/


static void
free_full_win(w)
   ASH_WINDOW w;
{
   if (w->full_win != NULL && !w->frame) {
      PROTECT;
      if (w->retained)
	 XDestroyWindow(DISPLAYOF(w),w->full_win);
      else
	 XFreePixmap(DISPLAYOF(w),w->full_win);
      UNPROTECT;
    };

   if (w->full_win != NULL) {
      PROTECT;
      w->full_win = NULL;
      ASH_free_full_drawinfo(w);
      UNPROTECT;
    };
};





/************************************************************************/
/*									*/
/*	initial_borders -- setup initial border table			*/
/*									*/
/************************************************************************/


static void
initial_borders()
{
   Integer i;

   for (i = 0; i < BORDER_TABLE_SIZE; ++i) {
      border_table[i].width = 0;
      border_table[i].pixel = -1;
      border_table[i].tiled = FALSE;
    };

   for (i = 1; i < 10; ++i) {
      ASHregister_border(i,i,-1,NULL);
    };
};





static void
copy_borders(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   DISPLAY d1;
   Integer i;
   String s;
   XColor cola;

   PROTECT;
   if (ffg) {
      for (i = 0; i < BORDER_TABLE_SIZE; ++i) {
	 d->data->border_tile[i] = NULL;
       };
      s = ASHinq_resource("borderColor");
      if (s != NULL) {
	 if (XParseColor(d->display,XDefaultColormapOfScreen(d->screen_id),s,&cola) &&
		XAllocColor(d->display,XDefaultColormapOfScreen(d->screen_id),&cola)) {
	    for (i = 1; i < BORDER_TABLE_SIZE; ++i) {
	       if (border_table[i].pixel < 0) border_table[i].pixel = cola.pixel;
	     };
	  };
       };
    }
   else {
      d1 = ASH_inq_top_display();
      for (i = 0; i < BORDER_TABLE_SIZE; ++i) {
	 if (border_table[i].tiled) {
	    d->data->border_tile[i] = ASH_copy_pixmap(d1,d1->data->border_tile[i],d);
	  }
	 else d->data->border_tile[i] = NULL;
       };
    };
   UNPROTECT;
};






/* end of ashwindow.c */

