/************************************************************************/
/*									*/
/*		ashutil.c						*/
/*									*/
/*	Utility functions for ASH					*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"





/************************************************************************/
/*									*/
/*	Spline drawing definitions					*/
/*									*/
/************************************************************************/


#define NUMPOINTS	200
#define SMOOTHNESS	1.0


typedef struct _LINELIST {
   Integer size;
   Integer count;
   Boolean mallocd;
   Integer * x;
   Integer * y;
} LINELIST;






/************************************************************************/
/*									*/
/*	Spline Macro routines						*/
/*									*/
/************************************************************************/


#define DOUBLE(x)	((Float)(x))
#define ROUND(x)	((Integer)((x)+0.5))


#define ThirdPoint(x0,y0,x1,y1,tx,ty) { 		\
   tx = (2*x0 + x1) / 3.0;				\
   ty = (2*y0 + y1) / 3.0;				\
}


#define Midpoint(x0,y0,x1,y1,mx,my) {			\
   mx = (x0 + x1) / 2.0;				\
   my = (y0 + y1) / 2.0;				\
}


#define INIT_LINELIST(ll,ct) {			       \
   (ll)->size = ct;				       \
   (ll)->count = 0;				       \
   (ll)->mallocd = FALSE;			       \
   (ll)->x = (Integer *) alloca(sizeof(Integer)*ct);   \
   (ll)->y = (Integer *) alloca(sizeof(Integer)*ct);   \
}


#define FREE_LINELIST(ll) {			       \
   if ((ll)->mallocd) { 			       \
      free((ll)->x);				       \
      free((ll)->y);				       \
    };						       \
}


#define AddLine(ll,x0,y0,x1,y1) {			\
   if (ll->count >= ll->size) { 			\
      GrowBuf(ll);					\
    };							\
   if (ll->count == 0) {				\
      ll->x[0] = ROUND(x0);				\
      ll->y[0] = ROUND(y0);				\
      ll->count = 1;					\
    };							\
   ll->x[ll->count] = ROUND(x1);			\
   ll->y[ll->count] = ROUND(y1);			\
   ++(ll->count);					\
}


#define CanApproxWithLine(x0,y0,x2,y2,x3,y3,fg) {			\
   Float triangleArea, sideSquared, dx, dy;				\
									\
   triangleArea = x0*y2 - x2*y0 + x2*y3 - x3*y2 + x3*y0 - x0*y3;	\
   triangleArea *= triangleArea;					\
   dx = x3 - x0;							\
   dy = y3 - y0;							\
   sideSquared = dx*dx + dy*dy; 					\
   fg = triangleArea <= SMOOTHNESS * sideSquared;			\
};





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		CreateOpenLineList();
static	void		CreateClosedLineList();
static	void		CalcBSpline();
static	void		AddBezierArc();
static	void		GrowBuf();





/************************************************************************/
/*									*/
/*	ASH_util_init -- initialize utility module			*/
/*									*/
/************************************************************************/


void
ASH_util_init()
{
};





/************************************************************************/
/*									*/
/*	ASHclear_box -- clear a box region				*/
/*	ASHfill_box -- fill a box region				*/
/*	ASHhilite_box -- hilite a boxed region				*/
/*	ASHstipple_box -- stipple a boxed region			*/
/*									*/
/************************************************************************/


void
ASHclear_box(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   Integer fill;
   Integer color;

   TRACE("ASHclear_box 0x%x %d %d %d %d",w,lx,by,rx,ty);

   if (L_PROTECT(w)) {
      fill = ASHfill(w,ASH_FILL_SOLID);
      color = ASHcolor(w,ASHinq_background_color(w));
      ASHrectangle(w,lx,by,rx,ty);
      ASHcolor(w,color);
      ASHfill(w,fill);
      L_UNPROTECT(w);
    };
};





void
ASHfill_box(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   Integer fill;

   TRACE("ASHfill_box 0x%x %d %d %d %d",w,lx,by,rx,ty);

   if (L_PROTECT(w)) {
      fill = ASHfill(w,ASH_FILL_SOLID);
      ASHrectangle(w,lx,by,rx,ty);
      ASHfill(w,fill);
      L_UNPROTECT(w);
    };
};





void
ASHhilite_box(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   Integer cr,pm;

   if (L_PROTECT(w)) {
      if (w->depth == 1) {
	 cr = ASHcombination_rule(w,10);
	 ASHrectangle(w,lx,by,rx,ty);
	 ASHcombination_rule(w,cr);
       }
      else {
	 pm= ASHplane_mask(w,ASH__flip_mask);
	 cr = ASHcombination_rule(w,10);
	 ASHrectangle(w,lx,by,rx,ty);
	 ASHcombination_rule(w,cr);
	 ASHplane_mask(w,pm);
       };
      L_UNPROTECT(w);
    };
};





void
ASHstipple_box(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   Integer bcol,col;
   Integer fill;

   TRACE("ASHstipple_box 0x%x %d %d %d %d",w,lx,by,rx,ty);

   if (L_PROTECT(w)) {
      bcol = ASHinq_background_color(w);
      col = ASHcolor(w,bcol);
      fill = ASHfill(w,ASH_FILL_HALFTONE_STIPPLE);
      ASHrectangle(w,lx,by,rx,ty);
      ASHfill(w,fill);
      ASHcolor(w,col);
      L_UNPROTECT(w);
    };
};





/************************************************************************/
/*									*/
/*	ASHdraw_hilite -- turn hilighting on or off for general drawing */
/*									*/
/************************************************************************/


void
ASHdraw_hilite(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   Integer ncr;
   Integer c1;

   if (fg) {
      c1 = ASHinq_color(w);
      ncr = (c1 == 0 ? 9 : 6);
      if (w->depth == 1) {
	 ASHcombination_rule(w,ncr);
       }
      else {
	 ASHplane_mask(w,ASH__flip_mask);
	 ASHcombination_rule(w,10);
       };
    }
   else {
      ASHcombination_rule(w,3);
      ASHplane_mask(w,~0);
    };
};





/************************************************************************/
/*									*/
/*	ASHcenter_text -- output text centered in region		*/
/*									*/
/************************************************************************/


void
ASHcenter_text(w,txt,lx,by,rx,ty)
   ASH_WINDOW w;
   String txt;
   Integer lx,by;
   Integer rx,ty;
{
   register Integer dx,dy,ln;
   Integer x,y;
   Character buf[1024];

   TRACE("ASHcenter_text 0x%x %s (%d,%d),(%d,%d)",w,(txt == NULL ? "" : txt),lx,by,rx,ty);

   ASHclear_box(w,lx,by,rx,ty);

   if (txt == NULL) return;

   dx = rx > lx ? 1 : -1;
   dy = ty > by ? 1 : -1;

   strcpy(buf,txt);
   ln = strlen(buf);

   do {
      ASHinq_text_extent(ASHinq_font(w),buf,&x,&y);
      x = (abs(rx-lx)+1-x)/2;
      y = (abs(ty-by)+1-y)/2;
      if (x >= 0) break;
      buf[--ln] = 0;
    }
   while (ln > 0);

   if (ln == 0) return;

   ASHtext(w,lx+dx*x,by+dy*y,buf);
};






/************************************************************************/
/*									*/
/*	ASHspline_arc -- add a spline-style arc 			*/
/*									*/
/************************************************************************/


void
ASHspline_arc(w,x0,y0,x1,y1,x2,y2,x3,y3)
   ASH_WINDOW w;
   Integer x0,y0;
   Integer x1,y1;
   Integer x2,y2;
   Integer x3,y3;
{
   LINELIST ll;

   INIT_LINELIST(&ll,NUMPOINTS);

   AddBezierArc(&ll, DOUBLE(x0), DOUBLE(y0), DOUBLE(x1), DOUBLE(y1),
		   DOUBLE(x2), DOUBLE(y2), DOUBLE(x3), DOUBLE(y3));
   ASHpolyline(w, ll.count, ll.x, ll.y);

   FREE_LINELIST(&ll);
};






/************************************************************************/
/*									*/
/*	ASHspline -- draw a spline through given points 		*/
/*									*/
/************************************************************************/


void
ASHspline(w,count,x,y)
   ASH_WINDOW w;
   Integer count;
   Integer x[];
   Integer y[];
{
   LINELIST ll;

   INIT_LINELIST(&ll,NUMPOINTS);

   CreateOpenLineList(&ll, x, y, count);
   ASHpolyline(w, ll.count, ll.x, ll.y);

   FREE_LINELIST(&ll);
};






/************************************************************************/
/*									*/
/*	ASHspline_closed -- draw a closed spline through points 	*/
/*									*/
/************************************************************************/


void
ASHspline_closed(w,count,x,y)
   ASH_WINDOW w;
   Integer count;
   Integer x[];
   Integer y[];
{
   LINELIST ll;

   if (count < 3) {
      ASHpolyline(w, count, x, y);
    }
   else {
      INIT_LINELIST(&ll,NUMPOINTS);
      CreateClosedLineList(&ll, x, y, count);
      ASHpolyline(w, ll.count, ll.x, ll.y);
      FREE_LINELIST(&ll);
    };
};





/************************************************************************/
/*									*/
/*	ASHspline_filled -- draw a filled region based on a spline	*/
/*									*/
/************************************************************************/


void
ASHspline_filled(w,count,x,y)
   ASH_WINDOW w;
   Integer count;
   Integer x[];
   Integer y[];
{
   LINELIST ll;

   if (count < 3) {
      ASHpolygon(w, count, x, y);
    }
   else {
      INIT_LINELIST(&ll,NUMPOINTS);
      CreateClosedLineList(&ll, x, y, count);
      ASHgeneral_polygon(w, ll.count, ll.x, ll.y);
      FREE_LINELIST(&ll);
    };
};




/************************************************************************/
/*									*/
/*	CreateOpenLineList -- create line list for open spline		*/
/*									*/
/************************************************************************/


static void
CreateOpenLineList(ll, cpx,cpy,cpcount)
   LINELIST * ll;
   Integer cpx[];
   Integer cpy[];
   Integer cpcount;
{
    register Integer cpi;

    ll->count = 0;

    CalcBSpline(ll, cpx[0], cpy[0], cpx[0], cpy[0], cpx[0], cpy[0], cpx[1], cpy[1] );
    CalcBSpline(ll, cpx[0], cpy[0], cpx[0], cpy[0], cpx[1], cpy[1], cpx[2], cpy[2] );

    for (cpi = 1; cpi < cpcount - 2; ++cpi) {
	CalcBSpline(ll, cpx[cpi - 1], cpy[cpi - 1], cpx[cpi], cpy[cpi],
			cpx[cpi + 1], cpy[cpi + 1], cpx[cpi + 2], cpy[cpi + 2] );
     };

    CalcBSpline(ll, cpx[cpi - 1], cpy[cpi - 1], cpx[cpi], cpy[cpi],
		    cpx[cpi + 1], cpy[cpi + 1], cpx[cpi + 1], cpy[cpi + 1] );
    CalcBSpline(ll, cpx[cpi], cpy[cpi], cpx[cpi + 1], cpy[cpi + 1],
		    cpx[cpi + 1], cpy[cpi + 1], cpx[cpi + 1], cpy[cpi + 1] );
};






/************************************************************************/
/*									*/
/*	CreateClosedLineList -- create line list for closed spline	*/
/*									*/
/************************************************************************/


static void
CreateClosedLineList(ll,cpx,cpy,cpcount)
   LINELIST *ll;
   Integer cpx[];
   Integer cpy[];
   Integer cpcount;
{
    register Integer cpi;

    ll->count = 0;

    CalcBSpline(ll, cpx[cpcount - 1], cpy[cpcount - 1], cpx[0], cpy[0],
		    cpx[1], cpy[1], cpx[2], cpy[2] );

    for (cpi = 1; cpi < cpcount - 2; ++cpi) {
	CalcBSpline(ll, cpx[cpi - 1], cpy[cpi - 1], cpx[cpi], cpy[cpi],
			cpx[cpi + 1], cpy[cpi + 1], cpx[cpi + 2], cpy[cpi + 2] );
     };

    CalcBSpline(ll, cpx[cpi - 1], cpy[cpi - 1], cpx[cpi], cpy[cpi],
		    cpx[cpi + 1], cpy[cpi + 1], cpx[0], cpy[0] );
    CalcBSpline(ll, cpx[cpi], cpy[cpi], cpx[cpi + 1], cpy[cpi + 1],
		    cpx[0], cpy[0], cpx[1], cpy[1] );
};





/************************************************************************/
/*									*/
/*	CalcBSpline -- add points for spline to list			*/
/*									*/
/************************************************************************/


static void
CalcBSpline(ll, cminus1x, cminus1y, cx, cy, cplus1x, cplus1y, cplus2x, cplus2y )
   LINELIST * ll;
   Integer cminus1x,cminus1y;
   Integer cx,cy;
   Integer cplus1x,cplus1y;
   Integer cplus2x,cplus2y;
{
   Float p0x, p1x, p2x, p3x, tempx, p0y, p1y, p2y, p3y, tempy;
   Float fcx,fcy,fc1x,fc1y,fcm1x,fcm1y,fc2x,fc2y;

   fcx = cx;
   fcy = cy;
   fc1x = cplus1x;
   fc1y = cplus1y;
   fcm1x = cminus1x;
   fcm1y = cminus1y;
   fc2x = cplus2x;
   fc2y = cplus2y;

   ThirdPoint(fcx,fcy,fc1x,fc1y, p1x,p1y );
   ThirdPoint(fc1x,fc1y,fcx,fcy, p2x,p2y );
   ThirdPoint(fcx,fcy,fcm1x,fcm1y, tempx,tempy);
   Midpoint(tempx, tempy, p1x, p1y, p0x, p0y);
   ThirdPoint(fc1x,fc1y,fc2x,fc2y, tempx,tempy );
   Midpoint(tempx, tempy, p2x, p2y, p3x, p3y);

   AddBezierArc(ll, p0x, p0y, p1x, p1y, p2x, p2y, p3x, p3y);
};






/************************************************************************/
/*									*/
/*	AddBezierArc -- add arc to					*/
/*									*/
/************************************************************************/


static void
AddBezierArc(ll,x0,y0,x1,y1,x2,y2,x3,y3)
   LINELIST * ll;
   Float x0,y0;
   Float x1,y1;
   Float x2,y2;
   Float x3,y3;
{
   Float midx01, midx12, midx23, midlsegx, midrsegx, cx;
   Float midy01, midy12, midy23, midlsegy, midrsegy, cy;
   Boolean fg;

   Midpoint(x0, y0, x1, y1, midx01, midy01);
   Midpoint(x1, y1, x2, y2, midx12, midy12);
   Midpoint(x2, y2, x3, y3, midx23, midy23);
   Midpoint(midx01, midy01, midx12, midy12, midlsegx, midlsegy);
   Midpoint(midx12, midy12, midx23, midy23, midrsegx, midrsegy);
   Midpoint(midlsegx, midlsegy, midrsegx, midrsegy, cx, cy);

   CanApproxWithLine(x0, y0, midlsegx, midlsegy, cx, cy, fg);
   if (fg) {
      AddLine(ll, x0, y0, cx, cy);
    }
   else if ( (midx01 != x1) || (midy01 != y1) ||
		(midlsegx != x2) || (midlsegy != y2) ||
		(cx != x3) || (cy != y3) ) {
	AddBezierArc(ll, x0, y0, midx01, midy01, midlsegx, midlsegy, cx, cy );
    };

   CanApproxWithLine(cx, cy, midx23, midy23, x3, y3, fg);
   if (fg) {
      AddLine(ll, cx, cy, x3, y3);
    }
   else if ( (cx != x0) || (cy != y0) ||
		(midrsegx != x1) || (midrsegy != y1) ||
		(midx23 != x2) || (midy23 != y2) ) {
	AddBezierArc(ll, cx, cy, midrsegx, midrsegy, midx23, midy23, x3, y3 );
    };
};






/************************************************************************/
/*									*/
/*	GrowBuf -- augment a line list					*/
/*									*/
/************************************************************************/


static void
GrowBuf(ll)
   LINELIST * ll;
{
   Integer newsize;
   Integer *new;

   newsize = ll->size*2;

   if (ll->mallocd) {
      ll->x = (Integer *) realloc(ll->x,newsize*sizeof(Integer));
      ll->y = (Integer *) realloc(ll->y,newsize*sizeof(Integer));
    }
   else {
      new = (Integer *) malloc(newsize*sizeof(Integer));
      bcopy(ll->x,new,ll->size*sizeof(Integer));
      ll->x = new;
      new = (Integer *) malloc(newsize*sizeof(Integer));
      bcopy(ll->y,new,ll->size*sizeof(Integer));
      ll->y = new;
      ll->mallocd = TRUE;
    };

   ll->size = newsize;
};





/* end of ashutil.c */
