/************************************************************************/
/*									*/
/*		ashscreen.c						*/
/*									*/
/*	Main entries for screen management				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		set_visible();





/************************************************************************/
/*									*/
/*	ASH_screen_init -- initialize this module			*/
/*									*/
/************************************************************************/


void
ASH_screen_init()
{
};






/************************************************************************/
/*									*/
/*	ASHvisible -- set visiblitity of window 			*/
/*	ASHvisible_after -- set visibility to be after given window	*/
/*									*/
/*	set_visible -- set visiblitity of window			*/
/*									*/
/************************************************************************/


void
ASHvisible(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   TRACE("ASHvisible 0x%x %d",w,fg);

   CHECKWIN(w,ASHvisible);

   set_visible(w,fg,NULL);
};





void
ASHvisible_after(w,after,fg)
   ASH_WINDOW w;
   ASH_WINDOW after;
   Boolean fg;
{
   TRACE("ASHvisible_after 0x%x 0x%x %d",w,after,fg);

   CHECKWIN(w,ASHvisible_after);
   if (after != NULL) {
      CHECKWIN(after,ASHvisible_after);
      if (after->parent != w->parent) {
	 ERROR("Visible after only applicable to sibling");
	 after = NULL;
       };
    };

   set_visible(w,fg,after);
};





static void
set_visible(w,fg,after)
   ASH_WINDOW w;
   Boolean fg;
   ASH_WINDOW after;
{
   Window win[2];

   TRACE("set_visible 0x%x %d 0x%x",w,fg,after);

   if (!w->inited) ASH_setup_window(w);

   if (w->our_court && fg && w->under_image == NULL) {
      ASH_save_under_image(w);
    };

   w->visible = fg;

   WINDRAW_BEGIN(w);

   if (fg && after == NULL) {
      if (w->parent == NULL)
	 XMapWindow(DISPLAYOF(w),w->x_win);
      else
	 XMapRaised(DISPLAYOF(w),w->x_win);
    }
   else if (fg) {
      win[0] = after->x_win;
      win[1] = w->x_win;
      XRestackWindows(DISPLAYOF(w),win,2);
      XMapWindow(DISPLAYOF(w),w->x_win);
    }
   else {
      XUnmapWindow(DISPLAYOF(w),w->x_win);
    };

   if (w->our_court && !fg && w->under_image != NULL) {
      ASH_restore_under_image(w);
    };

   WINDRAW_END(w);

   if (fg) ASHcontrol_msg(w,"ASH$VISIBLE");
   else ASHcontrol_msg(w,"ASH$INVISIBLE");
};





/************************************************************************/
/*									*/
/*	ASHinq_visible -- return flag indicating window visibility	*/
/*	ASHinq_viewable -- check if window is actually viewable re X11	*/
/*									*/
/************************************************************************/


int
ASHinq_visible(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p;

   TRACE("ASHinq_visible 0x%x",w);

   CHECKWIN(w,ASHinq_visible);

   for (p = w; p != NULL; p = p->parent) {
      if (!p->visible) break;
    };

   return (p == NULL);
};




int
ASHinq_viewable(w)
   ASH_WINDOW w;
{
   XWindowAttributes attrs;

   PROTECT;
   XSync(DISPLAYOF(w),0);
   XGetWindowAttributes(DISPLAYOF(w),w->x_win,&attrs);
   UNPROTECT;

   return (attrs.map_state == IsViewable);
};





/************************************************************************/
/*									*/
/*	ASHview -- change view for window				*/
/*									*/
/************************************************************************/


void
ASHview(w,plx,pby,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer plx,pby;
   Integer lx,by,rx,ty;
{
   Boolean fgx,fgy;

   TRACE("ASHview 0x%x %d %d %d %d %d %d",w,plx,pby,lx,by,rx,ty);

   CHECKWIN(w,ASHview);

   fgx = (rx < lx);
   fgy = (by < ty);
   if (fgx != w->inv_x || fgy != w->inv_y) {
      ERROR("Can't change scale/direction with a view");
      return;
    };

   w->base.x = plx;
   w->base.y = pby;
   w->view.lx = lx;
   w->view.by = by;
   w->view.rx = rx;
   w->view.ty = ty;

   if (w->inited) ASH_setup_maps(w,FALSE,FALSE,TRUE);

   if (w->inited && w->visible) XMapWindow(DISPLAYOF(w),w->x_win);

};





/************************************************************************/
/*									*/
/*	ASHresize -- change size of window				*/
/*									*/
/************************************************************************/


void
ASHresize(w,plx,pby,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer plx,pby;
   Integer lx,by,rx,ty;
{
   Integer vfg;

   TRACE("ASHresize 0x%x %d %d %d %d %d %d",w,plx,pby,lx,by,rx,ty);

   CHECKWIN(w,ASHresize);

   if (ASHcontrol_msg(w,"ASH$INQ_RESIZE") > 0) return;

   vfg = w->visible;

/* PROTECT; XUnmapWindow(DISPLAYOF(w),w->x_win); UNPROTECT; */

   w->base.x = plx;
   w->base.y = pby;
   w->coords.lx = lx;
   w->coords.by = by;
   w->coords.rx = rx;
   w->coords.ty = ty;
   w->view = w->coords;
   w->inv_x = (rx < lx);
   w->inv_y = (by < ty);

   if (w->inited) {
      ASH_setup_maps(w,TRUE,vfg,TRUE);
    };
};





/************************************************************************/
/*									*/
/*	ASHpop -- pop window to top of pile				*/
/*									*/
/************************************************************************/


void
ASHpop(w)
   ASH_WINDOW w;
{
   TRACE("ASHpop");

   CHECKWIN(w,ASHpop);

   if (!w->inited) ASH_setup_window(w);

   WINDRAW_BEGIN(w);
   XRaiseWindow(DISPLAYOF(w),w->x_win);
   WINDRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHpush -- push window to bottom of pile			*/
/*									*/
/************************************************************************/


void
ASHpush(w)
   ASH_WINDOW w;
{
   TRACE("ASHpush 0x%x",w);

   CHECKWIN(w,ASHpush);

   if (!w->inited) ASH_setup_window(w);

   WINDRAW_BEGIN(w);
   XLowerWindow(DISPLAYOF(w),w->x_win);
   WINDRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHrefresh -- refresh given window				*/
/*									*/
/************************************************************************/


void
ASHrefresh(w)
   ASH_WINDOW w;
{
   CHECKWIN(w,ASHrefresh);

   TRACE("ASHrefresh");

   ASH_call_for_tree(w,ASH_window_refresh,TRUE);
};






/************************************************************************/
/*									*/
/*	ASHmap -- map coordinates from one window to another		*/
/*									*/
/************************************************************************/


#define MAX_MAP_DEPTH		64


int
ASHmap(fw,x,y,tw,txp,typ)
   ASH_WINDOW fw;
   Integer x,y;
   ASH_WINDOW tw;
   Integer *txp,*typ;
{
   Integer nx,ny;
   Integer xx,yy;
   ASH_WINDOW par[MAX_MAP_DEPTH];
   register ASH_WINDOW p,w;
   register Integer i,ct;
   Window chld,root;
   Boolean fg;

   TRACE("ASHmap 0x%x (%d,%d) -> 0x%x",fw,x,y,tw);

   if (fw == NULL || fw->safety != SAFETY) return FALSE;
   if (tw == NULL || tw->safety != SAFETY) return FALSE;

   if (!fw->inited) ASH_setup_window(fw);
   if (!tw->inited) ASH_setup_window(tw);

   if (DISPLAYOF(fw) != DISPLAYOF(tw)) return FALSE;

   fg = TRUE;

   ct = 0;
   for (p = tw; p != NULL; p = p->parent) {
      par[ct++] = p;
      if (ct >= MAX_MAP_DEPTH) {
	 fg = FALSE;
	 break;
       };
    };

   if (fg) {
      xx = x;
      yy = y;
      for (w = fw; w != NULL; w = w->parent) {
	 for (i = 0; i < ct; ++i) {
	    if (w == par[i]) break;
	  };
	 if (i < ct) break;
	 nx = XMAPXVIEW(w,xx) + w->border_width;
	 ny = YMAPXVIEW(w,yy) - w->border_width;
	 ny = abs(w->view.ty - w->view.by) - ny;
	 if (w->parent != NULL) {
	    if (w->parent->inv_x) nx = -nx;
	    if (w->parent->inv_y) ny = -ny;
	  };
	 xx = w->base.x + nx;
	 yy = w->base.y - ny;
       };
      if (w == NULL) fg = FALSE;
    };

   if (fg) {
      for (i = i-1; i >= 0; --i) {
	 w = par[i];
	 p = par[i+1];
	 nx = xx - w->base.x;
	 if (p->inv_x) nx = -nx;
	 nx -= w->border_width;
	 ny = yy - w->base.y;
	 if (p->inv_y) ny = -ny;
	 ny += w->border_width; 	/* -bw + 2*bw */
	 ny += abs(w->view.ty - w->view.by);
	 xx = XUNMAPXVIEW(w,nx);
	 yy = YUNMAPXVIEW(w,ny);
       };
    };

/*
   if (!fg && XRootWindowOfScreen(fw->display->screen_id) == tw->view_win) {
      if (!fw->rootmap_valid) {
	 PROTECT;
	 XTranslateCoordinates(DISPLAYOF(fw),fw->view_win,tw->view_win,0,0,
				  &nx,&ny,&chld);
	 UNPROTECT;
	 fw->root_map.x = nx;
	 fw->root_map.y = ny;
	 fw->rootmap_valid = TRUE;
       };
      nx = XMAPVIEW(fw,x);
      ny = YMAPVIEW(fw,y);
      xx = fw->root_map.x + nx;
      yy = fw->root_map.y + ny;
      fg = TRUE;
    };
*/

   if (!fg) {
      root = XRootWindowOfScreen(fw->display->screen_id);
      if (root == XRootWindowOfScreen(tw->display->screen_id)) {
	 if (!fw->rootmap_valid) {
	    PROTECT;
	    XTranslateCoordinates(DISPLAYOF(fw),fw->view_win,root,0,0,
				     &nx,&ny,&chld);
	    UNPROTECT;
	    fw->root_map.x = nx;
	    fw->root_map.y = ny;
	    fw->rootmap_valid = TRUE;
	  };
	 if (!tw->rootmap_valid) {
	    PROTECT;
	    XTranslateCoordinates(DISPLAYOF(tw),tw->view_win,root,0,0,
				     &nx,&ny,&chld);
	    UNPROTECT;
	    tw->root_map.x = nx;
	    tw->root_map.y = ny;
	    tw->rootmap_valid = TRUE;
	  };
	 nx = XMAPVIEW(fw,x) + fw->root_map.x - tw->root_map.x;
	 ny = YMAPVIEW(fw,y) + fw->root_map.y - tw->root_map.y;
	 xx = XUNMAPVIEW(tw,nx);
	 yy = YUNMAPVIEW(tw,ny);
	 fg = TRUE;
       };
    };

   if (!fg) {
      PROTECT;
      XTranslateCoordinates(DISPLAYOF(fw),fw->view_win,tw->view_win,
			       XMAPVIEW(fw,x),YMAPVIEW(fw,y),
			       &nx,&ny,&chld);
      UNPROTECT;
      xx = XUNMAPVIEW(tw,nx);
      yy = YUNMAPVIEW(tw,ny);
    };

   if (txp != NULL) *txp = xx;
   if (typ != NULL) *typ = yy;

#ifdef DEBUG_MAP
   PROTECT;
   XTranslateCoordinates(DISPLAYOF(fw),fw->view_win,tw->view_win,
			    XMAPVIEW(fw,x),YMAPVIEW(fw,y),
			    &nx,&ny,&chld);
   UNPROTECT;
   nx = XUNMAPVIEW(tw,nx);
   ny = YUNMAPVIEW(tw,ny);
   if (nx != xx || ny != yy) {
      printf("Map 0x%x -> 0x%x : new: (%d,%d), old: (%d,%d)\n",fw,tw,xx,yy,nx,ny);
    };
#endif

   return TRUE;
};






/************************************************************************/
/*									*/
/*	ASHset_control, ASHremove_control, ASHcontrol_msg		*/
/*									*/
/*	    Functions for associating control with window		*/
/*									*/
/************************************************************************/


void
ASHset_control(w,rtn)
   ASH_WINDOW w;
   Function_Ptr rtn;
{
   register CONTROL c;

   TRACE("ASHset_control 0x%x 0x%x",w,rtn);

   CHECKWIN(w,ASHset_control);

   for (c = w->control; c != NULL; c = c->nextcontrol) {
      if (c->routine == rtn) return;
    };

   c = (CONTROL) malloc(sizeof(__CONTROL));
   c->routine = rtn;
   c->nextcontrol = w->control;
   w->control = c;
};





void
ASHremove_control(w,rtn)
   ASH_WINDOW w;
   Function_Ptr rtn;
{
   register CONTROL c,c1;

   TRACE("ASHremove_control 0x%x 0x%x",w,rtn);

   CHECKWIN(w,ASHremove_control);

   while (w->control != NULL && w->control->routine == rtn) {
      c = w->control;
      w->control = c->nextcontrol;
      free(c);
    };

   for (c = w->control;
	c != NULL && c->nextcontrol != NULL;
	c = c->nextcontrol) {
      while (c->nextcontrol != NULL && c->nextcontrol->routine == rtn) {
	 c1 = c->nextcontrol;
	 c->nextcontrol = c1->nextcontrol;
	 free(c1);
       };
    };
};




int
ASHcontrol_msg(w,msg)
   ASH_WINDOW w;
   String msg;
{
   register CONTROL c;
   register Integer sts;
   Function_Ptr con[MAX_CONTROL];
   register Integer i,j;

   TRACE("ASHcontrol_msg 0x%x %s",w,msg);

   if (w == NULL || w->safety != SAFETY) return ASH_CONTROL_REJECT;

   i = 0;
   for (c = w->control; c != NULL; c = c->nextcontrol) {
      if (c->routine != NULL) {
	 if (i < MAX_CONTROL) con[i++] = c->routine;
       };
    };

   sts = ASH_CONTROL_REJECT;
   if (L_PROTECT(w)) {
      for (j = 0; j < i; ++j) {
	 sts = (*con[j])(msg,w);
	 if (sts >= 0) break;
       };
      L_UNPROTECT(w);
    };

   return sts;
};






/************************************************************************/
/*									*/
/*	ASH_setup_maps -- define mappings for various windows		*/
/*									*/
/************************************************************************/


void
ASH_setup_maps(w,rsfg,vfg,conf)
   ASH_WINDOW w;
   Boolean rsfg;
   Boolean vfg;
   Boolean conf;
{
   XWindowChanges chgs;
   Integer mask;
   Drawable root;
   Integer px,py,wd,ht,bw,dp;
   Integer plx,pty,nw,nh,vlx,vty;
   Window chld;
   String msg;

   mask = 0;

   WINDRAW_BEGIN(w);

   nw = abs(w->view.rx - w->view.lx) + 1;
   nh = abs(w->view.by - w->view.ty) + 1;
   if (w->parent != NULL) {
      plx = XMAPVIEW(w->parent,w->base.x);
      pty = YMAPVIEW(w->parent,w->base.y);
    }
   else {
      plx = w->base.x;
      pty = w->base.y;
    }
   pty -= abs(w->view.by - w->view.ty) + 2*w->border_width;

   if (conf) {
      XGetGeometry(DISPLAYOF(w),w->x_win,&root,&px,&py,&wd,&ht,&bw,&dp);

      w->border_width = bw;

      if (nw != wd) {
	 chgs.width = nw;
	 mask |= CWWidth;
	 rsfg = TRUE;
       };
      if (nh != ht) {
	 chgs.height = nh;
	 mask |= CWHeight;
	 rsfg = TRUE;
       };

      if (plx != px || mask != 0) {
	 chgs.x = plx;
	 mask |= CWX;
       };
      if (pty != py || mask != 0) {
	 chgs.y = pty;
	 mask |= CWY;
       };

      if (mask != 0) {
	 w->resize = TRUE;
	 XConfigureWindow(DISPLAYOF(w),w->x_win,mask,&chgs);
       };
    }
   else {
      bw = w->border_width;
      wd = nw;
      ht = nh;
      px = plx;
      py = pty;
    };

   if (rsfg) {
      msg = "ASH$RESIZE";
    }
   else {
      msg = "ASH$VIEW";
    };

   if (!w->our_court) {
      w->view_map.x = w->view.lx;
      w->view_map.y = w->view.ty;
    }
   else {
      w->view_map.x = w->view.lx - plx;
      w->view_map.y = w->view.ty - pty;
    };

   if (w->full_win != NULL) {
      vlx = w->coords.lx - w->view.lx;
      if (w->inv_x) vlx = -vlx;
      vty = w->coords.ty - w->view.ty;
      if (w->inv_y) vty = -vty;
      XGetGeometry(DISPLAYOF(w),w->full_win,&root,&px,&py,&wd,&ht,&bw,&dp);
      nw = abs(w->coords.rx - w->coords.lx) + 1;
      nh = abs(w->coords.by - w->coords.ty) + 1;
      if (w->retained) {
	 mask = 0;
	 if (vlx != px) {
	    chgs.x = vlx;
	    mask |= CWX;
	  };
	 if (vty != py) {
	    chgs.y = vty;
	    mask |= CWY;
	  };
	 if (nw != wd) {
	    chgs.width = nw;
	    mask |= CWWidth;
	  };
	 if (nh != ht) {
	    chgs.height = nh;
	    mask |= CWHeight;
	  };
	 if (!w->frame && mask != 0) {
	    XConfigureWindow(DISPLAYOF(w),w->full_win,mask,&chgs);
	  };
       }
      else {
	 if (!w->frame && (nw != wd || nh != ht || w->depth != dp)) {
	    XFreePixmap(DISPLAYOF(w),w->full_win);
	    w->full_win = XCreatePixmap(DISPLAYOF(w),w->view_win,nw,nh,w->depth);
	  };
       };
      w->full_map.x = vlx;
      w->full_map.y = vty;
      if (w->frame) {
	 XTranslateCoordinates(DISPLAYOF(w),w->view_win,w->parent->view_win,
				  XMAPVIEW(w,w->coords.lx),
				  YMAPVIEW(w,w->coords.ty),
				  &vlx,&vty,&chld);
	 w->full_map.x = -XMAPVFULL(w->parent,vlx);
	 w->full_map.y = -YMAPVFULL(w->parent,vty);
       };
    };

   ASH_fix_clip(w);

   if (vfg) XMapRaised(DISPLAYOF(w),w->x_win);

   WINDRAW_END(w);

   if (conf) {
      PROTECT;
      XFlush(DISPLAYOF(w));
      UNPROTECT;
    };

   w->resize = FALSE;
   w->rootmap_valid = FALSE;

   ASHcontrol_msg(w,msg);
};





/* end of ashscreen.c */
