/************************************************************************/
/*									*/
/*		ashproperty.c						*/
/*									*/
/*	Routines for setting/getting propertys of ASH windows		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"
#include "ash_fill.h"

#include <varargs.h>



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


				/* mask for those things we change	*/
#define DFLT_GC_MASK	(GCFont|GCForeground|GCBackground)
#define DFLT_FONT_NAME	"fixed"
#define ALT_DFLT_FONT	"6x13"

#define BWE_FONT_PATH	"%s/lib/bwe/fonts/snf/"
#define BWE_FONT_PATH0	"%s/bwe/fonts/snf/"
#define BWE_FONT_PATH1	"%s/bwe/fonts/snf.%s/"




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


	Integer 	ASH__flip_mask;

typedef struct _FILL_INFO {
   Integer	style;
} FILL_INFO;

static	FILL_INFO	fill_table[FILL_TABLE_SIZE];

typedef struct _LINE_INFO {
   Integer	line_width;
   Integer	line_style;
   Integer	cap_style;
   Integer	join_style;
   Integer	dash_offset;
   Integer	dash_len;
   Character	dash_list[MAX_DASHES];
} LINE_INFO;

static	LINE_INFO	line_table[LINE_TABLE_SIZE];

typedef struct _FONT_INFO {
   String name;
} FONT_INFO;

static	FONT_INFO	font_table[MAX_FONTS];
static	Integer 	font_ctr;

static	String		default_font_name;

static	XColor *	initial_colors;
static	String *	color_names;
static	Integer 	num_initial_colors;
static	Integer 	max_initial_colors;

static	Boolean 	color_set;
static	Boolean 	color_install;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		set_line_style();
static	ASH_COLOR	enter_color();
static	void		define_fill_table();
static	void		pat_fill_entry();
static	void		pat_tile_entry();
static	void		load_font();
static	XFontStruct *	get_font_info();
static	void		initial_GC();
static	void		initial_color_table();
static	void		initial_fill_table();
static	void		setup_fill_table();
static	void		initial_line_table();
static	void		clear_clips();
static	void		add_clip_rect();





/************************************************************************/
/*									*/
/*	Macros for routine definition					*/
/*									*/
/************************************************************************/


#define SET_DRAW(w,d,rtn)						\
   if (w != NULL && w->safety == PROP_SAFETY) { 			\
      d = (ASH_DRAWINFO) w;						\
      w = d->window;							\
    }									\
   else {								\
     CHECKWIN(w,rtn)							\
     d = w->draw;							\
    }									\



#define CHECKCOLOR(w,c) 						\
   if ((c) < 0 || (c) >= ASHinq_configuration_colors(w))		\
      ERROR("Illegal color value")


#define SET_VALUE(from,rslt,name,mask)					\
 { XGCValues vals;							\
   rslt = d->context->values.name;					\
   if (rslt != from) {							\
      PROPERTY_BEGIN(w);						\
      vals.name = from; 						\
      XChangeGC(DISPLAYOF(w),d->context,mask,&vals);			\
      if (d->full_context != NULL)					\
	 XChangeGC(DISPLAYOF(w),d->full_context,mask,&vals);		\
      PROPERTY_END(w);							\
    };									\
 }





#define SET_TEXT_VALUE(from,rslt,name,mask)				\
 { XGCValues vals;							\
   rslt = d->text_context->values.name; 				\
   if (rslt != from) {							\
      PROPERTY_BEGIN(w);						\
      vals.name = from; 						\
      XChangeGC(DISPLAYOF(w),d->text_context,mask,&vals);		\
      if (d->full_text_context != NULL) 				\
	 XChangeGC(DISPLAYOF(w),d->full_text_context,mask,&vals);	\
      PROPERTY_END(w);							\
    };									\
 }





#ifndef __STDC__

#define INQ_VALUE(name,pname)			\
int						\
ASHinq_/**/name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_/**/name);		\
   return d->context->values.pname;		\
}






#define INQ_OUR_VALUE(name,pname)		\
int						\
ASHinq_/**/name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_/**/name);		\
   return d->pname;				\
}






#define INQ_TEXT_VALUE(name,pname)		\
int						\
ASHinq_/**/name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_/**/name);		\
   return d->text_context->values.pname;	\
}

#else

#define INQ_VALUE(name,pname)			\
int						\
ASHinq_ ## name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_ ## name);		\
   return d->context->values.pname;		\
}






#define INQ_OUR_VALUE(name,pname)		\
int						\
ASHinq_ ## name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_ ## name);		\
   return d->pname;				\
}






#define INQ_TEXT_VALUE(name,pname)		\
int						\
ASHinq_ ## name(w)				\
   ASH_WINDOW w;				\
{						\
   ASH_DRAWINFO d;				\
   TRACE("ASHinq_ name 0x%x",w);                \
   SET_DRAW(w,d,ASHinq_ ## name);		\
   return d->text_context->values.pname;	\
}

#endif




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


	String		ASH__known_colors[] = {
   "White",
   "Black",
   "White",
   "Red",
   "Blue",
   "Green",
   "SpringGreen",
   "Cyan",
   "SlateBlue",
   "MediumForestGreen",
   "DarkSlateBlue",
   "Salmon",
   "DarkTurquoise",
   "Aquamarine",
   "MediumTurquoise",
   "MediumSlateBlue",
   "MediumSpringGreen",
   "Sienna",
   "Maroon",
   "Firebrick",
   "LightSteelBlue",
   "PaleGreen",
   "MediumOrchid",
   "GreenYellow",
   "YellowGreen",
   "DarkOrchid",
   "Khaki",
   "BlueViolet",
   "Brown",
   "LightGray",
   "Turquoise",
   "Pink",
   "LightBlue",
   "Gray",
   "Gold",
   "VioletRed",
   "Orange",
   "Thistle",
   "Wheat",
   "MediumVioletRed",
   "Orchid",
   "Tan",
   "Goldenrod",
   "Plum",
   "MediumGoldenrod",
   "OrangeRed",
   "Magenta",
   "Coral",
   "Yellow",
   "SteelBlue",
   "SeaGreen",
   "ForestGreen",
   "Navy",
   "MidnightBlue",
   "DarkGreen",
   "DarkSlateGray",
   "SkyBlue",
   "MediumAquamarine",
   "LimeGreen",
   "MediumBlue",
   "MediumSeaGreen",
   "CornflowerBlue",
   "IndianRed",
   "Violet",
   "DarkOliveGreen",
   "DimGray",
   "CadetBlue",
   0
};





/************************************************************************/
/*									*/
/*	ASH_property_init -- module initialization			*/
/*	ASH_property_disp_init -- initialization for a new display	*/
/*									*/
/************************************************************************/


void
ASH_property_init()
{
   String s;

   font_ctr = 2;
   initial_colors = NULL;
   color_names = NULL;
   num_initial_colors = 0;
   max_initial_colors = 0;

   initial_fill_table();
   initial_line_table();

   default_font_name = ASHinq_resource("font");
   if (default_font_name == NULL) default_font_name = DFLT_FONT_NAME;
   font_table[0].name = default_font_name;
   font_table[1].name = ASH_ICON_FONT_NAME;

   color_set = TRUE;
   color_install = TRUE;
   s = ASHinq_resource("cmset");
   if (s != NULL && STREQL(s,"off")) color_set = FALSE;
   s = ASHinq_resource("cminstall");
   if (s != NULL && STREQL(s,"off")) color_install = FALSE;

   ASH__flip_mask = 1;
   s = ASHinq_resource("flip_mask");
   if (s != NULL) ASH__flip_mask = atol(s);
   if (ASH__flip_mask == 0) ASH__flip_mask = 1;
};





void
ASH_property_disp_init(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   register Integer i;
   String * paths;
   String * opaths;
   Character path[128];
   int ct;
   String s;

   s = getenv("BWE_FONT_PATH");
   if (s != NULL && access(s,1) >= 0) strcpy(path,s);
   else {
      sprintf(path,BWE_FONT_PATH,BWEbwe_project(),BWEarch());
      if (access(path,1) < 0) {
	 sprintf(path,BWE_FONT_PATH0,BWEbwe_project(),BWEarch());
	 if (access(path,1) < 0) {
	    sprintf(path,BWE_FONT_PATH1,BWEbwe_project(),BWEarch());   /* shd be server arch */
	  };
       };
    };

   if (getenv("NO_FONT_PATH") == NULL) {
      PROTECT;
      opaths = XGetFontPath(d->display,&ct);
      paths = (String *) alloca((ct+2)*sizeof(String));
      for (i = 0; i < ct; ++i) {
	 if (STREQL(opaths[i],path)) break;
	 paths[i] = opaths[i];
       };
      if (i >= ct) {
	 paths[ct] = path;
	 XSetFontPath(d->display,paths,ct+1);
       };
      XFreeFontPath(opaths);
      UNPROTECT;
    };

   initial_GC(d);
   initial_color_table(d);
   setup_fill_table(d,ffg);

   PROTECT;
   d->data->font_table[0] = d->data->default_GC.font;
   d->data->font_info[0] = XQueryFont(d->display,d->data->font_table[0]);
   UNPROTECT;

   if (d->color_visual != d->default_visual && d->top_colormap == NULL)
      color_set = FALSE;

   for (i = 1; i < font_ctr; ++i) {
      load_font(d,i);
    };
};





/************************************************************************/
/*									*/
/*	ASHcolor -- set drawing color for current window		*/
/*	ASHbackground_color -- set background color for current window	*/
/*	ASHcombination_rule -- set combination rule for current window	*/
/*	ASHplane_mask -- set plane mask 				*/
/*									*/
/************************************************************************/


ASH_COLOR
ASHcolor(w,color)
   ASH_WINDOW w;
   ASH_COLOR color;
{
   ASH_COLOR c;
   ASH_DRAWINFO d;

   TRACE("ASHcolor 0x%x %d",w,color);

   SET_DRAW(w,d,ASHcolor);

   CHECKCOLOR(w,color);

   SET_VALUE(color,c,foreground,GCForeground);

   return c;
};





ASH_COLOR
ASHbackground_color(w,color)
   ASH_WINDOW w;
   ASH_COLOR color;
{
   ASH_COLOR c;
   ASH_DRAWINFO d;

   TRACE("ASHbackground_color 0x%x %d",w,color);

   SET_DRAW(w,d,ASHbackground_color);

   CHECKCOLOR(w,color);

   SET_VALUE(color,c,background,GCBackground);

   return c;
};





int
ASHcombination_rule(w,rule)
   ASH_WINDOW w;
   Integer rule;
{
   register Integer r;
   ASH_DRAWINFO d;

   TRACE("ASHcombination_rule 0x%x %d",w,rule);

   if (rule < 0 || rule >= 16) {
      ERROR("Illegal combination rule");
      return rule;
    };

   SET_DRAW(w,d,ASHcombination_rule);

   SET_VALUE(rule,r,function,GCFunction);

   return r;
};






int
ASHplane_mask(w,mask)
   ASH_WINDOW w;
   Integer mask;
{
   register Integer m;
   ASH_DRAWINFO d;

   TRACE("ASHplane_mask 0x%x %d",w,mask);

   SET_DRAW(w,d,ASHplane_mask);
   SET_VALUE(mask,m,plane_mask,GCPlaneMask);

   return m;
};





/************************************************************************/
/*									*/
/*	ASHwindow_draw_thru -- make window draw thru			*/
/*									*/
/************************************************************************/


int
ASHwindow_draw_thru(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   Integer oval,nval;
   ASH_DRAWINFO d;

   TRACE("ASHwindow_draw_thru 0x%x %d",w,fg);

   SET_DRAW(w,d,ASHwindow_draw_thru);

   if (w->safety != PROP_SAFETY && w->our_court) fg = TRUE;

   nval = (fg ? IncludeInferiors : ClipByChildren);

   SET_VALUE(nval,oval,subwindow_mode,GCSubwindowMode);
   SET_TEXT_VALUE(nval,oval,subwindow_mode,GCSubwindowMode);

   fg = (oval == IncludeInferiors);

   return fg;
};





/************************************************************************/
/*									*/
/*	ASHfill -- set fill pattern for current window			*/
/*									*/
/************************************************************************/


ASH_FILL_STYLE
ASHfill(w,fill)
   ASH_WINDOW w;
   ASH_FILL_STYLE fill;
{
   ASH_FILL_STYLE f;
   XGCValues vals;
   Integer mask;
   ASH_DRAWINFO d;

   TRACE("ASHfill 0x%x %d",w,fill);

   SET_DRAW(w,d,ASHfill);

   f = d->fill_style;

   if (fill == f) return fill;

   if (fill < 0 || fill >= FILL_TABLE_SIZE) {
      ERROR("Illegal fill value");
      return fill;
    };

   PROPERTY_BEGIN(w);

   mask = GCFillStyle;
   vals.fill_style = fill_table[fill].style;
   if (fill_table[fill].style == FillTiled) {
      vals.tile = d->window->display->data->fill_tile[fill];
      mask |= GCTile;
    }
   else if (fill_table[fill].style != FillSolid) {
      vals.stipple = d->window->display->data->fill_tile[fill];
      mask |= GCStipple;
    };

   d->fill_style = fill;
   XChangeGC(PROPDISPLAYOF(d),d->context,mask,&vals);
   XChangeGC(PROPDISPLAYOF(d),d->text_context,mask,&vals);

   if (d->full_context != NULL) {
      XChangeGC(PROPDISPLAYOF(d),d->full_context,mask,&vals);
    };
   if (d->full_text_context != NULL) {
      XChangeGC(PROPDISPLAYOF(d),d->full_text_context,mask,&vals);
    };

   PROPERTY_END(w);

   if (d->line_style >= 0) set_line_style(w,d->line_style);

   return f;
};





/************************************************************************/
/*									*/
/*	ASHline_style -- set line style for current window		*/
/*	set_line_style -- handle setting line style			*/
/*									*/
/************************************************************************/


ASH_LINE_STYLE
ASHline_style(w,style)
   ASH_WINDOW w;
   ASH_LINE_STYLE style;
{
   ASH_LINE_STYLE s;
   ASH_DRAWINFO d;

   TRACE("ASHline_style 0x%x %d",w,style);

   SET_DRAW(w,d,ASHline_style);

   s = d->line_style;
   if (s == style) return style;

   if (style < 0 || style >= LINE_TABLE_SIZE) {
      ERROR("Illegal line style value");
      return style;
    };

   set_line_style(w,style);

   return s;
};





static void
set_line_style(w,style)
   ASH_WINDOW w;
   Integer style;
{
   register Integer wid;
   ASH_DRAWINFO d;

   PROPERTY_BEGIN(w);

   SET_DRAW(w,d,set_line_style);

   d->line_style = style;

   wid = line_table[style].line_width;
   if (wid == 0 && d->fill_style >= 0 &&
	  fill_table[d->fill_style].style != FillSolid) wid = 1;

   XSetLineAttributes(PROPDISPLAYOF(d),d->context,
			 wid,
			 line_table[style].line_style,
			 line_table[style].cap_style,
			 line_table[style].join_style);
   XSetDashes(PROPDISPLAYOF(d),d->context,
		 line_table[style].dash_offset,
		 line_table[style].dash_list,
		 line_table[style].dash_len);

   if (d->full_context != NULL) {
      XSetLineAttributes(PROPDISPLAYOF(d),d->full_context,
			    wid,
			    line_table[style].line_style,
			    line_table[style].cap_style,
			    line_table[style].join_style);
      XSetDashes(PROPDISPLAYOF(d),d->full_context,
		    line_table[style].dash_offset,
		    line_table[style].dash_list,
		    line_table[style].dash_len);
    };

   PROPERTY_END(w);
};





/************************************************************************/
/*									*/
/*	ASHclip -- set clipping for current window on or off		*/
/*									*/
/************************************************************************/


int
ASHclip(w,flag)
   ASH_WINDOW w;
   Integer flag;
{
   register Boolean fg;
   ASH_DRAWINFO d;

   TRACE("ASHclip 0x%x %d",w,flag);

   SET_DRAW(w,d,ASHclip);

   fg = d->use_clip;
   flag = (flag != 0);
   if (fg == flag) return flag;

   d->use_clip = flag;

   if (w->draw == d) ASH_fix_clip(w);

   return fg;
};






/************************************************************************/
/*									*/
/*	ASHclip_region -- set clipping region				*/
/*	ASHclip_add_region -- add another clip region			*/
/*	ASHclip_window -- set a clipping window 			*/
/*									*/
/************************************************************************/


void
ASHclip_region(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
{
   ASH_DRAWINFO d;

   TRACE("ASHclip_region 0x%x (%d,%d)->(%d,%d)",w,lx,by,rx,ty);

   SET_DRAW(w,d,ASHclip_region);

   clear_clips(w,&d->clip,FALSE);
   add_clip_rect(w,&d->clip,lx,by,rx,ty);

   if (d->use_clip && w->draw == d) ASH_fix_clip(w);
};





void
ASHclip_add_region(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   ASH_DRAWINFO d;

   TRACE("ASHclip_add_region 0x%x (%d,%d)->(%d,%d)",w,lx,by,rx,ty);

   SET_DRAW(w,d,ASHclip_add_region);

   add_clip_rect(w,&d->clip,lx,by,rx,ty);

   if (d->use_clip && w->draw == d) ASH_fix_clip(w);
};





void
ASHclip_window(w,cw)
   ASH_WINDOW w;
   ASH_WINDOW cw;
{
   ASH_DRAWINFO d;

   TRACE("ASHclip_window 0x%x 0x%x",w,cw);

   SET_DRAW(w,d,ASHclip_window);

   CHECKWIN(cw,ASHclip_window);

   if (cw->full_win == NULL || cw->depth != 1) {
      ERROR("Must specify depth 1 saved window for clip window");
      return;
    };

   clear_clips(w,&d->clip,FALSE);
   d->clip.type = CLIP_MAP;
   d->clip.map = cw;

   if (d->use_clip && w->draw == d) ASH_fix_clip(w);
};





/************************************************************************/
/*									*/
/*	ASHouter_clip -- turn on/off outer clipping			*/
/*	ASHouter_clip_region -- set outer clipping region		*/
/*									*/
/************************************************************************/


int
ASHouter_clip(w,flag)
   ASH_WINDOW w;
   Integer flag;
{
   register Boolean fg;
   ASH_DRAWINFO d;

   TRACE("ASHouter_clip 0x%x %d",w,flag);

   SET_DRAW(w,d,ASHouter_clip);

   fg = d->out_clip;
   flag = (flag != 0);
   if (fg == flag) return flag;

   d->out_clip = flag;

   ASH_fix_clip(w);

   return fg;
};






void
ASHouter_clip_region(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
{
   ASH_DRAWINFO d;

   TRACE("ASHouter_clip_region 0x%x (%d,%d)->(%d,%d)",w,lx,by,rx,ty);

   SET_DRAW(w,d,ASHouter_clip_region);

   clear_clips(w,&d->outer_clip,FALSE);
   add_clip_rect(w,&d->outer_clip,lx,by,rx,ty);

   if (d->out_clip && w->draw == d) ASH_fix_clip(w);
};





void
ASHouter_clip_add_region(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   ASH_DRAWINFO d;

   TRACE("ASHouter_clip_add_region 0x%x (%d,%d)->(%d,%d)",w,lx,by,rx,ty);

   SET_DRAW(w,d,ASHouter_clip_add_region);

   add_clip_rect(w,&d->outer_clip,lx,by,rx,ty);

   if (d->out_clip && w->draw == d) ASH_fix_clip(w);
};





void
ASHouter_clip_window(w,cw)
   ASH_WINDOW w;
   ASH_WINDOW cw;
{
   ASH_DRAWINFO d;

   TRACE("ASHouter_clip_window 0x%x 0x%x",w,cw);

   SET_DRAW(w,d,ASHouter_clip_window);

   CHECKWIN(cw,ASHclip_window);

   if (cw->full_win == NULL || cw->depth != 1) {
      ERROR("Must specify depth 1 saved window for clip window");
      return;
    };

   clear_clips(w,&d->outer_clip,FALSE);
   d->outer_clip.type = CLIP_MAP;
   d->outer_clip.map = cw;

   if (d->out_clip && w->draw == d) ASH_fix_clip(w);
};





/************************************************************************/
/*									*/
/*	ASHfont -- set text font for current window			*/
/*	ASHtext_color -- set text color for current window		*/
/*	ASHtext_background_color -- set text background color		*/
/*	ASHtext_background_clear -- set text background color/clear mode*/
/*	ASHtext_combination_rule -- set combination rule for text	*/
/*	ASHtext_plane_mask -- set plane mask				*/
/*									*/
/************************************************************************/


ASH_FONT
ASHfont(w,font)
   ASH_WINDOW w;
   ASH_FONT font;
{
   ASH_FONT f;
   Font ft,ft1;
   ASH_DRAWINFO d;

   TRACE("ASHfont 0x%x %d",w,font);

   SET_DRAW(w,d,ASHfont);

   if (d->font == font) return font;
   f = d->font;
   d->font = font;

   ft = (Integer) ASH_real_font(d->window->display,font);

   SET_TEXT_VALUE(ft,ft1,font,GCFont);

   return f;
};






ASH_COLOR
ASHtext_color(w,color)
   ASH_WINDOW w;
   Integer color;
{
   ASH_COLOR c;
   ASH_DRAWINFO d;

   TRACE("ASHtext_color 0x%x %d",w,color);

   SET_DRAW(w,d,ASHtext_color);

   CHECKCOLOR(w,color);

   SET_TEXT_VALUE(color,c,foreground,GCForeground);

   return c;
};





ASH_COLOR
ASHtext_background_color(w,color)
   ASH_WINDOW w;
   ASH_COLOR color;
{
   ASH_COLOR c;
   register Boolean fg;
   ASH_DRAWINFO d;

   TRACE("ASHtext_background_color 0x%x %d",w,color);

   if (color == -1) {
      color = 0;
      fg = TRUE;
    }
   else fg = FALSE;

   SET_DRAW(w,d,ASHtext_background_color);

   CHECKCOLOR(w,color);

   SET_TEXT_VALUE(color,c,background,GCBackground);

   if (d->clear_text) c = -1;
   d->clear_text = fg;

   return c;
};





ASH_COLOR
ASHtext_background_clear(w,color)
   ASH_WINDOW w;
   ASH_COLOR color;
{
   ASH_COLOR c;
   ASH_DRAWINFO d;

   TRACE("ASHtext_background_color 0x%x %d",w,color);

   SET_DRAW(w,d,ASHtext_background_clear);

   CHECKCOLOR(w,color);

   SET_TEXT_VALUE(color,c,background,GCBackground);

   d->clear_text = TRUE;

   return c;
};





int
ASHtext_combination_rule(w,rule)
   ASH_WINDOW w;
   Integer rule;
{
   register Integer r;
   ASH_DRAWINFO d;

   TRACE("ASHtext_combination_rule 0x%x %d",w,rule);

   if (rule < 0 || rule >= 16) {
      ERROR("Illegal combination rule");
      return rule;
    };

   SET_DRAW(w,d,ASHtext_combination_rule);

   SET_TEXT_VALUE(rule,r,function,GCFunction);

   return r;
};






int
ASHtext_plane_mask(w,mask)
   ASH_WINDOW w;
   Integer mask;
{
   register Integer m;
   ASH_DRAWINFO d;

   TRACE("ASHtext_plane_mask 0x%x %d",w,mask);

   SET_DRAW(w,d,ASHtext_plane_mask);

   SET_TEXT_VALUE(mask,m,plane_mask,GCPlaneMask);

   return m;
};





/************************************************************************/
/*									*/
/*	ASHsource -- set source window for blts 			*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHsource(w,win)
   ASH_WINDOW w;
   ASH_WINDOW win;
{
   register ASH_WINDOW ow;
   ASH_DRAWINFO d;

   TRACE("ASHsource 0x%x %x",w,win);

   SET_DRAW(w,d,ASHsource);

   CHECKWIN(win,ASHsource);

   ow = d->source;
   d->source = win;

   return ow;
};






/************************************************************************/
/*									*/
/*	ASHnew_color_table -- define new color table			*/
/*	ASHfill_color_table -- fill color table 			*/
/*	ASHset_color_table -- set color table for window (& activate)	*/
/*	ASHinq_color_table -- return color table of window		*/
/*	ASHlookup_color -- return color ID given name			*/
/*									*/
/************************************************************************/


ASH_COLOR_TABLE
ASHnew_color_table(w)
   ASH_WINDOW w;
{
   Colormap cmp;

   ENTER("ASHnew_color_table 0x%x",w);

   if (w == NULL) w = ASHinq_top();

   CHECKWIN(w,ASHnew_color_table);

   PROTECT;

   cmp = XCreateColormap(DISPLAYOF(w),w->x_win,
			    w->display->color_visual,
			    AllocAll);

   UNPROTECT;

   return (ASH_COLOR_TABLE) cmp;
};




void
ASHfill_color_table(w,ctbl,base,numcol,r,g,b)
   ASH_WINDOW w;
   ASH_COLOR_TABLE ctbl;
   ASH_COLOR base;
   Integer numcol;
   unsigned short r[];		 /* [ numcol ] */
   unsigned short g[];		 /* [ numcol ] */
   unsigned short b[];		 /* [ numcol ] */
{
   XColor defs[256];
   register Integer i;
   XWindowAttributes attrs;

   TRACE("ASHfill_color_table 0x%x 0x%x %d %d (%d,%d,%d),...",w,ctbl,base,numcol,r[0],g[0],b[0]);

   CHECKWIN(w,ASHwindow_color_table);

   for (i = 0; i < numcol; ++i) {
      defs[i].pixel = base+i;
      defs[i].red = r[i];
      defs[i].green = g[i];
      defs[i].blue = b[i];
      defs[i].flags = DoRed|DoGreen|DoBlue;
    };

   PROTECT;

   if (ctbl == NULL) {
      XGetWindowAttributes(DISPLAYOF(w),w->x_win,&attrs);
      ctbl = (ASH_COLOR_TABLE) attrs.colormap;
    };

   XStoreColors(DISPLAYOF(w),ctbl,defs,numcol);

   UNPROTECT;
};





ASH_COLOR_TABLE
ASHset_color_table(w,ctbl)
   ASH_WINDOW w;
   ASH_COLOR_TABLE ctbl;
{
   ASH_COLOR_TABLE cmp;

   TRACE("ASHset_color_table 0x%x 0x%x",w,ctbl);

   cmp = w->colormap;

   if (ctbl == NULL) {
      w->colormap = w->display->colormap;
    }
   else {
      w->colormap = ctbl;
    };

   PROTECT;
   if (color_set) XSetWindowColormap(DISPLAYOF(w),w->x_win,w->colormap);
   if (color_install || VendorRelease(DISPLAYOF(w)) < 3)
      XInstallColormap(DISPLAYOF(w),w->colormap);
   UNPROTECT;

   return cmp;
};





ASH_COLOR_TABLE
ASHinq_color_table(w)
   ASH_WINDOW w;
{
   TRACE("ASHinq_color_table 0x%x",w);

   return w->colormap;
};





ASH_COLOR
ASHlookup_color(w,name)
   ASH_WINDOW w;
   String name;
{
   XColor def;
   Integer i,j;
   Character buf[128];

   ENTER("ASHlookup_color 0x%x %s",w,name);

   if (w == NULL) w = ASHinq_top();

   if (name == NULL) {
      ERROR("Attempt to lookup <null> color");
      return 1;
    };

   CHECKWIN(w,ASHlookup_color);

   for (i = 0; i < num_initial_colors; ++i) {
      if (color_names[i] != NULL && STREQL(color_names[i],name)) {
	 return initial_colors[i].pixel;
       };
    };

   PROTECT;

   j = XParseColor(DISPLAYOF(w),w->colormap,name,&def);
   def.flags = DoRed|DoGreen|DoBlue;

   UNPROTECT;

   if (j == 0) {
      sprintf(buf,"Invalid color definition '%s'",name);
      ERROR(buf);
      return 1;
    };

   return enter_color(w,&def,name);
};





static ASH_COLOR
enter_color(w,defp,name)
   ASH_WINDOW w;
   XColor * defp;
   String name;
{
   XColor def,def1;
   Integer i,j;
   Character buf[128];
   Integer mask,pix;

   PROTECT;

   def = *defp;
   def.pixel = -1;

   for (i = 0; i < num_initial_colors; ++i) {
      if (color_names[i] != NULL && STREQL(color_names[i],name)) break;
    };
   if (i >= num_initial_colors) {
      i = num_initial_colors++;
      if (i >= max_initial_colors) {
	 max_initial_colors *= 2;
	 initial_colors = (XColor *) realloc(initial_colors,
						sizeof(XColor)*max_initial_colors);
	 color_names = (String *) realloc(color_names,
					     sizeof(String)*max_initial_colors);
       };
      color_names[i] = NULL;
      pix = -1;

      j = XAllocColorCells(DISPLAYOF(w),w->colormap,0,&mask,1,&pix,1);
      if (j == 0) {
	 j = XAllocColor(DISPLAYOF(w),w->colormap,&def);
	 if (j == 0) def.pixel = -1;
	 color_names[i] = SALLOC(name);
	 initial_colors[i] = def;
       }
      else {
	 ASH__flip_mask = 1;
	 def.pixel = pix;
	 XStoreColor(DISPLAYOF(w),w->colormap,&def);
	 color_names[i] = SALLOC(name);
	 def.pixel = pix;
	 initial_colors[i] = def;
	 def1 = def;
	 def1.pixel |= mask;
	 def1.red = 65535-def1.red;
	 def1.green = 65535-def1.green;
	 def1.blue = 65535-def1.blue;
	 XStoreColor(DISPLAYOF(w),w->colormap,&def1);
       }
    }
   else {
      def = initial_colors[i];
    };

   UNPROTECT;

   if (def.pixel == -1) {
      sprintf(buf,"Can't allocate color '%s'",name);
      ERROR(buf);
      def.pixel = 1;
    };

   return def.pixel;
};





/************************************************************************/
/*									*/
/*	ASHaverage_color -- get a color between two colors		*/
/*									*/
/************************************************************************/


ASH_COLOR
ASHaverage_color(w,c1,c2,d1,d2)
   ASH_WINDOW w;
   ASH_COLOR c1;
   ASH_COLOR c2;
   Integer d1,d2;
{
   XColor cols[2];
   Character nbuf[128];
   Integer i;

   if (w == NULL) w = ASHinq_top();

   cols[0].pixel = c1;
   cols[1].pixel = c2;
   sprintf(nbuf,"AVG_%d_%d_%d_%d",c1,c2,d1,d2);

   for (i = 0; i < num_initial_colors; ++i) {
      if (color_names[i] != NULL && STREQL(color_names[i],nbuf)) {
	 return initial_colors[i].pixel;
       };
    };

   PROTECT;
   XQueryColors(DISPLAYOF(w),w->colormap,cols,2);
   UNPROTECT;

   cols[0].red = (cols[0].red*d1 + cols[1].red*d2)/(d1+d2);
   cols[0].green = (cols[0].green*d1 + cols[1].green*d2)/(d1+d2);
   cols[0].blue = (cols[0].blue*d1 + cols[1].blue*d2)/(d1+d2);
   cols[0].flags |= cols[1].flags;

   return enter_color(w,&cols[0],nbuf);
};





/************************************************************************/
/*									*/
/*	ASHfill_table -- load fill pattern table			*/
/*	ASHfill_table_clear -- load fill pattern table w/ clear pats	*/
/*									*/
/*	define_fill_table -- add stipple patterns to table		*/
/*	pat_fill_entry -- define entry for one display			*/
/*									*/
/************************************************************************/


void
ASHfill_table(base,numpat,pats)
   Integer base;
   Integer numpat;
   Short pats[];	/* [ numpat ] */
{
   ENTER("ASHfill_table %d %d (0x%x),...",base,numpat,pats[0]);

   define_fill_table(base,numpat,pats,FillOpaqueStippled);
};






void
ASHfill_table_clear(base,numpat,pats)
   Integer base;
   Integer numpat;
   Short pats[];	/* [ numpat ] */
{
   ENTER("ASHfill_table_clear %d %d (0x%x),...",base,numpat,pats[0]);

   define_fill_table(base,numpat,pats,FillStippled);
};





static void
define_fill_table(base,ct,pats,sty)
   Integer base;
   Integer ct;
   Short pats[];
   Integer sty;
{
   register Integer i;

   if (base < 0) {
      ERROR("Bad fill index for table definition");
      return;
    };

   for (i = 0; i < ct; ++i) {
      if (base+i >= FILL_TABLE_SIZE) {
	 ERROR("Bad fill index for table definition");
	 break;
       };
      ASH_for_all_displays(pat_fill_entry,base+i,&pats[i*16],sty);
    };
}





static void
pat_fill_entry(d,b,pat,sty)
   DISPLAY d;
   Integer b;
   unsigned short pat[16];
   Integer sty;
{
   register Integer j;
   register Pixmap pix;

   PROTECT;

   for (j = 0; j < 16; ++j) if (pat[j] != 0xffff) break;
   if (j >= 16) pix = NULL;
   else pix = ASH_make_pixmap(d,pat);

   if (d->data->fill_tile[b] != NULL)
      XFreePixmap(d->display,d->data->fill_tile[b]);

   d->data->fill_tile[b] = pix;
   fill_table[b].style = (pix == NULL ? FillSolid : sty);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHfill_table_tile -- load fill tile from ASH_WINDOW		*/
/*									*/
/*	pat_tile_entry -- define this for one display			*/
/*									*/
/************************************************************************/


void
ASHfill_table_tile(base,w)
   Integer base;
   ASH_WINDOW w;
{
   TRACE("ASHfill_table_tile %d 0x%x",base,w);

   if (base < 0 || base >= FILL_TABLE_SIZE) {
      ERROR("Bad fill index for table definition");
      return;
    };

   ASH_for_all_displays(pat_tile_entry,base,w);
};





static void
pat_tile_entry(d,b,w)
   DISPLAY d;
   Integer b;
   ASH_WINDOW w;
{
   Pixmap pix;

   pix = ASH_window_to_pixmap(w,0,0,d);

   PROTECT;

   if (d->data->fill_tile[b] != NULL)
      XFreePixmap(d->display,d->data->fill_tile[b]);

   d->data->fill_tile[b] = pix;
   fill_table[b].style = FillTiled;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHline_table -- define line table entry			*/
/*									*/
/************************************************************************/


void
ASHline_table(va_alist)
   va_dcl
{
   va_list ap;
   register Integer id,i,j;

   va_start(ap);
   id = va_arg(ap,Integer);

   line_table[id].line_width = abs(va_arg(ap,Integer));
   line_table[id].line_style = va_arg(ap,Integer);
   line_table[id].cap_style = va_arg(ap,Integer);
   line_table[id].join_style = va_arg(ap,Integer);
   line_table[id].dash_offset = va_arg(ap,Integer);
   line_table[id].dash_len = va_arg(ap,Integer);

   if (line_table[id].dash_len > 0) {
      if (line_table[id].dash_len > MAX_DASHES) line_table[id].dash_len = MAX_DASHES;
      for (i = 0; i < line_table[id].dash_len; ++i) {
	 line_table[id].dash_list[i] = va_arg(ap,Integer);
	 if (line_table[id].dash_list[i] == 0) {
	    if (i == 0) {
	       line_table[id].dash_len = 2;
	       line_table[id].dash_list[0] = 4;
	       line_table[id].dash_list[1] = 4;
	     }
	    else line_table[id].dash_len = i;
	    break;
	  };
       };
    }
   else {
      line_table[id].dash_len = 2;
      line_table[id].dash_list[0] = 4;
      line_table[id].dash_list[1] = 4;
    };

   if (line_table[id].dash_len & 1) {
      for (i = 0; i < line_table[id].dash_len; ++i) {
	 j = line_table[id].dash_len + i;
	 if (j >= MAX_DASHES) break;
	 line_table[id].dash_list[j] = line_table[id].dash_list[i];
       };
      line_table[id].dash_len *= 2;
      if (line_table[id].dash_len > MAX_DASHES) line_table[id].dash_len = MAX_DASHES;
    };
};





/************************************************************************/
/*									*/
/*	ASHinq_base_font -- return base font name			*/
/*	ASHloadfont -- load the named font, return its id		*/
/*									*/
/************************************************************************/


String
ASHinq_base_font()
{
   return default_font_name;
};





ASH_FONT
ASHloadfont(nm)
   String nm;
{
   register Integer i;

   if (nm == NULL) nm = "";

   ENTER("ASHloadfont %s",nm);

   if (nm == NULL || *nm == 0) return 0;

   for (i = 1; i < font_ctr; ++i) {
      if (STREQL(nm,font_table[i].name)) return i;
    };

   if (font_ctr+1 >= MAX_FONTS) {
      ERROR("Too many fonts loaded");
      return 0;
    };

   PROTECT;
   i = font_ctr++;
   font_table[i].name = SALLOC(nm);
   UNPROTECT;

   ASH_for_all_displays(load_font,i);

   return (ASH_FONT) i;
};





static void
load_font(d,i)
   DISPLAY d;
   Integer i;
{
   Character buf[1024];

   if (i == ASH_ICON_FONT) return;

   PROTECT;

   buf[0] = 0;
   d->data->font_info[i] = XLoadQueryFont(d->display,font_table[i].name);
   if (d->data->font_info[i] == NULL) {
      sprintf(buf,"Couldn't load font %s",font_table[i].name);
      d->data->font_table[i] = NULL;
    }
   else d->data->font_table[i] = d->data->font_info[i]->fid;

   UNPROTECT;

   if (buf[0] != 0) ASH_error(buf);
};





/************************************************************************/
/*									*/
/*	ASHinq_text_extent -- return size of string			*/
/*									*/
/************************************************************************/


void
ASHinq_text_extent(aft,s,xp,yp)
   ASH_FONT aft;
   String s;
   Integer *xp,*yp;
{
   XFontStruct * xfs;
   Integer dir,asc,des;
   XCharStruct over;
   Integer ln,rt;
   DISPLAY d;

   TRACE("ASHinq_text_extent 0x%x <%s>",aft,s);

   aft = FONTOF(aft);
   if (aft == ASH_ICON_FONT) {
      ASHicon_size(s,xp,yp);
      return;
    };

   d = ASH_inq_top_display();
   xfs = get_font_info(d,aft);

   ln = strlen(s);
   if (ln == 0) {
      rt = asc = des = 0;
    }
   else {
      PROTECT;

      if (yp == NULL) {
	 rt = XTextWidth(xfs,s,ln);
       }
      else {
	 XTextExtents(xfs,s,ln,&dir,&asc,&des,&over);
	 rt = over.rbearing;
	 asc = over.ascent;
	 des = over.descent;
       };

      UNPROTECT;
    };

   if (xp != NULL) *xp = rt;
   if (yp != NULL) *yp = asc+des+1;
};





/************************************************************************/
/*									*/
/*	ASHinq_text_offset -- return offset for locating string 	*/
/*									*/
/************************************************************************/


void
ASHinq_text_offset(aft,s,xp,yp)
   ASH_FONT aft;
   String s;
   Integer *xp,*yp;
{
   XFontStruct * xfs;
   Integer dir,asc,des;
   XCharStruct over;
   Integer ln,lb;
   DISPLAY d;

   TRACE("ASHinq_text_offset 0x%x <%s>",aft,s);

   aft = FONTOF(aft);
   if (aft == ASH_ICON_FONT) {
      if (xp != NULL) *xp = 0;
      if (yp != NULL) *yp = 0;
      return;
    };

   d = ASH_inq_top_display();
   xfs = get_font_info(d,aft);

   ln = strlen(s);
   if (ln == 0) {
      lb = des = 0;
    }
   else {
      PROTECT;

      if (yp == NULL) ln = 1;

      XTextExtents(xfs,s,ln,&dir,&asc,&des,&over);
      lb = -over.lbearing;
      if (over.descent > des) des = over.descent;

      UNPROTECT;
    };

   if (xp != NULL) *xp = lb;
   if (yp != NULL) *yp = des;
};





/************************************************************************/
/*									*/
/*	ASHinq_text_next -- return location for next string		*/
/*									*/
/************************************************************************/


void
ASHinq_text_next(aft,s,xp,yp)
   ASH_FONT aft;
   String s;
   Integer *xp,*yp;
{
   XFontStruct * xfs;
   Integer ln,wd,ht;
   DISPLAY d;

   TRACE("ASHinq_text_next 0x%x <%s>",aft,s);

   aft = FONTOF(aft);
   if (aft == ASH_ICON_FONT) {
      ASHicon_size(s,xp,yp);
      return;
    };

   d = ASH_inq_top_display();
   xfs = get_font_info(d,aft);

   PROTECT;

   ln = strlen(s);
   wd = XTextWidth(xfs,s,ln);
   ht = xfs->ascent + xfs->descent;

   UNPROTECT;

   if (xp != NULL) *xp = wd;
   if (yp != NULL) *yp = ht+3;
};





/************************************************************************/
/*									*/
/*	ASHinq_text_info -- return lots of info regarding text string	*/
/*									*/
/************************************************************************/


void
ASHinq_text_info(aft,s,xszp,yszp,xoffp,yoffp,xnxp,ynxp)
   ASH_FONT aft;
   String s;
   Integer *xszp, *yszp;
   Integer *xoffp, *yoffp;
   Integer *xnxp, *ynxp;
{
   XFontStruct * xfs;
   Integer dir,asc,des;
   XCharStruct over;
   Integer ln,lb,rt,a,d;
   DISPLAY disp;

   TRACE("ASHinq_text_info 0x%x <%s>",aft,s);

   aft = FONTOF(aft);
   if (aft == ASH_ICON_FONT) {
      ASHicon_size(s,xszp,yszp);
      ASHicon_size(s,xnxp,ynxp);
      if (xoffp != NULL) *xoffp = 0;
      if (yoffp != NULL) *yoffp = 0;
      return;
    };

   disp = ASH_inq_top_display();
   xfs = get_font_info(disp,aft);

   ln = strlen(s);
   if (ln == 0) {
      lb = rt = asc = des = a = d = 0;
    }
   else {
      PROTECT;

      XTextExtents(xfs,s,ln,&dir,&asc,&des,&over);
      lb = over.lbearing;
      rt = over.rbearing;
      a = over.ascent;
      d = over.descent;

      UNPROTECT;
    };

   if (xszp != NULL) *xszp = rt;
   if (yszp != NULL) *yszp = a+d;
   if (xoffp != NULL) *xoffp = lb;
   if (yoffp != NULL) *yoffp = d-1;
   if (xnxp != NULL) *xnxp = rt;
   if (ynxp != NULL) *ynxp = asc+des;
};





/************************************************************************/
/*									*/
/*	ASHinq_OB4t_info -- get information about a font		*/
/*									*/
/************************************************************************/


Integer
ASHinq_font_info(nm,hspp,vspp,spp,wid,dwn,up)
   String nm;
   Integer * hspp;
   Integer * vspp;
   Integer * spp;
   Character wid[];	/* [ 128 ] */
   Character dwn[];	/* [ 128 ] */
   Character up[];	/* [ 128 ] */
{
   register XFontStruct *xfs;
   register Integer i,j;
   DISPLAY d;
   register XCharStruct *xch;

   d = ASH_inq_top_display();

   PROTECT;

   xfs = XLoadQueryFont(d->display,nm);

   if (xfs == NULL) {
      UNPROTECT;
      return FALSE;
    };

   if (hspp != NULL) *hspp = 0;
   if (vspp != NULL) {
      i = xfs->descent - xfs->max_bounds.descent;
      j = xfs->ascent - xfs->max_bounds.ascent;
      if (i < 0) i = 0;
      if (j < 0) j = 0;
      if (i < j) i = j;
      *vspp = i;
    };

   if (xfs->default_char < xfs->min_char_or_byte2 ||
	  xfs->default_char > xfs->max_char_or_byte2) xfs->default_char = ' ';

   j = ' ' - xfs->min_char_or_byte2;
   if (j >= 0 && xfs->per_char != NULL) {
      xch = &xfs->per_char[j];
      xch->ascent = xfs->max_bounds.ascent;
      xch->descent = xfs->max_bounds.descent;
    };

   for (i = 0; i < 128; ++i) {
      if (i < xfs->min_char_or_byte2) j = xfs->default_char;
      else if (i > xfs->max_char_or_byte2) j = xfs->default_char;
      else j = i;
      j -= xfs->min_char_or_byte2;
      if (xfs->per_char != NULL) xch = &xfs->per_char[j];
      else xch = &xfs->min_bounds;

      if (wid != NULL) wid[i] = xch->width;
      if (dwn != NULL) dwn[i] = xch->descent;
      if (up != NULL) up[i] = xch->ascent;
      if (i == ' ' && spp != NULL)
	 *spp = xch->lbearing + xch->rbearing;
    };

   UNPROTECT;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ASHinq_color -- return current color				*/
/*	ASHinq_background_color -- return current background color	*/
/*	ASHinq_combination_rule -- return current combination rule	*/
/*	ASHinq_plane_mask -- return current plane mask			*/
/*	ASHinq_fill -- return current fill pattern			*/
/*	ASHinq_line_style -- return current line style			*/
/*	ASHinq_clip -- return clipping flag				*/
/*	ASHinq_outer_clip -- return outer clipping flag 		*/
/*	ASHinq_font -- return current font				*/
/*	ASHinq_text_color -- return current color			*/
/*	ASHinq_text_background_color -- return current text background	*/
/*	ASHinq_text_combination_rule -- return current text function	*/
/*	ASHinq_text_plane_mask -- return current text plane mask	*/
/*									*/
/************************************************************************/


INQ_VALUE(color,foreground);
INQ_VALUE(background_color,background);
INQ_VALUE(combination_rule,function);
INQ_VALUE(plane_mask,plane_mask);

INQ_OUR_VALUE(fill,fill_style);
INQ_OUR_VALUE(line_style,line_style);
INQ_OUR_VALUE(clip,use_clip);
INQ_OUR_VALUE(outer_clip,out_clip);
INQ_OUR_VALUE(font,font);

INQ_TEXT_VALUE(text_color,foreground);
INQ_TEXT_VALUE(text_background_color,background);
INQ_TEXT_VALUE(text_combination_rule,function);
INQ_TEXT_VALUE(text_plane_mask,plane_mask);





/************************************************************************/
/*									*/
/*	ASHpush_drawinfo -- save old drawinfo and allocate new		*/
/*	ASHsave_drawinfo -- save old drawinfo but maintain it		*/
/*									*/
/************************************************************************/


void
ASHpush_drawinfo(w,d)
   ASH_WINDOW w;
   ASH_DRAWINFO d;
{
   TRACE("ASHpush_drawinfo 0x%x 0x%x",w,d);

   CHECKWIN(w,ASHpush_drawinfo);

   if (d == NULL) ASH_new_drawinfo(w);
   else if (d->window != w) {
      ERROR("DRAWINFO not for given WINDOW");
    }
   else {
      d->next = w->draw;
      w->draw = d;
      if (w->full_win != NULL && d->full_context == NULL) ASH_full_drawinfo(w);
    };
};





void
ASHsave_drawinfo(w)
   ASH_WINDOW w;
{
   ASH_DRAWINFO d;

   TRACE("ASHsave_drawinfo 0x%x",w);

   CHECKWIN(w,ASHpush_drawinfo);

   d = w->draw;
   ASH_new_drawinfo(w);
   ASHcopy_drawinfo(w,d);
};





/************************************************************************/
/*									*/
/*	ASHpop_drawinfo -- restore old drawinfo 			*/
/*									*/
/************************************************************************/


void
ASHpop_drawinfo(w)
   ASH_WINDOW w;
{
   ASH_DRAWINFO d;

   TRACE("ASHpop_drawinfo 0x%x",w);

   CHECKWIN(w,ASHpop_drawinfo);

   if (w->draw->next == NULL) return;

   d = w->draw->next;

   ASH_free_drawinfo(w);

   w->draw = d;
};






/************************************************************************/
/*									*/
/*	ASHnew_drawinfo -- get a new drawinfo structure for window	*/
/*									*/
/************************************************************************/


ASH_DRAWINFO
ASHnew_drawinfo(w)
   ASH_WINDOW w;
{
   ASH_DRAWINFO d;

   CHECKWIN(w,ASHnew_drawinfo);

   L_PROTECT(w);
   ASH_new_drawinfo(w);
   d = w->draw;
   d->nofree = TRUE;
   ASHpop_drawinfo(w);
   L_UNPROTECT(w);

   return d;
};





/************************************************************************/
/*									*/
/*	ASHset_drawinfo -- replace current drawinfo for window		*/
/*									*/
/************************************************************************/


ASH_DRAWINFO
ASHset_drawinfo(w,d)
   ASH_WINDOW w;
   ASH_DRAWINFO d;
{
   ASH_DRAWINFO od;

   CHECKWIN(w,ASHset_drawinfo);

   L_PROTECT(w);
   od = w->draw;
   if (d == NULL) {
      ASH_new_drawinfo(w);
      w->draw->next = NULL;
    }
   else {
      w->draw = d;
    };
   L_UNPROTECT(w);

   return od;
};





/************************************************************************/
/*									*/
/*	ASHcopy_drawinfo -- copy drawinfo from another window		*/
/*									*/
/************************************************************************/


void
ASHcopy_drawinfo(w,from)
   ASH_WINDOW w;
   ASH_WINDOW from;
{
   ASH_DRAWINFO d,dfrom;

   TRACE("ASHcopy_drawinfo 0x%x 0x%x",w,from);

   SET_DRAW(w,d,ASHcopy_drawinfo);
   SET_DRAW(from,dfrom,ASHcopy_drawinfo);

   PROTECT;

   XCopyGC(DISPLAYOF(w),dfrom->context,~0,d->context);
   XCopyGC(DISPLAYOF(w),dfrom->text_context,~0,d->text_context);

   UNPROTECT;

   if (w->full_win != NULL) ASH_full_drawinfo(w);

   d->clear_text = dfrom->clear_text;
   d->fill_style = dfrom->fill_style;
   d->line_style = dfrom->line_style;
   d->font = dfrom->font;
};






/************************************************************************/
/*									*/
/*	ASH_new_drawinfo -- set up default draw information		*/
/*									*/
/************************************************************************/


void
ASH_new_drawinfo(w)
   ASH_WINDOW w;
{
   ASH_DRAWINFO d, od;
   register ASH_WINDOW p;

   d = (ASH_DRAWINFO) malloc(sizeof(__DRAWINFO));
   d->safety = PROP_SAFETY;
   d->window = w;

   od = w->draw;

   p = ASHinq_top_window(w);

   PROTECT;
   d->context = XCreateGC(DISPLAYOF(w),p->x_win,DFLT_GC_MASK,
			     &w->display->data->default_GC);
   d->text_context = XCreateGC(DISPLAYOF(w),p->x_win,DFLT_GC_MASK,
			       &w->display->data->default_GC);

   if (w->full_win != NULL) {
      d->full_context = XCreateGC(DISPLAYOF(w),w->full_win,DFLT_GC_MASK,
				  &w->display->data->default_GC);
      d->full_text_context = XCreateGC(DISPLAYOF(w),w->full_win,DFLT_GC_MASK,
					  &w->display->data->default_GC);
    }
   else {
      d->full_context = NULL;
      d->full_text_context = NULL;
    };
   UNPROTECT;

   d->use_clip = FALSE;
   clear_clips(w,&d->clip,TRUE);

   d->out_clip = FALSE;
   clear_clips(w,&d->outer_clip,TRUE);

   d->clear_text = FALSE;
   d->fill_style = -1;
   d->line_style = -1;
   d->font = -1;
   d->source = w;
   d->next = od;
   d->nofree = FALSE;

   L_PROTECT(w);
   w->draw = d;

   ASHfill(w,ASH_FILL_SOLID);
   ASHline_style(w,ASH_STYLE_SOLID);
   ASHfont(w,ASH_DEFAULT_FONT);

   if (w->our_court) {
      ASHwindow_draw_thru(w,TRUE);
    };
   L_UNPROTECT(w);

   if (w->defaults != NULL) ASHcopy_drawinfo(w,w->defaults);
};





/************************************************************************/
/*									*/
/*	ASH_full_drawinfo -- setup drawinfo for full window		*/
/*									*/
/************************************************************************/


void
ASH_full_drawinfo(w)
   ASH_WINDOW w;
{
   Integer mask,maskt;

   mask = GCFunction|GCPlaneMask|GCForeground|GCBackground|GCLineWidth|
		GCLineStyle|GCCapStyle|GCJoinStyle|GCFillStyle|GCFillRule|
		GCTileStipXOrigin|GCTileStipYOrigin|GCFont|GCSubwindowMode|
		GCGraphicsExposures|GCDashOffset|GCDashList|GCArcMode;
   maskt = mask;
   if (w->draw->context->values.tile != None &&
	  w->draw->context->values.tile != ~0) mask |= GCTile;
   if (w->draw->context->values.stipple != None &&
	  w->draw->context->values.stipple != ~0) mask |= GCStipple;
   if (w->draw->text_context->values.tile != None &&
	  w->draw->text_context->values.tile != ~0) maskt |= GCTile;
   if (w->draw->text_context->values.stipple != None &&
	  w->draw->text_context->values.stipple != ~0) maskt |= GCStipple;

   PROTECT;
   w->draw->full_context = XCreateGC(DISPLAYOF(w),w->full_win,0,NULL);
   XCopyGC(DISPLAYOF(w),w->draw->context,mask,w->draw->full_context);

   w->draw->full_text_context = XCreateGC(DISPLAYOF(w),w->full_win,0,NULL);
   XCopyGC(DISPLAYOF(w),w->draw->text_context,maskt,w->draw->full_text_context);
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_free_drawinfo -- free drawinfo				*/
/*	ASH_free_full_drawinfo -- free full portion of drawinfo 	*/
/*									*/
/************************************************************************/


void
ASH_free_drawinfo(w)
   ASH_WINDOW w;
{
   ASH_free_full_drawinfo(w);

   PROTECT;

   if (!w->draw->nofree) {
      XFreeGC(DISPLAYOF(w),w->draw->context);
      XFreeGC(DISPLAYOF(w),w->draw->text_context);
      free(w->draw);
    };

   w->draw = NULL;

   UNPROTECT;
};





void
ASH_free_full_drawinfo(w)
   ASH_WINDOW w;
{
   PROTECT;

   if (w->draw->full_context != NULL) {
      XFreeGC(DISPLAYOF(w),w->draw->full_context);
      XFreeGC(DISPLAYOF(w),w->draw->full_text_context);
    };

   w->draw->full_context = NULL;
   w->draw->full_text_context = NULL;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_real_font -- map font id to X font				*/
/*	get_font_info -- get font info for font id			*/
/*									*/
/************************************************************************/


Font
ASH_real_font(d,id)
   DISPLAY d;
   ASH_FONT id;
{
   Font f;

   if (id == ASH_ICON_FONT) id = 0;

   f = d->data->font_table[id];
   if (f == NULL) f = d->data->font_table[0];

   return f;
};





static XFontStruct *
get_font_info(d,id)
   DISPLAY d;
   ASH_FONT id;
{
   XFontStruct * xfs;

   xfs = d->data->font_info[id];
   if (xfs == NULL) xfs = d->data->font_info[0];

   return xfs;
};





/************************************************************************/
/*									*/
/*	initial_GC -- setup default GC					*/
/*									*/
/************************************************************************/


static void
initial_GC(d)
   DISPLAY d;
{
   GC gc;
   XGCValues vals;
   Font ft;
   XFontStruct * fs;

   PROTECT;

   gc = XCreateGC(d->display,RootWindow(d->display,d->screen),0,&vals);

   fs = XLoadQueryFont(d->display,default_font_name);
   if (fs == NULL) {
      default_font_name = DFLT_FONT_NAME;
      fs = XLoadQueryFont(d->display,default_font_name);
    };
   if (fs == NULL) {
      default_font_name = ALT_DFLT_FONT;
      fs = XLoadQueryFont(d->display,default_font_name);
    };
   if (fs == NULL) {
      FATAL("Can't open default font");
    };

   ft = fs->fid;
   XSetFont(d->display,gc,ft);
   XSetForeground(d->display,gc,d->foreg);
   XSetBackground(d->display,gc,d->backg);

   UNPROTECT;

   d->data->default_GC = gc->values;
};





/********************************************************************************/
/*										*/
/*	initial_color_table -- setup color table				*/
/*										*/
/********************************************************************************/


static void
initial_color_table(d)
   DISPLAY d;
{
   Colormap cmp;
   Integer i,k;

   if (d->top_colormap != NULL) cmp = d->top_colormap;
   else cmp = DefaultColormap(d->display,d->screen);

   if (initial_colors == NULL) {
      max_initial_colors = 4;
      initial_colors = (XColor *) calloc(sizeof(XColor),max_initial_colors);
      color_names = (String *) calloc(sizeof(String),max_initial_colors);
      num_initial_colors = k = 2;
      for (i = 0; i < k; ++i) {
	 initial_colors[i].pixel = i;
	 color_names[i] = NULL;
       };
      i = BlackPixel(d->display,d->screen);
      if (i < k) color_names[i] = "black";
      i = WhitePixel(d->display,d->screen);
      if (i < k) color_names[i] = "white";
    };

   d->colormap = cmp;
};




/************************************************************************/
/*									*/
/*	initial_fill_table -- setup default fill styles 		*/
/*	setup_fill_table -- setup fill table for new display		*/
/*									*/
/************************************************************************/


static void
initial_fill_table()
{
   register Integer i;

   for (i = 0; i < FILL_TABLE_SIZE; ++i) {
      fill_table[i].style = FillSolid;
    };
};





static void
setup_fill_table(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   DISPLAY d1;
   Integer i;

   for (i = 0; i < FILL_TABLE_SIZE; ++i) d->data->fill_tile[i] = NULL;

   if (ffg) {
      for (i = 0; i < 16; ++i) {
	 ASH_for_all_displays(pat_fill_entry,i,&ASH__default_fillstyles[16*i],
				 FillOpaqueStippled);
	 ASH_for_all_displays(pat_fill_entry,i+ASH_FILL_STIPPLE_OFFSET,
				 &ASH__default_fillstyles[16*i],
				 FillStippled);
       };
    }
   else {
      d1 = ASH_inq_top_display();
      for (i = 0; i < FILL_TABLE_SIZE; ++i) {
	 if (d1->data->fill_tile[i] != NULL) {
	    d->data->fill_tile[i] =
	       ASH_copy_pixmap(d1,d1->data->fill_tile[i],d);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	initial_line_table -- setup default line styles 		*/
/*									*/
/************************************************************************/


static void
initial_line_table()
{
   register Integer i;

   for (i = 0; i < LINE_TABLE_SIZE; ++i) {
      line_table[i].line_width = 0;
      line_table[i].line_style = LineSolid;
      line_table[i].cap_style = CapButt;
      line_table[i].join_style = JoinMiter;
      line_table[i].dash_offset = 0;
      line_table[i].dash_len = 2;
      line_table[i].dash_list[0] = 4;
      line_table[i].dash_list[1] = 4;
    };

   ASHline_table(ASH_STYLE_DASHED,
		    0,LineOnOffDash,CapButt,JoinMiter,0,2,8,8);
   ASHline_table(ASH_STYLE_DOTTED,
		    0,LineOnOffDash,CapButt,JoinMiter,0,2,2,2);
   ASHline_table(ASH_STYLE_THICK,
		    2,LineSolid,CapButt,JoinMiter,0,2,4,4);
   ASHline_table(ASH_STYLE_THICK_DASHED,
		    2,LineOnOffDash,CapButt,JoinMiter,0,2,8,8);
   ASHline_table(ASH_STYLE_THICK_DOTTED,
		    2,LineOnOffDash,CapButt,JoinMiter,0,2,2,2);
   ASHline_table(ASH_STYLE_THICKER,
		    4,LineSolid,CapButt,JoinMiter,0,2,4,4);
   ASHline_table(ASH_STYLE_THICKER_DASHED,
		    4,LineOnOffDash,CapButt,JoinMiter,0,2,8,8);
   ASHline_table(ASH_STYLE_THICKER_DOTTED,
		    4,LineOnOffDash,CapButt,JoinMiter,0,2,2,2);
};





/************************************************************************/
/*									*/
/*	clear_clips -- clear cliping rectangles for window		*/
/*	add_clip_rect -- add clip rectangle to window			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
clear_clips(w,cp,new)
   ASH_WINDOW w;
   CLIP_INFO cp;
   Boolean new;
{
   CLIP_RECT cr,ncr;

   if (!new && cp->type == CLIP_RECTS) {
      for (cr = cp->rect; cr != NULL; cr = ncr) {
	 ncr = cr->next;
	 free(cr);
       };
    };

   cp->type = CLIP_NONE;
   cp->rect = NULL;
   cp->numrect = 0;
   cp->map = NULL;
};





static void
add_clip_rect(w,cp,lx,by,rx,ty)
   ASH_WINDOW w;
   CLIP_INFO cp;
   Integer lx,by,rx,ty;
{
   CLIP_RECT cr;

   if (cp->type == CLIP_MAP) clear_clips(w,cp,FALSE);

   if (cp->type == CLIP_NONE) {
      cp->type = CLIP_RECTS;
      cp->numrect = 0;
    };

   cr = PALLOC(CLIP_RECT_B);

   L_PROTECT(w);
   cr->next = cp->rect;
   cr->rect.lx = lx;
   cr->rect.by = by;
   cr->rect.rx = rx;
   cr->rect.ty = ty;
   cp->rect = cr;
   cp->numrect++;
   L_UNPROTECT(w);
};





/* end of ashproperty.c */
