/************************************************************************/
/*									*/
/*		ashmain.c						*/
/*									*/
/*	Main entries for using ASH in general				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#define ASH_MAIN


#include "ash_local.h"
#include <sys/types.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_GEOMETRY	"512x800+0+40"
#define MAX_DISPLAYS		4
#define DEFAULT_BORDER_WIDTH	3




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


	Integer ASH__debugfg = 0;	/* debugging level		*/
	Boolean ASH__initedflag = FALSE;/* have we been inited		*/
	FILE *	ASH__debugf;		/* debugging output		*/

	PROT_DECL;

static	DISPLAY 	displays = NULL;

static	ASH_WINDOW	top_window = NULL;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


#ifdef LIB
static			ash_cleanup_rtn();
#endif

static	DISPLAY 	startup_display();
static	void		setup_X();
static	Window		create_X_window();
static	int		error_handler();
static	int		io_error_handler();





/************************************************************************/
/*									*/
/*	ASHinit -- initialize ash					*/
/*									*/
/************************************************************************/



ASH_WINDOW
ASHinit(mode)
   ASH_MODE mode;
{
   Window window;
   DISPLAY display;
   String s;

   TRACE("ASHinit %d",mode);

   PROT_INIT;

   PROTECT;
   if (!ASH__initedflag) {
      displays = NULL;

      ASH__debugf = NULL;
      s = getenv("ASH_DEBUG");
      if (s != NULL) {
	 ASH__debugf = fopen(s,"w");
	 if (ASH__debugf == NULL) ASH__debugf = stdout;
	 setlinebuf(ASH__debugf);
       };

      ASH__initedflag = TRUE;

      ASH_rm_init();
      ASH_window_init();
      ASH_lock_init();
      ASH_screen_init();
      ASH_windraw_init();
      ASH_draw_init();
      ASH_property_init();
      ASH_cursor_init();
      ASH_input_init();
      ASH_util_init();
      ASH_anim_init();
      ASH_icon_init();
      ASH_font_init();

      BPIO_init();

      setup_X(mode,&display,&window);

      if (mode != ASH_MODE_NONE) {
	 top_window = ASH_define_window(display,window);
	 ASHvisible(top_window,TRUE);
       }
      else {
	 top_window = NULL;
       };

#ifdef LIB
      BROWNregister_cleanup(ash_cleanup_rtn);
#endif
    };
   UNPROTECT;

   return top_window;
};



#ifdef LIB

static
ash_cleanup_rtn()
{
   ASH_cursorfini();

   ASH__debugfg = 0;
   ASH__initedflag = FALSE;
};

#endif





/************************************************************************/
/*									*/
/*	ASHinq_configuration_depth -- return screen depth		*/
/*	ASHinq_configuration_width -- return screen width		*/
/*	ASHinq_configuration_height -- return screen height		*/
/*	ASHinq_configuration_colors -- return number of colors		*/
/*	ASHinq_configuration_black -- return black			*/
/*	ASHinq_configuration_white -- return white			*/
/*	ASHinq_foreground-- return foreground color for display 	*/
/*	ASHinq_background-- return background color for display 	*/
/*	ASHinq_read_mask -- return read mask for all displays		*/
/*									*/
/************************************************************************/


#ifndef __STDC__

#define DISPLAY_INQ(name,mname,typ)				\
typ								\
ASHinq_/**/name(w)						\
   ASH_WINDOW w;						\
{								\
   register DISPLAY d;						\
   ENTER("ASHinq_ name 0x%x",w);                                \
   if (w == NULL) d = ASH_inq_top_display();			\
   else d = w->display; 					\
   return (typ) mname(d->display,d->screen);			\
}

#else

#define DISPLAY_INQ(name,mname,typ)				\
typ								\
ASHinq_ ## name(w)						\
   ASH_WINDOW w;						\
{								\
   register DISPLAY d;						\
   ENTER("ASHinq_ name 0x%x",w);                                \
   if (w == NULL) d = ASH_inq_top_display();			\
   else d = w->display; 					\
   return (typ) mname(d->display,d->screen);			\
}

#endif




DISPLAY_INQ(configuration_depth,XDefaultDepth,int);
DISPLAY_INQ(configuration_width,XDisplayWidth,int);
DISPLAY_INQ(configuration_height,XDisplayHeight,int);
DISPLAY_INQ(configuration_colors,XDisplayCells,int);
DISPLAY_INQ(configuration_black,XBlackPixel,ASH_COLOR);
DISPLAY_INQ(configuration_white,XWhitePixel,ASH_COLOR);





Display *
ASHinq_display(w)
   ASH_WINDOW w;
{
   DISPLAY d;

   ENTER("ASHinq_display 0x%x",w);

   if (w == NULL) d = ASH_inq_top_display();
   else d = w->display;

   return d->display;
};





char *
ASHinq_display_name(w)
   ASH_WINDOW w;
{
   DISPLAY d;

   ENTER("ASHinq_display_name 0x%x",w);

   if (w == NULL) d = ASH_inq_top_display();
   else d = w->display;

   return d->name;
};





Integer
ASHinq_screen(w)
   ASH_WINDOW w;
{
   DISPLAY d;

   ENTER("ASHinq_screen 0x%x",w);

   if (w == NULL) d = ASH_inq_top_display();
   else d = w->display;

   return d->screen;
};





ASH_COLOR
ASHinq_foreground(w)
   ASH_WINDOW w;
{
   DISPLAY d;

   ENTER("ASHinq_foreground 0x%x",w);

   if (w == NULL) d = ASH_inq_top_display();
   else d = w->display;

   return d->foreg;
};





ASH_COLOR
ASHinq_background(w)
   ASH_WINDOW w;
{
   DISPLAY d;

   ENTER("ASHinq_background 0x%x",w);

   if (w == NULL) d = ASH_inq_top_display();
   else d = w->display;

   return d->backg;
};





int
ASHinq_read_mask(r)
   fd_set *r;
{
   register Integer ct;
   register DISPLAY d;

   ENTER("ASHinq_read_mask 0x%x",r);

   FD_ZERO(r);

   ct = 0;
   for (d = displays; d != NULL; d = d->next) {
      ++ct;
      FD_SET(ConnectionNumber(d->display),r);
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	ASHinq_top -- return top window 				*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHinq_top()
{
   ASHinit(ASH_MODE_DEFAULT);

   return top_window;
};





/************************************************************************/
/*									*/
/*	ASHroot_window -- create a new root window			*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHroot_window(display,args)
   String display;
   String args;
{
   register DISPLAY d;
   register Window w;
   register ASH_WINDOW aw;

   ASHinit(ASH_MODE_NONE);

   d = startup_display(display,NULL,-1);

   ASHset_application_string(args);

   w = create_X_window(d,NULL,TRUE);

   if (w != NULL) {
      aw = ASH_define_window(d,w);
      if (top_window == NULL) top_window = aw;
      ASHvisible(aw,TRUE);
    }
   else aw = NULL;

   ASHreset_application_string();

   return aw;
};





/************************************************************************/
/*									*/
/*	ASHuse_X_window -- transform X window to ASH window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHuse_X_window(xd,xw)
   Display * xd;
   Window xw;
{
   ASH_WINDOW aw;
   Window root;
   Integer x,y,wd,ht,bw,dp;
   DISPLAY d;
   register Integer i;

   ASHinit(ASH_MODE_NONE);

   if (xd == NULL) xd = displays->display;

   PROTECT;
   XGetGeometry(xd,xw,&root,&x,&y,&wd,&ht,&bw,&dp);
   UNPROTECT;

   for (i = 0; i < ScreenCount(xd); ++i) {
      if (RootWindow(xd,i) == root) break;
    };
   if (i >= ScreenCount(xd)) i = 0;
   d = startup_display(NULL,xd,i);

   aw = ASH_define_window(d,xw);
   if (top_window == NULL) top_window = aw;
   ASHvisible(aw,TRUE);

   return aw;
};





/************************************************************************/
/*									*/
/*	ASHroot_X_window -- generate X window for root of given window	*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHroot_X_window(w)
   ASH_WINDOW w;
{
   ASH_WINDOW rw;
   DISPLAY ds;

   if (w == NULL) ds = ASH_inq_top_display();
   else ds = w->display;

   rw = ASHuse_X_window(ds->display,XRootWindowOfScreen(ds->screen_id));

   return rw;
};





/************************************************************************/
/*									*/
/*	ASHwindow_info -- print information about window		*/
/*									*/
/************************************************************************/


void
ASHwindow_info(w)
   ASH_WINDOW w;
{
   XWindowAttributes attrs;

   PROTECT;
   XGetWindowAttributes(DISPLAYOF(w),w->x_win,&attrs);
   UNPROTECT;

   printf("Window 0x%x (x-win: 0x%x) :\n",w,w->x_win);
   printf("\troot = 0x%x\n",attrs.root);
   printf("\torigin = (%d,%d) width = %d, height = %d\n",
	     attrs.x,attrs.y,attrs.width,attrs.height);
   printf("\tborder width = %d; planes = %d\n",attrs.border_width,attrs.depth);
   printf("\tclass = %d; gravity = (%d,%d)\n",
	     attrs.class,attrs.bit_gravity,attrs.win_gravity);
   printf("\tmap state = %d\n",attrs.map_state);
   printf("\tevents = 0x%x 0x%x\n",attrs.your_event_mask,attrs.all_event_masks);
   printf("\tbacking pixel = 0x%x\n",attrs.backing_pixel);
   printf("\tbacking store = %d, backing_planes = %d\n",attrs.backing_store,attrs.backing_pixel);

   printf("\tBlack = 0x%x   white = 0x%x\n",ASHinq_configuration_black(w),
	     ASHinq_configuration_white(w));
};





/************************************************************************/
/*									*/
/*	ASHtrace -- handle tracing setup				*/
/*	ASHdebug -- set synchronization and other debugging options	*/
/*									*/
/************************************************************************/


void
ASHtrace(lvl)
   Integer lvl;
{
   ASH__debugfg = lvl;
};




void
ASHdebug(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   Display * d;

   d = ASHinq_display(w);

   PROTECT;
   if (fg) {
      XSynchronize(d,1);
    }
   else {
      XSynchronize(d,0);
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHnext_event -- return next BPIO event for any display 	*/
/*									*/
/************************************************************************/


int
ASHnext_event(e)
   BPIO_EVENT e;
{
   DISPLAY d;

   for (d = displays; d != NULL; d = d->next) {
      if (BPIOnext_event(d->display,e)) break;
    };

   return (d != NULL);
};





/************************************************************************/
/*									*/
/*	ASHstore_buffer_text -- store text into X buffer		*/
/*	ASHload_buffer_text -- load text from X buffer			*/
/*									*/
/************************************************************************/


void
ASHstore_buffer_text(w,txt)
   ASH_WINDOW w;
   String txt;
{
   Integer ln;

   ln = strlen(txt);

   PROTECT;
   XStoreBytes(DISPLAYOF(w),txt,ln);
   UNPROTECT;
};





char *
ASHload_buffer_text(w)
   ASH_WINDOW w;
{
   Integer ct;
   String s,xs;

   PROTECT;
   xs = XFetchBytes(DISPLAYOF(w),&ct);
   if (ct > 0) {
      s = SALLOC(xs);
      XFree(xs);
    }
   else s = NULL;
   UNPROTECT;

   return s;
};





/************************************************************************/
/*									*/
/*	ASH_close_display -- finish up with a top level window		*/
/*									*/
/************************************************************************/


void
ASH_close_display(topd)
   DISPLAY topd;
{
   DISPLAY d,ld,d1,od;
   Integer ct;

   if (--topd->root_cnt > 0) return;

   PROTECT;

   if (displays == topd && topd->next == NULL) {
      XCloseDisplay(topd->display);
    }
   else {
      ld = NULL;
      d1 = NULL;
      ct = 0;
      for (d = displays; d != NULL; d = d->next) {
	 if (d == topd) ld = d1;
	 else if (d->display == topd->display) {
	    ++ct;
	    od = d;
	  };
	 d1 = d;
       };
      if (displays == topd) displays = topd->next;
      else ld->next = topd->next;
      topd->next = NULL;
      if (ct == 0) {
	 XCloseDisplay(topd->display);
	 topd->display = NULL;
       }
      else if (!topd->dup || ct == 1) od->dup = FALSE;
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_find_display -- return ASH display for X display		*/
/*									*/
/************************************************************************/


DISPLAY
ASH_find_display(xd)
   Display * xd;
{
   DISPLAY d;

   for (d = displays; d != NULL; d = d->next) {
      if (d->display == xd) break;
    };

   return d;
};





/********************************************************************************/
/*										*/
/*	ASH_inq_top_display -- return top display				*/
/*										*/
/********************************************************************************/


DISPLAY
ASH_inq_top_display()
{
   return displays;
};





/************************************************************************/
/*									*/
/*	ASH_for_all_displays -- call a routine for each display data	*/
/*									*/
/************************************************************************/


/*VARARGS*/

void
ASH_for_all_displays(rtn,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10)
   void_Function_Ptr rtn;
   Universal a1,a2,a3,a4,a5,a6,a7,a8,a9,a10;
{
   DISPLAY d;

   for (d = displays; d != NULL; d = d->next) {
      (*rtn)(d,a1,a2,a3,a4,a5,a6,a7,a8,a9,a10);
    };
};





/************************************************************************/
/*									*/
/*	ASH_error -- handle ash error message			       */
/*									*/
/************************************************************************/


void
ASH_error(msg)
   String msg;
{
   fprintf(stderr,"ASH: ERROR: %s\n",msg);
   fflush(stderr);

   if (ASH__debugfg&ASH_TRACE_ERROR) abort();
};






/************************************************************************/
/*									*/
/*	ASH_trace -- handle tracing output			       */
/*									*/
/************************************************************************/


/* VARARGS1 */

void
ASH_trace(f,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9)
   String f;
   Integer a0,a1,a2,a3,a4,a5,a6,a7,a8,a9;
{
   Character buf[1024];

   sprintf(buf,f,a0,a1,a2,a3,a4,a5,a6,a7,a8,a9);

   printf("ASH: %s\n",buf);
   fflush(stdout);
};






/************************************************************************/
/*									*/
/*	startup_display -- get display handle for given string		*/
/*									*/
/************************************************************************/


static DISPLAY
startup_display(name,d,scr)
   String name;
   Display * d;
   Integer scr;
{
   DISPLAY ds,lds,od;
   Character buf[64];
   String s;
   Integer screen;
   Boolean first;
   XColor cola;
   Integer v,bufln;
   XVisualInfo * vin,vinit;
   Integer ct,i,j;
   XStandardColormap stdcmp;

   if (name == NULL && d != NULL) name = DisplayString(d);
   else if (name == NULL) {
      name = ASHinq_resource("display");
      if (name == NULL) name = "";
    };

   strcpy(buf,name);
   s = rindex(buf,'.');
   if (s == NULL) {
      screen = 0;
    }
   else {
      *s++ = 0;
      screen = atol(s);
    };

   if (scr >= 0) screen = scr;

   PROTECT;

   first = (displays == NULL);

   od = NULL;
   lds = NULL;
   bufln = strlen(buf);
   for (ds = displays; ds != NULL; ds = ds->next) {
      if (od == NULL && d != NULL && ds->display == d) od = ds;
      if (*buf == 0 && ds->dflt) break;
      else if (ds->display == d ||
		  (ds->name != NULL && strncmp(ds->name,buf,bufln) == 0)) {
	 if (ds->screen == screen) break;
	 else {
	    if (d == NULL) d = ds->display;
	    if (od == NULL) od = ds;
	  };
       };
      lds = ds;
    };

   if (ds == NULL) {
      ds = (DISPLAY) malloc(sizeof(__DISPLAY));
      ds->display =  (d == NULL ? XOpenDisplay(name) : d);
      ds->root_cnt = 0;

      if (ds->display == NULL) {
	 if (name != NULL && name[0] != 0) ds->name = name;
	 else ds->name = (String) getenv("DISPLAY");
	 if (ds->name == NULL)
	    fprintf(stderr,"No display to open\n");
	 else
	    fprintf(stderr,"Couldn't open display %s\n",ds->name);
	 exit(1);
       };

      if (USE_THREADS) THREADopen(ConnectionNumber(ds->display));
      ds->name = SALLOC(DisplayString(ds->display));

      s = rindex(ds->name,'.');
      if (s != NULL) {
	 ++s;
	 screen = atol(s);
       };

      ds->next = NULL;
      ds->screen = screen;
      ds->dflt = (*buf == 0);
      if (lds == NULL) displays = ds;
      else lds->next = ds;
      XSetErrorHandler(error_handler);
      XSetIOErrorHandler(io_error_handler);
      ds->screen_id = XScreenOfDisplay(ds->display,ds->screen);
      ds->backing_store = (XDoesBackingStore(ds->screen_id) == Always);
#ifdef OUR_SAVED_BITMAP
      ds->backing_store = FALSE;
#endif
      if (getenv("NO_BACKING_STORE") != NULL) ds->backing_store = FALSE;

      ds->save_under = XDoesSaveUnders(ds->screen_id);

      ds->foreg = XBlackPixel(ds->display,ds->screen);
      ds->backg = XWhitePixel(ds->display,ds->screen);

      s = ASHinq_resource("foreground");
      if (s != NULL) {
	 if (XParseColor(ds->display,XDefaultColormapOfScreen(ds->screen_id),s,&cola) &&
		XAllocColor(ds->display,XDefaultColormapOfScreen(ds->screen_id),&cola))
	    ds->foreg = cola.pixel;
       };
      s = ASHinq_resource("background");
      if (s != NULL) {
	 if (XParseColor(ds->display,XDefaultColormapOfScreen(ds->screen_id),s,&cola) &&
		XAllocColor(ds->display,XDefaultColormapOfScreen(ds->screen_id),&cola))
	    ds->backg = cola.pixel;
       };
      s = ASHinq_resource("reverseVideo");
      if (s != NULL && STREQL(s,"on")) {
	 v = ds->foreg;
	 ds->foreg = ds->backg;
	 ds->backg = v;
       };

      s = ASHinq_resource("synchronous");
      if (s != NULL && STREQL(s,"on")) XSynchronize(ds->display,1);

      vinit.screen = screen;
      vin = XGetVisualInfo(ds->display,VisualScreenMask,&vinit,&ct);
      ds->default_visual = XDefaultVisualOfScreen(ds->screen_id);
      ds->color_visual = NULL;
      if (vin != NULL) {
	 if (ds->default_visual == NULL) ds->default_visual = vin[0].visual;
	 for (i = 0; i < ct; ++i) {
	    if (vin[i].depth > 1 && vin[i].class == PseudoColor) {
	       ds->color_visual = vin[i].visual;
	       break;
	     };
	  };
       };
      XFree(vin);
      if (ds->color_visual == NULL) ds->color_visual = ds->default_visual;
      if (ds->color_visual == ds->default_visual) ds->top_colormap = NULL;
      else {
	 s = ASHinq_resource("colormap");
	 if (s == NULL) s = getenv("COLORMAP");
	 if (s == NULL || STREQL(s,"off")) ds->top_colormap = NULL;
	 else if (XGetStandardColormap(ds->display,
					  XRootWindowOfScreen(ds->screen_id),
					  &stdcmp,
					  XA_RGB_DEFAULT_MAP) != 0) {
	    if (XDefaultColormapOfScreen(ds->screen_id) == stdcmp.colormap)
	       ds->top_colormap = NULL;
	    else ds->top_colormap = stdcmp.colormap;
	  }
	 else {
	    XColor cols[256];
	    long pixels[256];

	    ds->top_colormap = XCreateColormap(ds->display,
						  XRootWindowOfScreen(ds->screen_id),
						  ds->color_visual,
						  AllocNone);
	    XAllocColorCells(ds->display,ds->top_colormap,0,0,0,pixels,256);
	    for (i = 0; i < 256; ++i) cols[i].pixel = i;
	    XQueryColors(ds->display,XDefaultColormapOfScreen(ds->screen_id),cols,256);
	    XStoreColors(ds->display,ds->top_colormap,cols,256);
	    j = 0;
	    for (i = 0; i < 252; ++i) { 	/* should cooperate with others    */
	       pixels[j++] = i;
	     };
	    XFreeColors(ds->display,ds->top_colormap,pixels,j,0);
	  };
       };

      if (od == NULL) {
	 ds->data = (DISPLAY_DATA) malloc(sizeof(__DISPLAY_DATA));
	 ASH_windraw_disp_init(ds,first);
	 ASH_property_disp_init(ds,first);
	 ASH_window_disp_init(ds,first);
	 ASH_cursor_disp_init(ds,first);
	 ASH_rm_disp_init(ds,first);
       }
      else {
	 ds->data = od->data;
	 ds->dup = TRUE;
       };
    };

   UNPROTECT;

   return ds;
};





/************************************************************************/
/*									*/
/*	setup_X -- setup X client					*/
/*									*/
/************************************************************************/


static void
setup_X(mode,d,xw)
   ASH_MODE mode;
   DISPLAY * d;
   Window * xw;
{
   DISPLAY topdisp;
   Window window;
   Integer x0,y0,xsize,ysize;
   String geom,s;
   Character buf[64];
   Window rwin,cwin;
   Integer bw,dp;
   Boolean brdr;

   topdisp = startup_display(NULL,NULL,-1);

   if (topdisp == NULL) {
      FATAL("Couldn't open display");
    };

   if (mode == ASH_MODE_DEFAULT) mode = ASH_MODE_INQUIRE;

   geom = NULL;
   brdr = TRUE;

   if (mode == ASH_MODE_DISPLAY) {
      xsize = XDisplayWidth(topdisp->display,topdisp->screen);
      ysize = XDisplayHeight(topdisp->display,topdisp->screen);
      x0 = 0;
      y0 = 0;
      sprintf(buf,"=%dx%d+%d+%d",xsize,ysize,x0,y0);
      geom = buf;
      brdr = FALSE;
    }
   else if (mode == ASH_MODE_WINDOW) {
      s = (String) getenv("WINDOWID");
      if (s != NULL) {
	 window = (Universal) atol(s);
	 PROTECT;
	 XGetGeometry(topdisp->display,window,&rwin,&x0,&y0,&xsize,&ysize,&bw,&dp);
	 XTranslateCoordinates(topdisp->display,window,rwin,0,0,&x0,&y0,&cwin);
	 s = ASHinq_resource("TopLevelShell.borderWidth");
	 if (s != NULL) bw = atol(s);
	 else bw = DEFAULT_BORDER_WIDTH;
	 x0 -= bw;
	 y0 -= bw;
	 UNPROTECT;
	 sprintf(buf,"=%dx%d+%d+%d",xsize,ysize,x0,y0);
	 geom = buf;
       }
    };

   if (mode != ASH_MODE_NONE) window = create_X_window(topdisp,geom,brdr);
   else window = NULL;

   *d = topdisp;
   *xw = window;
}




/************************************************************************/
/*									*/
/*	create_X_window -- create an X window				*/
/*									*/
/************************************************************************/


static Window
create_X_window(d,geo,brdr)
   DISPLAY d;
   String geo;
   Boolean brdr;
{
   Window window;
   Integer x,y,wd,ht;
   XWindowAttributes attrs;
   XSetWindowAttributes sattrs;
   XEvent xev;
   Integer bwid,qfg,bcol,afg;
   String s;
   String dgeo;
   XColor cola;
   Integer mask,depth;
   Visual * vis;

   PROTECT;

   dgeo = ASHinq_resource("TopLevelShell.geometry");
   if (dgeo == NULL) dgeo = DEFAULT_GEOMETRY;

   s = ASHinq_resource("TopLevelShell.borderWidth");
   if (s != NULL) bwid = atol(s);
   else bwid = DEFAULT_BORDER_WIDTH;
   if (!brdr) bwid = 0;
   s = ASHinq_resource("TopLevelShell.borderColor");
   bcol = d->foreg;
   if (s != NULL) {
      if (XParseColor(d->display,XDefaultColormapOfScreen(d->screen_id),s,&cola) &&
	     XAllocColor(d->display,XDefaultColormapOfScreen(d->screen_id),&cola))
	 bcol = cola.pixel;
    };

   wd = 16;
   ht = 16;
   x = 0;
   y = 0;

   afg = XGeometry(d->display,d->screen,geo,DEFAULT_GEOMETRY,bwid,1,1,16,16,&x,&y,&wd,&ht);
   afg = XGeometry(d->display,d->screen,geo,dgeo,bwid,1,1,16,16,&x,&y,&wd,&ht);

   vis = d->default_visual;
   sattrs.background_pixel = d->backg;
   sattrs.border_pixel = bcol;
   mask = CWBackPixel|CWBorderPixel;
   if (d->top_colormap != NULL) {
      vis = d->color_visual;
      sattrs.colormap = d->top_colormap;
      mask |= CWColormap;
    };
   sattrs.event_mask = ROOT_EVENT_MASK;
   mask |= CWEventMask;

   depth = 0;
   window = XCreateWindow(d->display,XRootWindow(d->display,d->screen),
			     x,y,wd,ht,
			     bwid,depth,InputOutput,vis,
			     mask,&sattrs);

   s = ASHinq_resource("TopLevelShell.query");
   if (s == NULL) qfg = (geo == NULL || geo[0] == 0);
   else if (STREQL(s,"on")) qfg = TRUE;
   else qfg = FALSE;

   ASH_set_properties(d,window,!qfg,afg);
   ASH_set_wm_properties(d,window);

   XMapWindow(d->display,window);

   for ( ; ; ) {
      while (XCheckWindowEvent(d->display,window,~0,&xev));
      XGetWindowAttributes(d->display,window,&attrs);
      if (attrs.map_state == IsViewable) break;
      UNPROTECT;
      BPIOwait(250000);
      PROTECT;
    };

   UNPROTECT;

   return window;
}



/************************************************************************/
/*									*/
/*	error_handler -- handle X errors				*/
/*	io_error_handler -- handle X I/O errors 			*/
/*									*/
/************************************************************************/


static int
error_handler(d,err)
   Display * d;
   XErrorEvent * err;
{
   Character buf[256];

   PROTECT;
   XGetErrorText(d,err->error_code,buf,256);
   UNPROTECT;

   ERROR(buf);
};





static int
io_error_handler(d)
   Display * d;
{
   ERROR("X I/O error");

   exit(1);
};





/* end of ashmain.c */

