/************************************************************************/
/*									*/
/*		ashlock.c						*/
/*									*/
/*	Locking routines for use with ASH				*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define WAIT_CURSOR		ASH_CURSOR_WATCH




/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct _ASH_LOCK_RQST * ASH_LOCK_RQST;



typedef struct _ASH_LOCK {
   ASH_WINDOW window;
   ASH_LOCK_RQST requests;
   ASH_CURSOR cursor;
   Boolean longlock;
} ASH_LOCK_INFO;




typedef struct _ASH_LOCK_RQST {
   ASH_LOCK_RQST next;
   ASH_WINDOW window;
   Boolean remove;
   Boolean refresh;
} ASH_LOCK_RQST_INFO;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static ASH_CURSOR	wait_cursor;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	ASH_WINDOW		find_lock_win();





/************************************************************************/
/*									*/
/*	ASH_lock_init -- module initialization				*/
/*									*/
/************************************************************************/


void
ASH_lock_init()
{
   wait_cursor = NULL;
};





/********************************************************************************/
/*										*/
/*	ASHlock -- lock window for this thread and from outside events		*/
/*	ASHunlock -- release window lock					*/
/*										*/
/********************************************************************************/


int
ASHlock(w)
   ASH_WINDOW w;
{
   TRACE("ASHlock 0x%x",w);

   if (w == NULL || w->safety != SAFETY) return FALSE;

   if (USE_THREADS && w->sema == NULL) {
      PROTECT;
      if (w->sema == NULL) {
	 w->sema = (BWE_SEMA *) calloc(1,sizeof(BWE_SEMA));
	 BWEthread_semaphore(w->sema);
       };
      UNPROTECT;
    };

   if (w->sema != NULL) {
      BWEthread_protect(w->sema);
      w->thread = THREADinq_current();
    };

   ASH_desensitize(w);
   ++w->threadct;

   if (w == NULL || w->safety != SAFETY) return FALSE;

   return TRUE;
};





void
ASHunlock(w)
   ASH_WINDOW w;
{
   TRACE("ASHunlock 0x%x",w);

   if (w == NULL || w->safety != SAFETY) return;

   --w->threadct;

   if (w->sema != NULL) {
      w->thread = NULL;
      BWEthread_unprotect(w->sema);
    };

   if (w->threadct == 0 && !ASH_inq_batched()) {
      ASH_flush_output(w);
    };
};





/************************************************************************/
/*									*/
/*	ASHinput_lock -- allow user or application to lock windows	*/
/*	ASHinput_unlock -- release window lock				*/
/*									*/
/*	This code works whether threads are running or not.  These locks*/
/*	do not nest however so they have to be used carefully.		*/
/*									*/
/************************************************************************/


ASH_LOCK_STATUS
ASHinput_lock(w,lng)
   ASH_WINDOW w;
   Boolean lng;
{
   ASH_WINDOW p;
   Integer sts;

   TRACE("ASHinput_lock 0x%x %d",w,lng);

   if (w == NULL || w->safety != SAFETY || w->removed) return ASH_LOCK_FAIL;

   if (!ASHlock(w)) return ASH_LOCK_FAIL;

   p = find_lock_win(w);

   L_PROTECT(p);
   if (p->lock == NULL) {
      p->lock = PALLOC(ASH_LOCK_INFO);
      p->lock->window = w;
      p->lock->requests = NULL;
      p->lock->longlock = lng;
      if (wait_cursor == NULL) wait_cursor = ASHcursor_standard(WAIT_CURSOR);
      if (lng) p->lock->cursor = ASHwindow_cursor(p,wait_cursor);
      else p->lock->cursor = NULL;
      sts = ASH_LOCK_GRANTED;
    }
   else if (p->lock->longlock) sts = ASH_LOCK_WAIT_LONG;
   else sts = ASH_LOCK_WAIT;
   L_UNPROTECT(p);

   if (sts != ASH_LOCK_GRANTED) ASHunlock(w);

   if (w == NULL || w->safety != SAFETY || w->removed) return ASH_LOCK_FAIL;

   return sts;
};





void
ASHinput_unlock(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p;
   ASH_LOCK_RQST rq,nrq;

   TRACE("ASHinput_unlock 0x%x",w);

   if (w == NULL || w->safety != SAFETY || w->removed) return;

   p = find_lock_win(w);

   if (p->lock == NULL) {
      ERROR("Releasing non-existent lock");
      return;
    };
   if (p->lock->window != w) {
      ERROR("Releasing lock for wrong window");
      return;
    };

   L_PROTECT(p);

   rq = p->lock->requests;
   if (p->lock->longlock) {
      ASHwindow_cursor(p,p->lock->cursor);
    };
   free(p->lock);
   p->lock = NULL;

   for ( ; rq != NULL; rq = nrq) {
      nrq = rq->next;
      if (ASHinq_valid_window(rq->window)) {
	 if (rq->remove) ASHremove(rq->window);
	 else if (rq->refresh) ASHfix_damage(rq->window);
       };
      free(rq);
    };

   L_UNPROTECT(p);

   ASH_flush_output(p);

   ASHunlock(w);
};





/************************************************************************/
/*									*/
/*	ASHinput_lock_makelong -- convert lock to a long one		*/
/*									*/
/************************************************************************/


void
ASHinput_lock_makelong(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p;

   TRACE("ASHinput_lock_makelong 0x%x",w);

   if (w == NULL || w->safety != SAFETY || w->removed) return;

   p = find_lock_win(w);

   PROTECT;
   if (p->lock != NULL && !p->lock->longlock) {
      if (wait_cursor == NULL) wait_cursor = ASHcursor_standard(WAIT_CURSOR);
      p->lock->cursor = ASHwindow_cursor(p,wait_cursor);
      p->lock->longlock = TRUE;
      XFlush(DISPLAYOF(p));
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHinput_lock_by -- associate window for locking		*/
/*									*/
/************************************************************************/


void
ASHinput_lock_by(w,by)
   ASH_WINDOW w;
   ASH_WINDOW by;
{
   TRACE("ASHinput_lock_by 0x%x 0x%x",w,by);

   if (w != NULL) w->lock_by = by;
};






/************************************************************************/
/*									*/
/*	ASH_lock_remove -- lock a window for remove			*/
/*									*/
/************************************************************************/


Boolean
ASH_lock_remove(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p,s;
   ASH_LOCK_RQST rq,lrq,nrq;
   Boolean fg;

   p = find_lock_win(w);

   L_PROTECT(p);

   if (p->lock) {
      lrq = NULL;
      for (rq = p->lock->requests; rq != NULL; rq = nrq) {
	 nrq = rq->next;
	 for (s = rq->window; s != NULL && s != w; s = s->parent);
	 if (s != NULL) {
	    if (lrq == NULL) p->lock->requests = nrq;
	    else lrq->next = nrq;
	    free(rq);
	  }
	 else lrq = rq;
       };
      for (rq = p->lock->requests; rq != NULL; rq = rq->next) {
	 if (rq->remove) {
	    for (s = w; s != NULL && s != rq->window; s = s->parent);
	    if (s != NULL) break;
	  };
       };
      if (rq == NULL) {
	 rq = PALLOC(ASH_LOCK_RQST_INFO);
	 rq->next = p->lock->requests;
	 p->lock->requests = rq;
	 rq->window = w;
	 rq->remove = TRUE;
	 rq->refresh = FALSE;
       };
      fg = FALSE;
    }
   else fg = TRUE;

   L_UNPROTECT(p);

   return fg;
};





/************************************************************************/
/*									*/
/*	ASH_lock_refresh -- lock a window for refreshing		*/
/*									*/
/************************************************************************/


Boolean
ASH_lock_refresh(w)
   ASH_WINDOW w;
{
   ASH_WINDOW p,s;
   ASH_LOCK_RQST rq;
   Boolean fg;

   p = find_lock_win(w);

   L_PROTECT(p);

   if (p->lock) {
      for (rq = p->lock->requests; rq != NULL; rq = rq->next) {
	 for (s = w; s != NULL && s != rq->window; s = s->parent);
	 if (s == w || (s != NULL && rq->remove)) break;
       };
      if (rq == NULL) {
	 rq = PALLOC(ASH_LOCK_RQST_INFO);
	 rq->next = p->lock->requests;
	 p->lock->requests = rq;
	 rq->window = w;
	 rq->remove = FALSE;
	 rq->refresh = TRUE;
       }
      else if (rq->window == w && !rq->remove) rq->refresh = TRUE;
      fg = FALSE;
    }
   else fg = TRUE;

   L_UNPROTECT(p);

   return fg;
};





/************************************************************************/
/*									*/
/*	find_lock_win -- find master lock window given window		*/
/*									*/
/************************************************************************/


static ASH_WINDOW
find_lock_win(w)
   ASH_WINDOW w;
{
   while (w != NULL) {
      if (w->lock_by != NULL && ASHinq_valid_window(w->lock_by)) w = w->lock_by;
      else if (w->independent || w->parent == NULL) break;
      else w = w->parent;
    };

   return w;
};





/* end of ashlock.c */
