/************************************************************************/
/*									*/
/*		ashinput.c						*/
/*									*/
/*	Main entries for input handling and correlation thru ash	*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_REFRESH_COUNT	2




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	ASH_SENSE	last_sense = NULL; /* last sensitive region	*/
static	ASH_SENSE	free_sense = NULL; /* free sensitive regions	*/
static	ASH_WINDOW	enter_win = NULL;
static	ASH_WINDOW	last_grab = NULL;
static	THREAD		grab_thread = NULL;
static	Boolean 	doing_sense = FALSE;



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		set_sense();
static	void		draw_sense();
static	void		handle_grab();
static	ASH_WINDOW	query_input();
static	Integer 	future_expose();
static	Boolean 	expose_check();
static	void		invalidate_root();




/************************************************************************/
/*									*/
/*	ASH_input_init -- initialize this module			*/
/*									*/
/************************************************************************/


void
ASH_input_init()
{
   last_sense = NULL;
   free_sense = NULL;
   last_grab = NULL;
   enter_win = NULL;
   grab_thread = NULL;
   doing_sense = FALSE;
};






/************************************************************************/
/*									*/
/*	ASHfind_child_window -- find window at given point		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHfind_child_window(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   Window xw,chld,nchld;
   Integer nx,ny,fg;

   TRACE("ASHfind_child_window 0x%x",w);

   CHECKWIN(w,ASHfind_child_window);

   PROTECT;

   x = XMAPVIEW(w,x);
   y = YMAPVIEW(w,y);
   xw = w->x_win;

   fg = XTranslateCoordinates(DISPLAYOF(w),xw,xw,x,y,&nx,&ny,&chld);

   while (fg != 0 && chld != NULL) {
      fg = XTranslateCoordinates(DISPLAYOF(w),xw,chld,x,y,&nx,&ny,&nchld);
      xw = chld;
      x = nx;
      y = ny;
      chld = nchld;
    };

   if (fg == 0) w = NULL;
   else w = ASH_find_ash_window(DISPLAYOF(w),xw);

   UNPROTECT;

   return w;
};






/************************************************************************/
/*									*/
/*	ASHsensitive_area -- define sensitive area for current window	*/
/*									*/
/************************************************************************/


ASH_SENSE
ASHsensitive_area(w,lx,by,rx,ty,value)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
   ASH_SENSE_TYPE value;
{
   register ASH_SENSE s;
   Integer mask;
   Integer x;

   TRACE("ASHsenstive_area 0x%x %d %d %d %d",w,lx,by,rx,ty);

   CHECKWIN(w,ASHsensitive_area);

   if ((w->inv_x && lx < rx) || (!w->inv_x && lx > rx)) { x = lx; lx = rx; rx = x; };
   if ((w->inv_y && ty < by) || (!w->inv_y && ty > by)) { x = ty; ty = by; by = x; };

   PROTECT;

   if (free_sense != NULL) {
      s = free_sense;
      free_sense = s->nextsense;
    }
   else s = (ASH_SENSE) malloc(sizeof(__SENSEBOX));

   UNPROTECT;

   if (w->sensitive == NULL && w->inputdata == NULL) {
      if (w->parent != NULL) mask = WINDOW_EVENT_MASK;
      else if (w->x_win == XRootWindowOfScreen(w->display->screen_id)) mask = XROOT_EVENT_MASK;
      else mask = ROOT_EVENT_MASK;
      mask |= SENSITIVE_MASK;
      PROTECT;
      XSelectInput(DISPLAYOF(w),w->x_win,mask);
      UNPROTECT;
    };

   s->region.lx = lx;
   s->region.by = by;
   s->region.rx = rx;
   s->region.ty = ty;
   s->window = w;
   s->value = value;

   s->nextsense = w->sensitive;
   w->sensitive = s;

   return s;
};





/************************************************************************/
/*									*/
/*	ASHsensitive_remove -- remove particular sensitive region	*/
/*									*/
/************************************************************************/


void
ASHsensitive_remove(s)
   ASH_SENSE s;
{
   register ASH_WINDOW w;
   register ASH_SENSE sa;

   TRACE("ASHsensitive_remove 0x%x",s);

   PROTECT;
   if (last_sense == s) set_sense(NULL,0,0);
   UNPROTECT;

   w = s->window;
   if (w == NULL) return;

   if (w->sensitive == s) {
      w->sensitive = s->nextsense;
    }
   else {
      for (sa = w->sensitive; sa != NULL && sa->nextsense != s; sa = sa->nextsense);
      if (sa != NULL) sa->nextsense = s->nextsense;
    };

   PROTECT;

   s->window = NULL;
   s->nextsense = free_sense;
   free_sense = s;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHsensitive_remove_all -- remove all sensitive regions 	*/
/*									*/
/************************************************************************/


void
ASHsensitive_remove_all(w)
   ASH_WINDOW w;
{
   register ASH_SENSE s;

   TRACE("ASHsensitive_remove_all 0x%x",w);

   PROTECT;
   if (last_sense != NULL && last_sense->window == w)
      set_sense(NULL,0,0);
   UNPROTECT;

   if (w->sensitive != NULL) {
      PROTECT;
      for (s = w->sensitive; s->nextsense != NULL; s = s->nextsense)
	 s->window = NULL;
      s->nextsense = free_sense;
      s->window = NULL;
      free_sense = w->sensitive;
      w->sensitive = NULL;
      UNPROTECT;
    };
};





/************************************************************************/
/*									*/
/*	ASHinq_sensitive -- return last sensitive region		*/
/*									*/
/************************************************************************/


ASH_SENSE
ASHinq_sensitive()
{
   TRACE("ASHinq_sensitive");

   return last_sense;
};





/************************************************************************/
/*									*/
/*	ASHinput_window -- mark window as an input window		*/
/*									*/
/************************************************************************/


void
ASHinput_window(w,data)
   ASH_WINDOW w;
   Universal data;
{
   Integer mask;

   CHECKWIN(w,ASHinput_window);

   w->inputdata = data;

   if (w->parent != NULL) mask = WINDOW_EVENT_MASK;
   else if (w->x_win == XRootWindowOfScreen(w->display->screen_id)) mask = XROOT_EVENT_MASK;
   else mask = ROOT_EVENT_MASK;
   if (data != NULL) mask |= INPUT_EVENT_MASK;
   else if (w->sensitive != NULL) mask |= SENSITIVE_MASK;

   PROTECT;

   XSelectInput(DISPLAYOF(w),w->x_win,mask);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHinq_input_data -- return input data for window		*/
/*									*/
/************************************************************************/


Universal
ASHinq_input_data(w)
   ASH_WINDOW w;
{
   while (w != NULL) {
      if (w->inputdata != NULL) return w->inputdata;
      w = w->parent;
    };

   return NULL;
};





/************************************************************************/
/*									*/
/*	ASHgrab_input -- grab input for a particular window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHgrab_input(w)
   ASH_WINDOW w;
{
   ASH_WINDOW ow;

   PROTECT;

   while (last_grab != NULL && THREADinq_current() != grab_thread) {
      UNPROTECT;
      THREADreschedule();
      PROTECT;
    };

   ow = last_grab;
   last_grab = w;
   grab_thread = THREADinq_current();

   UNPROTECT;

   return ow;
};





/********************************************************************************/
/*										*/
/*	ASHgrab_from -- grab input from a particular window			*/
/*	ASHgrab_text_from -- grab text input from a particular window		*/
/*										*/
/********************************************************************************/


ASH_WINDOW
ASHgrab_from(w,forw)
   ASH_WINDOW w;
   ASH_WINDOW forw;
{
   ASH_WINDOW ow;

   PROTECT;
   ow = w->grab_by;
   w->grab_by = forw;
   UNPROTECT;

   return ow;
};





ASH_WINDOW
ASHgrab_text_from(w,forw)
   ASH_WINDOW w;
   ASH_WINDOW forw;
{
   ASH_WINDOW ow;

   PROTECT;
   ow = w->grab_text_by;
   w->grab_text_by = forw;
   UNPROTECT;

   return ow;
};





/************************************************************************/
/*									*/
/*	ASHenter_window -- enter given window				*/
/*									*/
/************************************************************************/


void
ASHenter_window(w)
   ASH_WINDOW w;
{
   register ASH_WINDOW p,q;

   if (w == enter_win) return;

   for (p = enter_win; p != NULL; p = p->parent) {
      for (q = w; q != NULL && q != p; q = q->parent);
      if (q != NULL) break;
      if (p->threadct <= 0) ASHcontrol_msg(p,"ASH$EXIT");
    };

   for (p = enter_win; p != NULL; p = p->parent) {
      for (q = w; q != NULL && q != p; q = q->parent);
      if (q != NULL) break;
      if (p->threadct <= 0) ASHcontrol_msg(p,"ASH$EXIT");
    };

   for (q = w; q != NULL; q = q->parent) {
      if (q == p) break;
      if (q->threadct <= 0) ASHcontrol_msg(q,"ASH$ENTER");
    };

   enter_win = w;
};





/************************************************************************/
/*									*/
/*	ASHevent_process -- try to process X event			*/
/*									*/
/*		returns NULL to indicate event used			*/
/*		returns ASH_WINDOW for input if event shd be processed	*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHevent_process(ev)
   XEvent *ev;
{
   ASH_WINDOW w;
   Boolean used;
   Integer lx,by,rx,ty;

   used = TRUE;

   switch (ev->type) {
      case GraphicsExpose :
      case Expose : {
	 XExposeEvent * xe = &ev->xexpose;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && !ASH__exiting) {
	    if (L_PROTECT(w)) {
	       lx = XUNMAPVIEW(w,xe->x);
	       rx = XUNMAPVIEW(w,xe->x+xe->width-1);
	       ty = YUNMAPVIEW(w,xe->y);
	       by = YUNMAPVIEW(w,xe->y+xe->height-1);
	       used = ASHregister_damage(w,future_expose(xe->display,xe->window),lx,by,rx,ty);
	       L_UNPROTECT(w);
	     };
	  };
	 break;
       }
      case MapNotify : {
	 XMapEvent * xe = &ev->xmap;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 ASH_call_for_tree(w,invalidate_root,TRUE);
	 used = FALSE;
	 break;
       }
      case UnmapNotify : {
	 XUnmapEvent * xe = &ev->xunmap;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 used = FALSE;
	 break;
       }
      case EnterNotify : {
	 XEnterWindowEvent * xe = &ev->xcrossing;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && last_grab == NULL) ASHenter_window(w);
	 if (w != NULL && w->inputdata != NULL) used = FALSE;
	 break;
       }
      case LeaveNotify : {
	 XEnterWindowEvent * xe = &ev->xcrossing;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && w->parent == NULL) ASHenter_window(NULL);
	 if (w != NULL && w->inputdata != NULL) used = FALSE;
	 break;
       }
      case ConfigureNotify : {
	 XConfigureEvent * xe = &ev->xconfigure;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 ASH_call_for_tree(w,invalidate_root,TRUE);
	 used = FALSE;
	 break;
       }

      case KeyPress :
      case KeyRelease : {
	 XKeyEvent * xe = &ev->xkey;
	 if (last_grab != NULL) handle_grab(xe);
	 w = query_input(xe->display,xe->x,xe->y,xe->window,xe->subwindow,TRUE);
	 used = FALSE;
	 break;
       }

      case ButtonPress :
      case ButtonRelease :
      case MotionNotify : {
	 XKeyEvent * xe = &ev->xkey;
	 if (last_grab != NULL) handle_grab(xe);
	 w = query_input(xe->display,xe->x,xe->y,xe->window,xe->subwindow,FALSE);
	 used = FALSE;
	 break;
       }

      case DestroyNotify : {
	 XDestroyWindowEvent * xe = &ev->xdestroywindow;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && ASHinq_valid_window(w) && w->parent == NULL) ASHremove(w);
	 break;
       }

      default :
	 used = FALSE;
	 w = NULL;
	 break;
    };

   if (w == NULL) {
      PROTECT;
/*    XtDispatchEvent(ev);	* must have initialized toolkit */
      UNPROTECT;
    };

   if (used) w = NULL;

   return w;
};




/************************************************************************/
/*									*/
/*	ASHinput_script -- handle script input event			*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHinput_script(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   ASH_WINDOW p;

   ASHcursor_move(w,x,y);

   for (p = w; p != NULL; p = p->parent) {
      if (p->grab_by != NULL) {
	 ASHmap(w,x,y,p->grab_by,&x,&y);
	 w = p->grab_by;
	 break;
       };
    };

   set_sense(w,x,y);

   while (w != NULL && w->inputdata == NULL) w = w->parent;

   return w;
};





/************************************************************************/
/*									*/
/*	ASH_event_check_save -- check if we should save the event now	*/
/*									*/
/************************************************************************/


Boolean
ASH_event_check_save(ev,thp,wp)
   XEvent *ev;
   THREAD * thp;
   ASH_WINDOW * wp;
{
   ASH_WINDOW w;
   THREAD th;
   Boolean fg;

   *thp = NULL;
   *wp = NULL;

   if (!USE_THREADS) return FALSE;

   th = NULL;
   w = NULL;
   fg = FALSE;

   switch (ev->type) {
      case EnterNotify : {
	 XEnterWindowEvent * xe = &ev->xcrossing;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && last_grab == NULL) fg = TRUE;
	 break;
       }
      case LeaveNotify : {
	 XEnterWindowEvent * xe = &ev->xcrossing;
	 w = ASH_find_ash_window(xe->display,xe->window);
	 if (w != NULL && w->parent == NULL) fg = TRUE;
	 break;
       }
      default :
	 break;
    };

   if (fg) {
      if (w->threadct > 0 || ASH_inq_batched()) th = w->thread;
      else fg = FALSE;
    };

   *thp = th;
   *wp = w;

   return fg;
};




/************************************************************************/
/*									*/
/*	ASH_desensitize -- remove sensitive regions			*/
/*									*/
/************************************************************************/


void
ASH_desensitize(w)
   ASH_WINDOW w;
{
   PROTECT;
   if (last_sense != NULL && last_sense->window == w) {
      set_sense(NULL,0,0);
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASH_remove_enter -- handle remove of enter window		*/
/*									*/
/************************************************************************/


void
ASH_remove_enter(w)
   ASH_WINDOW w;
{
   if (w == enter_win) ASHenter_window(NULL);
};





/************************************************************************/
/*									*/
/*	set_sense -- set sensitive region				*/
/*									*/
/************************************************************************/


static void
set_sense(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   register ASH_SENSE s,ls;
   register Integer xfg,yfg;

   if (w != NULL && w->threadct > 0) return;
   if (doing_sense) return;
   doing_sense = TRUE;

   if (w != NULL) {
      xfg = (w->view.lx > w->view.rx);
      yfg = (w->view.by > w->view.ty);
      for (s = w->sensitive; s != NULL; s = s->nextsense) {
	 if (xfg) {
	    if (x < s->region.rx || x > s->region.lx) continue;
	  }
	 else {
	    if (x < s->region.lx || x > s->region.rx) continue;
	  };
	 if (yfg) {
	    if (y < s->region.ty || y > s->region.by) continue;
	  }
	 else {
	    if (y < s->region.by || y > s->region.ty) continue;
	  };
	 break;
       };
    }
   else s = NULL;

   if (s != last_sense) {
      ls = last_sense;
      last_sense = NULL;
      if (ls != NULL) draw_sense(ls,FALSE);
      if (s != NULL) draw_sense(s,TRUE);
      last_sense = s;
    };

   doing_sense = FALSE;
};





/************************************************************************/
/*									*/
/*	draw_sense -- draw for a sensitive region			*/
/*									*/
/************************************************************************/


static void
draw_sense(s,fg)
   ASH_SENSE s;
   Boolean fg;
{
   register Integer cr,pm,ls,co;
   register Integer clip;

   if (s == NULL) return;

   if (s->value != ASH_SENSE_NO_CHANGE) {
      ASHbatch_mode(TRUE);
      clip = ASHclip(s->window,FALSE);
      cr = ASHinq_combination_rule(s->window);
      pm = ASHinq_plane_mask(s->window);

      if (s->value == ASH_SENSE_FLIP) {
	 ASHhilite_box(s->window,s->region.lx,s->region.by,s->region.rx,s->region.ty);
       }
      else if (s->value  == ASH_SENSE_BOX) {
	 co = ASHcolor(s->window,1);
	 ASHdraw_hilite(s->window,TRUE);
	 ls = ASHline_style(s->window,ASH_STYLE_THICK);
	 ASHbox(s->window,s->region.lx,s->region.by,s->region.rx,s->region.ty);
	 ASHline_style(s->window,ls);
	 ASHdraw_hilite(s->window,FALSE);
	 ASHcolor(s->window,co);
       }
      else if (s->value == ASH_SENSE_HALF) {
	 ASHdraw_hilite(s->window,TRUE);
	 ASHstipple_box(s->window,s->region.lx,s->region.by,s->region.rx,s->region.ty);
	 ASHdraw_hilite(s->window,FALSE);
       }
      else {
	 (*s->value)(s->window,s->region.lx,s->region.by,s->region.rx,s->region.ty,fg,s);
       };

      ASHplane_mask(s->window,pm);
      ASHcombination_rule(s->window,cr);
      ASHclip(s->window,clip);
      ASHbatch_mode(FALSE);
    };
};





/************************************************************************/
/*									*/
/*	handle_grab -- handle grabbing events for input 		*/
/*									*/
/************************************************************************/


static void
handle_grab(ev)
   XKeyEvent * ev;
{
   Integer x,y;
   Window chld;
   ASH_WINDOW w;

   PROTECT;
   if (last_grab == NULL || last_grab->x_win == ev->window) {
      UNPROTECT;
      return;
    };

   w = ASH_find_ash_window(ev->display,ev->window);

   if (w != NULL && w->x_win == ev->window) {
      x = XUNMAPXVIEW(w,ev->x);
      y = YUNMAPXVIEW(w,ev->y);
      ASHmap(w,x,y,last_grab,&x,&y);
      chld = NULL;
    }
   else if (w != NULL) {
      XTranslateCoordinates(DISPLAYOF(last_grab),ev->window,last_grab->x_win,
			       ev->x,ev->y,&x,&y,&chld);
    };

   if (w != NULL) {
      ev->window = last_grab->x_win;
      ev->x = x;
      ev->y = y;
      ev->subwindow = chld;
    };

   UNPROTECT;
};








/************************************************************************/
/*									*/
/*	query_input -- handle input correlation with ASH		*/
/*									*/
/************************************************************************/


static ASH_WINDOW
query_input(xd,x,y,xw,sw,key)
   Display * xd;
   Integer x,y;
   Window xw,sw;
   Boolean key;
{
   register ASH_WINDOW w,p;
   Window chld;

   PROTECT;

   if (sw == NULL) sw = xw;

   w = ASH_find_ash_window(xd,xw);

   if (w != NULL) {
      if (xw != w->x_win) {
	 XTranslateCoordinates(DISPLAYOF(w),xw,w->x_win,x,y,&x,&y,&chld);
       };
      x = XUNMAPXVIEW(w,x);
      y = YUNMAPXVIEW(w,y);
    };

   for (p = w; p != NULL; p = p->parent) {
      if (p->grab_by != NULL) {
	 ASHmap(w,x,y,p->grab_by,&x,&y);
	 w = p->grab_by;
	 break;
       };
      if (key && p->grab_text_by != NULL) {
	 ASHmap(w,x,y,p->grab_by,&x,&y);
	 w = p->grab_text_by;
	 break;
       };
    };

   set_sense(w,x,y);

   UNPROTECT;

   while (w != NULL && w->inputdata == NULL) w = w->parent;

   return w;
};






/************************************************************************/
/*									*/
/*	future_expose -- check if there is another expose event in queue*/
/*									*/
/************************************************************************/


typedef struct _EX_INFO {
   Window w;
   Integer fg;
} EX_INFO;


static Integer
future_expose(d,w)
   Display * d;
   Window w;
{
   EX_INFO in;

   PROTECT;

   in.w = w;
   in.fg = 0;
   BPIO_future_events(d,expose_check,&in);

   UNPROTECT;

   return in.fg;
};




/*ARGSUSED*/

static Boolean
expose_check(d,ev,in)
   Display * d;
   XExposeEvent * ev;
   EX_INFO * in;
{
   if ((ev->type == Expose  || ev->type == GraphicsExpose) &&
	  ev->window == in->w) ++in->fg;

   return FALSE;
};





/************************************************************************/
/*									*/
/*	invalidate_root -- handle invalidating root mapping for window	*/
/*									*/
/************************************************************************/


static void
invalidate_root(w)
   ASH_WINDOW w;
{
   w->rootmap_valid = FALSE;
};





/* end of ashinput.c */
