/************************************************************************/
/*									*/
/*		ashfont.c						*/
/*									*/
/*	Entries for managing fonts with ASH				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/

#include "ash_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define FAMILY_FILE	"%s/lib/bwe/font.family"
#define FAMILY_FILE0	"%s/bwe/lib/font.family.%s"
#define FAMILY_FILE1	"%s/bwe/lib/font.family"

#define FONT_PER_FAM	1024		/* max fonts per family 	*/




/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct _FONT_FAMILY *	FONT_FAMILY;
typedef struct _FONT_DATA *	FONT_DATA;




typedef struct _FONT_FAMILY {
   String name;
   Integer numfont;
   FONT_DATA fonts;
   FONT_FAMILY next;
} FONT_FAMILY_INFO;





typedef struct _FONT_DATA {
   String name;
   Integer size;
   ASH_FONT_PROPS props;
} FONT_DATA_INFO;




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	FONT_FAMILY	all_familys;
static	FONT_DATA_INFO	dummy_data;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	FONT_DATA	find_font_data();
static	void		load_family_data();
static	void		add_family();
static	ASH_FONT_PROPS	scan_properties();





/************************************************************************/
/*									*/
/*	ASH_font_init -- module initialization				*/
/*									*/
/************************************************************************/


void
ASH_font_init()
{
   all_familys = NULL;

   dummy_data.name = ASHinq_base_font();
   dummy_data.size = 10;
   dummy_data.props = 0;
};






/************************************************************************/
/*									*/
/*	ASHfont_name -- return name of font matching properties 	*/
/*	ASHfont_id -- return id of font matching properties		*/
/*									*/
/************************************************************************/


char *
ASHfont_name(fam,size,prop)
   String fam;
   Integer size;
   ASH_FONT_PROPS prop;
{
   FONT_DATA fd;

   ENTER("ASHfont_name %s %d 0x%x",fam,size,prop);

   fd = find_font_data(fam,size,prop,TRUE);

   return fd->name;
};






char *
ASHfont_name_check(fam,size,prop)
   String fam;
   Integer size;
   ASH_FONT_PROPS prop;
{
   FONT_DATA fd;

   ENTER("ASHfont_name %s %d 0x%x",fam,size,prop);

   fd = find_font_data(fam,size,prop,FALSE);

   if (fd == NULL) return NULL;

   return fd->name;
};






ASH_FONT
ASHfont_id(fam,size,prop)
   String fam;
   Integer size;
   ASH_FONT_PROPS prop;
{
   FONT_DATA fd;

   ENTER("ASHfont_name %s %d 0x%x",fam,size,prop);

   fd = find_font_data(fam,size,prop,TRUE);

   return ASHloadfont(fd->name);
};






/************************************************************************/
/*									*/
/*	find_font_data -- find font matching user request		*/
/*									*/
/************************************************************************/


static FONT_DATA
find_font_data(family,size,prop,force)
   String family;
   Integer size;
   ASH_FONT_PROPS prop;
   Boolean force;
{
   FONT_FAMILY fam;
   Integer i;
   FONT_DATA font,fd;
   Character buf[1024];
   ASH_FONT_PROPS p1,p2;;

   if (all_familys == NULL) {
      load_family_data();
      if (all_familys == NULL) {
	 if (force) {
	    ERROR("No font familys defined");
	    return &dummy_data;
	  }
	 else return NULL;
       };
    };

   font = NULL;
   for (fam = all_familys; fam != NULL; fam = fam->next) {
      if (family == NULL || STREQL(family,fam->name)) {
	 p1 = prop;
	 if ((p1 & (ASH_FONT_ROMAN|ASH_FONT_ITALIC)) == 0) p1 |= ASH_FONT_ROMAN;
	 for ( ; ; ) {
	    for (i = 0; i < fam->numfont; ++i) {
	       fd = &fam->fonts[i];
	       if (size > 0 && fd->size > size) continue;
	       if ((p1 & ASH_FONT_ANY) == 0 && (p1 & fd->props) != p1) continue;
	       if (font == NULL || font->size < fd->size) font = fd;
	       if (font->size == size) break;
	     };
	    if (font != NULL) break;
	    else if ((p1 & ASH_FONT_ANY) != 0) break;
	    else if (p1 != prop) p1 = prop;
	    else if (family == NULL) break;
	    else {
	       p2 = p1;
	       p1 &= (ASH_FONT_ROMAN|ASH_FONT_ITALIC);
	       if (p1 == p2) p1 = NULL;
	       if (p1 == NULL && size > 0) p1 = ASH_FONT_ANY;
	       else if (p1 == NULL) break;
	     };
	  };
       };
      if (font != NULL && font->size == size) break;
    };

   if (font == NULL && force) {
      sprintf(buf,"Font family %s, size %d, props 0x%x not found",family,size,prop);
      ERROR(buf);
      font = &dummy_data;
    };

   return font;
};





/************************************************************************/
/*									*/
/*	load_family_data -- load the family definition file		*/
/*									*/
/************************************************************************/


static void
load_family_data()
{
   Character buf[1024],sbuf[1024],pbuf[1024];
   FILE * inf;
   FONT_FAMILY fam;
   FONT_DATA fd;
   FONT_DATA_INFO fts[FONT_PER_FAM];
   Integer sz,ct;
   String s;

   s = getenv("FONT_FAMILY");
   if (s != NULL) {
      strcpy(buf,s);
      if (access(buf,4) < 0) s = NULL;
    };
   if (s == NULL) {
      sprintf(buf,FAMILY_FILE,BWEbwe_project(),BWEarch());
      if (access(buf,4) < 0) {
	 sprintf(buf,FAMILY_FILE0,BWEbwe_project(),BWEarch());
	 if (access(buf,4) < 0) {
	    sprintf(buf,FAMILY_FILE1,BWEbwe_project(),BWEarch());
	  };
       };
    };

   inf = fopen(buf,"r");
   if (inf == NULL) return;

   fam = NULL;
   ct = 0;

   for ( ; ; ) {
      if (fgets(buf,1024,inf) == NULL) break;
      pbuf[0] = 0;
      if (sscanf(buf,"FAMILY %s\n",sbuf) == 1) {
	 add_family(fam,ct,fts);
	 ct = 0;
	 fam = PALLOC(FONT_FAMILY_INFO);
	 fam->name = SALLOC(sbuf);
	 fam->next = NULL;
	 fam->numfont = 0;
       }
      else if (sscanf(buf,"%d %s %s\n",&sz,pbuf,sbuf) == 3 ||
		  sscanf(buf,"%d %s\n",&sz,sbuf) == 2) {
	 if (sz > 0) {
	    fd = &fts[ct++];
	    fd->name = SALLOC(sbuf);
	    fd->size = sz;
	    fd->props = scan_properties(pbuf);
	  };
       };
    };

   fclose(inf);
   add_family(fam,ct,fts);
};






/************************************************************************/
/*									*/
/*	add_family -- add a new family					*/
/*									*/
/************************************************************************/


static void
add_family(fam,ct,fts)
   FONT_FAMILY fam;
   Integer ct;
   FONT_DATA_INFO fts[];
{
   Integer i;
   FONT_FAMILY f;

   if (fam == NULL) return;

   fam->numfont = ct;
   fam->fonts = (FONT_DATA) calloc(fam->numfont,sizeof(FONT_DATA_INFO));
   for (i = 0; i < ct; ++i) {
      fam->fonts[i] = fts[i];
    };

   if (all_familys == NULL) all_familys = fam;
   else {
      for (f = all_familys; f->next != NULL; f = f->next) ;
      f->next = fam;
    };
};





/************************************************************************/
/*									*/
/*	scan_properties -- scan font property names and convert 	*/
/*									*/
/************************************************************************/


static ASH_FONT_PROPS
scan_properties(p)
   String p;
{
   ASH_FONT_PROPS v;

   v = 0;

   for ( ; *p != 0; ++p) {
      switch (*p) {
	 case 'r' :
	 case 'R' :
	    v |= ASH_FONT_ROMAN;
	    break;
	 case 'i' :
	 case 'I' :
	    v |= ASH_FONT_ITALIC;
	    break;
	 case 'b' :
	 case 'B' :
	    v |= ASH_FONT_BOLD;
	    break;
	 case 'o' :
	 case 'O' :
	    v |= ASH_FONT_OUTLINE;
	    break;
	 case 's' :
	 case 'S' :
	    v |= ASH_FONT_SHADOW;
	    break;
	 default :
	    ERROR("Bad font property");
	    break;
       };
    };

   return v;
};





/* end of ashfont.c */
