/************************************************************************/
/*									*/
/*		ashdraw.c						*/
/*									*/
/*	Main entries for drawing in an ASH window			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"




/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef enum {
   POINTSET_POINT,
   POINTSET_LINE,
   POINTSET_POLYGON,
   POINTSET_CONVEX,
   POINTSET_GENERAL,
   ARCSET_LINE,
   ARCSET_CHORD,
   ARCSET_SLICE,
} POINTSET_TYPE;



/************************************************************************/
/*									*/
/*	Local macros							*/
/*									*/
/************************************************************************/


#define ANGLE(deg)	((deg)*64)

#define BOX_ADD 	0		/* add to abs(diff coords) to get size	*/
#define RECT_ADD	1		/* for boxes and rectangles		*/
#define COPY_ADD	0		/* for copyarea 			*/




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		round_rectangle();
static	void		draw_circle();
static	void		draw_pointset();
static	void		do_point_draw();
static	void		draw_arc();
static	void		do_arc_draw();
static	void		between_display_blt();
static	void		do_save();





/************************************************************************/
/*									*/
/*	Macro definitions						*/
/*									*/
/************************************************************************/


#define VIEW_DRAW(w)	((w)->use_view && (w)->visible)
#define FULL_DRAW(w)	((w)->full_win != NULL && (!(w)->frame || (w)->visible))





/************************************************************************/
/*									*/
/*	ASH_drawinit -- initialize for drawing				*/
/*									*/
/************************************************************************/


void
ASH_draw_init()
{
};






/************************************************************************/
/*									*/
/*	ASHline -- draw a line						*/
/*	ASHpolyline -- draw a sequence of lines 			*/
/*	ASHpoint -- draw a point					*/
/*	ASHpolypoint -- draw a number of points 			*/
/*	ASHpolygon -- draw a filled convex polygon			*/
/*	ASHconvex_polygon -- draw a filled convex polygon		*/
/*	ASHgeneral_polygon -- draw a filled arbitrary polygon		*/
/*									*/
/************************************************************************/


void
ASHline(w,x0,y0,x1,y1)
   ASH_WINDOW w;
   Integer x0,y0;
   Integer x1,y1;
{
   TRACE("ASHline 0x%x (%d,%d)->(%d,%d)",w,x0,y0,x1,y1);

   DRAW_BEGIN(w);

   x0 = XMAPVIEW(w,x0);
   y0 = YMAPVIEW(w,y0);
   x1 = XMAPVIEW(w,x1);
   y1 = YMAPVIEW(w,y1);

   if (VIEW_DRAW(w)) {
      XDrawLine(DISPLAYOF(w),w->view_win,w->draw->context,x0,y0,x1,y1);
    }

   if (FULL_DRAW(w)) {
      x0 = XMAPVFULL(w,x0);
      y0 = YMAPVFULL(w,y0);
      x1 = XMAPVFULL(w,x1);
      y1 = YMAPVFULL(w,y1);
      XDrawLine(DISPLAYOF(w),w->full_win,w->draw->full_context,x0,y0,x1,y1);
    };

   DRAW_END(w);
};





void
ASHpolyline(w,numpts,x,y)
   ASH_WINDOW w;
   Integer numpts;
   Integer x[]; 	/* [numpts] */
   Integer y[]; 	/* [numpts] */
{
   TRACE("ASHpolyline 0x%x %d (%d,%d)->(%d,%d)->...",w,numpts,x[0],y[0],x[1],y[1]);

   draw_pointset(w,POINTSET_LINE,numpts,x,y);
};





void
ASHpoint(w,x0,y0)
   ASH_WINDOW w;
   Integer x0,y0;
{
   TRACE("ASHpoint 0x%x (%d,%d)",w,x0,y0);

   DRAW_BEGIN(w);

   x0 = XMAPVIEW(w,x0);
   y0 = YMAPVIEW(w,y0);

   if (VIEW_DRAW(w)) {
      XDrawPoint(DISPLAYOF(w),w->view_win,w->draw->context,x0,y0);
    };

   if (FULL_DRAW(w)) {
      x0 = XMAPVFULL(w,x0);
      y0 = YMAPVFULL(w,y0);
      XDrawPoint(DISPLAYOF(w),w->full_win,w->draw->full_context,x0,y0);
    };

   DRAW_END(w);
};





void
ASHpolypoint(w,numpts,x,y)
   ASH_WINDOW w;
   Integer numpts;
   Integer x[]; 	/* [numpts] */
   Integer y[]; 	/* [numpts] */
{
   TRACE("ASHpolypoint 0x%x %d (%d,%d),...",w,numpts,x[0],y[0]);

   draw_pointset(w,POINTSET_POINT,numpts,x,y);
};





void
ASHpolygon(w,numpts,x,y)
   ASH_WINDOW w;
   Integer numpts;
   Integer x[]; 	/* [numpts] */
   Integer y[]; 	/* [numpts] */
{
   TRACE("ASHpolygon 0x%x %d (%d,%d)-(%d,%d)-...",w,numpts,x[0],y[0],x[1],y[1]);

   draw_pointset(w,POINTSET_POLYGON,numpts,x,y);
};





void
ASHconvex_polygon(w,numpts,x,y)
   ASH_WINDOW w;
   Integer numpts;
   Integer x[]; 	/* [numpts] */
   Integer y[]; 	/* [numpts] */
{
   TRACE("ASHconvex_polygon 0x%x %d (%d,%d)-(%d,%d)-...",w,numpts,x[0],y[0],x[1],y[1]);

   draw_pointset(w,POINTSET_CONVEX,numpts,x,y);
};





void
ASHgeneral_polygon(w,numpts,x,y)
   ASH_WINDOW w;
   Integer numpts;
   Integer x[]; 	/* [numpts] */
   Integer y[]; 	/* [numpts] */
{
   TRACE("ASHgeneral_polygon 0x%x %d (%d,%d)-(%d,%d)-...",w,numpts,x[0],y[0],x[1],y[1]);

   draw_pointset(w,POINTSET_GENERAL,numpts,x,y);
};





/************************************************************************/
/*									*/
/*	ASHarc -- draw an arc						*/
/*	ASHarc_chord -- draw a filled chordal region			*/
/*	ASHarc_slice -- draw a filled pie splice			*/
/*									*/
/************************************************************************/


void
ASHarc(w,cx,cy,rx,ry,a1,a2)
   ASH_WINDOW w;
   Integer cx,cy;
   Integer rx,ry;
   Integer a1,a2;
{
   TRACE("ASHarc 0x%x %d %d %d %d %d %d",w,cx,cy,rx,ry,a1,a2);

   draw_arc(ARCSET_LINE,w,cx,cy,rx,ry,a1,a2);
};





void
ASHarc_chord(w,cx,cy,rx,ry,a1,a2)
   ASH_WINDOW w;
   Integer cx,cy;
   Integer rx,ry;
   Integer a1,a2;
{
   TRACE("ASHarc_chord 0x%x %d %d %d %d %d %d",w,cx,cy,rx,ry,a1,a2);

   draw_arc(ARCSET_CHORD,w,cx,cy,rx,ry,a1,a2);
};





void
ASHarc_slice(w,cx,cy,rx,ry,a1,a2)
   ASH_WINDOW w;
   Integer cx,cy;
   Integer rx,ry;
   Integer a1,a2;
{
   TRACE("ASHarc_slice 0x%x %d %d %d %d %d %d",w,cx,cy,rx,ry,a1,a2);

   draw_arc(ARCSET_SLICE,w,cx,cy,rx,ry,a1,a2);
};





/************************************************************************/
/*									*/
/*	ASHbox -- draw an unfilled box					*/
/*	ASHrectangle -- draw a filled rectangle 			*/
/*	ASHround_box -- draw a round box				*/
/*	ASHround_rectangle -- draw a round rectangle			*/
/*									*/
/************************************************************************/


void
ASHbox(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
{
   register Integer wd,ht;

   TRACE("ASHbox 0x%x (%d,%d)-(%d,%d)",w,lx,by,rx,ty);

   DRAW_BEGIN(w);

   lx = XMAPVIEW(w,lx);
   by = YMAPVIEW(w,by);
   rx = XMAPVIEW(w,rx);
   ty = YMAPVIEW(w,ty);

   wd = abs(rx-lx)+BOX_ADD;
   ht = abs(ty-by)+BOX_ADD;
   lx = MIN(lx,rx);
   ty = MIN(by,ty);

   if (VIEW_DRAW(w)) {
      XDrawRectangle(DISPLAYOF(w),w->view_win,w->draw->context,lx,ty,wd,ht);
    };

   if (FULL_DRAW(w)) {
      lx = XMAPVFULL(w,lx);
      ty = YMAPVFULL(w,ty);
      XDrawRectangle(DISPLAYOF(w),w->full_win,w->draw->full_context,lx,ty,wd,ht);
    };

   DRAW_END(w);
};





void
ASHrectangle(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
{
   register Integer wd,ht;

   TRACE("ASHrectangle 0x%x (%d,%d)-(%d,%d)",w,lx,by,rx,ty);

   DRAW_BEGIN(w);

   lx = XMAPVIEW(w,lx);
   by = YMAPVIEW(w,by);
   rx = XMAPVIEW(w,rx);
   ty = YMAPVIEW(w,ty);

   wd = abs(rx-lx)+RECT_ADD;
   ht = abs(ty-by)+RECT_ADD;
   lx = MIN(lx,rx);
   ty = MIN(by,ty);

   if (VIEW_DRAW(w)) {
      XFillRectangle(DISPLAYOF(w),w->view_win,w->draw->context,lx,ty,wd,ht);
    };

   if (FULL_DRAW(w)) {
      lx = XMAPVFULL(w,lx);
      ty = YMAPVFULL(w,ty);
      XFillRectangle(DISPLAYOF(w),w->full_win,w->draw->full_context,lx,ty,wd,ht);
    };

   DRAW_END(w);
};





void
ASHround_box(w,lx,by,rx,ty,r)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
   Integer r;
{
   TRACE("ASHround_box 0x%x %d %d %d %d %d",w,lx,by,rx,ty,r);

   round_rectangle(w,FALSE,lx,by,rx,ty,r);
};





void
ASHround_rectangle(w,lx,by,rx,ty,r)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
   Integer r;
{
   TRACE("ASHround_rectangle 0x%x %d %d %d %d %d",w,lx,by,rx,ty,r);

   round_rectangle(w,TRUE,lx,by,rx,ty,r);
};





/************************************************************************/
/*									*/
/*	ASHcircle -- draw an unfilled circle				*/
/*	ASHfilled_circle -- draw a filled circle			*/
/*	ASHellipse -- draw a normal ellipse				*/
/*	ASHfilled_ellipse -- draw a normal filled ellipse		*/
/*									*/
/************************************************************************/


void
ASHcircle(w,x,y,r)
   ASH_WINDOW w;
   Integer x,y;
   Integer r;
{
   TRACE("ASHcircle 0x%x (%d,%d)@(%d)",w,x,y,r);

   draw_circle(w,FALSE,x,y,r,r);
};





void
ASHfilled_circle(w,x,y,r)
   ASH_WINDOW w;
   Integer x,y;
   Integer r;
{
   TRACE("ASHfilled_circle 0x%x (%d,%d)@(%d)",w,x,y,r);

   draw_circle(w,TRUE,x,y,r,r);
};





void
ASHellipse(w,x,y,rx,ry)
   ASH_WINDOW w;
   Integer x,y;
   Integer rx,ry;
{
   TRACE("ASHellipse 0x%x (%d,%d)@(%d,%d)",w,x,y,ry,rx);

   draw_circle(w,FALSE,x,y,rx,ry);
};





void
ASHfilled_ellipse(w,x,y,rx,ry)
   ASH_WINDOW w;
   Integer x,y;
   Integer rx,ry;
{
   TRACE("ASHfilled_ellipse 0x%x (%d,%d)@(%d,%d)",w,x,y,ry,rx);

   draw_circle(w,TRUE,x,y,rx,ry);
};





/************************************************************************/
/*									*/
/*	ASHclear -- clear current window				*/
/*									*/
/************************************************************************/


void
ASHclear(w)
   ASH_WINDOW w;
{
   register Integer wd,ht,lx,ty;
   GC gc;
   XGCValues vals;
   Integer mask;

   TRACE("ASHclear 0x%x",w);

   if (w->frame || w->our_court) {
      ASHclear_box(w,w->coords.lx,w->coords.by,w->coords.rx,w->coords.ty);
      return;
    };

   DRAW_BEGIN(w);

   lx = XMAPVIEW(w,w->view.lx);
   ty = YMAPVIEW(w,w->view.ty);

   if (VIEW_DRAW(w)) {
      wd = abs(w->view.lx - w->view.rx)+1;
      ht = abs(w->view.ty - w->view.by)+1;
      XClearArea(DISPLAYOF(w),w->view_win,lx,ty,wd,ht,FALSE);
    };

   if (FULL_DRAW(w)) {
      wd = abs(w->coords.lx - w->coords.rx)+1;
      ht = abs(w->coords.ty - w->coords.by)+1;
      if (w->retained) {
	 XClearArea(DISPLAYOF(w),w->full_win,0,0,wd,ht,FALSE);
       }
      else {
	 if (w->bkg_pixmap == NULL) {
	    vals.fill_style = FillSolid;
	    vals.foreground = w->bkg_pixel;
	    mask = GCFillStyle|GCForeground;
	  }
	 else {
	    vals.tile = w->bkg_pixmap;
	    vals.fill_style = FillTiled;
	    mask = GCFillStyle|GCTile;
	  };
	 gc = XCreateGC(DISPLAYOF(w),w->full_win,mask,&vals);
	 XFillRectangle(DISPLAYOF(w),w->full_win,gc,0,0,wd,ht);
	 XFreeGC(DISPLAYOF(w),gc);
       };
    };

   DRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHtext -- write some text out					*/
/*									*/
/************************************************************************/


void
ASHtext(w,x,y,text)
   ASH_WINDOW w;
   Integer x,y;
   String text;
{
   Function_Ptr fct;
   Integer ln;
   extern XDrawString();
   extern XDrawImageString();

   TRACE("ASHtext 0x%x (%d,%d) <%s>",w,x,y,text);

   if (w->draw->font == ASH_ICON_FONT) {
      ASHicon_draw(w,x,y,text);
      return;
    };

   DRAW_BEGIN(w);

   fct = (w->draw->clear_text ? XDrawString : XDrawImageString);

   ln = strlen(text);
   x = XMAPVIEW(w,x);
   y = YMAPVIEW(w,y);

   if (VIEW_DRAW(w)) {
      (*fct)(DISPLAYOF(w),w->view_win,w->draw->text_context,x,y,text,ln);
    };

   if (FULL_DRAW(w)) {
      x = XMAPVFULL(w,x);
      y = YMAPVFULL(w,y);
      (*fct)(DISPLAYOF(w),w->full_win,w->draw->full_text_context,x,y,text,ln);
    };

   DRAW_END(w);
};





/************************************************************************/
/*									*/
/*	ASHblt -- do a raster operation on current bitmap		*/
/*									*/
/************************************************************************/


void
ASHblt(w,lx,by,rx,ty,dlx,dby)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
   Integer dlx,dby;
{
   Drawable d;
   Integer wd,ht,z;

   TRACE("ASHblt 0x%x (%d,%d) (%d,%d) --> (%d,%d)",w,lx,by,rx,ty,dlx,dby);

   lx = XMAPVIEW(w->draw->source,lx);
   by = YMAPVIEW(w->draw->source,by);
   rx = XMAPVIEW(w->draw->source,rx);
   ty = YMAPVIEW(w->draw->source,ty);
   if (lx > rx) { z = lx; lx = rx; rx = z; };
   if (ty > by) { z = by; by = ty; ty = z; };

   wd = rx-lx+COPY_ADD;
   ht = by-ty+COPY_ADD;

   if (w->draw->source->full_win != NULL) {
      lx = XMAPVFULL(w->draw->source,lx);
      ty = XMAPVFULL(w->draw->source,ty);
      d = (Drawable) w->draw->source->full_win;
    }
   else {
      d = (Drawable) w->draw->source->view_win;
    };

   dlx = XMAPVIEW(w,dlx);
   dby = YMAPVIEW(w,dby);

   if (w->display->display != w->draw->source->display->display) {
      between_display_blt(w,d,lx,ty,wd,ht,dlx,dby-ht);
      return;
    };

   DRAW_BEGIN(w);
   DRAW_BEGIN(w->draw->source);

   if (VIEW_DRAW(w)) {
      XCopyArea(DISPLAYOF(w),d,w->view_win,w->draw->context,lx,ty,wd,ht,dlx,dby-ht);
    };

   if (FULL_DRAW(w)) {
      dlx = XMAPVFULL(w,dlx);
      dby = YMAPVFULL(w,dby);
      XCopyArea(DISPLAYOF(w),d,w->full_win,w->draw->full_context,
		   lx,ty,wd,ht,dlx,dby-ht);
    };

   DRAW_END(w->draw->source);
   DRAW_END(w);
};






/************************************************************************/
/*									*/
/*	ASHwrite_pixels -- write pixels in current bitmap(s)		*/
/*									*/
/************************************************************************/


void
ASHwrite_pixels(w,lx,by,rx,ty,pixels)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
   Integer pixels[];		/* [(abs(rx-lx)+1)*(abs(ty-by)+1)] */
{
   XImage * image;
   Integer wd,ht;

   TRACE("ASHwrite_pixels 0x%x (%d,%d) (%d,%d)",w,lx,by,rx,ty);

   DRAW_BEGIN(w);

   lx = XMAPVIEW(w,lx);
   by = YMAPVIEW(w,by);
   rx = XMAPVIEW(w,rx);
   ty = YMAPVIEW(w,ty);
   wd = rx-lx+1;
   ht = by-ty+1;

   image = XCreateImage(DISPLAYOF(w),
			   w->display->default_visual,
			   w->depth,
			   ZPixmap,
			   0,
			   pixels,
			   wd,ht,8,0);

   if (VIEW_DRAW(w)) {
      XPutImage(DISPLAYOF(w),w->view_win,w->draw->context,image,0,0,lx,ty,wd,ht);
    };

   if (FULL_DRAW(w)) {
      lx = XMAPVFULL(w,lx);
      ty = YMAPVFULL(w,ty);
      XPutImage(DISPLAYOF(w),w->full_win,w->draw->full_context,image,0,0,lx,ty,wd,ht);
    };

   free(image);

   DRAW_END(w);
};






/************************************************************************/
/*									*/
/*	ASHread_pixels -- read pixels in current bitmap 		*/
/*									*/
/************************************************************************/


void
ASHread_pixels(w,lx,by,rx,ty,pixels)
   ASH_WINDOW w;
   Integer lx,by;
   Integer rx,ty;
   Integer pixels[];		/* [(abs(rx-lx)+1)*(abs(ty-by)+1)] */
{
   XImage * image;
   register Integer i,j,k,l,wd,ht;
   Drawable d;
   unsigned char *pt;
   unsigned short *spt;
   unsigned int *ipt;

   TRACE("ASHread_pixels 0x%x (%d,%d) (%d,%d)",w,lx,by,rx,ty);

   DRAW_BEGIN(w);

   lx = XMAPVIEW(w,lx);
   by = YMAPVIEW(w,by);
   rx = XMAPVIEW(w,rx);
   ty = YMAPVIEW(w,ty);
   wd = rx-lx+1;
   ht = by-ty+1;

   if (w->full_win != NULL) {
      lx = XMAPVFULL(w,lx);
      by = XMAPVFULL(w,by);
      rx = XMAPVFULL(w,rx);
      ty = XMAPVFULL(w,ty);
      d = (Drawable) w->full_win;
    }
   else {
      d = (Drawable) w->view_win;
    };

   image = XGetImage(DISPLAYOF(w),d,lx,ty,wd,ht,~0,ZPixmap);

   if (image == NULL) {
      ASH_error("No image returned");
      DRAW_END(w);
      return;
    };

   l = 0;
   for (i = 0; i < ht; ++i) {
      pt = (unsigned char *) image->data;
      pt += (image->bytes_per_line)*i;
      switch (image->bits_per_pixel) {
	 case 1 :
	    for (j = 0; j < wd; ++j) {
	       if ((j % 8) == 0) k = *pt++;
	       pixels[l++] = (((k&0x80) != 0) ? 1 : 0);
	       k <<= 1;
	     };
	    break;
	 case 8 :
	    for (j = 0; j < wd; ++j) pixels[l++] = *pt++;
	    break;
	 case 16 :
	    spt = (unsigned short *) pt;
	    for (j = 0; j < wd; ++j) pixels[l++] = *spt++;
	    break;
	 case 32 :
	    ipt = (unsigned int *) pt;
	    for (j = 0; j < wd; ++j) pixels[l++] = *ipt++;
	    break;
	 default :
	    ASH_error("Bad image pixel size");
	    for (j = 0; j < wd; ++j) pixels[l++] = 0;
	    break;
       };
    };

   XDestroyImage(image);

   DRAW_END(w);
};






/************************************************************************/
/*									*/
/*	ASHsave_bitmap -- save bitmap for current window in a file	*/
/*	ASHsave_local -- save bitmap in machine local format		*/
/*	ASHsave_brim -- save bitmap for current window in BRIM format	*/
/*									*/
/************************************************************************/


void
ASHsave_bitmap(w,file)
   ASH_WINDOW w;
   String file;
{
   TRACE("ASHsave_bitmap 0x%x %s",w,file);

   do_save(w,file);
};






void
ASHsave_local(w,file)
   ASH_WINDOW w;
   String file;
{
   TRACE("ASHsave_local 0x%x %s",w,file);

   do_save(w,file);
};






void
ASHsave_brim(w,file)
   ASH_WINDOW w;
   String file;
{
   TRACE("ASHsave_brim 0x%x %s",w,file);
};






/************************************************************************/
/*									*/
/*	ASHload_bitmap -- load bitmap from file 			*/
/*									*/
/************************************************************************/


void
ASHload_bitmap(w,file)
   ASH_WINDOW w;
   String file;
{
   TRACE("ASHload_bitmap 0x%x %s",w,file);
};






/************************************************************************/
/*									*/
/*	ASHbell -- ring da bell 					*/
/*									*/
/************************************************************************/


void
ASHbell()
{
   TRACE("ASHbell");

   PROTECT;
   XBell(DISPLAYOF(ASHinq_top()),75);
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	round_rectangle -- draw a rounded rectangle			*/
/*									*/
/************************************************************************/


static void
round_rectangle(w,fill,lx,by,rx,ty,r)
   ASH_WINDOW w;
   Boolean fill;
   Integer lx,by;
   Integer rx,ty;
   Integer r;
{
   Integer vlx,vby,vrx,vty;
   Integer x0,x1,y0,y1;
   XArc arc[4];
   XRectangle rgn[3];
   XSegment lin[4];
   Integer i,delta;

   DRAW_BEGIN(w);

   if (fill) delta = 0;
   else delta = 1;

   vlx = XMAPVIEW(w,lx);
   vby = YMAPVIEW(w,by);
   vrx = XMAPVIEW(w,rx);
   vty = YMAPVIEW(w,ty);

   if (vlx > vrx) {
      x0 = vlx; vlx = vrx; vrx = x0;
    };
   if (vty > vby) {
      y0 = vty; vty = vby; vby = y0;
    };

   if (vby-vty > r+r) {
      y0 = vty + r;
      y1 = vby - r;
    }
   else {
      y0 = y1 = (vty+vby)/2;
    };

   if (vrx-vlx > r+r) {
      x0 = vlx + r;
      x1 = vrx - r;
    }
   else {
      x0 = x1 = (vlx+vrx)/2;
    };

   if (fill) {
      rgn[0].x = vlx;
      rgn[0].y = y0;
      rgn[0].width = vrx-vlx + RECT_ADD;
      rgn[0].height = y1-y0 + RECT_ADD;

      rgn[1].x = x0;
      rgn[1].y = vty;
      rgn[1].width = (x1-x0)-1 + RECT_ADD;
      rgn[1].height = y0-vty-1 + RECT_ADD;

      rgn[2].x = x0;
      rgn[2].y = y1+1;
      rgn[2].width = (x1-x0)-1 + RECT_ADD;
      rgn[2].height = vby-y1-1 + RECT_ADD;
    }
   else {
      lin[0].x1 = x0+1;
      lin[0].y1 = vty;
      lin[0].x2 = x1-1;
      lin[0].y2 = vty;

      lin[1].x1 = vrx;
      lin[1].y1 = y0+1-delta;
      lin[1].x2 = vrx;
      lin[1].y2 = y1-1;

      lin[2].x1 = x0+1;
      lin[2].y1 = vby;
      lin[2].x2 = x1-1;
      lin[2].y2 = vby;

      lin[3].x1 = vlx;
      lin[3].y1 = y0+1-delta;
      lin[3].x2 = vlx;
      lin[3].y2 = y1-1;
    };

   arc[0].x = vlx;
   arc[0].y = vty;
   arc[0].width = (x0-vlx)*2;
   arc[0].height = (y0-vty)*2;
   arc[0].angle1 = ANGLE(90);
   arc[0].angle2 = ANGLE(90);

   arc[1].x = vlx;
   arc[1].y = y1-(vby-y1)+1-delta;
   arc[1].width = (x0-vlx)*2;
   arc[1].height = (vby-y1)*2;
   arc[1].angle1 = ANGLE(180);
   arc[1].angle2 = ANGLE(90);

   arc[2].x = x1-(vrx-x1);
   arc[2].y = y1-(vby-y1)+1-delta;
   arc[2].width = (vrx-x1)*2;
   arc[2].height = (vby-y1)*2;
   arc[2].angle1 = ANGLE(270);
   arc[2].angle2 = ANGLE(90);

   arc[3].x = x1-(vrx-x1);
   arc[3].y = vty;
   arc[3].width = (vrx-x1)*2;
   arc[3].height = (y0-vty)*2;
   arc[3].angle1 = ANGLE(0);
   arc[3].angle2 = ANGLE(90);

   if (VIEW_DRAW(w)) {
      if (fill) {
	 XFillRectangles(DISPLAYOF(w),w->view_win,w->draw->context,rgn,3);
	 XFillArcs(DISPLAYOF(w),w->view_win,w->draw->context,arc,4);
       }
      else {
	 XDrawSegments(DISPLAYOF(w),w->view_win,w->draw->context,lin,4);
	 XDrawArcs(DISPLAYOF(w),w->view_win,w->draw->context,arc,4);
       };
    };

   if (FULL_DRAW(w)) {
      if (fill) {
	 for (i = 0; i < 3; ++i) {
	    rgn[i].x = XMAPVFULL(w,rgn[i].x);
	    rgn[i].y = YMAPVFULL(w,rgn[i].y);
	  };
       }
      else {
	 for (i = 0; i < 4; ++i) {
	    lin[i].x1 = XMAPVFULL(w,lin[i].x1);
	    lin[i].y1 = YMAPVFULL(w,lin[i].y1);
	    lin[i].x2 = XMAPVFULL(w,lin[i].x2);
	    lin[i].y2 = YMAPVFULL(w,lin[i].y2);
	  };
       };
      for (i = 0; i < 4; ++i) {
	 arc[i].x = XMAPVFULL(w,arc[i].x);
	 arc[i].y = YMAPVFULL(w,arc[i].y);
       };
      if (fill) {
	 XFillRectangles(DISPLAYOF(w),w->full_win,w->draw->full_context,rgn,3);
	 XFillArcs(DISPLAYOF(w),w->full_win,w->draw->full_context,arc,4);
       }
      else {
	 XDrawSegments(DISPLAYOF(w),w->full_win,w->draw->full_context,lin,4);
	 XDrawArcs(DISPLAYOF(w),w->full_win,w->draw->full_context,arc,4);
       };
    };

   DRAW_END(w);
};





/************************************************************************/
/*									*/
/*	draw_circle -- draw a circle/ellipse of some type		*/
/*									*/
/************************************************************************/


static void
draw_circle(w,fill,x,y,rx,ry)
   ASH_WINDOW w;
   Boolean fill;
   Integer x,y;
   Integer rx,ry;
{
   register Integer vx,vy;

   DRAW_BEGIN(w);

   vx = XMAPVIEW(w,x);
   vy = YMAPVIEW(w,y);

   if (VIEW_DRAW(w)) {
      if (fill) {
	 XFillArc(DISPLAYOF(w),w->view_win,w->draw->context,
		     vx-rx,vy-ry,rx+rx,ry+ry,ANGLE(0),ANGLE(360));
       }
      else {
	 XDrawArc(DISPLAYOF(w),w->view_win,w->draw->context,
		     vx-rx,vy-ry,rx+rx,ry+ry,ANGLE(0),ANGLE(360));
       };
    };

   if (FULL_DRAW(w)) {
      vx = XMAPVFULL(w,vx);
      vy = YMAPVFULL(w,vy);
      if (fill) {
	 XFillArc(DISPLAYOF(w),w->full_win,w->draw->full_context,
		     vx-rx,vy-ry,rx+rx,ry+ry,ANGLE(0),ANGLE(360));
       }
      else {
	 XDrawArc(DISPLAYOF(w),w->full_win,w->draw->full_context,
		     vx-rx,vy-ry,rx+rx,ry+ry,ANGLE(0),ANGLE(360));
       };
    };

   DRAW_END(w);
};





/************************************************************************/
/*									*/
/*	draw_pointset -- draw a set of points in various ways		*/
/*									*/
/************************************************************************/


static void
draw_pointset(w,mode,numpts,x,y)
   ASH_WINDOW w;
   POINTSET_TYPE mode;
   Integer numpts;
   register Integer *x;
   register Integer *y;
{
   XPoint pt[MAXPTS];
   XPoint *npt;
   register Integer i;

   if (numpts <= 0) return;

   if (numpts > MAXPTS) {
      npt = (XPoint *) alloca(sizeof(XPoint)*(numpts+1));
    }
   else {
      npt = pt;
    };

   DRAW_BEGIN(w);

   for (i = 0; i < numpts; ++i) {
      npt[i].x = XMAPVIEW(w,x[i]);
      npt[i].y = YMAPVIEW(w,y[i]);
    };

   if (VIEW_DRAW(w)) {
      do_point_draw(DISPLAYOF(w),w->view_win,w->draw->context,npt,numpts,mode);
    };

   if (FULL_DRAW(w)) {
      for (i = 0; i < numpts; ++i) {
	 npt[i].x = XMAPVFULL(w,npt[i].x);
	 npt[i].y = YMAPVFULL(w,npt[i].y);
       };
      do_point_draw(DISPLAYOF(w),w->full_win,w->draw->full_context,npt,numpts,mode);
    };

   DRAW_END(w);
};




static void
do_point_draw(d,w,c,npt,numpts,mode)
   Display * d;
   Drawable w;
   GC c;
   XPoint *npt;
   Integer numpts;
   POINTSET_TYPE mode;
{
   switch (mode) {
      case POINTSET_POINT :
	 XDrawPoints(d,w,c,npt,numpts,CoordModeOrigin);
	 break;

      case POINTSET_LINE :
	 XDrawLines(d,w,c,npt,numpts,CoordModeOrigin);
	 break;

      case POINTSET_POLYGON :
	 XFillPolygon(d,w,c,npt,numpts,Nonconvex,CoordModeOrigin);
	 break;

      case POINTSET_CONVEX :
	 XFillPolygon(d,w,c,npt,numpts,Convex,CoordModeOrigin);
	 break;

      case POINTSET_GENERAL :
	 XFillPolygon(d,w,c,npt,numpts,Complex,CoordModeOrigin);
	 break;
    };
};





/************************************************************************/
/*									*/
/*	draw_arc -- draw an arc of some type				*/
/*									*/
/************************************************************************/


static void
draw_arc(typ,w,cx,cy,rx,ry,a1,a2)
   POINTSET_TYPE typ;
   ASH_WINDOW w;
   Integer cx,cy;
   Integer rx,ry;
   Integer a1,a2;
{
   register Integer vx,vy;

   DRAW_BEGIN(w);

   vx = XMAPVIEW(w,cx);
   vy = YMAPVIEW(w,cy);
   vx -= rx;
   vy -= ry;

   if (VIEW_DRAW(w)) {
      do_arc_draw(DISPLAYOF(w),w->view_win,w->draw->context,
		     vx,vy,rx+rx,ry+ry,a1,a2,typ);
    };

   if (FULL_DRAW(w)) {
      vx = XMAPVFULL(w,vx);
      vy = YMAPVFULL(w,vy);
      do_arc_draw(DISPLAYOF(w),w->full_win,w->draw->full_context,
		     vx,vy,rx+rx,ry+ry,a1,a2,typ);
    };

   DRAW_END(w);
};





static void
do_arc_draw(d,w,c,x,y,wd,ht,a1,a2,mode)
   Display * d;
   Drawable w;
   GC c;
   Integer x,y,wd,ht,a1,a2;
   POINTSET_TYPE mode;
{
   Integer aty;
   XGCValues vals;
   Integer mask;

   if (mode == ARCSET_LINE) {
      XDrawArc(d,w,c,x,y,wd,ht,a1,a2);
    }
   else {
      aty = (mode == ARCSET_CHORD ? ArcChord : ArcPieSlice);
      if (c->values.arc_mode != aty) {
	 vals.arc_mode = aty;
	 mask = GCArcMode;
	 XChangeGC(d,c,mask,&vals);
       };
      XFillArc(d,w,c,x,y,wd,ht,a1,a2);
    };
};






/************************************************************************/
/*									*/
/*	between_display_blt -- do a blt between displays		*/
/*									*/
/************************************************************************/


static void
between_display_blt(w,d,lx,ty,wd,ht,dlx,dty)
   ASH_WINDOW w;
   Drawable d;
   Integer lx,ty;
   Integer wd,ht;
   Integer dlx,dty;
{
   XImage * im;
   Integer pm;

   pm = ASHinq_plane_mask(w->draw->source);

   DRAW_BEGIN(w->draw->source);

   im = XGetImage(DISPLAYOF(w->draw->source),d,lx,ty,wd,ht,pm,XYPixmap);

   DRAW_END(w->draw->source);

   DRAW_BEGIN(w);

   if (VIEW_DRAW(w)) {
      XPutImage(DISPLAYOF(w),w->view_win,w->draw->context,im,0,0,dlx,dty,wd,ht);
    };

   if (FULL_DRAW(w)) {
      dlx = XMAPVFULL(w,dlx);
      dty = YMAPVFULL(w,dty);
      XPutImage(DISPLAYOF(w),w->full_win,w->draw->full_context,im,0,0,dlx,dty,wd,ht);
    };

   DRAW_END(w);

   free(im);
};





/********************************************************************************/
/*										*/
/*	do_save -- save window in file						*/
/*										*/
/********************************************************************************/


static void
do_save(dw,file)
   ASH_WINDOW dw;
   String file;
{
   String sh;
   Integer pid;
   Character txt[1024],ftxt[256];
   ASH_WINDOW w;

   w = dw;
   if (dw == NULL) w = ASHinq_top();

   PROTECT;
   XSync(DISPLAYOF(w),0);
   UNPROTECT;

   if (file == NULL) {
      sprintf(ftxt,"xwddump.XXXXXX",w->x_win);
      mktemp(ftxt);
      file = ftxt;
    };

   if (dw == NULL)
      sprintf(txt,"xwd -display %s -root -nobdrs -out %s",w->display->name,file);
   else
      sprintf(txt,"xwd -display %s -id 0x%x -out %s",w->display->name,w->x_win,file);

   sh = getenv("SHELL");
   if (sh == NULL) sh = "/bin/csh";

   if ((pid = vfork()) == 0) {
      execl(sh,"sh","-c",txt,0);
      _exit(127);
    };

   while (getpgrp(pid) > 0) {
      CMPXselect(-2);
    };
};





/* end of ashdraw.c */
