/************************************************************************/
/*									*/
/*		ashcursor.c						*/
/*									*/
/*	Entries for managing the cursor from ASH			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/

#include "ash_local.h"




/************************************************************************/
/*									*/
/*	Constant Definitions						*/
/*									*/
/************************************************************************/


#define EMPTY_CURSOR	1
#define DFLT_CURSOR	0

#define CURSOR_UNDEF	(-1)
#define CURSOR_PAT	(-2)




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	cursor_ctr;
static	Integer 	cursor_standard[MAX_CURSORS];




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		std_cursor();
static	void		bitmap_cursor();
static	void		pattern_cursor();
static	void		char_cursor();
static	void		colormap();
static	Pixmap		cursor_pixmap();





/************************************************************************/
/*									*/
/*	ASH_cursor_init -- initialize this module			*/
/*	ASH_cursor_disp_init -- initialize display for cursors		*/
/*									*/
/************************************************************************/


void
ASH_cursor_init()
{
   register Integer i;

   cursor_ctr = 2;
   for (i = 0; i < MAX_CURSORS; ++i) cursor_standard[i] = CURSOR_UNDEF;
};





void
ASH_cursor_disp_init(d,ffg)
   DISPLAY d;
   Boolean ffg;
{
   register Integer i;
   DISPLAY d1;
   Short pat[16];

   for (i = 0; i < MAX_CURSORS; ++i) {
      d->data->cursor_table[i] = NULL;
    };

   std_cursor(d,0,DFLT_CURSOR);

   for (i = 0; i < 16; ++i) pat[i] = 0;
   pattern_cursor(d,EMPTY_CURSOR,pat,pat,0,0,0,0,16,16);

   if (!ffg) {
      d1 = ASH_inq_top_display();
      for (i = 2; i < MAX_CURSORS; ++i) {
	 if (d1->data->cursor_table[i] != NULL) {
	    if (cursor_standard[i] >= 0)
	       std_cursor(d,i,cursor_standard[i]);
	    else
	       d->data->cursor_table[i] = d->data->cursor_table[0];
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	ASHwindow_cursor -- set the cursor for a window 		*/
/*									*/
/************************************************************************/


ASH_CURSOR
ASHwindow_cursor(w,cur)
   ASH_WINDOW w;
   ASH_CURSOR cur;
{
   ASH_CURSOR old;
   Cursor c;

   TRACE("ASHwindow_cursor 0x%x",cur);

   CHECKWIN(w,ASHwindow_cursor);

   PROTECT;

   old = w->cursor;
   w->cursor = cur;

   if (w->cursoron) {
      if (cur >= 0) {
	 c = w->display->data->cursor_table[cur];
	 XDefineCursor(DISPLAYOF(w),w->x_win,c);
       }
      else {
	 XUndefineCursor(DISPLAYOF(w),w->x_win);
       };
    };

   UNPROTECT;

   return old;
};





/************************************************************************/
/*									*/
/*	ASHcursor_move -- move cursor to given screen location		*/
/*									*/
/************************************************************************/


void
ASHcursor_move(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   CHECKWIN(w,ASHcursor_move);

   if (!w->inited) ASH_setup_window(w);

   PROTECT;

   x = XMAPVIEW(w,x);
   y = YMAPVIEW(w,y);

   XWarpPointer(DISPLAYOF(w),NULL,w->view_win,0,0,0,0,x,y);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHcursor -- enable or disable cursor				*/
/*									*/
/************************************************************************/


int
ASHcursor_enable(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   register Boolean ofg;
   register Cursor c;

   ENTER("ASHcursor 0x%x %d",w,fg);

   if (w == NULL || w->safety != SAFETY || w->removed) return FALSE;

   ofg = w->cursoron;
   fg = (fg != 0);
   if (ofg == fg) return fg;

   PROTECT;

   w->cursoron = fg;

   if (fg) {
      if (w->cursor >= 0) {
	 c = w->display->data->cursor_table[w->cursor];
	 XDefineCursor(DISPLAYOF(w),w->x_win,c);
       }
      else {
	 XUndefineCursor(DISPLAYOF(w),w->x_win);
       };
    }
   else {
      c = w->display->data->cursor_table[EMPTY_CURSOR];
      XDefineCursor(DISPLAYOF(w),w->x_win,c);
    };

   UNPROTECT;

   return ofg;
};






/************************************************************************/
/*									*/
/*	ASHcursor_standard -- return id of a standard cursor		*/
/*	std_cursor -- define a standard cursor for a display		*/
/*									*/
/************************************************************************/


ASH_CURSOR
ASHcursor_standard(id)
   ASH_CURSOR_ID id;
{
   register Integer i;

   ENTER("ASHcursor_standard %d",id);

   PROTECT;
   for (i = 0; i < MAX_CURSORS; ++i) {
      if (cursor_standard[i] == id) break;
    };
   if (i >= MAX_CURSORS && cursor_ctr < MAX_CURSORS) {
      i = cursor_ctr++;
      cursor_standard[i] = id;
    };
   UNPROTECT;

   if (i >= MAX_CURSORS) {
      ERROR("Too many cursors defined");
      return -1;
    };

   ASH_for_all_displays(std_cursor,i,id);

   return (ASH_CURSOR) i;
};





static void
std_cursor(d,i,id)
   DISPLAY d;
   Integer i;
   Integer id;
{
   PROTECT;

   if (d->data->cursor_table[i] == NULL) {
      d->data->cursor_table[i] = XCreateFontCursor(d->display,id);
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHcursor_define_bitmap -- define bitmap cursor 		*/
/*	bitmap_cursor -- define a bitmap cursor for a display		*/
/*									*/
/************************************************************************/


ASH_CURSOR
ASHcursor_define_bitmap(w,wm,x,y,fgd,bgd)
   ASH_WINDOW w,wm;
   Integer x,y;
   Integer fgd,bgd;
{
   register Integer i;

   ENTER("ASHcursor_define_bitmap 0x%x 0x%x %d %d %d %d",w,wm,x,y,fgd,bgd);

   PROTECT;
   i = cursor_ctr++;
   cursor_standard[i] = CURSOR_PAT;
   UNPROTECT;

   if (i >= MAX_CURSORS) {
      ERROR("Too many cursors defined");
      return -1;
    };

   CHECKWIN(w,ASHcursor_define_bitmap);
   if (wm != NULL) CHECKWIN(wm,ASHcursor_define_bitmap);

   ASH_for_all_displays(bitmap_cursor,i,w,wm,fgd,bgd,x,y);

   return i;
};




static void
bitmap_cursor(d,i,w,wm,fgd,bgd,x,y)
   DISPLAY d;
   Integer i;
   ASH_WINDOW w,wm;
   Integer fgd,bgd;
   Integer x,y;
{
   Pixmap pix,pixm;
   XColor fc,bc;

   pix = ASH_window_to_pixmap(w,16,16,d);

   if (wm == NULL) pixm = NULL;
   else {
      pixm = ASH_window_to_pixmap(wm,16,16,d);
    };

   PROTECT;

   colormap(d,fgd,&fc);
   colormap(d,bgd,&bc);

   d->data->cursor_table[i] =
      XCreatePixmapCursor(DISPLAYOF(w),pix,pixm,&fc,&bc,x,y);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHcursor_define_pattern -- define cursor from user patterns	*/
/*	pattern_cursor -- define a patterned cursor for a display	*/
/*									*/
/************************************************************************/


ASH_CURSOR
ASHcursor_define_pattern(pat,mpat,wid,ht,x,y,fgd,bgd)
   Short * pat, * mpat;
   Integer wid,ht;
   Integer x,y;
   Integer fgd,bgd;
{
   register Integer i;

   ENTER("ASHcursor_define_pattern 0x%x 0x%x %d %d %d %d %d %d",
	    pat,mpat,wid,ht,x,y,fgd,bgd);

   PROTECT;
   i = cursor_ctr++;
   cursor_standard[i] = CURSOR_PAT;
   UNPROTECT;

   if (i >= MAX_CURSORS) {
      ERROR("Too many cursors defined");
      return -1;
    };

   ASH_for_all_displays(pattern_cursor,i,pat,mpat,fgd,bgd,x,y,wid,ht);

   return i;
};





static void
pattern_cursor(d,i,pat,mpat,fgd,bgd,x,y,wd,ht)
   DISPLAY d;
   Integer i;
   Short * pat, * mpat;
   Integer fgd,bgd;
   Integer x,y;
   Integer wd,ht;
{
   Pixmap pix,pixm;
   XColor fc,bc;

   pix = cursor_pixmap(d,pat,wd,ht);
   if (mpat != NULL) pixm = cursor_pixmap(d,mpat,wd,ht);
   else pixm = NULL;

   PROTECT;

   colormap(d,fgd,&fc);
   colormap(d,bgd,&bc);

   d->data->cursor_table[i] =
      XCreatePixmapCursor(d->display,pix,pixm,&fc,&bc,x,y);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	ASHcursor_define_char -- define a user cursor from a font	*/
/*	char_cursor -- define a character cursor for a display		*/
/*									*/
/************************************************************************/


ASH_CURSOR
ASHcursor_define_char(ch,font,mch,mfont,fgd,bgd)
   Integer   ch;
   Integer font;
   Integer mch;
   Integer mfont;
   Integer fgd;
   Integer bgd;
{
   register Integer i;

   ENTER("ASHcursor_define_char 0x%x 0x%x 0x%x 0x%x %d %d",
	    ch,font,mch,mfont,fgd,bgd);

   PROTECT;
   i = cursor_ctr++;
   cursor_standard[i] = CURSOR_PAT;
   UNPROTECT;

   if (i >= MAX_CURSORS) {
      ERROR("Too many cursors defined");
      return -1;
    };

   ASH_for_all_displays(char_cursor,i,ch,font,mch,mfont,fgd,bgd);

   return i;
};





static void
char_cursor(d,i,ch,font,mch,mfont,fgd,bgd)
   DISPLAY d;
   Integer i;
   Integer ch;
   Integer font;
   Integer mch;
   Integer mfont;
   Integer fgd,bgd;
{
   XColor fc,bc;

   PROTECT;

   colormap(d,fgd,&fc);
   colormap(d,bgd,&bc);

   d->data->cursor_table[i] =
      XCreateGlyphCursor(d->display,ASH_real_font(d,font),
			    ASH_real_font(d,mfont),ch,mch,&fc,&bc);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	colormap -- map pixel value to XColor				*/
/*									*/
/************************************************************************/


static void
colormap(d,pix,xc)
   DISPLAY d;
   Integer pix;
   XColor * xc;
{
   XWindowAttributes attrs;

   XGetWindowAttributes(d->display,RootWindow(d->display,d->screen),&attrs);
   if (attrs.colormap == NULL) {
      attrs.colormap = DefaultColormap(d->display,d->screen);
      XSetWindowColormap(d->display,RootWindow(d->display,d->screen),
			    attrs.colormap);
    };

   xc->pixel = pix;

   XQueryColor(d->display,attrs.colormap,xc);
};





/************************************************************************/
/*									*/
/*	cursor_pixmap -- make 1-plane pixmap from array of Shorts	*/
/*									*/
/************************************************************************/


static Pixmap
cursor_pixmap(d,pat,wd,ht)
   DISPLAY d;
   unsigned short * pat;
{
   XImage * im;
   Pixmap p;
   GC gc;
   XGCValues vals;

   PROTECT;

   im = XCreateImage(d->display,
			d->default_visual,
			1,
			XYPixmap,
			0,
			pat,
			wd, ht, 16, 0);

   im->byte_order = BYTE_ORDER;
   im->bitmap_bit_order = BIT_ORDER;

   p = XCreatePixmap(d->display,RootWindow(d->display,d->screen),wd,ht,1);
   gc = XCreateGC(d->display,p,0,&vals);

   XPutImage(d->display,p,gc,im,0,0,0,0,wd,ht);
   XFreeGC(d->display,gc);

   XFree(im);

   UNPROTECT;

   return p;
};





/* end of ashcursor.c */
