/************************************************************************/
/*									*/
/*		ashbpio.c						*/
/*									*/
/*	X-11 version of BPIO package					*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"

#include <sys/types.h>
#include <sys/time.h>
#include <sgtty.h>




/********************************************************************************/
/*										*/
/*	Local types								*/
/*										*/
/********************************************************************************/


typedef struct _BPIO_THREAD_QUEUE *  BPIO_THREAD_QUEUE;

typedef struct _BPIO_THREAD_QUEUE {
   THREAD thread;
   ASH_WINDOW window;
   XEvent event;
   BPIO_THREAD_QUEUE next;
   BPIO_THREAD_QUEUE last;
} BPIO_THREAD_QUEUE_INFO;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Boolean 	move_when_down = FALSE;
static	Integer 	button_state = 0;
static	Integer 	last_x = 0;
static	Integer 	last_y = 0;
static	BPIO_TIME	last_time = 0;
static	ASH_WINDOW	last_win = NULL;
static	BPIO_THREAD_QUEUE    thread_queue = NULL;



/************************************************************************/
/*									*/
/*	Macro definitions						*/
/*									*/
/************************************************************************/


#define STILL_COUNT	5
#define MOVE_STILL_COUNT	25

#define set_time(t) last_time = t




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/



static unsigned char funct_map[] = {
/* 0x00 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x10 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x20 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x30 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x40 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x50 */     40,41,42,43,44,45,46,47,48, 0, 0, 0, 0, 0, 0, 0,
/* 0x60 */     49,50,51,52, 0,53,54,55,56,57,58,59, 0, 0, 0, 0,
/* 0x70 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x80 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0x90 */	0,60,61,62,63, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0xa0 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0xb0 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2,
/* 0xc0 */	3, 4, 5, 6, 7, 8, 9,10,15,16,17,18,19,20,21,22,
/* 0xd0 */     23,24,25,26,27,28,29,30,31,32,33,34,35,36,37,38,
/* 0xe0 */     39, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
/* 0xf0 */	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
};




/************************************************************************/
/*									*/
/*	BPIO_init -- module initialization				*/
/*									*/
/************************************************************************/


void
BPIO_init()
{
   button_state = 0;
   last_x = 0;
   last_y = 0;
   last_time = 0;
   last_win = NULL;
   thread_queue = NULL;
   move_when_down = FALSE;
};






/************************************************************************/
/*									*/
/*	BPIOnext_event -- get next SUN event				*/
/*									*/
/************************************************************************/


int
BPIOnext_event(xd,e)
   Display * xd;
   BPIO_EVENT e;
{
   XEvent event;
   static Integer still_count = 0;
   Integer i,ct;
   Window xwin,chld;
   ASH_WINDOW w;
   BPIO_THREAD_QUEUE tq,ltq;
   THREAD th;
   Boolean thfg;

retry:
   PROTECT;


   if (ASH_inq_batched()) tq = NULL;
   else {
      for (tq = thread_queue; tq != NULL; tq = tq->next) {
	 if (tq->event.xany.display == xd && tq->window->threadct == 0) break;
       };
    };

   ct = XPending(xd);
   if (tq == NULL && ct == 0) {
      UNPROTECT;
      if (e != NULL && last_win != NULL && still_count++ == STILL_COUNT) {
	 e->BPIO_xpos = last_x;
	 e->BPIO_ypos = last_y;
	 e->BPIO_time = last_time;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = (button_state == 0 ? BPIO_EVENT_STILL :
					     BPIO_EVENT_STILL_DOWN);
	 if (move_when_down && button_state != 0) {
	    still_count = -MOVE_STILL_COUNT;
	    e->BPIO_data = BPIO_EVENT_MOVE_DOWN;
	  };
	 e->BPIO_window = last_win;
	 return TRUE;
       };
      if (still_count > 0 && (still_count % 1000) == 0) {
	 XSync(xd,0);		  /* check if we are still alive  */
       };
      return FALSE;
    };

   if (tq == NULL) {
      XNextEvent(xd,&event);
      if (ASH__debugf != NULL) {
	 fprintf(ASH__debugf,"ASH: event %d 0x%x\n",event.type,event.xany.window);
       };
    }
   else {
      if (tq->last != NULL) tq->last->next = tq->next;
      else thread_queue = tq->next;
      if (tq->next != NULL) tq->next->last = tq->last;
      event = tq->event;
      free(tq);
    };

   thfg = ASH_event_check_save(&event,&th,&w);

   if (thfg) {
      ltq = NULL;
      for (tq = thread_queue; tq != NULL; tq = tq->next) ltq = tq;
      tq = PALLOC(BPIO_THREAD_QUEUE_INFO);
      tq->event = event;
      tq->thread = th;
      tq->window = w;
      tq->next = NULL;
      tq->last = ltq;
      if (ltq == NULL) thread_queue = tq;
      else ltq->next = tq;
      UNPROTECT;
      goto retry;
    };

   UNPROTECT;

   w = ASHevent_process(&event);

   if (w == NULL || e == NULL) goto retry;

   still_count = 0;
   last_win = w;

   switch(event.type) {
      case KeyPress :
       { register XKeyEvent * xe = &event.xkey;
	 Character buf[16];
	 KeySym sym;
	 set_time(xe->time);
	 last_x = xe->x;
	 last_y = xe->y;
	 xwin = xe->window;
	 PROTECT;
	 ct = XLookupString(xe,buf,16,NULL,NULL);
	 UNPROTECT;
	 if (ct == 1) {
	    i = ((*buf)&0x7f);
	    e->BPIO_data = i;
	    if (xe->state & Mod1Mask) {
	       e->BPIO_class = BPIO_CLASS_META_ASCII;
	     }
	    else {
	       e->BPIO_class = BPIO_CLASS_ASCII;
	     };
	  }
	 else {
	    i = xe->keycode;
	    sym = XKeycodeToKeysym(xd,i,0);
	    if ((sym & 0xff00) != 0xff00) goto retry;
	    i = sym & 0xff;
	    i = funct_map[i];
	    if (i == 0) goto retry;
	    e->BPIO_data = i;
	    if (xe->state & ShiftMask)
	       e->BPIO_class = BPIO_CLASS_SHIFT_FKEY;
	    else if (xe->state & ControlMask)
	       e->BPIO_class = BPIO_CLASS_CTRL_FKEY;
	    else if (xe->state & Mod1Mask)
	       e->BPIO_class = BPIO_CLASS_META_FKEY;
	    else
	       e->BPIO_class = BPIO_CLASS_FKEY;
	  };
	 break;
       };

      case KeyRelease :
       { register XKeyEvent * xe = &event.xkey;
	 KeySym sym;
	 set_time(xe->time);
	 last_x = xe->x;
	 last_y = xe->y;
	 xwin = xe->window;
	 i = xe->keycode;
	 sym = XKeycodeToKeysym(xd,i,0);
	 if ((sym & 0xff00) != 0xff00) goto retry;
	 i = sym & 0xff;
	 i = funct_map[i];
	 if (i == 0) goto retry;
	 e->BPIO_data = i;
	 e->BPIO_class = BPIO_CLASS_UP_FKEY;
	 break;
       };

      case ButtonPress :
       { register XButtonEvent * xe = &event.xbutton;
	 set_time(xe->time);
	 last_x = xe->x;
	 last_y = xe->y;
	 xwin = xe->window;
	 if ((xe->state & ShiftMask) != 0)
	    e->BPIO_class = BPIO_CLASS_SHIFT_MOUSE;
	 else if ((xe->state & ControlMask) != 0)
	    e->BPIO_class = BPIO_CLASS_CTRL_MOUSE;
	 else if ((xe->state & Mod1Mask) != 0)
	    e->BPIO_class = BPIO_CLASS_META_MOUSE;
	 else
	    e->BPIO_class = BPIO_CLASS_MOUSE;
	 i = xe->button-1;
	 button_state |= (1<<i);
	 e->BPIO_data = BPIO_MOUSE_DATA(button_state,i);
	 break;
       };

      case ButtonRelease :
       { register XButtonEvent * xe = &event.xbutton;
	 set_time(xe->time);
	 last_x = xe->x;
	 last_y = xe->y;
	 xwin = xe->window;
	 e->BPIO_class = BPIO_CLASS_UP_MOUSE;
	 i = xe->button-1;
	 button_state &= ~(1<<i);
	 e->BPIO_data = BPIO_MOUSE_DATA(button_state,i);
	 break;
       };

      case MotionNotify :
       { register XMotionEvent * xe = &event.xmotion;
	 set_time(xe->time);
	 last_x = xe->x;
	 last_y = xe->y;
	 xwin = xe->window;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = (button_state == 0 ? BPIO_EVENT_MOVE :
					     BPIO_EVENT_MOVE_DOWN);
	 break;
       };

      case Expose :
      case GraphicsExpose :
       { xwin = NULL;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = BPIO_EVENT_REDRAW;
	 break;
       };

      case MapNotify :
       { xwin = NULL;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = BPIO_EVENT_VISIBLE;
	 break;
       };

      case UnmapNotify :
       { xwin = NULL;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = BPIO_EVENT_INVISIBLE;
	 break;
       };

      case ConfigureNotify :
       { xwin = NULL;
	 e->BPIO_class = BPIO_CLASS_EVENT;
	 e->BPIO_data = BPIO_EVENT_RESIZE;
	 break;
       };

      default :
	 goto retry;
    };

   if (xwin != NULL) {
      if (last_win->x_win != xwin) {
	 XTranslateCoordinates(DISPLAYOF(last_win),xwin,last_win->x_win,
				  last_x,last_y,
				  &last_x,&last_y,&chld);
       };
      last_x = XUNMAPXVIEW(last_win,last_x);
      last_y = YUNMAPXVIEW(last_win,last_y);
    };

   e->BPIO_time = last_time;
   e->BPIO_xpos = last_x;
   e->BPIO_ypos = last_y;
   e->BPIO_window = last_win;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	BPIOwait -- wait for desired period of time			*/
/*									*/
/************************************************************************/


void
BPIOwait(time)
   int time;
{
   struct timeval t;

   t.tv_sec = time>>20;
   t.tv_usec = time&0xfffff;
   if (t.tv_usec >= 1000000) {
      ++t.tv_sec;
      t.tv_usec -= 1000000;
    };

   select(0,0,0,0,&t);
};





/************************************************************************/
/*									*/
/*	BPIOmove_when_down -- set move when down flag			*/
/*									*/
/************************************************************************/


int
BPIOmove_when_down(fg)
   Boolean fg;
{
   Boolean ofg;

   ofg = move_when_down;
   move_when_down = fg;

   return ofg;
};






/************************************************************************/
/*									*/
/*	BPIO_future_events -- check future events			*/
/*									*/
/************************************************************************/


void
BPIO_future_events(d,fct,in)
   Display * d;
   Function_Ptr fct;
   Universal in;
{
   XEvent ev;
   BPIO_THREAD_QUEUE tq;
   Boolean fg;

   PROTECT;

   for (tq = thread_queue; tq != NULL; tq = tq->next) {
      if (tq->event.xany.display == d) (*fct)(d,&tq->event,in);
    };

   fg = XCheckIfEvent(d,&ev,fct,in);

   if (fg) abort();

   UNPROTECT;
};





/* end of bpiouvax.c */


