/************************************************************************/
/*									*/
/*		ashanim.c						*/
/*									*/
/*	Animation front end for multi-buffering 			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "ash_local.h"




/************************************************************************/
/*									*/
/*	Definitions							*/
/*									*/
/************************************************************************/


#define MAX_FRAME	2
#define MAX_COLOR	8

#define LOW_PUBLIC	128
#define HIGH_PUBLIC	192
#define SWAP_SCALE	1




/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _ANIMATE * ANIMATE;

typedef enum _ANIM_STYLE {
   ANIM_STYLE_COLORTABLE,
   ANIM_STYLE_OFFSCREEN,
   ANIM_STYLE_OVERLAY,
   ANIM_STYLE_OVERLAY_COPY,
} ANIM_STYLE;



typedef struct _ANIMATE {
   ANIMATE next;
   ASH_WINDOW window;
   Integer frame;
   Integer color;
   Integer backg;
   Boolean overlay;
   Boolean underlay_used;
   ASH_COLOR_TABLE orig_table;
   Integer orig_color;
   Integer orig_mask;
   Integer orig_backg;
   Integer orig_textc;
   Integer orig_textb;
   Integer ncolor;
   Integer nframe;
   ANIM_STYLE style;
   ASH_WINDOW * offwins;
   ASH_WINDOW draw_win;
   Boolean clear;
} __ANIMATE;




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	ASH_COLOR_TABLE color_tables[MAX_FRAME];
static	Integer 	plane_masks[MAX_FRAME];
static	Integer 	colors[MAX_FRAME][MAX_COLOR];
static	Integer 	common_planes;

static	ANIMATE 	all_animates;

static	ASH_COLOR_TABLE overlay_map;




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String		base_color_names[MAX_COLOR] = {
   "white",
   "black",
   "red",
   "orange",
   "yellow",
   "green",
   "blue",
   "maroon",
};


static	ASH_COLOR	base_color_values[MAX_COLOR];






/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		anim_initialize();
static	ANIMATE 	find_animation();





/************************************************************************/
/*									*/
/*	ASH_anim_init -- module initialization				*/
/*									*/
/************************************************************************/


void
ASH_anim_init()
{
   Integer i;

   for (i = 0; i < MAX_FRAME; ++i) {
      color_tables[i] = NULL;
    };

   for (i = 0; i < MAX_COLOR; ++i) base_color_values[i] = -1;

   all_animates = NULL;
};





/************************************************************************/
/*									*/
/*	ASHanim_setup -- start an animation in a window 		*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHanim_setup(w,ncol,nframe,clear)
   ASH_WINDOW w;
   Integer ncol;
   Integer nframe;
   Boolean clear;
{
   ANIMATE an;
   Integer fl,lx,by,rx,ty;
   String s;
   ANIM_STYLE sty;
   Integer i;

   TRACE("ASHanim_setup 0x%x %d %d",w,ncol,nframe);

   s = ASHinq_window_resource(w,"animstyle");
   if (s == NULL) s = getenv("ANIM_STYLE");
   if (s != NULL && STREQL(s,"OVERLAY") || (s == NULL && nframe < 0)) {
      if (nframe <= 0) {
	 nframe = 1;
	 sty = ANIM_STYLE_OVERLAY;
       }
      else sty = ANIM_STYLE_OVERLAY_COPY;
    }
   else if (s != NULL && STREQL(s,"OVERLAY_COPY")) {
      if (nframe <= 0) nframe = 1;
      sty = ANIM_STYLE_OVERLAY_COPY;
    }
   else if ((s == NULL || STRNEQ(s,"OFFSCREEN")) && ncol != 0 && nframe != 0) {
      if (ncol != MAX_COLOR || nframe != MAX_FRAME) {
	 ERROR("Can only animate with 8 colors and 2 frames");
	 ncol = MAX_COLOR;
	 nframe = MAX_FRAME;
       };
      sty = ANIM_STYLE_COLORTABLE;
    }
   else {
      if (nframe <= 0) nframe = 1;
      sty = ANIM_STYLE_OFFSCREEN;
    };

   anim_initialize(w,sty);

   an = (ANIMATE) calloc(sizeof(__ANIMATE),1);

   an->window = w;
   an->frame = 0;
   an->orig_table = w->colormap;
   an->orig_color = ASHinq_color(w);
   an->orig_mask = ASHinq_plane_mask(w);
   an->orig_backg = ASHinq_background_color(w);
   an->orig_textc = ASHinq_text_color(w);
   an->orig_textb = ASHinq_text_background_color(w);
   an->ncolor = ncol;
   an->nframe = nframe;
   an->clear = clear;
   an->style = sty;
   an->offwins = NULL;
   an->overlay = TRUE;
   an->underlay_used = TRUE;

   PROTECT;
   an->next = all_animates;
   all_animates = an;
   UNPROTECT;

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (sty != ANIM_STYLE_COLORTABLE) {
      an->offwins = (ASH_WINDOW *) calloc(nframe,sizeof(ASH_WINDOW));
      for (i = 0; i < nframe; ++i) {
	 an->offwins[i] = ASHcreate(w,lx,by,lx,by,rx,ty,0,
				       ASH_WINDOW_INVISIBLE|
				       ASH_WINDOW_OFF_SCREEN);
	 ASHclear_box(an->offwins[i],lx,by,rx,ty);
       };
     };

   if (sty != ANIM_STYLE_OFFSCREEN) {
      fl = ASHfill(w,1);
      ASHcolor(w,LOW_PUBLIC);
      ASHrectangle(w,lx,by,rx,ty);
      ASHcolor(w,an->orig_color);
      ASHfill(w,fl);
    };

   return ASHanim_next_frame(w);
};





/************************************************************************/
/*									*/
/*	ASHanim_done -- finish animation in a window			*/
/*									*/
/************************************************************************/


void
ASHanim_done(w)
   ASH_WINDOW w;
{
   ANIMATE an,an1;

   an = find_animation(w);
   if (an == NULL) return;

   ASHcolor(w,an->orig_color);
   ASHplane_mask(w,an->orig_mask);
   ASHbackground_color(w,an->orig_backg);
   ASHtext_color(w,an->orig_textc);
   ASHtext_plane_mask(w,an->orig_mask);
   ASHtext_background_color(w,an->orig_textb);
   ASHset_color_table(w,an->orig_table);

   PROTECT;
   if (all_animates == an) {
      all_animates = an->next;
    }
   else {
      for (an1 = all_animates; an1->next != an; an1 = an1->next);
      an1->next = an->next;
    };
   UNPROTECT;

   free(an);
};






/************************************************************************/
/*									*/
/*	ASHanim_next_frame -- move to next frame			*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHanim_next_frame(w)
   ASH_WINDOW w;
{
   ANIMATE an;
   Integer fr,fr0;
   Integer fl,cr,lx,by,rx,ty;
   ASH_WINDOW ow;
   Integer pmsk,tpmsk;

   an = find_animation(w);
   w = an->window;

   fr0 = an->frame % an->nframe;
   ++an->frame;
   fr = an->frame % an->nframe;

   if (an->style != ANIM_STYLE_COLORTABLE) {
      ASHsource(w,an->offwins[fr0]);
      ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      cr = ASHcombination_rule(w,3);
      if (an->style == ANIM_STYLE_OFFSCREEN) pmsk = ~0;
      else if (an->underlay_used) pmsk = plane_masks[0]|plane_masks[1];
      else pmsk = plane_masks[1];
      tpmsk = ASHtext_plane_mask(w,pmsk);
      pmsk = ASHplane_mask(w,pmsk);
      ASHblt(w,lx,by,rx,ty,lx,by);
      ASHcombination_rule(w,cr);
      ASHplane_mask(w,pmsk);
      ASHtext_plane_mask(w,tpmsk);
      if (an->clear) ASHclear_box(an->offwins[fr],lx,by,rx,ty);
      ow = an->offwins[fr];
      an->overlay = TRUE;
      an->underlay_used = FALSE;
      if (an->frame == 1 && an->style != ANIM_STYLE_OFFSCREEN) {
	 ASHset_color_table(w,overlay_map);
       };
    }
   else {
      ASHset_color_table(w,color_tables[fr0]);
      ASHplane_mask(w,plane_masks[fr]|common_planes);
      ASHtext_plane_mask(w,plane_masks[fr]|common_planes);
      ow = w;

      if (an->clear) {
	 fl = ASHfill(w,1);
	 ASHanim_color(w,-1);
	 ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
	 ASHrectangle(w,lx,by,rx,ty);
	 ASHfill(w,fl);
       };
    };

   an->draw_win = ow;

   ASHanim_color(w,-2);
   ASHanim_background_color(w,-1);
   ow = ASHanim_overlay(w,TRUE);

   return ow;
};





/************************************************************************/
/*									*/
/*	ASHanim_color -- set color for animation			*/
/*	ASHanim_background_color -- set background color		*/
/*									*/
/************************************************************************/


int
ASHanim_color(w,c)
   ASH_WINDOW w;
   int c;
{
   Integer oldc;
   Integer fr;
   ANIMATE an;

   an = find_animation(w);

   fr = an->frame % an->nframe;
   oldc = an->color;
   an->color = c;

   if (an->style == ANIM_STYLE_OFFSCREEN) {
      if (c < 0) c = base_color_values[-c-1];
      ASHcolor(an->offwins[fr],c);
      ASHtext_color(an->offwins[fr],c);
    }
   else if (an->style == ANIM_STYLE_OVERLAY || an->style == ANIM_STYLE_OVERLAY_COPY) {
      if (c < 0) c = -c-1;
      fr = (an->overlay ? 1 : 0);
      ASHcolor(w,colors[fr][c]);
      ASHtext_color(w,colors[fr][c]);
    }
   else {
      if (c < 0) c = -c-1;
      ASHcolor(w,colors[fr][c]);
      ASHtext_color(w,colors[fr][c]);
    };

   return oldc;
};





int
ASHanim_background_color(w,c)
   ASH_WINDOW w;
   int c;
{
   Integer oldc;
   Integer fr;
   ANIMATE an;

   an = find_animation(w);

   fr = an->frame % an->nframe;
   oldc = an->backg;
   an->backg = c;

   if (an->style == ANIM_STYLE_OFFSCREEN) {
      if (c < 0) c = base_color_values[-c-1];
      ASHbackground_color(an->offwins[fr],c);
      ASHtext_background_clear(an->offwins[fr],c);
    }
   else if (an->style == ANIM_STYLE_OVERLAY || an->style == ANIM_STYLE_OVERLAY_COPY) {
      if (c < 0) c = -c-1;
      fr = (an->overlay ? 1 : 0);
      ASHbackground_color(w,colors[fr][c]);
      ASHtext_background_clear(w,colors[fr][c]);
    }
   else {
      if (c < 0) c = -c-1;
      ASHbackground_color(w,colors[fr][c]);
      ASHtext_background_clear(w,colors[fr][c]);
    };

   return oldc;
};





/************************************************************************/
/*									*/
/*	ASHanim_overlay -- set/unset overlay				*/
/*									*/
/************************************************************************/


ASH_WINDOW
ASHanim_overlay(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   ANIMATE an;
   Integer i;
   ASH_WINDOW ow;

   an = find_animation(w);

   an->overlay = fg;
   if (an->style != ANIM_STYLE_OVERLAY && an->style != ANIM_STYLE_OVERLAY_COPY)
      return an->draw_win;

   if (an->style == ANIM_STYLE_OVERLAY && !fg) ow = an->window;
   else {
      if (!fg) an->underlay_used = TRUE;
      ow = an->draw_win;
    };

   i = (fg ? 1 : 0);
   ASHplane_mask(ow,plane_masks[i]);
   ASHtext_plane_mask(ow,plane_masks[i]);
   ASHanim_color(w,an->color);
   ASHanim_background_color(w,an->backg);

   return ow;
};





/************************************************************************/
/*									*/
/*	anim_initialize -- initialize when we must be used		*/
/*									*/
/************************************************************************/


static void
anim_initialize(w,sty)
   ASH_WINDOW w;
   ANIM_STYLE sty;
{
   XColor cols[MAX_COLOR],colx,zcol[LOW_PUBLIC];
   Integer i,j,k;

   if (sty == ANIM_STYLE_OFFSCREEN) {
      if (base_color_values[0] < 0) {
	 for (i = 0; i < MAX_COLOR; ++i) {
	    base_color_values[i] = ASHlookup_color(w,base_color_names[i]);
	  };
	 return;
       };
    };

   if (color_tables[0] != NULL) return;

   for (i = 0; i < MAX_FRAME; ++i) {
      color_tables[i] = ASHnew_color_table(w);
      if (color_tables[i] == NULL) return;
    };

   overlay_map = ASHnew_color_table(w);

   PROTECT;

   for (i = 0; i < MAX_COLOR; ++i) {
      XLookupColor(DISPLAYOF(w),w->colormap,base_color_names[i],&colx,&cols[i]);
      cols[i].pixel = -1;
      cols[i].flags = DoRed|DoGreen|DoBlue;
    };

   for (i = 0; i < LOW_PUBLIC; ++i) {
      zcol[i].pixel = i;
      zcol[i].flags = DoRed|DoGreen|DoBlue;
    };
   XQueryColors(DISPLAYOF(w),w->colormap,zcol,LOW_PUBLIC);

   for (i = 0; i < MAX_FRAME; ++i) {
      XStoreColors(DISPLAYOF(w),color_tables[i],zcol,LOW_PUBLIC);
    };
   XStoreColors(DISPLAYOF(w),overlay_map,zcol,LOW_PUBLIC);

   k = 256-HIGH_PUBLIC;
   for (i = 0; i < k; ++i) {
      zcol[i].pixel = i+HIGH_PUBLIC;
      zcol[i].flags = DoRed|DoGreen|DoBlue;
    };
   XQueryColors(DISPLAYOF(w),w->colormap,zcol,k);
   for (i = 0; i < MAX_FRAME; ++i) {
      XStoreColors(DISPLAYOF(w),color_tables[i],zcol,k);
    };
   XStoreColors(DISPLAYOF(w),overlay_map,zcol,k);

   for (i = 0; i < MAX_COLOR; ++i) {		/* only works if MAX_COLOR=8	*/
      for (j = 0; j < MAX_COLOR; ++j) { 	/* and if MAX_FRAME=2		*/
	 cols[i].pixel = LOW_PUBLIC+(j*MAX_COLOR+i)*SWAP_SCALE;
	 XStoreColor(DISPLAYOF(w),color_tables[0],&cols[i]);
	 cols[j].pixel = LOW_PUBLIC+(j*MAX_COLOR+i)*SWAP_SCALE;
	 XStoreColor(DISPLAYOF(w),color_tables[1],&cols[j]);
       };
    };
   plane_masks[0] = 0x07;
   plane_masks[1] = 0x38;
   common_planes = 0x80;
   for (i = 0; i < MAX_COLOR; ++i) {
      colors[0][i] = i*SWAP_SCALE+LOW_PUBLIC;
      colors[1][i] = MAX_COLOR*i*SWAP_SCALE+LOW_PUBLIC;
    };

   for (i = 0; i < MAX_COLOR; ++i) {
      cols[i].pixel = LOW_PUBLIC+i*SWAP_SCALE;
      XStoreColor(DISPLAYOF(w),overlay_map,&cols[i]);
      for (j = 1; j < MAX_COLOR; ++j) {
	 cols[j].pixel = LOW_PUBLIC + (j*MAX_COLOR+i)*SWAP_SCALE;
	 XStoreColor(DISPLAYOF(w),overlay_map,&cols[j]);
       };
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	find_animation -- find entry for window 			*/
/*									*/
/************************************************************************/


static ANIMATE
find_animation(w)
   ASH_WINDOW w;
{
   ANIMATE an;
   Integer i;

   for (an = all_animates; an != NULL; an = an->next) {
      if (an->window == w) break;
    };

   if (an == NULL) {
      for (an = all_animates; an != NULL; an = an->next) {
	 if (an->offwins != NULL) {
	    for (i = 0; i < an->nframe; ++i) {
	       if (an->offwins[i] == w) break;
	     };
	    if (i < an->nframe) break;
	  };
       };
    };

   return an;
};





/* end of ashanim.c */
