/*	Copyright 1989 Brown University -- Steven P. Reiss		*/

#ifndef ASH_ALREADY_INCLUDED
#define ASH_ALREADY_INCLUDED

/************************************************************************/
/*									*/
/*		ash.h							*/
/*									*/
/*	Include file for programs using ASH				*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	ASH_MODE -- initialization modes				*/
/*									*/
/************************************************************************/


typedef enum	_ASH_MODE {		/* display modes		*/
   ASH_MODE_DISPLAY,			/*    run in borrowed display	*/
   ASH_MODE_WINDOW,			/*    run in a borrowed window	*/
   ASH_MODE_INQUIRE,			/*    ask user for location	*/
   ASH_MODE_NONE,			/*    no explicit root		*/
   ASH_MODE_DEFAULT,			/*    default mode		*/
} ASH_MODE;






/************************************************************************/
/*									*/
/*	ASH_WINDOW -- window structure					*/
/*	ASH_DRAWINFO -- drawing information				*/
/*									*/
/************************************************************************/


typedef struct _ASH_WINDOW *	ASH_WINDOW;
typedef struct _ASH_DRAWINFO *	ASH_DRAWINFO;





/************************************************************************/
/*									*/
/*	Window creation flags						*/
/*									*/
/************************************************************************/


#define ASH_WINDOW_VISIBLE	0x1	/*    window is visible 	*/
#define ASH_WINDOW_INVISIBLE	0x2	/*    window is invisible	*/
#define ASH_WINDOW_FRAME	0x10	/*    window is a frame 	*/
#define ASH_WINDOW_COURTEOUS	0x20	/*    window is courteous	*/
#define ASH_WINDOW_NOSAVE	0x40	/*    window shouldn't be saved */
#define ASH_WINDOW_OFF_SCREEN	0x100	/*    off-screen primary window */
#define ASH_WINDOW_INDEPENDENT	0x200	/*    window is independent	*/
#define ASH_WINDOW_INPUT	0x400	/*    window will be used for input */
#define ASH_WINDOW_NOCLEAR	0x800	/*    don't clear initially     */
#define ASH_WINDOW_NOREDIRECT	0x1000	/*    no redirection		*/

#define ASH_WINDOW__DEPTH_MASK	 0x0f000000
#define ASH_WINDOW__DEPTH_BIT	 0x01000000
#define ASH_WINDOW_DEPTH(d)	(ASH_WINDOW__DEPTH_BIT*(d))

#define ASH_WINDOW_HIT_USE	0	/* for compatablity		*/




/************************************************************************/
/*									*/
/*	ASH_BORDERID -- border styles					*/
/*									*/
/************************************************************************/


typedef int ASH_BORDERID;

#define ASH_BORDER_NONE 	0
#define ASH_BORDER_THIN 	1
#define ASH_BORDER_WIDTH(n)	(n)		/* width < 10		*/




/************************************************************************/
/*									*/
/*	ASH_REFRESH_FUNCTION -- refresh function			*/
/*									*/
/************************************************************************/


typedef int	(*ASH_REFRESH_FUNCTION)();

#define ASH_REFRESH_WINDOW ((ASH_REFRESH_FUNCTION) -1)
#define ASH_REFRESH_CLEAR  ((ASH_REFRESH_FUNCTION) 0 )
#define ASH_REFRESH_IGNORE ((ASH_REFRESH_FUNCTION) -2)





/************************************************************************/
/*									*/
/*	ASH_SIZE_TYPE -- possible size requests 			*/
/*									*/
/************************************************************************/


typedef enum {
   ASH_SIZE_WINDOW,
   ASH_SIZE_VIEW,
   ASH_SIZE_SCREEN,
   ASH_SIZE_BORDER,
} ASH_SIZE_TYPE;





/************************************************************************/
/*									*/
/*	Control function definitions					*/
/*									*/
/************************************************************************/


typedef int	(*ASH_CONTROL_FUNCTION)();

#define ASH_CONTROL_NONE  ((ASH_CONTROL_FUNCTION) 0) /* no current control */


					/* returns from control functions */
#define ASH_CONTROL_REJECT	(-1)	/*    request not handled	*/
#define ASH_CONTROL_ACCEPT(sts) (sts)	/*    return given status	*/
#define ASH_STATUS_OK		0	/*    good status		*/
#define ASH_STATUS_BAD		1	/*    bad status		*/

#define ASH_CONTROL_OK ASH_CONTROL_ACCEPT(ASH_STATUS_OK)
#define ASH_CONTROL_BAD ASH_CONTROL_ACCEPT(ASH_STATUS_BAD)





/************************************************************************/
/*									*/
/*	Colors								*/
/*									*/
/************************************************************************/


typedef int	ASH_COLOR;

#define ASH_COLOR_BLACK 	1
#define ASH_COLOR_WHITE 	0

typedef int	ASH_COLOR_TABLE;






/************************************************************************/
/*									*/
/*	Fill styles							*/
/*									*/
/************************************************************************/


typedef int	ASH_FILL_STYLE;

#define ASH_FILL_HOLLOW 	0
#define ASH_FILL_SOLID		1
#define ASH_FILL_HALFTONE	2
#define ASH_FILL_LIGHT_HALFTONE 3
#define ASH_FILL_DARK_HALFTONE	4
#define ASH_FILL_VERY_LIGHT	5

#define ASH_FILL_STIPPLE_OFFSET 16
#define ASH_FILL_HALFTONE_STIPPLE	(ASH_FILL_HALFTONE+ASH_FILL_STIPPLE_OFFSET)
#define ASH_FILL_LIGHT_HALFTONE_STIPPLE (ASH_FILL_LIGHT_HALFTONE+ASH_FILL_STIPPLE_OFFSET)
#define ASH_FILL_DARK_HALFTONE_STIPPLE	(ASH_FILL_DARK_HALFTONE+ASH_FILL_STIPPLE_OFFSET)





/************************************************************************/
/*									*/
/*	Line styles							*/
/*									*/
/************************************************************************/


typedef int	ASH_LINE_STYLE;

#define ASH_STYLE_SOLID 	0
#define ASH_STYLE_DASHED	1
#define ASH_STYLE_DOTTED	2
#define ASH_STYLE_THICK 	3
#define ASH_STYLE_THICK_DASHED	4
#define ASH_STYLE_THICK_DOTTED	5
#define ASH_STYLE_THICKER	6
#define ASH_STYLE_THICKER_DASHED 7
#define ASH_STYLE_THICKER_DOTTED 8







/************************************************************************/
/*									*/
/*	Fonts								*/
/*									*/
/************************************************************************/


typedef int		ASH_FONT;

#define ASH_DEFAULT_FONT	0
#define ASH_ICON_FONT		1

#define ASH_ICON_FONT_NAME	"ASH_ICONS"





/************************************************************************/
/*									*/
/*	Cursors 							*/
/*									*/
/************************************************************************/


typedef int		ASH_CURSOR;
#define ASH_CURSOR_PARENT	(-1)

typedef int	ASH_CURSOR_ID;		/* standard cursor style	*/

#define ASH_CURSOR_ARROW_UR	44		/*	ur_angle	*/
#define ASH_CURSOR_ARROW_LL	44		/*	ll_angle	*/
#define ASH_CURSOR_ARROW_UL	132		/*	ul_angle	*/
#define ASH_CURSOR_ARROW_LR	132		/*	lr_angle	*/
#define ASH_CURSOR_GLASSES	94		/*	bogosity	*/
#define ASH_CURSOR_BULLSEYE	128		/*	target		*/
#define ASH_CURSOR_XHAIRS	32		/*	tcross		*/
#define ASH_CURSOR_DOT		38		/*	dot		*/
#define ASH_CURSOR_CROSS	34		/*	cross		*/
#define ASH_CURSOR_X		0		/*	center_ptr	*/
#define ASH_CURSOR_CIRCLE	24		/*	circle		*/
#define ASH_CURSOR_BETWEEN_L	152		/*	xterm		*/
#define ASH_CURSOR_HALF_ARROW_L 114		/*	sb_up_arrow	*/
#define ASH_CURSOR_HALF_ARROW_R 106		/*	sb_down_arrow	*/
#define ASH_CURSOR_MOUSE	86		/*	drawing 	*/
#define ASH_CURSOR_SMALL_XHAIRS 30
#define ASH_CURSOR_SMALL_ARROW	68
#define ASH_CURSOR_ARROWS	42
#define ASH_CURSOR_WATCH	150

#define ASH_CURSOR_SMALL_GLASSES	ASH_CURSOR_GLASSES
#define ASH_CURSOR_SMALL_BULLSEYE	ASH_CURSOR_BULLSEYE
#define ASH_CURSOR_SMALL_X		ASH_CURSOR_X
#define ASH_CURSOR_BETWEEN_SMALL_L	ASH_CURSOR_BETWEEN_L






/************************************************************************/
/*									*/
/*	Sense Boxes							*/
/*									*/
/************************************************************************/

typedef struct _SENSEBOX *ASH_SENSE;

typedef int		(*ASH_SENSE_TYPE)();

#define ASH_SENSE_NO_CHANGE	((ASH_SENSE_TYPE) 0)	/* sensitive region does nothing */
#define ASH_SENSE_FLIP		((ASH_SENSE_TYPE) 1)	/* sensitive region flips	 */
#define ASH_SENSE_HALF		((ASH_SENSE_TYPE) 2)	/* half tone flipping		 */
#define ASH_SENSE_BOX		((ASH_SENSE_TYPE) 3)	/* box flipping 		 */





/************************************************************************/
/*									*/
/*	Combination rule						*/
/*									*/
/************************************************************************/


#define ASH_CR_CLEAR		0x0		/* 0 */
#define ASH_CR_AND		0x1		/* src AND dst */
#define ASH_CR_AND_REVERSE	0x2		/* src AND NOT dst */
#define ASH_CR_COPY		0x3		/* src */
#define ASH_CR_AND_INVERTED	0x4		/* NOT src AND dst */
#define ASH_CR_NOOP		0x5		/* dst */
#define ASH_CR_XOR		0x6		/* src XOR dst */
#define ASH_CR_OR		0x7		/* src OR dst */
#define ASH_CR_NOR		0x8		/* NOT src AND NOT dst */
#define ASH_CR_EQUIV		0x9		/* NOT src XOR dst */
#define ASH_CR_INVERT		0xa		/* NOT dst */
#define ASH_CR_OR_REVERSED	0xb		/* src OR NOT dst */
#define ASH_CR_COPY_INVERTED	0xc		/* NOT src */
#define ASH_CR_OR_INVERTED	0xd		/* NOT src OR dst */
#define ASH_CR_NAND		0xe		/* NOT src OR NOT dst */
#define ASH_CR_SET		0xf		/* 1 */





/************************************************************************/
/*									*/
/*	Input Processing of events					*/
/*									*/
/************************************************************************/


#define ASH_INPUT_EVENT_REDRAW	0x1




/************************************************************************/
/*									*/
/*	FONT FAMILY INFORMATION 					*/
/*									*/
/************************************************************************/


typedef int		ASH_FONT_PROPS;

#define ASH_FONT_ROMAN		0x1
#define ASH_FONT_ITALIC 	0x2
#define ASH_FONT_BOLD		0x4
#define ASH_FONT_OUTLINE	0x8
#define ASH_FONT_SHADOW 	0x10

#define ASH_FONT_ANY		0x8000






/************************************************************************/
/*									*/
/*	Locking definitions						*/
/*									*/
/************************************************************************/


typedef int		ASH_LOCK_STATUS;

#define ASH_LOCK_GRANTED	0
#define ASH_LOCK_WAIT		1
#define ASH_LOCK_WAIT_LONG	2
#define ASH_LOCK_FAIL		3





/************************************************************************/
/*									*/
/*	Tracing definitions						*/
/*									*/
/************************************************************************/


#define ASH_TRACE_ERROR 64		/*    abort on errors		*/
#define ASH_TRACE_ON  1 		/*    trace external calls	*/
#define ASH_TRACE_OFF 0 		/*    stop tracing		*/





/************************************************************************/
/*									*/
/*	General functions						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_WINDOW	ASHinit();
extern	int		ASHinq_configuration_depth();
extern	int		ASHinq_configuration_width();
extern	int		ASHinq_configuration_height();
extern	int		ASHinq_configuration_colors();
extern	ASH_COLOR	ASHinq_configuration_black();
extern	ASH_COLOR	ASHinq_configuration_white();
extern	ASH_COLOR	ASHinq_foreground();
extern	ASH_COLOR	ASHinq_background();
#ifdef _XLIB_H_
extern	Display *	ASHinq_display();
#endif
extern	char *		ASHinq_display_name();
extern	ASH_WINDOW	ASHinq_top();
extern	int		ASHinq_read_mask();
extern	ASH_WINDOW	ASHroot_window();
extern	ASH_WINDOW	ASHuse_X_window();
extern	ASH_WINDOW	ASHroot_X_window();
extern	void		ASHtrace();
extern	int		ASHnext_event();
extern	void		ASHstore_buffer_text();
extern	char *		ASHload_buffer_text();

#else

extern "C" {
	ASH_WINDOW	ASHinit(ASH_MODE);
	int		ASHinq_configuration_depth(ASH_WINDOW);
	int		ASHinq_configuration_width(ASH_WINDOW);
	int		ASHinq_configuration_height(ASH_WINDOW);
	int		ASHinq_configuration_colors(ASH_WINDOW);
	ASH_COLOR	ASHinq_configuration_black(ASH_WINDOW);
	ASH_COLOR	ASHinq_configuration_white(ASH_WINDOW);
	ASH_COLOR	ASHinq_foreground(ASH_WINDOW);
	ASH_COLOR	ASHinq_background(ASH_WINDOW);
#ifdef _XLIB_H_
	Display *	ASHinq_display(ASH_WINDOW);
#endif
	char *		ASHinq_display_name(ASH_WINDOW);
	ASH_WINDOW	ASHinq_top();
	int		ASHinq_read_mask(...);
	ASH_WINDOW	ASHroot_window(char *, char *);
	ASH_WINDOW	ASHuse_X_window(...);
	ASH_WINDOW	ASHroot_X_window(ASH_WINDOW);
	void		ASHtrace(int);
	int		ASHnext_event(...);
	void		ASHstore_buffer_text(ASH_WINDOW,char *);
	char *		ASHload_buffer_text(ASH_WINDOW);
}

#endif




/************************************************************************/
/*									*/
/*	Window management functions					*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_WINDOW	ASHcreate();
extern	void		ASHremove();
extern	void		ASHwindow_background();
extern	void		ASHwindow_border_id();
extern	void		ASHwindow_border();
extern	void		ASHwindow_gravity();
extern	void		ASHinq_size();
extern	ASH_WINDOW	ASHinq_parent();
extern	ASH_WINDOW	ASHinq_top_window();
#ifdef _XLIB_H_
extern	Window		ASHinq_X_window();
#endif
extern	void		ASHset_refresh();
extern	void		ASHset_region_refresh();
extern	int		ASHinq_saved();
extern	void		ASHset_menu_data();
extern	int		ASHinq_menu_data();
extern	void		ASHset_user_data();
extern	int		ASHinq_user_data();
extern	void		ASHset_window_id();
extern	char *		ASHinq_window_id();
extern	void		ASHset_window_name();
extern	char *		ASHinq_window_name();
extern	void		ASHregister_border();
extern	int		ASHinq_valid_window();
extern	void		ASHexit();

#else

extern "C" {
	ASH_WINDOW	ASHcreate(ASH_WINDOW,int,int,int,int,int,int,int,int);
	void		ASHremove(ASH_WINDOW);
	void		ASHwindow_background(ASH_WINDOW,int,ASH_FILL_STYLE,ASH_WINDOW);
	void		ASHwindow_border_id(ASH_WINDOW,ASH_BORDERID);
	void		ASHwindow_border(ASH_WINDOW,int,ASH_COLOR,ASH_WINDOW);
	void		ASHwindow_gravity(ASH_WINDOW,int,int);
	void		ASHinq_size(ASH_WINDOW,ASH_SIZE_TYPE,int *,int *,int *,int *);
	ASH_WINDOW	ASHinq_parent(ASH_WINDOW);
	ASH_WINDOW	ASHinq_top_window(ASH_WINDOW);
#ifdef _XLIB_H_
	Window		ASHinq_X_window(ASH_WINDOW);
#endif
	void		ASHset_refresh(ASH_WINDOW ...);
	void		ASHset_region_refresh(ASH_WINDOW ...);
	int		ASHinq_saved(ASH_WINDOW);
	void		ASHset_menu_data(ASH_WINDOW ...);
	int		ASHinq_menu_data(ASH_WINDOW);
	void		ASHset_user_data(ASH_WINDOW ...);
	int		ASHinq_user_data(ASH_WINDOW);
	void		ASHset_window_id(ASH_WINDOW ...);
	char *		ASHinq_window_id(ASH_WINDOW);
	void		ASHset_window_name(ASH_WINDOW,char *);
	char *		ASHinq_window_name(ASH_WINDOW);
	void		ASHregister_border(ASH_BORDERID,int,ASH_COLOR,ASH_WINDOW);
	int		ASHinq_valid_window(ASH_WINDOW);
	void		ASHexit();
}

#endif




/************************************************************************/
/*									*/
/*	Screen management routines					*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	void		ASHvisible();
extern	void		ASHvisible_after();
extern	int		ASHinq_visible();
extern	int		ASHinq_viewable();
extern	void		ASHview();
extern	void		ASHresize();
extern	void		ASHpop();
extern	void		ASHpush();
extern	void		ASHrefresh();
extern	int		ASHmap();
extern	void		ASHset_control();
extern	void		ASHremove_control();
extern	int		ASHcontrol_msg();

#else

extern "C" {
	void		ASHvisible(ASH_WINDOW,int);
	void		ASHvisible_after(ASH_WINDOW,ASH_WINDOW,int);
	int		ASHinq_visible(ASH_WINDOW);
	int		ASHinq_viewable(ASH_WINDOW);
	void		ASHview(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHresize(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHpop(ASH_WINDOW);
	void		ASHpush(ASH_WINDOW);
	void		ASHrefresh(ASH_WINDOW);
	int		ASHmap(ASH_WINDOW,int,int,ASH_WINDOW,int *,int *);
	void		ASHset_control(ASH_WINDOW ...);
	void		ASHremove_control(ASH_WINDOW ...);
	int		ASHcontrol_msg(ASH_WINDOW,char *);
}

#endif




/************************************************************************/
/*									*/
/*	Window drawing aids						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	void		ASHbatch_mode();
extern	void		ASHsync();
extern	void		ASHclear_damage();
extern	void		ASHfix_damage();
extern	int		ASHregister_damage();
extern	void		ASHfix_visible();
extern	void		ASHfix_invisible();
extern	void		ASHfix_configure();

#else

extern "C" {
	void		ASHbatch_mode(int);
	void		ASHsync(ASH_WINDOW);
	void		ASHfix_damage(ASH_WINDOW);
	int		ASHregister_damage(ASH_WINDOW,int,int,int,int,int);
	void		ASHfix_visible(ASH_WINDOW);
	void		ASHfix_invisible(ASH_WINDOW);
	void		ASHfix_configure(ASH_WINDOW);
}

#endif




/************************************************************************/
/*									*/
/*	Drawing functions						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	void		ASHline();
extern	void		ASHpolyline();
extern	void		ASHpoint();
extern	void		ASHpolypoint();
extern	void		ASHpolygon();
extern	void		ASHconvex_polygon();
extern	void		ASHgeneral_polygon();
extern	void		ASHarc();
extern	void		ASHarc_chord();
extern	void		ASHarc_slice();
extern	void		ASHbox();
extern	void		ASHrectangle();
extern	void		ASHround_box();
extern	void		ASHround_rectangle();
extern	void		ASHcircle();
extern	void		ASHfilled_circle();
extern	void		ASHellipse();
extern	void		ASHfilled_ellipse();
extern	void		ASHclear();
extern	void		ASHtext();
extern	void		ASHblt();
extern	void		ASHwrite_pixels();
extern	void		ASHread_pixels();
extern	void		ASHsave_bitmap();
extern	void		ASHsave_brim();
extern	void		ASHsave_local();
extern	void		ASHload_bitmap();
extern	void		ASHbell();

#else

extern "C" {
	void		ASHline(ASH_WINDOW,int,int,int,int);
	void		ASHpolyline(ASH_WINDOW,int,int[],int[]);
	void		ASHpoint(ASH_WINDOW,int,int);
	void		ASHpolypoint(ASH_WINDOW,int,int[],int[]);
	void		ASHpolygon(ASH_WINDOW,int,int[],int[]);
	void		ASHconvex_polygon(ASH_WINDOW,int,int[],int[]);
	void		ASHgeneral_polygon(ASH_WINDOW,int,int[],int[]);
	void		ASHarc(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHarc_chord(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHarc_slice(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHbox(ASH_WINDOW,int,int,int,int);
	void		ASHrectangle(ASH_WINDOW,int,int,int,int);
	void		ASHround_box(ASH_WINDOW,int,int,int,int,int);
	void		ASHround_rectangle(ASH_WINDOW,int,int,int,int,int);
	void		ASHcircle(ASH_WINDOW,int,int,int);
	void		ASHfilled_circle(ASH_WINDOW,int,int,int);
	void		ASHellipse(ASH_WINDOW,int,int,int,int);
	void		ASHfilled_ellipse(ASH_WINDOW,int,int,int,int);
	void		ASHclear(ASH_WINDOW);
	void		ASHtext(ASH_WINDOW,int,int,char *);
	void		ASHblt(ASH_WINDOW,int,int,int,int,int,int);
	void		ASHwrite_pixels(ASH_WINDOW,int,int,int,int,int[]);
	void		ASHread_pixels(ASH_WINDOW,int,int,int,int,int[]);
	void		ASHsave_bitmap(ASH_WINDOW,char *);
	void		ASHsave_brim(ASH_WINDOW,char *);
	void		ASHsave_local(ASH_WINDOW,char *);
	void		ASHload_bitmap(ASH_WINDOW,char *);
	void		ASHbell();
}

#endif




/************************************************************************/
/*									*/
/*	Property-setting routines					*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_COLOR	ASHcolor();
extern	ASH_COLOR	ASHbackground_color();
extern	int		ASHcombination_rule();
extern	int		ASHplane_mask();
extern	int		ASHwindow_draw_thru();
extern	ASH_FILL_STYLE	ASHfill();
extern	int		ASHline_style();
extern	int		ASHclip();
extern	void		ASHclip_region();
extern	void		ASHclip_add_region();
extern	void		ASHclip_window();
extern	int		ASHouter_clip();
extern	void		ASHouter_clip_region();
extern	void		ASHouter_clip_add_region();
extern	void		ASHouter_clip_window();
extern	ASH_FONT	ASHfont();
extern	ASH_COLOR	ASHtext_color();
extern	ASH_COLOR	ASHtext_background_color();
extern	ASH_COLOR	ASHtext_background_clear();
extern	int		ASHtext_combination_rule();
extern	int		ASHtext_plane_mask();
extern	ASH_WINDOW	ASHsource();
extern	ASH_COLOR_TABLE ASHnew_color_table();
extern	void		ASHfill_color_table();
extern	ASH_COLOR_TABLE ASHset_color_table();
extern	ASH_COLOR_TABLE ASHinq_color_table();
extern	ASH_COLOR	ASHlookup_color();
extern	ASH_COLOR	ASHaverage_color();
extern	void		ASHfill_table();
extern	void		ASHfill_table_clear();
extern	void		ASHfill_table_tile();
extern	void		ASHline_table();
extern	char *		ASHinq_base_font();
extern	ASH_FONT	ASHloadfont();
extern	void		ASHinq_text_extent();
extern	void		ASHinq_text_offset();
extern	void		ASHinq_text_next();
extern	void		ASHinq_text_info();
extern	int		ASHinq_font_info();
extern	int		ASHinq_color();
extern	int		ASHinq_background_color();
extern	int		ASHinq_combination_rule();
extern	int		ASHinq_plane_mask();
extern	ASH_FILL_STYLE	ASHinq_fill();
extern	int		ASHinq_line_style();
extern	int		ASHinq_clip();
extern	int		ASHinq_outer_clip();
extern	ASH_FONT	ASHinq_font();
extern	int		ASHinq_text_color();
extern	int		ASHinq_text_background_color();
extern	int		ASHinq_text_combination_rule();
extern	int		ASHinq_text_plane_mask();
extern	void		ASHpush_drawinfo();
extern	void		ASHsave_drawinfo();
extern	void		ASHpop_drawinfo();
extern	ASH_DRAWINFO	ASHnew_drawinfo();
extern	ASH_DRAWINFO	ASHset_drawinfo();
extern	void		ASHcopy_drawinfo();

#else

extern "C" {
	ASH_COLOR	ASHcolor(ASH_WINDOW,ASH_COLOR);
	ASH_COLOR	ASHbackground_color(ASH_WINDOW,ASH_COLOR);
	int		ASHcombination_rule(ASH_WINDOW,int);
	int		ASHplane_mask(ASH_WINDOW,int);
	int		ASHwindow_draw_thru(ASH_WINDOW,int);
	ASH_FILL_STYLE	ASHfill(ASH_WINDOW,ASH_FILL_STYLE);
	int		ASHline_style(ASH_WINDOW,ASH_LINE_STYLE);
	int		ASHclip(ASH_WINDOW,int);
	void		ASHclip_region(ASH_WINDOW,int,int,int,int);
	void		ASHclip_add_region(ASH_WINDOW,int,int,int,int);
	void		ASHclip_window(ASH_WINDOW,ASH_WINDOW);
	int		ASHouter_clip(ASH_WINDOW,int);
	void		ASHouter_clip_region(ASH_WINDOW,int,int,int,int);
	void		ASHouter_clip_add_region(ASH_WINDOW,int,int,int,int);
	void		ASHouter_clip_window(ASH_WINDOW,ASH_WINDOW);
	ASH_FONT	ASHfont(ASH_WINDOW,ASH_FONT);
	ASH_COLOR	ASHtext_color(ASH_WINDOW,ASH_COLOR);
	ASH_COLOR	ASHtext_background_color(ASH_WINDOW,ASH_COLOR);
	ASH_COLOR	ASHtext_background_clear(ASH_WINDOW,ASH_COLOR);
	int		ASHtext_combination_rule(ASH_WINDOW,int);
	int		ASHtext_plane_mask(ASH_WINDOW,int);
	ASH_WINDOW	ASHsource(ASH_WINDOW,ASH_WINDOW);
	ASH_COLOR_TABLE ASHnew_color_table(ASH_WINDOW);
	void		ASHfill_color_table(ASH_WINDOW,ASH_COLOR_TABLE,ASH_COLOR,int,short[],short[],short[]);
	ASH_COLOR_TABLE ASHset_color_table(ASH_WINDOW,ASH_COLOR_TABLE);
	ASH_COLOR_TABLE ASHinq_color_table(ASH_WINDOW);
	ASH_COLOR	ASHlookup_color(ASH_WINDOW,char *);
	ASH_COLOR	ASHaverage_color(ASH_WINDOW,ASH_COLOR,ASH_COLOR);
	void		ASHfill_table(int,int,short *);
	void		ASHfill_table_clear(int,int,short *);
	void		ASHfill_table_tile(int,ASH_WINDOW);
	void		ASHline_table(int,int,ASH_LINE_STYLE,int,int,int,int ...);
	char *		ASHinq_base_font();
	ASH_FONT	ASHloadfont(char *);
	void		ASHinq_text_extent(...);
	void		ASHinq_text_offset(...);
	void		ASHinq_text_next(...);
	void		ASHinq_text_info(...);
	int		ASHinq_font_info(char *,int *,int *,int *,char[],char[],char[]);
	int		ASHinq_color(ASH_WINDOW);
	int		ASHinq_background_color(ASH_WINDOW);
	int		ASHinq_combination_rule(ASH_WINDOW);
	int		ASHinq_plane_mask(ASH_WINDOW);
	ASH_FILL_STYLE	ASHinq_fill(ASH_WINDOW);
	int		ASHinq_line_style(ASH_WINDOW);
	int		ASHinq_clip(ASH_WINDOW);
	int		ASHinq_outer_clip(ASH_WINDOW);
	ASH_FONT	ASHinq_font(ASH_WINDOW);
	int		ASHinq_text_color(ASH_WINDOW);
	int		ASHinq_text_background_color(ASH_WINDOW);
	int		ASHinq_text_combination_rule(ASH_WINDOW);
	int		ASHinq_text_plane_mask(ASH_WINDOW);
	void		ASHpush_drawinfo(ASH_WINDOW,ASH_DRAWINFO);
	void		ASHsave_drawinfo(ASH_WINDOW);
	void		ASHpop_drawinfo(ASH_WINDOW);
	ASH_DRAWINFO	ASHnew_drawinfo(ASH_WINDOW);
	ASH_DRAWINFO	ASHset_drawinfo(ASH_WINDOW,ASH_DRAWINFO);
	void		ASHcopy_drawinfo(ASH_WINDOW,ASH_WINDOW);
}

#endif






/************************************************************************/
/*									*/
/*	Cursor functions						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_CURSOR	ASHwindow_cursor();
extern	void		ASHcursor_move();
extern	int		ASHcursor_enable();
extern	ASH_CURSOR	ASHcursor_standard();
extern	ASH_CURSOR	ASHcursor_define_bitmap();
extern	ASH_CURSOR	ASHcursor_define_pattern();
extern	ASH_CURSOR	ASHcursor_define_char();

#else

extern "C" {
	ASH_CURSOR	ASHwindow_cursor(ASH_WINDOW,ASH_CURSOR);
	void		ASHcursor_move(ASH_WINDOW,int,int);
	int		ASHcursor_enable(ASH_WINDOW,int);
	ASH_CURSOR	ASHcursor_standard(ASH_CURSOR_ID);
	ASH_CURSOR	ASHcursor_define_bitmap(ASH_WINDOW,ASH_WINDOW,int,int,int,int);
	ASH_CURSOR	ASHcursor_define_pattern(short *,short *,int,int,int,int,int,int);
	ASH_CURSOR	ASHcursor_define_char(char,ASH_FONT,char,ASH_FONT,int,int);
}

#endif





/************************************************************************/
/*									*/
/*	Input interface routines					*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_WINDOW	ASHfind_child_window();
extern	ASH_SENSE	ASHsensitive_area();
extern	void		ASHsensitive_remove();
extern	void		ASHsensitive_remove_all();
extern	ASH_SENSE	ASHinq_sensitive();
extern	void		ASHinput_window();
extern	int		ASHinq_input_data();
extern	ASH_WINDOW	ASHgrab_input();
extern	ASH_WINDOW	ASHgrab_from();
extern	ASH_WINDOW	ASHgrab_text_from();
extern	void		ASHenter_window();
extern	ASH_WINDOW	ASHevent_process();
extern	ASH_WINDOW	ASHinput_script();

#else

extern "C" {
	ASH_WINDOW	ASHfind_child_window(ASH_WINDOW,int,int);
	ASH_SENSE	ASHsensitive_area(ASH_WINDOW,int,int,int,int,int);
	void		ASHsensitive_remove(ASH_SENSE);
	void		ASHsensitive_remove_all(ASH_WINDOW);
	ASH_SENSE	ASHinq_sensitive();
	void		ASHinput_window(ASH_WINDOW ...);
	int		ASHinq_input_data(ASH_WINDOW);
	ASH_WINDOW	ASHgrab_input(ASH_WINDOW);
	ASH_WINDOW	ASHgrab_from(ASH_WINDOW,ASH_WINDOW);
	ASH_WINDOW	ASHgrab_text_from(ASH_WINDOW,ASH_WINDOW);
	void		ASHenter_window(ASH_WINDOW);
	ASH_WINDOW	ASHevent_process(...);
	ASH_WINDOW	ASHinput_script(ASH_WINDOW,int,int);
};

#endif




/************************************************************************/
/*									*/
/*	Utility extensions						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	void		ASHclear_box();
extern	void		ASHfill_box();
extern	void		ASHhilite_box();
extern	void		ASHstipple_box();
extern	void		ASHdraw_hilite();
extern	void		ASHcenter_text();
extern	void		ASHspline_arc();
extern	void		ASHspline();
extern	void		ASHspline_closed();
extern	void		ASHspline_filled();

#else

extern "C" {
	void		ASHclear_box(ASH_WINDOW,int,int,int,int);
	void		ASHfill_box(ASH_WINDOW,int,int,int,int);
	void		ASHhilite_box(ASH_WINDOW,int,int,int,int);
	void		ASHstipple_box(ASH_WINDOW,int,int,int,int);
	void		ASHdraw_hilite(ASH_WINDOW,int);
	void		ASHcenter_text(ASH_WINDOW,char *,int,int,int,int);
	void		ASHspline_arc(ASH_WINDOW,int,int,int,int,int,int,int,int);
	void		ASHspline(ASH_WINDOW,int,int[],int[]);
	void		ASHspline_closed(ASH_WINDOW,int,int[],int[]);
	void		ASHspline_filled(ASH_WINDOW,int,int[],int[]);
};

#endif




/************************************************************************/
/*									*/
/*	Animation extensions						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	ASH_WINDOW	ASHanim_setup();
extern	void		ASHanim_done();
extern	ASH_WINDOW	ASHanim_next_frame();
extern	int		ASHanim_color();
extern	int		ASHanim_background_color();
extern	ASH_WINDOW	ASHanim_overlay();

#else

extern "C" {
	ASH_WINDOW	ASHanim_setup(ASH_WINDOW,int,int,int);
	void		ASHanim_done(ASH_WINDOW);
	ASH_WINDOW	ASHanim_next_frame(ASH_WINDOW);
	int		ASHanim_color(ASH_WINDOW,ASH_COLOR);
	int		ASHanim_background_color(ASH_WINDOW,ASH_COLOR);
	ASH_WINDOW	ASHanim_overlay(ASH_WINDOW,int);
};

#endif




/************************************************************************/
/*									*/
/*	Resource manager extensions					*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	int		ASHset_application();
extern	void		ASHset_application_string();
extern	void		ASHresource_file();
extern	char *		ASHinq_resource();
extern	char *		ASHinq_window_resource();
extern	char *		ASHinq_window_full_id();
extern	void		ASHset_window_defaults();

#else

extern "C" {
	int		ASHset_application(int,char **);
	void		ASHset_application_string(char *);
	void		ASHresource_file(char *,int);
	char *		ASHinq_resource(char *);
	char *		ASHinq_window_resource(ASH_WINDOW,char *);
	char *		ASHinq_window_full_id(ASH_WINDOW,char *);
	void		ASHset_window_defaults(ASH_WINDOW);
};

#endif



/************************************************************************/
/*									*/
/*	Locking routines						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	int		ASHlock();
extern	void		ASHunlock();
extern	ASH_LOCK_STATUS ASHinput_lock();
extern	void		ASHinput_unlock();
extern	void		ASHinput_lock_makelong();
extern	void		ASHinput_lock_by();

#else

extern "C" {
	int		ASHlock(ASH_WINDOW);
	void		ASHunlock(ASH_WINDOW);
	ASH_LOCK_STATUS ASHinput_lock(ASH_WINDOW,int);
	void		ASHinput_unlock(ASH_WINDOW);
	void		ASHinput_lock_makelong(ASH_WINDOW);
	void		ASHinput_lock_by(ASH_WINDOW,ASH_WINDOW);
};

#endif





/************************************************************************/
/*									*/
/*	Font family routines						*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	char *		ASHfont_name();
extern	char *		ASHfont_name_check();
extern	ASH_FONT	ASHfont_id();

#else

extern "C" {
	char *		ASHfont_name(char *,int,ASH_FONT_PROPS);
	char *		ASHfont_name_check(char *,int,ASH_FONT_PROPS);
	ASH_FONT	ASHfont_id(char *,int,ASH_FONT_PROPS);
};

#endif



/************************************************************************/
/*									*/
/*	Icon routines							*/
/*									*/
/************************************************************************/


#ifndef __cplusplus

extern	void		ASHicon_define();
extern	void		ASHicon_draw();
extern	void		ASHicon_size();

#else

extern "C" {
	void		ASHicon_define(char *,int,int,int *);
	void		ASHicon_draw(ASH_WINDOW,int,int,char *);
	void		ASHicon_size(char *,int *,int *);
};

#endif




#endif



/* end of ash.h */

