/************************************************************************/
/*									*/
/*		pearwin.c						*/
/*									*/
/*	Window management for PEAR object viewer package		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "pear_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define BTN_MOVE	2




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


	Sequence	PEAR__all_pear;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		pear_draw();
static	int		pear_hit();
static	Boolean 	pear_typein();
static	int		pear_control();
static	void		vert_scroll();
static	void		hor_scroll();
static	int		applic_hit();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define MENU_WIN	1
#define GELO_WIN	2
#define LR_CORNER	3
#define HSCROLL_WIN	4
#define VSCROLL_WIN	5
#define INSET_WIN	6



static LEAF_DATA	pear_leaf[] = {
   LEAF_ROOT(NULL),
   { MENU_WIN, LEAF_TYPE_PDM,
	{ LEAF_COORD_LX, LEAF_COORD_TEXT_LINE,
	     LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   { INSET_WIN, LEAF_TYPE_WINDOW|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_REL(50), LEAF_COORD_NEXT(HSCROLL_WIN),
	     LEAF_COORD_NEXT(VSCROLL_WIN), LEAF_COORD_REL(50) },
	pear_hit, NULL },
   { GELO_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_NEXT(HSCROLL_WIN),
	     LEAF_COORD_NEXT(VSCROLL_WIN), LEAF_COORD_NEXT(MENU_WIN) },
	pear_hit, NULL },
   { LR_CORNER, LEAF_TYPE_NONE,
	{ LEAF_COORD_SIZE(20), LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_SIZE(20) },
	applic_hit, NULL },
   { HSCROLL_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_NEXT_LINE(LR_CORNER), LEAF_COORD_SAME_LINE(LR_CORNER) },
	NULL, NULL },
   { VSCROLL_WIN, LEAF_TYPE_WINDOW,
	{ LEAF_COORD_SAME_LINE(LR_CORNER), LEAF_COORD_NEXT_LINE(LR_CORNER),
	     LEAF_COORD_RX, LEAF_COORD_NEXT(MENU_WIN) },
	NULL, NULL },
   LEAF_END
};





/************************************************************************/
/*									*/
/*	PEAR_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
PEAR_win_init()
{
   ITRACE("PEAR_win_init");

   PEAR__all_pear = NULL;
};





/************************************************************************/
/*									*/
/*	PEARedit -- set up a PEAR editor				*/
/*									*/
/************************************************************************/


void
PEARedit(w,o,t,id)
   ASH_WINDOW w;
   Universal o;
   Universal t;
   Universal id;
{
   register PEAR_DATA pd;
   register Integer i;
   ASH_LOCK_STATUS sts;

   ENTER("PEARedit 0x%x 0x%x 0x%x 0x%x",w,o,t,id);

   pd = (PEAR_DATA) calloc(1,sizeof(PEAR_DATA_B));
   pd->id = id;
   pd->window = w;
   pd->object = o;
   pd->type = t;
   pd->gelowin = NULL;
   pd->menuwin = NULL;
   pd->geloobj = NULL;
   pd->topgeloobj = NULL;
   pd->match = NULL;
   pd->redraw = TRUE;
   pd->last_x = -1;
   pd->last_y = -1;
   pd->havegelo = FALSE;
   pd->down_x = -1;
   pd->down_y = -1;
   pd->down_fg = 0;

   pd->insetwin = NULL;
   pd->showinset = FALSE;
   pd->inobject = NULL;
   pd->intype = NULL;
   pd->ingeloobj = NULL;
   pd->insetredraw = TRUE;
   pd->haveingelo = FALSE;
   pd->in_depend = FALSE;
   pd->in_changed = FALSE;

   for (i = 0; i < MAX_SELECT; ++i) {
      pd->selects[i] = NULL;
      pd->selinsets[i] = FALSE;
    };

   pd->update_delay = FALSE;
   pd->update_obj = NULL;
   pd->update_all = FALSE;

   PROTECT;
   PEAR__all_pear = APPEND(pd,PEAR__all_pear);
   UNPROTECT;

   BIOnew_input_window(w);
/* BIOset_cursor_standard(w,ASH_CURSOR_X);	*/

   LEAFsetup_window(w,pear_leaf,pd);
   LEAFset_control(w,pear_control);

   sts = ASHinput_lock(pd->window,TRUE);
   if (sts == ASH_LOCK_FAIL) return;

   pear_draw(pd);
   if (sts == ASH_LOCK_GRANTED) ASHinput_unlock(pd->window);
};






/************************************************************************/
/*									*/
/*	PEARupdate -- update pear display in the given window		*/
/*									*/
/************************************************************************/


Boolean
PEARupdate(w)
   ASH_WINDOW w;
{
   PEAR_DATA pd;
   GELO_OBJECT obj,nobj,onobj;
   Boolean cmpfg;
   Integer i;
   Boolean newfg;

   pd = (PEAR_DATA) ASHinq_user_data(w);
   if (pd == NULL) return FALSE;

   if (ASHinput_lock(pd->window,TRUE) != ASH_LOCK_GRANTED) return FALSE;

   for (i = 0; i < MAX_SELECT; ++i) {
      PEAR_select(pd,i,NULL,FALSE,FALSE,TRUE);
    };

   onobj = APPLEgelo_create(pd->object,pd->type,pd->match,FALSE,pd);

   if (pd->gelowin == NULL) cmpfg = TRUE;
   else cmpfg = GELOcompare(pd->topgeloobj,onobj,&obj,&nobj);

   if (pd->gelowin != NULL && !cmpfg) {
      for (i = 0; i < MAX_SELECT; ++i) {
	 PEAR_select(pd,i,NULL,FALSE,FALSE,TRUE);
       };
      newfg = (pd->topgeloobj != pd->geloobj);
      if (pd->topgeloobj == obj) pd->topgeloobj = nobj;
      pd->geloobj = pd->topgeloobj;
      GELOwindow_replace(pd->gelowin,obj,nobj);
      GELOwindow_free(pd->gelowin,obj);
      GELOwindow_free_except(pd->gelowin,onobj,nobj);
      if (pd->gelowin != NULL && newfg) {
	 pd->redraw = TRUE;
	 GELOlayout_validate(pd->geloobj,FALSE);
	 PEAR_refresh(pd,FALSE,TRUE,TRUE);
       };
    }
   else {
      GELOwindow_free(pd->gelowin,onobj);
    };

   ASHinput_unlock(pd->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	PEARremove -- remove pear editor from window			*/
/*									*/
/************************************************************************/


void
PEARremove(w)
   ASH_WINDOW w;
{
   PEAR_DATA pd;

   pd = (PEAR_DATA) ASHinq_user_data(w);
   if (pd == NULL) return;

   PEAR_finish(pd);
};





/************************************************************************/
/*									*/
/*	PEAR_refresh -- handle various refresh requests 		*/
/*	PEAR_insetrefresh --handle refresh of inset window only 	*/
/*									*/
/************************************************************************/


void
PEAR_refresh(pd,redrawfg,rescalefg,inredraw)
   PEAR_DATA pd;
   Boolean redrawfg;
   Boolean rescalefg;
   Boolean inredraw;
{
   Integer lx,by,rx,ty;

   ITRACE("PEAR_refresh 0x%x %d %d %d",pd,redrawfg,rescalefg,inredraw);

   if (redrawfg || inredraw) PEAR_ext_refresh(pd);

   if (redrawfg && pd->topgeloobj != NULL) {
      GELOwindow_free(pd->gelowin,pd->topgeloobj);
      pd->topgeloobj = NULL;
      pd->geloobj = NULL;
      pd->redraw = TRUE;
      pd->down_fg = 0;
    };

   if (rescalefg && pd->havegelo) {
      GELOwindow_inq_world(pd->gelowin,&lx,&by,&rx,&ty);
      GELOwindow_set_view(pd->gelowin,lx,by,rx,ty);
    };

   if (inredraw) {
      if (pd->ingeloobj != NULL) {
	 GELOwindow_free(pd->insetwin,pd->ingeloobj);
	 pd->ingeloobj = NULL;
       };
      pd->insetredraw = TRUE;
    };

   pear_draw(pd);
};





void
PEAR_insetrefresh(pd,inredraw)
   PEAR_DATA pd;
   Boolean inredraw;
{
   register Integer i;

   ITRACE("PEAR_insetrefresh 0x%x %d",pd,inredraw);

   if (inredraw) PEAR_ext_refresh(pd);

   if (inredraw) {
      if (pd->ingeloobj != NULL) {
	 GELOwindow_free(pd->insetwin,pd->ingeloobj);
	 pd->ingeloobj = NULL;
       };
      pd->insetredraw = TRUE;
    };

   if (pd->ingeloobj == NULL && pd->showinset) {
      pd->ingeloobj = APPLEgelo_create(pd->inobject,pd->intype,NULL,FALSE,pd);
      pd->insetredraw = TRUE;
    };

   if (pd->showinset && pd->ingeloobj != NULL) {
      if (!pd->haveingelo) {
	 GELOwindow_open(pd->insetwin);
	 pd->haveingelo = TRUE;
       };
      if (pd->insetredraw) {
	 for (i = 0; i < MAX_SELECT; ++i) {
	    if (pd->selinsets[i]) pd->selects[i] = NULL;
	  };
	 GELOwindow_draw(pd->insetwin,pd->ingeloobj);
       }
      else GELOwindow_redraw(pd->insetwin);
      pd->insetredraw = FALSE;
    };
};





/************************************************************************/
/*									*/
/*	PEAR_finish -- cleanup pear instance				*/
/*									*/
/************************************************************************/


void
PEAR_finish(pd)
   PEAR_DATA pd;
{
   ASHcontrol_msg(pd->window,"EDIT$SAVE_CLOSE");
   ASHset_user_data(pd->window,NULL);

   if (ASHinq_valid_window(pd->window)) {
      LEAFremove(pd->window);
      pd->gelowin = NULL;
      pd->menuwin = NULL;
      pd->vscroll = NULL;
      pd->hscroll = NULL;
      pd->insetwin = NULL;
    };

   PROTECT;
   PEAR__all_pear = REMOB(pd,PEAR__all_pear);
   UNPROTECT;

   ASHclear(pd->window);

   free(pd);
};





/************************************************************************/
/*									*/
/*	PEAR_select -- handle selection 				*/
/*									*/
/************************************************************************/


void
PEAR_select(pd,sty,obj,inset,lastok,force)
   PEAR_DATA pd;
   Integer sty;
   GELO_OBJECT obj;
   Boolean inset;
   Boolean lastok;
   Boolean force;
{
   register GELO_OBJECT par;
   Character buf[16];

   ITRACE("PEAR_select 0x%x %d 0x%x %d %d %d",pd,sty,obj,inset,lastok,force);

   if (obj != NULL && lastok && pd->selects[sty] != NULL) {
      par = GELOinq_parent(pd->selects[sty]);
      if (par != NULL) {
	 obj = par;
	 inset = pd->selinsets[sty];
       };
    }
   else if (!force && obj != NULL && obj == pd->selects[sty]) {
      obj = NULL;
    };

   if (obj != NULL) {
      sprintf(buf,"Mouse%d",sty);
      if (PEAR_edit_select(pd,"Mouse",buf,obj)) return;
    };

   if (pd->selects[sty] != NULL) {
      if (pd->selinsets[sty]) {
	 GELOwindow_select(pd->insetwin,pd->selects[sty],sty,FALSE);
       }
      else {
	 GELOwindow_select(pd->gelowin,pd->selects[sty],sty,FALSE);
       };
    };

   pd->selects[sty] = obj;
   pd->selinsets[sty] = inset;

   if (obj != NULL) {
      if (inset) {
	 GELOwindow_select(pd->insetwin,obj,sty,TRUE);
       }
      else {
	 GELOwindow_select(pd->gelowin,obj,sty,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	PEAR_set_scroll -- set scroll regions				*/
/*									*/
/************************************************************************/


void
PEAR_set_scroll(pd)
   PEAR_DATA pd;
{
   Integer lx,by,rx,ty;
   Integer vlx,vby,vrx,vty;

   ITRACE("PEAR_set_scroll 0x%x",pd);

   if (!pd->havegelo) {
      if (pd->vscroll != NULL)
	 STEMscroll_set(pd->vscroll,0,1,0,1);
      if (pd->hscroll != NULL)
	 STEMscroll_set(pd->hscroll,0,1,0,1);
    }
   else {
      GELOwindow_inq_world(pd->gelowin,&lx,&by,&rx,&ty);
      GELOwindow_inq_view(pd->gelowin,&vlx,&vby,&vrx,&vty);
      if (pd->vscroll != NULL)
	 STEMscroll_set(pd->vscroll,ty,by,vty,vby);
      if (pd->hscroll != NULL)
	 STEMscroll_set(pd->hscroll,lx,rx,vlx,vrx);
    };
};





/************************************************************************/
/*									*/
/*	pear_draw -- draw the given window				*/
/*									*/
/************************************************************************/


static void
pear_draw(pd)
   PEAR_DATA pd;
{
   register Integer i;

   DTRACE("pear_draw 0x%x",pd);

   LEAFredraw(pd->window);

   pd->last_x = -1;
   if (pd->down_fg != 0 && pd->selects[DOWN_SELECT] != NULL) {
      if (pd->selinsets[DOWN_SELECT])
	 GELOwindow_select(pd->insetwin,pd->selects[DOWN_SELECT],DOWN_SELECT,FALSE);
      else
	 GELOwindow_select(pd->gelowin,pd->selects[DOWN_SELECT],DOWN_SELECT,FALSE);
      pd->down_fg = 0;
      pd->selects[DOWN_SELECT] = NULL;
    };

   if (pd->menuwin == NULL) {
      pd->menuwin = LEAFinq_window(pd->window,MENU_WIN,0);
      PEAR_menu_define(pd);
    };

   if (pd->gelowin == NULL) {
      pd->gelowin = LEAFinq_window(pd->window,GELO_WIN,0);
      GELOwindow_open(pd->gelowin);
      pd->havegelo = TRUE;
      BIOnew_input_window(pd->gelowin);
    };

   if (pd->insetwin == NULL) {
      pd->insetwin = LEAFinq_window(pd->window,INSET_WIN,0);
      BIOnew_input_window(pd->insetwin);
      ASHvisible(pd->insetwin,pd->showinset);
    };

   if (pd->topgeloobj == NULL) {
      pd->topgeloobj = APPLEgelo_create(pd->object,pd->type,pd->match,FALSE,pd);
      pd->geloobj = pd->topgeloobj;
      pd->redraw = TRUE;
    };

   if (pd->ingeloobj == NULL && pd->showinset) {
      pd->ingeloobj = APPLEgelo_create(pd->inobject,pd->intype,NULL,FALSE,pd);
      pd->insetredraw = TRUE;
    };

   if (pd->vscroll == NULL) {
      pd->vscroll = LEAFinq_window(pd->window,VSCROLL_WIN,0);
      STEMscroll(pd->vscroll,vert_scroll,pd);
    };

   if (pd->hscroll == NULL) {
      pd->hscroll = LEAFinq_window(pd->window,HSCROLL_WIN,0);
      STEMscroll(pd->hscroll,hor_scroll,pd);
    };

   if (pd->geloobj == NULL) pd->geloobj = pd->topgeloobj;

   if (pd->geloobj != NULL) {
      if (pd->redraw) {
	 for (i = 0; i < MAX_SELECT; ++i) {
	    if (!pd->selinsets[i]) pd->selects[i] = NULL;
	  };
	 GELOwindow_draw(pd->gelowin,pd->geloobj);
       }
      else GELOwindow_redraw(pd->gelowin);
      pd->redraw = FALSE;
    };

   if (pd->showinset && pd->ingeloobj != NULL) {
      if (!pd->haveingelo) {
	 GELOwindow_open(pd->insetwin);
	 pd->haveingelo = TRUE;
       };
      if (pd->insetredraw) {
	 for (i = 0; i < MAX_SELECT; ++i) {
	    if (pd->selinsets[i]) pd->selects[i] = NULL;
	  };
	 GELOwindow_draw(pd->insetwin,pd->ingeloobj);
       }
      else GELOwindow_redraw(pd->insetwin);
      pd->insetredraw = FALSE;
    };

   if (!ASHinq_valid_window(pd->gelowin)) return;

   if (pd->gelowin != NULL) PEAR_set_scroll(pd);
};





/************************************************************************/
/*									*/
/*	pear_hit -- handle hits in the GELO window			*/
/*									*/
/************************************************************************/


static int
pear_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   PEAR_DATA pd;
   Boolean ifg,lfg;
   GELO_OBJECT nprv;
   ASH_WINDOW w;
   Integer sty;
   Boolean rfg;

   ITRACE("pear_hit %d %d 0x%x 0x%x 0x%x",x,y,ch,btn,rgn);

   pd = (PEAR_DATA) RIPinq_data(rgn);
   if (pd == NULL) return FALSE;
   w = RIPinq_window(rgn);
   if (pd->gelowin != w && pd->insetwin != w) return FALSE;

   if (pd->down_fg != 0 && pd->selects[DOWN_SELECT] != NULL) {
      if (pd->selinsets[DOWN_SELECT])
	 GELOwindow_select(pd->insetwin,pd->selects[DOWN_SELECT],DOWN_SELECT,FALSE);
      else
	 GELOwindow_select(pd->gelowin,pd->selects[DOWN_SELECT],DOWN_SELECT,FALSE);
    };

   w = RIPinq_window(rgn);
   ifg = (w == pd->insetwin);

   if (ifg) {
      ASHmap(ASHinq_top_window(pd->insetwin),x,y,pd->insetwin,&x,&y);
      nprv = GELOcorrelate(pd->insetwin,x,y);
    }
   else {
      ASHmap(ASHinq_top_window(pd->gelowin),x,y,pd->gelowin,&x,&y);
      nprv = GELOcorrelate(pd->gelowin,x,y);
    };

   while (nprv != NULL && GELOinq_flavor(nprv) == GELO_FLAVOR_EMPTY)
      nprv = GELOinq_parent(nprv);

   if (nprv == NULL) {
      rfg = FALSE;
      btn = 0;
    }
   else if (btn & RIP_BTN_NONE) {
      pd->down_fg = 0;
      pd->selects[DOWN_SELECT] = NULL;
      if (ch == '1') btn = RIP_BTN_LEFT|RIP_BTN_UP;
      else if (ch == '2') btn = RIP_BTN_MID|RIP_BTN_UP;
      else if (ch == '3') btn = RIP_BTN_RIGHT|RIP_BTN_UP;
      else {
	 rfg = pear_typein(pd,x,y,ch,nprv,ifg);
	 return rfg;
       };
    };

   if (btn & RIP_BTN_DOWN) {
      pd->down_fg = btn&RIP_BTN_ANY;
      pd->down_x = x;
      pd->down_y = y;
      pd->selects[DOWN_SELECT] = nprv;
      pd->selinsets[DOWN_SELECT] = ifg;
      if (ifg) GELOwindow_select(pd->insetwin,nprv,DOWN_SELECT,TRUE);
      else GELOwindow_select(pd->gelowin,nprv,DOWN_SELECT,TRUE);
      rfg = TRUE;
    }
   else if ((pd->down_fg & btn) != 0 &&
	       (abs(pd->down_x-x) > BTN_MOVE || abs(pd->down_y-y) > BTN_MOVE)) {
      pd->last_x = -1;
      pd->down_fg = 0;
      if (nprv == NULL || pd->selects[DOWN_SELECT] == NULL) rfg = FALSE;
      else {
	 if (btn & (RIP_BTN_LEFT|RIP_BTN_MID)) {
	    PEAR_select(pd,SELECT1,pd->selects[DOWN_SELECT],pd->selinsets[DOWN_SELECT],FALSE,TRUE);
	    PEAR_select(pd,SELECT2,nprv,ifg,FALSE,TRUE);
	    lfg = PEAR_edit_arc(pd,"BTNIN","Connect");
	  }
	 else {
	    PEAR_select(pd,SELECT2,pd->selects[DOWN_SELECT],pd->selinsets[DOWN_SELECT],FALSE,TRUE);
	    PEAR_select(pd,SELECT1,nprv,ifg,FALSE,TRUE);
	    lfg = PEAR_edit_insert(pd,"BTNIN","Append Copy");
	  };
	 pd->selects[DOWN_SELECT] = NULL;
	 rfg = lfg;
       };
    }
   else if (btn != 0) {
      if (btn & RIP_BTN_LEFT) sty = SELECT1;
      else if (btn & RIP_BTN_MID) sty = SELECT2;
      else if (btn & RIP_BTN_RIGHT) sty = SELECT3;
      else sty = SELECT1;

      lfg = (pd->last_x == x && pd->last_y == y);

      PEAR_select(pd,sty,nprv,ifg,lfg,FALSE);

      pd->last_x = x;
      pd->last_y = y;
      rfg = TRUE;
    };

   return rfg;
};






/************************************************************************/
/*									*/
/*	pear_typein -- handle typing in pear window			*/
/*									*/
/************************************************************************/


static Boolean
pear_typein(pd,x,y,ch,gobj,ifg)
   PEAR_DATA pd;
   Integer x,y;
   Integer ch;
   GELO_OBJECT gobj;
   Boolean ifg;
{
   register Boolean fg;
   register String menu;

   DTRACE("pear_typein 0x%x %d %d 0x%x 0x%x %d",pd,x,y,ch,gobj,ifg);

   menu = "TYPEIN";

   if (FKEY_TEST(ch)) {
      fg = PEAR_ext_typein(ch,gobj,pd->selects[SELECT1],pd->selects[SELECT2]);
      return fg;
    };

   switch (ch) {
      case 'a' :
	 fg = PEAR_edit_insert(pd,menu,"Append");
	 break;
      case 'A' :
	 fg = PEAR_edit_insert(pd,menu,"Append Copy");
	 break;
      case 'c' :
	 fg = PEAR_edit_arc(pd,menu,"Connect");
	 break;
      case 'C' :
	 fg = PEAR_edit_arc(pd,menu,"Connect New");
	 break;
      case 'd' :
      case 'D' :
	 fg = PEAR_edit_remove(pd,menu,"Delete");
	 break;
      case 'e' :
	 fg = PEAR_emphasize_btn(pd,menu,"Emphasize");
	 break;
      case 'E' :
	 fg = PEAR_emphasize_btn(pd,menu,"Deemphasize");
	 break;
      case 'h' :
      case 'H' :
	 fg = PEAR_inset_btn(pd,menu,"Hide Inset");
	 break;
      case 'i' :
	 fg = PEAR_edit_insert(pd,menu,"Insert");
	 break;
      case 'I' :
	 fg = PEAR_edit_insert(pd,menu,"Insert Copy");
	 break;
      case 'n' :
      case 'N' :
	 fg = PEAR_edit_replace(pd,menu,"New Value");
	 break;
      case 'r' :
      case 'R' :
	 fg = PEAR_edit_remove(pd,menu,"Remove");
	 break;
      case 's' :
      case 'S' :
	 fg = PEAR_inset_btn(pd,menu,"Show Item");
	 break;
      case 't' :
	 fg = PEAR_edit_replace(pd,menu,"Set Contents");
	 break;
      case 'T' :
	 fg = PEAR_edit_replace(pd,menu,"Copy Contents");
	 break;
      case 'v' :
	 fg = PEAR_edit_replace(pd,menu,"Set Value");
	 break;
      case 'V' :
	 fg = PEAR_edit_replace(pd,menu,"Copy Value");
	 break;
      case 'u' :
      case 'U' :
	 fg = PEAR_update_btn(pd,menu,"Update");
	 break;
      case 'z' :
	 fg = PEAR_zoom_btn(pd,menu,"Zoom In");
	 break;
      case 'Z' :
	 fg = PEAR_zoom_btn(pd,menu,"Zoom Out");
	 break;
      default :
	 fg = FALSE;
	 break;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	pear_control -- handle control messages on window		*/
/*									*/
/************************************************************************/


static int
pear_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register PEAR_DATA pd;

   ITRACE("pear_control %s 0x%x",msg,w);

   pd = (PEAR_DATA) ASHinq_user_data(w);
   if (pd == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      PEAR_finish(pd);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	vert_scroll -- handle vertical scroll request			*/
/*	hor_scroll -- handle horizontal scroll request			*/
/*									*/
/************************************************************************/


static void
vert_scroll(pd,dir,amt)
   PEAR_DATA pd;
   Integer dir;
   Integer amt;
{
   Integer vlx,vby,vrx,vty;
   Integer lx,by,rx,ty;
   register Integer dx;

   ITRACE("vert_scroll 0x%x %d %d",pd,dir,amt);

   if (!pd->havegelo) return;

   GELOwindow_inq_world(pd->gelowin,&lx,&by,&rx,&ty);
   GELOwindow_inq_view(pd->gelowin,&vlx,&vby,&vrx,&vty);

   if (dir > 0) {
      dx = (vby-vty)/3;
      if (dx < amt) dx = amt;
      if (vby+dx > by) dx = by-vby;
      vty += dx;
      vby += dx;
    }
   else if (dir < 0) {
      dx = (vby-vty)/3;
      if (dx < amt) dx = amt;
      if (vty-dx < ty) dx = vty-ty;
      vty -= dx;
      vby -= dx;
    }
   else {
      dx = vby-vty;
      if (amt < ty) amt = ty;
      else if (amt+dx > by) amt = by-dx;
      vty = amt;
      vby = vty+dx;
    }

   GELOwindow_set_view(pd->gelowin,vlx,vby,vrx,vty);

   PEAR_refresh(pd,FALSE,FALSE,FALSE);
};





static void
hor_scroll(pd,dir,amt)
   PEAR_DATA pd;
   Integer dir;
   Integer amt;
{
   Integer vlx,vby,vrx,vty;
   Integer lx,by,rx,ty;
   register Integer dx;

   ITRACE("hor_scroll 0x%x %d %d",pd,dir,amt);

   if (!pd->havegelo) return;

   GELOwindow_inq_world(pd->gelowin,&lx,&by,&rx,&ty);
   GELOwindow_inq_view(pd->gelowin,&vlx,&vby,&vrx,&vty);

   if (dir > 0) {
      dx = (vrx-vlx)/3;
      if (dx < amt) dx = amt;
      if (vrx+dx > rx) dx = rx-vrx;
      vlx += dx;
      vrx += dx;
    }
   else if (dir < 0) {
      dx = (vrx-vlx)/3;
      if (dx < amt) dx = amt;
      if (vlx-dx < lx) dx = vlx-lx;
      vlx -= dx;
      vrx -= dx;
    }
   else {
      dx = vrx-vlx;
      if (amt < lx) amt = lx;
      else if (amt+dx > rx) amt = rx-dx;
      vlx = amt;
      vrx = vlx+dx;
    }

   GELOwindow_set_view(pd->gelowin,vlx,vby,vrx,vty);

   PEAR_refresh(pd,FALSE,FALSE,FALSE);
};





/************************************************************************/
/*									*/
/*	applic_hit -- handle hit for application			*/
/*									*/
/************************************************************************/


static int
applic_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   register PEAR_DATA pd;
   register Boolean fg;

   ITRACE("applic_hit %d %d 0x%x 0x%x 0x%x",x,y,ch,btn,rgn);

   pd = (PEAR_DATA) RIPinq_data(rgn);
   if (pd == NULL) return FALSE;

   fg = PEAR_ext_hit(pd,x,y,ch,btn);

   return fg;
};





/* end of pearwin.c */


