/************************************************************************/
/*									*/
/*		pearmenu.c						*/
/*									*/
/*	Menu management for PEAR picture editor package 		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "pear_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define EDIT_MENU	"Edit"
#define LAYOUT_MENU	"Layout"
#define DISPLAY_MENU	"Display"
#define INSET_MENU	"Inset"
#define EVAL_MENU	"Eval"





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	int		heur_btn();
static	int		show_btn();
static	int		match_btn();
static	int		nochoice_btn();
static	int		eval_btn();
static	int		quit_btn();
static	int		reload_btn();
static	void		setup_inset();




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	STEM_PDM_DATA	pear_menu[] = {
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX, EDIT_MENU },
      { STEM_PSTATE_BTN, "Set Value", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Copy Value", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Set Contents", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Copy Contents", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Delete", PEAR_edit_remove },
      { STEM_PSTATE_BTN, "New Value", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Expand", PEAR_edit_replace },
      { STEM_PSTATE_BTN, "Make Current", PEAR_edit_select },
      { STEM_PSTATE_BTN, "Create", PEAR_edit_create },
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX, LAYOUT_MENU },
      { STEM_PSTATE_BTN, "Append", PEAR_edit_insert },
      { STEM_PSTATE_BTN, "Append Copy", PEAR_edit_insert },
      { STEM_PSTATE_BTN, "Insert", PEAR_edit_insert },
      { STEM_PSTATE_BTN, "Insert Copy", PEAR_edit_insert },
      { STEM_PSTATE_BTN, "Connect", PEAR_edit_arc },
      { STEM_PSTATE_BTN, "Connect New", PEAR_edit_arc },
      { STEM_PSTATE_BTN, "Remove", PEAR_edit_remove },
      { STEM_PSTATE_BTN, "Remove Arc", PEAR_edit_arc },
   { STEM_PSTATE_MENU, DISPLAY_MENU },
      { STEM_PSTATE_BTN, "Update", PEAR_update_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Emphasize", PEAR_emphasize_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Deemphasize", PEAR_emphasize_btn },
      { STEM_PSTATE_BTN, "Zoom In", PEAR_zoom_btn },
      { STEM_PSTATE_BTN, "Zoom Out", PEAR_zoom_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Heuristics", heur_btn },
      { STEM_PSTATE_BTN, "Show Item", show_btn },
      { STEM_PSTATE_BTN, "Show All", show_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_SIMPLE, "Set Value", PEAR_edit_replace },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Match", match_btn },
      { STEM_PSTATE_BTN, "Clear Choices", nochoice_btn },
   { STEM_PSTATE_MENU, INSET_MENU },
      { STEM_PSTATE_BTN, "Show Item", PEAR_inset_btn },
      { STEM_PSTATE_BTN, "Hide Inset", PEAR_inset_btn },
      { STEM_PSTATE_BTN, "Show Inset", PEAR_inset_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Clear", PEAR_inset_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Refresh", PEAR_inset_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "Resize", PEAR_inset_btn },
      { STEM_PSTATE_BTN|STEM_PSTATE_COMPLEX, "New Inset", PEAR_edit_create },
   { STEM_PSTATE_MENU|STEM_PSTATE_COMPLEX, EVAL_MENU },
      { STEM_PSTATE_BTN, "Evaluate", eval_btn },
      { STEM_PSTATE_BTN, "Inquire", eval_btn },
      { STEM_PSTATE_BTN, "Set Name", eval_btn },
      { STEM_PSTATE_BTN, "Fixup", PEAR_edit_update },
      { STEM_PSTATE_BTN, "Fixup Item", PEAR_edit_update },
      { STEM_PSTATE_BTN, "Reload Types", reload_btn },
   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	PEAR_menu_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
PEAR_menu_init()
{
   ITRACE("PEAR_menu_init");
};





/************************************************************************/
/*									*/
/*	PEARmenu_quit -- add quit to pear menu for window		*/
/*									*/
/************************************************************************/


void
PEARmenu_quit(w)
   ASH_WINDOW w;
{
   PEAR_DATA pd;

   pd = (PEAR_DATA) ASHinq_user_data(w);
   if (pd == NULL) return;
   if (pd->menuwin == NULL) return;
   STEMpdm_btn_add(pd->menuwin,EDIT_MENU,"Quit",quit_btn);
};





/************************************************************************/
/*									*/
/*	PEAR_menu_define -- define menu for apple window		*/
/*									*/
/************************************************************************/


void
PEAR_menu_define(pd)
   PEAR_DATA pd;
{
   ITRACE("PEAR_menu_define 0x%x",pd);

   if (pd->menuwin != NULL) {
      STEMpdm_define(pd->menuwin,pd,pear_menu);
    };
};





/************************************************************************/
/*									*/
/*	PEAR_update_btn -- handle user update request			*/
/*									*/
/************************************************************************/


int
PEAR_update_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   ITRACE("PEAR_update_btn 0x%x %s %s",pd,menu,btn);

   ASHinput_lock_makelong(pd->menuwin);

   PEAR_refresh(pd,TRUE,TRUE,TRUE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	PEAR_emphasize_btn -- handle user emphasize request		*/
/*									*/
/************************************************************************/


int
PEAR_emphasize_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   double mlt;
   GELO_OBJECT par,sel;
   ASH_WINDOW w;

   ITRACE("PEAR_emphasize_btn 0x%x %s %s",pd,menu,btn);

   pd->last_x = -1;

   sel = pd->selects[SELECT1];

   if (!pd->havegelo || sel == NULL) return FALSE;

   if (STREQL(btn,"Emphasize")) mlt = 2.000;
   else if (STREQL(btn,"Deemphasize")) mlt = 0.50;
   else mlt = 0.0;

   par = GELOinq_parent(sel);
   if (par == NULL) mlt = 0.0;

   if (mlt != 0) {
      GELOset_priority_x(sel,GELOinq_priority_x(sel)*mlt);
      GELOset_priority_y(sel,GELOinq_priority_y(sel)*mlt);

      w = (pd->selinsets[SELECT1] ? pd->insetwin : pd->gelowin);

      GELOwindow_resize_object(w,par);

      PEAR_refresh(pd,FALSE,FALSE,FALSE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	PEAR_zoom_btn -- handle user zoom request			*/
/*									*/
/************************************************************************/


int
PEAR_zoom_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   Integer lx,by,rx,ty;
   Integer rlx,rby,rrx,rty;
   Integer nlx,nby,nrx,nty;
   Integer d,dx;

   ITRACE("zoom_btn 0x%x %s %s",pd,menu,btn);

   pd->last_x = -1;

   if (!pd->havegelo) return FALSE;

   GELOwindow_inq_world(pd->gelowin,&rlx,&rby,&rrx,&rty);
   GELOwindow_inq_view(pd->gelowin,&lx,&by,&rx,&ty);

   if (STREQL(btn,"Zoom In")) {
      nlx = (3*lx+rx)/4;
      nrx = (lx+3*rx)/4;
      nty = (3*ty+by)/4;
      nby = (ty+3*by)/4;
    }
   else if (STREQL(btn,"Zoom Out")) {
      nlx = (3*lx-rx)/2;
      nrx = (3*rx-lx)/2;
      nty = (3*ty-by)/2;
      nby = (3*by-ty)/2;

      dx = (rlx < rrx ? 1 : -1);
      if ((nlx-rlx)*dx < 0) {
	 d = rlx - nlx;
	 nlx += d;
	 nrx += d;
       };
      if ((nrx-rrx)*dx > 0) {
	 d = rrx - nrx;
	 nlx += d;
	 nrx += d;
       };

      dx = (rby < rty ? 1 : -1);
      if ((nby-rby)*dx < 0) {
	 d = rby - nby;
	 nby += d;
	 nty += d;
       };
      if ((nty-rty)*dx > 0) {
	 d = rty - nty;
	 nby += d;
	 nty += d;
       };

      if (abs(lx-rx) >= abs(rlx-rrx)) {
	 nlx = rlx;
	 nrx = rrx;
	 nby = rby;
	 nty = rty;
       };
    }
   else {
      return FALSE;
    };

   GELOwindow_set_view(pd->gelowin,nlx,nby,nrx,nty);

   PEAR_set_scroll(pd);
   PEAR_refresh(pd,FALSE,FALSE,FALSE);


   return TRUE;
};





/************************************************************************/
/*									*/
/*	heur_btn -- handle user heuristics request			*/
/*									*/
/************************************************************************/


static int
heur_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   GELO_METHOD m;
   GELO_CONN_METHOD cm;
   Integer w;
   register GELO_OBJECT obj;
   register ASH_WINDOW aw;

   ITRACE("heur_btn 0x%x %s %s",pd,menu,btn);

   pd->last_x = -1;

   obj = pd->selects[SELECT1];
   aw = (pd->selinsets[SELECT1] ? pd->insetwin : pd->gelowin);
   if (obj == NULL) {
      obj = pd->geloobj;
      aw = pd->gelowin;
    };

   if (!pd->havegelo || obj == NULL) return FALSE;

   if (!GELOinq_layout_heuristics(obj,&m,&cm,&w)) return FALSE;
   if ((m & GELO_METHOD_SELECTABLE) == 0) return FALSE;

   if (!GELOrequest_layout_methods(pd->gelowin,&m,&cm,NULL,NULL,NULL,&w))
      return FALSE;

   GELOdefine_layout_white_space(obj,w);
   GELOdefine_layout_method(obj,m);
   GELOdefine_layout_conn_method(obj,cm);

   GELOwindow_resize_object(aw,obj);

   PEAR_refresh(pd,FALSE,FALSE,FALSE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	show_btn -- handle SHOW requests				*/
/*									*/
/************************************************************************/


static int
show_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   register GELO_OBJECT obj;
   Universal v,t;

   ITRACE("show_btn 0x%x %s %s",pd,menu,btn);

   pd->last_x = -1;

   if (!pd->havegelo) return FALSE;

   if (STREQL(btn,"Show Item")) {
      for (obj = pd->selects[SELECT1]; obj != NULL; obj = GELOinq_parent(obj)) {
	 if (PEAR_ext_gelo_value(obj,TRUE,&v,&t)) break;
       };
      if (obj == NULL) return FALSE;
      else {
	 pd->geloobj = obj;
	 pd->redraw = TRUE;
	 pd->selects[SELECT1] = NULL;
	 GELOlayout_validate(pd->geloobj,FALSE);
	 PEAR_refresh(pd,FALSE,TRUE,FALSE);
       };
    }
   else if (STREQL(btn,"Show All") && pd->topgeloobj != NULL) {
      pd->geloobj = pd->topgeloobj;
      pd->redraw = TRUE;
      pd->selects[SELECT1] = NULL;
      GELOlayout_validate(pd->geloobj,FALSE);
      PEAR_refresh(pd,FALSE,TRUE,FALSE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	match_btn -- handle MATCH string definition			*/
/*									*/
/************************************************************************/


static int
match_btn(pd,mnm,btn)
   PEAR_DATA pd;
   String mnm;
   String btn;
{
   Character buf[128];
   String menu[10];
   register Integer i;

   ITRACE("match_btn 0x%x %s %s",pd,mnm,btn);

   i = 0;
   menu[i++] = "%CSet Match String\n";
   menu[i++] = "%0.24t\n";
   menu[i++] = "   %a%M   %c";
   menu[i] = 0;

   if (pd->match == NULL) {
      buf[0] = 0;
    }
   else {
      strcpy(buf,pd->match);
    };

   if (!STEMdialog(pd->window,menu,buf)) return FALSE;

   if (buf[0] == 0 || buf[0] == ' ') pd->match = NULL;
   else pd->match = SALLOC(buf);

   PEAR_refresh(pd,TRUE,FALSE,FALSE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	nochoice_btn -- clear choices on display			*/
/*									*/
/************************************************************************/


static int
nochoice_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   register Integer i;

   ITRACE("nochoice_btn 0x%x %s %s",pd,menu,btn);

   for (i = 0; i < MAX_SELECT; ++i) {
      PEAR_select(pd,i,NULL,FALSE,FALSE,TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	PEAR_inset_btn -- handle inset commands 			*/
/*									*/
/************************************************************************/


int
PEAR_inset_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   register Boolean fg,rfg;
   Integer plx,pby;
   Integer lx,by,rx,ty;
   Integer nlx,nby,nrx,nty;

   ITRACE("PEAR_inset_btn 0x%x %s %s",pd,menu,btn);

   fg = TRUE;
   rfg = FALSE;

   if (STREQL(btn,"Show Inset")) {
      if (!pd->showinset) {
	 pd->showinset = TRUE;
	 ASHvisible(pd->insetwin,TRUE);
	 if (pd->inobject == NULL && pd->intype == NULL && pd->selects[SELECT1] != NULL) {
	    rfg = TRUE;
	    setup_inset(pd);
	  };
       };
    }
   else if (STREQL(btn,"Hide Inset")) {
      pd->showinset = FALSE;
      ASHvisible(pd->insetwin,FALSE);
      if (pd->in_depend && pd->in_changed && !PEAR_ext_inq_do_depend())
	 PEAR_refresh(pd,TRUE,FALSE,FALSE);
    }
   else if (STREQL(btn,"Show Item")) {
      if (!pd->showinset) {
	 pd->showinset = TRUE;
	 ASHvisible(pd->insetwin,TRUE);
       };
      if (pd->selects[SELECT1] != NULL) {
	 setup_inset(pd);
	 rfg = TRUE;
       };
    }
   else if (STREQL(btn,"Clear")) {
      pd->inobject = NULL;
      pd->intype = NULL;
      pd->insetredraw = TRUE;
      pd->in_depend = FALSE;
      pd->in_changed = FALSE;
      rfg = TRUE;
    }
   else if (STREQL(btn,"Refresh")) {
      rfg = TRUE;
    }
   else if (STREQL(btn,"Resize") && pd->insetwin != NULL) {
      RIPinput_box(pd->window,"Show size and location of inset window",
			   &lx,&by,&rx,&ty);
      if (abs(rx-lx) < 10 || abs(by-ty) < 10) fg = FALSE;
      plx = (lx < rx ? lx : rx);
      pby = (by < ty ? by : ty);
      rx = abs(rx-lx)-2;
      ty = abs(ty-by)-2;
      ASHinq_size(pd->window,ASH_SIZE_WINDOW,&nlx,&nby,&nrx,&nty);
      if (nlx < nrx) {
	 if (plx < nlx || plx+rx > nrx) fg = FALSE;
       }
      else {
	 plx = plx+rx+2;
	 if (plx > nlx || plx-rx < nrx) fg = FALSE;
       };
      if (nby < nty) {
	 if (pby < nby || pby+ty > nty) fg = FALSE;
       }
      else {
	 pby = pby+ty+2;
	 if (pby > nby || pby-ty < nty) fg = FALSE;
       };
      if (fg) ASHresize(pd->insetwin,plx,pby,0,0,rx,ty);
    }
   else fg = FALSE;

   if (fg) {
      PEAR_insetrefresh(pd,rfg);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	eval_btn -- handle evaluate requests				*/
/*									*/
/************************************************************************/


static int
eval_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   register GELO_OBJECT go;
   register Integer typ;
   register Universal obj;
   register Boolean fg;

   ITRACE("eval_btn 0x%x %s %s",pd,menu,btn);

   if (pd->selects[SELECT1] == NULL) {
      obj = pd->object;
    }
   else {
      for (go = pd->selects[SELECT1]; go != NULL; go = GELOinq_parent(go)) {
	 obj = GELOinq_user_structure(go);
	 if (obj != NULL) break;
       };
      if (go == NULL) btn = "BAD";
    };

   if (STREQL(btn,"Evaluate")) typ = 0;
   else if (STREQL(btn,"Inquire")) typ = 1;
   else if (STREQL(btn,"Set Name")) typ = 2;
   else typ = -1;

   if (typ >= 0) fg = PEAR_ext_evaluate(obj,typ);

   return fg;
};





/************************************************************************/
/*									*/
/*	quit_btn -- handle quit requests				*/
/*									*/
/************************************************************************/


static int
quit_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   ITRACE("quit_btn 0x%x %s %s",pd,menu,btn);

   ASHremove(pd->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	reload_btn -- handle reload type request			*/
/*									*/
/************************************************************************/


static int
reload_btn(pd,menu,btn)
   PEAR_DATA pd;
   String menu;
   String btn;
{
   ITRACE("reload_btn 0x%x %s %s",pd,menu,btn);

   APPLEsave_clear_all();

   return TRUE;
};





/************************************************************************/
/*									*/
/*	setup_inset -- set object and type for inset			*/
/*									*/
/************************************************************************/


static void
setup_inset(pd)
   PEAR_DATA pd;
{
   Universal v,t;
   register GELO_OBJECT obj;

   DTRACE("setup_inset 0x%x",pd);

   if (pd->selects[SELECT1] == NULL) {
      pd->inobject = NULL;
      pd->intype = NULL;
      pd->in_depend = FALSE;
      pd->in_changed = FALSE;
      return;
    };

   for (obj = pd->selects[SELECT1]; obj != NULL; obj = GELOinq_parent(obj)) {
      if (PEAR_ext_gelo_value(obj,TRUE,&v,&t)) break;
    };

   if (obj == NULL) {
      pd->inobject = NULL;
      pd->intype = NULL;
      pd->in_depend = FALSE;
      pd->in_changed = FALSE;
      return;
    };

   pd->inobject = v;
   pd->intype = t;
   pd->in_depend = TRUE;
   pd->in_changed = FALSE;
};





/* end of pearmenu.c */

