/************************************************************************/
/*									*/
/*		peardepend.c						*/
/*									*/
/*	Dependency maintenance for PEAR editor				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "pear_local.h"






/************************************************************************/
/*									*/
/*	Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _PEAR_DEPEND *	PEAR_DEPEND;


typedef struct _PEAR_DEPEND {
   PEAR_DATA	pear_data;
   GELO_OBJECT	gelo_object;
   Universal	user_object;
   Boolean	insetfg;
} PEAR_DEPEND_B;





/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/


#define MAX_WINDOWS	32
#define MAX_DO_DEPEND	32

#define HASH_SIZE	883
#define HASH_FCT(u)	(abs((u)%HASH_SIZE))

static	Sequence	pear_depend_table[HASH_SIZE];
static	PEAR_DEPEND	free_depend_list;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		pear_depend_notify();
static	void		fix_update();
static	Integer 	relate_obj();





/************************************************************************/
/*									*/
/*	PEAR_depend_init -- module initialization			*/
/*									*/
/************************************************************************/


void
PEAR_depend_init()
{
   register Integer i;

   ITRACE("PEAR_depend_init");

   for (i = 0; i < HASH_SIZE; ++i) pear_depend_table[i] = NULL;
   free_depend_list = NULL;

   GELOset_notify_fct(pear_depend_notify);
};





/************************************************************************/
/*									*/
/*	PEARdepend_update -- update any windows dependent on object	*/
/*									*/
/************************************************************************/


Boolean
PEARdepend_update(uo)
   Universal uo;
{
   register PEAR_DEPEND dep;
   register Sequence l;
   register Integer ct,i,dct;
   register Integer hash;
   PEAR_DEPEND dodep[MAX_DO_DEPEND];

   TRACE("PEARdepend_update 0x%x",uo);

   ct = 0;
   dct = 0;
   hash = HASH_FCT(uo);

   PROTECT;

   forin (dep,PEAR_DEPEND,l,pear_depend_table[hash]) {
      if (uo == dep->user_object) {
	 ++ct;
	 if (!dep->pear_data->update_delay) {
	    if (PEAR_ext_inq_cmd_update(dep->pear_data)) {
	       PEAR_update_begin(dep->pear_data);
	     };
	  };
	 if (!dep->pear_data->update_delay) {
	    dodep[dct++] = dep;
	  }
	 else {
	    fix_update(dep);
	  };
       };
    };

   UNPROTECT;

   for (i = 0; i < dct; ++i) {
      dep = dodep[i];
      PEAR_update_obj(dep->pear_data,dep->gelo_object,dep->insetfg);
    };

   return (ct != 0);
};





/************************************************************************/
/*									*/
/*	PEARdepend_hilite -- hilite chosen object			*/
/*									*/
/************************************************************************/


void
PEARdepend_hilite(uo,w,style,fg)
   Universal uo;
   ASH_WINDOW w;
   Integer style;
   Boolean fg;
{
   register PEAR_DEPEND dep;
   register Sequence l;
   register Integer hash;
   register PEAR_DATA pd;

   TRACE("PEARdepend_hilite 0x%x 0x%x %d",uo,w,style);

   if (w == NULL) pd = NULL;

   ASHbatch_mode(TRUE);

   if (uo == NULL) {
      if (pd != NULL) PEAR_select(pd,style,NULL,FALSE,FALSE,FALSE);
      else {
	 forin (pd,PEAR_DATA,l,PEAR__all_pear) {
	    PEAR_select(pd,style,NULL,FALSE,FALSE,FALSE);
	  };
       };
    }
   else {
      hash = HASH_FCT(uo);

      forin (dep,PEAR_DEPEND,l,pear_depend_table[hash]) {
	 if (uo == dep->user_object && (pd == NULL || pd == dep->pear_data)) {
	    if (fg) {
	       PEAR_select(dep->pear_data,style,dep->gelo_object,dep->insetfg,
			      FALSE,TRUE);
	     }
	    else {
	       PEAR_select(dep->pear_data,style,NULL,dep->insetfg,FALSE,FALSE);
	     };
	    if (pd != NULL) break;
	  };
       };
    };

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	pear_depend_notify -- handle GELO notifications 		*/
/*									*/
/************************************************************************/


static void
pear_depend_notify(pd,go,uo,w,newfg)
   PEAR_DATA pd;
   GELO_OBJECT go;
   Universal uo;
   ASH_WINDOW w;
   Boolean newfg;
{
   register PEAR_DEPEND dep,ndep;
   register Sequence l;
   register Integer ct;
   register Integer hash;
   register Boolean insetfg;
   register Boolean fg;

   ITRACE("pear_depend_notify 0x%x 0x%x 0x%x %d",pd,go,uo,newfg);

   if (pd->gelowin == w) insetfg = FALSE;
   else if (pd->insetwin == w) insetfg = TRUE;
   else return;

   hash = HASH_FCT(uo);
   dep = NULL;
   ct = 0;

   PROTECT;

   forin (ndep,PEAR_DEPEND,l,pear_depend_table[hash]) {
      if (uo == ndep->user_object) {
	 ++ct;
	 if (pd == ndep->pear_data && go == ndep->gelo_object &&
		insetfg == ndep->insetfg)
	    dep = ndep;
       };
    };

   if (newfg) {
      if (ct == 0) fg = PEAR_ext_depend_notify(pd,uo,TRUE);
      else fg = TRUE;
      if (fg && dep == NULL) {
	 if (free_depend_list == NULL) {
	    dep = (PEAR_DEPEND) malloc(sizeof(PEAR_DEPEND_B));
	  }
	 else {
	    dep = free_depend_list;
	    free_depend_list = (PEAR_DEPEND) dep->pear_data;
	  };
	 dep->pear_data = pd;
	 dep->gelo_object = go;
	 dep->user_object = uo;
	 dep->insetfg = insetfg;
	 pear_depend_table[hash] = CONS(dep,pear_depend_table[hash]);
       };
    }
   else {
      if (dep != NULL) {
	 if (ct == 1) PEAR_ext_depend_notify(pd,uo,FALSE);
	 pear_depend_table[hash] = REMOB(dep,pear_depend_table[hash]);
	 dep->pear_data = (PEAR_DATA) free_depend_list;
	 free_depend_list = dep;
       };
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	fix_update -- fixup for delayed update				*/
/*	relate_obj -- relate two objects				*/
/*									*/
/************************************************************************/


static void
fix_update(dep)
   PEAR_DEPEND dep;
{
   register PEAR_DATA pd;
   register Integer i;

   DTRACE("fix_update 0x%x",dep);

   pd = dep->pear_data;

   if (dep->insetfg) {
      if (pd->update_selall) return;
      i = relate_obj(pd->update_selobj,dep->gelo_object);
      if (i <= 0) return;
      else if (i == 1) pd->update_selobj = dep->gelo_object;
      else pd->update_selall = TRUE;
    }
   else {
      if (pd->update_all) return;
      i = relate_obj(pd->update_obj,dep->gelo_object);
      if (i <= 0) return;
      else if (i == 1) pd->update_obj = dep->gelo_object;
      else pd->update_all = TRUE;
    };
};





static Integer
relate_obj(o1,o2)
   GELO_OBJECT o1,o2;
{
   register GELO_OBJECT o;

   DTRACE("relate_obj 0x%x 0x%x",o1,o2);

   if (o1 == o2) return 0;
   if (o1 == NULL) return 1;
   if (o2 == NULL) return -1;

   for (o = GELOinq_parent(o1); o != NULL; o = GELOinq_parent(o)) {
      if (o == o2) return 1;
    };

   for (o = GELOinq_parent(o2); o != NULL; o = GELOinq_parent(o)) {
      if (o == o1) return -1;
    };

   return 2;
};






/* end of peardepend.c */
