/************************************************************************/
/*									*/
/*		applepic.c						*/
/*									*/
/*	Picture management for APPLE type picture editor package	*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_WIDTH	8
#define NULL_FILL	3
#define NULL_SHAPE	GELO_SHAPE_TRIANGLE_UP
#define DEFAULT_SHRINK	90




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Boolean 		simple;


/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void			fixup_layout_obj();
static	void			fixup_tiled_obj();
static	void			get_fields();
static	void			setup_picture();
static	APPLE_GELO		new_gelo_object();
static	APPLE_GELO_COMPONENT	new_component();
static	void			enlarge_layout();
static	Boolean 		intersect_component();





/************************************************************************/
/*									*/
/*	APPLE_pic_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
APPLE_pic_init()
{
   String s;

   ITRACE("APPLE_pic_init");

   s = ASHinq_resource("simple");
   if (s == NULL || STREQL(s,"off")) simple = FALSE;
   else simple = TRUE;
};





/************************************************************************/
/*									*/
/*	APPLE_get_picture -- get existing saved picture 		*/
/*									*/
/************************************************************************/


void
APPLE_get_picture(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_get_picture 0x%x",ad);

   if (ad->first_picture != NULL) APPLE_free_picture_data(ad);

   get_fields(ad);

   ad->picture = APPLE_ext_load_picture(ad->handle,NULL,FALSE);
   ad->first_picture = ad->picture;

   APPLE_set_picture_mode(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_default_picture -- get default picture			*/
/*									*/
/************************************************************************/


void
APPLE_default_picture(ad)
   APPLE_DATA ad;
{
   APPLE_GELO_OBJECT ago,la,lb;

   ITRACE("APPLE_default_picture 0x%x",ad);

   APPLE_new_picture(ad,APPLE_MODE_NONE);

   while (ad->grid < ad->numfield+2 || ad->grid < DEFAULT_WIDTH)
      ad->grid *= 2;
   ago = APPLE_default_pattern(ad->handle,NULL);

   ad->picture->object = ago->object;

   ago->object = NULL;
   if (ago->alternative != NULL) {
      ad->picture->cond_layout = ago->cond_layout;
      ad->picture->cond_array = ago->cond_array;
      ad->picture->cond_nested = ago->cond_nested;
      la = ad->picture->alternative;
      ad->picture->alternative = ago->alternative;
      ago->alternative = NULL;
      for (lb = ad->picture; lb->alternative != NULL; lb = lb->alternative);
      lb->alternative = la;
    };

   APPLEfree_picture(ago);

   APPLE_set_picture_mode(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_new_picture -- set up a new picture for item		*/
/*									*/
/************************************************************************/


void
APPLE_new_picture(ad,md)
   APPLE_DATA ad;
   APPLE_MODE md;
{
   ITRACE("APPLE_new_picture 0x%x %d",ad,md);

   APPLE_ext_notify(ad->handle);

   if (ad->picture != NULL && ad->picture->object != NULL) {
      APPLE_free_picture_elt(ad->picture->object);
      ad->picture->object = NULL;
      ad->picture->layout_obj = NULL;
      if (ad->picture->use != NULL) LFREE(ad->picture->use);
      ad->picture->use = NULL;
      if (ad->picture->usearc != NULL) LFREE(ad->picture->usearc);
      ad->picture->usearc = NULL;
    };

   get_fields(ad);

   if (ad->picture == NULL) {
      ad->picture = PALLOC(APPLE_GELO_OBJECT_B);
      ad->picture->cond_fct = NULL;
      ad->picture->cond_match = NULL;
      ad->picture->cond_null = FALSE;
      ad->picture->cond_layout = FALSE;
      ad->picture->cond_array = FALSE;
      ad->picture->cond_nested = FALSE;
      ad->picture->alternative = NULL;
      ad->picture->object = NULL;
      ad->picture->use = NULL;
      ad->picture->nodelete = FALSE;
      ad->picture->inuse = 0;
      ad->picture->layout_obj = NULL;
      setup_picture(ad);
    };

   ad->picture->object = new_gelo_object(md);

   if (md == APPLE_MODE_ARC) ad->picture->cond_layout = TRUE;

   APPLE_set_picture_mode(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_save_picture -- save current picture			*/
/*									*/
/************************************************************************/


void
APPLE_save_picture(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_save_picture 0x%x",ad);

   if (ad->picture != NULL) {
      APPLE_ext_save_picture(ad->handle,ad->first_picture);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_remove_alternative -- remove current alternative		*/
/*									*/
/************************************************************************/


void
APPLE_remove_alternative(ad)
   APPLE_DATA ad;
{
   register APPLE_GELO_OBJECT ao;

   ITRACE("APPLE_remove_alternative 0x%x",ad);

   if (ad->picture == NULL) return;

   APPLE_ext_notify(ad->handle);

   if (ad->picture == ad->first_picture) {
      ad->first_picture = ad->picture->alternative;
      ad->prev_picture = NULL;
    }
   else if (ad->prev_picture != NULL &&
	       ad->prev_picture->alternative == ad->picture) {
      ad->prev_picture->alternative = ad->picture->alternative;
    }
   else {
      for (ao = ad->first_picture;
	   ao->alternative != NULL && ao->alternative != ad->picture;
	   ao = ao->alternative);
      if (ao->alternative == ad->picture) {
	 ad->prev_picture = ao;
	 ao->alternative = ad->picture->alternative;
       };
    };

   APPLE_free_alternative(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_set_type -- set type of component 			*/
/*									*/
/************************************************************************/


void
APPLE_set_type(ad,blk,md)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT blk;
   APPLE_MODE md;
{
   ITRACE("APPLE_set_type 0x%x 0x%x %d",ad,blk,md);

   APPLE_ext_notify(ad->handle);

   if (blk->object != NULL) {
      if (blk->object->mode == md) return;
      APPLE_free_picture_elt(blk->object);
      blk->object = NULL;
    };

   if (md != APPLE_MODE_NONE) {
      blk->object = new_gelo_object(md);
    };

   fixup_layout_obj(ad->picture);
};





/************************************************************************/
/*									*/
/*	APPLE_new_field -- add new field to list of fields		*/
/*									*/
/************************************************************************/


Integer
APPLE_new_field(ad,name)
   APPLE_DATA ad;
   String name;
{
   register Integer i;
   register Boolean fg;

   ITRACE("APPLE_new_field %s",name);

   i = ad->numfield++;

   ad->fields[i].name = name;
   ad->fields[i].value = 0;

   fg = APPLE_ext_new_field(ad->handle,&ad->fields[i]);

   if (!fg) {
      -- ad->numfield;
      i = -1;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	APPLE_change_grid -- change grid size				*/
/*									*/
/************************************************************************/


Boolean
APPLE_change_grid(ad,v)
   APPLE_DATA ad;
   Integer v;
{
   register APPLE_GELO_TILED til;
   register Sequence oldcl,l;
   register APPLE_GELO_COMPONENT c,c1;
   register Integer oldg;
   Integer lx,by,rx,ty;

   ITRACE("APPLE_change_grid 0x%x %d",ad,v);

   if (v < MIN_GRID_SIZE) return FALSE;
   if (v > MAX_GRID_SIZE) return FALSE;

   til = (APPLE_GELO_TILED) ad->picture->object;
   oldcl = til->components;
   oldg = ad->grid;
   til->components = NULL;
   til->max_x = 0;
   til->max_y = 0;
   ad->grid = v;

   forin (c,APPLE_GELO_COMPONENT,l,oldcl) {
      lx = (c->lx*v)/oldg;
      rx = (c->rx*v)/oldg;
      ty = (c->ty*v)/oldg;
      by = (c->by*v)/oldg;
      c1 = APPLE_add_component(ad,lx,by,rx,ty);
      if (c1 != NULL) {
	 c1->fix_x = c->fix_x;
	 c1->fix_y = c->fix_y;
	 c1->shrinkage = c->shrinkage;
	 c1->object = c->object;
	 c->object = NULL;
       };
      APPLE_free_component(c);
    };

   LFREE(oldcl);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLE_add_component -- add component to layout			*/
/*									*/
/************************************************************************/


APPLE_GELO_COMPONENT
APPLE_add_component(ad,lx,by,rx,ty)
   APPLE_DATA ad;
   Integer lx,by,rx,ty;
{
   register APPLE_GELO_COMPONENT c,ca;
   register APPLE_GELO_TILED til;
   register Sequence l,ldel;

   ITRACE("APPLE_add_component 0x%x %d %d %d %d",ad,lx,by,rx,ty);

   if (lx == rx || by == ty) return NULL;

   til = (APPLE_GELO_TILED) ad->picture->object;
   if (til->mode != APPLE_MODE_TILED) return NULL;

   APPLE_ext_notify(ad->handle);

   c = new_component(lx,by,rx,ty);

   enlarge_layout(ad,c->rx,c->by);

   ldel = NULL;
   forin (ca,APPLE_GELO_COMPONENT,l,til->components) {
      if (!intersect_component(ad,ca,c)) ldel = CONS(ca,ldel);
    };
   if (ldel != NULL) {
      forin (ca,APPLE_GELO_COMPONENT,l,ldel) {
	 til->components = REMOB(ca,til->components);
	 APPLE_free_component(ca);
       };
      LFREE(ldel);
    };

   til->components = APPEND(c,til->components);

   fixup_layout_obj(ad->picture);
   fixup_tiled_obj(ad->picture);

   return c;
};





/************************************************************************/
/*									*/
/*	APPLE_change_component -- modify existing component		*/
/*									*/
/************************************************************************/


Boolean
APPLE_change_component(ad,c,lx,by,rx,ty)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT c;
   Integer lx,by,rx,ty;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_COMPONENT ca;
   register Sequence l,ldel;

   ITRACE("APPLE_change_component 0x%x 0x%x %d %d %d %d",ad,c,lx,by,rx,ty);

   til = (APPLE_GELO_TILED) ad->picture->object;
   if (til->mode != APPLE_MODE_TILED) return NULL;

   if (lx == rx || by == ty) return NULL;

   APPLE_ext_notify(ad->handle);

   ca = new_component(c->lx,c->by,c->rx,c->ty);

   c->lx = MIN(lx,rx);
   c->by = MAX(by,ty);
   c->rx = MAX(lx,rx);
   c->ty = MIN(by,ty);
   c->expand_x = abs(rx-lx);
   c->expand_y = abs(ty-by);
   c->shrinkage = DEFAULT_SHRINK;

   enlarge_layout(ad,c->rx,c->by);

   if (intersect_component(ad,ca,c)) {
      til->components = APPEND(ca,til->components);
    }
   else {
      free(ca);
    };

   ldel = NULL;
   forin (ca,APPLE_GELO_COMPONENT,l,til->components) {
      if (ca != c && !intersect_component(ad,ca,c)) ldel = CONS(ca,ldel);
    };
   if (ldel != NULL) {
      forin (ca,APPLE_GELO_COMPONENT,l,ldel) {
	 til->components = REMOB(ca,til->components);
	 APPLE_free_component(ca);
       };
      LFREE(ldel);
    };

   fixup_layout_obj(ad->picture);
   fixup_tiled_obj(ad->picture);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLE_set_picture_mode -- set mode for current picture		*/
/*									*/
/************************************************************************/


void
APPLE_set_picture_mode(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_set_picture_mode 0x%x",ad);

   if (ad->picture == NULL || ad->picture->object == NULL) {
      ad->mode = APPLE_MODE_NONE;
    }
   else {
      ad->mode = ad->picture->object->mode;
    };

   ad->selection = NULL;
   ad->use_select = USE_SELECT_NONE;
   ad->usearc_select = NULL;

   fixup_layout_obj(ad->picture);
};





/************************************************************************/
/*									*/
/*	APPLE_reget_fields -- get fields again				*/
/*									*/
/************************************************************************/


void
APPLE_reget_fields(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_reget_fields 0x%x",ad);

   get_fields(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_default_pattern -- build a default pattern based on fields*/
/*									*/
/************************************************************************/


APPLE_GELO_OBJECT
APPLE_default_pattern(h,typ)
   Universal h;
   Universal typ;
{
   register Integer i;
   register APPLE_GELO_OBJECT ago;
   register APPLE_GELO_TILED til;
   register APPLE_GELO_COMPONENT c;
   register APPLE_GELO_FIELD fld;
   register APPLE_GELO_BOX box;
   register APPLE_GELO_LAYOUT lay;
   register APPLE_VALUE val;
   APPLE_GELO_ARC ptr;
   APPLE_GELO_OBJECT alt;
   Integer nfld;
   APPLE_VALUE_B flds[MAX_FIELD];
   Boolean pfg,indfg,arrfg;

   ITRACE("APPLE_default_pattern 0x%x 0x%x",h,typ);

   nfld = APPLE_ext_inq_fields(h,typ,FALSE,MAX_FIELD,flds);

   arrfg = FALSE;
   pfg = FALSE;
   for (i = 0; i < nfld && !pfg; ++i)
      pfg = APPLE_ext_inq_pointer(NULL,h,typ,&flds[i]);

   indfg = !pfg;

   ago = PALLOC(APPLE_GELO_OBJECT_B);
   ago->cond_fct = NULL;
   ago->cond_match = NULL;
   ago->cond_null = FALSE;
   ago->cond_layout = FALSE;
   ago->cond_array = FALSE;
   ago->cond_nested = FALSE;
   ago->alternative = NULL;
   ago->nodelete = FALSE;
   ago->inuse = 0;
   ago->use = NULL;
   ago->layout_obj = NULL;

   if (nfld > 0) {
      til = (APPLE_GELO_TILED) new_gelo_object(APPLE_MODE_TILED);
      ago->object = (APPLE_GELO) til;

      c = new_component(0,1,DEFAULT_WIDTH,0);
      til->components = APPEND(c,til->components);
      box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
      c->object = (APPLE_GELO) box;
      c->fix_y = TRUE;
      box->value.name = APPLE_ext_type_name(h,typ);
      APPLE_ext_string(&box->value);

      for (i = 0; i < nfld; ++i) {
	 c = new_component((indfg ? 1 : 0),(i+2),DEFAULT_WIDTH,(i+1));
	 til->components = APPEND(c,til->components);
	 if (pfg && APPLE_ext_inq_pointer(NULL,h,typ,&flds[i])) {
	    ptr = (APPLE_GELO_ARC) new_gelo_object(APPLE_MODE_POINTER);
	    c->object = (APPLE_GELO) ptr;
	    ptr->to = flds[i];
	  }
	 else {
	    fld = (APPLE_GELO_FIELD) new_gelo_object(APPLE_MODE_FIELD);
	    c->object = (APPLE_GELO) fld;
	    fld->value = flds[i];
	  };
       };

      if (indfg) {
	 c = new_component(0,nfld+1,1,1);
	 til->components = APPEND(c,til->components);
	 box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
	 c->object = (APPLE_GELO) box;
/*	 c->fix_x = TRUE;			*/
	 box->value.name = NULL;
	 box->value.value = NULL;
       };

      til->max_x = DEFAULT_WIDTH;
      til->max_y = nfld+1;
      til->boxed = TRUE;

      if (pfg) {
	 ago->cond_layout = TRUE;
	 alt = PALLOC(APPLE_GELO_OBJECT_B);
	 ago->alternative = alt;
	 alt->cond_fct = NULL;
	 alt->cond_match = NULL;
	 alt->cond_null = FALSE;
	 alt->cond_layout = FALSE;
	 alt->cond_array = FALSE;
	 alt->cond_nested = FALSE;
	 alt->alternative = NULL;
	 alt->nodelete = FALSE;
	 alt->inuse = 0;
	 alt->use = NULL;
	 alt->layout_obj = NULL;

	 lay = (APPLE_GELO_LAYOUT) new_gelo_object(APPLE_MODE_LAYOUT);
	 alt->object = (APPLE_GELO) lay;
	 val = PALLOC(APPLE_VALUE_B);
	 val->name = APPLE_SELF_NAME;
	 val->value = 0;
	 lay->components = APPEND(APPLE_new_usefield(val),lay->components);
       };
    }
   else if (APPLE_ext_inq_use_contents(NULL,h,typ)) {
      arrfg = TRUE;
      lay = (APPLE_GELO_LAYOUT) new_gelo_object(APPLE_MODE_LAYOUT);
      ago->object = (APPLE_GELO) lay;
      val = PALLOC(APPLE_VALUE_B);
      val->name = APPLE_SELF_NAME;
      val->value = 0;
      lay->components = APPEND(APPLE_new_usefield(val),lay->components);
      lay->unique = FALSE;
      lay->simple = TRUE;
    }
   else if (typ == NULL && h == NULL) {
      box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
      box->fill = NULL_FILL;
      box->linestyle = 0;
      box->font = 0;
      box->shape = NULL_SHAPE;
      box->value.name = NULL;
      box->value.value = 0;
      ago->object = (APPLE_GELO) box;
    }
   else {
      box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
      box->value.name = APPLE_SELF_NAME;
      box->value.value = 0;
      ago->object = (APPLE_GELO) box;
    };

   if (simple && !arrfg) {
      alt = PALLOC(APPLE_GELO_OBJECT_B);
      alt->alternative = ago;
      alt->cond_fct = NULL;
      alt->cond_match = NULL;
      alt->cond_null = FALSE;
      alt->cond_layout = FALSE;
      alt->cond_array = FALSE;
      alt->cond_nested = TRUE;
      alt->nodelete = FALSE;
      alt->inuse = 0;
      alt->use = NULL;
      alt->layout_obj = NULL;

      box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
      box->value.name = APPLE_SELF_NAME;
      box->value.value = 0;
      alt->object = (APPLE_GELO) box;

      ago = alt;
    };

   alt = PALLOC(APPLE_GELO_OBJECT_B);
   alt->alternative = ago;
   alt->cond_fct = NULL;
   alt->cond_match = NULL;
   alt->cond_null = FALSE;
   alt->cond_layout = FALSE;
   alt->cond_array = TRUE;
   alt->cond_nested = FALSE;
   alt->nodelete = FALSE;
   alt->inuse = 0;
   alt->use = NULL;
   alt->layout_obj = NULL;

   til = (APPLE_GELO_TILED) new_gelo_object(APPLE_MODE_TILED);
   til->max_x = DEFAULT_WIDTH;
   til->max_y = 4;
   til->boxed = FALSE;
   alt->object = (APPLE_GELO) til;

   c = new_component(0,1,DEFAULT_WIDTH,0);
   til->components = APPEND(c,til->components);
   box = (APPLE_GELO_BOX) new_gelo_object(APPLE_MODE_BOX);
   box->shape = GELO_SHAPE_NONE;
   c->object = (APPLE_GELO) box;
   c->fix_y = TRUE;
   box->value.name = APPLE_INDEX_NAME;
   box->value.value = 0;

   c = new_component(0,4,DEFAULT_WIDTH,1);
   til->components = APPEND(c,til->components);
   fld = (APPLE_GELO_FIELD) new_gelo_object(APPLE_MODE_FIELD);
   c->object = (APPLE_GELO) fld;
   fld->value.name = APPLE_SELF_NAME;
   fld->value.value = 0;

   ago = alt;

   APPLE_fixup_load(ago);

   return ago;
};





/************************************************************************/
/*									*/
/*	APPLE_new_usefield -- create field for USE			*/
/*	APPLE_new_usearc -- create arc for USE				*/
/*	APPLE_layout_value -- return value from USE field/arc		*/
/*									*/
/************************************************************************/


APPLE_GELO
APPLE_new_usefield(v)
   APPLE_VALUE v;
{
   register APPLE_GELO_FIELD fld;

   ITRACE("APPLE_new_usefield 0x%x",v);

   fld = (APPLE_GELO_FIELD) new_gelo_object(APPLE_MODE_FIELD);
   fld->value = *v;

   return (APPLE_GELO) fld;
};





APPLE_GELO
APPLE_new_usearc(v,tofg)
   APPLE_VALUE v;
   Boolean tofg;
{
   register APPLE_GELO_ARC arc;

   ITRACE("APPLE_new_usearc 0x%x %d",v,tofg);

   arc = (APPLE_GELO_ARC) new_gelo_object(APPLE_MODE_ARC);

   if (tofg) {
      arc->to = *v;
      arc->from.name = APPLE_SELF_NAME;
    }
   else {
      arc->from = *v;
      arc->to.name = APPLE_SELF_NAME;
    };

   return (APPLE_GELO) arc;
};





APPLE_VALUE
APPLE_layout_value(ag)
   APPLE_GELO ag;
{
   register APPLE_GELO_FIELD fld;
   register APPLE_GELO_ARC arc;
   register APPLE_VALUE v;

   ITRACE("APPLE_layout_value 0x%x",ag);

   if (ag == NULL) return NULL;

   switch (ag->mode) {
      case APPLE_MODE_FIELD :
	 fld = (APPLE_GELO_FIELD) ag;
	 v = &(fld->value);
	 break;
      case APPLE_MODE_ARC :
	 arc = (APPLE_GELO_ARC) ag;
	 if (STREQL(arc->from.name,APPLE_SELF_NAME)) v = &(arc->to);
	 else if (STREQL(arc->to.name,APPLE_SELF_NAME)) v = &(arc->from);
	 else v = NULL;
	 break;
      default :
	 v = NULL;
	 break;
    };

   return v;
};





/************************************************************************/
/*									*/
/*	APPLE_tiled_connect -- add connection in a tiling		*/
/*	APPLE_tiled_conenct_remove -- remove connection 		*/
/*									*/
/************************************************************************/


APPLE_GELO_TARC
APPLE_tiled_connect(ad,from,to,fport,tport,style,arrow)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT from;
   APPLE_GELO_COMPONENT to;
   GELO_PORT fport;
   GELO_PORT tport;
   ASH_LINE_STYLE style;
   GELO_ARC_ARROW arrow;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_TARC arc;

   ITRACE("APPLE_tiled_connect 0x%x 0x%x 0x%x %d %d %d %d",
	     ad,from,to,fport,tport,style,arrow);

   til = (APPLE_GELO_TILED) ad->picture->object;

   arc = PALLOC(APPLE_GELO_TARC_B);
   arc->from = from;
   arc->to = to;
   arc->fport = fport;
   arc->tport = tport;
   arc->style = style;
   arc->arrow = arrow;

   til->arcs = APPEND(arc,til->arcs);

   return arc;
};




void
APPLE_tiled_connect_remove(ad,arc)
   APPLE_DATA ad;
   APPLE_GELO_TARC arc;
{
   register APPLE_GELO_TILED til;

   ITRACE("APPLE_tiled_connect_remove 0x%x 0x%x",ad,arc);

   til = (APPLE_GELO_TILED) ad->picture->object;

   til->arcs = REMOB(arc,til->arcs);
};





/************************************************************************/
/*									*/
/*	APPLE_tiled_constraint -- add constraint to a tiling		*/
/*	APPLE_tiled_constraint_remove -- remove constraint		*/
/*									*/
/************************************************************************/


APPLE_GELO_TCONST
APPLE_tiled_constraint(ad,from,to,fromy,toy,mult,add)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT from;
   APPLE_GELO_COMPONENT to;
   Boolean fromy;
   Boolean toy;
   Float mult;
   Float add;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_TCONST cnst;

   ITRACE("APPLE_tiled_constraint 0x%x 0x%x 0x%x %d %d %f %f",
	     ad,from,to,fromy,toy,mult,add);

   til = (APPLE_GELO_TILED) ad->picture->object;

   cnst = PALLOC(APPLE_GELO_TCONST_B);
   cnst->from = from;
   cnst->to = to;
   cnst->from_y = fromy;
   cnst->to_y = toy;
   cnst->multiplier = mult;
   cnst->addend = add;

   til->constraints = APPEND(cnst,til->constraints);

   return cnst;
};




void
APPLE_tiled_constraint_remove(ad,cnst)
   APPLE_DATA ad;
   APPLE_GELO_TCONST cnst;
{
   register APPLE_GELO_TILED til;

   ITRACE("APPLE_tiled_constraint_remove 0x%x 0x%x",ad,cnst);

   til = (APPLE_GELO_TILED) ad->picture->object;

   til->constraints = REMOB(cnst,til->constraints);
};





/************************************************************************/
/*									*/
/*	APPLE_fixup_load -- fixup APPLE_GELO_OBJECT after load		*/
/*									*/
/************************************************************************/


void
APPLE_fixup_load(ago)
   APPLE_GELO_OBJECT ago;
{
   register APPLE_GELO_OBJECT a1;

   ITRACE("APPLE_fixup_load 0x%x",ago);

   if (ago == NULL || ago->object == NULL) return;

   for (a1 = ago; a1 != NULL; a1 = a1->alternative) {
      fixup_layout_obj(a1);
      fixup_tiled_obj(a1);
    };
};





/************************************************************************/
/*									*/
/*	APPLE_new_object -- create new object of given mode		*/
/*									*/
/************************************************************************/


APPLE_GELO
APPLE_new_object(md)
   APPLE_MODE md;
{
   return new_gelo_object(md);
};





/************************************************************************/
/*									*/
/*	fixup_layout_obj -- fixup selection of layout_obj as needed	*/
/*									*/
/************************************************************************/


static void
fixup_layout_obj(ago)
   APPLE_GELO_OBJECT ago;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_COMPONENT c;
   register Sequence l;

   DTRACE("fixup_layout_obj 0x%x",ago);

   if (ago == NULL) return;

   ago->layout_obj = NULL;

   if (ago->object == NULL) return;

   if (ago->object->mode == APPLE_MODE_LAYOUT) {
      ago->layout_obj = ago->object;
    }
   else if (ago->object->mode == APPLE_MODE_TILED) {
      til = (APPLE_GELO_TILED) ago->object;
      forin (c,APPLE_GELO_COMPONENT,l,til->components) {
	 if (c->object != NULL && c->object->mode == APPLE_MODE_LAYOUT) {
	    ago->layout_obj = c->object;
	    break;
	  };
       };
    }
   else {
      ago->layout_obj = NULL;
    };
};





/************************************************************************/
/*									*/
/*	fixup_tiled_obj -- fixup tiling after object changes		*/
/*									*/
/************************************************************************/


static void
fixup_tiled_obj(ago)
   APPLE_GELO_OBJECT ago;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_TARC arc;
   register APPLE_GELO_TCONST cnst;
   register Sequence l;

   DTRACE("fixup_tiled_obj 0x%x",ago);

   if (ago == NULL || ago->object == NULL ||
	  ago->object->mode != APPLE_MODE_TILED)
      return;

   til = (APPLE_GELO_TILED) ago->object;

   do {
      forin (arc,APPLE_GELO_TARC,l,til->arcs) {
	 if (!MEMQ(arc->from,til->components) ||
		!MEMQ(arc->to,til->components)) {
	    til->arcs = REMOB(arc,til->arcs);
	    free(arc);
	    break;
	  };
       };
    }
   while (!EMPTY(l));

   do {
      forin (cnst,APPLE_GELO_TCONST,l,til->constraints) {
	 if (!MEMQ(cnst->from,til->components) ||
		!MEMQ(cnst->to,til->components)) {
	    til->constraints = REMOB(cnst,til->constraints);
	    free(cnst);
	    break;
	  };
       };
    }
   while (!EMPTY(l));
};





/************************************************************************/
/*									*/
/*	get_fields -- build list of fields for this object		*/
/*									*/
/************************************************************************/


static void
get_fields(ad)
   APPLE_DATA ad;
{
   APPLE_VALUE_B flds[MAX_FIELD];
   register Integer i,ct;

   DTRACE("get_fields 0x%x",ad);

   if (ad->fields != NULL) {
      free(ad->fields);
      ad->fields = NULL;
      ad->numfield = 0;
    };

   ct = APPLE_ext_inq_fields(ad->handle,NULL,TRUE,MAX_FIELD,flds);

   ad->fields = (APPLE_VALUE_B *) calloc(sizeof(APPLE_VALUE_B),(ct+MAX_NEW_FIELD));
   ad->numfield = ct;
   for (i = 0; i < ct; ++i) {
      ad->fields[i] = flds[i];
    };
};





/************************************************************************/
/*									*/
/*	setup_picture -- setup data descriptor for new picture		*/
/*									*/
/************************************************************************/


static void
setup_picture(ad)
   APPLE_DATA ad;
{
   DTRACE("setup_picture 0x%x",ad);

   APPLE_set_picture_mode(ad);

   if (ad->picture != NULL) {
      if (ad->prev_picture == NULL) {
	 ad->picture->alternative = ad->first_picture;
	 ad->first_picture = ad->picture;
       }
      else {
	 ad->picture->alternative = ad->prev_picture->alternative;
	 ad->prev_picture->alternative = ad->picture;
       };
    };
};




/************************************************************************/
/*									*/
/*	new_gelo_object -- create object of given mode			*/
/*									*/
/************************************************************************/


static APPLE_GELO
new_gelo_object(md)
   APPLE_MODE md;
{
   APPLE_GELO ag;

   DTRACE("new_gelo_object %d",md);

   switch (md) {
      case APPLE_MODE_TILED :
       { register APPLE_GELO_TILED til;
	 til = PALLOC(APPLE_GELO_TILED_B);
	 til->max_x = 0;
	 til->max_y = 0;
	 til->components = NULL;
	 til->boxed = TRUE;
	 til->arcs = NULL;
	 til->constraints = NULL;
	 ag = (APPLE_GELO) til;
	 break;
       };

      case APPLE_MODE_LAYOUT :
       { register APPLE_GELO_LAYOUT lay;
	 lay = PALLOC(APPLE_GELO_LAYOUT_B);
	 lay->method = GELO_METHOD_DFS2|GELO_METHOD_SELECTABLE;
	 lay->connmethod = GELO_CONN_METHOD_DIRECT_RECT;
	 lay->boxed = TRUE;
	 lay->components = NULL;
	 lay->white_space = 0;
	 lay->shrinkage = DEFAULT_SHRINK;
	 lay->unique = TRUE;
	 lay->fixed = FALSE;
	 lay->standard = FALSE;
	 lay->centered = FALSE;
	 lay->simple = FALSE;
	 ag = (APPLE_GELO) lay;
	 break;
       };

      case APPLE_MODE_BOX :
       { register APPLE_GELO_BOX box;
	 box = PALLOC(APPLE_GELO_BOX_B);
	 box->fill = 0;
	 box->linestyle = 0;
	 box->font = 0;
	 box->invert = 0;
	 box->value.name = NULL;
	 box->value.value = 0;
	 box->shape = GELO_SHAPE_RECTANGLE;
	 ag = (APPLE_GELO) box;
	 break;
       };

      case APPLE_MODE_ARC :
       { register APPLE_GELO_ARC arc;
	 arc = PALLOC(APPLE_GELO_ARC_B);
	 arc->style = ASH_STYLE_SOLID;
	 arc->arrow = GELO_ARROW_SINGLE_ALL;
	 arc->label_loc = GELO_LABEL_MIDDLE;
	 arc->label_pos = GELO_LABEL_LEFT;
	 arc->label_box = FALSE;
	 arc->fport = GELO_PORT_ANY;
	 arc->tport = GELO_PORT_ANY;
	 arc->from.name = NULL;
	 arc->from.value = 0;
	 arc->to.name = NULL;
	 arc->to.value = 0;
	 arc->label.name = NULL;
	 arc->label.value = 0;
	 arc->match = NULL;
	 ag = (APPLE_GELO) arc;
	 break;
       };

      case APPLE_MODE_POINTER :
       { register APPLE_GELO_ARC arc;
	 arc = PALLOC(APPLE_GELO_ARC_B);
	 arc->style = ASH_STYLE_SOLID;
	 arc->arrow = GELO_ARROW_SINGLE_ALL;
	 arc->label_loc = GELO_LABEL_MIDDLE;
	 arc->label_pos = GELO_LABEL_LEFT;
	 arc->label_box = FALSE;
	 arc->fport = GELO_PORT_CENTER;
	 arc->tport = GELO_PORT_ANY;
	 arc->from.name = APPLE_SELF_NAME;
	 arc->from.value = 0;
	 arc->to.name = NULL;
	 arc->to.value = 0;
	 arc->label.name = NULL;
	 arc->label.value = 0;
	 arc->match = NULL;
	 ag = (APPLE_GELO) arc;
	 break;
       };

      case APPLE_MODE_FIELD :
       { register APPLE_GELO_FIELD fld;
	 fld = PALLOC(APPLE_GELO_FIELD_B);
	 fld->value.name = NULL;
	 fld->value.value = 0;
	 fld->match = NULL;
	 fld->fport = GELO_PORT_CENTER;
	 ag = (APPLE_GELO) fld;
	 break;
       };

      case APPLE_MODE_IGNORE :
       { register APPLE_GELO_IGNORE ign;
	 ign = PALLOC(APPLE_GELO_IGNORE_B);
	 ag = (APPLE_GELO) ign;
	 break;
       };

      default :
	 ag = NULL;
	 break;
    };

   if (ag != NULL) {
      ag->mode = md;
      ag->fix_x = FALSE;
      ag->fix_y = FALSE;
    };

   return ag;
};





/************************************************************************/
/*									*/
/*	new_component -- create a new component 			*/
/*									*/
/************************************************************************/


static APPLE_GELO_COMPONENT
new_component(lx,by,rx,ty)
   Integer lx,by,rx,ty;
{
   register APPLE_GELO_COMPONENT c;

   DTRACE("new_component %d %d %d %d",lx,by,rx,ty);

   c = PALLOC(APPLE_GELO_COMPONENT_B);
   c->lx = MIN(lx,rx);
   c->by = MAX(by,ty);
   c->rx = MAX(lx,rx);
   c->ty = MIN(by,ty);

   c->fix_x = FALSE;
   c->fix_y = FALSE;
   c->expand_x = abs(rx-lx);
   c->expand_y = abs(ty-by);
   c->priority_x = 1.0;
   c->priority_y = 1.0;
   c->shrinkage = DEFAULT_SHRINK;

   c->object = NULL;

   return c;
};





/************************************************************************/
/*									*/
/*	enlarge_layout -- handle making the layout larger		*/
/*									*/
/************************************************************************/


static void
enlarge_layout(ad,x,y)
   APPLE_DATA ad;
   Integer x,y;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_COMPONENT c;

   DTRACE("enlarge_layout 0x%x %d %d",ad,x,y);

   til = (APPLE_GELO_TILED) ad->picture->object;

   if (x > til->max_x) {
      if (til->max_y > 0) {
	 c = new_component(til->max_x,til->max_y,x,0);
	 til->components = APPEND(c,til->components);
       };
      til->max_x = x;
    };

   if (y > til->max_y) {
      c = new_component(0,y,til->max_x,til->max_y);
      til->components = APPEND(c,til->components);
      til->max_y = y;
    };
};





/************************************************************************/
/*									*/
/*	intersect_component -- intersect two components 		*/
/*									*/
/************************************************************************/


static Boolean
intersect_component(ad,orig,new)
   APPLE_DATA ad;
   APPLE_GELO_COMPONENT orig;
   APPLE_GELO_COMPONENT new;
{
   register APPLE_GELO_TILED til;
   register APPLE_GELO_COMPONENT c;

   DTRACE("intersect_component 0x%x 0x%x 0x%x",ad,orig,new);

   til = (APPLE_GELO_TILED) ad->picture->object;

   DTRACE("\tstart: %d %d %d %d",orig->lx,orig->by,orig->rx,orig->ty);

   if (new->lx <= orig->lx && new->rx >= orig->rx &&
	  new->ty <= orig->ty && new->by >= orig->by) {
      return FALSE;
    };

   if (new->lx < orig->rx && new->rx > orig->lx) {
      if (new->ty > orig->ty && new->ty < orig->by) {
	 c = new_component(orig->lx,new->ty,orig->rx,orig->ty);
	 til->components = APPEND(c,til->components);
	 orig->ty = new->ty;
       };

      if (new->by > orig->ty && new->by < orig->by) {
	 if (new->lx > orig->lx) {
	    c = new_component(orig->lx,new->by,new->lx,orig->ty);
	    til->components = APPEND(c,til->components);
	  };
	 if (new->rx < orig->rx) {
	    c = new_component(new->rx,new->by,orig->rx,orig->ty);
	    til->components = APPEND(c,til->components);
	  };
	 orig->ty = new->by;
       };
    };

   if (new->ty < orig->by && new->by > orig->ty) {
      if (new->lx > orig->lx && new->lx < orig->rx &&
	     new->rx > orig->lx && new->rx < orig->rx) {
	 c = new_component(orig->lx,orig->by,new->lx,orig->ty);
	 til->components = APPEND(c,til->components);
	 orig->lx = new->rx;
       }
      else if (new->lx > orig->lx && new->lx < orig->rx) {
	 orig->rx = new->lx;
       }
      else if (new->rx > orig->lx && new->rx < orig->rx) {
	 orig->lx = new->rx;
       };
    };

   orig->expand_x = abs(orig->rx-orig->lx);
   orig->expand_y = abs(orig->ty-orig->by);

   DTRACE("\tresult: %d %d %d %d",orig->lx,orig->by,orig->rx,orig->ty);

   return TRUE;
};





/* end of applepic.c */
