/************************************************************************/
/*									*/
/*		applemenu.c						*/
/*									*/
/*	Menu management for APPLE type picture editor package		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "apple_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define FILE_MENU	"File"
#define TOP_MENU	"Top"
#define EDIT_MENU	"Edit"
#define PROP_MENU	"Props"

#define STRING_BTN	0
#define EMPTY_BTN	1
#define COMPUTED_BTN	2
#define FIELD_BTN	3
#define DATA_BTN	4
#define INDEX_BTN	5
#define SELF_BTN	6
#define NUM_PREFIX_BTN	7




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	int		save_btn();
static	int		abort_btn();
static	int		grid_btn();
static	int		top_option_btn();
static	int		arc_option_btn();
static	int		alt_btn();
static	int		cond_btn();
static	int		quit_btn();
static	Boolean 	get_conditions();
static	Boolean 	test_field_active();
static	Boolean 	set_field_value();
static	void		file_menu_setup();
static	void		top_menu_setup();
static	void		edit_menu_setup();
static	void		prop_menu_setup();
static	void		button_setup();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	STEM_PDM_DATA	apple_menu[] = {
   { STEM_PSTATE_MENU, FILE_MENU },
      { STEM_PSTATE_BTN, "Finer Grid", grid_btn },
      { STEM_PSTATE_BTN, "Courser Grid", grid_btn },
      { STEM_PSTATE_BTN, "Show Grid", grid_btn },
      { STEM_PSTATE_BTN, "Clear", abort_btn },
      { STEM_PSTATE_BTN, "Remove", abort_btn },
      { STEM_PSTATE_BTN, "Save", save_btn },
      { STEM_PSTATE_BTN, "Restart", abort_btn },
   { STEM_PSTATE_MENU, TOP_MENU },
      { STEM_PSTATE_BTN, "Default", APPLE_default_btn },
      { STEM_PSTATE_BTN, "Tiled", APPLE_design_btn },
      { STEM_PSTATE_BTN, "Layout", APPLE_design_btn },
      { STEM_PSTATE_BTN, "Box", APPLE_design_btn },
      { STEM_PSTATE_BTN, "Arc", APPLE_design_btn },
      { STEM_PSTATE_BTN, "Ignore", APPLE_design_btn },
      { STEM_PSTATE_BTN, "Next Alternative", alt_btn },
      { STEM_PSTATE_BTN, "First Alternative", alt_btn },
      { STEM_PSTATE_BTN, "New Alternative", alt_btn },
      { STEM_PSTATE_BTN, "Set Conditions", cond_btn },
   { STEM_PSTATE_MENU, EDIT_MENU },
      { STEM_PSTATE_BTN, "Box", APPLE_box_type_btn },
      { STEM_PSTATE_BTN, "Layout", APPLE_box_type_btn },
      { STEM_PSTATE_BTN, "Field", APPLE_box_type_btn },
      { STEM_PSTATE_BTN, "Pointer", APPLE_box_type_btn },
      { STEM_PSTATE_BTN, "Ignore", APPLE_box_type_btn },
      { STEM_PSTATE_BTN, "Empty", APPLE_box_type_btn },
   { STEM_PSTATE_MENU, PROP_MENU },
      { STEM_PSTATE_BTN, "Options", APPLE_option_btn },
   { STEM_PSTATE_END }
};





/************************************************************************/
/*									*/
/*	APPLE_menu_init -- module initialization			*/
/*									*/
/************************************************************************/


void
APPLE_menu_init()
{
   ITRACE("APPLE_menu_init");
};





/************************************************************************/
/*									*/
/*	APPLEmenu_quit -- add quit to file menu for window		*/
/*									*/
/************************************************************************/


void
APPLEmenu_quit(ad)
   APPLE_DATA ad;
{
   if (ad == NULL) return;
   if (ad->menu_win == NULL) return;

   STEMpdm_btn_add(ad->menu_win,FILE_MENU,"Quit",quit_btn);
};





/************************************************************************/
/*									*/
/*	APPLE_menu_define -- define menu for apple window		*/
/*									*/
/************************************************************************/


void
APPLE_menu_define(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_menu_define 0x%x",ad);

   if (ad->menu_win != NULL) {
      STEMpdm_define(ad->menu_win,ad,apple_menu);
    };

   APPLE_menu_setup(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_menu_setup -- setup menu by enabling/disabling as needed	*/
/*									*/
/************************************************************************/


void
APPLE_menu_setup(ad)
   APPLE_DATA ad;
{
   ITRACE("APPLE_menu_setup 0x%x",ad);

   file_menu_setup(ad);
   top_menu_setup(ad);
   APPLE_menu_prop_setup(ad);
};






/************************************************************************/
/*									*/
/*	APPLE_menu_prop_setup --setup property menus for selection	*/
/*									*/
/************************************************************************/


void
APPLE_menu_prop_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("APPLE_menu_prop_setup 0x%x",ad);

   edit_menu_setup(ad);
   prop_menu_setup(ad);
   button_setup(ad);
};





/************************************************************************/
/*									*/
/*	APPLE_field_menu_setup -- setup STEM menu for available fields	*/
/*									*/
/************************************************************************/


void
APPLE_field_menu_setup(ad)
   APPLE_DATA ad;
{
   STEM_BUTTON btns[MAX_FIELD];
   register Integer i;

   ITRACE("APPLE_field_menu_setup 0x%x");

   btns[STRING_BTN].name = "\"String\"";
   btns[STRING_BTN].flags = STEM_FLAG_NORMAL;
   btns[EMPTY_BTN].name = "<Empty>";
   btns[EMPTY_BTN].flags = STEM_FLAG_NORMAL;
   btns[COMPUTED_BTN].name = "<Computed>";
   btns[COMPUTED_BTN].flags = STEM_FLAG_NORMAL;

   btns[FIELD_BTN].name = "<New Field>";
   if (ad->newfields)
      btns[FIELD_BTN].flags = STEM_FLAG_NORMAL;
   else
      btns[FIELD_BTN].flags = STEM_FLAG_LOLITE;

   btns[DATA_BTN].name = "<From Source>";
   if (ad->picture != NULL && ad->picture->cond_layout) {
      if (test_field_active(ad,"$DATA"))
	 btns[DATA_BTN].flags = STEM_FLAG_CHOICE;
      else
	 btns[DATA_BTN].flags = STEM_FLAG_NORMAL;
    }
   else {
      btns[DATA_BTN].flags = STEM_FLAG_LOLITE;
    };

   btns[INDEX_BTN].name = "<From Index>";
   if (ad->picture != NULL && (ad->picture->cond_layout || ad->picture->cond_array)) {
      if (test_field_active(ad,"$INDEX"))
	 btns[INDEX_BTN].flags = STEM_FLAG_CHOICE;
      else
	 btns[INDEX_BTN].flags = STEM_FLAG_NORMAL;
    }
   else {
      btns[INDEX_BTN].flags = STEM_FLAG_LOLITE;
    };

   btns[SELF_BTN].name = "<From Self>";
   if (test_field_active(ad,"$SELF"))
      btns[SELF_BTN].flags = STEM_FLAG_CHOICE;
   else
      btns[SELF_BTN].flags = STEM_FLAG_NORMAL;

   for (i = 0; i < ad->numfield; ++i) {
      btns[NUM_PREFIX_BTN+i].name = ad->fields[i].name;
      if (test_field_active(ad,ad->fields[i].name))
	 btns[NUM_PREFIX_BTN+i].flags = STEM_FLAG_CHOICE;
      else
	 btns[NUM_PREFIX_BTN+i].flags = STEM_FLAG_NORMAL;
    };

   btns[NUM_PREFIX_BTN+i].name = 0;

   STEMmenu_refresh(ad->field_win,btns);
};





/************************************************************************/
/*									*/
/*	APPLE_field_button -- handle user selecting field button	*/
/*									*/
/************************************************************************/


int
APPLE_field_button(ad,btn,rbtn,w)
   APPLE_DATA ad;
   Integer btn;
   Integer rbtn;
   ASH_WINDOW w;
{
   String s,menu[10];
   Character buf[64];
   APPLE_VALUE_B v;
   register Integer i;
   register Boolean fg;

   ITRACE("APPLE_field_button 0x%x %d 0x%x 0x%x",ad,btn,rbtn,w);

   if (rbtn & RIP_BTN_DOWN) return FALSE;

   if (ad->picture == NULL) return FALSE;
   if (ad->picture->object == NULL) return FALSE;

   switch (ad->picture->object->mode) {
      case APPLE_MODE_NONE :
	 return FALSE;
      case APPLE_MODE_TILED :
	 if (ad->selection == NULL && ad->use_select == USE_SELECT_NONE)
	    return FALSE;
	 break;
      case APPLE_MODE_ARC :
	 if (ad->selection == NULL && ad->use_select == USE_SELECT_NONE)
	    return FALSE;
      default :
	 break;
    };

   switch (btn) {
      case STRING_BTN :
	 i = 0;
	 menu[i++] = "%CEnter String\n";
	 menu[i++] = "%0.48t\n";
	 menu[i++] = "   %a%M   %c";
	 menu[i] = 0;
	 buf[0] = 0;
	 if (!STEMdialog(ad->window,menu,buf)) return FALSE;
	 v.name = SALLOC(buf);
	 APPLE_ext_string(&v);
	 fg = set_field_value(ad,TRUE,&v);
	 break;

      case EMPTY_BTN :
	 v.name = NULL;
	 v.value = 0;
	 fg = set_field_value(ad,TRUE,&v);
	 break;

      case COMPUTED_BTN :
	 fg = APPLE_ext_inq_function(&v.value);
	 v.name = "<COMPUTED>";
	 if (fg) fg = set_field_value(ad,TRUE,&v);
	 break;

      case FIELD_BTN :
	 if (!ad->newfields) {
	    fg = FALSE;
	    break;
	  };
	 i = 0;
	 menu[i++] = "%CEnter Field Name\n";
	 menu[i++] = "%0.24t\n";
	 menu[i++] = "   %a%M   %c";
	 menu[i] = 0;
	 buf[0] = 0;
	 if (!STEMdialog(ad->window,menu,buf)) return FALSE;
	 s = SALLOC(buf);
	 i = APPLE_new_field(ad,s);
	 if (i < 0) fg = FALSE;
	 else fg = APPLE_field_button(ad,NUM_PREFIX_BTN+i,rbtn,w);
	 break;

      case DATA_BTN :
	 v.name = APPLE_DATA_NAME;
	 v.value = 0;
	 fg = set_field_value(ad,TRUE,&v);
	 break;

      case INDEX_BTN :
	 v.name = APPLE_INDEX_NAME;
	 v.value = 0;
	 fg = set_field_value(ad,TRUE,&v);
	 break;

      case SELF_BTN :
	 v.name = APPLE_SELF_NAME;
	 v.value = 0;
	 fg = set_field_value(ad,TRUE,&v);
	 break;

      default :
	 fg = set_field_value(ad,FALSE,&ad->fields[btn-NUM_PREFIX_BTN]);
	 break;
    };

   if (fg) {
      APPLE_menu_setup(ad);
      APPLE_field_menu_setup(ad);
      APPLE_redraw_component(ad,ad->selection);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	save_btn -- handle SAVE request 				*/
/*	abort_btn -- handle ABORT/CLEAR/REMOVE request			*/
/*									*/
/************************************************************************/


static int
save_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("save_btn 0x%x %s %s",ad,menu,btn);

   APPLE_save_picture(ad);

   APPLE_get_picture(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





static int
abort_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("abort_btn 0x%x %s %s",ad,menu,btn);

   if (STREQL(btn,"Restart")) {
      APPLE_get_picture(ad);
    }
   else if (STREQL(btn,"Clear")) {
      APPLE_new_picture(ad,APPLE_MODE_NONE);
    }
   else if (STREQL(btn,"Remove")) {
      APPLE_remove_alternative(ad);
    }
   else {
      return FALSE;
    };

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	grid_btn -- handle COURSER/FINER/SHOW grid requests		*/
/*									*/
/************************************************************************/


static int
grid_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   register Integer gd;
   register Boolean fg;

   ITRACE("grid_btn 0x%x %s %s",ad,menu,btn);

   if (STREQL(btn,"Show Grid")) {
      ad->show_grid = !ad->show_grid;
      APPLE_menu_setup(ad);
      APPLE_draw_picture(ad);
      return TRUE;
    };

   if (STREQL(btn,"Finer Grid")) gd = ad->grid*2;
   else gd = ad->grid/2;

   fg = APPLE_change_grid(ad,gd);

   if (fg) {
      ad->selection = NULL;
      ad->use_select = USE_SELECT_NONE;
      ad->usearc_select = NULL;
      APPLE_menu_setup(ad);
      APPLE_field_menu_setup(ad);
      APPLE_draw_picture(ad);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	APPLE_default_btn -- set up the default for this type		*/
/*									*/
/************************************************************************/


int
APPLE_default_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("APPLE_default_btn 0x%x %s %s",ad,menu,btn);

   APPLE_default_picture(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	APPLE_design_btn -- handle new design for the user		*/
/*									*/
/************************************************************************/


int
APPLE_design_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   register APPLE_MODE md;

   ITRACE("APPLE_design_btn 0x%x %s %s",ad,menu,btn);

   if (STREQL(btn,"Tiled")) md = APPLE_MODE_TILED;
   else if (STREQL(btn,"Layout")) md = APPLE_MODE_LAYOUT;
   else if (STREQL(btn,"Box")) md = APPLE_MODE_BOX;
   else if (STREQL(btn,"Arc")) md = APPLE_MODE_ARC;
   else if (STREQL(btn,"Ignore")) md = APPLE_MODE_IGNORE;
   else return FALSE;

   APPLE_new_picture(ad,md);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	APPLE_box_type_btn -- set box type from user			*/
/*									*/
/************************************************************************/


int
APPLE_box_type_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   register APPLE_MODE bty;

   ITRACE("APPLE_box_type_btn 0x%x %s %s",ad,menu,btn);

   if (ad->picture == NULL || ad->selection == NULL) return FALSE;

   if (STREQL(btn,"Box")) bty = APPLE_MODE_BOX;
   else if (STREQL(btn,"Layout")) bty = APPLE_MODE_LAYOUT;
   else if (STREQL(btn,"Field")) bty = APPLE_MODE_FIELD;
   else if (STREQL(btn,"Pointer")) bty = APPLE_MODE_POINTER;
   else if (STREQL(btn,"Empty")) bty = APPLE_MODE_NONE;
   else if (STREQL(btn,"Ignore")) bty = APPLE_MODE_IGNORE;
   else return FALSE;

   if (ad->picture->layout_obj != NULL && bty == APPLE_MODE_LAYOUT &&
	  (ad->selection->object == NULL ||
	      ad->selection->object->mode != APPLE_MODE_LAYOUT)) {
      APPLE_message(ad->window,"At most one layout field");
      return TRUE;
    };

   APPLE_set_type(ad,ad->selection,bty);

   APPLE_redraw_component(ad,ad->selection);
   APPLE_menu_prop_setup(ad);
   APPLE_field_menu_setup(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	APPLE_option_btn -- provide user with drawing options		*/
/*									*/
/************************************************************************/


int
APPLE_option_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   String btns[32];
   Integer i;
   Integer vals[30];
   Character fbuf[32],ebufx[16],ebufy[16],pbufx[16],pbufy[16],sbuf[32],csbuf[32];
   APPLE_GELO obj;
   APPLE_GELO_BOX box;
   APPLE_GELO_TILED til;
   APPLE_GELO_LAYOUT lay;
   APPLE_GELO_FIELD fld;
   APPLE_GELO_ARC arc;
   Integer v,fixx,fixy;
   float vf;

   ITRACE("APPLE_option_btn 0x%x %s %s",ad,menu,btn);

   if (ad->mode == APPLE_MODE_ARC) return arc_option_btn(ad,TRUE,FALSE);

   if (ad->use_select == USE_SELECT_NONE && ad->usearc_select != NULL)
      obj = ad->usearc_select;
   else if (ad->selection != NULL) obj = ad->selection->object;
   else if (ad->picture == NULL) obj = NULL;
   else if (ad->use_select == USE_SELECT_ONEUSE) obj = ad->usearc_select;
   else obj = ad->picture->object;

   if (obj == NULL) return top_option_btn(ad,TRUE);
   else if (obj->mode == APPLE_MODE_POINTER) return arc_option_btn(ad,FALSE,TRUE);
   else if (obj->mode == APPLE_MODE_ARC) return arc_option_btn(ad,FALSE,FALSE);

   i = 0;
   btns[i++] = "%CDrawing Options\n";

    switch (obj->mode) {
      case APPLE_MODE_BOX :
	 box = (APPLE_GELO_BOX) obj;
	 btns[i++] = "Fill style: %0.5t";
	 sprintf(fbuf,"%d",box->fill);
	 btns[i++] = "Invert:     %1o\n";
	 vals[0] = box->invert;
	 btns[i++] = "%2.0o Solid           %2.1o Dashed    %2.2o Dotted";
	 btns[i++] = "%2.3o Thick           %2.6o Thicker   %2.4o Thick Dashed\n";
	 vals[1] = box->linestyle;
	 btns[i++] = "%4.0o Default Font    %4.2o Small Font %4.3o Average Font";
	 btns[i++] = "%4.4o Large Font      %4.5o Huge Font\n";
	 vals[3] = box->font;
	 btns[i++] = "%3.0o Rectangle       %3.2o Square     %3.4o Circle";
	 btns[i++] = "%3.1o No Shape        %3.3o Ellipse    %3.5o Roundtangle";
	 btns[i++] = "%3.6o Triangle Up     %3.11o Hexagon    %3.7o Triangle Down";
	 btns[i++] = "%3.8o Triangle Left   %3.12o Octagon    %3.9o Triangle Right";
	 btns[i++] = "%3.10o Pentagon        %3.13o Diamond    %3.15o Vertical Line";
	 btns[i++] = "%3.16o SemiCircle Up   %3.17o Semi Down  %3.18o Semi Left";
	 btns[i++] = "%3.19o Semi Right";
	 vals[2] = (Integer) box->shape;
	 break;
      case APPLE_MODE_TILED :
	 til = (APPLE_GELO_TILED) obj;
	 btns[i++] = "Boxed:  %1o";
	 vals[0] = til->boxed;
	 break;
      case APPLE_MODE_LAYOUT :
	 lay = (APPLE_GELO_LAYOUT) obj;
	 btns[i++] = "Boxed:               %1o";
	 vals[0] = lay->boxed;
	 btns[i++] = "White Space:         %0.5t  %%";
	 sprintf(fbuf,"%d",lay->white_space);
	 btns[i++] = "Shrinkage:           %25.5t  %%";
	 sprintf(sbuf,"%d",lay->shrinkage);
	 btns[i++] = "Unique:              %2o";
	 vals[1] = lay->unique;
	 btns[i++] = "Fixed Components:    %7o";
	 vals[6] = lay->fixed;
	 btns[i++] = "Standard size:       %8o";
	 vals[7] = lay->standard;
	 btns[i++] = "Centered Components: %9o";
	 vals[8] = lay->centered;
	 btns[i++] = "Simple:              %20o";
	 vals[13] = lay->simple;
	 btns[i++] = "Layout Heuristics:";
	 btns[i++] = "   %3.0o Depth First   %3.1o Breadth First   %3.2o Ordered";
	 btns[i++] = "   %10o Averaged      %12o Bi-Directional  %13o Forward";
	 btns[i++] = "   %21o Level Graphs  %27o Pert Grids      %28o Pert Lattice";
	 btns[i++] = "   %29o Rectangular";
	 btns[i++] = "   %23o Local Optimization%M%24o Global Optimization";
	 btns[i++] = "   %22o User Selectable";
	 btns[i++] = "   %11.0o Horizontal    %11.1o Vertical        %11.2o Either";
	 v = (Integer) lay->method;
	 vals[2] = ((v&GELO_METHOD_BIT_DFS) != 0 ? 0 :
		       (v&GELO_METHOD_BIT_BFS) != 0 ? 1 : 2);
	 vals[9] = ((v&GELO_METHOD_BIT_AVG) != 0 ? 1 : 0);
	 vals[10] = ((v&GELO_METHOD_BIT_HOR) != 0 ? 0 :
		       (v&GELO_METHOD_BIT_VERT) != 0 ? 1 : 2);
	 vals[11] = ((v&GELO_METHOD_BIT_2WAY) != 0 ? 1 : 0);
	 vals[12] = ((v&GELO_METHOD_BIT_FWD) != 0 ? 1 : 0);
	 vals[14] = ((v&GELO_METHOD_BIT_LEVEL) != 0 ? 1 : 0);
	 vals[15] = ((v&GELO_METHOD_SELECTABLE) != 0 ? 1 : 0);
	 vals[16] = ((v&GELO_METHOD_BIT_OPT) != 0 ? 1 : 0);
	 vals[17] = ((v&GELO_METHOD_BIT_ANNEAL) != 0 ? 1 : 0);
	 vals[18] = ((v&GELO_METHOD_BIT_GRID) != 0 ? 1 : 0);
	 vals[19] = ((v&GELO_METHOD_BIT_LATTICE) != 0 ? 1 : 0);
	 vals[20] = ((v&GELO_METHOD_BIT_GIOTTO) != 0 ? 1 : 0);
	 btns[i++] = "%4.2o Best Placement";
	 btns[i++] = "   %4.0o Routed%M   %4.1o Direct";
	 vals[3] = (Integer) lay->connmethod;
	 break;
      case APPLE_MODE_FIELD :
	 fld = (APPLE_GELO_FIELD) obj;
	 btns[i++] = "Match String: %0.24t";
	 if (fld->match == NULL) fbuf[0] = 0;
	 else strcpy(fbuf,fld->match);
	 if (ad->usearc_select != NULL && ad->use_select == USE_SELECT_NONE) {
	    btns[i++] = "Portal Option:     %1.8o Not a portal";
	    btns[i++] = "   %1.1o Top Left     %1.2o Top       %1.3o Top Right";
	    btns[i++] = "   %1.0o Left         %1.14o Computed  %1.4o Right";
	    btns[i++] = "   %1.7o Bottom Left  %1.6o Bottom    %1.5o Bottom Right";
	    vals[0] = (Integer) fld->fport;
	  };
	 break;
      case APPLE_MODE_IGNORE :
	 break;
    };

   btns[i++] = "\nFix X:       %18o%MFix Y:       %19o";
   fixx = obj->fix_x;
   fixy = obj->fix_y;

   if (ad->use_select == USE_SELECT_NONE && ad->selection != NULL) {
      btns[i++] = "\nComp Fix X:  %5o%MComp Fix Y:  %6o";
      vals[4] = ad->selection->fix_x;
      vals[5] = ad->selection->fix_y;
      btns[i++] = "Shrinkage: %26.5t %%";
      sprintf(csbuf,"%d",ad->selection->shrinkage);
      btns[i++] = "Expansion X: %14.12t%MExpansion Y: %15.12t";
      sprintf(ebufx,"%g",ad->selection->expand_x);
      sprintf(ebufy,"%g",ad->selection->expand_y);
      btns[i++] = "Priority X:  %16.12t%MPriority Y:  %17.12t";
      sprintf(pbufx,"%g",ad->selection->priority_x);
      sprintf(pbufy,"%g",ad->selection->priority_y);
    }
   else if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select != NULL) {
      btns[i++] = "\nUSE as:";
      btns[i++] = "   %5.0o Node    %5.1o Arc To    %5.2o Arc From";
      vals[4] = 0;
    };

   btns[i++] = "\n   %a%M   %c";

   btns[i] = 0;

   if (!STEMdialog(ad->window,btns,fbuf,&vals[0],&vals[1],&vals[2],&vals[3],
		      &vals[4],&vals[5],&vals[6],&vals[7],&vals[8],
		      &vals[9],&vals[10],&vals[11],&vals[12],
		      ebufx,ebufy,pbufx,pbufy,&fixx,&fixy,
		      &vals[13],&vals[14],&vals[15],&vals[16],&vals[17],
		      sbuf,csbuf,&vals[18],&vals[19],&vals[20]))
      return FALSE;

   switch (obj->mode) {
      case APPLE_MODE_BOX :
	 box = (APPLE_GELO_BOX) obj;
	 if (sscanf(fbuf,"%d",&v) == 1) box->fill = v;
	 box->invert = vals[0];
	 box->linestyle = vals[1];
	 box->shape = (GELO_SHAPE) vals[2];
	 box->font = vals[3];
	 break;
      case APPLE_MODE_TILED :
	 til = (APPLE_GELO_TILED) obj;
	 til->boxed = vals[0];
	 break;
      case APPLE_MODE_LAYOUT :
	 lay = (APPLE_GELO_LAYOUT) obj;
	 lay->boxed = vals[0];
	 if (sscanf(fbuf,"%d",&v) == 1) lay->white_space = v;
	 if (sscanf(sbuf,"%d",&v) == 1 && v >= 0 && v <= 100) lay->shrinkage = v;
	 lay->unique = vals[1];
	 lay->connmethod = (GELO_CONN_METHOD) vals[3];
	 lay->fixed = (Boolean) vals[6];
	 lay->standard = (Boolean) vals[7];
	 lay->centered = (Boolean) vals[8];
	 lay->simple = (Boolean) vals[13];
	 v = 0;
	 if (vals[2] == 0) v |= GELO_METHOD_BIT_DFS;
	 else if (vals[2] == 1) v |= GELO_METHOD_BIT_BFS;
	 if (vals[9]) v |= GELO_METHOD_BIT_AVG;
	 if (vals[10] == 0) v |= GELO_METHOD_BIT_HOR;
	 else if (vals[10] == 1) v |= GELO_METHOD_BIT_VERT;
	 if (vals[11]) v |= GELO_METHOD_BIT_2WAY;
	 if (vals[12]) v |= GELO_METHOD_BIT_FWD;
	 if (vals[14]) v |= GELO_METHOD_BIT_LEVEL;
	 if (vals[15]) v |= GELO_METHOD_SELECTABLE;
	 if (vals[16]) v |= GELO_METHOD_BIT_OPT;
	 if (vals[17]) v |= GELO_METHOD_BIT_ANNEAL;
	 if (vals[18]) v |= GELO_METHOD_BIT_GRID;
	 if (vals[19]) v |= GELO_METHOD_BIT_LATTICE;
	 if (vals[20]) v |= GELO_METHOD_BIT_GIOTTO;
	 lay->method = (GELO_METHOD) v;
	 break;
      case APPLE_MODE_FIELD :
	 fld = (APPLE_GELO_FIELD) obj;
	 if (fbuf[0] == 0) fld->match = NULL;
	 else fld->match = SALLOC(fbuf);
	 if (ad->usearc_select != NULL && ad->use_select == USE_SELECT_NONE) {
	    fld->fport = (GELO_PORT) vals[0];
	  };
	 break;
      case APPLE_MODE_IGNORE :
	 break;
    };

   obj->fix_x = fixx;
   obj->fix_y = fixy;

   if (ad->use_select == USE_SELECT_NONE && ad->selection != NULL) {
      ad->selection->fix_x = vals[4];
      ad->selection->fix_y = vals[5];
      if (sscanf(csbuf,"%d",&v) == 1 && v >= 0 && v <= 100) ad->selection->shrinkage = v;
      if (sscanf(ebufx,"%f",&vf) == 1) ad->selection->expand_x = vf;
      if (sscanf(ebufy,"%f",&vf) == 1) ad->selection->expand_y = vf;
      if (sscanf(pbufx,"%f",&vf) == 1) ad->selection->priority_x = vf;
      if (sscanf(pbufy,"%f",&vf) == 1) ad->selection->priority_y = vf;
    }
   else if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select != NULL &&
	       vals[4] != 0) {
      ASHset_user_data(ad->use_window,-1);
      fld = (APPLE_GELO_FIELD) ad->usearc_select;
      ad->picture->use = REMOB(ad->usearc_select,ad->picture->use);
      arc = (APPLE_GELO_ARC) APPLE_new_usearc(&fld->value,(vals[4] == 1));
      arc->match = fld->match;
      ad->picture->use = APPEND(arc,ad->picture->use);
      ad->usearc_select = (APPLE_GELO) arc;
      APPLE_option_btn(ad,menu,btn);
    };

   APPLE_redraw_component(ad,ad->selection);
   APPLE_menu_prop_setup(ad);

   return TRUE;
};





static int
top_option_btn(ad,rdrfg)
   APPLE_DATA ad;
   Boolean rdrfg;
{
   String btns[16];
   register Integer i;
   Integer val;

   DTRACE("top_option_btn 0x%x",ad);

   i = 0;
   if (ad->selection == NULL)
      btns[i++] = "%CDrawing Selection\n";
   else
      btns[i++] = "%CField Drawing Selection\n";

   btns[i++] = "%0.4o Box";
   btns[i++] = "%0.1o Tiled";
   if (ad->selection == NULL || ad->picture->layout_obj == NULL)
      btns[i++] = "%0.3o Layout";
   if (ad->selection != NULL) {
      btns[i++] = "%0.2o Field";
      btns[i++] = "%0.6o Pointer";
    }
   else {
      btns[i++] = "%0.5o Arc";
    };
   btns[i++] = "%0.7o Ignore";

   btns[i++] = "   %a%M   %c";
   btns[i] = 0;

   val = (int) APPLE_MODE_BOX;

   if (!STEMdialog(ad->window,btns,&val)) return FALSE;

   if (val == 0) return FALSE;

   if (ad->selection == NULL) {
      APPLE_new_picture(ad,(APPLE_MODE) val);
      APPLE_menu_setup(ad);
      APPLE_field_menu_setup(ad);
      if (rdrfg) APPLE_draw_picture(ad);
    }
   else {
      APPLE_set_type(ad,ad->selection,(APPLE_MODE) val);
      APPLE_redraw_component(ad,ad->selection);
      APPLE_menu_prop_setup(ad);
      APPLE_field_menu_setup(ad);
    };

   return TRUE;
};




static int
arc_option_btn(ad,topfg,ptrfg)
   APPLE_DATA ad;
   Boolean topfg;
   Boolean ptrfg;
{
   String btns[50];
   Integer i;
   Integer fport,tport,style,arrow,lblloc,lblpos,lblbox,arctype;
   register APPLE_GELO_ARC arc;
   register APPLE_GELO_FIELD fld;
   register APPLE_VALUE av;
   Character match[64],label[64];
   Integer oarctype;

   ITRACE("arc_option_btn 0x%x %d %d",ad,topfg,ptrfg);

   i = 0;
   btns[i++] = "%CArc Options\n";

   btns[i++] = "From Port:         %0.9o ANY       %0.14o Computed";
   btns[i++] = "   %0.1o Top Left     %0.2o Top       %0.3o Top Right";
   btns[i++] = "   %0.0o Left         %0.8o Center    %0.4o Right";
   btns[i++] = "   %0.7o Bottom Left  %0.6o Bottom    %0.5o Bottom Right";
   btns[i++] = "   %0.10o Left Sequence     %0.12o Right Sequence";
   btns[i++] = "   %0.11o Top Sequence      %0.13o Bottom Sequence\n";

   btns[i++] = "To Port:           %1.9o ANY       %1.14o Computed";
   btns[i++] = "   %1.1o Top Left     %1.2o Top       %1.3o Top Right";
   btns[i++] = "   %1.0o Left         %1.8o Center    %1.4o Right";
   btns[i++] = "   %1.7o Bottom Left  %1.6o Bottom    %1.5o Bottom Right";
   btns[i++] = "   %1.10o Left Sequence     %1.12o Right Sequence";
   btns[i++] = "   %1.11o Top Sequence      %1.13o Bottom Sequence\n";

   btns[i++] = "%2.0o Solid%M%2.1o Dashed";
   btns[i++] = "%2.2o Dotted%M%2.3o Thick";
   btns[i++] = "%2.4o Thicker%M%2.5o Invisible\n";

   btns[i++] = "%3.0o No arrows%M%3.1o Single Arrows";
   btns[i++] = "%3.2o Double arrows%M%3.3o Directed Path\n";

   btns[i++] = "Label Information:%M%6o Boxed";
   btns[i++] = "   %4.1o Source      %4.2o Middle    %4.3o Destination";
   btns[i++] = "   %5.2o Left        %5.1o Centered  %5.3o Right\n";

   if (!topfg) {
      btns[i++] = "Label: %8.24t\n";
      btns[i++] = "Match: %7.24t\n";
    };

   if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select != NULL) {
      btns[i++] = "\nUSE as:";
      btns[i++] = "   %9.0o Node    %9.1o Arc To    %9.2o Arc From";
    };

   btns[i++] = "      %a%M   %c";
   btns[i] = 0;

   if (topfg) arc = (APPLE_GELO_ARC) ad->picture->object;
   else if (ptrfg && ad->selection->object != NULL)
      arc = (APPLE_GELO_ARC) ad->selection->object;
   else if (ptrfg) arc = (APPLE_GELO_ARC) ad->picture->object;
   else arc = (APPLE_GELO_ARC) ad->usearc_select;

   fport = (Integer) arc->fport;
   tport = (Integer) arc->tport;
   style = (Integer) arc->style;
   arrow = (Integer) arc->arrow;
   lblloc = (Integer) arc->label_loc;
   lblpos = (Integer) arc->label_pos;
   lblbox = arc->label_box;
   if (arc->match != NULL) strcpy(match,arc->match);
   else match[0] = 0;

   if (topfg || arc->label.name == NULL) label[0] = 0;
   else {
      strcpy(label,arc->label.name);
    };

   if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select != NULL) {
      if (STREQL(arc->from.name,APPLE_SELF_NAME)) arctype = 1;
      else arctype = 2;
      oarctype = arctype;
    };

   if (!STEMdialog(ad->window,btns,&fport,&tport,&style,&arrow,
		      &lblloc,&lblpos,&lblbox,match,label,&arctype))
      return FALSE;

   arc->fport = (GELO_PORT) fport;
   arc->tport = (GELO_PORT) tport;
   arc->style = (ASH_LINE_STYLE) style;
   arc->arrow = (GELO_ARC_ARROW) arrow;
   arc->label_loc = (GELO_ARC_LABEL_LOC) lblloc;
   arc->label_pos = (GELO_ARC_LABEL_POS) lblpos;
   arc->label_box = lblbox;

   if (match[0] == 0) arc->match = NULL;
   else if (arc->match == NULL || STRNEQ(arc->match,match))
      arc->match = SALLOC(match);

   if (!topfg) {
      if (label[0] == 0) {
	 arc->label.name = NULL;
	 arc->label.value = NULL;
       }
      else if (arc->label.name == NULL || STRNEQ(arc->label.name,label)) {
	 arc->label.name = SALLOC(label);
	 arc->label.value = NULL;
	 APPLE_ext_string(&arc->label);
       };
    };

   if (ad->use_select == USE_SELECT_ONEUSE && ad->usearc_select != NULL &&
	  oarctype != arctype) {
      ASHset_user_data(ad->use_window,-1);
      av = APPLE_layout_value(ad->usearc_select);
      switch (arctype) {
	 case 0 :
	    ad->picture->use = REMOB(ad->usearc_select,ad->picture->use);
	    fld = (APPLE_GELO_FIELD) APPLE_new_usefield(av);
	    fld->match = arc->match;
	    ad->picture->use = APPEND(fld,ad->picture->use);
	    ad->usearc_select = NULL;
	    break;
	 case 1 :
	 case 2 :
	    { APPLE_VALUE_B tmpv;
	      tmpv = arc->from;
	      arc->from = arc->to;
	      arc->to = tmpv;
	    };
	    break;
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	alt_btn -- handle alternative setting for current object	*/
/*									*/
/************************************************************************/


static int
alt_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("alt_btn 0x%x %s %s",ad,menu,btn);

   if (STREQL(btn,"First Alternative")) {
      ad->picture = ad->first_picture;
      ad->prev_picture = NULL;
    }
   else if (ad->picture == NULL) return FALSE;
   else if (STREQL(btn,"Next Alternative")) {
      if (ad->picture == NULL) {
	 APPLE_new_picture(ad,APPLE_MODE_NONE);
       };
      if (ad->picture->alternative != NULL) {
	 ad->prev_picture = ad->picture;
	 ad->picture = ad->picture->alternative;
       }
      else if (ad->picture->cond_fct != NULL || ad->picture->cond_match != NULL ||
		  ad->picture->cond_null || ad->picture->cond_layout ||
		  ad->picture->cond_array || ad->picture->cond_nested) {
	 ad->prev_picture = ad->picture;
	 ad->picture = NULL;
	 get_conditions(ad);
       }
      else {
	 APPLE_message(ad->window,"This is the default alternative");
	 return FALSE;
       };
    }
   else if (STREQL(btn,"New Alternative")) {
      ad->picture = NULL;
      get_conditions(ad);
    }
   else {
      return FALSE;
    };

   if (ad->picture == NULL || ad->mode == APPLE_MODE_NONE) top_option_btn(ad,FALSE);

   APPLE_set_picture_mode(ad);

   APPLE_menu_setup(ad);
   APPLE_field_menu_setup(ad);
   APPLE_draw_picture(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	cond_btn -- handle condition setting for current object 	*/
/*									*/
/************************************************************************/


static int
cond_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("cond_btn 0x%x %s %s",ad,menu,btn);

   if (!get_conditions(ad)) return FALSE;

   if (ad->picture == NULL || ad->mode == APPLE_MODE_NONE) top_option_btn(ad,TRUE);

   button_setup(ad);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	quit_btn -- handle quit request 				*/
/*									*/
/************************************************************************/


static int
quit_btn(ad,menu,btn)
   APPLE_DATA ad;
   String menu;
   String btn;
{
   ITRACE("quit_btn 0x%x %s %s",ad,menu,btn);

   ASHremove(ad->window);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	get_conditions -- dialog box for condition settings		*/
/*									*/
/************************************************************************/


static Boolean
get_conditions(ad)
   APPLE_DATA ad;
{
   String menu[20];
   register Integer ct;
   Integer p0,p1,p2,p3,p4,p5;
   Character match[64];
   Universal d;

   DTRACE("get_conditions 0x%x",ad);

   if (ad->picture == NULL) {
      APPLE_new_picture(ad,APPLE_MODE_NONE);
    };

   ct = 0;
   menu[ct++] = "%CConditions for this Alternative\n";

   menu[ct++] = "%0o If OBJECT is Null";
   menu[ct++] = "%4o Match String: %1.24t";
   if (ad->picture->cond_fct == NULL) {
      menu[ct++] = "%3.1o Based on User Conditon";
    }
   else {
      menu[ct++] = "%3.1o Based on New User Condition";
      menu[ct++] = "%3.2o Edit Current User Condition";
      menu[ct++] = "%3.3o Based on Current User Condition";
      menu[ct++] = "%3.0o No User Condition";
    };
   menu[ct++] = "";
   menu[ct++] = "%2o Use only in a layout";
   menu[ct++] = "%5o Use only if nested";
   menu[ct++] = "%6o Use only in an array";
   menu[ct++] = "";
   menu[ct++] = "   %a%M   %c";

   menu[ct] = 0;

   p0 = (ad->picture->cond_null);
   p1 = (ad->picture->cond_layout);
   p2 = (ad->picture->cond_fct == NULL ? 0 : 3);
   p4 = (ad->picture->cond_nested);
   p5 = (ad->picture->cond_array);

   if (ad->picture->cond_match != NULL) {
      strcpy(match,ad->picture->cond_match);
      p3 = TRUE;
    }
   else {
      match[0] = 0;
      p3 = FALSE;
    };

   if (!STEMdialog(ad->window,menu,&p0,match,&p1,&p2,&p3,&p4,&p5)) return FALSE;

   ad->picture->cond_null = p0;
   ad->picture->cond_layout = p1;
   ad->picture->cond_array = p5;
   ad->picture->cond_nested = p4;

   switch (p2) {
      default :
	 break;
      case 0 :
	 ad->picture->cond_fct = NULL;
	 break;
      case 1 :
	 d = NULL;
	 if (APPLE_ext_inq_condition(&d)) {
	    ad->picture->cond_fct = (Integer) d;
	  };
	 break;
      case 2 :
	 d = (Universal) ad->picture->cond_fct;
	 if (APPLE_ext_inq_condition(&d)) {
	    ad->picture->cond_fct = (Integer) d;
	  };
	 break;
    };

   if (ad->picture->cond_match != NULL && !p3) {
      ad->picture->cond_match = NULL;
    }
   else if (match[0] != 0) {
      if (ad->picture->cond_match == NULL ||
	     STRNEQ(match,ad->picture->cond_match)) {
	 ad->picture->cond_match = SALLOC(match);
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	test_field_active -- check if given field should be hilit	*/
/*									*/
/************************************************************************/


static Boolean
test_field_active(ad,fnm)
   APPLE_DATA ad;
   String fnm;
{
   register Sequence l;
   register APPLE_VALUE v;
   register APPLE_GELO_ARC arc;
   register APPLE_GELO_LAYOUT lay;
   register APPLE_GELO_FIELD fld;
   register APPLE_GELO_BOX box;
   register APPLE_GELO ag;

   DTRACE("test_field_active 0x%x %s",ad,fnm);

   if (ad->use_select == USE_SELECT_USE && ad->picture != NULL) {
      forin (ag,APPLE_GELO,l,ad->picture->use) {
	 v = APPLE_layout_value(ag);
	 if (v != NULL && v->name != NULL && STREQL(v->name,fnm)) {
	    return TRUE;
	  };
       };
    }
   else if (ad->usearc_select != NULL && ad->picture != NULL) {
      v = APPLE_layout_value(ad->usearc_select);
      if (v != NULL && v->name != NULL && STREQL(v->name,fnm)) {
	 return TRUE;
       };
    }
   else if (ad->mode == APPLE_MODE_LAYOUT) {
      lay = (APPLE_GELO_LAYOUT) ad->picture->object;
      forin (fld,APPLE_GELO_FIELD,l,lay->components) {
	 v = &(fld->value);
	 if (v->name != NULL && STREQL(v->name,fnm)) {
	    return TRUE;
	  };
       };
    }
   else if (ad->mode == APPLE_MODE_BOX) {
      box = (APPLE_GELO_BOX) ad->picture->object;
      if (box != NULL && box->value.name != NULL && STREQL(box->value.name,fnm)) {
	 return TRUE;
       };
    }
   else if (ad->selection != NULL) {
      if (ad->mode == APPLE_MODE_ARC) {
	 arc = (APPLE_GELO_ARC) ad->picture->object;
	 if (ad->selection == SELECT_FROM) v = &arc->from;
	 else if (ad->selection == SELECT_TO) v = &arc->to;
	 else if (ad->selection == SELECT_LABEL) v = &arc->label;
	 else v = NULL;
	 if (v != NULL && v->name != NULL && STREQL(v->name,fnm)) {
	    return TRUE;
	  };
       }
      else if (ad->selection->object != NULL) {
	 if (ad->selection->object->mode == APPLE_MODE_LAYOUT) {
	    lay = (APPLE_GELO_LAYOUT) ad->selection->object;
	    forin (fld,APPLE_GELO_FIELD,l,lay->components) {
	       v = &(fld->value);
	       if (v->name != NULL && STREQL(v->name,fnm)) {
		  return TRUE;
		};
	     }
	  }
	 else if (ad->selection->object->mode == APPLE_MODE_FIELD) {
	    fld = (APPLE_GELO_FIELD) ad->selection->object;
	    if (fld->value.name != NULL && STREQL(fld->value.name,fnm)) {
	       return TRUE;
	     };
	  }
	 else if (ad->selection->object->mode == APPLE_MODE_POINTER) {
	    arc = (APPLE_GELO_ARC) ad->selection->object;
	    if (arc->to.name != NULL && STREQL(arc->to.name,fnm)) {
	       return TRUE;
	     };
	  }
	 else if (ad->selection->object->mode == APPLE_MODE_BOX) {
	    box = (APPLE_GELO_BOX) ad->selection->object;
	    if (box->value.name != NULL && STREQL(box->value.name,fnm)) {
	       return TRUE;
	     };
	  };
       };
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	set_field_value -- handle setting field value			*/
/*									*/
/************************************************************************/


static Boolean
set_field_value(ad,boxfg,v)
   APPLE_DATA ad;
   Boolean boxfg;
   APPLE_VALUE v;
{
   register APPLE_VALUE nv;
   register APPLE_GELO obj;
   register Sequence l;
   register APPLE_GELO_ARC arc;
   register APPLE_GELO_FIELD fld;

   DTRACE("set_field_value 0x%x %d 0x%x",ad,boxfg,v);

   if (ad->selection != NULL && ad->mode != APPLE_MODE_ARC) {
      if (ad->selection->object == NULL) {
	 if (boxfg) APPLE_set_type(ad,ad->selection,APPLE_MODE_BOX);
	 else APPLE_set_type(ad,ad->selection,APPLE_MODE_FIELD);
       };
      obj = ad->selection->object;
    }
   else if (ad->use_select == USE_SELECT_ONEUSE) obj = NULL;
   else if (ad->use_select != USE_SELECT_NONE) obj = NULL;
   else {
      obj = ad->picture->object;
      if (obj == NULL) return FALSE;
    };

   if (ad->picture != NULL &&
	  (ad->use_select == USE_SELECT_USE || ad->use_select == USE_SELECT_ONEUSE)) {
      forin (obj,APPLE_GELO,l,ad->picture->use) {
	 nv = APPLE_layout_value(obj);
	 if (nv != NULL && nv->name != NULL && v->name != NULL &&
		STREQL(nv->name,v->name)) {
	    if (ad->use_select == USE_SELECT_ONEUSE &&
		   ad->usearc_select == obj) {
	       ad->picture->use = REMOB(obj,ad->picture->use);
	       ad->usearc_select = NULL;
	     }
	    else {
	       ad->use_select = USE_SELECT_ONEUSE;
	       ad->usearc_select = obj;
	     };
	    break;
	  };
       };
      if (EMPTY(l)) {
	 obj = APPLE_new_usefield(v);
	 ad->picture->use = APPEND(obj,ad->picture->use);
       };
      ASHset_user_data(ad->use_window,-1);
    }
   else if (obj == NULL) {
      if (ad->picture == NULL) return FALSE;
    }
   else switch (obj->mode) {
      case APPLE_MODE_BOX :
       { register APPLE_GELO_BOX box;
	 box = (APPLE_GELO_BOX) obj;
	 box->value = *v;
	 break;
       };
      case APPLE_MODE_LAYOUT :
       { register APPLE_GELO_LAYOUT lay;
	 lay = (APPLE_GELO_LAYOUT) obj;
	 forin (fld,APPLE_GELO_FIELD,l,lay->components) {
	    nv = &(fld->value);
	    if (nv->name != NULL && v->name != NULL && STREQL(nv->name,v->name)) {
	       if (ad->usearc_select != NULL &&
		      ((APPLE_GELO) fld) == ad->usearc_select) {
		  lay->components = REMOB(fld,lay->components);
		  ad->usearc_select = NULL;
		}
	       else {
		  ad->usearc_select = (APPLE_GELO) fld;
		};
	       break;
	     };
	  };
	 if (EMPTY(l)) {
	    nv = PALLOC(APPLE_VALUE_B);
	    *nv = *v;
	    lay->components = APPEND(APPLE_new_usefield(nv),lay->components);
	  };
	 break;
       };
      case APPLE_MODE_FIELD :
       { register APPLE_GELO_FIELD field;
	 if (ad->use_select == USE_SELECT_ONEUSE) return FALSE;
	 field = (APPLE_GELO_FIELD) obj;
	 field->value = *v;
	 break;
       };
      case APPLE_MODE_POINTER :
       { arc = (APPLE_GELO_ARC) obj;
	 arc->to = *v;
	 break;
       };
      case APPLE_MODE_ARC :
       { arc = (APPLE_GELO_ARC) obj;
	 if (ad->use_select == USE_SELECT_ONEUSE) {
	    arc->label = *v;
	  }
	 else {
	    if (ad->selection == SELECT_FROM) arc->from = *v;
	    else if (ad->selection == SELECT_TO) arc->to = *v;
	    else if (ad->selection == SELECT_LABEL) arc->label = *v;
	    else return FALSE;
	  };
	 break;
       };
      default :
	 return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	file_menu_setup -- setup buttons on the FILE menu		*/
/*									*/
/************************************************************************/


static void
file_menu_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("file_menu_setup 0x%x",ad);

   if (ad->mode != APPLE_MODE_TILED) {
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Show Grid",FALSE);
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Finer Grid",FALSE);
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Courser Grid",FALSE);
    }
   else {
      if (ad->grid < MAX_GRID_SIZE)
	 STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Finer Grid",TRUE);
      else
	 STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Finer Grid",FALSE);
      if (ad->grid > MIN_GRID_SIZE)
	 STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Courser Grid",TRUE);
      else
	 STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Courser Grid",FALSE);
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Show Grid",TRUE);
      if (ad->show_grid)
	 STEMpdm_btn_select(ad->menu_win,FILE_MENU,"Show Grid",TRUE);
      else
	 STEMpdm_btn_select(ad->menu_win,FILE_MENU,"Show Grid",FALSE);
    };

   if (ad->mode == APPLE_MODE_NONE) {
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Clear",FALSE);
    }
   else {
      STEMpdm_btn_enable(ad->menu_win,FILE_MENU,"Clear",TRUE);
    };
};






/************************************************************************/
/*									*/
/*	top_menu_setup -- setup top menu buttons			*/
/*									*/
/************************************************************************/


static void
top_menu_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("top_menu_setup 0x%x",ad);

   if (ad->mode == APPLE_MODE_NONE) {
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Default",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Design",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Tiled",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Layout",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Box",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Arc",TRUE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Ignore",TRUE);
    }
   else {
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Default",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Design",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Tiled",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Layout",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Box",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Arc",FALSE);
      STEMpdm_btn_enable(ad->menu_win,TOP_MENU,"Ignore",FALSE);
    };
};





/************************************************************************/
/*									*/
/*	edit_menu_setup -- setup buttons on the EDIT menu		*/
/*									*/
/************************************************************************/


static void
edit_menu_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("edit_menu_setup 0x%x",ad);

   if (ad->selection == NULL || ad->mode == APPLE_MODE_ARC) {
      STEMpdm_menu_enable(ad->menu_win,EDIT_MENU,FALSE);
    }
   else {
      STEMpdm_menu_enable(ad->menu_win,EDIT_MENU,TRUE);

      if (ad->picture == NULL || !ad->picture->cond_layout)
	 STEMpdm_btn_enable(ad->menu_win,EDIT_MENU,"Pointer",FALSE);
      else
	 STEMpdm_btn_enable(ad->menu_win,EDIT_MENU,"Pointer",TRUE);

      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Box",FALSE);
      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Layout",FALSE);
      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Field",FALSE);
      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Pointer",FALSE);
      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Ignore",FALSE);
      STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Empty",FALSE);

      if (ad->selection->object == NULL) {
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Empty",TRUE);
       }
      else switch (ad->selection->object->mode) {
	 case APPLE_MODE_BOX :
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Box",TRUE);
	    break;
	 case APPLE_MODE_FIELD :
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Field",TRUE);
	    break;
	 case APPLE_MODE_POINTER :
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Pointer",TRUE);
	    break;
	 case APPLE_MODE_IGNORE :
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Ignore",TRUE);
	    break;
	 case APPLE_MODE_LAYOUT :
	    STEMpdm_btn_select(ad->menu_win,EDIT_MENU,"Layout",TRUE);
	    break;
	 default :
	    break;
       };
    };
};





/************************************************************************/
/*									*/
/*	prop_menu_setup -- setup buttons on PROPERTY menu		*/
/*									*/
/************************************************************************/


static void
prop_menu_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("prop_menu_setup 0x%x",ad);
};





/************************************************************************/
/*									*/
/*	button_setup -- setup buttons for data/use			*/
/*									*/
/************************************************************************/


static void
button_setup(ad)
   APPLE_DATA ad;
{
   DTRACE("button_setup 0x%x",ad);
};





/* end of applemenu.c */

